/*
 * Line6 Linux USB driver - 0.8.1
 *
 * Copyright (C) 2004-2009 Markus Grabner (grabner@icg.tugraz.at)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 */

#include "driver.h"

#include <linux/usb.h>

#include "control.h"
#include "pod.h"
#include "usbdefs.h"
#include "variax.h"

#define DEVICE_ATTR2(_name1, _name2, _mode, _show, _store) \
struct device_attribute dev_attr_##_name1 = __ATTR(_name2, _mode, _show, _store)

#define LINE6_PARAM_R(PREFIX, prefix, type, param) \
static ssize_t prefix ## _get_ ## param(struct device *dev, \
			struct device_attribute *attr, char *buf) \
{ \
	return prefix ## _get_param_ ## type(dev, buf, PREFIX ## _ ## param); \
}

#define LINE6_PARAM_RW(PREFIX, prefix, type, param) \
LINE6_PARAM_R(PREFIX, prefix, type, param); \
static ssize_t prefix ## _set_ ## param(struct device *dev, \
		struct device_attribute *attr, const char *buf, size_t count) \
{ \
	return prefix ## _set_param_ ## type(dev, buf, count, PREFIX ## _ ## param); \
}

#define POD_PARAM_R(type, param) LINE6_PARAM_R(POD, pod, type, param)
#define POD_PARAM_RW(type, param) LINE6_PARAM_RW(POD, pod, type, param)
#define VARIAX_PARAM_R(type, param) LINE6_PARAM_R(VARIAX, variax, type, param)
#define VARIAX_PARAM_RW(type, param) LINE6_PARAM_RW(VARIAX, variax, type, param)


static ssize_t pod_get_param_int(struct device *dev, char *buf, int param)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_line6_pod *pod = usb_get_intfdata(interface);
	int retval = line6_wait_dump(&pod->dumpreq, 0);
	if (retval < 0)
		return retval;
	return sprintf(buf, "%d\n", pod->prog_data.control[param]);
}

static ssize_t pod_set_param_int(struct device *dev, const char *buf, size_t count, int param)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_line6_pod *pod = usb_get_intfdata(interface);
	int value = simple_strtoul(buf, NULL, 10);
	pod_transmit_parameter(pod, param, value);
	return count;
}

static ssize_t variax_get_param_int(struct device *dev, char *buf, int param)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_line6_variax *variax = usb_get_intfdata(interface);
	int retval = line6_wait_dump(&variax->dumpreq, 0);
	if (retval < 0)
		return retval;
	return sprintf(buf, "%d\n", variax->model_data.control[param]);
}

static ssize_t variax_get_param_float(struct device *dev, char *buf, int param)
{
	/*
		We do our own floating point handling here since at the time
		this code was written (Jan 2006) it was highly discouraged to
		use floating point arithmetic in the kernel. If you feel that
		this no longer applies, feel free to replace this by generic
		floating point code.
	*/

	static const int BIAS = 0x7f;
	static const int OFFSET = 0xf;
	static const int PRECISION = 1000;

	int len = 0;
	unsigned part_int, part_frac;
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_line6_variax *variax = usb_get_intfdata(interface);
	const unsigned char *p = variax->model_data.control + param;
	int retval = line6_wait_dump(&variax->dumpreq, 0);
	if (retval < 0)
		return retval;

	if ((p[0] == 0) && (p[1] == 0) && (p[2] == 0))
		part_int = part_frac = 0;
	else {
		int exponent = (((p[0] & 0x7f) << 1) | (p[1] >> 7)) - BIAS;
		unsigned mantissa = (p[1] << 8) | p[2] | 0x8000;
		exponent -= OFFSET;

		if (exponent >= 0) {
			part_int = mantissa << exponent;
			part_frac = 0;
		} else {
			part_int = mantissa >> -exponent;
			part_frac = (mantissa << (32 + exponent)) & 0xffffffff;
		}

		part_frac = (part_frac / ((1UL << 31) / (PRECISION / 2 * 10)) + 5) / 10;
	}

	len += sprintf(buf + len, "%s%d.%03d\n", ((p[0] & 0x80) ? "-" : ""), part_int, part_frac);
	return len;
}

