//////////////////////////////////////////////////////////////////////
// PSKMod.cpp: implementation of the PSKModulator class.
//
//////////////////////////////////////////////////////////////////////
//      PSK31/CW modulator
// Copyright 1999.    Moe Wheatley AE4JY  <ae4jy@mindspring.com>
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//////////////////////////////////////////////////////////////////////
//
// modified by Volker Schroer , DL1KSV for use in LinPsk

#include <math.h>
#include "psktable.h"
#include "pskmod.h"


// local defines.................

#define PHZ_0 0			//specify various signal phase states
#define PHZ_90 1
#define PHZ_180 2
#define PHZ_270 3
#define PHZ_OFF 4

#define SYM_NOCHANGE 0	//Stay the same phase
#define SYM_P90 1		//Plus 90  deg
#define SYM_P180 2		//Plus 180 deg
#define SYM_M90 3		//Minus 90 deg
#define SYM_OFF 4		//No output
#define SYM_ON 5		//constant output


#define SYMBOL_RATE 31.25		// 31.25 Symbols per Second

#define MAXRAMP_SIZE (((100*11100)/3125)+1) // max number of envelope ramp steps per symbol

#define TX_CONSTANT 23000.0		// TX Amplitude Factor

#define CW_SPEED 1		//bigger is slower. 1 is fastest.

// The use of static's here preclude having multiple instantiations
// of this class but should not be an issue since only one soundcard.

//Ramp shape tables that are loaded with cosine shaped functions at init
static double PSKShapeTbl_Z[MAXRAMP_SIZE];	// 0 
static double PSKShapeTbl_P[MAXRAMP_SIZE];	// +1
static double PSKShapeTbl_M[MAXRAMP_SIZE];	// -1
static double PSKShapeTbl_ZP[MAXRAMP_SIZE];	// 0 to +1
static double PSKShapeTbl_PZ[MAXRAMP_SIZE];	// +1 to 0
static double PSKShapeTbl_MZ[MAXRAMP_SIZE];	// -1 to 0
static double PSKShapeTbl_PM[MAXRAMP_SIZE];	// +1 to -1
static double PSKShapeTbl_MP[MAXRAMP_SIZE];	// -1 to +1

struct PSKStruct
{
	double* iptr;
	double* qptr;
	
	int	next;
};

typedef PSKStruct PSKSTATE;

//Lookup table for determining the next ramp shape depending on the
//  next symbol and the present output phase.
// indexing format is [symbol][presentPhase]
//returns the PSKSTRUCT containing the next phase and the I and Q
//   ramp table pointers.

static PSKSTATE PSKPhaseLookupTable[6][5]=
{
// SYMBOL = 0 = SYM_NOCHANGE
//   I ramp shape     Q ramp shape     Next Phase
	PSKShapeTbl_P, PSKShapeTbl_P, PHZ_0,	//present PHZ_0
	PSKShapeTbl_M, PSKShapeTbl_P, PHZ_90,	//present PHZ_90
	PSKShapeTbl_M, PSKShapeTbl_M, PHZ_180,	//present PHZ_180
	PSKShapeTbl_P, PSKShapeTbl_M, PHZ_270,	//present PHZ_270
	PSKShapeTbl_Z, PSKShapeTbl_Z, PHZ_OFF,	//present PHZ_OFF
// SYMBOL = 1 = SYM_P90 = Advance 90 degrees
//   I ramp shape     Q ramp shape     Next Phase
	PSKShapeTbl_PM, PSKShapeTbl_P, PHZ_90,	//present PHZ_0
	PSKShapeTbl_M, PSKShapeTbl_PM, PHZ_180,	//present PHZ_90
	PSKShapeTbl_MP, PSKShapeTbl_M, PHZ_270,	//present PHZ_180
	PSKShapeTbl_P, PSKShapeTbl_MP, PHZ_0,	//present PHZ_270
	PSKShapeTbl_ZP, PSKShapeTbl_ZP, PHZ_0,	//present PHZ_OFF
// SYMBOL = 2 = SYM_P180 = Advance 180 degrees
//   I ramp shape     Q ramp shape     Next Phase
	PSKShapeTbl_PM, PSKShapeTbl_PM, PHZ_180,//present PHZ_0
	PSKShapeTbl_MP, PSKShapeTbl_PM, PHZ_270,//present PHZ_90
	PSKShapeTbl_MP, PSKShapeTbl_MP, PHZ_0,	//present PHZ_180
	PSKShapeTbl_PM, PSKShapeTbl_MP, PHZ_90,	//present PHZ_270
	PSKShapeTbl_ZP, PSKShapeTbl_ZP, PHZ_0,	//present PHZ_OFF
// SYMBOL = 3 = SYM_M90	= retard 90 degrees
//   I ramp shape     Q ramp shape     Next Phase
	PSKShapeTbl_P, PSKShapeTbl_PM, PHZ_270,	//present PHZ_0
	PSKShapeTbl_MP, PSKShapeTbl_P, PHZ_0,	//present PHZ_90
	PSKShapeTbl_M, PSKShapeTbl_MP, PHZ_90,	//present PHZ_180
	PSKShapeTbl_PM, PSKShapeTbl_M, PHZ_180,	//present PHZ_270
	PSKShapeTbl_ZP, PSKShapeTbl_ZP, PHZ_0,	//present PHZ_OFF
// SYMBOL = 4 = SYM_OFF
//   I ramp shape     Q ramp shape     Next Phase
	PSKShapeTbl_PZ, PSKShapeTbl_PZ, PHZ_OFF,//present PHZ_0
	PSKShapeTbl_MZ, PSKShapeTbl_PZ, PHZ_OFF,//present PHZ_90
	PSKShapeTbl_MZ, PSKShapeTbl_MZ, PHZ_OFF,//present PHZ_180
	PSKShapeTbl_PZ, PSKShapeTbl_MZ, PHZ_OFF,//present PHZ_270
	PSKShapeTbl_Z, PSKShapeTbl_Z, PHZ_OFF,	//present PHZ_OFF
// SYMBOL = 5 = SYM_ON	
//   I ramp shape     Q ramp shape     Next Phase
	PSKShapeTbl_P, PSKShapeTbl_P, PHZ_0,	//present PHZ_0
	PSKShapeTbl_MP, PSKShapeTbl_P, PHZ_0,	//present PHZ_90
	PSKShapeTbl_MP, PSKShapeTbl_MP, PHZ_0,	//present PHZ_180
	PSKShapeTbl_P, PSKShapeTbl_MP, PHZ_0,	//present PHZ_270
	PSKShapeTbl_ZP, PSKShapeTbl_ZP, PHZ_0	//present PHZ_OFF
};
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

PSKModulator::PSKModulator(int FS,double freq,CTxdisplay *txwindow): CModulator(FS,txwindow)
{

	m_AmblePtr = 0;
int RampSize;
int i;


	m_PSKPhaseInc = PI2 * freq/(double)SampleRate;		//carrier frequency
	
//	if( settings.status == TX_CWID_STATE )
//  	m_PSKmode = CW_MODE;
	m_PSKSecPerSamp = 1.0/(double)SampleRate;
	m_PSKTime = 0.0;
	m_t = 0.0;
	m_Ramp = 0;
	m_PSKPeriodUpdate = 1.0/SYMBOL_RATE;	//symbol period
	m_Lastsymb = SYM_OFF;
	m_AddEndingZero = true;
	m_CWState = 0;
	m_CWtimer = 0;
// Generate cosine ramp envelope lookup tables
	RampSize =  (((100*SampleRate)/3125)+1); //  number of envelope ramp steps per symbol
	for( i=0; i<RampSize; i++)
	{
		PSKShapeTbl_Z[i] = 0.0;
		PSKShapeTbl_P[i] = 1.0;
		PSKShapeTbl_M[i] = -1.0;
		PSKShapeTbl_PM[i] = cos( (double)i*PI2/(RampSize*2) );
		PSKShapeTbl_MP[i] = -PSKShapeTbl_PM[i];

		if( i <RampSize/2 )
		{
			PSKShapeTbl_PZ[i] = cos( (double)i*PI2/(RampSize*2) );
			PSKShapeTbl_MZ[i] = -PSKShapeTbl_PZ[i];
			PSKShapeTbl_ZP[i] = 0.0;
		}
		else
		{
			PSKShapeTbl_ZP[i] = -cos( (double)i*PI2/(RampSize*2) );
			PSKShapeTbl_PZ[i] = 0.0;
			PSKShapeTbl_MZ[i] = 0.0;
		}

	}
	i = 0;
	while(i<32)		//create post/preamble tables
	{
		m_Preamble[i] = TXTOG_CODE;
		m_Postamble[i++] = TXON_CODE;
	}
	m_Preamble[i] = 0;		// null terminate these tables
	m_Postamble[i] = 0;
	
	m_pPSKtxI = PSKShapeTbl_Z;
	m_pPSKtxQ = PSKShapeTbl_Z;
	m_PresentPhase = PHZ_OFF;
	m_TxShiftReg = 0;
	m_TxCodeWord = 0;

}

PSKModulator::~PSKModulator()
{
}

///////////++++++++++++++++++++++++++++++++++////////////////
///////////   P S K 3 1   M O D U L A T O R  ////////////////
///////////++++++++++++++++++++++++++++++++++////////////////

/////////////////////////////////////////////////////////////
//Initialize the PSK/CW modulator with the sample frequncy Fs,
// the carrier frequency freq, and the PSK mode.
/////////////////////////////////////////////////////////////
/*)void PSKModulator::InitPSKModulator(double freq,CTxdisplay *txwindow)
{
int RampSize;
int i;

// Save Adresses of window- components
tx=txwindow;

	m_PSKPhaseInc = PI2 * freq/(double)SampleRate;		//carrier frequency
	
//	if( settings.status == TX_CWID_STATE )
//  	m_PSKmode = CW_MODE;
	m_PSKSecPerSamp = 1.0/(double)SampleRate;
	m_PSKTime = 0.0;
	m_t = 0.0;
	m_Ramp = 0;
	m_PSKPeriodUpdate = 1.0/SYMBOL_RATE;	//symbol period
	m_Lastsymb = SYM_OFF;
	m_AddEndingZero = true;
	m_CWState = 0;
	m_CWtimer = 0;
// Generate cosine ramp envelope lookup tables
	RampSize =  (((100*SampleRate)/3125)+1); //  number of envelope ramp steps per symbol
	for( i=0; i<RampSize; i++)
	{
		PSKShapeTbl_Z[i] = 0.0;
		PSKShapeTbl_P[i] = 1.0;
		PSKShapeTbl_M[i] = -1.0;
		PSKShapeTbl_PM[i] = cos( (double)i*PI2/(RampSize*2) );
		PSKShapeTbl_MP[i] = -PSKShapeTbl_PM[i];

		if( i <RampSize/2 )
		{
			PSKShapeTbl_PZ[i] = cos( (double)i*PI2/(RampSize*2) );
			PSKShapeTbl_MZ[i] = -PSKShapeTbl_PZ[i];
			PSKShapeTbl_ZP[i] = 0.0;
		}
		else
		{
			PSKShapeTbl_ZP[i] = -cos( (double)i*PI2/(RampSize*2) );
			PSKShapeTbl_PZ[i] = 0.0;
			PSKShapeTbl_MZ[i] = 0.0;
		}

	}
	i = 0;
	while(i<32)		//create post/preamble tables
	{
		m_Preamble[i] = TXTOG_CODE;
		m_Postamble[i++] = TXON_CODE;
	}
	m_Preamble[i] = 0;		// null terminate these tables
	m_Postamble[i] = 0;
	
	m_pPSKtxI = PSKShapeTbl_Z;
	m_pPSKtxQ = PSKShapeTbl_Z;
	m_PresentPhase = PHZ_OFF;
	m_TxShiftReg = 0;
	m_TxCodeWord = 0;
} */


/////////////////////////////////////////////////////////////
// generates n samples of psk31 waveform in data array pData
/////////////////////////////////////////////////////////////
unsigned int PSKModulator::CalcSignal(double *pData, int n)

{
int symbol;
int i;
/**	if( m_PSKmode != CW_MODE)
		m_PSKmode = mode;	//change mode on fly if not CW */
		m_RMSConstant = TX_CONSTANT;

	for( i=0; i<n; i++ )		//calculate n samples of tx data stream
	{
		m_t += m_PSKPhaseInc;			// increment radian phase count
		
// create sample from sin/cos and shape tables
		pData[i] = m_RMSConstant*( m_pPSKtxI[m_Ramp]*sin( m_t ) + m_pPSKtxQ[m_Ramp++]*cos( m_t ) );
		m_PSKTime += m_PSKSecPerSamp;
		if( m_PSKTime >= m_PSKPeriodUpdate )//if time to update envelope ramp index
		{
			m_PSKTime -= m_PSKPeriodUpdate;	//keep time bounded
			m_Ramp = 0;						// time to update symbol
			m_t = fmod(m_t,PI2);			//keep radian counter bounded
      if (settings.status == TX_CWID_STATE)
					symbol = GetNextCWSymbol();
			else				
			switch( PSKMode)				//get next symbol to send
				{
				case BPSK:
					symbol = GetNextBPSKSymbol();
					break;
				case QPSK:
					symbol = GetNextQPSKSymbol();
					break;
/*				case IQPSK:	
						symbol = GetNextQPSKSymbol();
						if(symbol==SYM_P90)		//rotate vectors the opposite way
							symbol = SYM_M90;
						else
							if(symbol==SYM_M90)
							symbol = SYM_P90;
					break; */
				}
			//get new I/Q ramp tables and next phase
			m_pPSKtxI = PSKPhaseLookupTable[symbol][m_PresentPhase].iptr;
			m_pPSKtxQ = PSKPhaseLookupTable[symbol][m_PresentPhase].qptr;
			m_PresentPhase = PSKPhaseLookupTable[symbol][m_PresentPhase].next;
		}
	if (settings.status == TX_END_STATE) // We have reached end of Transmission
		return i;
	}
return n;
}


/////////////////////////////////////////////////////////////
// called every symbol time to get next CW symbol and get the
// next character from the character Queue if no more symbols
// are left to send.
/////////////////////////////////////////////////////////////
char PSKModulator::GetNextCWSymbol(void)
{
char symb;
int ch;
	symb = m_Lastsymb;		//use last symbol unless it needs to change
	if( (m_TxShiftReg == 0 ) && (m_CWState == 0) )
	{
		ch = GetChar();			//get next character to xmit
		if( ch >=0 )			//if is not a control code
		{
			ch &= 0xFF;		
			ch = (int)toupper( (char)ch );	//make upper case
			if( ch>=' ' && ch<='Z')
				m_TxShiftReg = CW_TABLE[ ch-' '];	//look up pattern
		}
		else					// is a control code
		{
			if( ch == TXON_CODE )
				symb = SYM_ON;
			else
				symb = SYM_OFF;
			return symb;
		}
		m_CWState = 0;
	}
	switch( m_CWState )		// CW timing state machine
	{
		case 0:			//get next cw symbol state from pattern
			switch( m_TxShiftReg&0xC000 )
			{
				case 0x4000:	//dot
					m_CWState = 1;
					m_CWtimer = 1*CW_SPEED;
					symb = SYM_ON;
					break;
				case 0x8000:	//dash
					m_CWState = 1;
					m_CWtimer = 3*CW_SPEED;
					symb = SYM_ON;
					break;
				case 0xC000:	//inter char space 
					m_CWState = 2;
					m_CWtimer = 3*CW_SPEED;
					symb = SYM_OFF;
					break;
				default:
					symb = SYM_OFF;
					break;
			}
			m_TxShiftReg = m_TxShiftReg<<2;	//
			break;
		case 1:		//On time state
			if( --m_CWtimer <= 0 )
			{
				m_CWState = 2;
				m_CWtimer = 1*CW_SPEED;	//inter symbol time
				symb = SYM_OFF;
			}
			else
				symb = SYM_NOCHANGE;
			break;
		case 2:		//Off time state
			if( --m_CWtimer <= 0 )
				m_CWState = 0;
			break;
	}
	m_Lastsymb = symb;
	return symb;
}

/////////////////////////////////////////////////////////////
// called every symbol time to get next BPSK symbol and get the
// next character from the character Queue if no more symbols
// are left to send.
/////////////////////////////////////////////////////////////
char PSKModulator::GetNextBPSKSymbol(void)
{
char symb;
int ch;
	symb = m_Lastsymb;
	if( m_TxShiftReg == 0 )
	{
		if( m_AddEndingZero )		// if is end of code
		{
			symb = SYM_P180;		// end with a zero
			m_AddEndingZero = false;
		}
		else
		{
			ch = GetChar();			//get next character to xmit
			if( ch >=0 )			//if is not a control code
			{						//get next VARICODE codeword to send
				m_TxShiftReg = VARICODE_TABLE[ ch&0xFF ];
				symb = SYM_P180;	//Start with a zero
			}
			else					// is a control code
			{
				switch( ch )
				{
				case TXON_CODE:
					symb = SYM_ON;
					break;
				case TXTOG_CODE:
					symb = SYM_P180;
					break;
				case TXOFF_CODE:
					symb = SYM_OFF;
					break;
				}
			}
		}
	}
	else			// is not end of code word so send next bit
	{
		if( m_TxShiftReg&0x8000 )
			symb = SYM_NOCHANGE;
		else
			symb = SYM_P180;
		m_TxShiftReg = m_TxShiftReg<<1;	//point to next bit
		if( m_TxShiftReg == 0 )			// if at end of codeword
			m_AddEndingZero = true;		// need to send a zero nextime
	}
	m_Lastsymb = symb;
	return symb;
}

		
/////////////////////////////////////////////////////////////
// called every symbol time to get next QPSK symbol and get the
// next character from the character Queue if no more symbols
// are left to send.
/////////////////////////////////////////////////////////////
char PSKModulator::GetNextQPSKSymbol(void)
{
char symb;
int ch;
	symb = ConvolutionCodeTable[m_TxShiftReg&0x1F];	//get next convolution code
	m_TxShiftReg = m_TxShiftReg<<1;
	if( m_TxCodeWord == 0 )			//need to get next codeword
	{
		if( m_AddEndingZero )		//if need to add a zero
		{
			m_AddEndingZero = false;	//end with a zero
		}
		else
		{
			ch = GetChar();			//get next character to xmit
			if( ch >=0 )			//if not a control code
			{						//get next VARICODE codeword to send
				m_TxCodeWord = VARICODE_TABLE[ ch&0xFF ];
			}
			else					//is a control code
			{
				switch( ch )
				{
				case TXON_CODE:
					symb = SYM_ON;
					break;
				case TXTOG_CODE:
					m_TxCodeWord = 0;
					break;
				case TXOFF_CODE:
					symb = SYM_OFF;
					break;
				}
			}
		}
	}
	else
	{
		if(m_TxCodeWord&0x8000 )
		{
			m_TxShiftReg |= 1;
		}
		m_TxCodeWord = m_TxCodeWord<<1;
		if(m_TxCodeWord == 0)
			m_AddEndingZero = true;	//need to add another zero
	}
	return symb;
}

/////////////////////////////////////////////////////////////
//get next character/symbol depending on TX state.
/////////////////////////////////////////////////////////////
int PSKModulator::GetChar()
{
int ch;
static int last=0;
	switch( settings.status )
	{
		case TX_OFF_STATE:		//is receiving
			if( settings.DemoMode )	//test mode
				ch = ' '; //MakeTestChar();	//generate some test characters
			else
				ch = TXOFF_CODE;		//else turn off
			break;
		case TX_TUNE_STATE:
			ch = TXON_CODE;				// steady carrier
			break;
		case TX_PAUSED_STATE:
			ch = TXTOG_CODE;			// steady idle symbol
			break;
		case TX_POSTAMBLE_STATE:		// ending sequence
			if( !(ch = m_Postamble[m_AmblePtr++] ))
			{
				if(	settings.NeedCWid  )
				{
					settings.status = TX_CWID_STATE;
//					m_PSKmode = CW_MODE;
					settings.NeedCWid = false;
					m_AmblePtr = 0;
					ch = TXOFF_CODE;
				}
				else
				{
					
					m_AmblePtr = 0;
					ch = TXOFF_CODE;
					settings.status = TX_END_STATE;
					emit finished();
				}
			}
			break;
		case TX_PREAMBLE_STATE:			//starting sequence
			if( !(ch = m_Preamble[m_AmblePtr++] ))
			{
				settings.status = TX_SENDING_STATE;
				m_AmblePtr = 0;
				ch = TXTOG_CODE;
			}
			break;
		case TX_CWID_STATE:				// id sendingnCW ID
			if( m_AmblePtr >= settings.CWIdString.length() )
			{

				m_AmblePtr = 0;
				ch = TXOFF_CODE;
				settings.status = TX_END_STATE;
				emit finished();
			}
			else
			{
//				m_PSKmode = CW_MODE;
			
				ch = settings.CWIdString.at(m_AmblePtr++).cell();

			}
			break;
		case TX_SENDING_STATE:		//if sending text from TX window
		ch = tx->txwindow->getTxChar();
		if (ch >0)
			{
				emit charSend((char) ch);
				m_AmblePtr = 0;
			}
		else
			if ( ch == TXOFF_CODE)
			settings.status=TX_POSTAMBLE_STATE;	
			break;
    case TX_END_STATE:		// Should never be reached here
			break;
	}
	last = ch;
	return( ch );
}

