/****************************************************************
 *
 * vdictov: axis.H
 *
 * Copyright (C) Max Planck Institute 
 * for Human Cognitive and Brain Sciences, Leipzig
 *
 * Author Thomas Arnold, 2002, <lipsia@cbs.mpg.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * $Id: axis.H 687 2004-02-12 09:29:58Z karstenm $
 *
 *****************************************************************/


#ifndef AXIS_H_INCLUDED
#define AXIS_H_INCLUDED

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <float.h>
#include <math.h>
#include <time.h>
#include <assert.h>

extern "C"
{
   #include <Vlib.h>
   #include <VImage.h>
   #include <option.h>
   #include <mu.h>
}


/*------------------------------------------------------------------------------

Axis
====

This class stores a vector of cosines of angles to the x-, y- and z-axis, i.e.
the base point of the axis vector is located at the origin and the end point is
located on the unit sphere.

------------------------------------------------------------------------------*/

class Axis
{
   public:

      Axis ()                          {V[0] = 0;      V[1] = 0;      V[2] = 0;     };
      Axis (float X, float Y, float Z) {V[0] = X;      V[1] = Y;      V[2] = Z;     };
      Axis (const Axis& A)             {V[0] = A.V[0]; V[1] = A.V[1]; V[2] = A.V[2];};

      float&      operator[] (int i)         {return V[i];};
      const Axis& operator=  (const Axis& A) {V[0] = A.V[0]; V[1] = A.V[1]; V[2] = A.V[2]; return *this;};
      bool        operator== (const Axis& A) {return (V[0] == A.V[0]) && (V[1] == A.V[1]) && (V[2] == A.V[2]);};
      bool        operator!= (const Axis& A) {return !(*this == A);};
      bool        operator|  (const Axis& A) {return (*this == A) || (*this == -A);};
      Axis        operator-  () const        {return Axis (-V[0], -V[1], -V[2]);};

      friend Axis  operator* (const Axis& A, const Axis& B);   /* cross product                   */
      friend float Angle     (const Axis& A, const Axis& B);   /* angle between axes (degrees)    */
      friend Axis  Abs       (const Axis& A);                  /* flip to first octant            */
      friend Axis  Round     (const Axis& A);                  /* round to nearest principal axis */

   protected:

      float V[3];   /* cosines of angles (x, y, z) */

}; /* Axis */

/* principal axes */
const Axis NAxis (0, 0, 0);
const Axis XAxis (1, 0, 0);
const Axis YAxis (0, 1, 0);
const Axis ZAxis (0, 0, 1);

/*----------------------------------------------------------------------------*/

inline Axis operator* (const Axis& A, const Axis& B)
{
   Axis C;   /* cross product */


   /* multiply */
   C[0] = A.V[1] * B.V[2] - A.V[2] * B.V[1];
   C[1] = A.V[2] * B.V[0] - A.V[0] * B.V[2];
   C[2] = A.V[0] * B.V[1] - A.V[1] * B.V[0];

   return C;

} /* operator* */

/*----------------------------------------------------------------------------*/

inline float Angle (const Axis& A, const Axis& B)
{
   float angle;   /* angle */


   /* compute angle between axes */
   angle = A.V[0] * B.V[0] + A.V[1] * B.V[1] + A.V[2] * B.V[2];
   angle = acos (angle) * 180 / M_PI;

   return angle;

} /* Angle */

/*----------------------------------------------------------------------------*/

inline Axis Abs (const Axis& A)
{
   Axis C;   /* flipped axis */


   /* flip to first octant */
   C[0] = fabs (A.V[0]);
   C[1] = fabs (A.V[1]);
   C[2] = fabs (A.V[2]);

   return C;

} /* Abs */

/*----------------------------------------------------------------------------*/

inline Axis Round (const Axis& A)
{
   Axis C;   /* rounded axis */


   /* round to nearest principal axis */
   for (int i = 0; i < 3; i++)
      if      (A.V[i] <= -(1.0 / sqrt (2.0))) C[i] = -1;
      else if (A.V[i] >=  (1.0 / sqrt (2.0))) C[i] =  1;
      else                                    C[i] =  0;

   return C;

} /* Round */

/*----------------------------------------------------------------------------*/

#endif
