package Lire::ChartTypes::Scat;

use strict;

use base qw/Lire::ChartType/;

use Locale::TextDomain 'lire';

=pod

=head1 NAME

Lire::ChartTypes::Scat - Wrapper for the Ploticus scat prefab.

=head1 SYNOPSIS

  my $type = $chart_cfg->get( 'type' )->as_value();
  $type->write_chart( $chart_cfg, $subreport );

=head1 DESCRIPTION

This Lire::ChartType wraps the Ploticus scat prefab.

=cut

sub name {
    return 'scat';
}

sub title {
    return 'Scat chart';
}

sub description {
    return '<para>' . __( 'Scatterplot of two variables with optional regression line.' ) . '</para>';
}

sub prefab {
    return '-prefab scat'
}

sub get_vars {
    my ( $self, $chart_cfg ) = @_;

    return [ $chart_cfg->type_properties()->get( 'x' )->as_value(),
             $chart_cfg->type_properties()->get( 'y' )->as_value() ];
}

sub write_parameters {
    my ( $self, $chart_cfg, $fh ) = @_;

    # We do not link to our parent implemenation
    print $fh "#set title = ", $chart_cfg->title(), "\n"
      if $chart_cfg->title();
    print $fh "#set name = ", $chart_cfg->case_var()->label(), "\n";

    my $props = $chart_cfg->get('type')->get_properties();
    my $xlbl = ( $chart_cfg->xlabel() ? $chart_cfg->xlabel()
                 : $props->get( 'x' )->as_value()->label() );
    my $ylbl = ( $chart_cfg->ylabel() ? $chart_cfg->ylabel()
                 : $props->get( 'y' )->as_value()->label() );
    my $corr = $props->get( 'corr' )->as_value() ? 'yes' : 'no';
    print $fh <<EOF;
#set x = 2
#set xlbl = $xlbl
#set y = 3
#set ylbl = $ylbl
#set corr = $corr
EOF
    print $fh "#set id = 1\n"
      if $props->get( 'label' )->as_value();

    return;
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

 Lire::PluginManager(3pm) Lire::ChartType(3pm), ploticus(1)

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: Scat.pm,v 1.1 2004/07/22 16:46:18 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

