package Lire::DlfAnalyser;

use strict;

use base qw/Lire::Plugin/;

use Carp;

=pod

=head1 NAME

Lire::DlfAnalyser - Base interface for all DLF analysers.

=head1 SYNOPSIS

  use base qw/ Lire::DlfAnalyser /;


=head1 DESCRIPTION

This package defines the interface that must be implemented by all DLF
Analysers. All the methods defined in this package will throw a
'method not implemented' exception if they are called.

Analyser which produce DLF records for an extended schema should
probably use the simpler Lire::DlfCategoriser interface.

=head1 META INFORMATION METHODS

These methods provides information to the Lire framework about the DLF
analysers.

The Lire::DlfAnalyser interface extends the Lire::Plugin interface, so
they must implement all standard meta-information methods (name(),
title(), description()).

=pod

=head2 src_schema()

Returns the Dlf schema which is analysed by this analyser.

E.g. C<www>

=cut

sub src_schema {
    croak "src_schema() not implemented by ", ref $_[0] || $_[0];
}

=pod

=head2 dst_schema()

Returns the Dlf schema which is producted by this analyser.

E.g. C<www-user_session>

=cut

sub dst_schema {
    croak "dst_schema() not implemented by ", ref $_[0] || $_[0];
}

sub type { return 'dlf_analyser' }

=head1 IMPLEMENTATION METHOD

The analyser does its job using the analyse() method. 

=head2 analyse( $process, $config )

The $process parameter contains the Lire::DlfAnalyserProcess object
which is controlling the conversion process.

The $config parameter contains configuration data that was specified
in the AnalysisJob for that converter. To register configuration
specification for you DlfAnalyser, you just need to define a
configuration specification under the name
I<analyser_name>_properties. This should be either a RecordSpec or
ObjectSpec.

=cut

sub analyse {
    croak "analyse() not implemented by ", ref $_[0] || $_[0];
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::DlfAnalyserProcess(3pm), Lire::DlfStore(3pm),
Lire::PluginManager(3pm), Lire::DlfCategoriser(3pm).

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: DlfAnalyser.pm,v 1.3 2004/05/17 17:05:19 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

