package Lire::OutputFormats::HTML;

use strict;

use base qw/Lire::OutputFormat/;

use Locale::TextDomain 'lire';

use Lire::ReportParser::HTMLWriter;
use Lire::ReportParser::ReportBuilder;
use Lire::Utils qw/tempdir file_content/;

eval "use MIME::Entity";

=pod

=head1 NAME

Lire::OutputFormats::HTML - HTML output format.

=head1 SYNOPSIS

  my $format = Lire::PluginManager->get_plugin( 'output_format', 'html' );
  $format->format_report( $xml_file, $htmldir, $cfg );

=head1 DESCRIPTION

Format a XML report into HTML or XHTML with charts.

=cut

sub name {
    return 'html';
}

sub title {
    return __( '(X)HTML' );
}

sub description {
    return '<para>' . __( 'HTML Report with charts, if ploticus is available.' ) . '</para>';
}

sub format_report {
    my ( $self, $report_file, $output_file, $cfg ) = @_;

    my $parser = new Lire::ReportParser::ReportBuilder();
    my $report = $parser->parsefile( $report_file );
    my $writer =
      new Lire::ReportParser::HTMLWriter();
    $writer->write_report( $report, $output_file, %$cfg );

    return;
}

my %types = ( 'png' => 'image/png',
              'jpg' => 'image/jpeg',
              'jpeg' => 'image/jpeg',
              'gif' => 'image/gif',
              'css' => 'text/css',
            );

sub mime_report {
    my ( $self, $report_file, $cfg ) = @_;


    my $tmpdir = tempdir( __PACKAGE__ . '_XXXXXX', 'CLEANUP' => 1 );
    my %args = $cfg ? %$cfg : ();
    $args{'one_page'} = 1;
    $self->format_report( $report_file, $tmpdir, \%args );

    my $entity = MIME::Entity->build( 'Type' => 'multipart/related' );
    $entity->attach( 'Type' => 'text/html',
                     'Charset' => 'utf-8',
                     'Encoding' => 'quoted-printable',
                     'Data' => setcid( "$tmpdir/index.html" ) );
    opendir my $dh, $tmpdir
      or die "opendir '$tmpdir' failed: $!";
    foreach my $file ( sort readdir $dh ) {
        next if $file =~ /^(\.|\.\.|index\.html)$/;
        my ($ext) = $file =~ /\.(\w+$)/;

        my $type = $types{lc $ext} || 'application/octet-stream';
        $entity->attach( Path  => "$tmpdir/$file",
                         Id    => "<" . $file . ">",
                         Type  => $type,
                         Encoding => $type =~ /^text/ ? 'quoted-printable' : 'base64' ,
                       );
    }
    closedir $dh;

    return $entity;
}

sub relurl2cid {
    my ( $attr, $url ) = @_;

    if ( $url =~ /^([-.\w]+)(#.*)?$/ ) {
	$url = "cid:$1";
	$url .= $2 if defined $2;
    }
    return qq{$attr="$url"};
}

sub setcid {
    my $file = $_[0];

    my $content = file_content( $file );
    $content =~ s/(href|src)=['"](.*?)['"]/relurl2cid( $1, $2 )/eg; #
    return $content;
}

1;

__END__

=pod

=head1 SEE ALSO

 Lire::PluginManager(3pm) Lire::OutputFormat(3pm)

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: HTML.pm,v 1.1 2004/08/28 03:41:08 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

