package tests::RangegroupTest;

use strict;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::Rangegroup;
use Lire::ReportSpec;
use Lire::Report::Group;
use Lire::Report::TableInfo;
use Lire::DlfQuery;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_test_schema();

    $self->{'spec'} = new Lire::ReportSpec();
    $self->{'spec'}->superservice( 'test' );
    $self->{'spec'}->id( 'rangegroup-test' );

    $self->{'cfg'}{'lr_scale_numbers'} = 1;
    $self->{'cfg'}{'lr_scale_bytes'} = 1;
    $self->{'cfg'}{'lr_scale_seconds'} = 1;

    $self->{'rangegroup5'} =
      new Lire::Rangegroup( 'report_spec' => $self->{'spec'},
                            'range-size' => '10k',
                            'field' =>  'file_size',
                            'label' => 'Range',
                          );

    $self->{'rangegroup5_min'} =
      new Lire::Rangegroup( 'report_spec' => $self->{'spec'},
                            'range-size' => '10k',
                            'range-start' => '1k',
                            'min-value' => 0,
                            'max-value' => '10k',
                            'field' =>  'file_size',
                            'label' => 'Range',
                          );

    $self->{'rangegroup5_geo'} =
      new Lire::Rangegroup( 'report_spec' => $self->{'spec'},
                            'size-scale' => 5,
                            'range-size' => '1k',
                            'field' =>  'file_size',
                            'label' => 'Range', );

    return;
}

sub tear_down {
    my $self = $_[0];

    $self->SUPER::tear_down( @_ );

    return;
}

sub test_name {
    my $self = $_[0];

    $self->assert_equals( 'rangegroup:file_size',
                          $self->{'rangegroup5'}->name() );
}

sub test_create_categorical_info {
    my $self = $_[0];

    my $info = new Lire::Report::TableInfo();
    $self->{'rangegroup5'}->create_categorical_info( $info );

    my @cols = $info->children();
    $self->assert_equals( 1, scalar @cols );

    my $col = $cols[0];
    $self->assert_equals( 'categorical', $col->class() );
    $self->assert_equals( 'rangegroup:file_size', $col->name() );
    $self->assert_equals( 'bytes', $col->type() );
    $self->assert_equals( 'Range', $col->label() );
}

sub test_build_query {
    my $self = $_[0];

    foreach my $t ( [ '5', 'lr_rangegroup(file_size,0,10240,NULL,NULL)' ],
                    [ '5_min', 'lr_rangegroup(file_size,1024,10240,0,10240)' ],
                    [ '5_geo', 'lr_rangegroup_geo(file_size,0,1024,5,NULL,NULL)' ] )
    {
        my $rangegroup = $self->{ 'rangegroup' . $t->[0] };
        $rangegroup->{'field'} = 'file_size';
        my $e_query = new Lire::DlfQuery( 'test' );
        $e_query->add_aggr_field( '_lr_nrecords', 'count(*)' );
        $e_query->add_group_field( 'rangegroup:file_size', $t->[1] );
        $e_query->set_sort_spec( 'rangegroup:file_size' );

        my $query = new Lire::DlfQuery( 'test' );
        $rangegroup->build_query( $query );
        $self->assert_deep_equals( $e_query, $query );
    }
}

sub test_param_value {
    my $self = $_[0];

    my $op =  $self->{'rangegroup5'};

    $op->{'field'} = 'file_size';
    $op->{'range-start'} = '1k';
    $self->assert_equals( 1024, $op->_param_value( 'range-start' ) );

    $op->{'field'} = 'time_taken';
    $op->{'range-min'} = '10s';
    $self->assert_equals( 10, $op->_param_value( 'range-min' ) );

}

sub test_create_entry {
    my $self = $_[0];

    my $info = new Lire::Report::TableInfo();
    $self->{'rangegroup5'}->create_categorical_info( $info );

    foreach my $t ( [ '5', '[0k-10k>', 0, 10240 ],
                    [ '5_min', '[1k-11k>', 1024, 10240 ],
                    [ '5_geo', '[0k-1k>>', 0, 1024  ],
                    [ '5_geo', '[1k-6k>', 1024, 5*1024  ], )
    {
        my ( $range, $fmt_value, $value, $length ) = @$t;

        my $rangegroup = $self->{ 'rangegroup' . $range};
        my $dlf = { 'rangegroup:file_size' => $value };
        my $group = new Lire::Report::Group( bless( {}, 'Lire::Report::Entry'),
                                             $info );
        my $entry = $rangegroup->create_entry( $group, $dlf );

        $self->assert_not_null( $entry, "create_entry() returned undef" );
        my @data = $entry->data();
        $self->assert_equals( 1, scalar @data );
        $self->assert_equals( $fmt_value, $data[0]->{'content'} );
        $self->assert_equals( $value, $data[0]->{'value'} );
        $self->assert_equals( $length, $data[0]->{'range'} );
    }
}

sub test_create_entry_mc {
    my $self = $_[0];

    my $rangegroup = $self->{'rangegroup5'};
    my $info = new Lire::Report::TableInfo();
    $rangegroup->create_categorical_info( $info );
    my $group = new Lire::Report::Group( bless( {}, 'Lire::Report::Entry'),
                                         $info );

    my $entry = $rangegroup->create_entry( $group,
                                          { 'rangegroup:file_size' => undef,
                                            '_lr_nrecords' => 24 } );
    $self->assert_null( $entry, "create_entry() should have returned undef" );
    $self->assert_equals( 24, $group->missing_cases() );
}

1;
