package Lire::Field;

use strict;

use vars qw( $VERSION );

use Lire::DataTypes qw( :xml :special );

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.5 $' =~ m!Revision: ([.\d]+)!;
};

=pod

=head1 NAME

Lire::Field - Interface to Dlf field's specification.

=head1 SYNOPSIS

    my $field = $schema->field( 'time' );
    print "Name: ", $field->name, "\n", "Type: ", $field->type, "\n";

=head1 DESCRIPTION

This module represents Dlf field's specifications. You'll get
reference to this kind of object when you call the field(),
timestamp_field() or fields() methods on a Lire::DlfSchema(3pm)
object. You can obtain information about this field by querying its
methods.

=head1 OBJECT METHODS

=cut

sub new {
    my $proto = shift;
    my $class = ref( $proto) || $proto;


    my %params = @_;
    my $name	= $params{name};
    croak "invalid name for a field: $name"
      unless check_xml_name( $name );

    my $type	= $params{type};
    croak "invalid type for a field: $type"
      unless check_type( $type );

    my $pos	= $params{pos};
    croak "invalid position for a field: $pos"
      unless $pos =~ /^\d+$/;

    bless { name	=> $name,
	    type	=> $type,
	    pos		=> $pos,
	    description	=> $params{description},
	    default	=> $params{default},
	  }, $class;
}

=pod

=head2 name()

Returns the name of the field.

=cut

sub name {
    return $_[0]->{name};
}

=pod

=head2 pos()

Returns the field's position in the record. Field's positions are
indexed from 0.

=cut

sub pos {
    return $_[0]->{pos};
}

=pod

=head2 type()

Returns the field's type. This will be a string describing the type's
name as used in the XML specifications.

=cut

sub type {
    return $_[0]->{type};
}

=pod

=head2 default()

Returns the field's default value. If no default value was specificied
in the XML specification, this will 0 for numeric types and '-' for
other types.

=cut

sub default {
    return $_[0]->{default} if defined $_[0]->{default};
    is_numeric_type( $_[0]->{type} ) ? 0 : "-";
}

=pod

=head2 description()

This method will return the field's description. (This is the content
of the description element in the XML specification.) Be aware that
this will most likely contains DocBook markup.

=cut

sub description {
    $_[0]->{description};
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::DlfSchema(3pm) Lire::DataTypes(3pm)

=head1 VERSION

$Id: Field.pm,v 1.5 2002/02/12 21:47:28 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
