package Lire::ReportParser;

use strict;

use vars qw( $VERSION @ISA $SELF );

use XML::Parser;
use Carp;
use Lire::Program qw( :msg ); 

BEGIN {
    ($VERSION)	= '$Revision: 1.11 $' =~ m!Revision: ([.\d]+)!;
}

sub new {
    my $proto = shift;
    my $class = ref( $proto) || $proto;

    my $self = bless { skipping => 0, }, $class;
}

sub parse {
    my ( $self, $file ) = @_;

    my $parser = new XML::Parser ( Handlers	=> {
						    Init  => \&Init,
						    Final => \&Final,
						    Start => \&Start,
						    End   => \&End,
						    Char  => \&Char,
						   },
				   Namespaces => 1,
				   NoLWP      => 1,
				 );

    $SELF = $self;
    my $value = $parser->parse( $file );
    $SELF = undef;

    return $value;
}

use vars qw( $LRML_NS %LRML_ELEMENTS %LRML_PCDATA_ELEMENTS
	     %DBK_ELEMENTS %DBK_PCDATA_ELEMENTS );

BEGIN {
    $LRML_NS = "http://www.logreport.org/LRML/";

    my @elmnts = 

    %LRML_ELEMENTS = map { $_ => 1 } qw( report section subreport 
					 title description hostname timespan
					 table entry name value group );

    %LRML_PCDATA_ELEMENTS = map { $_ => 1 } qw( title hostname timespan
						name value );

}

sub known_dbk_elements {
    return [];
}

sub known_dbk_pcdata_elements {
    return [];
}

sub skip {
    my ( $self, $expat ) = @_;

    $self->{skipping}   = 1;
    $self->{skip_level} = $expat->depth;
}


sub Init {
    my ($expat) = @_;

    # Initialize DocBook processing
    %DBK_ELEMENTS = map { $_ => 1 } @{$SELF->known_dbk_elements() };
    %DBK_PCDATA_ELEMENTS = map { $_ => 1 } @{$SELF->known_dbk_pcdata_elements() };
    $SELF->parse_start( $expat );
}

sub Final {
    my ($expat) = @_;
    $SELF->parse_end( $expat );
}

sub error {
    my ( $expat, $msg ) = @_;

    # Remove other at line message
    # $msg =~ s/( at.*?line \d+\n*)//gm;

    my $line = $expat->current_line;

    die $msg, " at line ", $line, "\n";
}

sub warning {
    my ( $expat, $msg ) = @_;

    my $line = $expat->current_line;

    lr_warn($msg . " at line " . $line );
}

sub Start {
    my ( $expat, $name ) = @_;

    return if $SELF->{skipping};
    my $ns = $expat->namespace($name);
    $ns ||= ""; # Remove warning
    if ( $ns eq $LRML_NS ) {
	# This is one of our element
	error( $expat, "unknown element: $name" )
	  unless exists $LRML_ELEMENTS{$name};

	{
	    no strict 'refs';

	    my $sub = $name . "_start";
	    $sub =~ s/-/_/g;	# Hyphen aren't allowed in element name

	    eval {
		return if $SELF->element_start( @_ );
		$SELF->$sub( @_ );
	    };
	    error( $expat, $@ ) if $@;
	};
    } else {
	# If we are in lire:description, this is probably a
	# DocBook element
	my $lire_desc = $expat->generate_ns_name( "description", $LRML_NS );
	if ( $expat->within_element( $lire_desc ) ) {
	    return if eval { $SELF->element_start( @_ ) };
	    error( $expat, $@ ) if $@;

	    if ( exists $DBK_ELEMENTS{$name} ) {
		no strict 'refs';

		my $sub = "dbk_" . $name . "_start";
		$sub =~ s/-/_/g;	# Hyphen aren't allowed in element name

		eval {
		    $SELF->$sub( @_ );
		};
		error( $expat, $@ ) if $@;
	    } else {
		# Output warning only if there are known docbook elements
		warning( $expat, "unknown DobBook element: $name" )
		  if (%DBK_ELEMENTS);
	    }
	} else {
	    error( $expat, "unknown element: $name" );
	}
    }
}

sub End {
    my ( $expat, $name ) = @_;

    if ( $SELF->{skipping} ) {
	$SELF->{skipping} = 0
	  if $expat->depth == $SELF->{skip_level};
	return;
    }

    my $ns = $expat->namespace($name);
    $ns ||= ""; # Remove warning
    if ( $ns eq $LRML_NS ) {
	# This is one of our element
	error( $expat, "unknown element: $name" )
	  unless exists $LRML_ELEMENTS{$name};

	{
	    no strict 'refs';

	    my $sub = $name . "_end";
	    $sub =~ s/-/_/g;	# Hyphen aren't allowed in element name

	    eval {
		return if $SELF->element_end( @_ );
		$SELF->$sub( @_ );
	    };
	    error( $expat, $@ ) if $@;
	}
    } else {
	# If we are in lire:description, this is probably a
	# DocBook element
	my $lire_desc = $expat->generate_ns_name( "description", $LRML_NS );
	if ( $expat->within_element( $lire_desc ) ) {
	    return if eval { $SELF->element_end( @_ ) };
	    error( $expat, $@ ) if $@;
	    if ( exists $DBK_ELEMENTS{$name} ) {
		no strict 'refs';

		my $sub = "dbk_" . $name . "_end";
		$sub =~ s/-/_/g;	# Hyphen aren't allowed in element name

		eval {
		    $SELF->$sub( @_ );
		};
		error( $expat, $@ ) if $@;
	    } else {
		# Output warning only if there are known docbook elements
		warning( $expat, "unknown DobBook element: $name" )
		  if (%DBK_ELEMENTS);
	    }
	} else {
	    error( $expat, "unknown element: $name" );
	}
    }
}

sub Char {
    my ( $expat, $str ) = @_;

    return if $SELF->{skipping};
    # Character should only appear in title and description
    my $name = $expat->current_element;
    my $ns   = $expat->namespace($name);
    $ns ||= ""; # Remove warning
    if ( $ns eq $LRML_NS ) {
	# This is one of our element
	error( $expat, "character in unknown element: $name" )
	  unless exists $LRML_ELEMENTS{$name};

	if ( $LRML_PCDATA_ELEMENTS{$name} ) {
	    no strict 'refs';

	    my $sub = $name . "_char";
	    $sub =~ s/-/_/g;	# Hyphen aren't allowed in element name

	    eval {
		return if $SELF->pcdata( @_ );
		$SELF->$sub( @_ );
	    };
	    error( $expat, $@ ) if $@;
	} else {
	    eval { $SELF->ignorable_ws( @_ ) };
	    error( $expat, $@ ) if $@;
	}
    } else {
	# If we are in lire:description, this is probably a
	# DocBook element
	my $lire_desc = $expat->generate_ns_name( "description", $LRML_NS );
	if ( $expat->within_element( $lire_desc ) ) {
	    if ( ! %DBK_PCDATA_ELEMENTS ) {
		# Nothing is known about DocBook
		eval { $SELF->pcdata( @_ ) };
		error( $expat, $@ ) if $@;
		return;
	    }

	    if ( exists $DBK_PCDATA_ELEMENTS{$name} ) {
		no strict 'refs';

		my $sub = "dbk_" . $name . "_char";
		$sub =~ s/-/_/g;	# Hyphen aren't allowed in element name

		eval {
		    $SELF->$sub( @_ );
		};
		error( $expat, $@ ) if $@;
	    } else {
		eval { $SELF->ignorable_ws( @_ ) };
		error( $expat, $@ ) if $@;
	    }
	}
    }
}

sub parse_start {}

sub parse_end {}

sub element_start {
    my ( $self, $expat, $name, %attr ) = @_;
    return 0;
}

sub element_end {
    my ( $self, $expat, $name ) = @_;
    return 0;
}

sub pcdata {
    my ( $self, $expat, $text ) = @_;
    return 0;
}

sub ignorable_ws {
    my ( $self, $expat, $text ) = @_;
    return 0;
}

sub report_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->{subreport_count} = 0;
}

sub report_end {}

sub title_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->{curr_title} = "";
}

sub title_char {
    my ( $self, $expat, $char ) = @_;

    $self->{curr_title} .= $char;
}

sub title_end {
    my ( $self, $expat, $name ) = @_;

    $self->{curr_title} =~ s/\s+/ /g;
    $self->handle_title( $expat, $self->{curr_title} );
}

sub handle_title {
    my ( $self, $expat, $title ) = @_;
}

sub description_start {}
sub description_end {}

sub hostname_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->{curr_hostname} = "";
}

sub hostname_char {
    my ( $self, $expat, $char ) = @_;

    $self->{curr_hostname} .= $char;
}

sub hostname_end {
    my ( $self, $expat, $name ) = @_;

    $self->{curr_hostname} =~ s/\s+/ /g;
    $self->handle_hostname( $expat, $self->{curr_title} );
}


sub handle_hostname {
    my ( $self, $expat, $hostname ) = @_;
}

sub timespan_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->{curr_timespan} = "";
}

sub timespan_char {
    my ( $self, $expat, $char ) = @_;

    $self->{curr_timespan} .= $char;
}

sub timespan_end {
    my ( $self, $expat, $name ) = @_;

    $self->handle_timespan( $expat, $self->{curr_timespan} );
}

sub handle_timespan {}

sub current_subreport_count {
    return $_[0]{subreport_count};
}

sub current_group_level {
    return $_[0]{group_level};
}

sub current_type {
    return $_[0]{subreport_type};
}

sub current_superservice {
    return $_[0]{subreport_superservice};
}

sub current_charttype {
    return $_[0]{subreport_charttype};
}

sub current_section_subreport_count {
    return $_[0]{section_subreport_count};
}

sub in_section_intro {
    return $_[0]{in_section_intro};
}

sub current_table_entry_count {
    return $_[0]{curr_table_entry_count};
}

sub section_start {
    my ( $self, $expat, $name, %attr ) = @_;
    $self->{in_section_intro}		= 1;
    $self->{section_subreport_count}   = 0;
}

sub section_end {
    my ( $self, $expat, $name ) = @_;
    $self->{in_section_intro}		= 0;
    $self->{section_subreport_count}	= 0;
}

sub subreport_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->{in_section_intro}	    = 0;
    $self->{section_subreport_count}++;

    $self->{group_level}	    = 0;
    $self->{subreport_type}	    = $attr{type};
    $self->{subreport_superservice} = $attr{superservice};
    $self->{subreport_charttype}    = undef;
}

sub subreport_end {
    my ( $self, $expat, $name ) = @_;

    $self->{group_level}	    = undef;
    $self->{subreport_type}	    = undef;
    $self->{subreport_superservice} = undef;
    $self->{subreport_charttype}    = undef;
    $self->{subreport_count}++;
}

sub table_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->{subreport_charttype}	= $attr{charttype};
    $self->{curr_table_entry_count}	= 0;
}

sub table_end {
    my ( $self, $expat, $name ) = @_;

    $self->{curr_table_entry_count}	= undef;
}

sub entry_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->{curr_table_entry_count}++;
}

sub entry_end {}

sub group_start {
    my ($self, $expat, $name, %attr ) = @_;

    $self->{group_level}++;
}

sub group_end {
    my ($self, $expat, $name ) = @_;
    $self->{group_level}--;
}

sub name_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->{curr_name} = "";
}

sub name_char {
    my ( $self, $expat, $char ) = @_;

    $self->{curr_name} .= $char;
}

sub name_end{
    my ( $self, $expat, $name ) = @_;

    $self->handle_name( $expat, $self->{curr_name} );
}

sub handle_name {}

sub value_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->{curr_value} = "";
    $self->{curr_num_value} = $attr{value};
}

sub value_char {
    my ( $self, $expat, $char ) = @_;

    $self->{curr_value} .= $char;
}

sub value_end{
    my ( $self, $expat, $name ) = @_;

    $self->{curr_num_value} = $self->{curr_value}
      unless defined $self->{curr_num_value};
    $self->handle_value( $expat, $self->{curr_value},
			 $self->{curr_num_value} );
}

sub handle_value {}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::ReportParser -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: ReportParser.pm,v 1.11 2002/01/20 21:59:02 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
