package Lire::ReportParser::ChartWriter;

use strict;

use vars qw( $VERSION @ISA );

use Lire::Config;
use Lire::ReportParser;
use Lire::Program qw( :msg );

use GD::Graph::pie;
use GD::Graph::bars;
use GD::Graph::lines;

use File::Basename;
use Carp;

use constant NUMBER_OF_Y_TICKS => 5;
use constant GRAPH_HEIGHT => 300;
use constant GRAPH_WIDTH  => 400;
use constant MAX_LABEL_LENGTH => 20;

BEGIN {
    ($VERSION)	= '$Revision: 1.5 $' =~ m!Revision: ([.\d]+)!;
    @ISA = qw( Lire::ReportParser );
}

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;
    my $self  = $class->SUPER::new( @_ );

    my %args  = @_;

    $args{format}    ||= "png";
    $args{outputdir} ||= ".";

    die "unsupported file format: $args{format} (png, eps, gif or jpg)\n"
      unless $args{format} =~ /^(PNG|EPS|JPG|GIF)$/i;

    die "$args{outputdir} is not a directory\n"
      unless -d $args{outputdir};

    die "can't write in $args{outputdir}\n"
      unless -w $args{outputdir};

    $self->{format} = lc $args{format};
    $self->{outputdir} = $args{outputdir};
    $self;
}

sub parse_start {
    my ($self, $expat) = @_;
    # Register handler for XMLDcl and DocType event, so that 
    # we output them as is

    my $print_it = sub {
	print $_[0]->original_string(), "\n";
    };

    $expat->setHandlers( XMLDecl    => $print_it,
			 Comment    => $print_it,
		       );

    1;
}

# Print an identical copy of the XML report.
# We will add image and file element when we create graphics.
sub element_start {
    my ( $self, $expat, $name, %attr ) = @_;

    print $expat->original_string;

    return 0; # Continue normal processing
}

sub element_end {
    my ( $self, $expat, $name ) = @_;

    print $expat->original_string;

    return 0; # Continue normal processing
}

sub pcdata {
    my ( $self, $expat, $str ) = @_;

    print $expat->original_string;
    return 0; # Continue normal processing
}

sub ignorable_ws {
    my ( $self, $expat, $str ) = @_;

    print $expat->original_string;
    return 0; # Continue normal processing
}

sub table_start {
    my ( $self, $expat, $name, %attr ) = @_;

    $self->SUPER::table_start( $expat, $name, %attr );

    $self->{write_chart} = defined $self->current_charttype;

    if ($self->{write_chart}) {
	$self->{x_data} = [];
	$self->{y_data} = [];
    }
}

sub table_end {
    my ( $self, $expat, $name ) = @_;

    $self->SUPER::table_end( $expat, $name );

    return unless $self->{write_chart};
    
    unless ( @{$self->{x_data}} && @{$self->{y_data}} ) {
	lr_info( "skipping chart for ", $self->current_type , " because table is empty" );
	return;
    }

    my $graph;
    my %common_opts = ( x_labels_vertical => 1,
			transparent => 1,
		      );
  SWITCH:
    for ( $self->current_charttype ) {
	/^bars$/ && do {
	    $graph = new GD::Graph::bars( GRAPH_WIDTH, GRAPH_HEIGHT );
	    $graph->set( bar_spacing  => 4,
			 shadow_depth => 2,
			 %common_opts );
	    last SWITCH;
	};
	/^lines$/ && do {
	    $graph = new GD::Graph::lines( GRAPH_WIDTH, GRAPH_HEIGHT );
	    $graph->set( %common_opts );
	    last SWITCH;
	};
	/^pie$/ && do {
	    $graph = new GD::Graph::pie( GRAPH_WIDTH, GRAPH_HEIGHT );
	    $graph->set( start_angle => "240",
			 %common_opts );
	    last SWITCH;
	};
	/^histogram$/ && do {
	    $graph = new GD::Graph::bars( GRAPH_WIDTH, GRAPH_HEIGHT );
	    $graph->set( bar_spacing => 0,
			 %common_opts );
	    last SWITCH;
	};
	croak "Unknown chart type: $_\n";
    }

    if ( $self->current_charttype =~ /^bars|lines|histogram$/) {
	my $maxy = 0;
	foreach my $val ( @{$self->{y_data}} ) {
	    $maxy = $val if $val > $maxy;
	}

	$maxy = int( $maxy * 1.1);
	$maxy = $maxy + ( NUMBER_OF_Y_TICKS - ($maxy % NUMBER_OF_Y_TICKS));
	$graph->set( y_max_value    => $maxy,
		     y_tick_number  => NUMBER_OF_Y_TICKS );
    }

    my $img = $graph->plot( [ $self->{x_data}, $self->{y_data} ] );
    unless ( $img ) {
	warn "failed to generate chart ", $self->current_type, " (", 
	  $self->current_charttype, "): ", $graph->error, "\n";
	delete $self->{x_data};
	delete $self->{y_data};
	return;
    }
    my $basename = $self->current_subreport_count() . "-" . $self->current_type();
    $basename =~ s/\./_/g;

    my $png_file = $self->{outputdir} . "/" . $basename . ".png";
    open ( GRAPH, "> $png_file" )
      or die "can't create $png_file: $!\n";
    binmode GRAPH;
    print GRAPH $img->png()
      or die "can't convert graph to PNG: ", $graph->error, "\n";
    close GRAPH;

    my $target_file;
    if ( $self->{format} ne "png" ) {
	$target_file = $self->{outputdir} . "/" . $basename . $self->{format};
	system ( $Lire::Config::Convert, $png_file, $target_file );
	die "error converting PNG to $self->{format}\n";
	unlink $png_file;
    } else {
	$target_file = $png_file;
    }


    my $format = uc $self->{format}; # DBX Notation are in uppercase
    my $file = basename ( $target_file );
    print qq{\n  <lire:image>\n};
    print qq{   <lire:file format="$format">$file</lire:file>\n};
    print qq{  </lire:image>};

    lr_info( "writing chart $target_file" );

    delete $self->{x_data};
    delete $self->{y_data};
}

sub handle_name {
    my ( $self, $expat, $name ) = @_;

    return unless $self->{write_chart};

    # Crop $name if necessary
    $name = substr( $name, 0, MAX_LABEL_LENGTH ) . "..."
      if length $name > MAX_LABEL_LENGTH;

    push @{$self->{x_data}}, $name;
}

sub handle_value {
    my ( $self, $expat, $value, $num_value ) = @_;

    return unless $self->{write_chart};

    push @{$self->{y_data}}, $num_value;
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::ChartWriter -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: ChartWriter.pm,v 1.5 2001/12/13 21:08:17 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
