package Lire::ReportSection;

use strict;

use vars qw( $VERSION );

use Lire::DataTypes qw/:special /;

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.2 $' =~ m!Revision: ([.\d]+)!;
}

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;

    my ( $super, $title ) = @_;

    croak "invalid superservice: $super"
      unless check_superservice( $super );
    
    bless { superservice    => $super,
	    filters	    => [],
	    title	    => $title || "Section",
	    reports_by_key  => {},
	    reports	    => [],
	  }, $class;
}

sub superservice {
    my ( $self ) = @_;

    $self->{superservice};
}

sub title {
    my ( $self, $title ) = @_;

    if ( defined $title ) {
	$self->{title} = $title;
    }

    $self->{title};
}

sub expanded_display_description {
    my ( $self ) = @_;
    
    return "" unless $self->filters;

    my $desc = "<para>";
    if ( $self->filters == 1 ) {
	my ($f) = $self->filters;
	$desc .= "Applied filter in this section: " . 
	  $f->expanded_display_title;
    } else {
	$desc .= "Filters applied in this section:\n" .
	  q{{ <itemizedlist spacing="compact">\n}};

	foreach my $filter ( $self->filters ) {
	    $desc .= " <listitem>\n  <para>" .
	      $filter->expanded_display_title . "</para>\n </listitem>\n";
	}
	$desc .= " </itemizedlist>\n";
    }
    
    $desc .= "</para>\n";

    return $desc;
}

sub filters {
    my ( $self ) = @_;

    return @{ $self->{filters} };
}

sub add_filter {
    my ( $self, $filter_spec ) = @_;

    croak ( "filter_spec must be of type Lire::FilterSpec (not $filter_spec)" )
      unless UNIVERSAL::isa( $filter_spec, 'Lire::FilterSpec' );

    # Make sure all the report's schema are compatible
    # with with this filter
    my $schema = $filter_spec->schema->id;
    foreach my $r ( $self->reports ) {
	die "filter ", $filter_spec->id, "'s schema is incompatible ",
	  "with report ", $r->id, "\n"
	    unless $r->schema->is_schema_compatible( $schema );
    }
    push @{$self->{filters}},$filter_spec;
}

sub reports {
    my ( $self ) = @_;

    return @{ $self->{reports} };
}

sub add_report {
    my ( $self, $report ) = @_;

    croak "report must of type Lire::ReportSpec (not $report)"
      unless UNIVERSAL::isa( $report, "Lire::ReportSpec" );

    croak ( "report is of wrong superservice: ", $report->superservice ,
	    " != ", $self->superservice )
      if $self->superservice ne $report->superservice;

    # Make sure the report's schema is compatible
    # with all the filters' schema
    my $schema = $report->schema;
    foreach my $f ( $self->filters ) {
	die "report ", $report->id, "'s schema is incompatible with filter ",
	  $f->id, "\n"
	    unless $schema->is_schema_compatible( $f->schema->id );
    }
    push @{$self->{reports}},$report;
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::ReportSection -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: ReportSection.pm,v 1.2 2002/01/19 20:15:54 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
