/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */

#include "include/support.h"
#include "include/pixmap.h"
#include "include/environment.h"

/*! \brief Konstruktor

	Tworzy ikonę
*/	
LWPixmap::LWPixmap (): name(NULL), pixbuf(NULL)
{
}

/*! \brief Destruktor

	Niszczy ikonę
*/	
LWPixmap::~LWPixmap ()
{
	if (pixbuf != NULL) 
		g_object_unref (G_OBJECT(pixbuf));
	
	if (name != NULL)
		g_free (name);
}

/*! \brief Konstruktor kopiujący

	Kopiuje ikonę
	\param pixmap Kopiowana ikona
*/	
LWPixmap::LWPixmap (const LWPixmap *pixmap)
{
	pixbuf = ((LWPixmap *) pixmap)->getPixbuf();
	name = g_strdup (((LWPixmap *) pixmap)->getName());
}

/*! \brief Odtwarza ikonę z węzła XML

	Odtwarza ikonę na podstawie opisu zawartego w węźle drzewa XML
	\param node Węzeł na podstawie zostanie dokonana rekonstrukcja
*/
void LWPixmap::restoreFromXML (xmlNode *node)
{
GError *error=NULL;

	g_return_if_fail (!xmlStrcasecmp (node->name, (xmlChar *) "Pixmap"));
	
	/* Pobieranie atrybutu name, wynik jest zaalakowany w pamieci */
	name = (gchar *) xmlGetProp (node, (xmlChar *) "name");
	
	/* Pobieranie atrybutu src, wynik jest zaalakowany w pamieci */
	xmlChar *srcutf = xmlGetProp (node, (xmlChar *) "src");
	
	/* Dekodowanie nazwy pliku do kodowania uzywanego w systemie plików */
	gchar *src = g_filename_from_utf8 ((gchar *) srcutf, -1, NULL, NULL, NULL);
	
	/* Zwolnianie pamięci zaalakowanej podczas pobierania atrybutu */
	xmlFree (srcutf);

	/* Zbudowanie pełnej nazwy ze ścieżką do pliku */
	gchar *filename = LWEnvironment::buildPixmapFilename (src);

	/* Zwonienie pamięci zaalakowanej podczas konwersji kodowania */
	g_free (src);
	
	/* Odczytanie pliku obrazka z dysku */
	pixbuf = gdk_pixbuf_new_from_file (filename, &error);

	/* Upewnij się, że obrazek jest w odpowiedniej rozdzielczości */
	if ((ICON_XSIZE != gdk_pixbuf_get_width(pixbuf)) || 
		 (ICON_YSIZE != gdk_pixbuf_get_height(pixbuf))) {
			GdkPixbuf *tmp = gdk_pixbuf_scale_simple (pixbuf, ICON_XSIZE, ICON_YSIZE, GDK_INTERP_HYPER);
			g_object_unref (pixbuf);
			pixbuf = tmp;
		 }

	/* Zwonienie pamięci zaalakowanej podczas budowania ścieżki */
	g_free (filename);
}

/*! \brief Pobranie danych ikony

	Pobiera dane ikony w postaci takiej jaka jest używana wewnętrznie
	przez bibliotekę Gdk-Pixbuf
	\return Zwrócone dane ikony
*/	
GdkPixbuf *LWPixmap::getPixbuf ()
{
	g_return_val_if_fail (pixbuf != NULL, NULL);
	
	return pixbuf;
}

/*! \brief Pobranie nazwy ikony

	Pobiera związaną z ikoną nazwę
	\return Zwrócona nazwa
*/	
gchar *LWPixmap::getName ()
{
	g_return_val_if_fail (name != NULL, NULL);

	return name;
}
