/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */

#include "include/support.h"
#include "include/parser.h"
#include "include/interpreter.h"
#include "include/board.h"
#include "include/environment.h"

static gboolean delete_cb (LWProgram *program)
{
	if (TRUE == program->isFinished())
		delete program;
	else
		program->stop ();
	
	return TRUE;
}

/*! \brief Konstruktor

	Tworzy program
*/
LWProgram::LWProgram () 
: world(NULL), interp(NULL), program_data(NULL)
{
	LWEnvironment::registerProgram (this);
}

/*! \brief Destruktor
	Destroying a program
*/	
LWProgram::~LWProgram ()
{
GtkWidget *window=NULL;
	
	if (world != NULL)
		window = gtk_widget_get_ancestor(world->getWidget(), GTK_TYPE_WINDOW);

	if (interp != NULL)
		delete interp;
	
	if (world != NULL)
		delete world;
	
	if (program_data != NULL)
		delete program_data;
	
	if (window != NULL)
		gtk_widget_destroy (window);
	
	LWEnvironment::unregisterProgram ();
}


/* \brief Wyświetla komunikat

	Wyświetla komunikat
	\msg Komunikat do wyświetlenia
*/
void LWProgram::showMessage (LWMessage *msg)
{
		GtkWidget *window = gtk_widget_get_ancestor (world->getWidget(), GTK_TYPE_WINDOW);
		gtk_window_set_modal (GTK_WINDOW(window), FALSE);	
	
		if (interp != NULL) {
			delete interp;
			interp = NULL;
			showRunTimeMessage (msg);
			delete this;
		} else {
			showParseMessage (msg);	
		}
	
		delete msg;
}

void LWProgram::showParseMessage (LWMessage *msg)
{
	msg->show();
}

void LWProgram::showRunTimeMessage (LWMessage *msg)
{
	msg->show();
}

/*! \brief Analiza programu

	Przeprowadza analizę planszy z programem,
	tworząc drzewo składniowe. W razie błędów
	metoda tworzy okno z błędem i zaznacza ikonę
	w związku z którą wystąpił

	\param program Plansza z programem
	\return Prawda jęśli analiza zakończyła się sukcesem
	w przeciwnym wypadku fałsz.
*/
gboolean LWProgram::parse (LWBoard *program, gboolean enable_debug)
{
	program->unmarkPiece ();

	LWParser parser(this);
	parser.enableDebug(enable_debug);
	program_data = parser.parse( program);
	
	return (program_data != NULL) ? TRUE : FALSE;
}

/*! \brief Przydzielenie świata

	Przydziela świat, na którym będzie operował program.
	Metoda tworzy kopię tego świata i umieszcza w lewym górnym
	rogu czarodzieja, który po nim będzie się poruszał.

	\param a_world Plansza ze światem
*/
void LWProgram::setWorld (LWBoard *a_world)
{
	g_return_if_fail (world == NULL);
	g_return_if_fail (a_world != NULL);
	g_return_if_fail (a_world->getType() == LW_TYPE_WORLD);
	g_return_if_fail (a_world->getRowNth(0) != NULL);
	g_return_if_fail (a_world->getRowNth(0)->getPieceNth(0) != NULL);
	
	world = new LWBoard (a_world, LW_TYPE_PLAYGROUND);

	GtkWidget *window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_resizable (GTK_WINDOW(window), FALSE);	
	
	if (NULL != LWEnvironment::getWidget()) {
		GtkWidget *parentwindow = gtk_widget_get_ancestor(
			LWEnvironment::getWidget(), GTK_TYPE_WINDOW);
		
		if (parentwindow != NULL)
			gtk_window_set_transient_for (GTK_WINDOW(window), GTK_WINDOW(parentwindow));
	}
	
    gtk_container_add (GTK_CONTAINER (window), world->getWidget());
	gtk_widget_show (window);
	g_signal_connect_swapped (G_OBJECT (window), "delete_event",
		G_CALLBACK (delete_cb), (gpointer) this);
}

LWBoard *LWProgram::getWorld ()
{
	return world;
}

/*! \brief Wykonanie programu

	Wykonuje program. Metoda może być wykonana dopiero po stworzeniu
	drzewa składniowego przez metodę parse.

	\param node Węzeł do wykonania 
*/
void LWProgram::execute ()
{
	g_return_if_fail (program_data != NULL);
	g_return_if_fail (interp == NULL);
		
	interp = new LWInterpreter (this, program_data);
	interp->execute();
}			

void LWProgram::finish ()
{
}

gboolean LWProgram::isFinished()
{
	if (interp == NULL)
		return TRUE;
		
	return interp->isRunning() ? FALSE : TRUE;
}

void LWProgram::stop()
{
	if (interp != NULL) {
		interp->stop();
		delete interp;
		interp = NULL;
	}
}
