"""This file defines the EnhancedNewMatchPathValueComboDetector
detector to extract values from LogAtoms and check, if the value
combination was already seen before."""

import time

from aminer.analysis.NewMatchPathValueComboDetector import NewMatchPathValueComboDetector
from aminer.util import PersistencyUtil

class EnhancedNewMatchPathValueComboDetector(NewMatchPathValueComboDetector):
  """This class creates events when a new value combination for
  a given list of match data pathes were found. It is similar
  to the NewMatchPathValueComboDetector basic detector but also
  provides support for storing meta information about each detected
  value combination, e.g.
  * the first time a tuple was detected using the LogAtom default
    timestamp.
  * the last time a tuple was seen
  * the number of times the tuple was seen
  * user data for annotation.
  Due to the additional features, this detector is slower than
  the basic detector."""

  def __init__(
      self, aminerConfig, targetPathList, anomalyEventHandlers,
      peristenceId='Default', allowMissingValuesFlag=False,
      autoIncludeFlag=False, tupleTransformationFunction=None):
    """Initialize the detector. This will also trigger reading
    or creation of persistence storage location.
    @param targetPathList the list of values to extract from each
    match to create the value combination to be checked.
    @param allowMissingValuesFlag when set to True, the detector
    will also use matches, where one of the pathes from targetPathList
    does not refer to an existing parsed data object.
    @param autoIncludeFlag when set to True, this detector will
    report a new value only the first time before including it
    in the known values set automatically.
    @param tupleTransformationFunction when not None, this function
    will be invoked on each extracted value combination list to
    transform it. It may modify the list directly or create a
    new one to return it."""
    super(EnhancedNewMatchPathValueComboDetector, self).__init__(
        aminerConfig, targetPathList, anomalyEventHandlers, peristenceId,
        allowMissingValuesFlag, autoIncludeFlag)
    self.tupleTransformationFunction = tupleTransformationFunction


  def loadPersistencyData(self):
    """Load the persistency data from storage."""
    self.knownValuesDict = {}
    persistenceData = PersistencyUtil.loadJson(self.persistenceFileName)
    if persistenceData != None:
# Dictionary and tuples were stored as list of lists. Transform
# the first lists to tuples to allow hash operation needed by set.
      for valueTuple, extraData in persistenceData:
        self.knownValuesDict[tuple(valueTuple)] = extraData


  def receiveAtom(self, logAtom):
    """Receive on parsed atom and the information about the parser
    match.
    @return True if a value combination was extracted and checked
    against the list of known combinations, no matter if the checked
    values were new or not."""
    matchDict = logAtom.parserMatch.getMatchDictionary()
    matchValueList = []
    for targetPath in self.targetPathList:
      matchElement = matchDict.get(targetPath, None)
      if matchElement is None:
        if not self.allowMissingValuesFlag:
          return False
        matchValueList.append(None)
      else:
        matchValueList.append(matchElement.matchObject)

    if self.tupleTransformationFunction != None:
      matchValueList = self.tupleTransformationFunction(matchValueList)
    matchValueTuple = tuple(matchValueList)

    currentTimestamp = logAtom.getTimestamp()
    extraData = self.knownValuesDict.get(matchValueTuple, None)
    if extraData != None:
      extraData[1] = currentTimestamp
      extraData[2] += 1
    else:
      if self.autoIncludeFlag:
        self.knownValuesDict[matchValueTuple] = [
            currentTimestamp, currentTimestamp, 1, None]
        if self.nextPersistTime is None:
          self.nextPersistTime = time.time()+600
      for listener in self.anomalyEventHandlers:
        listener.receiveEvent(
            'Analysis.%s' % self.__class__.__name__,
            'New value combination for path(es) %s: %s' % (
                ', '.join(self.targetPathList), repr(matchValueTuple)),
            [logAtom.rawData], (logAtom, matchValueTuple), self)
    return True


  def doPersist(self):
    """Immediately write persistence data to storage."""
    persistencyData = []
    for dictRecord in self.knownValuesDict.items():
      persistencyData.append(dictRecord)
    PersistencyUtil.storeJson(self.persistenceFileName, persistencyData)
    self.nextPersistTime = None


  def whitelistEvent(
      self, eventType, sortedLogLines, eventData, whitelistingData):
    """Whitelist an event generated by this source using the information
    emitted when generating the event.
    @return a message with information about whitelisting
    @throws Exception when whitelisting of this special event
    using given whitelistingData was not possible."""
    if eventType != 'Analysis.%s' % self.__class__.__name__:
      raise Exception('Event not from this source')
    if whitelistingData != None:
      raise Exception('Whitelisting data not understood by this detector')
    currentTimestamp = eventData[0].getTimestamp()
    self.knownValuesDict[eventData[1]] = [
        currentTimestamp, currentTimestamp, 1, None]
    return 'Whitelisted path(es) %s with %s in %s' % (
        ', '.join(self.targetPathList), eventData[1], sortedLogLines[0])
