/* Copyright (C) 2000/2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <stdlib.h>
#include <sys/time.h>
#include <time.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "interface.h"
#include "support.h"
#include "commands.h"
#include "callbacks.h"
#include "connection.h"
#include "global.h"
#include "chat.h"
#include "preferences.h"
#include "search.h"
#include "transfer.h"
#include "share.h"
#include "hotlist.h"
#include "browse.h"
#include "log.h"
#include "scheme.h"
#include "handler.h"
#include "server.h"
#include "resume.h"
#include "whois.h"
#include "statistic.h"
#include "string_list.h"
#include "subscription.h"
#include "userinfo.h"
#ifdef ENABLE_WHITE_BOARD
#include "whiteboard.h"
#endif

static HANDLER_ENTRY Protocol[] = {
  {CMD_SERVER_ERROR, cmd_server_error},	// 0
  {CMD_SERVER_EMAIL, cmd_login_ack},	// 3
  {CMD_SERVER_REGISTER_OK, cmd_do_nothing},	// 8
  {CMD_SERVER_REGISTER_FAIL, cmd_register_fail},	// 9
  {CMD_SERVER_BAD_NICK, cmd_bad_nick},	// 10
  {CMD_CLIENT_UNSHARE_ALL, cmd_do_nothing},	// 110
  {CMD_SERVER_SEARCH_RESULT, cmd_search_response},	// 201
  {CMD_SERVER_SEARCH_END, cmd_end_of_search},	// 202
  {CMD_SERVER_FILE_READY, cmd_download_ack},	// 204
  {CMD_CLIENT_PRIVMSG, cmd_private_message},	// 205
  {CMD_SERVER_SEND_ERROR, cmd_get_error},	// 206
  {CMD_SERVER_USER_SIGNON, cmd_user_sign_on},	// 209
  {CMD_SERVER_USER_SIGNOFF, cmd_user_sign_off},	// 210
  {CMD_SERVER_BROWSE_RESPONSE, cmd_browse_response},	// 212
  {CMD_SERVER_BROWSE_END, cmd_browse_end},	// 213
  {CMD_SERVER_STATS, cmd_server_stats},	// 214
  {CMD_SERVER_RESUME_MATCH, cmd_resume_match},	// 216
  {CMD_SERVER_RESUME_MATCH_END, cmd_do_nothing},	// 217
  {CMD_SERVER_HOTLIST_ACK, cmd_hotlist_ack},	// 301
  {CMD_HOTLIST_ERROR, cmd_hotlist_error},	// 302
  {CMD_SERVER_DISCONNECTING, cmd_disconnecting},	// 316
  {CMD_CLIENT_IGNORE_LIST, cmd_do_nothing},	// 320
  {CMD_SERVER_IGNORE_ENTRY, cmd_ignore_user},	// 321
  {CMD_CLIENT_IGNORE_USER, cmd_ignore_user},	// 322
  {CMD_CLIENT_UNIGNORE_USER, cmd_unignore_user},	// 323
  {CMD_SERVER_NOT_IGNORED, cmd_do_nothing},	// 324
  {CMD_SERVER_ALREADY_IGNORED, cmd_do_nothing},	// 325
  {CMD_CLIENT_PART, cmd_part_channel},	// 401
  {CMD_SERVER_PUBLIC, cmd_public_message},	// 403
  {CMD_SERVER_NOSUCH, cmd_error_message},	// 404
  {CMD_SERVER_JOIN_ACK, cmd_create_channel},	// 405
  {CMD_SERVER_JOIN, cmd_join_message},	// 406
  {CMD_SERVER_PART, cmd_part_message},	// 407
  {CMD_SERVER_CHANNEL_USER_LIST, cmd_user_online},	// 408
  {CMD_SERVER_CHANNEL_USER_LIST_END, cmd_do_nothing},	// 409
  {CMD_SERVER_TOPIC, cmd_channel_topic},	// 410
  {CMD_CLIENT_CHANNEL_BAN_LIST, cmd_do_nothing},	// 420
  {CMD_SERVER_CHANNEL_BAN_LIST, cmd_banlist_entry},	// 421
  {CMD_CHANNEL_ALT_TOPIC, cmd_channel_alt_topic},	// 425
  {CMD_SERVER_UPLOAD_FIREWALL, cmd_alternate_ack},	// 501
  {CMD_SERVER_USER_SPEED, cmd_linkspeed_response},	// 601
  {CMD_SERVER_WHOIS_RESPONSE, cmd_eval_whois},	// 604
  {CMD_SERVER_WHOWAS, cmd_eval_whowas},	// 605
  {CMD_SERVER_UPLOAD_REQUEST, cmd_upload_request},	// 607
  {CMD_GET_ERROR, cmd_get_error},	// 609
  {CMD_CLIENT_ALTER_PORT, cmd_set_port},	// 613
  {CMD_CLIENT_BANLIST, cmd_do_nothing},	// 615
  {CMD_SERVER_IP_BANLIST, cmd_banlist_entry},	// 616
  {CMD_SERVER_CHANNEL_LIST_END, cmd_channel_list_end},	// 617
  {CMD_SERVER_CHANNEL_LIST, cmd_channel_list},	// 618
  {CMD_SERVER_LIMIT, cmd_remote_queued},	// 620
  {CMD_SERVER_MOTD, cmd_motd},	// 621
  {CMD_SERVER_DATA_PORT_ERROR, cmd_data_port_error},	// 626
  {CMD_SERVER_WALLOP, cmd_operator_message},	// 627
  {CMD_SERVER_ANNOUNCE, cmd_global_message},	// 628
  {CMD_SERVER_NICK_BANLIST, cmd_nick_banlist},	// 629
  {CMD_CLIENT_BROWSE_DIRECT, cmd_browse_direct},	// 640
  {CMD_SERVER_BROWSE_DIRECT_OK, cmd_browse_direct_ok},	// 641
  {CMD_SERVER_BROWSE_DIRECT_ERR, cmd_browse_direct_err},	// 642
  {CMD_SERVER_GHOST, cmd_ghost},	// 748
  {CMD_CLIENT_PING_SERVER, cmd_sping},	// 750
  {CMD_SERVER_PING, cmd_ping},	// 751
  {CMD_SERVER_PONG, cmd_pong},	// 752
  {CMD_CLIENT_REDIRECT, cmd_redirect},	// 821
  {CMD_CLIENT_CYCLE, cmd_cycle},	// 822
  {CMD_CLIENT_EMOTE, cmd_emote},	// 824
  {CMD_SERVER_NAMES_LIST, cmd_user_online},	// 825
  {CMD_CLIENT_FULL_CHANNEL_LIST, cmd_channel_list_end},	// 827
  {CMD_SERVER_FULL_CHANNEL_INFO, cmd_channel_list_entry},	// 828
  {CMD_SERVER_NAMES_LIST_END, cmd_do_nothing},	// 830
  {CMD_CLIENT_GLOBAL_USER_LIST, cmd_global_user_end},	// 831
  {CMD_SERVER_GLOBAL_USER_LIST, cmd_global_user},	// 832
  {CMD_SERVER_LINKS, cmd_server_links},	// 10112
  {CMD_CLIENT_WHO_WAS, cmd_whowas},	// 10121
  {CMD_CLIENT_HISTOGRAM, cmd_histogram},	// 10123
  {CMD_SERVER_HISTOGRAM, cmd_histogram_end},	// 10124
  {CMD_SERVER_USAGE_STATS, cmd_usage_stats},	// 10115
  {CMD_CLIENT_VERSION_STATS, cmd_version_stats},	// 10118
  {CMD_SERVER_USER_MODE, cmd_user_mode},	// 10203
  {CMD_CLIENT_CLASS_LIST, cmd_acl_list},	// 10252
  {CMD_CLIENT_ILINE_LIST, cmd_acl_list},	// 10255
  {CMD_CLIENT_DLINE_LIST, cmd_acl_list},	// 10258
  {CMD_CLIENT_ELINE_LIST, cmd_acl_list},	// 10261
  {CMD_SERVER_BROWSE_RESULT_NEW, cmd_browse_new},	// 10302
  {CMD_CLIENT_BROWSE_NEW, cmd_browse_end}	// 10301
};

static int Protocol_Size = sizeof(Protocol) / sizeof(HANDLER_ENTRY);

char Mode[NO_MODES][20] = {
  "ERROR",
  "BAN",
  "CHANGE",
  "KILL",
  "LEVEL",
  "SERVER",
  "MUZZLE",
  "PORT",
  "WALLOP",
  "CLOAK",
  "FLOOD",
  "PING",
  "MSG",
  "WHOIS"
};

int mode2int(char *mode)
{
  int no;

  for (no = 0; no < NO_MODES; no++)
    if (!l_strcasecmp(mode, Mode[no]))
      return no;

  return -1;
}

void not_implemented(int type, char *data)
{
#ifdef PROTOCOL_DEBUG
  log("protocol", LOG_OTHER, "** Unknown\n");
#endif
  client_message("Error", "** not implemented yet: %d: [%s]", type, data);
}

int search_handler(int code)
{
  int i1;

  // for now do a linear search
  for (i1 = 0; i1 < Protocol_Size; i1++) {
    if (Protocol[i1].code == code)
      return i1;
  }
  return -1;
}

void handle_command(int type, char *data)
{
  int i1;
  char dummy[] = "";

  if (!data) data = dummy;

  i1 = search_handler(type);
  if (i1 < 0) {
    not_implemented(type, data);
  } else {
    (Protocol[i1].handler) (data);
  }
}

HANDLER(cmd_do_nothing)
{
  (void)data;
}

HANDLER(cmd_server_error)
{
  cmd_error_message(data);
}

HANDLER(cmd_login_ack)
{
  int i1;
  char *text;
  GList *dlist;
  hot_t *hot;
  GList *channels;
  chat_page_t *page;
  search_pattern_t* pattern;
  char* prefix;

  (void)data;
  if (global.clientinfo) {
    text =
      l_strdup_printf("Lopster %s (alias %s)", VERSION,
		      global.clientinfo);
    gtk_window_set_title(GTK_WINDOW(global.win), text);
  } else {
    text = l_strdup_printf("Lopster %s", VERSION);
    gtk_window_set_title(GTK_WINDOW(global.win), text);
  }
  l_free(text);

  global.user.level = L_USER;
  global.user.status = STATUS_ACTIVE;

  page = chat_page_search("MOTD", P_OTHER);
  if (page) chat_page_clear(page, 1, 1);

  connect_success();

  if ((global.options.login_mode & L_NOT_SHARE) == 0) {
    lib_set_flag(FLAG_TO_SHARE, NULL);
    // now share one file to prevent getting killed from server
    // at login time.
    lib_start_sharing();
  } else {
    lib_del_flag(FLAG_TO_SHARE, NULL);
  }

  // activate some chat pages and printing server join message
  for (dlist = global.chat_pages; dlist; dlist = dlist->next) {
    page = (chat_page_t *) (dlist->data);
    if (page->type != P_FIX) {
      chat_print_time_stamp(page, M_PUBLIC);
      prefix = cparse(global.scheme->client_prefix);
      chat_print_colored(page, M_PUBLIC, "message", prefix);
      chat_print_channel(page, M_PUBLIC, "message",
			 _("------------- Joined Server ------------- ("));
      chat_print_channel(page, M_PUBLIC, "error",
			 (SERVER->description) ? (SERVER->description) : (SERVER->address));
      chat_print_channel(page, M_PUBLIC, "message", ")\n");
      
      if (page->type == P_PUBLIC) {
	gtk_clist_clear(GTK_CLIST(page->online));
      } else {
	chat_page_set_active(page, 1);
      }
    }
  }

#ifdef PROTOCOL_DEBUG
  log("protocol", LOG_OTHER, _("Connected to %s:%d\n"), SERVER->address,
      SERVER->port);
#endif
  channels = NULL;
  if ((global.options.login_mode & L_NOT_JOIN) == 0) {
    client_message(_("Message"), _("Joining channels...."));

    for (dlist = global.string_list[LIST_AUTOJOIN]; dlist;
	 dlist = dlist->next) {
      channels = g_list_append(channels, dlist->data);
    }
  } else {
    client_message(_("Message"), _("Autojoin channels suppressed!"));
  }

  if (global.options.channel_rejoin) {
    for (dlist = global.chat_pages; dlist; dlist = dlist->next) {
      page = (chat_page_t *) (dlist->data);
      if (page->type != P_PUBLIC) continue;
      if (!is_string_in_list(channels, page->name)) {
	channels = g_list_append(channels, page->name);
      }
    }
  }
  for (dlist = channels; dlist; dlist = dlist->next) {
    join_channel(dlist->data);
  }
  if (channels) g_list_free(channels);

  client_message(_("Message"), _("Adding Hotlist entries...."));
  for (dlist = global.hotlist; dlist; dlist = dlist->next) {
    hot = (hot_t *) (dlist->data);
    send_command(CMD_CLIENT_ADD_HOTLIST_SEQ, hot->user);
  }

  /*
     chat_print("message", _("Ignoring enemy users....\n"));
     for (dlist = global.enemyuser; dlist; dlist = dlist->next) {
     send_command(CMD_CLIENT_IGNORE_USER, dlist->data);
     }
   */

  set_last_server(SERVER);
  //  whois_request(SERVER->nick, WHOIS_NONE);

  client_message(_("Message"), _("Sending Transfer Status...."));
  for (i1 = 0; i1 < global.limit.cur_real_downloads; i1++) {
    send_command(CMD_CLIENT_DOWNLOAD_START, "");
  }
  for (i1 = 0; i1 < global.limit.cur_real_uploads; i1++) {
    send_command(CMD_CLIENT_UPLOAD_START, "");
  }

  /*
  // resetting incomplete files
  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume_t* resume;
    resume = (resume_t *) (dlist->data);
    resume_remove_search(resume);
  }

  // reset left searches
  while (global.searches) {
    search_remove((search_t *) (global.searches->data));
  }
  */

  resume_search_all(0);
  for (dlist = global.search_pattern; dlist; dlist = dlist->next) {
    pattern = dlist->data;
    if (pattern->level == PATTERN_SEARCH) {
      search_pattern_search(pattern);
    }
  }
}

HANDLER(cmd_register_fail)
{
  (void)data;
  socket_destroy(global.napster, (void *) _("Nick already registered"));
}

HANDLER(cmd_bad_nick)
{
  (void)data;
  socket_destroy(global.napster, (void *) _("Invalid Nick"));
}

HANDLER(cmd_search_response)
{
  file_t *file;

  file = file_create_from_search_response(data);
  if (!file) return;

  //  resume = resume_list_search_file(file);
  file_insert_search(file, DEST_NAPSTER);

  destroy_file_row(file);
}

HANDLER(cmd_end_of_search)
{
  (void)data;
  search_finish_oldest(0);
}

HANDLER(cmd_download_ack)
{
  socket_t *socket;
  transfer_t *transfer;
  char *user;
  int port;
  unsigned long ip_long;
  char *winname;
  char *md5;
  int linespeed;
  char t2[2048] = "";

  user = arg(data, 0);
  ip_long = strtoul(arg(NULL, 0), NULL, 10);
  port = atoi(arg(NULL, 0));
  winname = arg(NULL, 0);
  md5 = arg(NULL, 0);
  linespeed = atoi(arg(NULL, 0));

  socket = download_search_mapable(user, winname);
  if (!socket) return;

  if (socket->timeout >= 0) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout = -1;
  }
  
  transfer = (transfer_t *) (socket->data);
  // dont allow peer to start DCCs
  if (transfer->is_dcc) return;

  // try to grab resume
  if (!transfer_grab_resume(socket, transfer->resume)) {
    socket_end(socket, &(SocketStatus[S_QUEUED]));
    return;
  }

  socket->port = htons(port);
  socket->ip_long = BSWAP32(ip_long);
  if (transfer->md5)
    l_free(transfer->md5);
  transfer->md5 = l_strdup(md5);
  transfer->user_info->linespeed = linespeed;

  if (socket->port == 0) {
    if ((global.network.port == 0) ||global.network.firewall) {
      socket_end(socket, &(SocketStatus[S_FIREWALL]));
    } else {
      strcpy(t2, transfer->user_info->user);
      strcat(t2, " \"");
      strcat(t2, transfer->winname);
      strcat(t2, "\"");
      send_command(CMD_CLIENT_DOWNLOAD_FIREWALL, t2);
    }
  } else {
#ifdef TRANAFER_DEBUG
    printf("download ack [%s][%s]\n", user, winname);
#endif
    transfer_connect_and_start(socket);
  }
}

HANDLER(cmd_private_message)
{
  char *t;
  char *user;
  char *message;
  GtkWidget *temp;
  char *dummy = "";
  chat_page_t *page;

  user = arg(data, 0);
  if (!user)
    return;
  message = arg(NULL, 1);
  if (!message)
    message = dummy;

  if (!strcmp(message, "//WantQueue"))
    return;
  if (!strncasecmp(message, "!Subscription", 13)) {
    subscription_send_info(user);
    return;
  }
  /*
  if (!strncasecmp(message, "!Status", 7)) {
    send_user_status(user);
    return;
  }
  */
  if (dcc_check_message(user, message))
    return;
#ifdef ENABLE_WHITE_BOARD
  if (whiteboard_check_message(user, message))
    return;
#endif

  log(user, LOG_PRIVATE, "<%s> %s\n", user, message);
  if ((page = chat_page_search(user, P_PRIVATE))) {
    t = l_strdup_printf("%s %s %s", user, user, message);
    cmd_public_message_no_log(t, M_PUBLIC);
    l_free(t);
  } else if (global.afk.message && (global.options.piping & PIPE_AWAY_LOG)) {
    page = chat_page_search("AwayLog", P_OTHER);
    if (!page)
      page = create_other_page("AwayLog", "Away Log");
    t = l_strdup_printf("AwayLog %s %s", user, message);
    cmd_public_message_no_log(t, M_PUBLIC);
    l_free(t);
  } else if (global.options.piping & PIPE_PRIVATE) {
    create_private_page(user);
    t = l_strdup_printf("%s %s %s", user, user, message);
    cmd_public_message_no_log(t, M_PUBLIC);
    l_free(t);
  } else {
    chat_print_time_stamp(global.current_page, M_PUBLIC);
    chat_print("whisper", _("<from "));
    chat_print("whisper", user);
    chat_print("whisper", "> ");
    chat_print_ln("whisper", message);
  }
  if (global.reply)
    l_free(global.reply);
  global.reply = l_strdup(user);
  temp = lookup_widget(global.win, "notebook1");
  if (gtk_notebook_get_current_page(GTK_NOTEBOOK(temp)) != 1)
    update_status_line(1);
  else
    update_status_line(0);

  if (global.afk.message)
    send_afk(user, &global.afk);
}

HANDLER(cmd_get_error)
{
  socket_t *socket;
  char *user;
  char *winname;

  user = arg(data, 0);
  winname = arg(NULL, 0);
  if (!user || !winname) return;

  socket = download_search_mapable(user, winname);
  if (socket) {
    printf("get error: %s %s\n", user, winname);
    socket_end(socket, &(SocketStatus[S_UNAVAILABLE]));
  } else {
    printf("get error, not found: %s %s\n", user, winname);
  }
}

HANDLER(cmd_user_sign_on)
{
  char *nick;
  int speed;
  hot_t *hot;

  nick = arg(data, 0);
  speed = atoi(arg(NULL, 0));

  hot = hotlist_search_user(nick);
  if (!hot)
    return;
  hot->speed = speed;
  hotlist_user_online(hot);
}

HANDLER(cmd_user_sign_off)
{
  char *nick;
  hot_t *hot;

  nick = arg(data, 0);
  hot = hotlist_search_user(nick);

  if (!hot)
    return;

  hotlist_user_offline(hot);
}

HANDLER(cmd_browse_response)
{
  file_t *file;

  file = file_create_from_browse_response(data);
  browse_insert_file(file);
}

HANDLER(cmd_browse_end)
{
  user_t *userinfo;
  char *nick;
  char *ip;
  GList *dlist;
  file_t *file;
  unsigned long ip_long;

  nick = arg(data, 0);
  ip = arg(NULL, 0);

  userinfo = browse_search_nick(nick);
  if (!userinfo)
    return;
  browse_show(userinfo, MIME_SIZE);

  if (!ip) return;

  ip_long = BSWAP32(strtoul(ip, NULL, 10));
  for (dlist = userinfo->files; dlist; dlist = dlist->next) {
    file = (file_t *) (dlist->data);
    file->ip = ip_long;
  }
  if (!l_strcasecmp(nick, SERVER->nick)) global.my_ip = ip_long;
}

HANDLER(cmd_server_stats)
{
  GtkWidget *temp;
  int songs, gics, users;
  char t[500];

  sscanf(data, "%d %d %d", &users, &songs, &gics);
  temp = lookup_widget(global.win, "label280");
  sprintf(t, _("%d Users"), users);
  gtk_label_set_text(GTK_LABEL(temp), t);
  temp = lookup_widget(global.win, "label281");
  sprintf(t, _("%d Files"), songs);
  gtk_label_set_text(GTK_LABEL(temp), t);
  temp = lookup_widget(global.win, "label282");
  sprintf(t, _("%d GB"), gics);
  gtk_label_set_text(GTK_LABEL(temp), t);
}

HANDLER(cmd_resume_match)
{
  (void)data;

  /*
     sscanf(data, "%s %*s %*s %s %*s %s %s",
     tstr[1], tstr[0], tstr[2], tstr[3]);
     strcpy(tstr[3], LineSpeed(atoi(tstr[3])));
   */
  /*
     temp = lookup_widget(global.resume_win, "clist2");
     gtk_clist_append(GTK_CLIST(temp), list);
   */
}

HANDLER(cmd_hotlist_ack)
{
  GtkCList *clist;
  int row;
  char *nick;
  hot_t *hot;
  chat_page_t *page;

  nick = arg(data, 0);
  if (!nick)
    return;

  hot = hotlist_search_user(nick);
  if (!hot)
    return;

  if (hot->visible) {
    clist = GTK_CLIST(lookup_widget(global.win, "clist16"));
    row = gtk_clist_find_row_from_data(clist, hot);
    if (row >= 0) {
      gtk_clist_set_pixtext(clist, row, 0, nick, 5,
			    global.pix.user2, global.pix.user2b);
    }
  }
  hot->online = HOT_OFFLINE;

  page = chat_page_search(nick, P_PRIVATE);
  if (!page)
    return;
  chat_page_set_active(page, 0);
}

HANDLER(cmd_hotlist_error)
{
  client_message(_("Error"), data);
}

HANDLER(cmd_disconnecting)
{
  (void)data;
  client_message(_("Message"), _("You will be disconnected"));
}

HANDLER(cmd_ignore_user)
{
  ignore_add(data);
}

HANDLER(cmd_unignore_user)
{
  ignore_remove(data);
}

HANDLER(cmd_part_channel)
{
  chat_page_t *page;

  page = chat_page_search(data, P_PUBLIC);
  if (!page) return;
  if (page->close_on_part)
    chat_page_destroy(page);
  else
    chat_page_set_active(page, 0);
}

HANDLER(cmd_public_message)
{
  char *data2 = NULL;
  char *page;
  char *user;
  char *message;
  GtkWidget *temp;

  if (data)
    data2 = l_strdup(data);

  page = arg(data2, 0);
  user = arg(NULL, 0);
  if (!page || !user)
    return;
  message = arg(NULL, 1);

  log(page, LOG_CHANNEL, "<%s> %s\n", user, message ? message : "");

  temp = lookup_widget(global.win, "notebook1");
  if (message && l_strcasecmp(user, SERVER->nick) && strcasestr(message, SERVER->nick)) {
    if (global.reply)
      l_free(global.reply);
    global.reply = l_strdup(user);
    if (gtk_notebook_get_current_page(GTK_NOTEBOOK(temp)) != 1)
      update_status_line(1);
    else
      update_status_line(0);
  }

  l_free(data2);
  cmd_public_message_no_log(data, M_PUBLIC);
}

int check_not_known1(char *data)
{
  char *text;
  char *user;
  char *data2;

  if (!strncmp("User is not currently online", data, 28))
    return 1;

  data2 = l_strdup(data);
  text = arg(data2, 0);

  if (!text) {
    l_free(data2);
    return 0;
  }
  if (l_strcasecmp("user", text)) {
    l_free(data2);
    return 0;
  }
  user = arg(NULL, 0);
  if (!user) {
    l_free(data2);
    return 0;
  }
  text = arg(NULL, 1);
  if (!text) {
    l_free(data2);
    return 0;
  }
  if (l_strcasecmp("is not a known user", text)) {
    l_free(data2);
    return 0;
  }

  whois_eval_not_online(user);
  l_free(data2);
  return 1;
}

int check_not_known2(char *data)
{
  char *text;
  char *user;
  char *data2;

  data2 = l_strdup(data);
  text = arg(data2, 0);
  if (!text) {
    l_free(data2);
    return 0;
  }
  if (l_strcasecmp("information", text)) {
    l_free(data2);
    return 0;
  }

  text = arg(NULL, 0);
  if (!text) {
    l_free(data2);
    return 0;
  }
  if (l_strcasecmp("on", text)) {
    l_free(data2);
    return 0;
  }

  user = arg(NULL, 0);
  if (!user) {
    l_free(data2);
    return 0;
  }
  text = arg(NULL, 1);
  if (!text) {
    l_free(data2);
    return 0;
  }
  if (strncmp("is not available", text, 16)) {
    l_free(data2);
    return 0;
  }

  whois_eval_not_online(user);
  l_free(data2);
  return 1;
}

void check_ping(char *data)
{
  char *text;
  char *user;
  char *data2;
  user_timestamp_t *stamp;

  data2 = l_strdup(data);
  text = arg(data2, 0);
  if (!text) {
    l_free(data2);
    return;
  }
  if (l_strcasecmp("ping", text)) {
    l_free(data2);
    return;
  }

  text = arg(NULL, 0);
  if (!text) {
    l_free(data2);
    return;
  }
  if (l_strcasecmp("failed,", text)) {
    l_free(data2);
    return;
  }

  user = arg(NULL, 0);
  if (!user) {
    l_free(data2);
    return;
  }

  stamp = timestamp_search(global.userstamp, user);
  if (!stamp)
    return;
  global.userstamp = g_list_remove(global.userstamp, stamp);
  l_free(stamp->user);
  l_free(stamp);

  return;
}

void check_chat_page(char *data)
{
  GList* dlist;
  chat_page_t *page;
  char *prefix;

  for (dlist = global.chat_pages; dlist; dlist = dlist->next) {
    page = dlist->data;
    if ((page->type != P_PUBLIC) && (page->type != P_PRIVATE)) continue;
    if (!strcasestr(data, page->name)) continue;
    chat_print_time_stamp(page, M_PUBLIC);
    prefix = cparse(global.scheme->server_prefix);
    chat_print_colored(page, M_PUBLIC, "error", prefix);
    chat_print_colored(page, M_PUBLIC, "error", data);
    chat_print_colored(page, M_PUBLIC, "error", "\n");
    if (page == global.current_page) return;
  }
  if (global.current_page && strstr(data, SERVER->nick)) {
    chat_print_time_stamp(global.current_page, M_PUBLIC);
    prefix = cparse(global.scheme->server_prefix);
    chat_print_colored(global.current_page, M_PUBLIC, "error", prefix);
    chat_print_colored(global.current_page, M_PUBLIC, "error", data);
    chat_print_colored(global.current_page, M_PUBLIC, "error", "\n");
  }
}

void check_channel_kick(char* data) {
  char* temp;
  char* channel;
  char* who;

  if (!global.options.rejoin_when_kicked) return;

  if (!strncmp("You were kicked from channel ", data, 29)) {
    temp = l_strdup(data+29);
    channel = arg(temp, 0);
    who = arg(NULL, 1);
    if (!strcasestr(who, "server") && channel) {
      join_channel(channel);
    }
    l_free(temp);
  }
}

void check_level_set(char* data) {

  if (strstr(data, "your user level to")) {
    whois_request(SERVER->nick, WHOIS_NONE);
  }
}

void check_channel_mode(char* data) {
  char* channel;
  char* mode;
  char* temp;
  int chmode = 0;

  if (strncmp("mode for channel ", data, 17)) return;
  temp = l_strdup(data+17);
  channel = arg(temp, 0);

  if (!channel) return;
  while ((mode = arg(NULL, 0)) != NULL) {
    chmode |= channelmode2int(mode);
  }

  channel_set_mode(channel, 0xff, 0, 0);
  channel_set_mode(channel, chmode, 1, 0);
  l_free(temp);
}

void check_channel_mode2(char* data) {
  char* pos;
  char* temp;
  char* channel;
  char* mode;
  int chmode1 = 0;
  int chmode2 = 0;

  pos = strstr(data, "changed mode on channel ");
  if (!pos) return;
  printf("modify channel mode [%s]\n",data);

  temp = l_strdup(pos+24);
  channel = arg(temp, 0);
  if (!channel) {
    l_free(temp);
    return;
  }
  channel[strlen(channel)-1] = 0;

  while ((mode = arg(NULL, 0)) != NULL) {
    if (*mode == '+')
      chmode1 |= channelmode2int(mode);
    else  if (*mode == '-')
      chmode2 |= channelmode2int(mode);
  }

  channel_set_mode(channel, chmode1, 1, 0);
  channel_set_mode(channel, chmode2, 0, 0);

  l_free(temp);
}

void check_channel_limit_level(char* data) {
  char* channel;
  char* temp;
  char* rest;

  if (strncmp("Channel ", data, 8)) return;
  temp = l_strdup(data+8);
  channel = arg(temp, 0);
  if (!channel) {
    l_free(temp);
    return;
  }
  rest = arg(NULL, 1);
  if (!rest) {
    l_free(temp);
    return;
  }
  if (!strncmp("has limit ", rest, 10)) {
    channel_set_limit(channel, atoi(rest+10), 0);
    l_free(temp);
    return;
  }
  if (!strncmp("is level ", rest, 9)) {
    channel_set_level(channel, level2int(rest+9), 0);
    l_free(temp);
    return;
  }
  
  l_free(temp);
  return;
}

void check_channel_limit(char* data) {
  char* pos1;
  char* channel;
  char* limit;
  char* temp;

  pos1 = strstr(data, "set limit on channel ");
  if (!pos1) return;
  temp = l_strdup(pos1+21);
  
  channel = arg(temp, 0);
  pos1 = arg(NULL, 0);
  limit = arg(NULL, 0);
  if (!channel || !limit) {
    l_free(temp);
    return;
  }
  channel_set_limit(channel, atoi(limit), 0);
  l_free(temp);
}

void check_channel_level(char* data) {
  char* pos1;
  char* pos2;
  char* channel;
  char* level;
  char* temp;

  pos1 = strstr(data, "set channel ");
  pos2 = strstr(data, "to level ");
  if (!pos1 || !pos2) return;

  temp = l_strdup(pos1+12);
  channel = arg(temp, 0);
  pos1 = arg(NULL, 0);
  pos1 = arg(NULL, 0);
  level = arg(NULL, 0);
  if (!channel || !level) {
    l_free(temp);
    return;
  }
  printf("%s %d\n", channel, level2int(level));
  channel_set_level(channel, level2int(level), 0);
  l_free(temp);
}

HANDLER(cmd_error_message)
{
  if (check_channel_wallop(data))
    return;

  check_server_join_quit(data);
  check_server_pong(data);
  //  check_get_motd(data);
  check_op(data);
  check_pending_searches(data);
  check_share_limit(data);
  if (check_not_known1(data)) return;
  if (check_not_known2(data)) return;
  check_ping(data);
  check_chat_page(data);
  check_channel_kick(data);
  check_level_set(data);
  check_channel_mode(data);
  check_channel_mode2(data);
  check_channel_limit_level(data);
  check_channel_level(data);
  check_channel_limit(data);

  server_message(NULL, data);
  
  if (global.status.connection == 1)
    napster_disconnect(data);
  log("Messages", LOG_OTHER, "%s\n", data);

}

void
on_chmode_clicked(GtkToggleButton * togglebutton, gpointer user_data) {
  int set;
  chat_page_t* page;

  if (gtk_toggle_button_get_active(togglebutton)) set = 1;
  else set = 0;

  page = gtk_object_get_data(GTK_OBJECT(togglebutton), "page");
  if (!page) {
    printf("page not found\n");
    return;
  }
  channel_set_mode(page->name, (int)user_data, set, 1);
}

void on_limit_activate(GtkEditable * editable, gpointer user_data) {
  char* str;
  chat_page_t* page = user_data;

  str = gtk_entry_get_text(GTK_ENTRY(editable));
  channel_set_limit(page->name, atoi(str), 1);
}

void on_level_changed(GtkEditable * editable, gpointer user_data) {
  char* str;
  chat_page_t* page = user_data;

  str = gtk_entry_get_text(GTK_ENTRY(editable));
  channel_set_level(page->name, level2int(str), 1);
}

HANDLER(cmd_create_channel)
{
  GtkWidget *channel_widget;
  GtkWidget *channel_widget2;
  GtkWidget *scrolledwindow27;
  GtkWidget *text4;
  GtkWidget *topic;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *scrolledwindow26;
  GtkWidget *clist1;
  GtkWidget *label96;
  GtkWidget *label97;
  GtkWidget *label98;
  GtkWidget *tab_label;
  GtkWidget *entry18;
  GtkWidget *button;
  GtkWidget *frame;
  GtkNotebook *notebook;
  GtkTooltips *tips;
  GtkWidget *button2;
  GtkWidget *pix1;
  GtkWidget *vbox2;
  GtkWidget *hbox2;
  GtkWidget *hbox3;
  GtkWidget *hbox4;
  GtkWidget *label;
  GtkWidget *header;
  GtkWidget *main_;
  GtkWidget *wallop;
  GtkWidget *hbox_temp;
  GtkWidget *combo;
  GtkWidget *combo_entry;
  GList *combo_items;
  GtkWidget *wid;

  chat_page_t *page;
  style_t *style;

  tips = (GtkTooltips *) gtk_object_get_data(GTK_OBJECT(global.win),
					     "tooltips");

  page = chat_page_search(data, P_PUBLIC);

  if (page) {
    send_command(CMD_CLIENT_CHANNEL_MODE, page->name);
    send_command(CMD_CLIENT_CHANNEL_LIMIT, page->name);
    send_command(CMD_CLIENT_SET_CHAN_LEVEL, page->name);
    if (page->active) return;			// should not happen
    chat_page_set_active(page, 1);
    return;
  }
  page = chat_page_new(data, P_PUBLIC);
  send_command(CMD_CLIENT_CHANNEL_MODE, page->name);
  send_command(CMD_CLIENT_CHANNEL_LIMIT, page->name);
  send_command(CMD_CLIENT_SET_CHAN_LEVEL, page->name);

  channel_widget = gtk_hpaned_new();
  gtk_paned_set_gutter_size(GTK_PANED(channel_widget), 10);
  gtk_paned_set_position(GTK_PANED(channel_widget), global.paned_pos);
  gtk_widget_show(channel_widget);

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox);
  gtk_paned_pack1(GTK_PANED(channel_widget), vbox, TRUE, TRUE);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_object_set_data(GTK_OBJECT(global.win), "hbox", hbox);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  topic = gtk_text_new(NULL, NULL);
  header = topic;
  gtk_widget_set_style(topic, global.style[2]);
  gtk_widget_show(topic);
  gtk_signal_connect_after(GTK_OBJECT(topic), "button_press_event",
			   GTK_SIGNAL_FUNC(on_text_button_press_event),
			   NULL);

  style = style_get(global.scheme, "text");
  if (style->font)
    gtk_widget_set_usize(topic, -1,
			 style->font->ascent + style->font->descent + 7);
  else
    gtk_widget_set_usize(topic, -1, 22);
  gtk_text_set_word_wrap(GTK_TEXT(topic), 1);
  gtk_box_pack_start(GTK_BOX(hbox), topic, TRUE, TRUE, 0);

  //
  button2 = gtk_button_new();
  gtk_widget_show(button2);
  gtk_box_pack_start(GTK_BOX(hbox), button2, FALSE, FALSE, 0);
  GTK_WIDGET_UNSET_FLAGS(button2, GTK_CAN_FOCUS);
  gtk_button_set_relief(GTK_BUTTON(button2), GTK_RELIEF_NONE);

  vbox2 = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vbox2);
  gtk_container_add(GTK_CONTAINER(button2), vbox2);

  hbox2 = gtk_hbox_new(FALSE, 5);
  hbox_temp = hbox2;
  gtk_widget_show(hbox2);
  gtk_box_pack_start(GTK_BOX(vbox2), hbox2, FALSE, FALSE, 0);

  label = gtk_label_new(_("hide users"));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);

  pix1 = create_pixmap(global.win, "arrowr.xpm");
  gtk_widget_show(pix1);
  gtk_box_pack_start(GTK_BOX(hbox2), pix1, FALSE, FALSE, 0);

  hbox2 = gtk_hbox_new(FALSE, 5);
  //gtk_widget_show (hbox2);
  gtk_box_pack_start(GTK_BOX(vbox2), hbox2, FALSE, FALSE, 0);

  pix1 = create_pixmap(global.win, "arrowl.xpm");
  gtk_widget_show(pix1);
  gtk_box_pack_start(GTK_BOX(hbox2), pix1, FALSE, FALSE, 0);

  gtk_signal_connect(GTK_OBJECT(button2), "clicked",
		     GTK_SIGNAL_FUNC(on_button_users_clicked), NULL);
  label = gtk_label_new(_("show users"));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);

  // new
  channel_widget2 = gtk_vpaned_new();
  gtk_widget_show(channel_widget2);
  if (global.user.level > L_USER)
    gtk_paned_set_position(GTK_PANED(channel_widget2), global.paned_pos2);
  else
    gtk_paned_set_position(GTK_PANED(channel_widget2), 0);
  gtk_box_pack_start(GTK_BOX(vbox), channel_widget2, TRUE, TRUE, 0);

  //
  scrolledwindow27 = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow27);

  gtk_paned_pack1(GTK_PANED(channel_widget2), scrolledwindow27, TRUE,
		  TRUE);

  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow27),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  gtk_range_set_update_policy(GTK_RANGE
			      (GTK_SCROLLED_WINDOW(scrolledwindow27)->
			       hscrollbar), GTK_UPDATE_CONTINUOUS);
  gtk_range_set_update_policy(GTK_RANGE
			      (GTK_SCROLLED_WINDOW(scrolledwindow27)->
			       vscrollbar), GTK_UPDATE_CONTINUOUS);

  text4 = gtk_text_new(NULL, NULL);
  wallop = text4;
  gtk_widget_set_style(text4, global.style[2]);
  gtk_signal_connect_after(GTK_OBJECT(text4), "button_press_event",
			   GTK_SIGNAL_FUNC(on_text_button_press_event),
			   NULL);

  gtk_widget_show(text4);
  gtk_text_set_word_wrap(GTK_TEXT(text4), 1);
  gtk_container_add(GTK_CONTAINER(scrolledwindow27), text4);

  //
  scrolledwindow27 = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow27);

  gtk_paned_pack2(GTK_PANED(channel_widget2), scrolledwindow27, TRUE,
		  TRUE);
  //  gtk_box_pack_start (GTK_BOX (vbox), scrolledwindow27, TRUE, TRUE, 0);

  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow27),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  gtk_range_set_update_policy(GTK_RANGE
			      (GTK_SCROLLED_WINDOW(scrolledwindow27)->
			       hscrollbar), GTK_UPDATE_CONTINUOUS);
  gtk_range_set_update_policy(GTK_RANGE
			      (GTK_SCROLLED_WINDOW(scrolledwindow27)->
			       vscrollbar), GTK_UPDATE_CONTINUOUS);

  text4 = gtk_text_new(NULL, NULL);
  main_ = text4;
  gtk_widget_set_style(text4, global.style[2]);
  gtk_signal_connect_after(GTK_OBJECT(text4), "button_press_event",
			   GTK_SIGNAL_FUNC(on_text_button_press_event),
			   NULL);

  gtk_widget_show(text4);
  gtk_text_set_word_wrap(GTK_TEXT(text4), 1);
  gtk_container_add(GTK_CONTAINER(scrolledwindow27), text4);

  ///

  vbox = gtk_vbox_new(FALSE, 0);
  wid = vbox;
  gtk_widget_show(vbox);
  gtk_paned_pack2(GTK_PANED(channel_widget), vbox, TRUE, TRUE);

  hbox3 = gtk_hbox_new(FALSE, 0);
  gtk_widget_show (hbox3);
  gtk_box_pack_start(GTK_BOX(vbox), hbox3, FALSE, FALSE, 0);

  frame = gtk_frame_new(NULL);
  gtk_widget_show(frame);
  gtk_box_pack_start(GTK_BOX(hbox3), frame, FALSE, FALSE, 0);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);

  hbox4 = gtk_hbox_new(TRUE, 0);
  gtk_widget_show (hbox4);
  gtk_container_add(GTK_CONTAINER(frame), hbox4);

  button = gtk_toggle_button_new_with_label("T");
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(hbox4), button, TRUE, TRUE, 0);
  gtk_object_set_data(GTK_OBJECT(main_), "MT", button);
  gtk_signal_connect_after(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(on_chmode_clicked), (gpointer)CMODE_TOPIC);
  gtk_object_set_data(GTK_OBJECT(button), "page", (gpointer)page);
  if (global.user.level <= L_USER)
    gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
  gtk_tooltips_set_tip (tips, button, 
			_("Topic is set-able by all users"), NULL);

  button = gtk_toggle_button_new_with_label("R");
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(hbox4), button, TRUE, TRUE, 0);
  gtk_object_set_data(GTK_OBJECT(main_), "MR", button);
  gtk_signal_connect_after(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(on_chmode_clicked), (gpointer)CMODE_REGISTERED);
  gtk_object_set_data(GTK_OBJECT(button), "page", (gpointer)page);
  if (global.user.level <= L_USER)
    gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
  gtk_tooltips_set_tip (tips, button, 
			_("Channel is registered"), NULL);

  button = gtk_toggle_button_new_with_label("P");
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(hbox4), button, TRUE, TRUE, 0);
  gtk_object_set_data(GTK_OBJECT(main_), "MP", button);
  gtk_signal_connect_after(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(on_chmode_clicked), (gpointer)CMODE_PRIVATE);
  gtk_object_set_data(GTK_OBJECT(button), "page", (gpointer)page);
  if (global.user.level <= L_USER)
    gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
  gtk_tooltips_set_tip (tips, button, 
			_("Channel is private"), NULL);

  button = gtk_toggle_button_new_with_label("I");
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(hbox4), button, TRUE, TRUE, 0);
  gtk_object_set_data(GTK_OBJECT(main_), "MI", button);
  gtk_signal_connect_after(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(on_chmode_clicked), (gpointer)CMODE_INVITE);
  gtk_object_set_data(GTK_OBJECT(button), "page", (gpointer)page);
  if (global.user.level <= L_USER)
    gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
  gtk_tooltips_set_tip (tips, button, 
			_("Channel is invite only"), NULL);

  button = gtk_toggle_button_new_with_label("M");
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(hbox4), button, TRUE, TRUE, 0);
  gtk_object_set_data(GTK_OBJECT(main_), "MM", button);
  gtk_signal_connect_after(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(on_chmode_clicked), (gpointer)CMODE_MODERATED);
  gtk_object_set_data(GTK_OBJECT(button), "page", (gpointer)page);
  if (global.user.level <= L_USER)
    gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
  gtk_tooltips_set_tip (tips, button, 
			_("Channel is moderated"), NULL);

  /*
  button = gtk_toggle_button_new_with_label("Q");
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(hbox4), button, TRUE, TRUE, 0);
  gtk_object_set_data(GTK_OBJECT(main_), "MQ", button);
  gtk_signal_connect_after(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(on_chmode_clicked), (gpointer)CMODE_QUIET);
  gtk_object_set_data(GTK_OBJECT(button), "page", (gpointer)page);
  if (global.user.level <= L_USER)
    gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
  */

  hbox4 = gtk_hbox_new(FALSE, 0);
  gtk_widget_show (hbox4);
  gtk_box_pack_start(GTK_BOX(hbox3), hbox4, TRUE, TRUE, 0);

  entry18 = gtk_entry_new();
  gtk_tooltips_set_tip (tips, entry18, 
			_("Channel user limit"), NULL);
  gtk_widget_show (entry18);
  gtk_box_pack_start(GTK_BOX(hbox4), entry18, TRUE, TRUE, 0);
  if (global.user.level <= L_USER)
    gtk_widget_set_sensitive(GTK_WIDGET(entry18), FALSE);
  gtk_object_set_data(GTK_OBJECT(main_), "limit", entry18);
  gtk_signal_connect (GTK_OBJECT (entry18), "activate",
                      GTK_SIGNAL_FUNC (on_limit_activate),
                      (gpointer)page);

  //////////

  combo = gtk_combo_new ();
  gtk_box_pack_start(GTK_BOX(hbox4), combo, FALSE, FALSE, 0);
  gtk_widget_set_usize (combo, 70, -2);

  gtk_widget_show (combo);
  combo_items = NULL;
  combo_items = g_list_append (combo_items, LevelShort(0));
  combo_items = g_list_append (combo_items, LevelShort(1));
  combo_items = g_list_append (combo_items, LevelShort(2));
  combo_items = g_list_append (combo_items, LevelShort(3));
  combo_items = g_list_append (combo_items, LevelShort(4));
  gtk_combo_set_popdown_strings (GTK_COMBO (combo), combo_items);
  g_list_free (combo_items);
  
  if (global.user.level <= L_USER)
    gtk_widget_set_sensitive(GTK_WIDGET(combo), FALSE);

  gtk_object_set_data(GTK_OBJECT(main_), "level", combo);

  combo_entry = GTK_COMBO (combo)->entry;
  gtk_widget_show (combo_entry);
  gtk_tooltips_set_tip (tips, combo_entry, 
			_("Channel level"), NULL);
  gtk_entry_set_editable (GTK_ENTRY (combo_entry), FALSE);
  gtk_entry_set_text(GTK_ENTRY(combo_entry), LevelShort(page->level));
  gtk_signal_connect (GTK_OBJECT (combo_entry), "changed",
                      GTK_SIGNAL_FUNC (on_level_changed),
                      (gpointer)page);

  scrolledwindow26 = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow26);
  gtk_box_pack_start(GTK_BOX(vbox), scrolledwindow26, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow26),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);

  clist1 = gtk_clist_new(3);

  gtk_widget_show(clist1);
  gtk_container_add(GTK_CONTAINER(scrolledwindow26), clist1);
  gtk_clist_set_column_width(GTK_CLIST(clist1), 0, global.online_width[0]);
  gtk_clist_set_column_width(GTK_CLIST(clist1), 1, global.online_width[1]);
  gtk_clist_set_column_width(GTK_CLIST(clist1), 2, global.online_width[2]);
  gtk_clist_set_column_visibility(GTK_CLIST(clist1), 0,
				  global.online_show[0]);
  gtk_clist_set_column_visibility(GTK_CLIST(clist1), 1,
				  global.online_show[1]);
  gtk_clist_set_column_visibility(GTK_CLIST(clist1), 2,
				  global.online_show[2]);
  gtk_clist_column_titles_show(GTK_CLIST(clist1));

  label96 = gtk_label_new(_("User"));
  gtk_widget_show(label96);
  gtk_clist_set_column_widget(GTK_CLIST(clist1), 0, label96);

  label97 = gtk_label_new(_("Share"));
  gtk_widget_show(label97);
  gtk_clist_set_column_widget(GTK_CLIST(clist1), 1, label97);

  label98 = gtk_label_new(_("Speed"));
  gtk_widget_show(label98);
  gtk_clist_set_column_widget(GTK_CLIST(clist1), 2, label98);


  hbox = gtk_hbox_new(FALSE, 0);
  gtk_object_set_data(GTK_OBJECT(global.win), "hbox", hbox);
  gtk_widget_show(hbox);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);


  entry18 = gtk_entry_new();
  gtk_widget_show(entry18);
  gtk_box_pack_start(GTK_BOX(hbox), entry18, TRUE, TRUE, 0);

  frame = gtk_frame_new(NULL);
  gtk_widget_show(frame);
  gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, FALSE, 0);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);

  button = gtk_button_new_with_label(_("Save"));
  gtk_widget_show(button);
  gtk_container_add(GTK_CONTAINER(frame), button);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(on_online_save_clicked), NULL);

  gtk_tooltips_set_tip(tips, button, _("Save Chat Page Layout"), NULL);

  tab_label = gtk_label_new(data);
  gtk_widget_ref(tab_label);
  gtk_widget_show(tab_label);

  gtk_signal_connect(GTK_OBJECT(clist1), "button_press_event",
		     GTK_SIGNAL_FUNC(on_online_button_press_event), NULL);
  gtk_signal_connect(GTK_OBJECT(clist1), "click_column",
		     GTK_SIGNAL_FUNC(on_list_click_column), NULL);
  gtk_clist_set_sort_column(GTK_CLIST(clist1), 0);
  gtk_clist_set_auto_sort(GTK_CLIST(clist1), TRUE);
  gtk_clist_set_compare_func(GTK_CLIST(clist1), user_compare);

  notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook3"));
  gtk_notebook_append_page(notebook, channel_widget, tab_label);
  page->header = header;
  page->main = main_;
  page->wallop = wallop;
  page->tlabel = tab_label;
  page->mlabel = tab_label;
  page->online = clist1;
  page->no = entry18;
  page->toplevel = channel_widget;
  page->paned = channel_widget2;
  gtk_object_set_data(GTK_OBJECT(tab_label), "page", page);

  gtk_notebook_set_page(GTK_NOTEBOOK(notebook),
			gtk_notebook_page_num(GTK_NOTEBOOK(notebook),
					      channel_widget));

  log(data, LOG_CHANNEL, "[................] %s", ctime(&global.current_time));

  set_up_button2(button2, hbox2, hbox_temp, wid);
  gtk_signal_connect(GTK_OBJECT(button2), "clicked",
		     GTK_SIGNAL_FUNC(on_hide_show_clicked), NULL);

  if (global.user.level > L_USER)
    opchannel_add(page->name);

}

HANDLER(cmd_join_message)
{
  char *channel;
  char *user;
  char *link;
  char *share;
  char *friend;
  chat_page_t *page;
  char *temp;
  char *data2 = l_strdup(data);
  char* prefix;

  channel = arg(data2, 0);
  user = arg(NULL, 0);
  share = arg(NULL, 0);
  link = arg(NULL, 0);

  if (!link) {
    l_free(data2);
    return;
  }

  friend = string_list_search(LIST_FRIEND, user);
  page = chat_page_search(channel, P_PUBLIC);
  if (page && ((global.options.show_joins == 1) ||friend)) {
    chat_print_time_stamp(page, M_PUBLIC);
    prefix = cparse(global.scheme->join_prefix);
    chat_print_colored(page, M_PUBLIC, "error", prefix);
    chat_print_channel(page, M_PUBLIC, "user", "[");
    if (friend)
      chat_print_channel(page, M_PUBLIC, "friend", user);
    else
      chat_print_channel(page, M_PUBLIC, "join", user);
    chat_print_channel(page, M_PUBLIC, "join", " has joined");
    chat_print_channel(page, M_PUBLIC, "user", "]");
    temp =
	l_strdup_printf(" (%s)(%d)\n",
			LineSpeedShort[verify_speed(atoi(link))],
			atoi(share));
    chat_print_channel(page, M_PUBLIC, "NULL", temp);
    l_free(temp);
  }
  cmd_user_online(data);
  l_free(data2);
}

HANDLER(cmd_part_message)
{
  char *channel;
  char *user;
  char *link;
  char *share;
  GtkWidget *temp;
  GtkWidget *temp2;
  char temp_str[1024];
  char *pos;
  chat_page_t *page;
  user_timestamp_t *stamp;
  int row;
  char* prefix;

  channel = arg(data, 0);
  user = arg(NULL, 0);
  share = arg(NULL, 0);
  link = arg(NULL, 0);

  if (!link)
    return;

  pos = string_list_search(LIST_FRIEND, user);
  page = chat_page_search(channel, P_PUBLIC);
  if (!page)
    return;

  stamp = timestamp_search(global.appearance, user);

  if ((stamp && (timestamp_difference(stamp) < 1000 * global.options.recent_timeout) &&(global.options.show_parts == 1)) ||	// recent
      (global.options.show_parts == 2) ||	// show all
      pos) {			// friend
    chat_print_time_stamp(page, M_PUBLIC);
    prefix = cparse(global.scheme->part_prefix);
    chat_print_colored(page, M_PUBLIC, "error", prefix);
    chat_print_channel(page, M_PUBLIC, "user", "[");
    if (pos)
      chat_print_channel(page, M_PUBLIC, "friend", user);
    else
      chat_print_channel(page, M_PUBLIC, "part", user);
    chat_print_channel(page, M_PUBLIC, "part", " has left");
    chat_print_channel(page, M_PUBLIC, "user", "]");
    sprintf(temp_str, " (%s)(%d)\n",
	    LineSpeedShort[verify_speed(atoi(link))], atoi(share));
    chat_print_channel(page, M_PUBLIC, "NULL", temp_str);
  }

  temp = page->online;
  row = search_user_in_list(GTK_CLIST(temp), user);
  if (row < 0) {
    g_warning("user not in list %s", user);
  } else {
    gtk_clist_remove(GTK_CLIST(temp), row);
  }

  temp2 = page->no;
  sprintf(temp_str, _("%d users online !"), GTK_CLIST(temp)->rows);
  gtk_entry_set_text(GTK_ENTRY(temp2), temp_str);
}

HANDLER(cmd_user_online)
{
  char *t2;
  GtkWidget *temp;
  GtkWidget *temp2;
  int row;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  char *channel;
  char *user;
  char *share;
  char *speed;
  chat_page_t *page;

  channel = arg(data, 0);
  user = arg(NULL, 0);
  share = arg(NULL, 0);
  speed = arg(NULL, 0);

  if (!speed)
    return;

  page = chat_page_search(channel, P_PUBLIC);
  if (!page)
    return;

  temp = page->online;

  row = search_user_in_list(GTK_CLIST(temp), user);
  strcpy(tstr[0], user);
  strcpy(tstr[1], share);
  strcpy(tstr[2], LineSpeed(atoi(speed)));
  if (row == -1)
    row = gtk_clist_append(GTK_CLIST(temp), list);

  detect_line_pixs(atoi(speed), &pixmap, &bitmap);
  gtk_clist_set_pixtext(GTK_CLIST(temp), row, 2,
			tstr[2], 5, pixmap, bitmap);
  gtk_clist_set_pixtext(GTK_CLIST(temp), row, 0,
			tstr[0], 5, global.pix.dummy, global.pix.dummyb);
  gtk_clist_set_text(GTK_CLIST(temp), row, 1, tstr[1]);

  temp2 = page->no;
  t2 = l_strdup_printf(_("%d users online !"), GTK_CLIST(temp)->rows);
  gtk_entry_set_text(GTK_ENTRY(temp2), t2);
  l_free(t2);

  update_user(user);
}

HANDLER(cmd_channel_topic)
{
  char *room;
  char *topic;
  chat_page_t *page;
  GtkWidget *temp;

  room = arg(data, 0);
  topic = arg(NULL, 1);

  page = chat_page_search(room, P_PUBLIC);
  if (!page) return;

  temp = page->header;
  gtk_text_set_point(GTK_TEXT(temp), 0);
  gtk_text_forward_delete(GTK_TEXT(temp),
			  gtk_text_get_length(GTK_TEXT(temp)));
  if (topic)
    chat_print_colored(page, M_TOPIC, "text", topic);
}

HANDLER(cmd_banlist_entry)
{
  char* who;
  char* sender;
  char* reason;
  char* when;
  char* timeout;
  time_t l1;
  time_t l2;
  GtkWidget *temp;

  if (!global.ban_win)
    return;

  temp = lookup_widget(global.ban_win, "clist3");
  
  who = arg(data, 0);
  sender = arg(NULL, 0);
  reason = arg(NULL, 0);
  when = arg(NULL, 0);
  timeout = arg(NULL, 0);

  if (!who || !sender || !reason || !when) return;

  strcpy(tstr[0], who);
  strcpy(tstr[1], sender);
  strcpy(tstr[3], reason);
  l1 = strtoul(when, NULL, 10);
  if (timeout) {
    l2 = strtoul(timeout, NULL, 10);
    strcpy(tstr[2], ctime(&l1));
    tstr[2][strlen(tstr[2])-1] = 0;
    if (l2 > 0) {
      l2 += l1;
      strcat(tstr[2], " (expires ");
      strcat(tstr[2], ctime(&l2));
      tstr[2][strlen(tstr[2])-1] = 0;
      strcat(tstr[2], ")");
    }
  } else {
    sprintf(tstr[2], "%s", ctime(&l1));
  }
  gtk_clist_append(GTK_CLIST(temp), list);
}

HANDLER(cmd_channel_alt_topic)
{
  client_message(_("Alternate Topic"), data);
}

HANDLER(cmd_alternate_ack)
{
  transfer_t *transfer;
  socket_t *socket;

  char *user;
  char *ip;
  char *port;
  char *winname;
  char *md5;
  char *linespeed;
  file_t *file;

  user = arg(data, 0);
  ip = arg(NULL, 0);
  port = arg(NULL, 0);
  winname = arg(NULL, 0);
  md5 = arg(NULL, 0);
  linespeed = arg(NULL, 0);

  socket = upload_search_mapable(user, winname);
  if (!socket) return;
  transfer = (transfer_t *) (socket->data);

  if (!transfer_in_progress(transfer))
    return;

  transfer->user_info->linespeed = atoi(linespeed);
  if (transfer->md5)
    l_free(transfer->md5);
  transfer->md5 = l_strdup(md5);
  socket->port = htons(atoi(port));
  socket->ip_long = BSWAP32(strtoul(ip, NULL, 10));

  file = lib_search_transfer(transfer);
  if (!file) {
    // this is the first lib check
    // should send someting to the user?
    g_warning("file not found in lib\n");
    return;
  }
  transfer->size = file->size;
  transfer->mime_type = file->mime_type;

  if (socket->port == 0) {
    socket_end(socket, &(SocketStatus[S_FIREWALL]));
  } else {
    transfer_connect_and_start(socket);
  }
}

HANDLER(cmd_linkspeed_response)
{
  char *nick;
  char *linespeed;
  speed_t *speed;

  //  printf("**** linkspeed response [%s]\n", data);
  nick = arg(data, 0);
  linespeed = arg(NULL, 0);
  if (!nick || !linespeed)
    return;
  speed = speed_search(nick, SPEED_ALL);
  if (!speed) return;

  speed->speed = atoi(linespeed);
  speed_action(speed);
}

HANDLER(cmd_eval_whois)
{
  whois_eval_whois(data);
}

HANDLER(cmd_eval_whowas)
{
  whois_eval_whowas(data);
}

HANDLER(cmd_upload_request)
{
  transfer_t *new_trans;
  socket_t *socket;
  char *user;
  char *winname;
  char *speed;
  //  file_t* file;
  subscription_t* sub;
  char* found;

  user = arg(data, 0);
  winname = arg(NULL, 0);
  speed = arg(NULL, 0);

  if (!user || !winname) return;

  // do not allow uploads from enemies
  if (string_list_search(LIST_ENEMY, user))
    return;
  // do not allow uploads on safe exit
  if (global.status.exiting == E_SAFE)
    return;

  socket = upload_search_mapable(user, winname);

  if (!socket) {
    new_trans = transfer_new();
    new_trans->winname = l_strdup(winname);
    new_trans->longname = l_strdup(winname);
    convert_to_unix(new_trans->longname);
    new_trans->shortname = l_strdup(extract_short_name(new_trans->longname));
    transfer_detect_user_info(new_trans, user);
    new_trans->type = T_UPLOAD;
    new_trans->mime_type = get_mimetype(winname);

    socket = socket_new(S_TRANSFER);
    socket->data = new_trans;

    // better check this when connected to user?
    /*
       file = lib_search_transfer(new_trans);
       if (file == NULL) {
       g_warning("requested file not shared");
       transfer_destroy(new_trans);
       return;
       }
    */
    if (speed) new_trans->user_info->linespeed = atoi(speed);
    
    //    new_trans->size = file->size;
    upload_show(socket);

    sub = subscription_lookup_file(user, new_trans->longname, &found);
    if (sub && !found) {
      client_message(NULL, _("<%s> has requested a not subscribed file [%s]"), 
		     user, new_trans->shortname);
      found = l_strdup_printf("Automatic Notification: File [%s] was not subscribed by you, this request will be queued", 
			      new_trans->shortname);
      send_private(user, found, 0);
      l_free(found);
      send_private(user, "To view your subscriptions, send me a private message: '!Subscription'", 0);
    }
  } else {
    new_trans = socket->data;
  }

  access_new_request(new_trans);
  user_info_get(new_trans->user_info);
  upload_start(socket, 0);
}

HANDLER(cmd_set_port)
{
  int port;

  port = atoi(data);
  client_message(_("Message"),
		 _("Server has set your data port to %d!"), port);
  create_upload_port(port, FALSE);
}

HANDLER(cmd_channel_list_end)
{
  GtkWidget *temp;

  (void)data;
  temp = lookup_widget(global.win, "channel_list");
  gtk_clist_thaw(GTK_CLIST(temp));
}

HANDLER(cmd_channel_list)
{
  GtkWidget *temp;
  char *pos;

  pos = strchr(data, ' ');
  pos[0] = 0;
  strcpy(tstr[0], data);
  data = pos + 1;
  pos = strchr(data, ' ');
  pos[0] = 0;
  strcpy(tstr[1], data);
  data = pos + 1;

  strcpy(tstr[4], data);

  tstr[2][0] = tstr[3][0] = 0;
  temp = lookup_widget(global.win, "channel_list");
  gtk_clist_append(GTK_CLIST(temp), list);
}

HANDLER(cmd_remote_queued)
{
  GList *dlist;
  GList *result;
  socket_t *socket;
  transfer_t *transfer;
  char *user;
  char *winname;
  char* queue;

  user = arg(data, 0);
  winname = arg(NULL, 0);
  queue = arg(NULL, 0);
  queue = arg(NULL, 0);

  if (!user || !winname)
    return;

  //  printf("remote [%s][%s]\n", user, winname);

  socket = download_search_mapable(user, winname);
  if (!socket)
    return;
  transfer = (transfer_t *) (socket->data);

  if (queue) transfer->queue = atoi(queue);
  // test/ dont autodecrease limit any more.
  //  user_info_change(transfer->user_info, -3, 1);

  if (transfer->status == S_WAITING) {
    socket_end(socket, &(SocketStatus[S_REMOTE]));
  }

  return;

  //////////////////////////////////////////////
  // find all queued transfer from this user
  result = NULL;
  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    if (socket->type != S_TRANSFER)
      continue;
    transfer = (transfer_t *) (socket->data);
    if (!transfer)
      continue;
    if (transfer->type != T_DOWNLOAD)
      continue;
    if (transfer->status != S_QUEUED)
      continue;
    if (strcmp(transfer->user_info->user, user))
      continue;
    result = g_list_append(result, socket);
  }

  // now remote queue the transfers
  for (dlist = result; dlist; dlist = dlist->next) {
    result = g_list_find(global.sockets, dlist->data);
    if (!result)
      continue;			// socket not in downloads any more
    socket = (socket_t *) (result->data);
    transfer = (transfer_t *) (socket->data);
    if (transfer->status != S_QUEUED)
      continue;			//transfer not queued anymore
    socket_end(socket, &(SocketStatus[S_REMOTE]));
  }
}

HANDLER(cmd_motd)
{
  char *mode;
  chat_page_t *page;

  if (SERVER->network == N_UNKNOWN) {
    if (get_version(data) != N_NAPSTER) {
      if (global.usermode) {
	mode = make_string_from_list(global.usermode, " ");
	send_command(CMD_CLIENT_USER_MODE, "NONE");
	send_command(CMD_CLIENT_USER_MODE, mode);
      }
      lopster_links(NULL);
    } else {
      global.limit.max_searches_real = 1;
    }
    update_status_line(0);
    setup_sensitive(0);
    // add timeout that finishes the share after library was completely built
    gtk_timeout_add(5000, lib_share_timeout, NULL);

    global.options.login_mode &= (0xff ^ L_WAIT_MOTD);
  }
  if (!global.links)
    get_server(data);

  page = chat_page_search("MOTD", P_OTHER);
  if (!page)
    page = create_other_page("MOTD", "MOTD");
  if (!page)
    return;
  chat_print_colored(page, M_PUBLIC, "text", data);
  chat_print_colored(page, M_PUBLIC, "text", "\n");
}

HANDLER(cmd_data_port_error)
{
  client_message(_("Message"),
		 _("<%s> reports that your dataport is not reachable!"),
		 data);
}

HANDLER(cmd_operator_message)
{
  char *user;
  char *message;
  chat_page_t *page;
  char *command;

  if (global.options.piping & PIPE_WALLOP) {
    page = chat_page_search("Wallop", P_OTHER);
    if (!page)
      page = create_other_page("Wallop", _("Operator"));
    command = l_strdup_printf("Wallop %s", data);
    cmd_public_message_no_log(command, M_PUBLIC);
    l_free(command);
  } else {
    int old_stamp = global.options.timestamps;
    page = global.current_page;
    chat_print_time_stamp(page, M_PUBLIC);
    chat_print("error", _("(Operator) "));
    global.options.timestamps = TIME_NONE;
    command = l_strdup_printf("%s %s", page->name, data);
    cmd_public_message_no_log(command, M_PUBLIC);
    l_free(command);
    global.options.timestamps = old_stamp;
  }

  user = arg(data, 0);
  message = arg(NULL, 1);
  log("Operator", LOG_OTHER, "<%s> %s\n", user, message);
}

HANDLER(cmd_global_message)
{
  char *user;
  char *message;
  chat_page_t *page;
  char *command;

  if (global.options.piping & PIPE_GLOBAL) {
    page = chat_page_search("Global", P_OTHER);
    if (!page)
      page = create_other_page("Global", _("Global"));
    command = l_strdup_printf("Global %s", data);
    cmd_public_message_no_log(command, M_PUBLIC);
    l_free(command);
  } else {
    int old_stamp = global.options.timestamps;
    page = global.current_page;
    chat_print_time_stamp(page, M_PUBLIC);
    chat_print("error", _("(Global) "));
    global.options.timestamps = TIME_NONE;
    command = l_strdup_printf("%s %s", page->name, data);
    cmd_public_message_no_log(command, M_PUBLIC);
    l_free(command);
    global.options.timestamps = old_stamp;
  }

  user = arg(data, 0);
  message = arg(NULL, 1);
  log("Global", LOG_OTHER, "<%s> %s\n", user, message);
}

HANDLER(cmd_nick_banlist)
{
  GtkWidget *temp;

  if (!global.ban_win)
    return;
  temp = lookup_widget(global.ban_win, "clist3");
  strcpy(tstr[0], data);
  tstr[1][0] = 0;
  tstr[2][0] = 0;
  tstr[3][0] = 0;
  gtk_clist_append(GTK_CLIST(temp), list);
}

HANDLER(cmd_browse_direct)
{
  socket_t *socket;
  char *nick;
  char *ip;
  char *port;

  if (!data)
    return;

  nick = arg(data, 0);
  if (!nick)
    return;
  ip = arg(NULL, 0);
  port = arg(NULL, 0);

  client_message(NULL, _("<%s> is trying to browse your files"), nick);
  if (ip && port) {
    socket = socket_new(S_SHARE);
    //    socket->data = l_strdup(nick);    // this was a bug, wasnt it?
    socket->port = htons(atoi(port));
    socket->ip_long = BSWAP32(strtoul(ip, NULL, 10));
    if (!connect_socket(socket, "TCP", SOCK_STREAM)) {
      socket_destroy(socket, NULL);
      return;
    }
    socket->input =
	gdk_input_add(socket->fd, GDK_INPUT_READ,
		      GTK_SIGNAL_FUNC(await_conn_ack3), socket);
  } else {
    send_command(CMD_SERVER_BROWSE_DIRECT_OK, nick);
  }
}

HANDLER(cmd_browse_direct_ok)
{
  char *nick;
  char *ip;
  char *port;
  socket_t *socket;

  nick = arg(data, 0);
  ip = arg(NULL, 0);
  port = arg(NULL, 0);

  if (!nick || !ip || !port)
    return;

  if (atoi(port) > 0) {
    socket = browse_search(nick);
    if (!socket)
      return;
    socket->port = htons(atoi(port));
    socket->ip_long = BSWAP32(strtoul(ip, NULL, 10));
    browse_connect_and_start(socket);
  }
}

HANDLER(cmd_browse_direct_err)
{
  char *nick;
  char *message;
  socket_t *socket;

  nick = arg(data, 0);
  message = arg(NULL, 0);

  socket = browse_search(nick);
  if (!socket)
    return;

  client_message("Browse", message);
  socket_destroy(socket, NULL);
}

HANDLER(cmd_ghost)
{
  (void)data;
  client_message(_("Message"), _("Someone else tries to login with your Nick"));
}

HANDLER(cmd_sping)
{
  char *server;
  user_timestamp_t *stamp;

  server = arg(data, 0);
  if (!server)
    return;

  stamp = timestamp_search(global.userstamp, server);
  if (!stamp) {
    //    client_message(_("Message"), _("Unrequested pong from %s"), nick);
  } else {
    client_message(_("Message"), _("Pong from server <%s> [%d ms]"),
		   server, timestamp_difference(stamp));
    global.userstamp = g_list_remove(global.userstamp, stamp);
    l_free(stamp->user);
    l_free(stamp);
  }
}

HANDLER(cmd_ping)
{
  char* nick;
  char* add;

  nick = arg(data, 0);
  add = arg(NULL, 1);

  if (add)
    client_message(_("Message"), _("<%s> (%s) has pinged you!"), nick, add);
  else
    client_message(_("Message"), _("<%s> has pinged you!"), nick);
  if (nick) send_command(CMD_CLIENT_PONG, nick);
}

HANDLER(cmd_pong)
{
  char *nick;
  user_timestamp_t *stamp;
  char* add;

  nick = arg(data, 0);
  if (!nick) return;
  add = arg(NULL, 1);

  stamp = timestamp_search(global.userstamp, nick);
  if (!stamp) return;

  if (add)
    client_message(_("Message"), _("Pong from <%s> (%s)[%d ms]"), nick, add,
		   timestamp_difference(stamp));
  else
    client_message(_("Message"), _("Pong from <%s> [%d ms]"), nick,
		   timestamp_difference(stamp));
  global.userstamp = g_list_remove(global.userstamp, stamp);
  l_free(stamp->user);
  l_free(stamp);
}

HANDLER(cmd_emote)
{
  char *channel;
  char *user;
  char *message;
  chat_page_t *page;
  user_timestamp_t *stamp;
  char style[1024];
  int colored = 0;

  channel = arg(data, 0);
  user = arg(NULL, 0);
  message = arg(NULL, 0);	// quoted

  if (!user || !channel || !message) return;

  log(channel, LOG_CHANNEL, "<%s %s>\n", user, message);

  if (global.options.public_ignore &&
      string_list_search(LIST_IGNORE, user))
    return;

  stamp = timestamp_search(global.appearance, user);
  if (!stamp) {
    stamp = timestamp_new(user);
    global.appearance = g_list_append(global.appearance, stamp);
  } else
    timestamp_touch(stamp);

  page = chat_page_search(channel, P_PUBLIC);
  if (!page)
    return;

  chat_print_time_stamp(page, M_PUBLIC);

  chat_print_channel(page, M_PUBLIC, "user", "<");
  
  if (!l_strcasecmp(SERVER->nick, user))
    strcpy(style, "yourself");
  else if (global.options.colored_nicks) {
    strcpy(style, "nick");
    colored = 1;
  } else if (string_list_search(LIST_FRIEND, user))
    strcpy(style, "friend");
  else
    strcpy(style, "user");

  chat_print_channel(page, M_PUBLIC, style, user);
  if (colored) {
    chat_print_channel(page, M_PUBLIC, NULL, " ");
    // critical here
    chat_print_colored(page, M_PUBLIC, NULL, message);
  } else {
    chat_print_channel(page, M_PUBLIC, style, " ");
    chat_print_colored(page, M_PUBLIC, style, message);
  }
  chat_print_channel(page, M_PUBLIC, "user", ">\n");

}

HANDLER(cmd_channel_list_entry)
{
  char *pos;
  GtkWidget *temp;

  pos = strchr(data, ' ');
  pos[0] = 0;
  strcpy(tstr[0], data);
  data = pos + 1;
  pos = strchr(data, ' ');
  pos[0] = 0;
  strcpy(tstr[1], data);
  data = pos + 1;

  data = strchr(data, ' ') + 1;
  pos = strchr(data, ' ');
  pos[0] = 0;
  strcpy(tstr[3], data);
  data = pos + 1;

  pos = strchr(data, ' ');
  pos[0] = 0;
  strcpy(tstr[2], data);
  data = pos + 2;

  pos = strrchr(data, '\"');
  pos[0] = 0;
  strcpy(tstr[4], data);

  temp = lookup_widget(global.win, "channel_list");
  gtk_clist_append(GTK_CLIST(temp), list);
}

HANDLER(cmd_global_user_end)
{
  GtkWidget *temp;

  (void)data;
  if (!global.user_win)
    return;

  temp = lookup_widget(global.user_win, "button164");
  gtk_widget_set_sensitive(temp, TRUE);
}

HANDLER(cmd_global_user)
{
  char *nick;
  char *ip;
  GtkCList *clist;
  clone_t *clone;

  nick = arg(data, 0);
  ip = arg(NULL, 0);

  if ((clone = is_ip_in_clones(global.clones, ip)) != NULL)
  {
    clone->nicks = g_list_prepend(clone->nicks, l_strdup(nick));
  } else {
    clone = (clone_t *) l_malloc(sizeof(clone_t));
    clone->ip = l_strdup(ip);
    clone->nicks = g_list_prepend(NULL, l_strdup(nick));
    global.clones = g_list_prepend(global.clones, clone);
  }

  if (!global.user_win)
    return;

  clist = GTK_CLIST(lookup_widget(global.user_win, "clist12"));
  strcpy(tstr[0], nick);
  strcpy(tstr[1], ip);
  gtk_clist_append(clist, list);
}

HANDLER(cmd_server_links)
{
  static GList *ldata = NULL;
  char *s1, *s2;
  link_t *new_link;
  link_t *new_link2;
  GList *dlist;
  int cnt;
  int who;

  if (strlen(data) > 0) {
    ldata = g_list_append(ldata, l_strdup(data));
    return;
  }

  if (!ldata)
    return;

  if (!global.links) {
    free_links(global.links);
    global.links = NULL;
  }
  global.links = (link_t *) l_malloc(sizeof(link_t));
  global.links->server = l_strdup(get_s1(ldata->data));
  global.links->links = NULL;
  global.links->ping = NULL;

  while (ldata) {
    cnt = 0;
    for (dlist = ldata; dlist; dlist = dlist->next) {
      who = 0;
      data = (char *) (dlist->data);
      s1 = l_strdup(get_s1(data));
      s2 = l_strdup(get_s2(data));
      new_link = link_search_rec(global.links, s1);
      if (new_link) {
	who = 1;
      } else {
	new_link = link_search_rec(global.links, s2);
	if (new_link)
	  who = 2;
      }
      if (who) {
	new_link2 = (link_t *) l_malloc(sizeof(link_t));
	if (who == 1)
	  new_link2->server = l_strdup(s2);
	else
	  new_link2->server = l_strdup(s1);
	new_link2->links = NULL;
	new_link2->ping = NULL;
	new_link->links = g_list_append(new_link->links, new_link2);

	add_allowed_link(new_link->server, new_link2->server);
	add_allowed_link(new_link2->server, new_link->server);

	cnt++;
	ldata = g_list_remove(ldata, data);
	free(data);
	break;
      }
      l_free(s1);
      l_free(s2);
    }
    if (cnt == 0)
      break;
  }
  if (ldata) {
    client_message(_("Error"), _("Could not calculate Link tree"));
    for (dlist = ldata; dlist; dlist = dlist->next) {
      l_free(dlist->data);
    }
    g_list_free(ldata);
    ldata = NULL;
  }
  //  print_links(global.links, 0);
}

HANDLER(cmd_whowas)
{
  char *pos1;
  GtkEntry *entry;
  GtkWidget *temp;
  time_t last_seen;
  char str[200];

  if (global.whois_win == NULL) {
    global.whois_win = create_whois_win();
  }
  gtk_widget_show(global.whois_win);

  gtk_window_set_title(GTK_WINDOW(global.whois_win), _("Whowas User"));

  temp = lookup_widget(global.whois_win, "label116");
  gtk_label_set_text(GTK_LABEL(temp), _("Last Seen"));

  pos1 = arg(data, 0);
  temp = lookup_widget(global.whois_win, "label527");
  gtk_label_set_text(GTK_LABEL(temp), pos1);

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry61"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, ntoa(strtoul(pos1, NULL, 10)));

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry62"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  gtk_entry_set_text(entry, pos1);

  pos1 = arg(NULL, 0);
  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry26"));
  gtk_widget_set_sensitive(GTK_WIDGET(entry), TRUE);
  last_seen = strtoul(pos1, NULL, 10);
  sprintf(str, "%s", ctime(&last_seen));
  gtk_entry_set_text(entry, str);

  entry = GTK_ENTRY(lookup_widget(global.whois_win, "entry34"));
  gtk_entry_set_text(entry, "");

  // --
  temp = lookup_widget(global.whois_win, "entry25");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.whois_win, "entry28");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.whois_win, "entry60");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.whois_win, "entry27");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.whois_win, "entry30");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.whois_win, "entry59");
  gtk_entry_set_text(GTK_ENTRY(temp), "");
  gtk_widget_set_sensitive(temp, FALSE);

  temp = lookup_widget(global.whois_win, "table18");
  gtk_widget_hide(temp);
}

HANDLER(cmd_usage_stats)
{
  char *pos;
  char *pos2;
  time_t stime;
  int t1;
  char str[1024];
  double gigs;

  pos = arg(data, 0);
  client_message(_("Local Clients"), pos);

  pos = arg(NULL, 0);
  client_message(_("Local Servers"), pos);

  pos = arg(NULL, 0);
  client_message(_("Users"), pos);

  pos = arg(NULL, 0);
  pos2 = arg(NULL, 0);
  gigs = strtod(pos2, NULL);
  print_size(str, gigs);
  client_message(_("Files"), "%s (%s)", pos, str);

  pos = arg(NULL, 0);
  client_message(_("Channels"), pos);

  pos = arg(NULL, 0);
  sscanf(pos, "%ld", &stime);
  strcpy(str, ctime(&stime));
  str[strlen(str) - 1] = 0;
  client_message(_("Boot Time"), str);

  pos = arg(NULL, 0);
  t1 = strtoul(pos, NULL, 10);
  sprintf(str, _("%d days %d hours %d minutes %d seconds"),
	  t1 / (60 * 60 * 24),
	  (t1 % (60 * 60 * 24)) / (60 * 60),
	  (t1 % (60 * 60)) / 60, t1 % 60);
  client_message(_("Uptime"), str);

  pos = arg(NULL, 0);
  if (!pos)
    return;
  client_message(_("Memory"), pos);

  pos = arg(NULL, 0);
  if (!pos)
    return;
  client_message(_("Registered Users"), pos);

  pos = arg(NULL, 0);
  if (!pos)
    return;
  client_message(_("KBytes In/sec"), pos);

  pos = arg(NULL, 0);
  if (!pos)
    return;
  client_message(_("KBytes Out/sec"), pos);

  pos = arg(NULL, 0);
  if (!pos)
    return;
  client_message(_("Searches/sec"), pos);

  pos = arg(NULL, 0);
  if (!pos)
    return;
  gigs = strtod(pos, NULL);
  print_size(str, gigs);
  client_message(_("Traffic In"), str);

  pos = arg(NULL, 0);
  if (!pos)
    return;
  gigs = strtod(pos, NULL);
  print_size(str, gigs);
  client_message(_("Traffic Out"), str);

  pos = arg(NULL, 0);
  if (!pos)
    return;
  client_message(_("Pending Searches"), pos);
}

HANDLER(cmd_version_stats)
{
  GtkCList *clist;
  GdkColor color = { 0, 0x9900, 0x0f00, 0x0f00 };
  GtkWidget *temp;
  GList *dlist;
  char *info;
  char *number;
  client_t *client;
  client_t *client2;
  char *short_info;
  int flag;
  int threshold;
  int row;

  if (!global.client_win)
    return;

  clist = GTK_CLIST(lookup_widget(global.client_win, "clist8"));
  if (strlen(data) > 0) {
    info = arg(data, 0);
    number = arg(NULL, 0);
    client = (client_t *) l_malloc(sizeof(client_t));
    client->cnt = 0;
    client->info = l_strdup(info);
    if (number)
      client->logins = atoi(number);
    else
      client->logins = 0;

    global.client_list = g_list_append(global.client_list, client);

    short_info = strtok(info, " -");
    if (!short_info)
      short_info = info;
    client2 = is_client_in_list(global.client_list, short_info);
    if (!client2) {
      client2 = (client_t *) l_malloc(sizeof(client_t));
      client2->cnt = 1;
      client2->info = l_strdup(short_info);
      client2->logins = client->logins;
      global.client_list = g_list_append(global.client_list, client2);
    } else {
      client2->cnt++;
      client2->logins += client->logins;
    }
    return;
  }

  temp = lookup_widget(global.client_win, "checkbutton30");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp))) {
    temp = lookup_widget(global.client_win, "spinbutton19");
    threshold = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(temp));
  } else {
    threshold = 0;
  }

  flag = 0;
  temp = lookup_widget(global.client_win, "checkbutton32");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp)))
    flag |= 1;			// single
  temp = lookup_widget(global.client_win, "checkbutton31");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp)))
    flag |= 2;			// sum

  gtk_clist_freeze(clist);
  dlist = g_list_first(global.client_list);
  while (dlist) {
    client = (client_t *) (dlist->data);
    if (client->logins >= threshold) {
      if ((client->cnt == 0) && (flag & 1)) {
	strcpy(tstr[1], client->info);
	sprintf(tstr[0], "%d", client->logins);
	row = gtk_clist_append(clist, list);
      }
      if ((client->cnt > 0) && (flag & 2)) {
	sprintf(tstr[1], "%s%s", client->info,
		(flag & 1) ? _(" (SUM)") : "");
	sprintf(tstr[0], "%d", client->logins);
	row = gtk_clist_append(clist, list);
	if (flag & 1) {
	  gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(clist)),
			  &color);
	  gtk_clist_set_foreground(clist, row, &color);
	}
      }
    }
    dlist = dlist->next;
  }
  gtk_clist_thaw(clist);
}

HANDLER(cmd_user_mode)
{
  GList *dlist;
  char *mode;

  client_message(_("Usermode"), data);
  for (dlist = global.usermode; dlist; dlist = dlist->next) {
    l_free(dlist->data);
  }
  if (global.usermode)
    g_list_free(global.usermode);
  global.usermode = NULL;

  mode = arg(data, 0);
  while (mode) {
    global.usermode = g_list_append(global.usermode, l_strdup(mode));
    mode = arg(NULL, 0);
  }
}

// deprecated
HANDLER(cmd_browse_new)
{
  file_create_from_browse_new(data);
}

HANDLER(cmd_redirect)
{
  client_message(_("Ignoring redirect"), "[%s]", data);
}

HANDLER(cmd_cycle)
{
  client_message(_("Ignoring cycle"), "[%s]", data);
}

HANDLER(cmd_histogram)
{
  int id;
  unsigned long count;
  double bytes;
  char *arg1;
  char str[1024];

  arg1 = arg(data, 0);
  if (!arg1)
    return;
  id = atoi(arg1);
  arg1 = arg(NULL, 0);
  if (!arg1)
    return;
  count = strtoul(arg1, NULL, 10);
  arg1 = arg(NULL, 0);
  if (!arg1)
    return;
  bytes = strtod(arg1, NULL);

  if (count > 0) {
    arg1 = l_strdup_printf("%5d", id);
    print_size(str, bytes);
    client_message(arg1, "%10lu %10s", count, str);
    l_free(arg1);
  }
}

HANDLER(cmd_histogram_end)
{
  int id;
  unsigned long count;
  double bytes;
  char *arg1;
  char str[1024];

  arg1 = arg(data, 0);
  if (!arg1)
    return;
  id = atoi(arg1);
  arg1 = arg(NULL, 0);
  if (!arg1)
    return;
  count = strtoul(arg1, NULL, 10);
  arg1 = arg(NULL, 0);
  if (!arg1)
    return;
  bytes = strtod(arg1, NULL);

  print_size(str, bytes);
  arg1 = l_strdup_printf("%5d", id);
  client_message(arg1, "%10lu %10s", count, str);
  l_free(arg1);

  client_message("-ID--", "----calls- ----bytes-");

  arg1 = arg(NULL, 0);
  if (!arg1)
    return;
  count = strtoul(arg1, NULL, 10);
  arg1 = arg(NULL, 0);
  if (!arg1)
    return;
  bytes = strtod(arg1, NULL);
  print_size(str, bytes);
  client_message("-SUM-", "%10lu %10s", count, str);
}

HANDLER(cmd_acl_list)
{
  char *ip;
  char *count;
  char *pos;
  char str[1024];

  ip = arg(data, 0);
  count = arg(NULL, 0);
  if (!ip) {
    client_message("End of List", "");
    return;
  }
  if (!count)
    return;
  pos = strchr(ip, '/');
  sprintf(str, "%2s", count);
  if (pos) {
    *pos = 0;
    pos++;
    client_message(str, "%15s %s", ip, pos);
  } else {
    client_message(str, "%15s", ip);
  }
}
