/* Copyright (C) 2000/2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>
#include <strings.h>
#include <string.h>
#include <stdio.h>
#include <sys/time.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "lopster.h"
#include "callbacks.h"
#include "connection.h"
#include "global.h"
#include "search.h"
#include "share.h"
#include "browse.h"
#include "handler.h"
#include "support.h"
#include "transfer.h"
#include "resume.h"
#include "string_list.h"
#include "whois.h"
#include "ping.h"

typedef int (*IntFunc) (char *str);

#define WORD_CHAR(c) \
        (isalnum((unsigned char)(c))||(c)=='\''||(unsigned char)(c) > 128)

search_pattern_t *search_pattern_create();
void search_pattern_destroy(search_pattern_t * search);
void search_pattern_update(search_pattern_t * search);

char *Destination(int id)
{
  switch (id) {
  case DEST_NAPSTER:
    return _("Napster");
  case DEST_HOTLIST:
    return _("Hotlist");
  case DEST_LIBRARY:
    return _("Library");
  case DEST_SEARCH:
    return _("Results");
  default:
    return _("Unknown");
  }
  return "??";
}

char *get_real_mediatype(char *trans)
{
  if (!l_strcasecmp(trans, _("mp3")))
    return l_strdup("mp3");
  if (!l_strcasecmp(trans, _("audio")))
    return l_strdup("audio");
  if (!l_strcasecmp(trans, _("video")))
    return l_strdup("video");
  if (!l_strcasecmp(trans, _("application")))
    return l_strdup("application");
  if (!l_strcasecmp(trans, _("image")))
    return l_strdup("image");
  if (!l_strcasecmp(trans, _("text")))
    return l_strdup("text");
  if (!l_strcasecmp(trans, _("any")))
    return l_strdup("any");
  return l_strdup("mp3");
}

char *get_real_speed(char *trans)
{
  if (!l_strcasecmp(trans, _("not specified")))
    return NULL;
  if (!l_strcasecmp(trans, _("unknown")))
    return l_strdup("unknown");
  else
    return l_strdup(trans);
}

char *get_real_entry(char *trans)
{
  if (!l_strcasecmp(trans, _("not specified")))
    return NULL;
  else
    return l_strdup(trans);
}

char *Destination_long(int id)
{
  switch (id) {
  case DEST_NAPSTER:
    return _("Search on Napster");
  case DEST_HOTLIST:
    return _("Search in Hotlist");
  case DEST_LIBRARY:
    return _("Search in Library");
  case DEST_SEARCH:
    return _("Search in Results");
  default:
    return _("Unknown");
  }
  return "??";
}

int life2int(char *dest)
{
  if (!l_strcasecmp(dest, _("Ignore")))
    return PATTERN_IGNORE;
  else if (!l_strcasecmp(dest, _("Save")))
    return PATTERN_SAVE;
  else if (!l_strcasecmp(dest, _("Save+Search on relogin")))
    return PATTERN_SEARCH;
  else
    return PATTERN_TEMPORARY;
}

char* LifeTime(int id) {
  switch (id) {
  case PATTERN_IGNORE:
    return _("Ignore");
  case PATTERN_SAVE:
    return _("Save");
  case PATTERN_SEARCH:
    return _("Save+Search on relogin");
  case PATTERN_TEMPORARY:
  default:
    return _("Temporary");
  }
  return "??";
}

int destination2int(char *dest)
{
  if (!strcmp(_("Search on Napster"), dest))
    return DEST_NAPSTER;
  else if (!strcmp(_("Search in Library"), dest))
    return DEST_LIBRARY;
  else if (!strcmp(_("Search in Hotlist"), dest))
    return DEST_HOTLIST;
  else if (!strcmp(_("Search in Results"), dest))
    return DEST_SEARCH;
  else
    return DEST_NAPSTER;
}

char *search_arg(char *data)
{
  static char *string;
  char *pos;

  if (data)
    string = data;
  if (!string)
    return NULL;

  while (*string && !WORD_CHAR(*string))
    string++;

  pos = string;
  while (WORD_CHAR(*string)) {
    string++;
  }
  if (pos == string) {
    string = NULL;
    return NULL;
  } else {
    if (*string != 0)
      *string++ = 0;
    return pos;
  }
}

void search_create_page(search_t * search)
{
  GtkWidget *scrolledwindow;
  GtkWidget *search_list;
  GtkWidget *label;
  search_pattern_t *pattern;
  GtkWidget *tab_label;
  char *str;
  GtkNotebook *notebook;

  pattern = search->pattern;
  if (!pattern) {
    printf("no pattern in search\n");
    return;
  }

  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scrolledwindow);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);

  search->link = (void *) scrolledwindow;

  search_list = gtk_clist_new(8);
  gtk_widget_show(search_list);
  gtk_widget_set_events(search_list,
			GDK_POINTER_MOTION_MASK | GDK_LEAVE_NOTIFY_MASK);

  gtk_clist_set_sort_column(GTK_CLIST(search_list), 0);
  //  gtk_clist_set_auto_sort(GTK_CLIST(search_list), TRUE);
  gtk_clist_set_compare_func(GTK_CLIST(search_list), search_compare);

  gtk_container_add(GTK_CONTAINER(scrolledwindow), search_list);
  gtk_clist_set_column_width(GTK_CLIST(search_list), 0,
			     global.search_width[0]);
  gtk_clist_set_column_visibility(GTK_CLIST(search_list), 0,
				  global.search_show[0]);
  gtk_clist_set_column_width(GTK_CLIST(search_list), 1,
			     global.search_width[1]);
  gtk_clist_set_column_visibility(GTK_CLIST(search_list), 1,
				  global.search_show[1]);
  gtk_clist_set_column_width(GTK_CLIST(search_list), 2,
			     global.search_width[2]);
  gtk_clist_set_column_visibility(GTK_CLIST(search_list), 2,
				  global.search_show[2]);
  gtk_clist_set_column_width(GTK_CLIST(search_list), 3,
			     global.search_width[3]);
  gtk_clist_set_column_visibility(GTK_CLIST(search_list), 3,
				  global.search_show[3]);
  gtk_clist_set_column_width(GTK_CLIST(search_list), 4,
			     global.search_width[4]);
  gtk_clist_set_column_visibility(GTK_CLIST(search_list), 4,
				  global.search_show[4]);
  gtk_clist_set_column_width(GTK_CLIST(search_list), 5,
			     global.search_width[5]);
  gtk_clist_set_column_visibility(GTK_CLIST(search_list), 5,
				  global.search_show[5]);
  gtk_clist_set_selection_mode(GTK_CLIST(search_list),
			       GTK_SELECTION_EXTENDED);
  gtk_clist_column_titles_show(GTK_CLIST(search_list));

  label = gtk_label_new(_("Filename"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(search_list), 0, label);

  label = gtk_label_new(_("Size"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(search_list), 1, label);

  label = gtk_label_new(_("Bitrate"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(search_list), 2, label);

  label = gtk_label_new(_("Length"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(search_list), 3, label);

  label = gtk_label_new(_("User"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(search_list), 4, label);

  label = gtk_label_new(_("IP"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(search_list), 5, label);

  label = gtk_label_new(_("Line Speed"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(search_list), 6, label);

  label = gtk_label_new(_("Ping"));
  gtk_widget_show(label);
  gtk_clist_set_column_widget(GTK_CLIST(search_list), 7, label);

  gtk_signal_connect(GTK_OBJECT(search_list), "click_column",
		     GTK_SIGNAL_FUNC(on_list_click_column), NULL);
  gtk_signal_connect(GTK_OBJECT(search_list), "button_press_event",
		     GTK_SIGNAL_FUNC(on_search_list_button_press_event),
		     NULL);
  /*
  gtk_signal_connect(GTK_OBJECT(search_list), "select_row",
		     GTK_SIGNAL_FUNC(on_search_select_row), NULL);
  */
  gtk_signal_connect(GTK_OBJECT(search_list), "motion_notify_event",
		     GTK_SIGNAL_FUNC(on_search_motion_notify_event), NULL);
  gtk_signal_connect(GTK_OBJECT(search_list), "leave_notify_event",
		     GTK_SIGNAL_FUNC(on_list_leave_notify_event), NULL);

  str = l_strdup_printf("%s: %s\n%s",
			Destination(pattern->destination),
			_(pattern->media_type), pattern->include);
  tab_label = gtk_label_new(str);
  l_free(str);

  gtk_object_set_data(GTK_OBJECT(search_list), "search", search);
  gtk_object_set_data(GTK_OBJECT(scrolledwindow), "clist", search_list);

  notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook5"));
  gtk_notebook_append_page(notebook, scrolledwindow, tab_label);
  gtk_notebook_set_page(notebook,
			gtk_notebook_page_num(GTK_NOTEBOOK(notebook),
					      scrolledwindow));
}

GtkWidget *create_search_list_popup(file_t * file)
{
  GtkWidget *popup;
  GtkWidget *user_popup;
  GtkWidget *item;
  GtkAccelGroup *popup_accels;
  GtkWidget *download;
  GtkWidget *download_selected;
  GtkWidget *clear_selected;
  GtkWidget *clear_users_files;
  GtkWidget *clear_all;
  GtkWidget *trennlinie13;
  GtkWidget *trennlinie17;
  GtkWidget *customize_list3;
  GtkCList *clist;
  char item_str[1024];
  int item_num;
  GList* resume_list;
  GList* dlist;
  resume_t* resume;

  popup = gtk_menu_new();
  popup_accels = gtk_menu_ensure_uline_accel_group(GTK_MENU(popup));
  clist = GTK_CLIST(global.popup_list);
  item_num = g_list_length(clist->selection);

  if (file) {
    if (file->local) {
      download = gtk_menu_item_new_with_label(_("Open file"));
      gtk_widget_show(download);
      gtk_container_add(GTK_CONTAINER(popup), download);
      gtk_signal_connect(GTK_OBJECT(download), "activate",
			 GTK_SIGNAL_FUNC(on_play_file3_activate), NULL);
    } else if (item_num < 2) {
      download = gtk_menu_item_new_with_label(_("Download"));
      gtk_widget_show(download);
      gtk_container_add(GTK_CONTAINER(popup), download);
      gtk_signal_connect(GTK_OBJECT(download), "activate",
			 GTK_SIGNAL_FUNC(on_download_activate),
			 (void *) 0);
      download =
	  gtk_menu_item_new_with_label(_("Download with Foldername"));
      gtk_widget_show(download);
      gtk_container_add(GTK_CONTAINER(popup), download);
      gtk_signal_connect(GTK_OBJECT(download), "activate",
			 GTK_SIGNAL_FUNC(on_download_activate),
			 (void *) 1);
    } else {
      sprintf(item_str, _("Download Selected (%d)"), item_num);
      download_selected = gtk_menu_item_new_with_label(item_str);
      gtk_widget_show(download_selected);
      gtk_container_add(GTK_CONTAINER(popup), download_selected);
      gtk_signal_connect(GTK_OBJECT(download_selected), "activate",
			 GTK_SIGNAL_FUNC(on_download_selected_activate),
			 (void *) 0);
      sprintf(item_str, _("Download Selected (%d) with Foldername"),
	      item_num);
      download_selected = gtk_menu_item_new_with_label(item_str);
      gtk_widget_show(download_selected);
      gtk_container_add(GTK_CONTAINER(popup), download_selected);
      gtk_signal_connect(GTK_OBJECT(download_selected), "activate",
			 GTK_SIGNAL_FUNC(on_download_selected_activate),
			 (void *) 1);
    }
    if ((resume_list = resume_search_size(file->size)) != NULL) {
      item = gtk_menu_item_new_with_label(_("Resume File"));
      gtk_widget_show(item);
      gtk_container_add(GTK_CONTAINER(popup), item);
      
      user_popup = gtk_menu_new();
      gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), user_popup);
      for (dlist = resume_list; dlist; dlist = dlist->next) {
	resume = dlist->data;
	item = gtk_menu_item_new_with_label(extract_filename(resume->filename));
	gtk_widget_show(item);
	gtk_container_add(GTK_CONTAINER(user_popup), item);
	gtk_signal_connect(GTK_OBJECT(item), "activate",
			   GTK_SIGNAL_FUNC(on_download_resume_activate),
			   (gpointer)resume);
      }
      g_list_free(resume_list);
    }

    trennlinie17 = gtk_menu_item_new();
    gtk_widget_show(trennlinie17);
    gtk_container_add(GTK_CONTAINER(popup), trennlinie17);
    gtk_widget_set_sensitive(trennlinie17, FALSE);

    sprintf(item_str, _("Remove Selected (%d)"), item_num);
    clear_selected = gtk_menu_item_new_with_label(item_str);
    gtk_widget_show(clear_selected);
    gtk_container_add(GTK_CONTAINER(popup), clear_selected);
    gtk_signal_connect(GTK_OBJECT(clear_selected), "activate",
		       GTK_SIGNAL_FUNC(on_clear_selected_activate), NULL);

    clear_users_files =
	gtk_menu_item_new_with_label(_("Remove Users Files"));
    gtk_widget_show(clear_users_files);
    gtk_container_add(GTK_CONTAINER(popup), clear_users_files);
    gtk_signal_connect(GTK_OBJECT(clear_users_files), "activate",
		       GTK_SIGNAL_FUNC(on_clear_users_files_activate),
		       NULL);
    if (file->local)
      gtk_widget_set_sensitive(clear_users_files, FALSE);
  }
  clear_all = gtk_menu_item_new_with_label(_("Remove Search"));
  gtk_widget_show(clear_all);
  gtk_container_add(GTK_CONTAINER(popup), clear_all);
  gtk_signal_connect(GTK_OBJECT(clear_all), "activate",
		     GTK_SIGNAL_FUNC(on_clear_all_activate), NULL);
  clear_all = gtk_menu_item_new_with_label(_("Remove All Searches"));
  gtk_widget_show(clear_all);
  gtk_container_add(GTK_CONTAINER(popup), clear_all);
  gtk_signal_connect(GTK_OBJECT(clear_all), "activate",
		     GTK_SIGNAL_FUNC(on_clear_all_searches_activate), NULL);

  trennlinie13 = gtk_menu_item_new();
  gtk_widget_show(trennlinie13);
  gtk_container_add(GTK_CONTAINER(popup), trennlinie13);
  gtk_widget_set_sensitive(trennlinie13, FALSE);

  clear_all = gtk_menu_item_new_with_label(_("Search again"));
  gtk_widget_show(clear_all);
  gtk_container_add(GTK_CONTAINER(popup), clear_all);
  gtk_signal_connect(GTK_OBJECT(clear_all), "activate",
		     GTK_SIGNAL_FUNC(on_search_again_activate), NULL);

  clear_all = gtk_menu_item_new_with_label(_("Search all again"));
  gtk_widget_show(clear_all);
  gtk_container_add(GTK_CONTAINER(popup), clear_all);
  gtk_signal_connect(GTK_OBJECT(clear_all), "activate",
		     GTK_SIGNAL_FUNC(on_search_all_again_activate), NULL);

  trennlinie17 = gtk_menu_item_new();
  gtk_widget_show(trennlinie17);
  gtk_container_add(GTK_CONTAINER(popup), trennlinie17);
  gtk_widget_set_sensitive(trennlinie17, FALSE);

  if (file && !file->local) {
    item = gtk_menu_item_new_with_label(_("User Menu"));
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);

    user_popup = create_user_popup(M_SEARCH);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), user_popup);

    trennlinie17 = gtk_menu_item_new();
    gtk_widget_show(trennlinie17);
    gtk_container_add(GTK_CONTAINER(popup), trennlinie17);
    gtk_widget_set_sensitive(trennlinie17, FALSE);
  }

  customize_list3 = gtk_menu_item_new_with_label(_("Customize List"));
  gtk_widget_show(customize_list3);
  gtk_container_add(GTK_CONTAINER(popup), customize_list3);
  gtk_signal_connect(GTK_OBJECT(customize_list3), "activate",
		     GTK_SIGNAL_FUNC(on_customize_list_activate), NULL);

  return popup;
}

gint search_compare(GtkCList * clist,
		    gconstpointer ptr1, gconstpointer ptr2)
{
  file_t *file1;
  file_t *file2;

  GtkCListRow *row1 = (GtkCListRow *) ptr1;
  GtkCListRow *row2 = (GtkCListRow *) ptr2;

  file1 = row1->data;
  file2 = row2->data;

  if (!file2)
    return (file1 != NULL);
  if (!file1)
    return -1;

  if (clist->sort_column == 0) {
    return l_strcasecmp(file1->shortname, file2->shortname);
  } else if (clist->sort_column == 1) {
    if (file1->size < file2->size)
      return -1;
    if (file1->size > file2->size)
      return 1;
    return 0;
  } else if (clist->sort_column == 2) {
    if (file1->bitrate < file2->bitrate)
      return -1;
    if (file1->bitrate > file2->bitrate)
      return 1;
    return 0;
  } else if (clist->sort_column == 3) {
    if (file1->duration < file2->duration)
      return -1;
    if (file1->duration > file2->duration)
      return 1;
    return 0;
  } else if (clist->sort_column == 4) {
    return l_strcasecmp(file1->user, file2->user);
  } else if (clist->sort_column == 5) {
    unsigned long ip1 = ntohl(file1->ip);
    unsigned long ip2 = ntohl(file2->ip);
    if (ip1 < ip2) return -1;
    if (ip1 > ip2) return +1;
    return 0;
  } else if (clist->sort_column == 6) {
    if (file1->linespeed < file2->linespeed)
      return -1;
    if (file1->linespeed > file2->linespeed)
      return +1;
    return 0;
  } else if (clist->sort_column == 7) {
    if (!file1->ping)
      return (file2->ping != 0);
    if (!file2->ping)
      return -1;
    if (file1->ping < file2->ping)
      return -1;
    if (file1->ping > file2->ping)
      return 1;
    return 0;
  } else {
    return 0;
  }
}

file_t *file_create_from_search_response(char *data)
{
  file_t *new_file;
  char *pos1;
  char* name;
  char* md5;
  char* size;
  char* rate;
  char* freq;
  char* dur;
  char* user;
  char* ip;
  char* speed;

  name = arg(data, 0);
  md5 = arg(NULL, 0);
  size = arg(NULL, 0);
  rate = arg(NULL, 0);
  freq = arg(NULL, 0);
  dur = arg(NULL, 0);
  user = arg(NULL, 0);
  ip = arg(NULL, 0);
  speed = arg(NULL, 0);

  if (!name || !md5 || !size || !rate || !freq ||
      !dur || !user || !ip || !speed) return NULL;

  new_file = (file_t *) l_malloc(sizeof(file_t));

  new_file->winname = l_strdup(name);
  new_file->md5 = l_strdup(name);
  new_file->size = strtoul(size, NULL, 10);
  new_file->bitrate = atoi(rate);
  new_file->frequency = atoi(freq);
  new_file->duration = atoi(dur);
  new_file->user = l_strdup(user);
  new_file->ip = BSWAP32(strtoul(ip, NULL, 10));
  new_file->linespeed = atoi(speed);
  
  new_file->longname = l_strdup(new_file->winname);
  convert_to_unix(new_file->longname);
  pos1 = extract_short_name(new_file->longname);
  new_file->shortname = l_strdup(pos1);
  new_file->mime_type = get_mimetype(new_file->shortname);

  new_file->flags = 0;
  new_file->local = 0;		//remote file
  new_file->ping = 0;

  return new_file;
}

void append_field(search_pattern_t * pattern, char *data, int field)
{
  int flag = 0;
  char *s1, *s2;
  char t[1024];
  IntFunc func = NULL;
  char str[1024];

  s1 = s2 = NULL;
  switch (field) {
  case 0:
    s1 = pattern->speed_lo;
    s2 = pattern->speed_hi;
    strcpy(t, "LINESPEED");
    func = (IntFunc) (speed2int);
    break;
  case 1:
    s1 = pattern->bitrate_lo;
    s2 = pattern->bitrate_hi;
    strcpy(t, "BITRATE");
    func = (IntFunc) (atoi);
    break;
  case 2:
    s1 = pattern->freq_lo;
    s2 = pattern->freq_hi;
    strcpy(t, "FREQ");
    func = (IntFunc) (atoi);
    break;
  }
  if (s1 && s2 && !strcmp(s1, s2)) {
    flag = 4;
  } else if (s2) {
    if (opennap_version(0, 30) && s1)
      flag = 3;
    else
      flag = 2;
  } else if (s1) {
    flag = 1;
  }
  if (flag & 1) {
    sprintf(str, " %s \"AT LEAST\" %d", t, func(s1));
    strcat(data, str);
  }
  if (flag & 2) {
    sprintf(str, " %s \"AT BEST\" %d", t, func(s2));
    strcat(data, str);
  }
  if (flag & 4) {
    sprintf(str, " %s \"EQUAL TO\" %d", t, func(s1));
    strcat(data, str);
  }
}

void append_field2(search_pattern_t * pattern, char *data, int field)
{
  long s1, s2;
  char t[1024];
  char str[1024];

  s1 = s2 = 0;
  switch (field) {
  case 0:
    s1 = pattern->duration_lo;
    s2 = pattern->duration_hi;
    strcpy(t, "DURATION");
    break;
  case 1:
    s1 = pattern->size_lo;
    s2 = pattern->size_hi;
    strcpy(t, "SIZE");
    break;
  }

  if ((s1 == s2) && (s1 > 0)) {
    sprintf(str, " %s \"EQUAL TO\" \"%ld\"", t, s1);
    strcat(data, str);
  } else {
    if (s1 > 0) {
      sprintf(str, " %s \"AT LEAST\" \"%ld\"", t, s1);
      strcat(data, str);
    }
    if (s2 > 0) {
      sprintf(str, " %s \"AT BEST\" \"%ld\"", t, s2);
      strcat(data, str);
    }
  }
}

void napster_search(search_t * search)
{
  char data[2048];
  char str[1024];
  int opennap;
  char *excl;
  search_pattern_t* pattern;

  if (!search->pattern) return;
  pattern = search->pattern;

  if (global.status.connection < 2) {
    g_warning("not connected!");
    return;
  }
  if (strlen(search->pattern->include) == 0) {
    search_finish(search);
    return;
  }

  //  setup_sensitive(-1);

  if (opennap_version(0, 30)) opennap = 1;
  else opennap = 0;

  *data = 0;
  if (opennap) {
    sprintf(str, "FILENAME CONTAINS \"%s\"", pattern->include);
    strcat(data, str);
    if (strlen(pattern->exclude) > 0) {
      strcat(data, str);
      sprintf(str, " FILENAME EXCLUDES \"%s\"", pattern->exclude);
    }
  } else {
    sprintf(str, "FILENAME CONTAINS \"%s", pattern->include);
    strcat(data, str);
    strcpy(str, pattern->exclude);
    excl = arg(str, 0);
    while (excl) {
      strcat(data, " -");
      strcat(data, excl);
      excl = arg(NULL, 0);
    }
    strcat(data, "\"");
  }

  sprintf(str, " MAX_RESULTS %d", pattern->max_results);
  strcat(data, str);

  append_field(pattern, data, 0);
  append_field(pattern, data, 1);
  append_field(pattern, data, 2);

  if (opennap) {
    append_field2(pattern, data, 0);
    append_field2(pattern, data, 1);
    if (pattern->media_type && strcmp(pattern->media_type, "mp3")) {
      sprintf(str, " TYPE %s", pattern->media_type);
      strcat(data, str);
    }
    if (search->local) strcat(data, " LOCAL_ONLY");
  }
  send_command_unimportant(CMD_CLIENT_SEARCH, data);
}

void search_pattern_insert(search_pattern_t* pattern, int save) {
  GtkCList* clist;
  int row;
  
  if (pattern->level < PATTERN_TEMPORARY) {
    search_pattern_destroy(pattern);
    return;
  }
  clist = GTK_CLIST(lookup_widget(global.win, "clist34"));
  global.search_pattern = g_list_prepend(global.search_pattern, pattern);
  strcpy(tstr[0], pattern->name);
  sprintf(tstr[1], "%s: %s",
	  Destination(pattern->destination),
	  _(pattern->media_type));
  strcpy(tstr[2], LifeTime(pattern->level));

  row = gtk_clist_append(clist, list);
  gtk_clist_set_row_data(clist, row, (gpointer) pattern);
  if (save && (pattern->level >= PATTERN_SAVE)) search_pattern_save();
}

int search_pattern_cmp(search_pattern_t* p1, search_pattern_t* p2) {
  if (l_strcasecmp(p1->include, p2->include)) return 1;
  if (l_strcasecmp(p1->exclude, p2->exclude)) return 1;
  if (p1->max_results != p2->max_results) return 1;
  if (l_strcasecmp(p1->media_type, p2->media_type)) return 1;
  if (p1->destination != p2->destination) return 1;
  if (p1->bitrate_lo != p2->bitrate_lo) return 1;
  if (p1->bitrate_hi != p2->bitrate_hi) return 1;
  if (p1->freq_lo != p2->freq_lo) return 1;
  if (p1->freq_hi != p2->freq_hi) return 1;
  if (p1->speed_lo != p2->speed_lo) return 1;
  if (p1->speed_hi != p2->speed_hi) return 1;
  if (p1->size_lo != p2->size_lo) return 1;
  if (p1->size_hi != p2->size_hi) return 1;
  if (p1->duration_lo != p2->duration_lo) return 1;
  if (p1->duration_hi != p2->duration_hi) return 1;
  return 0;
}

search_pattern_t* search_pattern_find_same(search_pattern_t* pattern) {
  search_pattern_t* result;
  search_t* search;
  GList* dlist;

  for (dlist = global.search_pattern; dlist; dlist = dlist->next) {
    result = dlist->data;
    if (!search_pattern_cmp(pattern, result))
      return result;
  }
  for (dlist = global.searches; dlist; dlist = dlist->next) {
    search = dlist->data;
    if (search->resume) continue;
    if (!search_pattern_cmp(pattern, search->pattern))
      return search->pattern;
  }
  return NULL;
}

void send_search_request(gboolean local)
{
  search_pattern_t *pattern;
  search_pattern_t *pattern2;
  search_t *search;

  pattern = search_pattern_create();
  pattern2 = search_pattern_find_same(pattern);
  if (pattern2) {
    search_pattern_destroy(pattern);
    search_pattern_search(pattern2);
    return;
  }
  search_pattern_insert(search_pattern_copy(pattern), 1);

  //////
  search = search_new();
  search->pattern = pattern;
  search->local = local;
  search_create_page(search);
  /////

  search_queue(search);
}

void send_search_again(search_t* search) {
  if (!search) return;

  search_clear(search);
  search_queue(search);
}

void search_insert_file(search_t* search, file_t * file) {
  file = file_dup(file);
  search->results = g_list_append(search->results, file);
  if (search->resume) {
    // creating a transfer for the result
    resume_insert_file((resume_t*)(search->link), file);
  }
}

void search_insert_file_real(GtkCList* clist, file_t * file)
{
  int row;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  GtkWidget *temp;
  ping_t *p;

  strcpy(tstr[0], file->shortname);
  sprintf(tstr[1], "%.2f MB", (double) (file->size) / 1024 / 1024);
  sprintf(tstr[2], "%d", file->bitrate);
  print_time_short(tstr[3], file->duration);
  strcpy(tstr[4], file->user);
  if (file->local) strcpy(tstr[5], "localhost");
  else sprintf(tstr[5], "%s", ntoa(file->ip));
  
  if (file->local) strcpy(tstr[6], _("Local File"));
  else strcpy(tstr[6], LineSpeed(file->linespeed));

  temp = lookup_widget(global.win, "checkbutton45");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp))) {
    if ((p = ping_search(file->ip))) {
      file->ping = p->time;
      if (p->pid >= 0)
	sprintf(tstr[7], "eval..");
      else if (file->ping)
	sprintf(tstr[7], "%d ms", file->ping);
      else
	sprintf(tstr[7], "n/a");
    } else {
      ping(file->ip);
      strcpy(tstr[7], "eval..");
    }
  } else {
    strcpy(tstr[7], "");
  }

  row = gtk_clist_append(clist, &list[0]);
  gtk_clist_set_row_data(clist, row, (gpointer) file);

  detect_line_pixs(file->linespeed, &pixmap, &bitmap);

  gtk_clist_set_pixtext(clist, row, 0, tstr[0], 5, pixmap, bitmap);
}


int file_cmp(file_t * file1, file_t * file2)
{
  if (file1->size != file2->size)
    return 1;
  if (strcmp(file1->winname, file2->winname))
    return 2;
  if (l_strcasecmp(file1->user, file2->user))
    return 3;
  //  if (file1->bitrate != file2->bitrate) return 4;
  //  if (file1->frequency != file2->frequency) return 5;
  //  if (file1->duration != file2->duration) return 6;

  return 0;
}

file_t *search_find_file(search_t * search, file_t * file)
{
  GList *dlist;
  file_t *file2;

  if (!search)
    return NULL;
  if (!file)
    return NULL;

  for (dlist = search->results; dlist; dlist = dlist->next) {
    file2 = (file_t *) (dlist->data);
    if (!file_cmp(file, file2))
      return file2;
  }

  return NULL;
}

// destination should always be DEST_NAPSTER, as the other search routines
// do not use this function
// inserts a file in all possible searches
void file_insert_search(file_t * file, int destination)
{
  search_t *search;
  GList *dlist;
  int cnt = 0;
  resume_t *resume;

  for (dlist = global.searches; dlist; dlist = dlist->next) {
    search = dlist->data;
    // only insert in running searches
    if (search->status != SEARCH_ACTIVE) continue;
    // widget must be present
    if (!search->link) continue;
    if (search->pattern->destination != destination)
      continue;

    if (search->resume) {
      resume = search->link;
      if ((resume->size == file->size) &&
	  ((resume->local_size > 100) || //only check filename if local size is low, rest is down by resume checker
	   search_pattern_fits_file(search->pattern, file, 0))) {
	cnt++;
	if (!search_find_file(search, file)) {
	  search_insert_file(search, file);
	}
      }
    } else {
      if (search_pattern_fits_file(search->pattern, file, 0)) {
	cnt++;
	if (!search_find_file(search, file)) {
	  search_insert_file(search, file);
	}
      }
    }
  }

#ifdef SEARCH_DEBUG
  if (cnt == 0) {
    //    printf("could not find search for [%s]\n", file->longname);
  }
#endif
}

void search_pattern_load()
{
  FILE *file;
  char* filename;
  char line[500];
  search_pattern_t *search = NULL;
  char *pos;
  char *pos2;

  filename = l_strdup_printf("%s/searches.save", global.options.config_dir);
  if ((file = fopen(filename, "r")) == NULL) {
    global.status.search_read = 1;
    l_free(filename);
    return;
  }
  l_free(filename);

  search = NULL;
  while (fgets(line, 1023, file)) {
    pos = strchr(line, '\n');
    if (!pos) {
      g_warning("search_pattern_load(): no line end found in line [%s]", line);
      continue;
    }
    *pos = 0;
    if ((line[0] == '#') || (line[0] == ' ') || (line[0] == 0)) {
    } else if (!strncasecmp(line, "Pattern", 7)) {
      pos = strchr(line, '[') + 1;
      pos2 = strrchr(line, ']');
      if (!pos || !pos2) {
	g_warning("corrupted searches.save");
      } else {
	pos2[0] = 0;
	if (search) search_pattern_insert(search, 0);
	search = search_pattern_new();
	search->name = l_strdup(pos);
      }
    } else if (search) {
      if (!strncasecmp(line, "contains", 8)) {
	pos = strchr(line, '=') + 1;
	search->include = l_strdup(pos);
      } else if (!strncasecmp(line, "excludes", 8)) {
	pos = strchr(line, '=') + 1;
	search->exclude = l_strdup(pos);
      } else if (!strncasecmp(line, "MaxResults", 10)) {
	pos = strchr(line, '=') + 1;
	search->max_results = atoi(pos);
      } else if (!strncasecmp(line, "MediaType", 9)) {
	pos = strchr(line, '=') + 1;
	if (strlen(pos) == 0)	// backward compatible
	  search->media_type = l_strdup("mp3");
	else
	  search->media_type = l_strdup(pos);
      } else if (!strncasecmp(line, "Destination", 11)) {
	pos = strchr(line, '=') + 1;
	search->destination = atoi(pos);
      } else if (!strncasecmp(line, "Bitrate", 7)) {
	pos = strchr(line, '=') + 1;
	pos2 = strchr(pos, ':');
	pos2[0] = 0;
	pos2++;
	if (*pos)
	  search->bitrate_lo = l_strdup(pos);
	if (*pos2)
	  search->bitrate_hi = l_strdup(pos2);
      } else if (!strncasecmp(line, "Frequency", 9)) {
	pos = strchr(line, '=') + 1;
	pos2 = strchr(pos, ':');
	pos2[0] = 0;
	pos2++;
	if (*pos)
	  search->freq_lo = l_strdup(pos);
	if (*pos2)
	  search->freq_hi = l_strdup(pos2);
      } else if (!strncasecmp(line, "LineSpeed", 9)) {
	pos = strchr(line, '=') + 1;
	pos2 = strchr(pos, ':');
	pos2[0] = 0;
	pos2++;
	if (*pos)
	  search->speed_lo = l_strdup(pos);
	if (*pos2)
	  search->speed_hi = l_strdup(pos2);
      } else if (!strncasecmp(line, "Size", 4)) {
	pos = strchr(line, '=') + 1;
	pos2 = strchr(pos, ':');
	pos2[0] = 0;
	pos2++;
	search->size_lo = strtoul(pos, NULL, 10);
	search->size_hi = strtoul(pos2, NULL, 10);
      } else if (!strncasecmp(line, "Duration", 8)) {
	pos = strchr(line, '=') + 1;
	pos2 = strchr(pos, ':');
	pos2[0] = 0;
	pos2++;
	search->duration_lo = atoi(pos);
	search->duration_hi = atoi(pos2);
      } else if (!strncasecmp(line, "Level", 5)) {
	pos = strchr(line, '=') + 1;
	search->level = atoi(pos);
      } else {
	g_warning(_("unknown tag in section Search Pattern [%s]\n[%s]"),
		  search->name, line);
      }
    } else {
      g_warning("no search name is given");
    }
  }

  if (search) search_pattern_insert(search, 0);
  fclose(file);
  global.status.search_read = 1;
}

void search_pattern_save()
{
  GList *dlist;
  search_pattern_t *search;
  FILE *file;
  char* filename;
  char *pos;
  char *pos2;

  filename = l_strdup_printf("%s/searches.save", global.options.config_dir);
  if ((file = fopen(filename, "w")) == NULL) {
    g_warning("Could not write searches");
    l_free(filename);
    return;
  }
  l_free(filename);

  for (dlist = global.search_pattern; dlist; dlist = dlist->next) {
    search = (search_pattern_t *) (dlist->data);
    if (search->level < PATTERN_SAVE) continue;
    pos2 = l_strdup(search->name);
    for (pos = pos2; *pos; pos++)
      if (*pos == '\n')
	*pos = ' ';
    fprintf(file, "PATTERN [%s]\n", pos2);
    l_free(pos2);
    fprintf(file, "Contains=%s\n", search->include);
    fprintf(file, "Excludes=%s\n", search->exclude);
    fprintf(file, "MaxResults=%d\n", search->max_results);
    fprintf(file, "MediaType=%s\n",
	    (search->media_type) ? (search->media_type) : "");
    fprintf(file, "Destination=%d\n", search->destination);
    fprintf(file, "Bitrate=%s:%s\n",
	    (search->bitrate_lo) ? (search->bitrate_lo) : "",
	    (search->bitrate_hi) ? (search->bitrate_hi) : "");
    fprintf(file, "Frequency=%s:%s\n",
	    (search->freq_lo) ? (search->freq_lo) : "",
	    (search->freq_hi) ? (search->freq_hi) : "");
    fprintf(file, "LineSpeed=%s:%s\n",
	    (search->speed_lo) ? (search->speed_lo) : "",
	    (search->speed_hi) ? (search->speed_hi) : "");
    fprintf(file, "Size=%ld:%ld\n", search->size_lo, search->size_hi);
    fprintf(file, "Duration=%d:%d\n", search->duration_lo,
	    search->duration_hi);
    fprintf(file, "Level=%d\n\n", search->level);
  }

  fclose(file);
}

void search_pattern_get(search_pattern_t * search)
{
  GtkWidget *temp;

  temp = lookup_widget(global.win, "search_artist");
  search->include = l_strdup(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "entry69");
  search->exclude = l_strdup(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "search_results");
  search->max_results =
      gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(temp));

  temp = lookup_widget(global.win, "combo_entry17");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->media_type =
	get_real_mediatype(gtk_entry_get_text(GTK_ENTRY(temp)));
  else
    search->media_type = l_strdup("mp3");

  temp = lookup_widget(global.win, "combo_entry19");
  search->destination =
      destination2int(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "entry76");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->bitrate_lo =
      get_real_entry(gtk_entry_get_text(GTK_ENTRY(temp)));
  
  temp = lookup_widget(global.win, "combo_bitrate2");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->bitrate_hi =
	get_real_entry(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "entry77");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->freq_lo = get_real_entry(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "combo_freq2");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->freq_hi = get_real_entry(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "entry78");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->speed_lo = get_real_speed(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "combo_speed2");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->speed_hi = get_real_speed(gtk_entry_get_text(GTK_ENTRY(temp)));

  temp = lookup_widget(global.win, "entry127");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->size_lo = extract_bytes(gtk_entry_get_text(GTK_ENTRY(temp)));
  else
    search->size_lo = 0;

  temp = lookup_widget(global.win, "entry129");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->size_hi = extract_bytes(gtk_entry_get_text(GTK_ENTRY(temp)));
  else
    search->size_hi = 0;

  temp = lookup_widget(global.win, "entry128");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->duration_lo = extract_duration(gtk_entry_get_text(GTK_ENTRY(temp)));
  else
    search->duration_lo = 0;

  temp = lookup_widget(global.win, "entry130");
  if (GTK_WIDGET_IS_SENSITIVE(temp))
    search->duration_hi = extract_duration(gtk_entry_get_text(GTK_ENTRY(temp)));
  else
    search->duration_hi = 0;

  temp = lookup_widget(global.win, "combo_entry28");
  search->level =
    life2int(gtk_entry_get_text(GTK_ENTRY(temp)));
}

search_pattern_t* find_pattern_id(int id) {
  search_pattern_t* pattern;
  GList* dlist;

  for (dlist = global.search_pattern; dlist; dlist = dlist->next) {
    pattern = dlist->data;
    if (id == pattern->id) return pattern;
  }
  return NULL;
}

int find_free_pattern_id() {
  int id = 0;

  while (1) {
    if (!find_pattern_id(id)) return id;
    id++;
  }
  return -1;
}

search_pattern_t *search_pattern_new()
{
  search_pattern_t *pattern;

  pattern = l_malloc(sizeof(*pattern));

  pattern->name = NULL;
  pattern->include = NULL;
  pattern->exclude = NULL;
  pattern->media_type = NULL;
  pattern->destination = DEST_NAPSTER;
  pattern->bitrate_lo = NULL;
  pattern->bitrate_hi = NULL;
  pattern->freq_lo = NULL;
  pattern->freq_hi = NULL;
  pattern->speed_lo = NULL;
  pattern->speed_hi = NULL;
  pattern->max_results = 0;
  pattern->size_lo = 0;
  pattern->size_hi = 0;
  pattern->duration_lo = 0;
  pattern->duration_hi = 0;
  pattern->id = find_free_pattern_id();  //find l_free id
  pattern->level = PATTERN_IGNORE;
 
  return pattern;
}

search_pattern_t *search_pattern_create()
{
  search_pattern_t *search;

  search = search_pattern_new();
  search_pattern_get(search);
  search->name = l_strdup(search->include);

  return search;
}

void search_pattern_update(search_pattern_t * search)
{
  if (search->include)
    l_free(search->include);
  search->include = NULL;
  if (search->exclude)
    l_free(search->exclude);
  search->exclude = NULL;
  if (search->media_type)
    l_free(search->media_type);
  search->media_type = NULL;
  if (search->bitrate_lo)
    l_free(search->bitrate_lo);
  search->bitrate_lo = NULL;
  if (search->bitrate_hi)
    l_free(search->bitrate_hi);
  search->bitrate_hi = NULL;
  if (search->freq_lo)
    l_free(search->freq_lo);
  search->freq_lo = NULL;
  if (search->freq_hi)
    l_free(search->freq_hi);
  search->freq_hi = NULL;
  if (search->speed_lo)
    l_free(search->speed_lo);
  search->speed_lo = NULL;
  if (search->speed_hi)
    l_free(search->speed_hi);
  search->speed_hi = NULL;

  search_pattern_get(search);
}

void search_pattern_destroy(search_pattern_t * search)
{
  if (!search)
    return;
  if (search->name)
    l_free(search->name);
  if (search->include)
    l_free(search->include);
  if (search->exclude)
    l_free(search->exclude);
  if (search->media_type)
    l_free(search->media_type);
  if (search->bitrate_lo)
    l_free(search->bitrate_lo);
  if (search->bitrate_hi)
    l_free(search->bitrate_hi);
  if (search->freq_lo)
    l_free(search->freq_lo);
  if (search->freq_hi)
    l_free(search->freq_hi);
  if (search->speed_lo)
    l_free(search->speed_lo);
  if (search->speed_hi)
    l_free(search->speed_hi);
  l_free(search);
}

int search_pattern_fits_file(search_pattern_t * search, file_t * file,
			     int with_type)
{
  static char *pos;
  static char *text;
  static long val1;
  static long val2;

  text = l_strdup(search->include);
  pos = search_arg(text);
  while (pos) {
    if (!strcasestr(file->longname, pos)) {
      l_free(text);
      return 0;
    }
    pos = search_arg(NULL);
  }
  l_free(text);

  text = l_strdup(search->exclude);
  pos = arg(text, 0);
  while (pos) {
    if (strcasestr(file->longname, pos))
      return 0;
    pos = arg(NULL, 0);
  }
  l_free(text);

  if (with_type && search->media_type &&
      strcmp(search->media_type, "any") &&
      (mime2int_(search->media_type) != file->mime_type))
    return 0;

  val1 = (search->bitrate_lo) ? atoi(search->bitrate_lo) : 0;
  val2 = (search->bitrate_hi) ? atoi(search->bitrate_hi) : 0;
  if (val1 && (file->bitrate < val1))
    return 0;
  if (val2 && (file->bitrate > val2))
    return 0;

  val1 = (search->freq_lo) ? atoi(search->freq_lo) : 0;
  val2 = (search->freq_hi) ? atoi(search->freq_hi) : 0;
  if (val1 && (file->frequency < val1))
    return 0;
  if (val2 && (file->frequency > val2))
    return 0;

  val1 = search->size_lo;
  val2 = search->size_hi;
  if (val1 && (file->size < (unsigned long)val1))
    return 0;
  if (val2 && (file->size > (unsigned long)val2))
    return 0;

  val1 = search->duration_lo;
  val2 = search->duration_hi;
  if (val1 && (file->duration < val1))
    return 0;
  if (val2 && (file->duration > val2))
    return 0;

  if (!file->local) {
    val1 = (search->speed_lo) ? speed2int(search->speed_lo) : 0;
    val2 = (search->speed_hi) ? speed2int(search->speed_hi) : 10;
    if (file->linespeed < val1)
      return 0;
    if (file->linespeed > val2)
      return 0;
  }
  return 1;
}

void search_pattern_show(search_pattern_t * pattern)
{
  GtkWidget *temp;
  char *text;
  char str[1024];

  if (pattern) {
    temp = lookup_widget(global.win, "search_artist");
    gtk_entry_set_text(GTK_ENTRY(temp), pattern->include);
    temp = lookup_widget(global.win, "entry69");
    gtk_entry_set_text(GTK_ENTRY(temp), pattern->exclude);
    temp = lookup_widget(global.win, "search_results");
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp), pattern->max_results);
    temp = lookup_widget(global.win, "combo_entry17");
    gtk_entry_set_text(GTK_ENTRY(temp), _(pattern->media_type));
    temp = lookup_widget(global.win, "combo_entry19");
    gtk_entry_set_text(GTK_ENTRY(temp),
		       Destination_long(pattern->destination));
    temp = lookup_widget(global.win, "entry76");
    gtk_entry_set_text(GTK_ENTRY(temp),
		       (pattern->bitrate_lo) ? (pattern->
						bitrate_lo) :
		       _("not specified"));
    temp = lookup_widget(global.win, "combo_bitrate2");
    gtk_entry_set_text(GTK_ENTRY(temp),
		       (pattern->bitrate_hi) ? (pattern->
						bitrate_hi) :
		       _("not specified"));
    temp = lookup_widget(global.win, "entry77");
    gtk_entry_set_text(GTK_ENTRY(temp),
		       (pattern->freq_lo) ? (pattern->
					     freq_lo) :
		       _("not specified"));
    temp = lookup_widget(global.win, "combo_freq2");
    gtk_entry_set_text(GTK_ENTRY(temp),
		       (pattern->freq_hi) ? (pattern->
					     freq_hi) :
		       _("not specified"));
    temp = lookup_widget(global.win, "entry78");
    gtk_entry_set_text(GTK_ENTRY(temp),
		       (pattern->speed_lo) ? (pattern->
					      speed_lo) :
		       _("not specified"));
    temp = lookup_widget(global.win, "combo_speed2");
    gtk_entry_set_text(GTK_ENTRY(temp),
		       (pattern->speed_hi) ? (pattern->speed_hi) :_("not specified"));

    temp = lookup_widget(global.win, "entry127");
    print_bytes(str, pattern->size_lo);
    gtk_entry_set_text(GTK_ENTRY(temp), str);

    temp = lookup_widget(global.win, "entry129");
    if (pattern->size_hi == 0) sprintf(str, _("unlimited"));
    else print_bytes(str, pattern->size_hi);
    gtk_entry_set_text(GTK_ENTRY(temp), str);

    temp = lookup_widget(global.win, "entry128");
    print_duration(str, pattern->duration_lo);
    gtk_entry_set_text(GTK_ENTRY(temp), str);

    temp = lookup_widget(global.win, "entry130");
    if (pattern->duration_hi == 0) sprintf(str, _("unlimited"));
    else print_duration(str, pattern->duration_hi);
    gtk_entry_set_text(GTK_ENTRY(temp), str);

    temp = lookup_widget(global.win, "combo_entry28");
    gtk_entry_set_text(GTK_ENTRY(temp), LifeTime(pattern->level));
  } else {
    temp = lookup_widget(global.win, "search_artist");
    text = gtk_entry_get_text(GTK_ENTRY(temp));
    if (!text || !(*text)) {
      temp = lookup_widget(global.win, "search_results");
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(temp), 100);
      temp = lookup_widget(global.win, "combo_entry17");
      gtk_entry_set_text(GTK_ENTRY(temp), _("mp3"));
      temp = lookup_widget(global.win, "combo_entry19");
      gtk_entry_set_text(GTK_ENTRY(temp), Destination_long(DEST_NAPSTER));
      temp = lookup_widget(global.win, "entry76");
      gtk_entry_set_text(GTK_ENTRY(temp), _("not specified"));
      temp = lookup_widget(global.win, "combo_bitrate2");
      gtk_entry_set_text(GTK_ENTRY(temp), _("not specified"));
      temp = lookup_widget(global.win, "entry77");
      gtk_entry_set_text(GTK_ENTRY(temp), _("not specified"));
      temp = lookup_widget(global.win, "combo_freq2");
      gtk_entry_set_text(GTK_ENTRY(temp), _("not specified"));
      temp = lookup_widget(global.win, "entry78");
      gtk_entry_set_text(GTK_ENTRY(temp), _("not specified"));
      temp = lookup_widget(global.win, "combo_speed2");
      gtk_entry_set_text(GTK_ENTRY(temp), _("not specified"));
      temp = lookup_widget(global.win, "entry127");
      gtk_entry_set_text(GTK_ENTRY(temp), "0B");
      temp = lookup_widget(global.win, "entry129");
      gtk_entry_set_text(GTK_ENTRY(temp), _("unlimited"));
      temp = lookup_widget(global.win, "entry128");
      gtk_entry_set_text(GTK_ENTRY(temp), "0s");
      temp = lookup_widget(global.win, "entry130");
      gtk_entry_set_text(GTK_ENTRY(temp), _("unlimited"));
      temp = lookup_widget(global.win, "combo_entry28");
      gtk_entry_set_text(GTK_ENTRY(temp), LifeTime(PATTERN_TEMPORARY));
    }
    temp = lookup_widget(global.win, "search_artist");
    gtk_entry_set_text(GTK_ENTRY(temp), "");
    temp = lookup_widget(global.win, "entry69");
    gtk_entry_set_text(GTK_ENTRY(temp), "");
  }
}

search_pattern_t *search_pattern_copy(search_pattern_t * search)
{
  search_pattern_t *pattern;

  pattern = search_pattern_new();
  if (search->name)
    pattern->name = l_strdup(search->name);
  if (search->include)
    pattern->include = l_strdup(search->include);
  if (search->exclude)
    pattern->exclude = l_strdup(search->exclude);
  if (search->media_type)
    pattern->media_type = l_strdup(search->media_type);
  pattern->destination = search->destination;
  if (search->bitrate_lo)
    pattern->bitrate_lo = l_strdup(search->bitrate_lo);
  if (search->bitrate_hi)
    pattern->bitrate_hi = l_strdup(search->bitrate_hi);
  if (search->freq_lo)
    pattern->freq_lo = l_strdup(search->freq_lo);
  if (search->freq_hi)
    pattern->freq_hi = l_strdup(search->freq_hi);
  if (search->speed_lo)
    pattern->speed_lo = l_strdup(search->speed_lo);
  if (search->speed_hi)
    pattern->speed_hi = l_strdup(search->speed_hi);
  pattern->max_results = search->max_results;
  pattern->size_lo = search->size_lo;
  pattern->size_hi = search->size_hi;
  pattern->duration_lo = search->duration_lo;
  pattern->duration_hi = search->duration_hi;
  pattern->id = search->id;
  pattern->level = search->level;
  return pattern;
}

search_t *search_new()
{
  search_t *search;

  search = (search_t *) l_malloc(sizeof(search_t));
  search->results = NULL;
  search->resume = 0;
  search->status = SEARCH_ACTIVE;
  search->local = 0;
  search->pattern = NULL;
  search->link = NULL;
  search->timestamp = 0;

  global.searches = g_list_append(global.searches, search);

  return search;
}

void search_clear(search_t * search)
{
  GList *dlist;
  file_t *file;
  GtkCList *clist;

  for (dlist = search->results; dlist; dlist = dlist->next) {
    file = (file_t *) (dlist->data);
    destroy_file_row(file);
  }
  g_list_free(search->results);
  search->results = NULL;

  if (!search->resume) {
    clist = GTK_CLIST(gtk_object_get_data(GTK_OBJECT(search->link), "clist"));
    gtk_clist_clear(clist);
  }

  search_update_stats(search, 1);
}

void search_destroy(search_t * search)
{

  if (!search) return;
  // releasing file list
  search_clear(search);
  search_pattern_destroy(search->pattern);
  l_free(search);
}

void search_show(search_t* search) {
  GtkCList* clist;
  GList* dlist;
  file_t* file;

  if (!search) return;
  if (search->status != SEARCH_FINISHED) return;
  if (search->resume) return;
  if (!search->link) return;

  clist = GTK_CLIST(gtk_object_get_data(GTK_OBJECT(search->link), "clist"));
  gtk_clist_clear(clist);
  gtk_clist_freeze(clist);
  for (dlist = search->results; dlist; dlist = dlist->next) {
    file = dlist->data;
    search_insert_file_real(clist, file);
  }
  gtk_clist_thaw(clist);
  search_update_stats(search, 1);
}

void search_remove(search_t * search)
{
  GtkNotebook *notebook;
  int i1;

  global.searches = g_list_remove(global.searches, search);

  if (!search->resume) {
    notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook5"));
    i1 = gtk_notebook_page_num(notebook, GTK_WIDGET(search->link));
    search_destroy(search);
    gtk_notebook_remove_page(notebook, i1);
  } else {
    search_destroy(search);
  }
}

void search_update_stats(search_t * search, int check)
{
  char t[1024];
  GtkWidget *label;
  GtkNotebook* notebook;
  int row;
  GtkWidget* temp;

  if (!search || search->resume) return;

  notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook5"));
  row = gtk_notebook_get_current_page(notebook);
  temp = gtk_notebook_get_nth_page(notebook, row);
  if (!check || (temp == search->link)) {
    label = lookup_widget(global.win, "label597");
    sprintf(t, "%d", g_list_length(search->results));
    gtk_label_set_text(GTK_LABEL(label), t);
  }
}

void search_update_counter()
{
  GtkWidget *label;
  char t[1024];

  label = lookup_widget(global.win, "label598");
  sprintf(t, "%d", global.status.searching);
  gtk_label_set_text(GTK_LABEL(label), t);
}

gboolean
on_search_fields_delete(GtkWidget * widget,
			GdkEvent * event ATTR_UNUSED, gpointer user_data ATTR_UNUSED)
{
  GtkWidget *win;
  GtkWidget *temp;
  GtkWidget *vbox;

  win = widget;

  if (win->window) {
    gdk_window_get_origin(win->window,
			  &global.geometry[G_SIZE-1].x, 
			  &global.geometry[G_SIZE-1].y);
  }
  global.geometry[G_SIZE-1].width = win->allocation.width;
  global.geometry[G_SIZE-1].height = win->allocation.height;

  temp = GTK_BIN(win)->child;
  if (!temp)
    return FALSE;

  gtk_widget_ref(temp);
  gtk_container_remove(GTK_CONTAINER(win), temp);

  vbox = lookup_widget(global.win, "vbox5");
  gtk_box_pack_start(GTK_BOX(vbox), temp, FALSE, FALSE, 0);
  gtk_box_reorder_child(GTK_BOX(vbox), temp, 1);
  gtk_widget_unref(temp);

  widget = lookup_widget(global.win, "button115");
  gtk_widget_set_sensitive(widget, TRUE);
  widget = lookup_widget(global.win, "button218");
  gtk_widget_set_sensitive(widget, TRUE);
  widget = lookup_widget(global.win, "frame82");
  gtk_widget_show(widget);
  global.extra_win &= (0xFFF) ^ (1 << (G_SIZE-1));

  return FALSE;
}

GtkWidget *search_fields_detach()
{
  GtkWidget *window;
  GtkWidget *widget;

  widget = lookup_widget(global.win, "hbox710");
  if (!widget)
    return NULL;
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width(GTK_CONTAINER(window), 5);
  gtk_object_set_data(GTK_OBJECT(global.win), "fields", window);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, TRUE, TRUE);
  gtk_window_set_title(GTK_WINDOW(window), _("Search Fields"));

  gtk_my_widget_show(window);

  gtk_widget_ref(widget);
  gtk_container_remove(GTK_CONTAINER(widget->parent), widget);
  gtk_container_add(GTK_CONTAINER(window), widget);
  gtk_widget_unref(widget);

  gtk_signal_connect(GTK_OBJECT(window), "delete_event",
		     GTK_SIGNAL_FUNC(on_search_fields_delete), NULL);
  widget = lookup_widget(global.win, "button115");
  set_button_state(widget, 1);
  gtk_widget_set_sensitive(widget, FALSE);
  widget = lookup_widget(global.win, "frame82");
  gtk_widget_hide(widget);
  widget = lookup_widget(global.win, "button218");
  gtk_widget_set_sensitive(widget, FALSE);
  global.extra_win |= (1 << (G_SIZE - 1));

  if ((global.geometry[G_SIZE-1].x >= 0) && (global.geometry[G_SIZE-1].y >= 0)) {
    gtk_window_reposition(GTK_WINDOW(window),
			  global.geometry[G_SIZE - 1].x,
			  global.geometry[G_SIZE - 1].y);
  }
  if ((global.geometry[G_SIZE-1].width >= 0) && 
      (global.geometry[G_SIZE-1].height >= 0)) {
    gtk_window_set_default_size(GTK_WINDOW(window),
				global.geometry[G_SIZE - 1].width,
				global.geometry[G_SIZE - 1].height);
  }
  return window;
}

gboolean
on_search_fields_delete2(GtkWidget * widget,
			GdkEvent * event ATTR_UNUSED, gpointer user_data ATTR_UNUSED)
{
  GtkWidget *win;
  GtkWidget *temp;
  GtkWidget *vbox;

  win = widget;

  if (win->window) {
    gdk_window_get_origin(win->window,
			  &global.geometry[G_SIZE-2].x, 
			  &global.geometry[G_SIZE-2].y);
  }
  global.geometry[G_SIZE-2].width = win->allocation.width;
  global.geometry[G_SIZE-2].height = win->allocation.height;
  
  temp = GTK_BIN(win)->child;
  if (!temp)
    return FALSE;

  gtk_widget_ref(temp);
  gtk_container_remove(GTK_CONTAINER(win), temp);

  vbox = lookup_widget(global.win, "hbox710");
  gtk_box_pack_start(GTK_BOX(vbox), temp, TRUE, TRUE, 0);
  //  gtk_box_reorder_child(GTK_BOX(vbox), temp, 1);
  gtk_widget_unref(temp);

  widget = lookup_widget(global.win, "button337");
  gtk_widget_set_sensitive(widget, TRUE);
  widget = lookup_widget(global.win, "button331");
  gtk_widget_set_sensitive(widget, TRUE);
  widget = lookup_widget(global.win, "vbox205");
  gtk_widget_show(widget);
  global.extra_win &= (0xFFF) ^ (1 << (G_SIZE-2));

  return FALSE;
}

GtkWidget *search_fields_detach2()
{
  GtkWidget *window;
  GtkWidget *widget;

  widget = lookup_widget(global.win, "scrolledwindow96");
  if (!widget) return NULL;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width(GTK_CONTAINER(window), 5);
  gtk_object_set_data(GTK_OBJECT(global.win), "fields2", window);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, TRUE, TRUE);
  gtk_window_set_title(GTK_WINDOW(window), _("Search History"));

  gtk_my_widget_show(window);

  gtk_widget_ref(widget);
  gtk_container_remove(GTK_CONTAINER(widget->parent), widget);
  gtk_container_add(GTK_CONTAINER(window), widget);
  gtk_widget_unref(widget);

  gtk_signal_connect(GTK_OBJECT(window), "delete_event",
		     GTK_SIGNAL_FUNC(on_search_fields_delete2), NULL);
  widget = lookup_widget(global.win, "button331");
  set_button_state(widget, 1);
  gtk_widget_set_sensitive(widget, FALSE);
  widget = lookup_widget(global.win, "vbox205");
  gtk_widget_hide(widget);
  widget = lookup_widget(global.win, "button337");
  gtk_widget_set_sensitive(widget, FALSE);
  global.extra_win |= (1 << (G_SIZE - 2));

  if ((global.geometry[G_SIZE-2].x >= 0) && (global.geometry[G_SIZE-2].y >= 0)) {
    gtk_window_reposition(GTK_WINDOW(window),
			  global.geometry[G_SIZE - 2].x,
			  global.geometry[G_SIZE - 2].y);
  }
  if ((global.geometry[G_SIZE-2].width >= 0) && 
      (global.geometry[G_SIZE-2].height >= 0)) {
    gtk_window_set_default_size(GTK_WINDOW(window),
				global.geometry[G_SIZE - 2].width,
				global.geometry[G_SIZE - 2].height);
  }
  return window;
}

void search_search(search_t * search)
{
  GList *dlist;
  GList *dlist2;
  file_t *file;
  int cnt = 0;
  search_t *tsearch;

  for (dlist = global.searches; dlist; dlist = dlist->next) {
    tsearch = dlist->data;
    if (tsearch->resume) continue;
    if (tsearch == search) continue;
    for (dlist2 = tsearch->results; dlist2; dlist2 = dlist2->next) {
      file = dlist2->data;
      if (search_pattern_fits_file(search->pattern, file, 1)) {
	search_insert_file(search, file);
	cnt++;
      }
      if (search->pattern->max_results && (cnt >= search->pattern->max_results))
	break;
    }
    if (search->pattern->max_results && (cnt >= search->pattern->max_results))
      break;
  }

  search_finish(search);
}

void search_queue(search_t * search)
{
  GtkCList *clist;

  // dont queue local searches
  if (search->pattern->destination != DEST_NAPSTER) {
    search_do(search);
    return;
  }

  if (search->results) {
    printf("cant queue search: already got results\n");
    return;
  }

  search->status = SEARCH_QUEUED;

  if (search->resume) {
    resume_update((resume_t *) (search->link));
  } else {
    clist =
      GTK_CLIST(gtk_object_get_data(GTK_OBJECT(search->link), "clist"));
    strcpy(tstr[0], _("Search queued...."));
    tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = tstr[5][0] =
	tstr[6][0] = tstr[7][0] = 0;
    gtk_clist_append(clist, &list[0]);
  }
}

int search_unqueue(search_t * search)
{
  GtkCList *clist;

  search->status = SEARCH_ACTIVE;
  if (search->resume) {
    resume_update((resume_t *) (search->link));
  } else {
    clist =
	GTK_CLIST(gtk_object_get_data(GTK_OBJECT(search->link), "clist"));
    gtk_clist_clear(clist);
  }

  return 1;
}

void search_do(search_t * search)
{
  search_pattern_t *pattern;
  resume_t *resume;
  GtkCList* clist;

  pattern = search->pattern;

  if ((pattern->destination == DEST_NAPSTER) && (global.status.connection < 2))
    return;

  if (!search_unqueue(search))
    return;

  search->status = SEARCH_ACTIVE;

  if (search->resume) {
    resume = search->link;
    resume_update(resume);
  } else {
    clist = GTK_CLIST(gtk_object_get_data(GTK_OBJECT(search->link), "clist"));
    strcpy(tstr[0], _("Search in progress...."));
    tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = tstr[5][0] =
      tstr[6][0] = tstr[7][0] = 0;
    gtk_clist_append(clist, &list[0]);
  }

  search->timestamp = global.current_time;
  global.status.searching++;
  search_update_counter();

  if (pattern->destination == DEST_LIBRARY) {
    lib_search(search);
  } else if (pattern->destination == DEST_HOTLIST) {
    browse_do_search(search);
  } else if (pattern->destination == DEST_NAPSTER) {
    napster_search(search);
  } else if (pattern->destination == DEST_SEARCH) {
    search_search(search);
  } else {
    printf("unknown search destination");
  }
}

void search_finish(search_t* search) {
  if (search->status != SEARCH_ACTIVE) return;

  if (global.status.searching > 0) global.status.searching--;
#ifdef SEARCH_DEBUG
  else printf("search_finish: search counter <= 0 !\n");
#endif

  search_update_counter();
  
  search->status = SEARCH_FINISHED;
  // now show the results;
  if (search->resume) {
    resume_update((resume_t *) (search->link));
  } else {
    search_show(search);
  }
  // starting another search if possible
  search_next();
}

void search_next() {
  search_t *search1;
  search_t *search2;
  search_t *tsearch;
  GList* dlist;

  if (global.status.searching >= global.limit.max_searches_real) return;

  search1 = search2 = NULL;
  for (dlist = global.searches; dlist; dlist = dlist->next) {
    tsearch = dlist->data;
    if (tsearch->status == SEARCH_QUEUED) {
      if (tsearch->resume) {
	if (!search2) search2 = tsearch;
      } else {
	if (!search1) search1 = tsearch;
      }
    }
  }
  if (search1) search_do(search1);
  else if (search2) search_do(search2);
}

search_t* search_finish_oldest(int latest) {
  search_t* search;
  GList* dlist;
  search_t* to_finish = NULL;

  for (dlist = global.searches; dlist; dlist = dlist->next) {
    search = dlist->data;
    if (!search->pattern) continue;
    if (search->pattern->destination != DEST_NAPSTER) continue;
    if (search->status != SEARCH_ACTIVE) continue;

    if (latest) {
      if (!to_finish || (search->timestamp >= to_finish->timestamp))
	to_finish = search;
    } else {
      if (!to_finish || (search->timestamp < to_finish->timestamp))
	to_finish = search;
    }
  }

  if (!to_finish) {
#ifdef SEARCH_DEBUG
    printf("search_finish_oldest: no search found to finish\n");
#endif  
    // dont forget to decrease the counter in this case.
    if (global.status.searching > 0) global.status.searching--;
    search_update_counter();
    return NULL;
  }

  search_finish(to_finish);
  return to_finish;
}  

#define ROW_TOP_YPIXEL(clist, row) (((clist)->row_height * (row)) + \
                                    ((row) + 1) + \
                                    (clist)->voffset)

void search_pattern_visual_update(search_pattern_t* pattern) {
  GtkCList* clist;
  int row;

  clist = GTK_CLIST(lookup_widget(global.win, "clist34"));
  row = gtk_clist_find_row_from_data(clist, pattern);
  if (row < 0) return;
  
  gtk_clist_set_text(clist, row, 0, pattern->name);
  strcpy(tstr[3], LifeTime(pattern->level));

  gtk_clist_set_text(clist, row, 2, tstr[3]);
}

void on_rename_activate(GtkEditable * editable, gpointer user_data)
{
  char* text;
  search_pattern_t* pattern;

  gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "activate");
  text = gtk_entry_get_text(GTK_ENTRY(editable));
  pattern = gtk_clist_get_row_data(GTK_CLIST(global.popup_list), global.popup_row);
  l_free(pattern->name);
  pattern->name = l_strdup(text);
  gtk_widget_destroy(GTK_WIDGET(user_data));
  gtk_clist_set_text(GTK_CLIST(global.popup_list), global.popup_row, 0, 
		     pattern->name);
}

gboolean
on_escape_press_event             (GtkWidget       *widget ATTR_UNUSED,
				   GdkEventKey     *event,
				   gpointer         user_data) {

  if ((event->type == GDK_KEY_PRESS) && (event->keyval == GDK_Escape)) {
    gtk_widget_destroy(GTK_WIDGET(user_data));
  }
  return FALSE;
}

void search_pattern_rename(search_pattern_t* pattern) {
  GtkWidget* win;
  GtkWidget* entry;
  int x, y;
  int w, h;

  win = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_title(GTK_WINDOW(win), _("Edit"));
  gtk_window_set_modal (GTK_WINDOW (win), TRUE);
  gtk_window_set_policy(GTK_WINDOW(win), FALSE, FALSE, FALSE);

  entry = gtk_entry_new ();
  gtk_entry_set_text(GTK_ENTRY(entry), pattern->name);
  gtk_widget_show(entry);
  gtk_container_add (GTK_CONTAINER (win), entry);
  gtk_entry_select_region(GTK_ENTRY(entry), 0, -1);
  
  gtk_signal_connect (GTK_OBJECT (entry), "activate",
                      GTK_SIGNAL_FUNC (on_rename_activate),
                      (gpointer)win);
  gtk_signal_connect (GTK_OBJECT (entry), "key_press_event",
                      GTK_SIGNAL_FUNC (on_escape_press_event),
                      (gpointer)win);
  
  gdk_window_get_origin(GTK_CLIST(global.popup_list)->clist_window, &x, &y);
  x += GTK_CLIST(global.popup_list)->column[0].area.x;
  y += ROW_TOP_YPIXEL(GTK_CLIST(global.popup_list), global.popup_row);
  w = GTK_CLIST(global.popup_list)->column[1].area.x -
    GTK_CLIST(global.popup_list)->column[0].area.x;
  h = GTK_CLIST(global.popup_list)->row_height+1;
  y -= 3;
  x -= 4;
  gtk_widget_realize(win);
  if (win->window) gdk_window_set_decorations(win->window, 0);
  gtk_widget_set_usize(win, w, -2);

  gtk_widget_set_uposition(win, x, y);
  gtk_widget_show(win);
  //  gtk_window_reposition(GTK_WINDOW(win), x, y);
  gtk_widget_grab_focus(entry);
}

search_t* search_find_id(int id) {
  GList* dlist;
  search_t* search;

  for (dlist = global.searches; dlist; dlist = dlist->next) {
    search = dlist->data;
    if (search->resume) continue;
    if (search->pattern->id == id) return search;
  }
  return NULL;
}

void search_pattern_search(search_pattern_t* pattern) {
  search_t *search;
  GtkNotebook* notebook;

  // first search for pattern id, if found do not create new search
  search = search_find_id(pattern->id);
  if (search) {
    search_clear(search);
    notebook = GTK_NOTEBOOK(lookup_widget(global.win, "notebook5"));
    gtk_notebook_set_page(notebook,
			  gtk_notebook_page_num(GTK_NOTEBOOK(notebook),
						GTK_WIDGET(search->link)));
  } else {
    search = search_new();
    search->pattern = search_pattern_copy(pattern);
    search_create_page(search);
  }
  search_queue(search);
}

void
on_search_pattern_rename(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data) {
  search_pattern_rename((search_pattern_t*)user_data);
}

void search_pattern_delete(search_pattern_t* pattern) {
  GtkCList* clist;
  int row;

  clist = GTK_CLIST(lookup_widget(global.win, "clist34"));
  row = gtk_clist_find_row_from_data(clist, pattern);
  if (row < 0) return;
  gtk_clist_remove(clist, row);
  global.search_pattern = g_list_remove(global.search_pattern, pattern);
}

void
on_search_pattern_delete(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data ATTR_UNUSED) {

  GtkCList *clist;
  GList *row_list;
  search_pattern_t *pattern;
  int row;

  clist = GTK_CLIST(global.popup_list);
  row_list = clist->selection;

  gtk_clist_freeze(clist);
  while (row_list) {
    row = (int) row_list->data;
    row_list = row_list->next;
    pattern = gtk_clist_get_row_data(clist, row);
    gtk_clist_remove(clist, row);
    global.search_pattern = g_list_remove(global.search_pattern, pattern);
    search_pattern_destroy(pattern);
  }
  gtk_clist_thaw(clist);
  search_pattern_save();
}

void
on_search_pattern_search(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data ATTR_UNUSED) {
  GtkCList *clist;
  GList *row_list;
  search_pattern_t *pattern;
  int row;

  clist = GTK_CLIST(global.popup_list);
  row_list = clist->selection;

  while (row_list) {
    row = (int) row_list->data;
    pattern = gtk_clist_get_row_data(clist, row);
    search_pattern_search(pattern);
    row_list = row_list->next;
  }
}

void
on_search_pattern_search_all(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data ATTR_UNUSED) {
  GList *dlist;
  search_pattern_t *pattern;

  for (dlist = global.search_pattern; dlist; dlist = dlist->next) {
    pattern = dlist->data;
    search_pattern_search(pattern);
  }
}

void
on_search_pattern_level(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data) {
  GtkCList *clist;
  GList *row_list;
  search_pattern_t *pattern;
  int row;

  clist = GTK_CLIST(global.popup_list);
  row_list = clist->selection;
  
  while (row_list) {
    row = (int) row_list->data;
    pattern = gtk_clist_get_row_data(clist, row);
    pattern->level = (int)user_data;
    search_pattern_visual_update(pattern);
    row_list = row_list->next;
  }
  search_pattern_save();
}

GtkWidget *create_search_pattern_popup(search_pattern_t *pattern)
{
  GtkWidget *popup;
  GtkWidget *item;
  GtkWidget *item2;
  GtkWidget *item3;
  GtkAccelGroup *popup_accels;
  char item_str[1024];
  int item_num;
  GtkCList* clist;
  GSList *level_group = NULL;
  
  if (!pattern) return NULL;

  popup = gtk_menu_new();
  popup_accels = gtk_menu_ensure_uline_accel_group(GTK_MENU(popup));
  clist = GTK_CLIST(global.popup_list);
  item_num = g_list_length(clist->selection);

  if (item_num > 1)
    sprintf(item_str, _("Search Selected (%d)"), item_num);
  else strcpy(item_str, _("Search"));
  item = gtk_menu_item_new_with_label(item_str);
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_search_pattern_search), NULL);
  
  item = gtk_menu_item_new_with_label(_("Search All"));
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_search_pattern_search_all), NULL);

  item = gtk_menu_item_new();
  gtk_widget_show(item);
  gtk_widget_set_sensitive(item, FALSE);
  gtk_container_add(GTK_CONTAINER(popup), item);

  if (item_num > 1)
    sprintf(item_str, _("Delete Selected (%d)"), item_num);
  else strcpy(item_str, _("Delete"));
  item = gtk_menu_item_new_with_label(item_str);
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_search_pattern_delete), NULL);
  
  item = gtk_menu_item_new_with_label(_("Rename"));
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_search_pattern_rename),
		     (gpointer)pattern);
  item = gtk_menu_item_new();
  gtk_widget_show(item);
  gtk_widget_set_sensitive(item, FALSE);
  gtk_container_add(GTK_CONTAINER(popup), item);
  
  item = gtk_radio_menu_item_new_with_label (level_group, LifeTime(PATTERN_TEMPORARY));
  level_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (item));
  gtk_widget_show (item);
  gtk_container_add (GTK_CONTAINER (popup), item);

  if (pattern->level == PATTERN_TEMPORARY)
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (item), TRUE);
  gtk_check_menu_item_set_show_toggle (GTK_CHECK_MENU_ITEM (item), TRUE);

  item2 = gtk_radio_menu_item_new_with_label (level_group, LifeTime(PATTERN_SAVE));
  level_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (item2));
  gtk_widget_show (item2);
  gtk_container_add (GTK_CONTAINER (popup), item2);

  if (pattern->level == PATTERN_SAVE)
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (item2), TRUE);
  gtk_check_menu_item_set_show_toggle (GTK_CHECK_MENU_ITEM (item2), TRUE);

  item3 = gtk_radio_menu_item_new_with_label (level_group, LifeTime(PATTERN_SEARCH));
  level_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (item3));
  gtk_widget_show (item3);
  gtk_container_add (GTK_CONTAINER (popup), item3);

  if (pattern->level == PATTERN_SEARCH)
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (item3), TRUE);
  gtk_check_menu_item_set_show_toggle (GTK_CHECK_MENU_ITEM (item3), TRUE);

  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_search_pattern_level),
		     (gpointer)PATTERN_TEMPORARY);
  gtk_signal_connect(GTK_OBJECT(item2), "activate",
		     GTK_SIGNAL_FUNC(on_search_pattern_level),
		     (gpointer)PATTERN_SAVE);
  gtk_signal_connect(GTK_OBJECT(item3), "activate",
		     GTK_SIGNAL_FUNC(on_search_pattern_level),
		     (gpointer)PATTERN_SEARCH);

  return popup;
}
