/* Copyright (C) 2000/2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include <netdb.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "callbacks.h"
#include "support.h"
#include "connection.h"
#include "global.h"
#include "commands.h"
#include "handler.h"
#include "dialog.h"
#include "chat.h"
#include "interface.h"
#include "server.h"

server_t *server_new()
{
  server_t *server;

  server = (server_t *) l_malloc(sizeof(server_t));
  server->nick = NULL;
  server->passwd = NULL;
  server->description = NULL;
  server->address = NULL;
  server->meta = 0;
  server->network = N_UNKNOWN;
  server->major = server->minor = 0;

  return server;
}

server_t *server_copy(server_t * s)
{
  server_t *server;

  server = (server_t *) l_malloc(sizeof(server_t));
  server->nick = NULL;
  server->passwd = NULL;
  server->address = NULL;
  server->description = NULL;
  server->meta = s->meta;
  server->network = s->network;
  server->major = s->major;
  server->minor = s->minor;
  server->port = s->port;
  if (s->nick)
    server->nick = l_strdup(s->nick);
  if (s->passwd)
    server->passwd = l_strdup(s->passwd);
  if (s->address)
    server->address = l_strdup(s->address);
  if (s->description)
    server->description = l_strdup(s->description);

  return server;
}

void server_destroy(server_t * server)
{
  if (server->nick)
    l_free(server->nick);
  if (server->passwd)
    l_free(server->passwd);
  l_free(server);
}

void server_set_nick(server_t * server, char *nick)
{
  if (!server || !nick)
    return;

  if (server->nick)
    l_free(server->nick);
  if (strlen(nick) < 1)
    server->nick = l_strdup(global.user.username);
  else
    server->nick = l_strdup(nick);
}

void server_set_passwd(server_t * server, char *passwd)
{
  if (!server || !passwd)
    return;

  if (server->passwd)
    l_free(server->passwd);
  if (strlen(passwd) < 1)
    server->passwd = l_strdup(global.user.password);
  else
    server->passwd = l_strdup(passwd);
}

void server_set_address(server_t * server, char *address)
{
  if (!server || !address)
    return;

  if (server->address)
    l_free(server->address);
  server->address = l_strdup(address);
}

void server_set_description(server_t * server, char *description)
{
  if (!server || !description)
    return;

  if (server->description)
    l_free(server->description);
  server->description = l_strdup(description);
}


gint
server_compare(GtkCList * clist, gconstpointer ptr1, gconstpointer ptr2)
{

  char *text1 = NULL;
  char *text2 = NULL;
  long u1, u2;
  double d1, d2;

  GtkCListRow *row1 = (GtkCListRow *) ptr1;
  GtkCListRow *row2 = (GtkCListRow *) ptr2;

  text1 = GTK_CELL_TEXT(row1->cell[clist->sort_column])->text;
  text2 = GTK_CELL_TEXT(row2->cell[clist->sort_column])->text;

  if (!text2)
    return (text1 != NULL);

  if (!text1)
    return -1;

  if (clist->sort_column == 0) {
    return l_strcasecmp(text1, text2);
  } else if ((clist->sort_column == 2) || (clist->sort_column == 3) ||
	     (clist->sort_column == 4) || (clist->sort_column == 5)) {
    sscanf(text1, "%ld", &u1);
    sscanf(text2, "%ld", &u2);
    if (u1 < u2)
      return -1;
    if (u1 > u2)
      return 1;
    return 0;
  } else if ((clist->sort_column == 6) || (clist->sort_column == 7) ||
	     (clist->sort_column == 8)) {
    sscanf(text1, "%lf", &d1);
    sscanf(text2, "%lf", &d2);
    if (d1 < d2)
      return -1;
    if (d1 > d2)
      return 1;
    return 0;
  } else {
    return 0;
  }
}

server_t *search_server()
{
  GtkWidget *temp;
  server_t *server;
  char *des;
  int port;
  char *text;
  GList *dlist;

  temp = lookup_widget(global.connect_win, "combo_entry12");
  text = l_strdup(gtk_entry_get_text(GTK_ENTRY(temp)));

  if (!l_strcasecmp(_("Last Server"), text) && global.last_server) {
    l_free(text);
    return global.last_server;
  } else {
    des = strrchr(text, ' ');
    *des = 0;
    port = atoi(des + 1);
    des = text;
  }

  for (dlist = global.servers; dlist; dlist = dlist->next) {
    server = (server_t *) (dlist->data);
    if (!l_strcasecmp(server->description, des))
      if (server->port == port) {
	l_free(text);
	return server;
      }
  }
  l_free(text);
  return NULL;
}

void add_server(int meta)
{
  GtkWidget *temp1;
  GtkWidget *temp2;
  server_t *server;
  char *addr = NULL;
  char *des = NULL;
  unsigned long uaddr;
  char t[2048];

  temp1 = lookup_widget(global.win, "entry73");
  addr = gtk_entry_get_text(GTK_ENTRY(temp1));
  temp2 = lookup_widget(global.win, "entry72");
  des = gtk_entry_get_text(GTK_ENTRY(temp2));

  if (!addr || !(*addr)) {
    sprintf(t, _("Enter a server address!\n"));
    temp1 = lookup_widget(global.win, "text14");
    gtk_text_insert(GTK_TEXT(temp1), NULL, NULL, NULL, t, strlen(t));
    return;
  }

  uaddr = resolv_address(addr);
  if (uaddr) {
    server = server_new();
    server_set_address(server, addr);
    server->port =
	gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					 (lookup_widget
					  (global.win, "spinbutton14")));
    server->meta = meta;
    server_set_nick(server, "");
    server_set_passwd(server, "");
    if (strlen(des) > 0) {
      server_set_description(server, des);
    } else {
      server_set_description(server, addr);
    }

    if (!add_server_to_list(server)) {
      // already in list, do not insert
      sprintf(t, _("Server [%s] [%s] already in list\n"),
	      server->address, server->description);
      temp1 = lookup_widget(global.win, "text14");
      gtk_text_insert(GTK_TEXT(temp1), NULL, NULL, NULL, t, strlen(t));
      server_destroy(server);
    } else {
      gtk_entry_set_text(GTK_ENTRY(temp2), "");
      gtk_entry_set_text(GTK_ENTRY(temp1), "");
    }
    write_rc();
  } else {
    sprintf(t, _("Address [%s] is not valid (no internet connection?)\n"),
	    addr);
    temp1 = lookup_widget(global.win, "text14");
    gtk_text_insert(GTK_TEXT(temp1), NULL, NULL, NULL, t, strlen(t));
  }
}

GtkCTreeNode *napigator_search_network(char *network)
{
  GtkCTree *ctree;
  GtkCTreeNode *node;
  char *text;

  ctree = GTK_CTREE(lookup_widget(global.win, "server_ctree"));

  node = GTK_CTREE_NODE(GTK_CLIST(ctree)->row_list);
  while (node) {
    text = GTK_CELL_PIXTEXT
	(GTK_CTREE_ROW(node)->row.cell[ctree->tree_column])->text;
    if (!strcmp(network, text))
      return node;
    node = GTK_CTREE_ROW(node)->sibling;
  }
  return NULL;
}

void napigator_insert_server(char *network, char *server,
			     char *ip, char *port,
			     char *users, char *files, char *gigs)
{
  GtkCTree *ctree;
  GtkCTreeNode *node;
  unsigned long igb, iusers, ifiles;

  ctree = GTK_CTREE(lookup_widget(global.win, "server_ctree"));
  node = napigator_search_network(network);
  if (!node) {
    strcpy(tstr[0], network);
    tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 
      tstr[5][0] = tstr[6][0] = tstr[7][0] = tstr[8][0] = 0;
    node = gtk_ctree_insert_node(ctree, node, NULL, list, 5,
				 global.pix.folder,
				 global.pix.folderb,
				 global.pix.folder_open,
				 global.pix.folder_openb, FALSE, FALSE);
  }
  strcpy(tstr[0], server);
  strcpy(tstr[1], ip);
  strcpy(tstr[2], port);
  strcpy(tstr[3], users);
  strcpy(tstr[4], files);
  strcpy(tstr[5], gigs);
  igb = strtoul(gigs, NULL, 10);
  iusers = strtoul(users, NULL, 10);
  ifiles = strtoul(files, NULL, 10);
  if (iusers) {
    sprintf(tstr[6], "%.1f", (double)igb/(double)iusers);
    sprintf(tstr[7], "%.1f", (double)ifiles/(double)iusers);
  } else {
    sprintf(tstr[6], "n/a");
    sprintf(tstr[7], "n/a");
  }
  if (ifiles)
    sprintf(tstr[8], "%.1f", (double)igb*1024./(double)ifiles);
  else
    sprintf(tstr[8], "n/a");
  gtk_ctree_insert_node(ctree, node, NULL, list, 5,
			NULL, NULL, NULL, NULL, TRUE, FALSE);
}

void napigator_update_network_entry(GtkCTree * ctree, GtkCTreeNode * node)
{
  long users = 0;
  long files = 0;
  long gigs = 0;
  int cnt = 0;
  GtkCTreeNode *childs;
  char *text;

  childs = GTK_CTREE_ROW(node)->children;

  while (childs) {
    gtk_ctree_node_get_text(ctree, childs, 3, &text);
    users += strtoul(text, NULL, 10);
    gtk_ctree_node_get_text(ctree, childs, 4, &text);
    files += strtoul(text, NULL, 10);
    gtk_ctree_node_get_text(ctree, childs, 5, &text);
    gigs += strtol(text, NULL, 10);
    cnt++;
    childs = GTK_CTREE_ROW(childs)->sibling;
  }

  sprintf(tstr[0], "%ld", users / cnt);
  gtk_ctree_node_set_text(ctree, node, 3, tstr[0]);
  sprintf(tstr[0], "%ld", files / cnt);
  gtk_ctree_node_set_text(ctree, node, 4, tstr[0]);
  sprintf(tstr[0], "%ld", gigs / cnt);
  gtk_ctree_node_set_text(ctree, node, 5, tstr[0]);
  if (users)
    sprintf(tstr[0], "%.1f", (double)gigs/(double)users);
  else strcpy(tstr[0], "n/a");
  gtk_ctree_node_set_text(ctree, node, 6, tstr[0]);
  if (users)
    sprintf(tstr[0], "%.1f", (double)files/(double)users);
  else strcpy(tstr[0], "n/a");
  gtk_ctree_node_set_text(ctree, node, 7, tstr[0]);
  if (files)
    sprintf(tstr[0], "%.1f", (double)gigs*1024./(double)files);
  else strcpy(tstr[0], "n/a");
  gtk_ctree_node_set_text(ctree, node, 8, tstr[0]);
}

void napigator_update_network()
{
  GtkCTree *ctree;
  GtkCTreeNode *node;

  ctree = GTK_CTREE(lookup_widget(global.win, "server_ctree"));

  node = GTK_CTREE_NODE(GTK_CLIST(ctree)->row_list);
  while (node) {
    napigator_update_network_entry(ctree, node);
    node = GTK_CTREE_ROW(node)->sibling;
  }
}

void napigator_read(gpointer data, int source, GdkInputCondition condition)
{
  int res;
  char buf[1025];
  char buf2[1025];
  char *pos1;
  char *pos2;
  char *ip;
  char *port;
  char *network;
  char *users;
  char *files;
  char *gigs;
  char *server;
  socket_t *socket = (socket_t *) data;
  static time_t tim = 0;
  
  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, 0);
    return;
  }

  switch (res = recv(source, buf, 1024, MSG_PEEK)) {
  case -1:
    socket_destroy(socket, 0);
    return;
  case 0:
    socket_destroy(socket, 0);
    napigator_update_network();
    return;
  default:
    break;
  }

  buf[res] = 0;
  socket->cnt = 0;
  if (tim == 0) tim = time(NULL);

  pos1 = buf;
  while ((pos2 = strchr(pos1, '\n')) != NULL) {
    res = pos2 - pos1 + 1;
    recv(source, buf2, res, 0);
    buf2[res - 1] = 0;
    //    printf(": [%s]\n", buf2);
    ip = arg(buf2, 0);
    if (!ip)
      goto close;
    port = arg(NULL, 0);
    if (!port)
      goto close;
    network = arg(NULL, 0);
    if (!network)
      goto close;
    users = arg(NULL, 0);
    if (!users)
      goto close;
    files = arg(NULL, 0);
    if (files) {
      gigs = arg(NULL, 0);
      if (!gigs)
	goto close;
      server = arg(NULL, 0);
      if (!server)
	goto close;
      napigator_insert_server(network, server, ip, port,
			      users, files, gigs);
    } else {
      /*
         printf("napi: server: %s users: %s files: %s, gigs: %s\n",
         ip, port, network, users);
       */
    }
    pos1 = pos2 + 1;
    tim = time(NULL);
  }
  
  // close if nothing received for 10 seconds.
  if (tim+10 < global.current_time) goto close;
  return;

close:
  socket_destroy(socket, 0);
}

void napigator_read_header(gpointer data, int source,
			   GdkInputCondition condition)
{
  int res;
  int n;
  char buf[1025];
  char buf2[1025];
  char *pos1;
  char *pos2;
  socket_t *socket = (socket_t *) data;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, 0);
    return;
  }
  switch (res = recv(source, buf, 1024, MSG_PEEK)) {
  case -1:
    socket_destroy(socket, 0);
    return;
  case 0:
    socket_destroy(socket, 0);
    return;
  default:
    break;
  }

  pos1 = buf;
  while ((pos2 = strchr(pos1, '\n')) != NULL) {
    n = pos2 - pos1 + 1;
    recv(source, buf2, n, 0);
    buf2[n - 2] = 0;
    if (strlen(buf2) == 0) {
      gdk_input_remove(socket->input);
      socket->input =
	  gdk_input_add(socket->fd, GDK_INPUT_READ,
			GTK_SIGNAL_FUNC(napigator_read), socket);

      return;
    }
    pos1 = pos2 + 1;
  }
}

void napigator_request(gpointer data, int source,
		       GdkInputCondition condition)
{
  socket_t *socket = (socket_t *) data;
  GtkWidget *temp;
  char t[1024];

  if (condition != GDK_INPUT_WRITE) {
    sprintf(t, _("Could not connect to www.napigator.com\n"));
    temp = lookup_widget(global.win, "text14");
    gtk_text_insert(GTK_TEXT(temp), NULL, NULL, NULL, t, strlen(t));
    socket_destroy(socket, 0);
    return;
  }

  gdk_input_remove(socket->input);

  if (global.napigator.use_proxy) {
    send(source,
	 "GET http://www.napigator.com/servers.php?version=107&client=lopster HTTP/1.0\nHost: www.napigator.com\n\n",
	 strlen
	 ("GET http://www.napigator.com/servers.php?version=107&client=lopster HTTP/1.0\nHost: www.napigator.com\n\n"),
	 0);
  } else {
    send(source,
	 "GET /servers.php?version=107&client=lopster HTTP/1.0\nHost: www.napigator.com\n\n",
	 strlen
	 ("GET /servers.php?version=107&client=lopster HTTP/1.0\nHost: www.napigator.com\n\n"),
	 0);
  }

  socket->input =
      gdk_input_add(source, GDK_INPUT_READ, napigator_read_header, socket);
}

void napigator_get_list()
{
  unsigned long addr;
  GtkWidget *temp;
  char t[1024];

  socket_t *socket = socket_new(S_HTTP);

  if (global.napigator.use_proxy) {
    addr = resolv_address(global.napigator.proxy_url);
    socket->ip_long = addr;
    socket->port = htons(global.napigator.proxy_port);
  } else {
    addr = resolv_address("www.napigator.com");
    if (!addr) {
      socket->ip_long = inet_addr("209.25.178.23");
    } else {
      socket->ip_long = addr;
    }
    socket->port = htons(80);
  }

  if (!connect_socket(socket, "TCP", SOCK_STREAM)) {
    sprintf(t, _("Could not connect to www.napigator.com\n"));
    temp = lookup_widget(global.win, "text14");
    gtk_text_insert(GTK_TEXT(temp), NULL, NULL, NULL, t, strlen(t));
    socket_destroy(socket, NULL);
    return;
  }

  socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_WRITE,
		    GTK_SIGNAL_FUNC(napigator_request), socket);
}

server_t *search_server_in_glist(server_t * server)
{
  server_t *server2;
  GList *dlist;

  if (global.servers) {
    dlist = g_list_first(global.servers);
    while (dlist) {
      server2 = (server_t *) (dlist->data);
      if (!l_strcasecmp(server2->address, server->address))
	if (server2->port == server->port)
	  return server2;
      dlist = g_list_next(dlist);
    }
  }

  return NULL;
}

int add_server_to_list(server_t * server)
{
  GtkWidget *temp;
  int row;

  if (search_server_in_glist(server))
    return 0;

  strcpy(tstr[0], server->address);
  sprintf(tstr[1], "%d", server->port);
  strcpy(tstr[2], server->description);
  sprintf(tstr[3], "%d", server->meta);
  strcpy(tstr[4], server->nick);

  temp = lookup_widget(global.win, "clist11");

  row = gtk_clist_append(GTK_CLIST(temp), list);
  gtk_clist_set_row_data(GTK_CLIST(temp), row, server);

  global.servers = g_list_append(global.servers, server);

  return 1;
}

GtkWidget *create_napigator_popup(int row)
{
  GtkWidget *popup;
  GtkAccelGroup *popup_accels;
  GtkWidget *menu_entry;
  GtkWidget *separator;

  popup = gtk_menu_new();
  gtk_object_set_data(GTK_OBJECT(popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group(GTK_MENU(popup));

  menu_entry = gtk_menu_item_new_with_label(_("Refresh List"));
  gtk_widget_ref(menu_entry);
  gtk_object_set_data_full(GTK_OBJECT(popup), "menu_entry", menu_entry,
			   (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show(menu_entry);
  gtk_container_add(GTK_CONTAINER(popup), menu_entry);
  gtk_signal_connect(GTK_OBJECT(menu_entry), "activate",
		     GTK_SIGNAL_FUNC(on_refresh_activate), NULL);

  if (row >= 0) {
    separator = gtk_menu_item_new();
    gtk_widget_ref(separator);
    gtk_object_set_data_full(GTK_OBJECT(popup), "separator", separator,
			     (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);

    menu_entry = gtk_menu_item_new_with_label(_("Connect"));
    gtk_widget_ref(menu_entry);
    gtk_object_set_data_full(GTK_OBJECT(popup), "menu_entry", menu_entry,
			     (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show(menu_entry);
    gtk_container_add(GTK_CONTAINER(popup), menu_entry);
    gtk_signal_connect(GTK_OBJECT(menu_entry), "activate",
		       GTK_SIGNAL_FUNC(on_connect3_activate), NULL);
    separator = gtk_menu_item_new();
    gtk_widget_ref(separator);
    gtk_object_set_data_full(GTK_OBJECT(popup), "separator", separator,
			     (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);

    menu_entry = gtk_menu_item_new_with_label(_("Add Selected"));
    gtk_widget_ref(menu_entry);
    gtk_object_set_data_full(GTK_OBJECT(popup), "menu_entry", menu_entry,
			     (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show(menu_entry);
    gtk_container_add(GTK_CONTAINER(popup), menu_entry);
    gtk_signal_connect(GTK_OBJECT(menu_entry), "activate",
		       GTK_SIGNAL_FUNC(on_add_servers_activate), NULL);
  }

  separator = gtk_menu_item_new();
  gtk_widget_ref(separator);
  gtk_object_set_data_full(GTK_OBJECT(popup), "separator", separator,
			   (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show(separator);
  gtk_container_add(GTK_CONTAINER(popup), separator);
  gtk_widget_set_sensitive(separator, FALSE);

  menu_entry = gtk_menu_item_new_with_label(_("Customize List"));
  gtk_widget_ref(menu_entry);
  gtk_object_set_data_full(GTK_OBJECT(popup), "menu_entry", menu_entry,
			   (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show(menu_entry);
  gtk_container_add(GTK_CONTAINER(popup), menu_entry);
  gtk_signal_connect(GTK_OBJECT(menu_entry), "activate",
		     GTK_SIGNAL_FUNC(on_customize_list_activate), NULL);

  return popup;
}

GtkWidget *create_server_popup(int row)
{
  GtkWidget *popup;
  GtkAccelGroup *popup_accels;
  GtkWidget *menu_entry;
  GtkWidget *separator;


  popup = gtk_menu_new();
  gtk_object_set_data(GTK_OBJECT(popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group(GTK_MENU(popup));

  if ((row >= 0) && (global.status.connection != 1)) {
    if (global.status.connection > 1)
      menu_entry = gtk_menu_item_new_with_label(_("Reconnect"));
    else
      menu_entry = gtk_menu_item_new_with_label(_("Connect"));
    gtk_widget_ref(menu_entry);
    gtk_object_set_data_full(GTK_OBJECT(popup), "menu_entry", menu_entry,
			     (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show(menu_entry);
    gtk_container_add(GTK_CONTAINER(popup), menu_entry);
    gtk_signal_connect(GTK_OBJECT(menu_entry), "activate",
		       GTK_SIGNAL_FUNC(on_connect2_activate), NULL);

    separator = gtk_menu_item_new();
    gtk_widget_ref(separator);
    gtk_object_set_data_full(GTK_OBJECT(popup), "separator", separator,
			     (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);
  }

  menu_entry = gtk_menu_item_new_with_label(_("Delete Selected"));
  gtk_widget_ref(menu_entry);
  gtk_object_set_data_full(GTK_OBJECT(popup), "menu_entry", menu_entry,
			   (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show(menu_entry);
  gtk_container_add(GTK_CONTAINER(popup), menu_entry);
  gtk_signal_connect(GTK_OBJECT(menu_entry), "activate",
		     GTK_SIGNAL_FUNC(on_delete_selected_activate), NULL);

  separator = gtk_menu_item_new();
  gtk_widget_ref(separator);
  gtk_object_set_data_full(GTK_OBJECT(popup), "separator", separator,
			   (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show(separator);
  gtk_container_add(GTK_CONTAINER(popup), separator);
  gtk_widget_set_sensitive(separator, FALSE);

  menu_entry = gtk_menu_item_new_with_label(_("Customize List"));
  gtk_widget_ref(menu_entry);
  gtk_object_set_data_full(GTK_OBJECT(popup), "menu_entry", menu_entry,
			   (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show(menu_entry);
  gtk_container_add(GTK_CONTAINER(popup), menu_entry);
  gtk_signal_connect(GTK_OBJECT(menu_entry), "activate",
		     GTK_SIGNAL_FUNC(on_customize_list_activate), NULL);

  return popup;
}


char *get_s1(char *data)
{
  char result[1024];
  char *pos;

  strcpy(result, data);
  pos = arg(result, 0);
  return pos;
}

char *get_s2(char *data)
{
  char result[1024];
  char *pos;

  strcpy(result, data);
  pos = arg(result, 0);
  pos = arg(NULL, 0);
  pos = arg(NULL, 0);
  return pos;
}

link_t *link_search_rec(link_t * link, char *name)
{
  link_t *result;
  GList *dlist;
  link_t *temp;

  if (!link) {
    g_warning("link_search_rec: link == NULL");
    return NULL;
  }
  if (!l_strcasecmp(link->server, name))
    return link;
  for (dlist = link->links; dlist; dlist = dlist->next) {
    temp = (link_t *) (dlist->data);
    result = link_search_rec(temp, name);
    if (result)
      return result;
  }
  return NULL;
}

link_t *link_search(link_t * link, char *name)
{
  GList *dlist;
  link_t *temp;

  for (dlist = link->links; dlist; dlist = dlist->next) {
    temp = (link_t *) (dlist->data);
    if (!strcmp(temp->server, name))
      return temp;
  }
  return NULL;
}

link_t *link_remove(link_t * link, char *name)
{
  link_t *result;
  GList *dlist;
  link_t *temp;

  if (!strcmp(link->server, name))
    return link;

  for (dlist = link->links; dlist; dlist = dlist->next) {
    temp = (link_t *) (dlist->data);
    result = link_remove(temp, name);
    if (result) {
      link->links = g_list_remove(link->links, temp);
      free_links(temp);
      return NULL;
    }
  }
  return NULL;
}

void link_message(char *pre, char *server, int depth)
{
  char *prefix;
  int i1;
  
  chat_print_time_stamp(global.current_page, M_PUBLIC);
  prefix = cparse(global.scheme->client_prefix);
  chat_print("message", prefix);
  for (i1 = 0; i1 < depth; i1++)
    chat_print("message", "  ");
  chat_print("error", "[");
  chat_print("message", pre);
  chat_print("error", "] ");
  chat_print_ln("message", server);
}

void print_links(link_t * link, int depth)
{
  GList *dlist;
  link_t *temp;
  char text[1024];
  char t2[100];
  static int cnt = 0;
  int no_links;

  if (depth == 0) {
    chat_print_time_stamp(global.current_page, M_PUBLIC);
    chat_print("error", _("[Linked servers]\n"));
    cnt = 0;
  }
  if (!link) {
    client_message(NULL, _("No Servers Linked"));
    return;
  }

  *text = 0;

  no_links = g_list_length(link->links);
  if (no_links > 0) {
    if (depth == 0) {
      if (no_links > 1)
	sprintf(t2, "%2d", no_links);
      else
	sprintf(t2, "->");
    } else {
      if (no_links > 0)
	sprintf(t2, "%2d", no_links + 1);
      else
	sprintf(t2, "->");
    }
  } else
    sprintf(t2, "->");
  link_message(t2, link->server, depth);
  cnt++;

  for (dlist = link->links; dlist; dlist = dlist->next) {
    temp = (link_t *) (dlist->data);
    print_links(temp, depth + 1);
  }

  if (depth == 0)
    client_message(_("Sum"), _("%d servers linked!"), cnt - 1);
}

void free_links(link_t * link)
{
  GList *dlist;
  link_t *temp;

  if (!link)
    return;
  l_free(link->server);
  link->server = NULL;
  for (dlist = link->links; dlist; dlist = dlist->next) {
    temp = (link_t *) (dlist->data);
    free_links(temp);
  }
  g_list_free(link->links);
  l_free(link);
}

void on_server_links(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data)
{
  lopster_links((char *) user_data);
}

void on_server_delink(GtkMenuItem * menuitem, gpointer user_data ATTR_UNUSED)
{
  GtkLabel *label;

  label = GTK_LABEL(GTK_BIN(menuitem)->child);
  send_command(CMD_CLIENT_DISCONNECT, label->label);
}

void on_server_shutdown(GtkMenuItem * menuitem, gpointer user_data ATTR_UNUSED)
{
  GtkWidget *win;
  GtkWidget *temp;
  GtkLabel *label;

  win = create_shutdown_win();

  temp = lookup_widget(win, "label529");
  label = GTK_LABEL(GTK_BIN(menuitem)->child);
  gtk_label_set_text(GTK_LABEL(temp), label->label);
  gtk_widget_show(win);

}

void on_server_link(GtkMenuItem * menuitem, gpointer user_data ATTR_UNUSED)
{
  GtkLabel *label;
  GtkWidget *temp;
  char *s1, *s2;
  char *command;

  temp = gtk_object_get_user_data(GTK_OBJECT(menuitem));
  if (temp) {
    label = GTK_LABEL(GTK_BIN(temp)->child);
    s2 = label->label;
  } else {
    s2 = NULL;
  }

  label = GTK_LABEL(GTK_BIN(menuitem)->child);
  s1 = label->label;
  if (s2) {
    command = l_strdup_printf("%s %s", s1, s2);
    send_command(CMD_CLIENT_CONNECT, command);
    l_free(command);
  } else {
    send_command(CMD_CLIENT_CONNECT, s1);
  }
}

void add_server_entry(GtkWidget * popup, link_t * server,
		      GtkSignalFunc func, char *action)
{
  GtkWidget *menu_entry;
  GtkWidget *menu;
  GList *sub;
  link_t *entry;
  char *name;

  if (server->links) {
    if (action) {
      menu_entry = gtk_menu_item_new_with_label(action);
    } else if (server->ping && !func) {
      name = l_strdup_printf("%s (%s)", server->server, server->ping);
      menu_entry = gtk_menu_item_new_with_label(name);
      l_free(name);
    } else {
      menu_entry = gtk_menu_item_new_with_label(server->server);
    }
    gtk_widget_show(menu_entry);
    gtk_container_add(GTK_CONTAINER(popup), menu_entry);

    menu = gtk_menu_new();
    gtk_widget_show(menu);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_entry), menu);

    menu_entry = gtk_menu_item_new_with_label(server->server);
    gtk_widget_show(menu_entry);
    gtk_container_add(GTK_CONTAINER(menu), menu_entry);

    if (func)
      gtk_signal_connect(GTK_OBJECT(menu_entry), "activate", func, NULL);
    menu_entry = gtk_menu_item_new();
    gtk_widget_show(menu_entry);
    gtk_container_add(GTK_CONTAINER(menu), menu_entry);
    gtk_widget_set_sensitive(menu_entry, FALSE);


    for (sub = server->links; sub; sub = sub->next) {
      entry = (link_t *) (sub->data);
      add_server_entry(menu, entry, func, NULL);
    }
  } else if (action) {
    menu_entry = gtk_menu_item_new_with_label(action);
    gtk_widget_show(menu_entry);
    gtk_container_add(GTK_CONTAINER(popup), menu_entry);

    menu = gtk_menu_new();
    gtk_widget_show(menu);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_entry), menu);

    menu_entry = gtk_menu_item_new_with_label(server->server);
    gtk_widget_show(menu_entry);
    gtk_container_add(GTK_CONTAINER(menu), menu_entry);

    if (func)
      gtk_signal_connect(GTK_OBJECT(menu_entry), "activate", func, NULL);
  } else {
    if (server->ping && !func) {
      name = l_strdup_printf("%s (%s)", server->server, server->ping);
      menu_entry = gtk_menu_item_new_with_label(name);
      l_free(name);
    } else
      menu_entry = gtk_menu_item_new_with_label(server->server);
    gtk_widget_show(menu_entry);
    gtk_container_add(GTK_CONTAINER(popup), menu_entry);

    if (func)
      gtk_signal_connect(GTK_OBJECT(menu_entry), "activate", func, NULL);
  }
}

int any_unlinked(link_t * entry1)
{
  GList *dlist2;
  link_t *entry2;

  for (dlist2 = entry1->links; dlist2; dlist2 = dlist2->next) {
    entry2 = (link_t *) (dlist2->data);

    if (!link_search_rec(global.links, entry2->server))
       return 1;
  }
  return 0;
}

int is_local_server(link_t * link)
{
  if (!link || !global.links || strcmp(global.links->server, link->server))
     return 0;

  return 1;
}

void add_server_entry2(GtkWidget * popup, link_t * server)
{
  GtkWidget *menu_entry;
  GtkWidget *temp_entry;
  GtkWidget *menu;
  GList *dlist;
  GList *dlist2;
  link_t *entry1;
  link_t *entry2;
  int cnt = 0;

  if (!server)
    return;

  menu_entry = gtk_menu_item_new_with_label(_("Link Server"));
  gtk_widget_show(menu_entry);
  gtk_container_add(GTK_CONTAINER(popup), menu_entry);

  popup = gtk_menu_new();
  gtk_widget_show(popup);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_entry), popup);

  for (dlist = server->links; dlist; dlist = dlist->next) {
    entry1 = (link_t *) (dlist->data);
    if (!link_search_rec(global.links, entry1->server))
       continue;
    if (!any_unlinked(entry1))
      continue;

    cnt++;
    menu_entry = gtk_menu_item_new_with_label(entry1->server);
    gtk_widget_show(menu_entry);
    gtk_container_add(GTK_CONTAINER(popup), menu_entry);
    temp_entry = menu_entry;

    menu = gtk_menu_new();
    gtk_widget_show(menu);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_entry), menu);

    for (dlist2 = entry1->links; dlist2; dlist2 = dlist2->next) {
      entry2 = (link_t *) (dlist2->data);

      menu_entry = gtk_menu_item_new_with_label(entry2->server);
      gtk_widget_show(menu_entry);
      gtk_container_add(GTK_CONTAINER(menu), menu_entry);
      if (!is_local_server(entry1))
	gtk_object_set_user_data(GTK_OBJECT(menu_entry), temp_entry);

      if (link_search_rec(global.links, entry2->server))
      {
	gtk_widget_set_sensitive(menu_entry, FALSE);
      }
      gtk_signal_connect(GTK_OBJECT(menu_entry), "activate",
			 on_server_link, NULL);
    }
  }
  if (cnt == 0) {
    menu_entry = gtk_menu_item_new_with_label(_("All Linked"));
    gtk_widget_show(menu_entry);
    gtk_container_add(GTK_CONTAINER(popup), menu_entry);
    gtk_widget_set_sensitive(menu_entry, FALSE);
  }
}

void on_show_idle_activate(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data ATTR_UNUSED)
{
  GtkWidget *temp;

  temp = lookup_widget(global.win, "frame320");
  if (GTK_WIDGET_VISIBLE(temp))
    gtk_widget_hide(temp);
  else
    gtk_widget_show(temp);
}

GtkWidget *create_server2_popup(void)
{
  GtkWidget *popup;
  GtkAccelGroup *popup_accels;
  GtkWidget *mode;

  popup = gtk_menu_new();
  gtk_object_set_data(GTK_OBJECT(popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group(GTK_MENU(popup));

  if (global.status.connection < 2) {
    mode = gtk_menu_item_new_with_label(_("Not connected"));
    gtk_widget_show(mode);
    gtk_container_add(GTK_CONTAINER(popup), mode);
    gtk_widget_set_sensitive(mode, FALSE);
  } else if (!opennap_version(0, 0)) {
    mode = gtk_menu_item_new_with_label(_("Opennap only"));
    gtk_widget_show(mode);
    gtk_container_add(GTK_CONTAINER(popup), mode);
    gtk_widget_set_sensitive(mode, FALSE);
  } else {
    if (global.links) {
      if (global.user.level > L_MOD) {
	if (opennap_version(0, 37)) {
	  add_server_entry2(popup, global.allowed_links);

	  mode = gtk_menu_item_new();
	  gtk_widget_show(mode);
	  gtk_container_add(GTK_CONTAINER(popup), mode);
	  gtk_widget_set_sensitive(mode, FALSE);
	}

	add_server_entry(popup, global.links, on_server_delink,
			 _("Delink Server"));

	mode = gtk_menu_item_new();
	gtk_widget_ref(mode);
	gtk_widget_show(mode);
	gtk_container_add(GTK_CONTAINER(popup), mode);
	gtk_widget_set_sensitive(mode, FALSE);

	if (global.user.level > L_ADMIN) {
	  add_server_entry(popup, global.links, on_server_shutdown,
			   _("Shutdown Server"));
	  mode = gtk_menu_item_new();
	  gtk_widget_show(mode);
	  gtk_container_add(GTK_CONTAINER(popup), mode);
	  gtk_widget_set_sensitive(mode, FALSE);
	}
      }

      add_server_entry(popup, global.links, NULL, _("Linked Servers"));

      mode = gtk_menu_item_new();
      gtk_widget_show(mode);
      gtk_container_add(GTK_CONTAINER(popup), mode);
      gtk_widget_set_sensitive(mode, FALSE);
    }

    if (global.links)
      mode = gtk_menu_item_new_with_label(_("Refresh linked servers"));
    else
      mode = gtk_menu_item_new_with_label(_("Get linked servers"));
    gtk_widget_show(mode);
    gtk_container_add(GTK_CONTAINER(popup), mode);
    gtk_signal_connect(GTK_OBJECT(mode), "activate",
		       GTK_SIGNAL_FUNC(on_server_links), NULL);

    mode = gtk_menu_item_new_with_label(_("Show server stats"));
    gtk_widget_show(mode);
    gtk_container_add(GTK_CONTAINER(popup), mode);
    gtk_signal_connect(GTK_OBJECT(mode), "activate",
		       GTK_SIGNAL_FUNC(lopster_sstats), NULL);
  }
  {
    GtkWidget *temp;

    mode = gtk_menu_item_new();
    gtk_widget_show(mode);
    gtk_container_add(GTK_CONTAINER(popup), mode);
    gtk_widget_set_sensitive(mode, FALSE);

    mode = gtk_check_menu_item_new_with_label(_("Show server idle bar"));
    gtk_widget_show(mode);
    temp = lookup_widget(global.win, "frame320");
    if (GTK_WIDGET_VISIBLE(temp)) {
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
    }
    gtk_container_add(GTK_CONTAINER(popup), mode);
    gtk_signal_connect(GTK_OBJECT(mode), "activate",
		       GTK_SIGNAL_FUNC(on_show_idle_activate), NULL);
  }

  return popup;
}

void add_allowed_link(char *server1, char *server2)
{
  link_t *link;
  link_t *link2;

  link = link_search(global.allowed_links, server1);
  if (!link) {
    link = (link_t *) l_malloc(sizeof(link_t));
    link->server = l_strdup(server1);
    link->links = NULL;
    global.allowed_links->links =
	g_list_append(global.allowed_links->links, link);
  }

  if (link_search(link, server2))
    return;

  link2 = (link_t *) l_malloc(sizeof(link_t));
  link2->server = l_strdup(server2);
  link2->links = NULL;


  link->links = g_list_append(link->links, link2);
}

void remove_allowed_link(char *server1 ATTR_UNUSED, char *server2 ATTR_UNUSED)
{
}

void read_allowed_links()
{
  char *File;
  FILE *file;
  char line[500];
  char *server1;
  char *server2;

  File = l_strdup_printf("%s/links.allow", global.options.config_dir);

  global.allowed_links = (link_t *) l_malloc(sizeof(link_t));
  global.allowed_links->server = l_strdup(".");
  global.allowed_links->links = NULL;

  if ((file = fopen(File, "r")) == NULL) {
    l_free(File);
    return;
  }
  l_free(File);

  while (fgets(line, 400, file)) {
    char* cr = strchr(line, '\n');
    if (!cr) continue;
    *cr = 0;
    if ((line[0] == '#') || (line[0] == ' ')) continue;

    server1 = arg(line, 0);
    server2 = arg(NULL, 0);
    
    if (!server1 || !server2) {
      g_warning("unparsable line in links.allow [%s]", line);
      continue;
    }
    add_allowed_link(server1, server2);
    add_allowed_link(server2, server1);
  }
  fclose(file);
}

void write_allowed_links()
{
  char* File;
  FILE *file;
  GList *dlist;
  GList *dlist2;
  link_t *link;
  link_t *link2;

  if (!global.allowed_links)
    return;

  File = l_strdup_printf("%s/links.allow", global.options.config_dir);

  if ((file = fopen(File, "w")) == NULL) {
    g_warning("Could not write links.allow");
    l_free(File);
    return;
  }
  l_free(File);

  for (dlist = global.allowed_links->links; dlist; dlist = dlist->next) {
    link = (link_t *) (dlist->data);
    for (dlist2 = link->links; dlist2; dlist2 = dlist2->next) {
      link2 = (link_t *) (dlist2->data);
      fprintf(file, "%s %s\n", link->server, link2->server);
    }
  }
  fclose(file);
}

server_t *create_server_from_napigator(GtkCTree * ctree,
				       GtkCTreeNode * node)
{
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  guint8 space;
  server_t *server;
  char *text;

  server = server_new();

  gtk_ctree_node_get_pixtext(ctree, node, 0,
			     &text, &space, &pixmap, &bitmap);
  server->address = l_strdup(text);
  server->description = l_strdup(text);

  gtk_ctree_node_get_text(ctree, node, 2, &text);
  server->port = atoi(text);
  server->meta = 0;

  gtk_ctree_node_get_text(ctree, node, 1, &text);
  if (strlen(text) > 0) {
    server_set_nick(server, "");
    server_set_passwd(server, "");

    return server;
  } else {
    server_destroy(server);
    return NULL;
  }
}

void problems_read(gpointer data, int source, GdkInputCondition condition)
{
  GtkText *text;
  int res;
  char buf[1025];
  char buf2[1025];
  char *pos1;
  char *pos2;
  socket_t *socket = (socket_t *) data;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, 0);
    return;
  }
  switch (res = recv(source, buf, 1024, MSG_PEEK)) {
  case -1:
    socket_destroy(socket, 0);
    return;
  case 0:
    socket_destroy(socket, 0);
    return;
  default:
    break;
  }

  buf[res] = 0;
  socket->cnt = 0;

  pos1 = buf;
  while ((pos2 = strchr(pos1, '\n')) != NULL) {
    res = pos2 - pos1 + 1;
    recv(source, buf2, res, 0);
    buf2[res] = 0;
    text = GTK_TEXT(socket->data);
    text_print_text(text, buf2);
    pos1 = pos2 + 1;
  }
  return;
}

void problems_read_header(gpointer data, int source,
			  GdkInputCondition condition)
{
  GtkWidget *win;
  GtkText *text;
  int res;
  int n;
  char buf[1025];
  char buf2[1025];
  char *pos1;
  char *pos2;
  socket_t *socket = (socket_t *) data;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, 0);
    return;
  }
  switch (res = recv(source, buf, 1024, MSG_PEEK)) {
  case -1:
    socket_destroy(socket, 0);
    return;
  case 0:
    socket_destroy(socket, 0);
    return;
  default:
    break;
  }

  pos1 = buf;
  while ((pos2 = strchr(pos1, '\n')) != NULL) {
    n = pos2 - pos1 + 1;
    recv(source, buf2, n, 0);
    buf2[n - 2] = 0;
    if (strlen(buf2) == 0) {
      win = create_information_win();
      gtk_widget_show(win);
      gtk_window_set_title(GTK_WINDOW(win), _("News"));
      text = GTK_TEXT(lookup_widget(win, "text15"));
      gtk_text_set_word_wrap(text, 1);
      text_print_text(text,
		      "$r-------------------------\n News ($b%s$r)\n-------------------------\n\n",
		      VERSION);
      socket->data = text;
      gdk_input_remove(socket->input);
      socket->input =
	  gdk_input_add(socket->fd, GDK_INPUT_READ,
			GTK_SIGNAL_FUNC(problems_read), socket);

      return;
    }
    pos1 = pos2 + 1;
  }
}

void problems_request(gpointer data, int source,
		      GdkInputCondition condition)
{
  socket_t *socket = (socket_t *) data;
  char t[1024];

  client_message(NULL, _("Requesting News........"));

  if (condition != GDK_INPUT_WRITE) {
    client_message(_("Error"),
		   _("Could not connect to lopster.sourceforge.net\n"));
    socket_destroy(socket, 0);
    return;
  }

  gdk_input_remove(socket->input);

  sprintf(t,
	  "GET /news.php3?version=%s HTTP/1.0\nHost: lopster.sourceforge.net\n\n",
	  VERSION);
  send(source, t, strlen(t), 0);

  socket->input =
      gdk_input_add(source, GDK_INPUT_READ, problems_read_header, socket);
}

void problems_get()
{
  long addr;

  socket_t *socket = socket_new(S_HTTP);

  addr = resolv_address("lopster.sourceforge.net");
  if (!addr) {
    client_message(_("Error"),
		   _("Could not resolve lopster.sourceforge.net\n"));
    return;
  }

  socket->ip_long = addr;
  socket->port = htons(80);

  if (!connect_socket(socket, "TCP", SOCK_STREAM)) {
    client_message(_("Error"),
		   _("Could not connect to lopster.sourceforge.net\n"));
    socket_destroy(socket, NULL);
    return;
  }

  socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_WRITE,
		    GTK_SIGNAL_FUNC(problems_request), socket);
}

/////////////////////////////////

void latest_version_read(gpointer data, int source,
			 GdkInputCondition condition)
{
  int res;
  char buf[1025];
  char buf2[1025];
  char *pos1;
  char *pos2;
  char *version;
  socket_t *socket = (socket_t *) data;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, 0);
    return;
  }
  switch (res = recv(source, buf, 1024, MSG_PEEK)) {
  case -1:
    socket_destroy(socket, 0);
    return;
  case 0:
    socket_destroy(socket, 0);
    return;
  default:
    break;
  }

  buf[res] = 0;
  socket->cnt = 0;

  pos1 = buf;
  while ((pos2 = strchr(pos1, '\n')) != NULL) {
    res = pos2 - pos1 + 1;
    recv(source, buf2, res, 0);
    buf2[res - 1] = 0;
    /////
    version = arg(buf2, 0);
    if (version && !l_strcasecmp(version, "Version")) {
      version = arg(NULL, 0);
      if (version) {
	if (socket->data) {
	  check_version_dialog(version);
	} else if (!version_is_up_to_date(version)) {
	  new_version_dialog(version);
	}
	socket_destroy(socket, 0);
	return;
      }
    }
    /////
    pos1 = pos2 + 1;
  }
  return;
}

void latest_version_read_header(gpointer data, int source,
				GdkInputCondition condition)
{
  int res;
  int n;
  char buf[1025];
  char buf2[1025];
  char *pos1;
  char *pos2;
  char *pos3;
  socket_t *socket = (socket_t *) data;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, 0);
    return;
  }
  switch (res = recv(source, buf, 1024, MSG_PEEK)) {
  case -1:
    socket_destroy(socket, 0);
    return;
  case 0:
    socket_destroy(socket, 0);
    return;
  default:
    break;
  }

  pos1 = buf;
  while ((pos2 = strchr(pos1, '\n')) != NULL) {
    n = pos2 - pos1 + 1;
    recv(source, buf2, n, 0);
    buf2[n - 2] = 0;
    if (strlen(buf2) == 0) {
      gdk_input_remove(socket->input);
      socket->input =
	  gdk_input_add(socket->fd, GDK_INPUT_READ,
			GTK_SIGNAL_FUNC(latest_version_read), socket);

      return;
    }
    if (!strncasecmp(buf2, "HTTP", 4)) {
      pos3 = arg(buf2, 0);
      pos3 = arg(NULL, 0);
      if (!pos3 || strcmp(pos3, "200")) {
	//      printf("error getting latest version [%s]\n", pos3?pos3:"???");
	socket_destroy(socket, 0);
	return;
      }
    }
    pos1 = pos2 + 1;
  }
}

void latest_version_request(gpointer data, int source,
			    GdkInputCondition condition)
{
  socket_t *socket = (socket_t *) data;
  char t[1024];

  if (condition != GDK_INPUT_WRITE) {
    socket_destroy(socket, 0);
    return;
  }

  gdk_input_remove(socket->input);

  strcpy(t,
	 "GET /LATEST_VERSION HTTP/1.0\nHost: lopster.sourceforge.net\n\n");
  send(source, t, strlen(t), 0);

  socket->input =
      gdk_input_add(source, GDK_INPUT_READ,
		    latest_version_read_header, socket);
}

void latest_version_get(int just_check)
{
  long addr;

  socket_t *socket = socket_new(S_HTTP);
  socket->data = (void *) just_check;

  addr = resolv_address("lopster.sourceforge.net");
  if (!addr)
    return;

  socket->ip_long = addr;
  socket->port = htons(80);

  if (!connect_socket(socket, "TCP", SOCK_STREAM)) {
    socket_destroy(socket, NULL);
    return;
  }

  socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_WRITE,
		    GTK_SIGNAL_FUNC(latest_version_request), socket);
}
