/* Copyright (C) 2000/2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#include <sys/types.h>
#include <sys/socket.h>
#ifndef CYGWIN
#include <dirent.h>
#include <unistd.h>
#endif
#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "callbacks.h"
#include "connection.h"
#include "global.h"
#include "support.h"
#include "search.h"
#include "browse.h"
#include "share.h"
#include "scheme.h"
#include "md5.h"
#include "handler.h"

#define FRAMES_FLAG     0x0001
#define BYTES_FLAG      0x0002
#define TOC_FLAG        0x0004
#define VBR_SCALE_FLAG  0x0008

/*
  load -------------.
                     >-- commit
  refresh -- remove-
*/

static GList *cur_file = NULL;

gint lib_load_idle(gpointer data);
gint lib_apply_idle(gpointer data);

file_t *lib_search_file(char *longname);


unsigned int get_bits(unsigned char *buffer, int start, int no)
{
  unsigned int res = 0;

  res |= buffer[0] << 24 | buffer[1] << 16 | buffer[2] << 8 | buffer[3];

  res <<= start;
  res >>= (32 - no);
  return res;
}

static int ExtractI4(unsigned char *buf)
{
  int x;
  /* big endian extract */
  x = buf[0];
  x <<= 8;
  x |= buf[1];
  x <<= 8;
  x |= buf[2];
  x <<= 8;
  x |= buf[3];

  return x;
}

char *read_xing(FILE * fd, int offset)
{
  unsigned char buffer[1024];
  unsigned char *buf;

  buf = buffer;
  if ((int)fread(buffer, 1, offset + 120, fd) < offset + 120)
    return NULL;

  buf += offset;
  if (buf[0] != 'X')
    return 0;			/* fail */
  if (buf[1] != 'i')
    return 0;			/* header not found */
  if (buf[2] != 'n')
    return 0;
  if (buf[3] != 'g')
    return 0;

  buf += 4;
  return buf;
}

void file_parse_mp3_header(file_t * file)
{
  FILE *fd;
  unsigned char buffer[4];	// header
  unsigned char *buf;
  int mpeg_version;
  int mpeg_layer;
  int res;
  int index;
  long totalframes;
  int padding;
  int stereo;
  int protection;
  int lsf;
  double bpf;
  static int bs[4] = { 0, 384, 1152, 1152 };
  static int s1[4] = { 0, 48000, 144000, 144000 };
  double tpf;
  int xing_offset;
  int pos;
  short br[2][4][16] = {
    {				// mpeg 2
     {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
     {0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0},	// 
     {0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, 0},	// 
     {0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256, 0}	// 
     },
    {				// mpeg1
     {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
     {0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, 0},	// layer I
     {0, 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384, 0},	// layer II
     {0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 0}	// layer III
     }
  };

  int freq[2][2][4] = {
    {
     {11025, 12000, 8000, 0},	// mpeg 2.5
     {0, 0, 0},			// reserved
     },
    {
     {22050, 24000, 16000, 0},	// mpeg 2
     {44100, 48000, 32000, 0}	// mpeg 1
     },
  };

  if ((fd = fopen(file->longname, "r")) != NULL) {
    if (fread(buffer, sizeof(char), 4, fd) != 4) {
      printf("could not read header [%s]\n", file->longname);
      return;
    }
    while (get_bits(buffer, 0, 11) != 0x7ff) {
      buffer[0] = buffer[1];
      buffer[1] = buffer[2];
      buffer[2] = buffer[3];
      if (fread(&buffer[3], sizeof(char), 1, fd) != 1) {
	printf("could not read header [%s]\n", file->longname);
	return;
      }
    }
  } else {
    //    printf("could not open file for reading header [%s]\n", file->longname);
    return;
  }

  // sync word
  res = get_bits(buffer, 0, 11);

  // sync word
  index = get_bits(buffer, 11, 1);

  // MPEG Version
  mpeg_version = get_bits(buffer, 12, 1);
  if (mpeg_version)
    lsf = 0x0;
  else
    lsf = 0x1;

  // MPEG Layer
  mpeg_layer = get_bits(buffer, 13, 2);
  mpeg_layer = 4 - mpeg_layer;
  if (mpeg_layer > 3)
    return;

  // Protection Bit
  protection = get_bits(buffer, 15, 1);

  // Bitrate
  res = get_bits(buffer, 16, 4);
  if (res > 14)
    return;
  file->bitrate = br[mpeg_version][mpeg_layer][res];

  // Fequency
  res = get_bits(buffer, 20, 2);
  file->frequency = freq[index][mpeg_version][res];
  if ((file->bitrate == 0) || (file->frequency == 0)) {
    //    printf("corrupted header! [%s]\n", file->longname);
    return;
  }
  // Padding Bit
  padding = get_bits(buffer, 22, 1);

  // Private Bit
  res = get_bits(buffer, 23, 1);

  // Mode
  res = get_bits(buffer, 24, 2);
  if (res == 3)
    stereo = 0;
  else
    stereo = 1;

  tpf = (double) bs[mpeg_layer] / (file->frequency << lsf);

  if (mpeg_version) {		/* mpeg1 */
    if (stereo)
      xing_offset = 32;
    else
      xing_offset = 17;
  } else {			/* mpeg2 */
    if (stereo)
      xing_offset = 17;
    else
      xing_offset = 9;
  }

  pos = ftell(fd);
  if ((buf = read_xing(fd, xing_offset))) {
    int head_flags;
    int vbr_scale = -1;
    int bytes;

    head_flags = ExtractI4(buf);
    buf += 4;
    if (head_flags & FRAMES_FLAG) {
      totalframes = ExtractI4(buf);
      buf += 4;
    } else
      totalframes = 1;
    if (head_flags & BYTES_FLAG) {
      bytes = ExtractI4(buf);
      buf += 4;
    } else
      bytes = 1;
    if (head_flags & TOC_FLAG) {
      buf += 100;
    }
    if (head_flags & VBR_SCALE_FLAG) {
      vbr_scale = ExtractI4(buf);
      buf += 4;
    }
    file->bitrate = (gint) ((bytes * 8) / (tpf * totalframes * 1000));
    bpf = (double) (file->size - pos) / totalframes;
    file->vbr = 1;
  } else {
    bpf = file->bitrate * s1[mpeg_layer] / (file->frequency << lsf);
    totalframes = (int) ((file->size - pos) / bpf);
    file->vbr = 0;
  }

  file->duration = (time_t) (totalframes * tpf);

#ifdef SHARE_DEBUG
  printf
      ("mpeg %d layer %d, br %d, freq %d bpf %.1f, frames %ld, duration %d, vbr:%d\n",
       lsf + 1, mpeg_layer, file->bitrate, file->frequency, bpf,
       totalframes, file->duration, file->vbr);
#endif

  // Mode Extention
  res = get_bits(buffer, 26, 2);

  // Copyright
  res = get_bits(buffer, 28, 1);

  // Original Home
  res = get_bits(buffer, 29, 1);

  // Enphasis
  res = get_bits(buffer, 30, 2);

  fclose(fd);
}

void file_parse_header(file_t * file)
{
  char *suf;

  suf = strrchr(file->longname, '.');
  if (suf)
    suf++;

  if (!suf) {
    file->bitrate = 32;
    file->frequency = 32000;
    file->duration = 60;
  } else if (!l_strcasecmp(suf, "MP3")) {
    file_parse_mp3_header(file);
  } else if (!l_strcasecmp(suf, "OGG")) {
    // for now set dummy values
    file->bitrate = 32;
    file->frequency = 32000;
    file->duration = 60;
  } else {
    file->bitrate = 32;
    file->frequency = 32000;
    file->duration = 60;
  }
}

void file_calc_md5(file_t * file)
{
  int i1;
  char str[33];
  char hex[16] = "0123456789abcdef";

  if (file->md5)
    l_free(file->md5);
  file->md5 = NULL;
  if (global.options.dummy_md5) {
    for (i1 = 0; i1 < 32; i1++)
      str[i1] = hex[random() % 16];
    str[32] = 0;
    file->md5 = l_strdup_printf("%s-%ld", str, file->size);
  } else {
    file->md5 = MD5File(file->longname, file->md5);
  }
}

suffix_t *check_suffix(char *name, int mime)
{
  GList *dlist;
  char *suffix;
  suffix_t *suf;

  suffix = strrchr(name, '.');
  if (!suffix)
    return NULL;
  else
    suffix++;

  for (dlist = global.mimetype[mime].suffixes; dlist; dlist = dlist->next) {
    suf = (suffix_t *) (dlist->data);
    if (!l_strcasecmp(suffix, suf->suffix))
      return suf;
  }
  return NULL;
}

/////////////////////////////////////////////////////7

void lib_update_file(file_t * file)
{
  struct stat stats;
  unsigned long size;

  if (stat(file->longname, &stats) < 0) {
    size = file->size;
  } else {
    size = stats.st_size;
  }
  if (file->size != size) {
    global.user.bytes -= file->size;
    file->size = size;
    global.user.bytes += file->size;
    file_parse_header(file);
    file_calc_md5(file);
    if (file->flags & FLAG_SHARED) {
      lib_unshare_file(file);
      lib_share_file(file, 0);
    }      
  }
}

void lib_share_file(file_t * file, int important)
{
  char t2[2048];
  char t[1024];
  suffix_t *suffix;

  if (global.status.connection < 2) return;
  /*
  if (global.options.login_mode & L_NOT_SHARE)
    global.options.login_mode &= (0xff^L_NOT_SHARE);
  */
  suffix = get_suffix(file->winname);
  if ((suffix && suffix->as_mp3) || (!opennap_version(0, 37))) {
    strcpy(t2, "\"");
    strcat(t2, file->winname);
    strcat(t2, "\" ");
    strcat(t2, file->md5);
    if (opennap_version(0, 0))
      sprintf(t, " %ld %d %d %d",
	      file->size, find_valid_bitrate(file->bitrate),
	      file->frequency, file->duration);
    else
      sprintf(t, " %ld %d %d %d", file->size, file->bitrate,
	      file->frequency, file->duration);
    strcat(t2, t);
    if (important) send_command(CMD_CLIENT_ADD_FILE, t2);
    else send_command_unimportant(CMD_CLIENT_ADD_FILE, t2);
  } else {
    strcpy(t2, "\"");
    strcat(t2, file->winname);
    strcat(t2, "\"");
    sprintf(t, " %ld %s %s",
	    file->size, file->md5,
	    MimeNames_((file->mime_type ==
			MIME_NONE) ? (MIME_APPLICATION) : (file->mime_type)));
    strcat(t2, t);
    if (important) send_command(CMD_CLIENT_SHARE_FILE, t2);
    else send_command_unimportant(CMD_CLIENT_SHARE_FILE, t2);
  }
  file->flags &= (0xff ^ FLAG_TO_SHARE);
  file->flags |= FLAG_SHARED;
  global.user.unshared--;
}


int lib_share_timeout(gpointer data ATTR_UNUSED) {
  GList* dlist;
  file_t* file;

  if (lib_is_busy()) return 1;

  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = dlist->data;
    if (file->flags & FLAG_TO_SHARE) {
      lib_share_file(file, 0);
      lib_tree_update_file_fb(file, NULL);
      lib_update_stats();
    }
  }
  return 0;
}

void lib_unshare_file(file_t * file)
{
  char *command;

#ifdef SHARE_DEBUG
  printf("unshare [%s]\n", file->longname);
#endif
  if (global.status.connection > 1) {
    /*
    if (opennap_version(0, 37) || !opennap_version(0, 0))
      command = l_strdup_printf("\"%s\"", file->winname);
    else
    */
      command = l_strdup_printf("%s", file->winname);
    send_command_unimportant(CMD_CLIENT_REMOVE_FILE, command);
    l_free(command);
  }
  file->flags &= (0xff ^ FLAG_SHARED);
  file->flags &= (0xff ^ FLAG_TO_SHARE);
  file->flags &= (0xff ^ FLAG_TO_UNSHARE);
  global.user.unshared++;
}

void lib_remove_file(file_t * file)
{

#ifdef SHARE_DEBUG
  printf("removing [%s]\n", file->longname);
#endif
  if (file->flags & FLAG_SHARED) lib_unshare_file(file);

  global.user.unshared--;
  global.user.shared--;
  global.user.bytes -= file->size;
  global.user.files = g_list_remove(global.user.files, file);
  destroy_file_row(file);
}

void lib_delete_file(file_t * file)
{
#ifdef SHARE_DEBUG
  printf("deleting [%s]\n", file->longname);
#endif
  unlink(file->longname);
  lib_remove_file(file);
}

void lib_insert_file(file_t * file)
{
  if (!file) return;

#ifdef SHARE_DEBUG
  printf("adding [%s] %lu\n", file->longname, file->size);
#endif
  global.user.shared++;
  global.user.unshared++;
  global.user.bytes += file->size;
  if ((global.options.login_mode & L_NOT_SHARE) == 0) {
    file->flags |= FLAG_TO_SHARE;
  }

  global.user.files = g_list_prepend(global.user.files, file);
}

void lib_apply_file(file_t * file) {

  lib_update_file(file);
  
  if (file->flags & FLAG_TO_DELETE)
    lib_delete_file(file);
  else if (file->flags & FLAG_TO_REMOVE)
    lib_remove_file(file);
  else if (file->flags & FLAG_TO_UNSHARE)
    lib_unshare_file(file);
  else if (file->flags & FLAG_TO_SHARE)
    lib_share_file(file, 0);
  lib_update_stats();
}


int add_shared_directory(int mime, char *dirname)
{
  DIR *dir;
  struct dirent *entry;
  struct stat buf;
  char name[2048];
  file_t *file;

#ifdef SHARE_DEBUG
  printf("adding dir [%s] (add_shared)\n", dirname);
#endif

  if ((dir = opendir(dirname)) == NULL) {
#ifdef SHARE_DEBUG
    printf("could not open dir [%s] (add_shared)\n", dirname);
#endif
    return 0;
  }

  while ((entry = readdir(dir)) != NULL) {
    if (!strncmp(entry->d_name, ".", 1))
      continue;
    if (!strncmp(entry->d_name, "..", 2))
      continue;

    strcpy(name, dirname);
    strcat(name, "/");
    strcat(name, entry->d_name);
    stat(name, &buf);
    if (buf.st_mode & S_IFDIR) {
      // links are followed
      //      printf("mode : %s:%x:%x\n", name, buf.st_mode, S_IFLNK);
      //      if ((buf.st_mode&S_IFLNK) == 0)
      add_shared_directory(mime, name);
    } else {
      if ((mime == MIME_NONE) || check_suffix(entry->d_name, mime)) {
	file = lib_search_file(name);
	if (!file) {
	  file = file_create_from_local(name);
	  if (file)
	    lib_insert_file(file);
	} else {
#ifdef SHARE_DEBUG
	  printf("updating [%s]\n", file->longname);
#endif
	  file->flags &= 0xff ^ FLAG_TO_REMOVE;
	}
      }
      if (gtk_events_pending())
	gtk_main_iteration();
    }
  }
  closedir(dir);

  return 1;
}

void lib_refresh()
{
  int i1 = 0;
  GList *dlist = NULL;
  char *text;

  if (lib_is_busy()) return;
  lib_enter(_("Refreshing Library"));
#ifdef SHARE_DEBUG
  printf("refresh\n");
#endif
  lib_set_flag(FLAG_TO_REMOVE, NULL);

  for (i1 = 0; i1 < MIME_SIZE; i1++) {
    for (dlist = global.mimetype[i1].shared; dlist; dlist = dlist->next) {
      text = (char *) (dlist->data);
#ifdef SHARE_DEBUG
      printf("adding [%s]\n", text);
#endif
      add_shared_directory(i1, text);
    }
  }

  lib_enter(_("Updating Library"));
  gtk_idle_add(lib_apply_idle, NULL);

  return;
}

file_t *file_create_from_local(char *longname)
{
  file_t *new_file;
  char *pos1;
  int mime;

  mime = get_mimetype(longname);
  new_file = (file_t *) l_malloc(sizeof(file_t));

  new_file->mime_type = mime;
  new_file->longname = l_strdup(longname);
  pos1 = extract_short_name(new_file->longname);
  new_file->shortname = l_strdup(pos1);
  new_file->winname = l_strdup(new_file->longname);
  convert_to_win(new_file->winname);

  new_file->size = 0;
  new_file->user = l_strdup("");
  new_file->ip = 0;

  new_file->md5 = l_strdup("00000000000000000000000000000000");

  new_file->bitrate = 32;
  new_file->frequency = 32000;
  new_file->duration = 60;
  new_file->linespeed = 0;
  new_file->flags = 0;
  new_file->local = 1;
  new_file->vbr = 0;

  return new_file;
}

void destroy_dir_row(gpointer data)
{
  char *dir;

  dir = (char *) data;
  l_free(dir);
}

void lib_insert_file_real(file_t * file, GtkCTree* ctree)
{
  GtkWidget *temp;
  char *text;
  char t[500];
  char *pos;
  GtkCTreeNode *node;
  GtkCTreeNode *node2 = NULL;
  char dir[2048];
  int tt;
  style_t* style;

  //----------------------- new hotlist ctree -------------------------

  if (!file && !ctree) return;
  
  tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 0;

  temp = lookup_widget(global.win, "checkbutton19");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp))) {
    strcpy(t, file->longname);
    dir[0] = 0;
  } else {
    strcpy(t, file->shortname);
    tt = strlen(file->longname) - strlen(file->shortname);
    strncpy(dir, file->longname, tt);
    dir[tt] = 0;
    if (tt > 0 && dir[tt - 1] == '/')
      dir[tt - 1] = 0;
  }

  /////////////////
  pos = strtok(t, "/");
  node = NULL;
  while (pos) {
    if (strlen(pos) <= 0) {
      pos = strtok(NULL, "/");
      continue;
    }
    node2 = node;
    if (!node)
      node = GTK_CTREE_NODE(GTK_CLIST(ctree)->row_list);
    else
      node = GTK_CTREE_ROW(node)->children;
    while (node) {
      text = GTK_CELL_PIXTEXT
	  (GTK_CTREE_ROW(node)->row.cell[ctree->tree_column])->text;
      if (strcmp(pos, text) == 0)
	break;
      node = GTK_CTREE_ROW(node)->sibling;
    }
    if (!node)
      break;
    if (GTK_CTREE_ROW(node)->is_leaf)
      break;
    strcat(dir, "/");
    strcat(dir, pos);
    pos = strtok(NULL, "/");
  }

  // do not insert identical files
  /*
     if (node && GTK_CTREE_ROW (node)->is_leaf) {
     file2 = (file_t*)gtk_ctree_node_get_row_data(ctree, node);    
     if (!strcmp(file2->longname, file->longname)) return 0;
     }
   */

  node = node2;
  do {
    strcpy(tstr[0], pos);
    strcat(dir, "/");
    strcat(dir, pos);
    pos = strtok(NULL, "/");
    if (pos) {
      node =
	  gtk_ctree_insert_node(ctree, node, NULL, list, 5,
				global.pix.folder,
				global.pix.folderb,
				global.pix.folder_open,
				global.pix.folder_openb, FALSE, FALSE);
      gtk_ctree_node_set_row_data_full(ctree, node, (gpointer) l_strdup(dir),
				       (GtkDestroyNotify) destroy_dir_row);
    } else {
      sprintf(tstr[1], "%.2f MB", (double) (file->size) / 1024 / 1024);
      if (file->vbr)
	sprintf(tstr[2], "av. %d", file->bitrate);
      else
	sprintf(tstr[2], "%d", file->bitrate);
      sprintf(tstr[3], "%d", file->frequency);
      print_time_short(tstr[4], file->duration);

      node = gtk_ctree_insert_node(ctree,
				   node, NULL, list, 5,
				   NULL, NULL, NULL, NULL, TRUE, FALSE);

      style = style_get(global.scheme, "library_unshared");
      if (style) {
	if (file->flags & FLAG_SHARED) {
	  gtk_ctree_node_set_foreground(ctree, node, NULL);
	  gtk_ctree_node_set_background(ctree, node, NULL);
	} else {
	  gtk_ctree_node_set_foreground(ctree, node, style->fore);
	  gtk_ctree_node_set_background(ctree, node, style->back);
	}
      }

      gtk_ctree_node_set_row_data(ctree, node, (gpointer) file);
    }
  } while (pos);
}

void lib_show() {
  GtkCTree *ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  GList* dlist;
  file_t* file;

  gtk_clist_clear(GTK_CLIST(ctree));
  gtk_clist_freeze(GTK_CLIST(ctree));
  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = dlist->data;
    lib_insert_file_real(file, ctree);
  }
  gtk_clist_thaw(GTK_CLIST(ctree));
  gtk_ctree_sort_recursive(ctree, NULL);
}

void lib_tree_update_file_fb(file_t * file, GtkCTreeNode* node)
{
  GtkCTree *ctree;
  style_t *style;

  style = style_get(global.scheme, "library_unshared");
  if (!style) return;

  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  if (!node) node = gtk_ctree_find_by_row_data(ctree, NULL, file);
  if (node == NULL) return;

  if (file->flags & FLAG_SHARED) {
    gtk_ctree_node_set_foreground(ctree, node, NULL);
    gtk_ctree_node_set_background(ctree, node, NULL);
  } else {
    gtk_ctree_node_set_foreground(ctree, node, style->fore);
    gtk_ctree_node_set_background(ctree, node, style->back);
  }
  
}

void lib_tree_update_file(file_t * file, GtkCTreeNode* node)
{
  GtkCTree *ctree;
  style_t *style;

  style = style_get(global.scheme, "library_unshared");

  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  if (!node) node = gtk_ctree_find_by_row_data(ctree, NULL, file);
  if (node == NULL)
    return;

  sprintf(tstr[0], "%.2f MB", (double) (file->size) / 1024 / 1024);
  gtk_ctree_node_set_text(ctree, node, 1, tstr[0]);
  if (file->vbr)
    sprintf(tstr[0], "av. %d", file->bitrate);
  else
    sprintf(tstr[0], "%d", file->bitrate);
  gtk_ctree_node_set_text(ctree, node, 2, tstr[0]);
  sprintf(tstr[0], "%d", file->frequency);
  gtk_ctree_node_set_text(ctree, node, 3, tstr[0]);
  sprintf(tstr[0], "%d:%s%d", file->duration / 60,
	  (file->duration % 60) < 10 ? "0" : "", file->duration % 60);
  gtk_ctree_node_set_text(ctree, node, 4, tstr[0]);

  if (!style)
    return;

  if (file->flags & FLAG_SHARED) {
    gtk_ctree_node_set_foreground(ctree, node, NULL);
    gtk_ctree_node_set_background(ctree, node, NULL);
  } else {
    gtk_ctree_node_set_foreground(ctree, node, style->fore);
    gtk_ctree_node_set_background(ctree, node, style->back);
  }
  
}

void rec_apply_node(GtkCTree * ctree, GtkCTreeNode * node, gpointer data ATTR_UNUSED) {
  file_t *file;

  if (!GTK_CTREE_ROW(node)->is_leaf)
    return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  if (file) {
    lib_apply_file(file);
    lib_tree_update_file_fb(file, node);
  }
}

void lib_enter(char* info) {
  GtkWidget* temp;
  GtkCTree *ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));

  gtk_clist_clear(GTK_CLIST(ctree));

  tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = 0;
  strcpy(tstr[0], info);
  gtk_clist_append(GTK_CLIST(ctree), list);

  global.status.building = 1;
  temp = lookup_widget(global.win, "button232");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.win, "button231");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.win, "button272");
  gtk_widget_set_sensitive(temp, FALSE);
  temp = lookup_widget(global.win, "button273");
  gtk_widget_set_sensitive(temp, FALSE);
}

int lib_is_busy() {
  if (global.status.building) return 1;
  else return 0;
}
void lib_leave() {
  GtkWidget* temp;

  global.status.building = 0;
  temp = lookup_widget(global.win, "button232");
  gtk_widget_set_sensitive(temp, TRUE);
  temp = lookup_widget(global.win, "button231");
  gtk_widget_set_sensitive(temp, TRUE);
  temp = lookup_widget(global.win, "button272");
  gtk_widget_set_sensitive(temp, TRUE);
  temp = lookup_widget(global.win, "button273");
  gtk_widget_set_sensitive(temp, TRUE);
}

void lib_apply_node(GtkCTreeNode * node) {
  GtkCTree *ctree;

  if (lib_is_busy()) return;
  ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
  gtk_ctree_post_recursive(ctree, node, (GtkCTreeFunc) rec_apply_node, NULL);
  lib_update_stats();

  if (global.auto_save) lib_save(global.auto_save);
  else lib_save("temp.list");
}

void rec_set_flag(GtkCTree * ctree, GtkCTreeNode * node, int *data)
{
  file_t *file;

  if (!GTK_CTREE_ROW(node)->is_leaf)
    return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  if (file) file->flags |= (*data);
}

void lib_set_flag(int val, GtkCTreeNode * node)
{
  GtkCTree *ctree;
  GList* dlist;
  file_t* file;

  if (node) {
    ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
    gtk_ctree_post_recursive(ctree, node, (GtkCTreeFunc) rec_set_flag, &val);
  } else {
    for (dlist = global.user.files; dlist; dlist = dlist->next) {
      file = dlist->data;
      if (file) file->flags |= val;
    }
  }
}

void rec_del_flag(GtkCTree * ctree, GtkCTreeNode * node, int *data)
{
  file_t *file;

  if (!GTK_CTREE_ROW(node)->is_leaf)
    return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  if (file) file->flags &= (0xff ^ (*data));
}

void lib_del_flag(int val, GtkCTreeNode * node)
{
  GList* dlist;
  file_t* file;
  GtkCTree *ctree;

  if (node) {
    ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
    gtk_ctree_post_recursive(ctree, node, (GtkCTreeFunc) rec_del_flag, &val);
  } else {
    for (dlist = global.user.files; dlist; dlist = dlist->next) {
      file = dlist->data;
      if (file) file->flags &= (0xff ^ val);
    }
  }
}

void rec_replace_flag(GtkCTree * ctree, GtkCTreeNode * node, int *data)
{
  file_t *file;
  char *vals;

  vals = (char *) data;

  if (!GTK_CTREE_ROW(node)->is_leaf)
    return;

  file = gtk_ctree_node_get_row_data(ctree, node);
  if (file && (file->flags & vals[0])) {
    file->flags &= (0xff ^ (vals[0]));
    file->flags |= vals[1];
  }
}

void lib_replace_flag(int val1, int val2, GtkCTreeNode * node)
{
  GtkCTree *ctree;
  GList* dlist;
  file_t* file;
  char data[2];

  if (node) {
    data[0] = val1;
    data[1] = val2;
    ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
    gtk_ctree_post_recursive(ctree, node,
			     (GtkCTreeFunc) rec_replace_flag, data);
  } else {
    for (dlist = global.user.files; dlist; dlist = dlist->next) {
      file = dlist->data;
      if (file && (file->flags & val1)) {
	file->flags &= (0xff ^ val1);
	file->flags |= val2;
      }
    }
  }
}

void rec_xor_flag(GtkCTree * ctree, GtkCTreeNode * node, int *data)
{
  file_t *file;
  char *vals;

  vals = (char *) data;

  if (!GTK_CTREE_ROW(node)->is_leaf)
    return;
  
  file = gtk_ctree_node_get_row_data(ctree, node);
  if (!file) return;
  if (file->flags & vals[0]) {
    file->flags &= (0xff ^ vals[1]);
  } else {
    file->flags |= vals[1];
  }
}

void lib_xor_flag(int val1, int val2, GtkCTreeNode * node)
{
  GtkCTree *ctree;
  char data[2];
  GList* dlist;
  file_t* file;

  if (node) {
    data[0] = val1;
    data[1] = val2;
    ctree = GTK_CTREE(lookup_widget(global.win, "lib_tree"));
    gtk_ctree_post_recursive(ctree, node, (GtkCTreeFunc) rec_xor_flag, data);
  } else {
    for (dlist = global.user.files; dlist; dlist = dlist->next) {
      file = dlist->data;
      if (file->flags & val1) {
	file->flags &= (0xff ^ val2);
      } else {
	file->flags |= val2;
      }
    }
  }

}

file_t *lib_search_file(char *longname)
{
  GList *dlist;
  file_t *file;

  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t *) (dlist->data);
    if (!strcmp(longname, file->longname))
      return file;
  }
  return NULL;
}

void lib_search(search_t * search)
{
  GList *dlist;
  file_t *file;
  int cnt = 0;

  if (!search->pattern) return;

  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = dlist->data;
    if (search_pattern_fits_file(search->pattern, file, 1)) {
      search_insert_file(search, file);
      cnt++;
    }
    if (search->pattern->max_results && (cnt >= search->pattern->max_results))
      break;
  }

  search_finish(search);
}

void lib_set_auto_save(char* name) {
  GtkWidget* temp;
  char* str;

  if (name) {
    str = l_strdup(name);
    if (global.auto_save) l_free(global.auto_save);
    global.auto_save = l_strdup(str);
    l_free(str);
    temp = lookup_widget(global.win, "entry84");
    gtk_entry_set_text(GTK_ENTRY(temp), global.auto_save);
  } else {
    if (global.auto_save) l_free(global.auto_save);
    global.auto_save = NULL;
  }

}

file_t *load_file_entry(FILE * fd)
{
  char *filename;
  char *md5;
  char *size;
  char *bitrate;
  char *freq;
  char *duration;
  char *mime;

  char line[2048];
  file_t *file;

  if (!fgets(line, 2048 - 1, fd))
    return NULL;

  file = (file_t *) l_malloc(sizeof(file_t));
  filename = arg(line, 0);
  md5 = arg(NULL, 0);
  size = arg(NULL, 0);
  bitrate = arg(NULL, 0);
  freq = arg(NULL, 0);
  duration = arg(NULL, 0);
  mime = arg(NULL, 0);

  if (!filename || !md5 || !size || !bitrate || !freq || !duration)
    return NULL;

  file->longname = l_strdup(filename);
  file->shortname = l_strdup(extract_short_name(file->longname));
  file->winname = l_strdup(file->longname);
  convert_to_win(file->winname);
  file->md5 = l_strdup(md5);
  file->size = strtoul(size, NULL, 10);
  if (!strncmp(bitrate, "vbr", 3)) {
    file->bitrate = atoi(bitrate + 3);
    file->vbr = 1;
  } else {
    file->bitrate = atoi(bitrate);
    file->vbr = 0;
  }
  file->frequency = atoi(freq);
  file->duration = atoi(duration);
  if (!mime)
    file->mime_type = MIME_MP3;
  else
    file->mime_type = atoi(mime);
  file->user = l_strdup("");
  file->ip = 0;
  file->linespeed = 0;
  file->flags = 0;
  file->local = 1;		//local file

  return file;
}

gint lib_load_idle(gpointer data)
{
  FILE *fd = (FILE *) data;
  file_t *file;
  file_t *file2;
  int i1;

#ifdef SHARE_DEBUG
  printf("load idle\n");
#endif

  for (i1 = 0; i1 < 10; i1++) {
    if (!(file = load_file_entry(fd)))
      goto finish;
    file2 = lib_search_file(file->longname);
    if (!file2) {
      lib_insert_file(file);
    } else {
      file2->flags &= 0xff ^ FLAG_TO_REMOVE;
      destroy_file_row(file);
    }
  }

  return 1;

finish:
  fclose(fd);
  gtk_idle_add(lib_apply_idle, NULL);
  
  return 0;
}

int lib_load(char *fname, int add)
{
  char filename[1024];
  FILE *fd = NULL;
  char* str;
  
  if (lib_is_busy()) return 0;
  if (!fname || !(*fname)) return 0;

  str = l_strdup_printf(_("Loading Library [%s]\n"), fname);
  lib_enter(str);
  l_free(str);
  
#ifdef SHARE_DEBUG
  printf("lib_load %d\n", add);
#endif
  sprintf(filename, "%s/share/%s", global.options.config_dir, fname);
  fd = fopen(filename, "r");
  if (!fd) {
    if (!add) lib_set_auto_save(NULL);
    lib_show();
    lib_leave();
    return 0;
  }

  if (!add) {
    lib_set_auto_save(fname);
    lib_set_flag(FLAG_TO_REMOVE, NULL);
  }
  gtk_idle_add(lib_load_idle, (gpointer) fd);
  
  return 1;
}

void lib_save(char *name)
{
  char filename[1024];
  FILE *fd;
  GList *dlist;
  file_t *file;

  if (lib_is_busy()) return;
  if (!name || !(*name)) return;
  
  sprintf(filename, "%s/share/%s", global.options.config_dir, name);

  if ((fd = fopen(filename, "w")) == NULL) {
    g_warning(_("could not save shared files %s"), filename);
    return;
  }
  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t *) (dlist->data);
    fprintf(fd, "\"%s\" %s %ld %s%d %d %d %d\n",
	    file->longname, file->md5, file->size, file->vbr ? "vbr" : "",
	    file->bitrate, file->frequency, file->duration,
	    file->mime_type);
  }
  fclose(fd);
  
  lib_set_auto_save(name);
}

void lib_apply() {
  if (lib_is_busy()) return;
  lib_enter(_("Updating Library"));
  gtk_idle_add(lib_apply_idle, NULL);
}

gint lib_apply_idle(gpointer data ATTR_UNUSED)
{
  file_t *file;
  int i1;

  if (!cur_file) cur_file = global.user.files;

  for (i1 = 0; i1 < 10; i1++) {
    if (!cur_file) break;
    file = cur_file->data;
    cur_file = cur_file->next;
    lib_apply_file(file);
  }
  if (cur_file) return 1;
  
  lib_show();
  lib_leave();

  if (global.auto_save) lib_save(global.auto_save);
  else lib_save("temp.list");

  return 0;
}

void lib_delete(char *name)
{
  char filename[1024];

  if (!name || !(*name))
    return;

  sprintf(filename, "%s/share/%s", global.options.config_dir, name);
  unlink(filename);
}

void on_delete_list_activate(GtkMenuItem * menuitem, gpointer user_data ATTR_UNUSED)
{

  GtkLabel *label;
  char *s1;

  label = GTK_LABEL(GTK_BIN(menuitem)->child);
  s1 = label->label;
  lib_delete(s1);
}

void lib_update_stats()
{
  GtkWidget *temp;
  char comm[1024];

  temp = lookup_widget(global.win, "label233");
  sprintf(comm, "%d/%d", global.user.shared - global.user.unshared,
	  global.user.shared);
  gtk_label_set_text(GTK_LABEL(temp), comm);
  temp = lookup_widget(global.win, "label235");
  sprintf(comm, "%.0f", global.user.bytes / 1048576.);
  gtk_label_set_text(GTK_LABEL(temp), comm);

}

void on_load_list_activate(GtkMenuItem * menuitem, gpointer user_data ATTR_UNUSED)
{

  GtkLabel *label;
  char *s1;

  label = GTK_LABEL(GTK_BIN(menuitem)->child);
  s1 = label->label;
  lib_load(s1, 0);
}

void on_add_list_activate(GtkMenuItem * menuitem, gpointer user_data ATTR_UNUSED)
{

  GtkLabel *label;
  char *s1;

  label = GTK_LABEL(GTK_BIN(menuitem)->child);
  s1 = label->label;
  lib_load(s1, 1);
}

GtkWidget *create_files_popup()
{
  GtkWidget *popup;
  GtkWidget *menu1 = NULL;
  GtkWidget *menu2 = NULL;
  GtkWidget *menu3 = NULL;
  GtkWidget *item;
  GtkWidget *item1;
  GtkWidget *item2;
  GtkWidget *item3;
  GtkAccelGroup *popup_accels;
  char dirname[2048];
  struct dirent *entry;
  struct stat buf;
  char filename[2048];
  DIR *dir;
  int cnt = 0;

  sprintf(dirname, "%s/share", global.options.config_dir);

  popup = gtk_menu_new();
  gtk_object_set_data(GTK_OBJECT(popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group(GTK_MENU(popup));

  item1 = gtk_menu_item_new_with_label(_("Delete List"));
  gtk_widget_show(item1);
  gtk_container_add(GTK_CONTAINER(popup), item1);

  item = gtk_menu_item_new();
  gtk_widget_show(item);
  gtk_widget_set_sensitive(item, FALSE);
  gtk_container_add(GTK_CONTAINER(popup), item);

  item3 = gtk_menu_item_new_with_label(_("Add List"));
  gtk_widget_show(item3);
  if (lib_is_busy()) gtk_widget_set_sensitive(item3, FALSE);
  gtk_container_add(GTK_CONTAINER(popup), item3);

  item2 = gtk_menu_item_new_with_label(_("Load List"));
  gtk_widget_show(item2);
  if (lib_is_busy()) gtk_widget_set_sensitive(item2, FALSE);
  gtk_container_add(GTK_CONTAINER(popup), item2);


  if ((dir = opendir(dirname)) == NULL) {
    g_warning("could not open dir [%s]", dirname);
  } else {
    while ((entry = readdir(dir)) != NULL) {
      //      if (!strncmp(entry->d_name, ".", 1)) continue;
      //      if (!strncmp(entry->d_name, "..", 2)) continue;
      sprintf(filename, "%s/%s", dirname, entry->d_name);
      stat(filename, &buf);
      if (S_ISREG(buf.st_mode)) {
	if (!cnt) {
	  menu1 = gtk_menu_new();
	  gtk_widget_show(menu1);
	  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item1), menu1);
	  menu2 = gtk_menu_new();
	  gtk_widget_show(menu2);
	  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item2), menu2);
	  menu3 = gtk_menu_new();
	  gtk_widget_show(menu3);
	  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item3), menu3);
	}
	item = gtk_menu_item_new_with_label(entry->d_name);
	gtk_widget_show(item);
	gtk_container_add(GTK_CONTAINER(menu1), item);
	gtk_signal_connect(GTK_OBJECT(item), "activate",
			   on_delete_list_activate, NULL);
	item = gtk_menu_item_new_with_label(entry->d_name);
	gtk_widget_show(item);
	gtk_container_add(GTK_CONTAINER(menu2), item);
	gtk_signal_connect(GTK_OBJECT(item), "activate",
			   on_load_list_activate, NULL);
	item = gtk_menu_item_new_with_label(entry->d_name);
	gtk_widget_show(item);
	gtk_container_add(GTK_CONTAINER(menu3), item);
	gtk_signal_connect(GTK_OBJECT(item), "activate",
			   on_add_list_activate, NULL);
	cnt++;
      }
    }
    closedir(dir);
  }

  if (!cnt) {
    gtk_widget_set_sensitive(item1, FALSE);
    gtk_widget_set_sensitive(item2, FALSE);
    gtk_widget_set_sensitive(item3, FALSE);
  }

  return popup;
}

GtkWidget *create_lib_popup2(void)
{
  GtkWidget *lib_popup;
  GtkAccelGroup *lib_popup_accels;
  GtkWidget *item;
  GtkWidget *separator;
  GtkCTree *ctree;
  GtkCTreeNode *node;
  file_t *file;

  ctree = GTK_CTREE(global.popup_list);
  node = gtk_ctree_node_nth(ctree, global.popup_row);
  if (node) {
    file = (file_t *) gtk_ctree_node_get_row_data(ctree, node);
  } else {
    file = NULL;
  }

  lib_popup = gtk_menu_new();
  gtk_container_set_border_width(GTK_CONTAINER(lib_popup), 2);
  lib_popup_accels =
      gtk_menu_ensure_uline_accel_group(GTK_MENU(lib_popup));

  if (file) {
    item = gtk_menu_item_new_with_label(_("Open File"));
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(lib_popup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_play_file2_activate), NULL);

    separator = gtk_menu_item_new();
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(lib_popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);
  }
  item = gtk_menu_item_new_with_label(_("Refresh Shared List"));
  gtk_widget_show(item);
  if (lib_is_busy()) gtk_widget_set_sensitive(item, FALSE);
  gtk_container_add(GTK_CONTAINER(lib_popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_refresh1_activate), NULL);

  separator = gtk_menu_item_new();
  gtk_widget_show(separator);
  gtk_container_add(GTK_CONTAINER(lib_popup), separator);
  gtk_widget_set_sensitive(separator, FALSE);

  if (file) {
    item = gtk_menu_item_new_with_label(_("Remove File"));
    gtk_widget_show(item);
    if (lib_is_busy()) gtk_widget_set_sensitive(item, FALSE);
    gtk_container_add(GTK_CONTAINER(lib_popup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_remove_activate), NULL);

    item = gtk_menu_item_new_with_label(_("Remove All"));
    if (lib_is_busy()) gtk_widget_set_sensitive(item, FALSE);
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(lib_popup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_remove_all_activate), NULL);

    separator = gtk_menu_item_new();
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(lib_popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);

    if (file->flags & FLAG_SHARED) {
      item = gtk_menu_item_new_with_label(_("Unshare File"));
      if (lib_is_busy()) gtk_widget_set_sensitive(item, FALSE);
      gtk_widget_show(item);
      gtk_container_add(GTK_CONTAINER(lib_popup), item);
      gtk_signal_connect(GTK_OBJECT(item), "activate",
			 GTK_SIGNAL_FUNC(on_unshare_activate), NULL);
      
    }
    item = gtk_menu_item_new_with_label(_("Unshare All"));
    if (lib_is_busy()) gtk_widget_set_sensitive(item, FALSE);
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(lib_popup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_unshare_all_activate), NULL);
    separator = gtk_menu_item_new();
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(lib_popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);
    
    if (!(file->flags & FLAG_SHARED)) {
      item = gtk_menu_item_new_with_label(_("Share File"));
      if (lib_is_busy()) gtk_widget_set_sensitive(item, FALSE);
      gtk_widget_show(item);
      gtk_container_add(GTK_CONTAINER(lib_popup), item);
      gtk_signal_connect(GTK_OBJECT(item), "activate",
			 GTK_SIGNAL_FUNC(on_unshare_activate), NULL);
      
    }
    item = gtk_menu_item_new_with_label(_("Share All"));
    gtk_widget_show(item);
    if (lib_is_busy()) gtk_widget_set_sensitive(item, FALSE);
    gtk_container_add(GTK_CONTAINER(lib_popup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_share_all_activate), NULL);
    
    separator = gtk_menu_item_new();
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(lib_popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);
    
    item = gtk_menu_item_new_with_label(_("Delete File (from Disk)"));
    if (lib_is_busy()) gtk_widget_set_sensitive(item, FALSE);
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(lib_popup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_delete_file_activate), NULL);
    
    separator = gtk_menu_item_new();
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(lib_popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);
  }
  item = gtk_menu_item_new_with_label(_("Customize List"));
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(lib_popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_customize_list_activate), NULL);

  return lib_popup;
}

void lib_compare_with_browse(user_t * userinfo)
{
  GList *dlist;
  file_t *file;

  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t *) (dlist->data);
    if (!browse_search_file(userinfo, file->winname) &&
	(file->flags & FLAG_SHARED)) {
      file->flags &= 0xff ^ FLAG_SHARED;
      global.user.unshared++;
      lib_tree_update_file_fb(file, NULL);
    }
  }
  lib_update_stats();
}

file_t *lib_search_transfer(transfer_t * transfer)
{
  GList *dlist;
  file_t *file;

  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = (file_t *) (dlist->data);
    if (!(file->flags & FLAG_TO_SHARE) && !(file->flags & FLAG_SHARED))
      continue;
    if (!strcmp(transfer->winname, file->winname))
      return file;
  }
  return NULL;
}

GList* lib_search_file_type(file_t* file) {
  GList* result = NULL;
  GList *dlist;
  file_t* file2;

  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file2 = (file_t *) (dlist->data);
    if (file2->size == file->size)
      result = g_list_prepend(result, file2);
  }
  return result;
}

void lib_start_sharing() {
  GList* dlist;
  file_t* file;

  if (global.options.login_mode & L_NOT_SHARE) return;

  for (dlist = global.user.files; dlist; dlist = dlist->next) {
    file = dlist->data;
    lib_share_file(file, 1);
    lib_tree_update_file_fb(file, NULL);
    lib_update_stats();
    return;
  }
}
