/* Copyright (C) 2000/2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

// cleaned up

#include <ctype.h>
#include <time.h>
#include <stdio.h>

#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <errno.h>

#ifndef CYGWIN
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/time.h>
#endif

#include <gtk/gtk.h>
#include <gdk/gdkprivate.h>

#include "lopster.h"
#include "global.h"
#include "support.h"
#include "transfer.h"
#include "connection.h"
#include "handler.h"
#include "log.h"
#include "chat.h"
#include "resume.h"
#include "userinfo.h"
#include "scheme.h"
#include "dialog.h"
#include "string_list.h"
#include "whois.h"
#include "share.h"
#include "callbacks.h"
#include "ping.h"
#include "interface.h"
#include "subscription.h"
#include "statistic.h"
#include "browse.h"
#include "search.h"
/*
#include "commands.h"
#include "hotlist.h"
*/

#define PING_LIFE     60
#define SPEED_LIFE    60

void create_percent_win();

const int StatusInfo[S_NUMBER] = {
  T_NONE, T_CURRENT, T_CURRENT, T_NONE,
  T_TRANSFER, T_TRANSFER, T_NONE, T_NONE,
  T_NONE, T_NONE, T_CURRENT, T_CURRENT,
  T_CURRENT, T_CURRENT, T_NONE, T_NONE,
  T_NONE, T_NONE, T_NONE, T_NONE,
  T_NONE, T_NONE
};

const int StatusDist[4][4] = {
  {0, 1, 0, 1},
  {-1, 0, -1, 0},
  {0, 1, 0, 1},
  {-1, 0, -1, 0}
};

char *status_names(int status)
{
  switch (status) {
  case S_INACTIVE:
    return _("Inactive");
  case S_CONNECTING:
    return _("Connecting...");
  case S_INFO:
    return _("Getting info...");
  case S_DOWNLOADING:
    return _("Downloading");
  case S_CANCELED:
    return _("Canceled");
  case S_FINISHED:
    return _("Finished");
  case S_TIMEOUT:
    return _("Timeout");
  case S_REJECT:
    return _("Rejected");
  case S_INCOMPLETE:
    return _("Incomplete");
  case S_UPLOADING:
    return _("Uploading");
  case S_INFO1:
    return _("Getting info");
  case S_INFO2:
    return _("Getting info");
  case S_INFO3:
    return _("Getting info");
  case S_WAITING:
    return _("Waiting");
  case S_FIREWALL:
    return _("Both Firewalled");
  case S_CONERROR:
    return _("Connection Error");
  case S_QUEUED:
    return _("Queued");
  case S_REMOTE:
    return _("Remotely Queued");
  case S_UNAVAILABLE:
    return _("Unavailable");
  case S_RESUME_ERR:
    return _("Resume Error");
  case S_DELETE:
    return _("Deleted");
  case S_IO:
    return _("IO error");
  default:
    return _("unknown state");
  }
}

/////////////////////////////////////

transfer_t *transfer_new()
{
  transfer_t *transfer;
  int i1;

  transfer = (transfer_t *) l_malloc(sizeof(transfer_t));

  transfer->longname = NULL;	//
  transfer->shortname = NULL;	//
  transfer->winname = NULL;	//
  transfer->download_dir = NULL;
  transfer->md5 = NULL;
  transfer->progress = 0;
  transfer->rate = 0;
  transfer->timeleft = 0;
  transfer->start_time = 0;
  transfer->status = S_INACTIVE;
  transfer->size = 0;

  transfer->file = NULL;
  transfer->type = T_UNKNOWN;
  transfer->resume_check = NULL;
  transfer->check_length = 0;
  transfer->resume = NULL;
  transfer->user_info = NULL;
  transfer->is_dcc = FALSE;
  transfer->mime_type = MIME_NONE;
  transfer->start_size = 0;
  transfer->queue = 0;

  for (i1 = 0; i1 < TRANSFER_HISTORY_TIME; i1++)
    transfer->history[i1] = transfer->start_size;
  transfer->hist_pos = 0;
  transfer->hist_cnt = 0;

  return transfer;
}

void transfer_destroy(transfer_t * transfer)
{
  if (!transfer)
    return;
#ifdef TRANSFER_DEBUG
  printf("destroying transfer %p\n", transfer);
#endif
  if (transfer->longname) l_free(transfer->longname);
  if (transfer->shortname) l_free(transfer->shortname);
  if (transfer->winname) l_free(transfer->winname);
  if (transfer->download_dir) l_free(transfer->download_dir);
  if (transfer->md5) l_free(transfer->md5);
  l_free(transfer);
}

void transfer_hide(socket_t * socket)
{
  transfer_t *transfer;

  if (!socket) return;
  if (socket->type != S_TRANSFER) return;

  transfer = socket->data;
  if (!transfer) {
    g_warning("transfer_hide(): no transfer");
    return;
  }
  if (transfer->type == T_DOWNLOAD)
    download_hide(socket);
  else if (transfer->type == T_UPLOAD)
    upload_hide(socket);
  else
    g_warning("transfer_hide(): no valid transfer_status");
}

int transfer_in_progress(transfer_t * transfer)
{
  if ((StatusInfo[transfer->status] == T_CURRENT) ||
      (StatusInfo[transfer->status] == T_TRANSFER))
    return 1;
  else
    return 0;
}

int transfer_status_dist(int s1, int s2)
{
  return StatusDist[StatusInfo[s1]][StatusInfo[s2]];
}

void transfer_status_set(socket_t * socket, int status)
{
  transfer_t *transfer;
  int add;
  style_t *style;
  double val;
  char str[1024];
  char *prefix;
  chat_page_t *page;

  if (!socket)
    return;
  transfer = (transfer_t *) (socket->data);
  if (!transfer)
    return;
  add = transfer_status_dist(transfer->status, status);

#ifdef TRANSFER_DEBUG
  printf("old/new %d/%d :%d\n", transfer->status, status, add);
#endif
  if (transfer->type == T_DOWNLOAD) {
    transfer->user_info->cur[0] += add;
    global.limit.cur_downloads += add;
#ifdef TRANSFER_DEBUG
    printf("down now %d\n", global.limit.cur_downloads);
#endif
    if (status == S_DOWNLOADING) {
      global.statistic.no_download[S_NUMBER]++;
      global.statistic.no_download[S_DOWNLOADING]++;

      send_command(CMD_CLIENT_DOWNLOAD_START, "");
      global.limit.cur_real_downloads++;
      if (transfer->user_info) transfer->user_info->real_cur[0]++;
    }
    if (transfer->status == S_DOWNLOADING) {
      global.statistic.no_download[S_DOWNLOADING]--;
      global.statistic.no_download[status]++;

      send_command(CMD_CLIENT_DOWNLOAD_END, "");
      global.limit.cur_real_downloads--;
      transfer->user_info->real_cur[0]--;
    }
    // test that allows only to request x files if remotely queued
    if (status == S_REMOTE)
      transfer->user_info->remotes++;
    if (transfer->status == S_REMOTE)
      transfer->user_info->remotes--;
    // test end
  } else {
    if (transfer->user_info)
      transfer->user_info->cur[1] += add;
    global.limit.cur_uploads += add;
    if (transfer->size >= (unsigned long) (global.limit.large_size) *1024 * 1024)
      global.limit.cur_large += add;
#ifdef TRANSFER_DEBUG
    printf("up now %d\n", global.limit.cur_uploads);
#endif
    if (status == S_UPLOADING) {
      global.statistic.no_upload[S_NUMBER]++;
      global.statistic.no_upload[S_UPLOADING]++;

      send_command(CMD_CLIENT_UPLOAD_START, "");
      global.limit.cur_real_uploads++;
      if (transfer->user_info) transfer->user_info->real_cur[1]++;
    }
    if (transfer->status == S_UPLOADING) {
      global.statistic.no_upload[S_UPLOADING]--;
      global.statistic.no_upload[status]++;

      send_command(CMD_CLIENT_UPLOAD_END, "");
      global.limit.cur_real_uploads--;
      transfer->user_info->real_cur[1]--;
    }
  }

  if (transfer->type == T_UPLOAD) {
    if (status == S_UPLOADING) {
      log("uploads", LOG_OTHER, "Uploading [%s] to [%s]%s at [%ld]\n",
	  transfer->shortname, transfer->user_info->user,
	  transfer->is_dcc ? " (DCC)" : "", transfer->start_size);

      if ((global.options.no_piping & NOPIPE_UPLOAD) == 0) {
	if (global.options.piping & PIPE_UPLOAD) {
	  page = chat_page_search("Uploads", P_OTHER);
	  if (!page)
	    page = create_other_page("Uploads", _("Uploads"));
	} else {
	  page = global.current_page;
	}
	
	chat_print_time_stamp(page, M_PUBLIC);
	prefix = cparse(global.scheme->client_prefix);
	chat_print_colored(page, M_PUBLIC, "message", prefix);
	chat_print_colored(page, M_PUBLIC, "message", "(");
	chat_print_colored(page, M_PUBLIC, "text", "upload");
	chat_print_colored(page, M_PUBLIC, "message", "!");
	chat_print_colored(page, M_PUBLIC, "text",
			   transfer->start_size ? "resume" : "start");
	chat_print_colored(page, M_PUBLIC, "message", "!");
	print_size(str,
		   transfer->start_size ? transfer->start_size : transfer->size);
	chat_print_colored(page, M_PUBLIC, "text", str);
	chat_print_colored(page, M_PUBLIC, "message", ") ");
	chat_print_colored(page, M_PUBLIC, "user", "<");
	chat_print_colored(page, M_PUBLIC, "nick", transfer->user_info->user);
	chat_print_colored(page, M_PUBLIC, "user", ">");
	chat_print_colored(page, M_PUBLIC, "message", " (");
	chat_print_colored(page, M_PUBLIC, "text", transfer->shortname);
	chat_print_colored(page, M_PUBLIC, "message", ")\n");
      }
    } else if (transfer->status == S_UPLOADING) {
      if (transfer->end_time - transfer->start_time > 0)
	val = (transfer->progress - transfer->start_size) /
	    (transfer->end_time - transfer->start_time);
      else
	val = 0;
      print_speed(str, val, 1);
      log("uploads", LOG_OTHER, "Ending [%s] to [%s]%s :%s: [%ld] (%s)\n",
	  (transfer->shortname) ? transfer->shortname : "???",
	  transfer->user_info->user,
	  transfer->is_dcc ? " (DCC)" : "", status_names(status),
	  transfer->progress - transfer->start_size, str);

      if ((global.options.no_piping & NOPIPE_UPLOAD) == 0) {
	if (global.options.piping & PIPE_UPLOAD) {
	  page = chat_page_search("Uploads", P_OTHER);
	  if (!page)
	    page = create_other_page("Uploads", _("Uploads"));
	} else {
	  page = global.current_page;
	}
	
	chat_print_time_stamp(page, M_PUBLIC);
	prefix = cparse(global.scheme->client_prefix);
	chat_print_colored(page, M_PUBLIC, "message", prefix);
	chat_print_colored(page, M_PUBLIC, "message", "(");
	chat_print_colored(page, M_PUBLIC, "text", "upload");
	chat_print_colored(page, M_PUBLIC, "message", "!");
	chat_print_colored(page, M_PUBLIC, "text", status_names(status));
	chat_print_colored(page, M_PUBLIC, "message", "!");
	chat_print_colored(page, M_PUBLIC, "text", str);
	chat_print_colored(page, M_PUBLIC, "message", ") ");
	chat_print_colored(page, M_PUBLIC, "user", "<");
	chat_print_colored(page, M_PUBLIC, "nick", transfer->user_info->user);
	chat_print_colored(page, M_PUBLIC, "user", ">");
	chat_print_colored(page, M_PUBLIC, "message", " (");
	chat_print_colored(page, M_PUBLIC, "text", transfer->shortname);
	chat_print_colored(page, M_PUBLIC, "message", ")\n");
      }
    }
  } else if (transfer->type == T_UNKNOWN) {
    log("unknown_transfer", LOG_OTHER, "Ending [%s] [%s]%s\n",
	transfer->shortname, transfer->user_info->user,
	transfer->is_dcc ? " (DCC)" : "");
  } else {
    if (status == S_DOWNLOADING) {
      transfer->resume->last_access = global.current_time;
      if (!transfer->resume->user || !(transfer->resume->user[0])) {
	if (transfer->resume->user) l_free(transfer->resume->user);
	transfer->resume->user = l_strdup(transfer->user_info->user);
	if (transfer->resume->org_file) l_free(transfer->resume->org_file);
	transfer->resume->org_file = l_strdup(transfer->winname);
      }
      resume_save();
      log("downloads", LOG_OTHER,
	  "Downloading [%s] from [%s]%s at [%ld]\n", transfer->shortname,
	  transfer->user_info->user, transfer->is_dcc ? " (DCC)" : "",
	  transfer->start_size);

      if ((global.options.no_piping & NOPIPE_DOWNLOAD) == 0) {
	if (global.options.piping & PIPE_DOWNLOAD) {
	  page = chat_page_search("Downloads", P_OTHER);
	  if (!page)
	    page = create_other_page("Downloads", _("Downloads"));
	} else {
	  page = global.current_page;
	}
	
	chat_print_time_stamp(page, M_PUBLIC);
	prefix = cparse(global.scheme->client_prefix);
	chat_print_colored(page, M_PUBLIC, "message", prefix);
	chat_print_colored(page, M_PUBLIC, "message", "(");
	chat_print_colored(page, M_PUBLIC, "text", "download");
	chat_print_colored(page, M_PUBLIC, "message", "!");
	chat_print_colored(page, M_PUBLIC, "text",
			   transfer->start_size ? "resume" : "start");
	chat_print_colored(page, M_PUBLIC, "message", "!");
	print_size(str,
		   transfer->start_size ? transfer->start_size : transfer->
		   size);
	chat_print_colored(page, M_PUBLIC, "text", str);
	chat_print_colored(page, M_PUBLIC, "message", ") ");
	chat_print_colored(page, M_PUBLIC, "user", "<");
	chat_print_colored(page, M_PUBLIC, "nick", transfer->user_info->user);
	chat_print_colored(page, M_PUBLIC, "user", ">");
	chat_print_colored(page, M_PUBLIC, "message", " (");
	chat_print_colored(page, M_PUBLIC, "text", transfer->shortname);
	chat_print_colored(page, M_PUBLIC, "message", ")\n");
      }
    } else if (transfer->status == S_DOWNLOADING) {
      if (transfer->end_time - transfer->start_time > 0)
	val = (transfer->progress - transfer->start_size) /
	    (transfer->end_time - transfer->start_time);
      else
	val = 0;
      print_speed(str, val, 1);
      log("downloads", LOG_OTHER,
	  "Ending [%s] from [%s]%s :%s: [%ld] (%s)\n",
	  (transfer->shortname) ? transfer->shortname : "???",
	  transfer->user_info->user,
	  transfer->is_dcc ? " (DCC)" : "", status_names(status),
	  transfer->progress - transfer->start_size, str);

      if ((global.options.no_piping & NOPIPE_DOWNLOAD) == 0) {
	if (global.options.piping & PIPE_DOWNLOAD) {
	  page = chat_page_search("Downloads", P_OTHER);
	  if (!page)
	    page = create_other_page("Downloads", _("Downloads"));
	} else {
	  page = global.current_page;
	}
	
	chat_print_time_stamp(page, M_PUBLIC);
	prefix = cparse(global.scheme->client_prefix);
	chat_print_colored(page, M_PUBLIC, "message", prefix);
	chat_print_colored(page, M_PUBLIC, "message", "(");
	chat_print_colored(page, M_PUBLIC, "text", "download");
	chat_print_colored(page, M_PUBLIC, "message", "!");
	if (status == S_IO || status == S_RESUME_ERR)
	  chat_print_colored(page, M_PUBLIC, "error", status_names(status));
	else chat_print_colored(page, M_PUBLIC, "text", status_names(status));
	chat_print_colored(page, M_PUBLIC, "message", "!");
	chat_print_colored(page, M_PUBLIC, "text", str);
	chat_print_colored(page, M_PUBLIC, "message", ") ");
	chat_print_colored(page, M_PUBLIC, "user", "<");
	chat_print_colored(page, M_PUBLIC, "nick", transfer->user_info->user);
	chat_print_colored(page, M_PUBLIC, "user", ">");
	chat_print_colored(page, M_PUBLIC, "message", " (");
	chat_print_colored(page, M_PUBLIC, "text", transfer->shortname);
	chat_print_colored(page, M_PUBLIC, "message", ")\n");
      }
    }
  }

  transfer->status = status;
  transfer_update(socket, 1);

  if (transfer->type == T_UPLOAD) {
    GtkCList *clist;
    int row;

    clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
    row = gtk_clist_find_row_from_data(clist, socket);
    if (row < 0)
      return;

    if ((socket->timeout >= 0) || (transfer->status == S_QUEUED)) {
      style = style_get(global.scheme, "transfer_waiting");
      if (style) {
	gtk_clist_set_background(clist, row, style->back);
	gtk_clist_set_foreground(clist, row, style->fore);
      }
    } else {
      if (transfer_in_progress(transfer)) {
	style = style_get(global.scheme, "transfer_running");
	if (style) {
	  gtk_clist_set_background(clist, row, style->back);
	  gtk_clist_set_foreground(clist, row, style->fore);
	}
      } else {
	gtk_clist_set_background(clist, row, NULL);
	gtk_clist_set_foreground(clist, row, NULL);
      }
    }
  } else {
    GtkCTree *ctree;
    GtkCTreeNode *node;

    if (!transfer->resume || (transfer->resume->socket != socket) ||
	!transfer->resume->ctree || !transfer->resume->node)
      return;
    ctree = transfer->resume->ctree;
    node = transfer->resume->node;

    if (transfer->resume->status != R_ACTIVE) {
      gtk_ctree_node_set_background(ctree, node, NULL);
      gtk_ctree_node_set_foreground(ctree, node, NULL);
    } else if (transfer_in_progress(transfer)) {
      style = style_get(global.scheme, "transfer_running");
      if (style) {
	gtk_ctree_node_set_background(ctree, node, style->back);
	gtk_ctree_node_set_foreground(ctree, node, style->fore);
      }
    } else if (resume_has_potentials(transfer->resume)) {
      style = style_get(global.scheme, "transfer_waiting");
      if (style) {
	gtk_ctree_node_set_background(ctree, node, style->back);
	gtk_ctree_node_set_foreground(ctree, node, style->fore);
      }
    } else {
      gtk_ctree_node_set_background(ctree, node, NULL);
      gtk_ctree_node_set_foreground(ctree, node, NULL);
    }
  }
}

void transfer_connect_and_start(socket_t * socket)
{
  transfer_t *transfer = (transfer_t *) (socket->data);

  transfer_status_set(socket, S_CONNECTING);
  if (!connect_socket(socket, "TCP", SOCK_STREAM)) {
    send_command(CMD_CLIENT_DATA_PORT_ERROR, transfer->user_info->user);
    return;
  }

  socket->input =
    gdk_input_add(socket->fd, GDK_INPUT_READ,
		  GTK_SIGNAL_FUNC(await_conn_ack), socket);
}

socket_t *download_search_mapable(char *user, char *winname)
{
  socket_t *socket;
  transfer_t *transfer;
  GList *dlist;

  //  printf("searching [%s][%s]\n", user, winname);
  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = dlist->data;
    if (socket->type != S_TRANSFER) continue;
    transfer = socket->data;
    if (!transfer) continue;
    if (transfer->type != T_DOWNLOAD) continue;

    //    printf("dest [%s][%s] %d\n", transfer->user, transfer->winname, transfer->status);
    if (!l_strcasecmp(transfer->winname, winname) &&
	!l_strcasecmp(transfer->user_info->user, user) && 
	(!transfer_in_progress(transfer) || (transfer->status == S_WAITING))
	&& (transfer->status != S_FINISHED)
	&& (transfer->status != S_DELETE))
      return socket;
  }

  return NULL;
}


socket_t *upload_search_mapable(char *user, char *winname)
{
  
  socket_t *socket;
  transfer_t *transfer;
  GList *dlist;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    if (socket->type != S_TRANSFER)
      continue;
    transfer = (transfer_t *) (socket->data);
    if (!transfer)
      continue;
    if (transfer->type != T_UPLOAD)
      continue;
    //    printf("dest [%s][%s] %d\n", transfer->user, transfer->winname, transfer->status);
    if (!l_strcasecmp(transfer->winname, winname) &&
	!l_strcasecmp(transfer->user_info->user, user) && 
	(!transfer_in_progress(transfer) || (transfer->status == S_WAITING))
	&& (transfer->status != S_FINISHED)
	&& (transfer->status != S_CANCELED)
	&& (transfer->status != S_DELETE))
      return socket;
  }

  return NULL;
}

/*
  thanks to Matthew Pratt <mattpratt@yahoo.com> for this code
  I made a few changes....
*/
GdkPixmap *transfer_draw_progress(GtkCList * clist, GdkPixmap * pixmap,
				  double percent, int width, int active)
{
  int height;
  char text[128];
  static GdkGC *gc = NULL;
  GdkColor background = { 0, 0xbf00, 0xbf00, 0xbf00 };
  GdkWindowPrivate *temp_pix;
  style_t *style;
  GdkFont *font;

  if (active) style = style_get(global.scheme, "transfer_bar1");
  else style = style_get(global.scheme, "transfer_bar2");
  if (!style) return NULL;

  if (!gdk_color_alloc
      (gtk_widget_get_colormap(GTK_WIDGET(clist)), &background)
      || !gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(clist)),
			  style->fore)
      || !gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(clist)),
			  style->back)) {
    g_error("couldn't allocate colour");
  }

  height = clist->row_height;
  temp_pix = (GdkWindowPrivate *) pixmap;

  if (!pixmap || (temp_pix->width != width) ||
      (temp_pix->height != height)) {
    if (pixmap) {
      gdk_pixmap_unref(pixmap);
      temp_pix = NULL;
      pixmap = NULL;
    }
    pixmap = gdk_pixmap_new(global.win->window, width,
			    clist->row_height, -1);
  }

  if (!gc) gc = gdk_gc_new(global.win->window);

  // draw the upper/left black lines
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
  gdk_draw_rectangle(pixmap, gc, TRUE, 0, 0, width, height);

  // draw the lower/right white lines
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->white_gc);
  gdk_draw_rectangle(pixmap, gc, TRUE, 1, 1, width - 1, height - 1);

  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
  // draw the background (unfilled progress)
  gdk_gc_set_foreground(gc, &background);
  gdk_draw_rectangle(pixmap, gc, TRUE, 1, 1, width - 2, height - 2);

  // draw the actual progress bar
  // draw the lower/right white lines
  if ((int) ((width - 2) * percent) > 0) {
    gdk_gc_copy(gc, GTK_WIDGET(clist)->style->white_gc);
    gdk_draw_rectangle(pixmap, gc, TRUE, 1, 1,
		       (int) ((width - 2) * percent), height - 2);
  }
  // draw the upper/left black lines
  if ((int) ((width - 2) * percent) - 1 > 0) {
    gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
    gdk_draw_rectangle(pixmap, gc, TRUE, 2, 2,
		       (int) ((width - 2) * percent) - 1, height - 3);
  }

  if ((int) ((width - 2) * percent) - 2 > 0) {
    gdk_gc_set_foreground(gc, style->back);
    gdk_draw_rectangle(pixmap, gc, TRUE, 2, 2,
		       (int) ((width - 2) * percent) - 2, height - 4);
  }
  // draw the text on top
  g_snprintf(text, sizeof(text), "%.1f%%", percent * 100);
  gdk_gc_copy(gc, GTK_WIDGET(clist)->style->black_gc);
  gdk_gc_set_foreground(gc, style->fore);

  font = GTK_WIDGET(clist)->style->font;
  gdk_draw_text(pixmap, font, gc,
		(width - gdk_string_width(font, text)) / 2,
		height - font->descent, text, strlen(text));

  return pixmap;
}

void transfer_update(socket_t * socket, int full ATTR_UNUSED)
{
  transfer_t *transfer;

  transfer = (transfer_t *) (socket->data);
  if (transfer->type == T_DOWNLOAD)
    download_update(socket);
  else if (transfer->type == T_UPLOAD)
    upload_update(socket);
}

void transfer_start_queued()
{
  GtkCList* clist;
  GtkCTree* ctree;
  GList *dlist;
  socket_t *socket;
  resume_t *resume;
  transfer_t *transfer;
  int row;
  GtkCTreeNode* node;
  socket_t* winner = NULL;
  int winner_pri = -1;
  int temp;

  // start queued uploads
  if (global.options.advanced_upload & AU_RESTART) {
    clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
    row = 0;
    while (1) {
      if (row >= clist->rows) break;
      socket = gtk_clist_get_row_data(clist, row);
      transfer = (socket->data);
      if ((transfer->status == S_QUEUED) && upload_allowed(transfer, 0)) {
	temp = user_info_priority(transfer->user_info);
	if (temp > winner_pri) {
	  winner_pri = temp;
	  winner = socket;
	}
      }
      row++;
    }
    if (winner) upload_start(winner, FALSE);
  }

  // start queued downloads
  ctree = GTK_CTREE(lookup_widget(global.win, "ctree4"));
  node = GTK_CTREE_NODE (GTK_CLIST (ctree)->row_list);
  while (node) {
    resume = (resume_t *) gtk_ctree_node_get_row_data(ctree, node);
    node = GTK_CTREE_ROW (node)->sibling;
    for (dlist = resume->downloads; dlist; dlist = dlist->next) {
      socket = dlist->data;
      transfer = (transfer_t *) (socket->data);
      // only start one download per turn
      if ((transfer->status == S_QUEUED && download_allowed(socket) &&
	   !transfer->is_dcc)) {
	download_start(socket, FALSE);
	break;
      }
    }
  }

  if (global.options.auto_search) {
    for (dlist = global.incomplete; dlist; dlist = dlist->next) {
      resume = dlist->data;
      if (resume->is_dcc) continue;
      if (!resume->search) resume_search(resume, 1);
    }
  }

  if (!global.options.auto_resume) return;
  
  // start autoresume inactive files
  ctree = GTK_CTREE(lookup_widget(global.win, "ctree4"));
  node = GTK_CTREE_NODE (GTK_CLIST (ctree)->row_list);
  while (node) {
    resume = (resume_t *) gtk_ctree_node_get_row_data(ctree, node);
    node = GTK_CTREE_ROW (node)->sibling;
    if (resume->status != R_ACTIVE) continue;
    if (resume->is_dcc) continue;
    if (resume->socket) continue;
    if (!global.options.autoresume_download && 
	(resume_has_potentials(resume)&5)) continue;

    for (dlist = resume->downloads; dlist; dlist = dlist->next) {
      socket = dlist->data;
      transfer = (transfer_t *) (socket->data);
      if ((transfer->status == S_INACTIVE) && !transfer->is_dcc) {
	// only start one download per turn
	if (string_list_search(LIST_NODOWNLOAD, transfer->user_info->user))
	  socket_end(socket, &(SocketStatus[S_CANCELED]));
	else
	  download_start(socket, FALSE);
	break;
      }
    }
  }

  ctree = GTK_CTREE(lookup_widget(global.win, "ctree5"));
  node = GTK_CTREE_NODE (GTK_CLIST (ctree)->row_list);
  while (node) {
    resume = (resume_t *) gtk_ctree_node_get_row_data(ctree, node);
    node = GTK_CTREE_ROW (node)->sibling;
    if (resume->status != R_ACTIVE) continue;
    if (resume->socket) continue;
    if (resume->is_dcc) continue;

    for (dlist = resume->downloads; dlist; dlist = dlist->next) {
      socket = dlist->data;
      transfer = (transfer_t *) (socket->data);
      if ((transfer->status == S_INACTIVE) && !transfer->is_dcc) {
	// only start one download per turn
	if (string_list_search(LIST_NODOWNLOAD, transfer->user_info->user))
	  socket_end(socket, &(SocketStatus[S_CANCELED]));
	else
	  download_start(socket, FALSE);
	break;
      }
    }
  }
}

void transfer_end(socket_t * socket, int mode)
{
  transfer_t *transfer = (transfer_t *) (socket->data);

  if (!transfer)
    return;
#ifdef TRANSFER_DEBUG
  printf("ending %d [%s][%s]\n", mode,
	 transfer->winname ? transfer->winname : "_unknown_", 
	 (transfer->user_info)?transfer->user_info->user:"_unknown_");
#endif

  transfer->end_time = global.current_time;

  if (transfer->file) {
    fclose(transfer->file);
    transfer->file = NULL;
  }

  if (transfer->type == T_DOWNLOAD) {
    download_end(socket, mode);
  } else if (transfer->type == T_UPLOAD) {
    upload_end(socket, mode);
  }
}

gint transfer_send_info(gpointer data, gint source,
			GdkInputCondition condition)
{
  char *temp_str = NULL;
  socket_t *socket;
  transfer_t *transfer;

  socket = (socket_t *) data;
  transfer = (transfer_t *) (socket->data);

  if (condition != GDK_INPUT_WRITE) {
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  }

  gdk_input_remove(socket->input);

  if (transfer->type == T_DOWNLOAD) {
#ifdef TRANSFER_DEBUG
    printf("sending [GET]\n");
#endif
    send(source, "GET", 3, 0);
    temp_str = l_strdup_printf("%s \"%s\" %lu",
			       SERVER->nick,
			       transfer->winname, transfer->progress);
  } else if (transfer->type == T_UPLOAD) {
    send(source, "SEND", 4, 0);
#ifdef TRANSFER_DEBUG
    printf("sending [SEND]\n");
#endif
    temp_str = l_strdup_printf("%s \"%s\" %lu",
			       SERVER->nick,
			       transfer->winname, transfer->size);
    transfer->progress = 0;
  }
  if (temp_str) {
#ifdef TRANSFER_DEBUG
    printf("sending [%s]\n", temp_str);
#endif
    send(source, temp_str, strlen(temp_str), 0);
    l_free(temp_str);
  } else {
    g_warning("Should not happen: transfer.c");
    return 1;
  }

  transfer_status_set(socket, S_INFO);

  socket->input =
    gdk_input_add(socket->fd, GDK_INPUT_READ,
		    GTK_SIGNAL_FUNC(transfer_get_info), socket);

  return 1;
}

gint transfer_get_info(gpointer data, gint source,
		       GdkInputCondition condition)
{
  socket_t *socket;
  transfer_t *transfer;
  unsigned char buffer[1024];
  int res;
  int cnt;
  int i1;
  unsigned long size;

  socket = (socket_t *) data;
  transfer = (transfer_t *) (socket->data);

  if (condition != GDK_INPUT_READ) {
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  }

  gdk_input_remove(socket->input);

  // 128 bytes should do it, as we are trying to get the filessize.
  switch ((res = recv(source, buffer, 128, MSG_PEEK))) {
  case -1:
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  case 0:
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  default:
    break;
  }

  buffer[res] = 0;
#ifdef TRANSFER_DEBUG
  printf("read %d bytes\n", res);
#endif
  cnt = 0;
  while (isdigit(buffer[cnt]))
    cnt++;
#ifdef TRANSFER_DEBUG
  if (cnt == 0) {
    printf("read [%s]\n", buffer);
  }
#endif

  // now pop the size packet from the socket.
  if (recv(source, buffer, cnt, 0) != cnt) {
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  }
  buffer[cnt] = 0;

  if (transfer->type == T_UPLOAD) {
    transfer->progress = strtoul(buffer, NULL, 10);
#ifdef TRANSFER_DEBUG
    printf("got progress %ld\n", transfer->progress);
#endif
    for (i1 = 0; i1 < TRANSFER_HISTORY_TIME; i1++)
      transfer->history[i1] = transfer->progress;
    transfer->start_size = transfer->progress;
  } else {
    size = strtoul(buffer, NULL, 10);
    if (size == 0) {
#ifdef TRANSFER_DEBUG
      printf("got size 0\n");
#endif
      socket_end(socket, &(SocketStatus[S_REJECT]));
      return 1;
    } else {
      transfer->size = size;
    }
#ifdef TRANSFER_DEBUG
    printf("got size [%ld] [%d]\n", transfer->size, cnt);
#endif
  }

  transfer->start_time = global.current_time;
  if (transfer->type == T_UPLOAD) {
    if (!upload_open_file(transfer)) {
      send(socket->fd, "FILE NOT FOUND", strlen("FILE NOT FOUND"), 0);
      socket_end(socket, &(SocketStatus[S_IO]));
      return 1;
    }
    transfer_status_set(socket, S_UPLOADING);
    socket->input =
	gdk_input_add(socket->fd, GDK_INPUT_WRITE,
		      GTK_SIGNAL_FUNC(upload_push_output), socket);
  } else {
    if (!download_open_file(transfer)) {
      socket_end(socket, &(SocketStatus[S_IO]));
      return 1;
    }
    transfer_status_set(socket, S_DOWNLOADING);
    transfer->type = T_DOWNLOAD;
    socket->input =
	gdk_input_add(socket->fd, GDK_INPUT_READ,
		      GTK_SIGNAL_FUNC(download_get_input), socket);
  }

  return 1;
}

void transfer_detect_user_info(transfer_t * transfer, char* user)
{
  user_info_t *user_info;
  int download = 0;

  if (transfer->type == T_DOWNLOAD)
    download = 1;

  user_info = user_info_search(user, download);

  if (!user_info) {
    user_info = user_info_new(user);
    whois_request(user_info->user, WHOIS_NONE);
  } else if (user_info->timestamp &&
	     (global.current_time - user_info->timestamp > 60*60)) {
    whois_request(user_info->user, WHOIS_NONE);
  }

  transfer->user_info = user_info;
  return;
}

int transfer_to_destroy(socket_t * socket, int mode)
{
  transfer_t *transfer;

  if (socket->timeout != -1)
    return 0;

  transfer = (transfer_t *) (socket->data);
  if (transfer->type == T_UPLOAD) {
    switch (mode) {
    case S_FINISHED:
      if (global.options.ul_autoremove & REMOVE_U_FINISHED)
	return 1;
      break;
    case S_REJECT:
    case S_REMOTE:
    case S_TIMEOUT:
    case S_INCOMPLETE:
    case S_FIREWALL:
    case S_CONERROR:
    case S_UNAVAILABLE:
    case S_RESUME_ERR:
    case S_IO:
    case S_CANCELED:
      if (global.options.ul_autoremove & REMOVE_U_ABORTED)
	return 1;
      break;
    case S_QUEUED:
      if (global.options.ul_autoremove & REMOVE_U_QUEUED)
	return 1;
      break;
    case S_DELETE:
      return 1;
      break;
    }
  } else {
    return 0;
    //    if (mode == S_DELETE) return 1;
  }
  return 0;
}

int transfer_to_retry(int mode)
{
  switch (mode) {
  case S_TIMEOUT:
    if (global.options.auto_retry & RETRY_TIMEOUT)
      return global.options.retry_timeout[0] * 1000;
    break;
  case S_REJECT:
    if (global.options.auto_retry & RETRY_REJECT)
      return global.options.retry_timeout[1] * 1000;
    break;
  case S_INCOMPLETE:
    if (global.options.auto_retry & RETRY_INCOMPLETE)
      return global.options.retry_timeout[2] * 1000;
    break;
  case S_CONERROR:
    if (global.options.auto_retry & RETRY_CONERROR)
      return global.options.retry_timeout[3] * 1000;
    break;
  case S_REMOTE:
    if (global.options.auto_retry & RETRY_REMOTE)
      return global.options.retry_timeout[4] * 1000;
    break;
  case S_UNAVAILABLE:
    if (global.options.auto_retry & RETRY_UNAVAILABLE)
      return global.options.retry_timeout[5] * 1000;
    break;
  case S_QUEUED:
    //    return 1000;
    return -1;
    break;
  }

  return -1;
}

int transfer_grab_resume(socket_t * socket, resume_t * resume)
{
  struct stat st;
  int i1;
  transfer_t *transfer = (transfer_t *) (socket->data);

  if (resume->status != R_ACTIVE)
    return 0;

#ifdef TRANSFER_DEBUG
  printf("s:%p grabbing r:%p\n", socket, resume);
#endif
  transfer->resume = resume;

  if (resume->socket) {
    if (socket != resume->socket) {
#ifdef TRANSFER_DEBUG
      printf("already grabbed\n");
#endif
      return 0;
    } else {
      // already grabbed by this transfer
      return 1;
    }
  }

  resume->socket = socket;

  if (transfer->download_dir)
    l_free(transfer->download_dir);
  if (resume->dirname)
    transfer->download_dir = l_strdup(resume->dirname);
  else
    transfer->download_dir = NULL;

  if (stat(resume->filename, &st) < 0) {
    transfer->progress = 0;
    transfer->start_size = 0;
  } else {
    if (st.st_size > 100)
      transfer->progress = st.st_size - 100;
    else
      transfer->progress = 0;
    transfer->start_size = transfer->progress;
    for (i1 = 0; i1 < TRANSFER_HISTORY_TIME; i1++)
      transfer->history[i1] = transfer->progress;
  }

  resume_update(resume);

  return 1;
}

void transfer_ungrab_resume(socket_t * socket)
{
  resume_t *resume;
  transfer_t *transfer = (transfer_t *) (socket->data);

  if (!transfer->resume) {
    g_warning("nothing grabbed\n");
    return;
  }
#ifdef TRANSFER_DEBUG
  printf("s:%p ungrabbing r:%p\n", socket, transfer->resume);
#endif

  resume = transfer->resume;
  resume->socket = NULL;
  resume_update(resume);
}

resume_t *transfer_create_resume(transfer_t * transfer)
{
  struct stat st;
  char *text;
  resume_t *resume;
  char *filename;
  char* f2;

  text = extract_filename(transfer->longname);

  filename = l_strdup_printf("%s/%s", global.incomplete_path, text);
  convert_local_name(filename);
  if (resume_search_long_file(filename)) {
    // resume entry already exists, renaming
    f2 = rename_file(filename);
    if (!f2) return NULL;
    filename = f2;
  } else if (stat(filename, &st) >= 0) {
    // file exists, using.
  }

  resume = resume_new();
  resume->filename = filename;
  //  printf("download %s\n", resume->filename);
  resume->size = transfer->size;
  if (transfer->download_dir)
    resume->dirname = l_strdup(transfer->download_dir);
  else
    resume->dirname = NULL;
  resume->user = l_strdup(transfer->user_info->user);
  resume->org_file = l_strdup(transfer->winname);
  resume->online = 1;
  resume->is_dcc = transfer->is_dcc;

  resume_show(resume, 2);
  resume_save();
  return resume;
}

int transfer_search_queued(resume_t * resume)
{
  GList *dlist;
  socket_t *socket;
  transfer_t *transfer;
  int result = 0;

  if (!resume->socket)
    return 0;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    if (!socket) {
      printf("socket->data = NULL\n");
      continue;
    }
    if (socket->type != S_TRANSFER)
      continue;
    transfer = (transfer_t *) (socket->data);
    if (!transfer)
      continue;
    if (transfer->type != T_DOWNLOAD)
      continue;
    if ((transfer->resume == resume) && (transfer->status == S_QUEUED))
      result++;
  }
  return result;
}

void transfer_cancel_running()
{
  GList *dlist;
  socket_t *socket;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    if (socket->type != S_TRANSFER)
      continue;
    socket_end(socket, &(SocketStatus[S_CANCELED]));
  }
}

////////////////////
// download stuff

void download_update_download(socket_t * socket)
{
  GtkCTree *ctree;
  GtkCTreeNode *node;
  char str[200];
  double trate;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  int sec;
  transfer_t *transfer;

  if (!socket)
    return;
  transfer = (transfer_t *) (socket->data);
  if (!transfer->resume)
    return;

  ctree = transfer->resume->ctree;
  node = transfer->resume->node;
  if (!ctree || !node)
    return;
  node = gtk_ctree_find_by_row_data(ctree, node, socket);
  if (!node) return;

  // filesize
  sprintf(str, "%ld/%ld", transfer->progress, transfer->size);
  gtk_ctree_node_set_text(ctree, node, 1, str);

  // user
  if (transfer->user_info->user)
    strcpy(str, transfer->user_info->user);
  else
    strcpy(str, _("Not known yet"));
  gtk_ctree_node_set_text(ctree, node, 2, str);

  // status
  if ((transfer->status < 0) || (transfer->status >= S_NUMBER)) {
    g_warning("transfer_update: invalid status %d", transfer->status);
    return;
  }
  if (transfer->status == S_WAITING)
    sprintf(str, _("Waiting %d secs"), (socket->max_cnt - socket->cnt));
  else if (transfer->status == S_REMOTE)
    sprintf(str, _("%d Remotely queued"), transfer->queue);
  else
    strcpy(str, status_names(transfer->status));
  gtk_ctree_node_set_text(ctree, node, 3, str);

  // speed
  gtk_ctree_node_set_text(ctree, node, 4, LineSpeed(transfer->user_info->linespeed));

  // progress
  if ((transfer->status == S_DOWNLOADING) ||
      (transfer->status == S_FINISHED) ||
      (transfer->status == S_UPLOADING)) {
    int last_prog;
    int first_prog;

    if (transfer->size > 0) {
      gtk_ctree_node_get_pixmap(ctree, node, 5, &pixmap, &bitmap);

      pixmap =
	transfer_draw_progress(GTK_CLIST(ctree), pixmap,
			       (double) transfer->progress /
			       (double) transfer->size,
			       GTK_CLIST(ctree)->column[5].width,
			       (transfer->status != S_FINISHED));
      gtk_ctree_node_set_pixmap(ctree, node, 5, pixmap, NULL);
    } else {
      gtk_ctree_node_set_text(ctree, node, 5, "");
    }

    transfer->hist_pos++;
    if (transfer->hist_pos >= TRANSFER_HISTORY_TIME)
      transfer->hist_pos = 0;
    transfer->history[transfer->hist_pos] = transfer->progress;
    if (transfer->hist_cnt <
	(TRANSFER_HISTORY_TIME / global.network.transfer_delay) -1)
      transfer->hist_cnt++;

    last_prog = transfer->hist_pos;
    first_prog = (last_prog + TRANSFER_HISTORY_TIME - transfer->hist_cnt) %
	TRANSFER_HISTORY_TIME;

    if (transfer->status == S_FINISHED) {
      if (transfer->end_time - transfer->start_time == 0)
	transfer->end_time++;
      trate = (transfer->size - transfer->start_size) /
	  (transfer->end_time - transfer->start_time);
    } else {
      trate = (double) (transfer->history[last_prog] -
			transfer->history[first_prog]) /
	  transfer->hist_cnt / global.network.transfer_delay;
    }
#ifdef TRANSFER_DEBUG
    if (trate < 0) {
      printf("%d/%d/%d\n", first_prog, last_prog, TRANSFER_HISTORY_TIME);
    }
#endif
    print_speed(str, (int) trate, 1);

    transfer->rate = trate;
    detect_speed_pixs(trate, &pixmap, &bitmap);
    gtk_ctree_node_set_pixtext(ctree, node, 6, str, 5, pixmap, bitmap);

    if ((transfer->status == S_FINISHED) || (trate > 0)) {
      if (transfer->status == S_FINISHED)
	sec = transfer->end_time - transfer->start_time;
      else
	sec = (int) ((transfer->size - transfer->progress) / trate);
      transfer->timeleft = sec;

      print_time_average(str, sec);
    } else {
      sprintf(str, _("stalled"));
    }
    gtk_ctree_node_set_text(ctree, node, 7, str);
  } else {
    gtk_ctree_node_set_text(ctree, node, 6, "");
    gtk_ctree_node_set_text(ctree, node, 7, "");
  }

  if (socket == transfer->resume->socket) {
    resume_update(transfer->resume);
  }
  return;
}

void download_update_resume(socket_t * socket)
{
  GtkCTree *ctree;
  GtkCTreeNode *node;
  char str[200];
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  transfer_t *transfer;

  if (!socket)
    return;
  transfer = (transfer_t *) (socket->data);
  if (!transfer->resume)
    return;

  ctree = transfer->resume->ctree;
  node = transfer->resume->node;
  if (!ctree || !node)
    return;
  node = gtk_ctree_find_by_row_data(ctree, node, socket);
  if (!node) return;

  // filesize
  sprintf(str, "%ld", transfer->size);
  gtk_ctree_node_set_text(ctree, node, 1, str);

  // user
  if (transfer->user_info->user)
    strcpy(str, transfer->user_info->user);
  else
    strcpy(str, _("Not known yet"));
  gtk_ctree_node_set_text(ctree, node, 2, str);

  // status
  if ((transfer->status < 0) || (transfer->status >= S_NUMBER)) {
    g_warning("transfer_update: invalid status %d", transfer->status);
    return;
  }
  if (transfer->status == S_WAITING)
    sprintf(str, _("Waiting %d secs"), (socket->max_cnt - socket->cnt));
  else if (transfer->status == S_REMOTE)
    sprintf(str, _("%d Remotely queued"), transfer->queue);
  else
    strcpy(str, status_names(transfer->status));
  gtk_ctree_node_set_text(ctree, node, 4, str);

  // progress
  if ((transfer->status == S_DOWNLOADING) ||
      (transfer->status == S_FINISHED) ||
      (transfer->status == S_UPLOADING)) {

    if (transfer->size > 0) {
      gtk_ctree_node_get_pixmap(ctree, node, 3, &pixmap, &bitmap);

      pixmap =
	  transfer_draw_progress(GTK_CLIST(ctree), pixmap,
				 (double) transfer->progress /
				 (double) transfer->size,
				 GTK_CLIST(ctree)->column[3].width,
				 (transfer->status != S_FINISHED));
      gtk_ctree_node_set_pixmap(ctree, node, 3, pixmap, NULL);
    } else {
      gtk_ctree_node_set_text(ctree, node, 3, "");
    }
  }

  return;
}

void download_update(socket_t * socket)
{
  GtkCTree *ctree1;
  GtkCTree *ctree2;
  transfer_t *transfer;
  resume_t* resume;

  if (!socket)
    return;
  transfer = (transfer_t *) (socket->data);
  if (!transfer || !transfer->resume)
    return;
  resume = transfer->resume;
  if (!resume->ctree || !resume->node)
    return;

  ctree1 = GTK_CTREE(lookup_widget(global.win, "ctree4"));
  ctree2 = GTK_CTREE(lookup_widget(global.win, "ctree5"));

  if (resume->ctree == ctree1) {
    download_update_download(socket);
  } else if (resume->ctree == ctree2) {
    download_update_resume(socket);
#ifdef RESUME_DEBUG
  } else {
    g_warning("download_update(): resume not correctly shown");
#endif
  }
  return;
}

socket_t *download_search(char *user, char *winname)
{
  socket_t *socket;
  transfer_t *transfer;
  GList *dlist;

  //  printf("searching [%s][%s]\n", user, winname);
  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    if (socket->type != S_TRANSFER)
      continue;
    transfer = (transfer_t *) (socket->data);
    if (!transfer) {
      g_warning("download_search(): transfer == NULL");
      continue;
    }
    if (transfer->type != T_DOWNLOAD)
      continue;
    //    printf("dest [%s][%s] %d\n", transfer->user, transfer->winname, transfer->status);
    if (!strcmp(transfer->winname, winname) &&
	!strcmp(transfer->user_info->user, user))
      return socket;
  }

  return NULL;
}

GList *download_search_all(char *user, char *winname) {
  socket_t *socket;
  transfer_t *transfer;
  GList *dlist;
  GList* result = NULL;

  //  printf("searching [%s][%s]\n", user, winname);
  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = dlist->data;
    if (socket->type != S_TRANSFER) continue;
    transfer = socket->data;
    if (!transfer) continue;
    if (transfer->type != T_DOWNLOAD) continue;
    if (!strcmp(transfer->winname, winname) &&
	!l_strcasecmp(transfer->user_info->user, user))
      result = g_list_append(result, socket);
  }

  return result;
}

socket_t *download_create(file_t * file, int withdir, resume_t* resume)
{
  transfer_t *new_trans;
  char *dir;
  char *folder;
  socket_t *socket;
  GList* dlist;

  if ((folder = valid_download_folder(file->mime_type)) == NULL) {
    download_dialog(file->shortname, file->mime_type);
    return NULL;
  }

  // only test for existing transfer, if it should be mapped to an existing resume
  if (resume) {
    for (dlist = resume->downloads; dlist; dlist = dlist->next) {
      socket = dlist->data;
      new_trans = socket->data;
      if (!strcmp(new_trans->winname, file->winname) &&
	  !l_strcasecmp(new_trans->user_info->user, file->user))
	return socket;
    }
  }

  new_trans = transfer_new();
  new_trans->longname = l_strdup(file->longname);
  new_trans->shortname = l_strdup(file->shortname);
  new_trans->winname = l_strdup(file->winname);
  new_trans->size = file->size;
  new_trans->type = T_DOWNLOAD;
  new_trans->mime_type = file->mime_type;
  transfer_detect_user_info(new_trans, file->user);
  new_trans->user_info->linespeed = file->linespeed;
  
  socket = socket_new(S_TRANSFER);
  socket->data = new_trans;
  socket->ip_long = file->ip;

  if (withdir) {
    dir = extract_last_dir(new_trans->longname);
    if (dir) {
      new_trans->download_dir = l_strdup(dir);
    } else
      new_trans->download_dir = NULL;
  } else {
    new_trans->download_dir = NULL;
  }
  
#ifdef TRANSFER_DEBUG
  printf("folder = [%s][%d]\n",
	 (new_trans->download_dir) ? (new_trans->download_dir) : "(null)",
	 file->mime_type);
#endif
  if (resume)
    new_trans->resume = resume;
  else
    new_trans->resume = transfer_create_resume(new_trans);

  if (!new_trans->resume) {
    socket_destroy(socket, &(SocketStatus[S_DELETE]));
    return NULL;
  } else {
    new_trans->resume->downloads =
      g_list_prepend(new_trans->resume->downloads, socket);
  }
  download_show(socket);

  return socket;
}

void download_file(file_t * file, int withdir)
{
  GList* result;
  file_t* file2;
  file_t* file3;
  GtkCTree* ctree;
  GtkCTreeNode* node1;
  GtkCTreeNode* node2;
  GList* dlist;

  if (!global.options.check_lib_for_download) {
    download_file_without_check(file, withdir);
    return;
  }

  result = lib_search_file_type(file);
  if (!result) {
    download_file_without_check(file, withdir);
    return;
  }
  
  if (global.queue_win) {
    gtk_widget_show(global.queue_win);
    if (global.queue_win->window)
      gdk_window_raise(global.queue_win->window);
  } else {
    global.queue_win = create_queue_win();
    gtk_widget_show(global.queue_win);
  }
  
  file2 = file_dup(file);
  file2->flags = withdir;
  ctree = GTK_CTREE(lookup_widget(global.queue_win, "ctree6"));

  strcpy(tstr[0], file2->shortname);
  strcpy(tstr[1], file2->user);
  sprintf(tstr[2], "%lu", file2->size);
  node1 = gtk_ctree_insert_node(ctree, NULL, NULL, list, 5,
			       NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data_full(ctree, node1, file2, 
				   (GtkDestroyNotify)destroy_file_row);
  for (dlist = result; dlist; dlist = dlist->next) {
    file3 = dlist->data;
    strcpy(tstr[0], file3->shortname);
    strcpy(tstr[1], _("local"));
    sprintf(tstr[2], "%lu", file3->size);
    node2 = gtk_ctree_insert_node(ctree, node1, NULL, list, 5,
				  NULL, NULL, NULL, NULL, FALSE, FALSE);
  }
}

void download_file_without_check(file_t* file, int withdir) {
  socket_t *socket;

  socket = download_create(file, withdir, NULL);
  if (!socket)
    return;
  download_start(socket, FALSE);
}

int download_allowed(socket_t * socket)
{
  transfer_t *transfer = (transfer_t *) (socket->data);
  int add = transfer_status_dist(transfer->status, S_INACTIVE);
  int add2 = -(transfer->status == S_REMOTE);
  

  if (transfer->resume && transfer->resume->socket &&
      transfer->resume->socket != socket)
    return 0;

  if (global.limit.max_downloads <= global.limit.cur_downloads + add)
    return 0;

  if (transfer->user_info) {
    if (transfer->user_info->max[0] == -1) {
      // only try one at a time from winmx users
      if ((transfer->user_info->client == C_WINMX) &&
	  (transfer->user_info->cur[0] + add + transfer->user_info->remotes + add2 >= 1))
	return 0;
      if (transfer->user_info->cur[0] + add + transfer->user_info->remotes + add2 >= 
	  global.limit.default_downloads) return 0;
    } else {
      if (transfer->user_info->cur[0] + add + transfer->user_info->remotes + add2 >=
	  transfer->user_info->max[0])
	return 0;
    }
  } else {
    g_warning("no user info found");
  }
  return 1;
}

void download_show(socket_t * socket)
{
  transfer_t *transfer = (transfer_t *) (socket->data);
  resume_t* resume = transfer->resume;
  GtkCTreeNode *node;

  if (!resume)
    return;
  if (!resume->ctree || !resume->node)
    return;

  strcpy(tstr[0], transfer->shortname);

  tstr[1][0] = tstr[2][0] = tstr[3][0] = tstr[4][0] = tstr[5][0] = tstr[6][0] = 0;
  node = gtk_ctree_insert_node(resume->ctree, resume->node, NULL, list, 5,
			       NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(resume->ctree, node, (gpointer) socket);
  
  download_update(socket);
}

void download_hide(socket_t * socket)
{
  transfer_t *transfer = (transfer_t *) (socket->data);
  resume_t* resume = transfer->resume;
  GtkCTreeNode *node;

  if (!resume)
    return;
  if (!resume->ctree || !resume->node)
    return;

  node = gtk_ctree_find_by_row_data(resume->ctree, resume->node, socket);
  if (!node) {
    g_warning("download_hide(): transfer not found");
    return;
  }
  gtk_ctree_remove_node(resume->ctree, node);
}

void download_start(socket_t * socket, int force)
{
  char *t2;
  transfer_t *transfer = (transfer_t *) (socket->data);
  //  chat_page_t* page;
  //  char* prefix;
  
  if (!transfer->resume || (transfer->resume->status != R_ACTIVE)) {
    socket_end(socket, &(SocketStatus[S_CANCELED]));
    g_warning("download_start(): could not start download [%s]", transfer->longname);
    return;
  }
  
  if (global.status.connection < 2) {
    socket_end(socket, &(SocketStatus[S_QUEUED]));
    return;
  }
  
  // testing
  if (transfer_in_progress(transfer)) return;
  // end testing

  // removing timeout if set
  if (socket->timeout >= 0) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout = -1;
  }
#ifdef TRANSFER_DEBUG
  printf("starting down [%s]\n", transfer->winname);
#endif
  transfer->hist_cnt = 0;
  socket->cnt = 0;
  socket->max_cnt = global.network.transfer_timeout;

  user_info_get(transfer->user_info);

  // trying to grab/create resume
  if (!transfer_grab_resume(socket, transfer->resume)) {
    socket_end(socket, &(SocketStatus[S_QUEUED]));
    return;
  }
  // now starting the download
  if (transfer->is_dcc) {
    if (force || (download_allowed(socket))) {
      transfer_connect_and_start(socket);
    } else {
      socket_end(socket, &(SocketStatus[S_QUEUED]));
    }
  } else if (force || (download_allowed(socket))) {
    if (global.options.emulate_winmx) {
      if (transfer->user_info && (transfer->user_info->client == C_WINMX)) {
	send_private(transfer->user_info->user, "//WantQueue", 0);
      }
    }
    t2 = l_strdup_printf("%s \"%s\"", transfer->user_info->user,
			 transfer->winname);
    send_command(CMD_CLIENT_DOWNLOAD, t2);
    l_free(t2);
    transfer_status_set(socket, S_WAITING);
  } else {
    socket_end(socket, &(SocketStatus[S_QUEUED]));
  }
}

FILE *download_open_file(transfer_t * transfer)
{
  int cnt;
  struct stat st;
  resume_t *resume;
  int exist;

  if (!transfer->resume) {
#ifdef TRANSFER_DEBUG
    printf("ops, no resume\n");
#endif
    return NULL;
  }
  resume = transfer->resume;

  exist = (stat(resume->filename, &st) >= 0);
  transfer->check_length = 0;

  if (exist && st.st_size > 1000) {
    transfer->progress = st.st_size - 100;
#ifdef TRANSFER_DEBUG
    printf("resume file does exist! [%s]\n", resume->filename);
#endif
    transfer->file = fopen(resume->filename, "r");
    if (transfer->file == NULL) {
#ifdef TRANSFER_DEBUG
      printf("could not open incomplete file\n");
#endif
      return NULL;
    }

    cnt = st.st_size - transfer->progress;  // should be always 100

    fseek(transfer->file, transfer->progress, SEEK_SET);
#ifdef TRANSFER_DEBUG
    printf("set offset to %ld %d\n", transfer->progress, cnt);
#endif
    transfer->resume_check = l_malloc(cnt * sizeof(char));
    transfer->check_length = read(fileno(transfer->file),
				  transfer->resume_check, cnt);
    fclose(transfer->file);

    if (transfer->check_length != cnt) {
      transfer->file = NULL;
      printf("Warning: resume check length is unsufficient, aborting!!\n");
    } else {
      transfer->file = fopen(resume->filename, "a");
    }
  } else {
    if (exist) {
#ifdef TRANSFER_DEBUG
      printf("resume file does not exist, but is quite small, deleting! [%s]\n", resume->filename);
#endif
      unlink(resume->filename);
    } else {
#ifdef TRANSFER_DEBUG
      printf("resume file does not exist! [%s]\n", resume->filename);
#endif
    }

    //    resume->size = transfer->size;
    cnt = 0;
    transfer->file = fopen(resume->filename, "w");
    transfer->progress = 0;
  }

  if (transfer->shortname) l_free(transfer->shortname);
  transfer->shortname = l_strdup(extract_short_name(resume->filename));

  if (transfer->file == NULL) {
    printf("could not open file for downloading [%s]\n", resume->filename);
  }
  return transfer->file;
}

void download_end(socket_t * socket, int mode)
{
  transfer_t *transfer = (transfer_t *) (socket->data);
  resume_t *resume;
  char *command;
  char *pos;
  int mime;
  char *folder;
  struct stat st;
  GtkToggleButton* toggle;

#ifdef TRANSFER_DEBUG
  printf("download_end, mode %d\n", mode);
#endif

  if (transfer->resume_check) {
    l_free(transfer->resume_check);
    transfer->resume_check = NULL;
    transfer->check_length = 0;
  }

  resume = transfer->resume;
  // return if transfer is not linked to resume // (should never be)
  if (!resume) {
    g_warning("download_end(): no resume found");
    return;
  }

  switch (mode) {
  case S_FINISHED:
    mime = get_mimetype(transfer->resume->filename);
    folder = global.mimetype[mime].download;
    
    if (transfer->download_dir)
      l_free(transfer->download_dir);
    
#ifdef TRANSFER_DEBUG
    if (folder)
      printf("Folder: [%s]\n", folder);
    if (transfer->resume->dirname)
      printf("Transfer->Resume->Dirname: [%s]\n",
	     transfer->resume->dirname);
#endif
    
    if (transfer->resume->dirname) {
      if (transfer->resume->dirname[0] == '/')
	transfer->download_dir = l_strdup(transfer->resume->dirname);
      else if (folder && (*folder))
	transfer->download_dir =
	  l_strdup_printf("%s/%s", folder, transfer->resume->dirname);
      else
	transfer->download_dir =
	  l_strdup_printf("./%s", transfer->resume->dirname);
    } else {
      if (folder && (*folder))
	transfer->download_dir = l_strdup(folder);
      else
	transfer->download_dir = l_strdup(".");
    }
    
    create_dir(transfer->download_dir);
    command = l_strdup_printf("%s/%s", transfer->download_dir,
			      extract_filename(transfer->resume->filename));
    if (stat(command, &st) >= 0) {
      pos = rename_file(command);
    } else
      pos = command;
    if (strcmp(transfer->download_dir, global.incomplete_path)) {
      file_move(transfer->resume->filename, pos);
      l_free(transfer->resume->filename);
      transfer->resume->filename = l_strdup(pos);
    }
    l_free(pos);

    resume->status = R_FINISHED;
    //    resume_update(resume);
    resume_save();
    break;
  case S_IO:
    // deactivate auto resume and autosearch
    if (global.options.auto_resume) {
      toggle = GTK_TOGGLE_BUTTON(lookup_widget(global.win, "checkbutton25"));
      if (toggle) gtk_toggle_button_set_active(toggle, 0);
    }
    if (global.options.auto_search) {
      toggle = GTK_TOGGLE_BUTTON(lookup_widget(global.win, "checkbutton46"));
      if (toggle) gtk_toggle_button_set_active(toggle, 0);
    }
    break;
  case S_DELETE:
  case S_QUEUED:
  case S_REMOTE:
  case S_TIMEOUT:
  case S_INCOMPLETE:
  case S_CANCELED:
  default:
    break;
  }

  transfer_status_set(socket, mode);
  
  // ungrab resume
  if (resume->socket == socket) {
    transfer_ungrab_resume(socket);
  } else if (resume->ctree != resume_belongs_to(resume)) {
    resume_update(resume);
  }
}

gint download_get_info(gpointer data, gint source,
		       GdkInputCondition condition)
{
  char buffer[1025];
  int cnt;
  socket_t *socket = (socket_t *) data;
  socket_t *socket2;
  char *user;
  char *filename;
  char *size;
  transfer_t *transfer;
  char *temp_str;
  unsigned long lsize;

  gdk_input_remove(socket->input);
  socket->input = -1;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, 0);
    return 1;
  }

  cnt = 0;
  switch (cnt = recv(source, buffer, 1024, 0)) {
  case -1:
    socket_destroy(socket, 0);
    return 1;
  case 0:
    socket_destroy(socket, 0);
    return 1;
  default:
    break;
  }

  buffer[cnt] = 0;
#ifdef TRANSFER_DEBUG
  printf("download_info [%s]\n", buffer);
#endif

  user = arg(buffer, 0);
  filename = arg(NULL, 0);
  size = arg(NULL, 0);
  if (!user || !filename || !size) {
    socket_destroy(socket, 0);
    return 1;
  }
#ifdef TRANSFER_DEBUG
  printf("down [%s][%s][%s]\n", user, filename, size);
#endif
  socket2 = download_search_mapable(user, filename);
  if (!socket2) {
    socket_destroy(socket, &(SocketStatus[S_INACTIVE]));
    return 1;
  }

  transfer = (transfer_t *) (socket2->data);
  // try to grab resume
  if (!transfer_grab_resume(socket2, transfer->resume)) {
    socket_destroy(socket, &(SocketStatus[S_INACTIVE]));
    return 1;
  }

  // make sure to reset destination transfer first
  if (socket2->fd >= 0) close(socket2->fd);
  socket2->fd = socket->fd;
  socket->fd = -1;
  if (socket2->input >= 0) gdk_input_remove(socket2->input);
  socket2->input = socket->input;
  socket->input = -1;
  if (socket2->timeout >= 0) gtk_timeout_remove(socket2->timeout);
  socket2->timeout = socket->timeout;
  socket->timeout = -1;
  socket2->ip_long = socket->ip_long;
  socket2->port = socket->port;

  socket_destroy(socket, &(SocketStatus[S_INACTIVE]));
  socket = socket2;
  socket->cnt = 0;
  transfer_status_set(socket, S_INFO);
  lsize = strtoul(size, NULL, 10);
  if (lsize == 0) {
#ifdef TRANSFER_DEBUG
    printf("size == 0\n");
#endif
    socket_end(socket, &(SocketStatus[S_UNAVAILABLE]));
    return 1;
  } else {
    transfer->size = lsize;
  }

  temp_str = l_strdup_printf("%lu", transfer->progress);
#ifdef TRANSFER_DEBUG
  printf("sending offset %s\n", temp_str);
#endif
  send(socket->fd, temp_str, strlen(temp_str), 0);
  l_free(temp_str);

  transfer->start_time = global.current_time;
  if (!download_open_file(transfer)) {
    socket_end(socket, &(SocketStatus[S_IO]));
    return 1;
  }

  transfer_status_set(socket, S_DOWNLOADING);
  socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_READ,
		    GTK_SIGNAL_FUNC(download_get_input), socket);
  return 1;
}

#define MAX_GET   10240

long download_calc_max(socket_t* socket) {
  long m1;
  long m2;
  long limit;
  transfer_t* transfer = socket->data;

  if (transfer->user_info->limit[0]) {
    if (transfer->user_info->limit[0] <= transfer->user_info->bytes[0]) {
      gdk_input_remove(socket->input);
      socket->input = -1;
      return 0;
    } else {
      m1 = transfer->user_info->limit[0] - transfer->user_info->bytes[0];
    }
  } else m1 = MAX_GET;

  if (transfer->user_info->ignore_global[0]) {
    m2 = MAX_GET;
  } else if (global.down_width.bytes >= global.down_width.limit) {
    downloads_disable();
    return 0;
  } else {
    m2 = global.down_width.limit - global.down_width.bytes;
  }

  if (m1 > m2) m1 = m2;
  limit = global.down_width.limit/global.limit.cur_real_downloads/4;
  if (limit > MAX_GET) limit = MAX_GET;

  if (m1 > limit) m1 = limit;
  return m1;
}

gint download_get_input(gpointer data, gint source,
			GdkInputCondition condition)
{
  char buf[MAX_GET + 1];
  int n, n2;
  socket_t *socket = (socket_t *) data;
  transfer_t *transfer = (transfer_t *) (socket->data);
  char *cmp_buf;
#ifdef TRANSFER_DEBUG
  int cnt;
#endif
  long maxget;

  if (condition != GDK_INPUT_READ) {
    socket_end(socket, &(SocketStatus[S_INCOMPLETE]));
    return 1;
  }

  maxget = download_calc_max(socket);
  if (!maxget) return 1;
  
  if ((maxget < 101) && (transfer->check_length > 0)) {
    // read at least 101 bytes for resume check
    maxget = 101;
  }

  //  if (global.down_width.current > global.down_width.limit) return 1;
  //  bzero(buf, sizeof(buf));
  n = recv(source, buf, maxget, 0);
  if (n <= 0) {
    if (transfer->progress >= transfer->size) {
      socket_end(socket, &(SocketStatus[S_FINISHED]));
    } else {
      socket_end(socket, &(SocketStatus[S_INCOMPLETE]));
    }
    return 0;
  } else {
    buf[n] = 0;
  }

  global.statistic.total[0] += (double) n;
  global.down_width.bytes += n;
  transfer->user_info->bytes[0] += n;

  socket->cnt = 0;
  // resume check
  if ((transfer->check_length > 0) || (transfer->progress == 0)) {
    if (!strncasecmp(buf, "FILE NOT", 8)) {
      buf[n] = 0;
      printf("got [%s]\n", buf);
      socket_end(socket, &(SocketStatus[S_UNAVAILABLE]));
      return 0;
    }
  }

  if (transfer->check_length > 0) {
    if (n == 1) {
      // assuming that this is FN behaviour with one bytes
      // sent before file data, just ignoring it.
#ifdef TRANSFER_DEBUG
      printf("read only one byte (%c): assuming FN behaviour, ignoring byte\n",
	     *buf);
      return 0;
#endif
    } else if (transfer->check_length > n) {
#ifdef TRANSFER_DEBUG
      printf("did not read enough bytes [%p][%d][%d]\n",
	     transfer, n, transfer->check_length);
      printf("cannot do resume check, aborting\n");
      client_message("Warning", "Did not read enough bytes (%d) for resume check [%s]",
		     n, transfer->longname);
#endif
      socket_end(socket, &(SocketStatus[S_RESUME_ERR]));
      return 0;
    }
    if (buf[0] == 'A') {
      printf("*** omitting first byte [A], dont know whether is is correct (FN/Rapigator user?)\n");
      client_message("Warning", "Omitting first byte [A] in resume [%s], i hope there is no resume error",
		     transfer->shortname);
      cmp_buf = buf + 1;
      n--;
    } else
      cmp_buf = buf;

    /*
    if (n < transfer->check_length) {
#ifdef TRANSFER_DEBUG
      printf("now there are to less bytes for resume check, aborting\n");
#endif
      socket_end(socket, &(SocketStatus[S_RESUME_ERR]));
      return 0;
    }
    */

#ifdef TRANSFER_DEBUG
    printf("checking %d [%d] bytes\n", transfer->check_length, n);
#endif

    if (memcmp(cmp_buf, transfer->resume_check, transfer->check_length)) {
#ifdef TRANSFER_DEBUG
      printf("resume check failed, ending download\n");
      for (cnt = 0; cnt < transfer->check_length; cnt++)
	printf("%4d %4d\n", cmp_buf[cnt], transfer->resume_check[cnt]);
#endif
      socket_end(socket, &(SocketStatus[S_RESUME_ERR]));
      return 0;
    }

    l_free(transfer->resume_check);
    transfer->resume_check = NULL;

    if (n == transfer->check_length) {
      transfer->check_length = 0;
      transfer->progress += n;    // should be always 100 (or 99) bytes
      return 0;
    }

    // writing pending bytes
    if ((n2 = fwrite(cmp_buf + transfer->check_length,
		     n - transfer->check_length,
		     sizeof(char), transfer->file)) <= 0) {
      printf("could not write data to file2 [%d]\n", n2);
      socket_end(socket, &(SocketStatus[S_IO]));

      transfer->check_length = 0;
      return 0;
    }

    transfer->check_length = 0;
  } else {
    if ((transfer->progress == 0) && (buf[0] == 'A')) {
      printf("*** omitting first byte [A], dont know whether is is correct (FN/Rapigator user?)\n");
      client_message("Warning", "Omitting first byte [A] in download [%s], i hope there is no resume error later",
		     transfer->shortname);
      cmp_buf = buf + 1;
      n--;
    } else
      cmp_buf = buf;

    if (n <= 0) return 0;

    if ((n2 = fwrite(cmp_buf, n, sizeof(char), transfer->file)) <= 0) {
      printf("could not write data to file [%d][%d]\n", n2, n);
      socket_end(socket, &(SocketStatus[S_IO]));
      return 0;
    }
  }

  transfer->progress += n;
  if (transfer->progress >= transfer->size) {
    socket_end(socket, &(SocketStatus[S_FINISHED]));
  }

  return 0;
}

int download_timeout(gpointer data)
{
  socket_t *socket;
  transfer_t *transfer;

  socket = (socket_t *) data;
  transfer = (transfer_t *) (socket->data);

  // test
  if (transfer_in_progress(transfer)) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout = -1;
    return 0;
  }
  // test end

  if (download_allowed(socket)) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout = -1;
    download_start(socket, FALSE);
  } else if (transfer->status != S_QUEUED) {
    gtk_timeout_remove(socket->timeout);
    socket->timeout = -1;
    socket_end(socket, &(SocketStatus[S_QUEUED]));
  }

  return 1;
}

void downloads_disable()
{
  GList *dlist;
  socket_t *socket;
  transfer_t *transfer;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = dlist->data;
    if (socket->type == S_TRANSFER) {
      transfer = socket->data;
      if (!transfer || (transfer->type != T_DOWNLOAD) ||
	  (transfer->status != S_DOWNLOADING)) continue;
      if (transfer->user_info && transfer->user_info->ignore_global[0]) continue;
      if (socket->input >= 0) {
	gdk_input_remove(socket->input);
	socket->input = -1;
      }
    } else if (socket->type == S_BROWSE) {
      if (socket->data && (socket->input >= 0)) {
	gdk_input_remove(socket->input);
	socket->input = -1;
      }
    }

  }
}

void downloads_enable()
{
  GList *dlist;
  socket_t *socket;
  transfer_t *transfer;
  user_info_t* userinfo;
  
  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    if (socket->type == S_TRANSFER) {
      transfer = (transfer_t *) (socket->data);
      if (!transfer || (transfer->type != T_DOWNLOAD) ||
	  (transfer->status != S_DOWNLOADING))
	continue;
      if (socket->input == -1) {
	socket->input =
	  gdk_input_add(socket->fd, GDK_INPUT_READ,
			GTK_SIGNAL_FUNC(download_get_input), socket);
      }
    } else if (socket->type == S_BROWSE) {
      if (socket->data && (socket->input == -1)) {
	socket->input =
	  gdk_input_add(socket->fd, GDK_INPUT_WRITE,
			GTK_SIGNAL_FUNC(get_browse_files), socket);
      }
    }
  }
  global.down_width.bytes = 0;

  for (dlist = global.userinfo; dlist; dlist = dlist->next) {
    userinfo = dlist->data;
    userinfo->bytes[0] = 0;
  }
}

int upload_update(socket_t * socket)
{
  GtkCList *clist;
  int row;
  char str[200];
  double trate;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  int sec;
  transfer_t *transfer;

  if (!socket)
    return 1;
  transfer = (transfer_t *) (socket->data);

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  row = gtk_clist_find_row_from_data(clist, socket);
  if (row < 0)
    return 1;

  strcpy(str, LineSpeed(transfer->user_info->linespeed));
  gtk_clist_set_text(clist, row, 4, str);

  // filesize
  sprintf(str, "%ld/%ld", transfer->progress, transfer->size);
  gtk_clist_set_text(clist, row, 1, str);

  // status
  if ((transfer->status < 0) || (transfer->status >= S_NUMBER)) {
    g_warning("transfer_update: invalid status %d", transfer->status);
    return 1;
  }
  if (transfer->status == S_WAITING)
    sprintf(str, _("Waiting %d secs"), (socket->max_cnt - socket->cnt));
  else
    strcpy(str, status_names(transfer->status));
  gtk_clist_set_text(clist, row, 3, str);

  // progress
  if ((transfer->status == S_DOWNLOADING) ||
      (transfer->status == S_FINISHED) ||
      (transfer->status == S_UPLOADING)) {
    int last_prog;
    int first_prog;

    if (transfer->size > 0) {
      gtk_clist_get_pixmap(clist, row, 5, &pixmap, &bitmap);

      pixmap =
	transfer_draw_progress(clist, pixmap,
			       (double) transfer->progress /
			       (double) transfer->size,
			       clist->column[5].width,
			       (transfer->status != S_FINISHED));
      gtk_clist_set_pixmap(clist, row, 5, pixmap, NULL);
    } else {
      gtk_clist_set_pixmap(clist, row, 5, NULL, NULL);
    }

    transfer->hist_pos++;
    if (transfer->hist_pos >= TRANSFER_HISTORY_TIME)
      transfer->hist_pos = 0;
    transfer->history[transfer->hist_pos] = transfer->progress;
    if (transfer->hist_cnt <
	(TRANSFER_HISTORY_TIME / global.network.transfer_delay) -1)
      transfer->hist_cnt++;

    last_prog = transfer->hist_pos;
    first_prog = (last_prog + TRANSFER_HISTORY_TIME - transfer->hist_cnt) %
	TRANSFER_HISTORY_TIME;

    if (transfer->status == S_FINISHED) {
      if (transfer->end_time - transfer->start_time == 0)
	transfer->end_time++;
      trate = (transfer->size - transfer->start_size) /
	  (transfer->end_time - transfer->start_time);
    } else {
      trate = (double) (transfer->history[last_prog] -
			transfer->history[first_prog]) /
	  transfer->hist_cnt / global.network.transfer_delay;
    }
#ifdef TRANSFER_DEBUG
    if (trate < 0) {
      printf("%d/%d/%d\n", first_prog, last_prog, TRANSFER_HISTORY_TIME);
    }
#endif
    print_speed(str, (int) trate, 1);

    
    transfer->rate = trate;
    detect_speed_pixs(trate, &pixmap, &bitmap);
    gtk_clist_set_pixtext(clist, row, 6, str, 5, pixmap, bitmap);

    if ((transfer->status == S_FINISHED) || (trate > 0)) {
      if (transfer->status == S_FINISHED)
	sec = transfer->end_time - transfer->start_time;
      else
	sec = (int) ((transfer->size - transfer->progress) / trate);
      transfer->timeleft = sec;

      print_time_average(str, sec);
    } else {
      sprintf(str, _("stalled"));
    }
    gtk_clist_set_text(clist, row, 7, str);
  } else {
    gtk_clist_set_text(clist, row, 6, "");
    gtk_clist_set_text(clist, row, 7, "");
  }

  return 1;
}

int eject_upload(transfer_t* request, int large_only) {
  GList* dlist;
  socket_t* socket;
  transfer_t* transfer;
  socket_t* loser = NULL;
  transfer_t* losert = NULL;
  int loser_priority;
  int pri;
  chat_page_t* page;
  char* prefix;

  loser_priority = user_info_priority(request->user_info);
  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = dlist->data;
    if (!socket) continue;
    transfer = socket->data;
    if (transfer == request) continue;

    // searching for a started/running upload
    if (!transfer || (transfer->type != T_UPLOAD) ||
	!transfer_in_progress(transfer)) continue;

    if (large_only &&
	(transfer->size <= (unsigned long) global.limit.large_size * 1024 * 1024))
      continue;

    pri = user_info_priority(transfer->user_info);
    if (transfer->user_info && (loser_priority > pri)) {
      loser = socket;
      losert = transfer;
      loser_priority = pri;
    }
  }
  if (loser) {
    log("uploads", LOG_OTHER, "Ejecting [%s] for [%s]\n",
	losert->user_info->user, request->user_info->user);
    if ((global.options.no_piping & NOPIPE_UPLOAD) == 0) {
      if (global.options.piping & PIPE_UPLOAD) {
	page = chat_page_search("Uploads", P_OTHER);
	if (!page)
	  page = create_other_page("Uploads", _("Uploads"));
      } else {
	page = global.current_page;
      }
      chat_print_time_stamp(page, M_PUBLIC);
      prefix = cparse(global.scheme->client_prefix);
      chat_print_colored(page, M_PUBLIC, "message", prefix);
      chat_print_colored(page, M_PUBLIC, "message", "(");
      chat_print_colored(page, M_PUBLIC, "text", "upload");
      chat_print_colored(page, M_PUBLIC, "message", "!");
      chat_print_colored(page, M_PUBLIC, "text", "eject");
      chat_print_colored(page, M_PUBLIC, "message", ") ");
      chat_print_colored(page, M_PUBLIC, "user", "<");
      chat_print_colored(page, M_PUBLIC, "nick", losert->user_info->user);
      chat_print_colored(page, M_PUBLIC, "user", ">");
      chat_print_colored(page, M_PUBLIC, "text", " for ");
      chat_print_colored(page, M_PUBLIC, "user", "<");
      chat_print_colored(page, M_PUBLIC, "nick", request->user_info->user);
      chat_print_colored(page, M_PUBLIC, "user", ">\n");
    }
    socket_end(loser, &(SocketStatus[S_DELETE]));
    return 1;
  } else {
    return 0;
  }
}

int eject_upload_generic(transfer_t * transfer, int large, int free_large)
{
  if (large) {
    if (free_large) {
      return eject_upload(transfer, 0);   // eject any upload;
    } else {
      return eject_upload(transfer, 1);   // eject any large upload;
    }
  } else {
    return eject_upload(transfer, 0);
  }    
  return 0;
}

int free_slot(int large, int free_up, int free_large) {
  if (large && free_up && free_large) return 1;
  else if (!large && free_up) return 1;
  else return 0;
}

int upload_allowed(transfer_t * transfer, int eject)
{
  struct stat st;
  int add = transfer_status_dist(transfer->status, S_INACTIVE);
  int large;
  int free_up, free_large;
  char* found;
  subscription_t* sub;

  if (stat(transfer->longname, &st) >= 0) {
    transfer->size = st.st_size;
  }
  
  if (!transfer->user_info) {
    g_warning("no user info found");
    return 0;
  }

  // check per user limit
  if ((transfer->user_info->max[1] == -1) &&
      (transfer->user_info->cur[1] + add >= global.limit.default_uploads))
    return 0;
  if ((transfer->user_info->max[1] >= 0) &&
      (transfer->user_info->cur[1] + add >= transfer->user_info->max[0]))
    return 0;

  // now check for subscriptions
  sub = subscription_lookup_file(transfer->user_info->user, transfer->longname, &found);
  if (sub && !found) return 0;

  if (transfer->size > (unsigned long) global.limit.large_size*1024*1024) large = 1;
  else large = 0;
  if (global.limit.max_uploads <= global.limit.cur_uploads + add) free_up = 0;
  else free_up = 1;
  if (global.limit.max_large <= global.limit.cur_large + add) free_large = 0;
  else free_large = 1;

  // if free slot return now
  if (free_slot(large, free_up, free_large)) return 1;

  // if no free slot, try to eject an upload
  if (eject && eject_upload_generic(transfer, large, free_large)) return 1;

  // at last test whether it is a friend, always allow him, even if
  // there was no upload ejected
  if (string_list_search(LIST_FRIEND, transfer->user_info->user)) return 1;
  if (sub && sub->friend_while_subs) return 1;

  // all other cases, return 0;
  return 0;
}

void upload_show(socket_t * socket)
{
  GtkCList *temp;
  int row;
  transfer_t *transfer = (transfer_t *) (socket->data);

  temp = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  row = gtk_clist_find_row_from_data(temp, socket);
  if (row >= 0)
    return;

  strcpy(tstr[0], transfer->shortname);
  strcpy(tstr[2], transfer->user_info->user);
  strcpy(tstr[4], LineSpeed(transfer->user_info->linespeed));

  tstr[1][0] = tstr[3][0] = tstr[5][0] = tstr[6][0] = 0;
  row = gtk_clist_append(temp, list);
  gtk_clist_set_row_data(temp, row, (gpointer) socket);

  transfer_update(socket, 0);
}

void upload_hide(socket_t * socket)
{
  GtkCList *clist;
  int row;

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  row = gtk_clist_find_row_from_data(clist, socket);
  if (row < 0) {
    g_warning("upload_hide(): transfer not found");
    return;
  }
  
  gtk_clist_remove(clist, row);
}

void upload_start(socket_t * socket, int force)
{
  char *t2;
  transfer_t *transfer = (transfer_t *) (socket->data);

  socket->cnt = 0;
  socket->max_cnt = global.network.transfer_timeout;

#ifdef TRANSFER_DEBUG
  printf("starting up [%s]\n", transfer->winname);
#endif
  if (transfer->is_dcc) {
    transfer_status_set(socket, S_WAITING);
  } else {
    if (upload_allowed(transfer, 1) || force) {
      t2 = l_strdup_printf("%s \"%s\"", transfer->user_info->user,
			   transfer->winname);
      send_command(CMD_CLIENT_UPLOAD_OK, t2);

      transfer_status_set(socket, S_WAITING);
      /////////////
      //      access_new_request(transfer);
      /////////////
    } else {
      t2 = l_strdup_printf("%s \"%s\" %d", transfer->user_info->user,
			   transfer->winname, global.limit.cur_uploads);
      send_command(CMD_CLIENT_LIMIT, t2);

      //      socket_end(socket, S_REJECT);
      socket_end(socket, &(SocketStatus[S_QUEUED]));
    }
    l_free(t2);
  }
}

FILE *upload_open_file(transfer_t * transfer)
{
  char *filename;
  struct stat st;

  filename = transfer->longname;
  stat(filename, &st);
  transfer->file = fopen(filename, "r");

  if (transfer->file != NULL)
    transfer->size = st.st_size;
  return transfer->file;
}

void upload_end(socket_t * socket, int mode)
{
  transfer_t *transfer = (transfer_t *) (socket->data);
  subscription_t* sub;
  char* file;

#ifdef TRANSFER_DEBUG
  printf("upload_end, mode %d old %d\n", mode, transfer->status);
#endif
  transfer_status_set(socket, mode);

  if (mode == S_FINISHED) {
    sub = subscription_lookup_file(transfer->user_info->user, transfer->longname, &file);
    if (sub && file) subscription_file_remove(sub, file);

    // now tell access list, that a upload has finished
    access_finished_request(transfer);
  }
}

gint upload_get_info(gpointer data, gint source,
		     GdkInputCondition condition)
{
  char buffer[1025];
  int cnt;
  socket_t *socket = (socket_t *) data;
  socket_t *socket2;
  char *user;
  char *filename;
  char *progress;
  file_t *file;
  transfer_t *transfer;
  struct stat st;
  char *temp_str;
  int i1;

  if (condition != GDK_INPUT_READ) {
    socket_destroy(socket, 0);
    return 1;
  }

  gdk_input_remove(socket->input);
  socket->input = -1;

  cnt = 0;
  switch (cnt = recv(source, buffer, 1024, 0)) {
  case -1:
    socket_destroy(socket, 0);
    return 1;
  case 0:
    socket_destroy(socket, 0);
    return 1;
  default:
    break;
  }

  buffer[cnt] = 0;
#ifdef TRANSFER_DEBUG
  printf("upload_info [%s]\n", buffer);
#endif

  user = arg(buffer, 0);
  filename = arg(NULL, 0);
  progress = arg(NULL, 0);
  
  if (!user || !filename || !progress) {
    socket_destroy(socket, 0);
    return 1;
  }

  if (user && string_list_search(LIST_ENEMY, user)) {
    socket_destroy(socket, 0);
    return 1;
  }
#ifdef TRANSFER_DEBUG
  printf("up [%s][%s][%s]\n", user, filename, progress);
#endif

  socket2 = upload_search_mapable(user, filename);
  if (socket2) {
    transfer = socket2->data;
    if (transfer->status != S_WAITING) {
      log("uploads", LOG_OTHER, "alert [!waiting] (incoming) %s %s\n", user, filename);
      g_warning("transfer doesn't have status \"Waiting\"");
      socket_destroy(socket, &(SocketStatus[S_INACTIVE]));
      return 1;
    }
#ifdef TRANSFER_DEBUG
    printf("transfer: type [%d]\n", transfer->type);
#endif
    // make sure to reset the destination transfer first
    if (socket2->fd >= 0) close(socket->fd);
    socket2->fd = socket->fd;             // should be -1
    socket->fd = -1;
    if (socket2->input >= 0) gdk_input_remove(socket2->input);
    socket2->input = socket->input;       // should be -1
    socket->input = -1;
    if (socket->timeout >= 0) gtk_timeout_remove(socket2->timeout);
    socket2->timeout = socket->timeout;   // should be -1
    socket->timeout = -1;
    socket2->ip_long = socket->ip_long;
    socket2->port = socket->port;

    socket_destroy(socket, &(SocketStatus[S_INACTIVE]));
    socket = socket2;
    socket->cnt = 0;
    transfer_status_set(socket, S_INFO);
    transfer->progress = strtoul(progress, NULL, 10);
    for (i1 = 0; i1 < TRANSFER_HISTORY_TIME; i1++)
      transfer->history[i1] = transfer->progress;
    transfer->start_size = transfer->progress;
  } else {
    log("invalid_requests", LOG_OTHER, "alert [!found] (incoming) %s %s\n", user, filename);
    g_warning("transfer [%s] not found, already deleted?", filename);
    socket_destroy(socket, &(SocketStatus[S_INACTIVE]));
    return 1;
  }

  file = lib_search_transfer(transfer);
  if (stat(transfer->longname, &st) < 0) {
#ifdef TRANSFER_DEBUG
    printf("**sending FILE NOT FOUND\n");
#endif
    //    send(transfer->sock->fd, "0", 1,0);
    send(socket->fd, "FILE NOT FOUND", strlen("FILE NOT FOUND"), 0);
    socket_end(socket, &(SocketStatus[S_IO]));
    return 1;
  } else if (!file && !transfer->is_dcc) {
#ifdef TRANSFER_DEBUG
    printf("**sending FILE NOT SHARED\n");
#endif
    send(socket->fd, "FILE NOT SHARED", strlen("FILE NOT SHARED"), 0);
    socket_end(socket, &(SocketStatus[S_UNAVAILABLE]));
    return 1;
  }
  transfer->size = st.st_size;
  if (file)
    transfer->mime_type = file->mime_type;

  temp_str = l_strdup_printf("%lu", transfer->size);
#ifdef TRANSFER_DEBUG
  printf("sending size %s\n", temp_str);
#endif
  send(socket->fd, temp_str, strlen(temp_str), 0);
  l_free(temp_str);

  transfer->start_time = global.current_time;
  if (!upload_open_file(transfer)) {
    send(socket->fd, "FILE NOT FOUND", strlen("FILE NOT FOUND"), 0);
    socket_end(socket, &(SocketStatus[S_IO]));
    return 1;
  }
  transfer_status_set(socket, S_UPLOADING);
  socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_WRITE,
		    GTK_SIGNAL_FUNC(upload_push_output), socket);
  return 1;
}

#define MAX_PUSH  10240

long upload_calc_max(socket_t* socket) {
  long m1;
  long m2;
  long limit;
  transfer_t* transfer = socket->data;
  int act_limit = global.up_width.limit - 
    global.down_width.current/100*global.limit.download_percent;

  // check per user limit
  if (transfer->user_info->limit[1]) {
    if (transfer->user_info->limit[1] <= transfer->user_info->bytes[1]) {
      // if user limit exceeded, then disable upload
      gdk_input_remove(socket->input);
      socket->input = -1;
      return 0;
    } else {
      // else set max bytes to differnce to limit-already_downloaded
      m1 = transfer->user_info->limit[1] - transfer->user_info->bytes[1];
    }
  } else m1 = MAX_PUSH;
  
  // check global limit
  if (transfer->user_info->ignore_global[1]) {
    // ignore limit for user
    m2 = MAX_PUSH;
  } else if (global.up_width.bytes >= act_limit) {
    uploads_disable();
    return 0;
  } else {
    m2 = act_limit - global.up_width.bytes;
  }

  if (m2 < m1) m1 = m2;

  // set max bytes
  limit = act_limit/global.limit.cur_real_uploads/4;
  if (limit > MAX_PUSH) limit = MAX_PUSH;

  if (m1 > limit) m1 = limit;
  if (m1 > MAX_PUSH) m1 = MAX_PUSH;

  return m1;
}

gint upload_push_output(gpointer data, gint source,
			GdkInputCondition condition)
{
  socket_t *socket = (socket_t *) data;
  transfer_t *transfer = (transfer_t *) (socket->data);
  char buf[MAX_PUSH + 1];
  int n, sent;
  long maxpush;
  
  if (condition != GDK_INPUT_WRITE) {
    socket_end(socket, &(SocketStatus[S_INCOMPLETE]));
    return 1;
  }
  maxpush = upload_calc_max(socket);
  if (!maxpush) return 1;

  if (transfer->progress < transfer->size) {
    lseek(fileno(transfer->file), transfer->progress, SEEK_SET);
    n = read(fileno(transfer->file), buf, maxpush);
    if (n <= 0) {
#ifdef TRANSFER_DEBUG
      printf("read <= 0\n");
#endif
      socket_end(socket, &(SocketStatus[S_IO]));
      return 1;
    }
    sent = send(source, buf, n, 0);
    if (sent <= 0) {
#ifdef TRANSFER_DEBUG
      printf("sent error\n");
#endif
      socket_end(socket, &(SocketStatus[S_INCOMPLETE]));
    } else {
      socket->cnt = 0;
      transfer->progress += sent;
      global.up_width.bytes += sent;
      transfer->user_info->bytes[1] += sent;
      global.statistic.total[1] += (double) sent;
    }
  } else {
    socket_end(socket, &(SocketStatus[S_FINISHED]));
  }

  return 1;
}

void uploads_disable()
{
  GList *dlist;
  socket_t *socket;
  transfer_t *transfer;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    if (socket->type == S_TRANSFER) {
      transfer = (transfer_t *) (socket->data);
      if (!transfer || (transfer->type != T_UPLOAD) ||
	  (transfer->status != S_UPLOADING))
	continue;
      if (transfer->user_info && transfer->user_info->ignore_global[1]) continue;
      if (socket->input >= 0) {
	gdk_input_remove(socket->input);
	socket->input = -1;
      }
    } else if (socket->type == S_SHARE) {
      if (socket->data && (socket->input >= 0)) {
	gdk_input_remove(socket->input);
	socket->input = -1;
      }
    }
  }
}

void uploads_enable()
{
  GList *dlist;
  socket_t *socket;
  transfer_t *transfer;
  user_info_t* userinfo;

  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    if (socket->type == S_TRANSFER) {
      transfer = (transfer_t *) (socket->data);
      if (!transfer || (transfer->type != T_UPLOAD) ||
	  (transfer->status != S_UPLOADING))
	continue;
      if (socket->input == -1) {
	socket->input =
	  gdk_input_add(socket->fd, GDK_INPUT_WRITE,
			GTK_SIGNAL_FUNC(upload_push_output), socket);
      }
    } else if (socket->type == S_SHARE) {
      if (socket->data && (socket->input == -1)) {
	socket->input =
	  gdk_input_add(socket->fd, GDK_INPUT_WRITE,
			GTK_SIGNAL_FUNC(send_browse_files), socket);
      }
    }
  }
  global.up_width.bytes = 0;

  // updating all userinfos should not be necessary
  for (dlist = global.userinfo; dlist; dlist = dlist->next) {
    userinfo = dlist->data;
    userinfo->bytes[1] = 0;
  }
}

void remove_dead_uploads()
{
  GList *dlist;
  GList *result1 = NULL;
  socket_t *socket;
  transfer_t *transfer;
  
  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    if (!socket || (socket->type != S_TRANSFER)) continue;
    transfer = (transfer_t *) (socket->data);
    if (!transfer) continue;

    if ((transfer->type == T_UPLOAD) && 
	(transfer->status == S_QUEUED) &&
	!g_list_find(result1, transfer->user_info))
      result1 = g_list_prepend(result1, transfer->user_info);
  }
  
  for (dlist = result1; dlist; dlist = dlist->next) {
    speed_request(((user_info_t*)(dlist->data))->user, SPEED_UPLOAD);
  }
  if (result1) g_list_free(result1);
}


void remove_dead_downloads()
{
  GList *dlist;
  GList *result2 = NULL;
  socket_t *socket;
  transfer_t *transfer;
  
  for (dlist = global.sockets; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    if (!socket || (socket->type != S_TRANSFER))
      continue;
    transfer = (transfer_t *) (socket->data);
    if (!transfer) continue;

    if ((transfer->type == T_DOWNLOAD) &&
	!transfer_in_progress(transfer) &&
	(transfer->status != S_QUEUED) &&
	(socket->timeout == -1) &&
	!g_list_find(result2, transfer->user_info))
      result2 = g_list_prepend(result2, transfer->user_info);
  }

  for (dlist = result2; dlist; dlist = dlist->next) {
    speed_request(((user_info_t*)(dlist->data))->user, SPEED_DOWNLOAD);
  }
  if (result2) g_list_free(result2);
}

int remove_dead_transfers(gpointer data ATTR_UNUSED)
{
  if (global.options.ul_autoremove & REMOVE_U_OFFLINE)
    remove_dead_uploads();
  if (global.options.dl_autoremove & REMOVE_D_OFFLINE)
    remove_dead_downloads();
  return 1;
}


//////////////////////
// widgets
//////////////////////

void on_toggle_dl_remove(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data)
{
  int no = (int) user_data;

  global.options.dl_autoremove = global.options.dl_autoremove ^ no;
}

GtkWidget *create_dl_remove_popup(int val)
{
  GtkWidget *mode_popup;
  GtkAccelGroup *mode_popup_accels;
  GtkWidget *mode;

  mode_popup = gtk_menu_new();
  gtk_object_set_data(GTK_OBJECT(mode_popup), "mode_popup", mode_popup);
  mode_popup_accels =
    gtk_menu_ensure_uline_accel_group(GTK_MENU(mode_popup));

  mode = gtk_check_menu_item_new_with_label(_("Finished"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & REMOVE_D_FINISHED)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_dl_remove,
		     (gpointer)REMOVE_D_FINISHED);

  mode = gtk_check_menu_item_new_with_label(_("Aborted"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & REMOVE_D_ABORTED)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_dl_remove,
		     (gpointer)REMOVE_D_ABORTED);

  mode = gtk_check_menu_item_new_with_label(_("Offline Users"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & REMOVE_D_OFFLINE)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_dl_remove,
		     (gpointer)REMOVE_D_OFFLINE);

  return mode_popup;
}

void on_toggle_autoretry(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data)
{
  int no = (int) user_data;

  global.options.auto_retry = global.options.auto_retry ^ (1 << no);
}

void on_retry_configure(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data ATTR_UNUSED)
{
  GtkWidget *win;
  GtkWidget *widget;

  win = create_retry_win();
  widget = lookup_widget(win, "spinbutton43");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget),
			    (gfloat) (global.options.retry_timeout[0]));
  widget = lookup_widget(win, "spinbutton44");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget),
			    (gfloat) (global.options.retry_timeout[1]));
  widget = lookup_widget(win, "spinbutton45");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget),
			    (gfloat) (global.options.retry_timeout[2]));
  widget = lookup_widget(win, "spinbutton46");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget),
			    (gfloat) (global.options.retry_timeout[3]));
  widget = lookup_widget(win, "spinbutton47");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget),
			    (gfloat) (global.options.retry_timeout[4]));
  widget = lookup_widget(win, "spinbutton58");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget),
			    (gfloat) (global.options.retry_timeout[5]));

  gtk_widget_show(win);
}

GtkWidget *create_retry_popup(int val)
{
  GtkWidget *mode_popup;
  GtkAccelGroup *mode_popup_accels;
  GtkWidget *mode;

  mode_popup = gtk_menu_new();
  gtk_object_set_data(GTK_OBJECT(mode_popup), "mode_popup", mode_popup);
  mode_popup_accels =
      gtk_menu_ensure_uline_accel_group(GTK_MENU(mode_popup));

  mode = gtk_check_menu_item_new_with_label(_("Timed out"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & (1 << 0))
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_autoretry, (gpointer) 0);

  mode = gtk_check_menu_item_new_with_label(_("Rejected"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & (1 << 1))
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_autoretry, (gpointer) 1);

  mode = gtk_check_menu_item_new_with_label(_("Incomplete"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & (1 << 2))
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_autoretry, (gpointer) 2);

  mode = gtk_check_menu_item_new_with_label(_("Connection error"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & (1 << 3))
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_autoretry, (gpointer) 3);

  mode = gtk_check_menu_item_new_with_label(_("Remotely queued"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & (1 << 4))
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_autoretry, (gpointer) 4);

  mode = gtk_check_menu_item_new_with_label(_("Unavailable"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & (1 << 5))
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_autoretry, (gpointer) 5);

  mode = gtk_menu_item_new();
  gtk_widget_ref(mode);
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  gtk_widget_set_sensitive(mode, FALSE);

  mode = gtk_menu_item_new_with_label(_("Configure..."));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_retry_configure, NULL);

  return mode_popup;
}

void on_toggle_ul_remove(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data)
{
  int no = (int) user_data;

  global.options.ul_autoremove = global.options.ul_autoremove ^ no;
}

GtkWidget *create_ul_remove_popup(int val)
{
  GtkWidget *mode_popup;
  GtkAccelGroup *mode_popup_accels;
  GtkWidget *mode;

  mode_popup = gtk_menu_new();
  gtk_object_set_data(GTK_OBJECT(mode_popup), "mode_popup", mode_popup);
  mode_popup_accels =
    gtk_menu_ensure_uline_accel_group(GTK_MENU(mode_popup));

  mode = gtk_check_menu_item_new_with_label(_("Finished"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & REMOVE_U_FINISHED)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_ul_remove, 
		     (gpointer)REMOVE_U_FINISHED);

  mode = gtk_check_menu_item_new_with_label(_("Aborted"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & REMOVE_U_ABORTED)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_ul_remove, 
		     (gpointer)REMOVE_U_ABORTED);

  mode = gtk_check_menu_item_new_with_label(_("Queued"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & REMOVE_U_QUEUED)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_ul_remove, 
		     (gpointer)REMOVE_U_QUEUED);

  mode = gtk_check_menu_item_new_with_label(_("Offline Users"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & REMOVE_U_OFFLINE)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_ul_remove, 
		     (gpointer)REMOVE_U_OFFLINE);

  return mode_popup;
}

void on_toggle_au(GtkMenuItem * menuitem ATTR_UNUSED, gpointer user_data)
{
  int no = (int) user_data;

  global.options.advanced_upload = global.options.advanced_upload ^ no;
}

void
on_dynamic_bandwidth_activate(GtkMenuItem * menuitem ATTR_UNUSED, 
			      gpointer user_data ATTR_UNUSED)
{
  create_percent_win();
}

void
on_priority_activate(GtkMenuItem * menuitem ATTR_UNUSED, 
		     gpointer user_data ATTR_UNUSED)
{
  GtkWidget* win;
  GtkWidget* widget;

  win = create_priority_win();
  gtk_widget_show(win);

  widget = lookup_widget(win, "spinbutton63");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget),
			    (gfloat) (global.options.upload_priority[PRI_FRIEND]));
  widget = lookup_widget(win, "spinbutton64");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget),
			    (gfloat) (global.options.upload_priority[PRI_DOWNLOAD]));
  widget = lookup_widget(win, "spinbutton65");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget),
			    (gfloat) (global.options.upload_priority[PRI_SHARE]));
  widget = lookup_widget(win, "spinbutton66");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget),
			    (gfloat) (global.options.upload_priority[PRI_NONE]));
}


GtkWidget *create_advanced_popup(int val)
{
  GtkWidget *mode_popup;
  GtkAccelGroup *mode_popup_accels;
  GtkWidget *mode;

  mode_popup = gtk_menu_new();
  gtk_object_set_data(GTK_OBJECT(mode_popup), "mode_popup", mode_popup);
  mode_popup_accels =
    gtk_menu_ensure_uline_accel_group(GTK_MENU(mode_popup));
  
  mode =
    gtk_check_menu_item_new_with_label(_("Restart queued if free slot"));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  if (val & AU_RESTART)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mode), TRUE);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_toggle_au, (gpointer) AU_RESTART);

  mode = gtk_menu_item_new();
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  gtk_widget_set_sensitive(mode, FALSE);

  mode =
    gtk_menu_item_new_with_label(_("Dynamic bandwidth settings..."));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_dynamic_bandwidth_activate, NULL);

  mode =
    gtk_menu_item_new_with_label(_("User priority settings..."));
  gtk_widget_show(mode);
  gtk_container_add(GTK_CONTAINER(mode_popup), mode);
  gtk_signal_connect(GTK_OBJECT(mode), "activate",
		     (GtkSignalFunc) on_priority_activate, NULL);

  return mode_popup;
}

void on_add_subscription_activate(GtkMenuItem * menuitem, gpointer user_data);

GtkWidget *create_upload_popup(transfer_t * transfer)
{
  GtkWidget *popup;
  GtkWidget *user_popup;
  GtkWidget *item;
  GtkAccelGroup *popup_accels;
  GtkWidget *delete_upload;
  GtkWidget *separator;
  GtkWidget *trennlinie16;
  GtkWidget *customize_list2;
  GtkCList *clist;
  int item_num;
  char item_str[1024];

  popup = gtk_menu_new();
  gtk_object_set_data(GTK_OBJECT(popup), "popup", popup);
  popup_accels = gtk_menu_ensure_uline_accel_group(GTK_MENU(popup));

  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));
  item_num = g_list_length(clist->selection);

  if (transfer) {
    if (item_num > 1)
      sprintf(item_str, _("Delete Selected (%d)"), item_num);
    else
      sprintf(item_str, _("Delete Upload"));
    delete_upload = gtk_menu_item_new_with_label(item_str);
    gtk_widget_show(delete_upload);
    gtk_container_add(GTK_CONTAINER(popup), delete_upload);
    gtk_signal_connect(GTK_OBJECT(delete_upload), "activate",
		       GTK_SIGNAL_FUNC(on_delete_upload_activate), NULL);

    separator = gtk_menu_item_new();
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);

    if (!transfer_in_progress(transfer)
	&& (transfer->status != S_FINISHED)) {
      if (item_num > 1)
	sprintf(item_str, _("Allow Selected (%d)"), item_num);
      else
	sprintf(item_str, _("Allow Upload"));
      delete_upload = gtk_menu_item_new_with_label(item_str);
      gtk_widget_show(delete_upload);
      gtk_container_add(GTK_CONTAINER(popup), delete_upload);
      gtk_signal_connect(GTK_OBJECT(delete_upload), "activate",
			 GTK_SIGNAL_FUNC(on_allow_upload_activate), NULL);

    }

    if (item_num > 1)
      sprintf(item_str, _("Add to Subscription list (%d)"), item_num);
    else
      sprintf(item_str, _("Add to Subscription list"));
    delete_upload = gtk_menu_item_new_with_label(item_str);
    gtk_widget_show(delete_upload);
    gtk_container_add(GTK_CONTAINER(popup), delete_upload);
    gtk_signal_connect(GTK_OBJECT(delete_upload), "activate",
		       GTK_SIGNAL_FUNC(on_add_subscription_activate), NULL);

    separator = gtk_menu_item_new();
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);

    delete_upload = gtk_menu_item_new_with_label(_("Open File"));
    gtk_widget_show(delete_upload);
    gtk_container_add(GTK_CONTAINER(popup), delete_upload);

    gtk_signal_connect(GTK_OBJECT(delete_upload), "activate",
		       GTK_SIGNAL_FUNC(on_play_file_activate), NULL);

    separator = gtk_menu_item_new();
    gtk_widget_show(separator);
    gtk_container_add(GTK_CONTAINER(popup), separator);
    gtk_widget_set_sensitive(separator, FALSE);

    item = gtk_menu_item_new_with_label(_("User Menu"));
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);

    user_popup = create_user_popup(M_TRANSFER);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), user_popup);

    trennlinie16 = gtk_menu_item_new();
    gtk_widget_show(trennlinie16);
    gtk_container_add(GTK_CONTAINER(popup), trennlinie16);
    gtk_widget_set_sensitive(trennlinie16, FALSE);
  }

  customize_list2 = gtk_menu_item_new_with_label(_("Customize List"));
  gtk_widget_show(customize_list2);
  gtk_container_add(GTK_CONTAINER(popup), customize_list2);
  gtk_signal_connect(GTK_OBJECT(customize_list2), "activate",
		     GTK_SIGNAL_FUNC(on_customize_list_activate), NULL);

  return popup;
}

GtkWidget *create_transfer_popup(transfer_t * transfer, int num)
{
  GtkWidget *user_popup;
  GtkWidget *popup;
  GtkWidget *item;
  char item_str[1024];

  popup = gtk_menu_new();
  if (!transfer)
    return popup;

  if (num > 1)
    sprintf(item_str, _("Cancel Selected (%d)"), num);
  else
    sprintf(item_str, _("Cancel Download"));
  item = gtk_menu_item_new_with_label(item_str);
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_cancel_download_activate), NULL);
  
  item = gtk_menu_item_new();
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_widget_set_sensitive(item, FALSE);

  if (transfer_in_progress(transfer)) {
    if (num > 1)
      sprintf(item_str, _("Restart Selected (%d)"), num);
    else
      sprintf(item_str, _("Restart Download"));
    item = gtk_menu_item_new_with_label(item_str);
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);

    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_retry_download_activate), NULL);
  } else if (transfer->resume->socket || (transfer->status == S_QUEUED)) {
    item = gtk_menu_item_new_with_label(_("Force Download"));
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);

    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_force_download_activate), NULL);
  } else if (transfer->status != S_FINISHED) {
    if (transfer->is_dcc)
      sprintf(item_str, _("Start DCC"));
    else if (num > 1)
      sprintf(item_str, _("Retry Selected (%d)"), num);
    else
      sprintf(item_str, _("Retry Download"));
    item = gtk_menu_item_new_with_label(item_str);
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);

    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_retry_download_activate), NULL);
  } else {
    sprintf(item_str, _("Retry Download"));
    item = gtk_menu_item_new_with_label(item_str);
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);
    gtk_widget_set_sensitive(item, FALSE);
  }

  item = gtk_menu_item_new();
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_widget_set_sensitive(item, FALSE);
  
  item = gtk_menu_item_new_with_label(_("User Menu"));
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);

  user_popup = create_user_popup(M_TRANSFER);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), user_popup);

  item = gtk_menu_item_new();
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_widget_set_sensitive(item, FALSE);

  return popup;
}

GtkWidget *create_resume_popup(resume_t * resume, int num)
{
  GtkWidget *popup;
  GtkWidget *user_popup;
  GtkWidget *item;
  char item_str[1024];

  popup = gtk_menu_new();
  if (!resume)
    return popup;

  if (num > 1)
    sprintf(item_str, _("Cancel Selected (%d)"), num);
  else
    sprintf(item_str, _("Cancel File"));
  item = gtk_menu_item_new_with_label(item_str);
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_resume_cancel_activate), NULL);
  if (resume->status == R_FINISHED) 
    gtk_widget_set_sensitive(item, FALSE);
  
  if (num > 1)
    sprintf(item_str, _("Delete Selected (%d)"), num);
  else
    sprintf(item_str, _("Delete File"));
  item = gtk_menu_item_new_with_label(item_str);
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_resume_delete_activate), NULL);

  if (resume->status == R_ACTIVE) {
    if (num > 1)
      sprintf(item_str, _("Deactivate Selected (%d)"), num);
    else
      sprintf(item_str, _("Deactivate File"));
    item = gtk_menu_item_new_with_label(item_str);
  } else if (resume->status == R_QUEUED) {
    if (num > 1)
      sprintf(item_str, _("Activate Selected (%d)"), num);
    else
      sprintf(item_str, _("Activate File"));
    item = gtk_menu_item_new_with_label(item_str);
  } else {
    if (num > 1)
      sprintf(item_str, _("Activate Selected (%d)"), num);
    else
      sprintf(item_str, _("Activate File"));
    item = gtk_menu_item_new_with_label(item_str);
    gtk_widget_set_sensitive(item, FALSE);
  }

  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  if (resume->status == R_ACTIVE) {
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_resume_freeze_activate), NULL);
  } else {
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_resume_thaw_activate), NULL);
  }
  item = gtk_menu_item_new();
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_widget_set_sensitive(item, FALSE);

  item = gtk_menu_item_new_with_label(_("Open File"));
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_resume_play_activate), NULL);

  item = gtk_menu_item_new();
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_widget_set_sensitive(item, FALSE);

  if (num > 1)
    sprintf(item_str, _("Search Selected (%d)"), num);
  else
    sprintf(item_str, _("Search File"));
  item = gtk_menu_item_new_with_label(item_str);
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_resume_search_activate), NULL);

  item = gtk_menu_item_new();
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_widget_set_sensitive(item, FALSE);
  item = gtk_menu_item_new();
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_widget_set_sensitive(item, FALSE);

  if (resume->socket || (resume->user && *(resume->user))) {
    item = gtk_menu_item_new_with_label(_("User Menu"));
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);

    user_popup = create_user_popup(M_RESUME);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), user_popup);

    item = gtk_menu_item_new();
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);
    gtk_widget_set_sensitive(item, FALSE);
  }

  item = gtk_menu_item_new_with_label(_("Configure Entry..."));
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_resume_configure), resume);

  /*
  if (resume->user && resume->user[0] &&
      (!resume->search_string || l_strcasecmp("__DONT_SEARCH__", resume->search_string))) {
    item = gtk_menu_item_new_with_label(_("Only download from orig. user"));
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_resume_dont_search), resume);
  } else if (resume->search_string && !l_strcasecmp("__DONT_SEARCH__", resume->search_string)) {
    item = gtk_menu_item_new_with_label(_("Allow download from any user"));
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(on_resume_allow_search), resume);
  }
  */

  item = gtk_menu_item_new();
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_widget_set_sensitive(item, FALSE);

  return popup;
}

GtkWidget *create_download_popup()
{
  GtkWidget *popup = NULL;
  GtkWidget *item;
  int item_num;
  socket_t *socket = NULL;
  transfer_t *transfer = NULL;
  resume_t *resume = NULL;
  GtkCTree *ctree;
  GtkCTreeNode *node;

  ctree = GTK_CTREE(global.popup_list);
  node = gtk_ctree_node_nth(ctree, global.popup_row);

  item_num = g_list_length(GTK_CLIST(ctree)->selection);

  if (node) {
    if (GTK_CTREE_ROW(node)->parent != NULL) {
      socket = (socket_t *) gtk_ctree_node_get_row_data(ctree, node);
      transfer = (transfer_t *) (socket->data);
    } else
      resume = (resume_t *) gtk_ctree_node_get_row_data(ctree, node);
  }

  if (transfer)
    popup = create_transfer_popup(transfer, item_num);
  else
    popup = create_resume_popup(resume, item_num);

  if (transfer || resume) {
    item = gtk_menu_item_new();
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(popup), item);
    gtk_widget_set_sensitive(item, FALSE);
  }

  item = gtk_menu_item_new_with_label(_("Customize List"));
  gtk_widget_show(item);
  gtk_container_add(GTK_CONTAINER(popup), item);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(on_customize_list_activate), NULL);

  return popup;
}

char *valid_download_folder(int mime)
{
  if (!global.incomplete_path || !directory_exists(global.incomplete_path))
     return NULL;
  if (!global.mimetype[mime].download ||
      !directory_exists(global.mimetype[mime].download))
     return NULL;
  return global.mimetype[mime].download;
}

void on_cancel2_clicked(GtkButton * button ATTR_UNUSED, gpointer user_data ATTR_UNUSED)
{
  char *nick;

  nick = (char *) gtk_object_get_data(GTK_OBJECT(global.file_win), "nick");
  gtk_object_set_data(GTK_OBJECT(global.file_win), "nick", NULL);
  if (nick) l_free(nick);
  gtk_widget_hide(global.file_win);
}

void on_ok2_clicked(GtkButton * button, gpointer user_data)
{
  char *filename;
  char *nick;

  filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(global.file_win));
  nick = (char *) gtk_object_get_data(GTK_OBJECT(global.file_win), "nick");
  if (filename && *filename && nick)
    dcc_send_file(nick, filename);
  on_cancel2_clicked(button, user_data);
}

#define CTCP_DELIM_CHAR '\001'

void dcc_send_file(char *nick, char *file)
{
  char *command;
  struct stat st;
  transfer_t *new_trans;
  struct sockaddr_in localaddr;
  int len = sizeof(struct sockaddr_in);
  socket_t *socket;
  unsigned long ip_long;

  if (!nick || !file) {
    printf("no nick/file\n");
    return;
  }
  if (stat(file, &st) < 0) {
    client_message(_("DCC"), _("File does not exist!"));
    return;
  }
  if (global.status.connection < 2) {
    client_message(_("DCC"), _("Not connected!"));
    return;
  }
  if (!global.upload_socket) {
    client_message(_("DCC"),
		   _("You cannot send files, if you are firewalled (port 0)"));
    return;
  }

  new_trans = transfer_new();
  new_trans->longname = l_strdup(file);
  new_trans->winname = l_strdup(file);
  convert_to_win(new_trans->winname);
  new_trans->shortname = l_strdup(extract_short_name(new_trans->longname));
  new_trans->type = T_UPLOAD;
  new_trans->is_dcc = TRUE;
  new_trans->mime_type = get_mimetype(file);
  transfer_detect_user_info(new_trans, nick);

  send_command(CMD_CLIENT_USERSPEED, new_trans->user_info->user);
  new_trans->size = st.st_size;
  socket = socket_new(S_TRANSFER);
  socket->data = new_trans;

  upload_show(socket);
  upload_start(socket, 1);	// 1 is needless here, cause dcc is always started

  getsockname(global.napster->fd, (struct sockaddr *)&localaddr, &len);
  ip_long = (unsigned long) BSWAP32(localaddr.sin_addr.s_addr);
  if (global.my_ip && (global.my_ip != ip_long)) {
    client_message("DCC", _("Lopster detected local ip [%s]"), ntoa(ip_long));
    client_message("DCC", _("But the server browse reported ip [%s], using this one for DCC file transfer"), ntoa(global.my_ip));
    ip_long = global.my_ip;
  }
  command =
    l_strdup_printf("%s %cSEND %s %lu %d \"%s\" %lu %s %d%c",
		    nick, CTCP_DELIM_CHAR, SERVER->nick,
		    ip_long, global.network.port, new_trans->winname,
		    (unsigned long) st.st_size, "checksum", 0,
		    CTCP_DELIM_CHAR);
  send_command(CMD_CLIENT_PRIVMSG, command);
  client_message("DCC", _("Sending %s to <%s>"), new_trans->longname, 
		 new_trans->user_info->user);
  l_free(command);
}

void dcc_select_file(char *nick)
{
  GtkWidget *temp;
  char *str;

  if (!nick) return;

  if (!global.file_win) {
    str = l_strdup_printf(_("Select file to send [%s]"), nick);
    global.file_win = gtk_file_selection_new(str);
    l_free(str);
    gtk_widget_show(global.file_win);
    temp = GTK_FILE_SELECTION(global.file_win)->ok_button;
    if (temp)
      gtk_signal_connect(GTK_OBJECT(temp), "clicked",
			 GTK_SIGNAL_FUNC(on_ok2_clicked), NULL);
    temp = GTK_FILE_SELECTION(global.file_win)->cancel_button;
    if (temp)
      gtk_signal_connect(GTK_OBJECT(temp), "clicked",
			 GTK_SIGNAL_FUNC(on_cancel2_clicked), NULL);
  } else {
    gtk_widget_show(global.file_win);
  }
  str = gtk_object_get_data(GTK_OBJECT(global.file_win), "nick");
  if (str) l_free(str);
  gtk_object_set_data(GTK_OBJECT(global.file_win), "nick", l_strdup(nick));
}

int dcc_check_message(char *from ATTR_UNUSED, char *message)
{
  char *nick;
  unsigned long ip_long;
  int port;
  char *filename;
  unsigned long filesize;
  char *command;
  char *checksum;
  int speed;
  transfer_t *new_trans;
  char *folder;
  socket_t *socket;
  GList* resume_list;

  if (!message || strlen(message) < 2)
    return 0;

  if (*message == CTCP_DELIM_CHAR &&
      message[strlen(message) - 1] == CTCP_DELIM_CHAR) {
    message++;
    message[strlen(message) - 1] = 0;

    command = arg(message, 0);
    if (command && strncmp(command, "SEND", 4))
      return 0;
    nick = arg(NULL, 0);
    ip_long = strtoul(arg(NULL, 0), NULL, 10);
    port = atoi(arg(NULL, 0));
    filename = arg(NULL, 0);
    filesize = strtoul(arg(NULL, 0), NULL, 10);
    checksum = arg(NULL, 0);
    speed = atoi(arg(NULL, 0));

    if (filesize == 0) {
      client_message("DCC", _("<%s> is sending a 0 byte %s, ignoring"),
		     nick, filename);
      return 1;
    } else {
      client_message("DCC", _("<%s> is sending %s"), nick, filename);
    }
    if (port == 0 && global.network.port == 0) {
      client_message("Message",
		     _("Both systems are firewalled. Unable to comply"));
      return 1;
    }

    new_trans = transfer_new();
    new_trans->winname = l_strdup(filename);
    new_trans->mime_type = get_mimetype(filename);
    new_trans->longname = l_strdup(new_trans->winname);
    convert_to_unix(new_trans->longname);
    new_trans->shortname = l_strdup(extract_short_name(new_trans->longname));
    new_trans->size = filesize;
    new_trans->type = T_DOWNLOAD;
    transfer_detect_user_info(new_trans, nick);
    new_trans->user_info->linespeed = speed;

    if ((folder = valid_download_folder(new_trans->mime_type)) == NULL) {
      transfer_destroy(new_trans);
      download_dialog(new_trans->shortname, new_trans->mime_type);
      return 1;
    }

    new_trans->download_dir = l_strdup(folder);
    new_trans->md5 = l_strdup(checksum);
    new_trans->is_dcc = TRUE;
    socket = socket_new(S_TRANSFER);
    socket->data = new_trans;
    socket->port = htons(port);
    //    printf("port %d %d\n", port, socket->port);
    socket->ip_long = BSWAP32(ip_long);

    resume_list = resume_search_size(new_trans->size);
    if (resume_list) {
      new_trans->resume = resume_list->data;
      g_list_free(resume_list);
    } else {
      new_trans->resume = transfer_create_resume(new_trans);
    }

    if (!new_trans->resume) {
      socket_destroy(socket, &(SocketStatus[S_DELETE]));
      return 1;
    } else {
      new_trans->resume->downloads =
	g_list_prepend(new_trans->resume->downloads, socket);
    }
    download_show(socket);

    if (!global.options.allow_dcc) {
      client_message(NULL,
		     _("At the moment you do not allow dcc file transfers"),
		     nick, filename);
      client_message(NULL,
		     _("If you want to accept the transfer, go to the transfer page and continue download"));
      send_private(nick,
		   "DCC transfers are not allowed at the moment, maybe i will start the download manually",
		   1);
      return 1;
    } else {
      download_start(socket, TRUE);
    }

    return 1;
  }
  return 0;
}

int global_timer(gpointer data ATTR_UNUSED)
{
  char comm[2048];
  GtkWidget *temp;
  GList *dlist;
  socket_t *socket;
  transfer_t *transfer;
  static int cnt = 0;
  ping_t *ping;
  user_timestamp_t *stamp;
  struct timeval tv;
  speed_t *speed;
  cnt++;

  global.current_time = time(NULL);

  if (global.status.exiting == E_SAFE) {
    if (global.limit.cur_uploads + global.limit.cur_downloads == 0)
      global.status.exiting = E_NORMAL;
  } else if (global.status.exiting == E_NORMAL) {
    global_exit();
  }

  if (global.socket_win) {
    for (dlist = global.sockets; dlist; dlist = dlist->next) {
      socket = (socket_t *) (dlist->data);
      socket_update_clist(socket);
    }
  }

  if (global.options.time_display)
    print_topright_corner();

  gettimeofday(&tv, 0);

  dlist = global.userstamp;
  while (dlist) {
    stamp = (user_timestamp_t *) (dlist->data);
    dlist = dlist->next;
    if (tv.tv_sec - stamp->tv.tv_sec > 100) {
      client_message(_("Message"),
		     _("Ping to <%s> timed out after 100 seconds!"),
		     stamp->user);
      global.userstamp = g_list_remove(global.userstamp, stamp);
      l_free(stamp->user);
      l_free(stamp);
    }
  }

  dlist = global.speed_requests;
  while (dlist) {
    speed = (speed_t *) (dlist->data);
    dlist = dlist->next;
    if (global.current_time - speed->sent > SPEED_LIFE)
      speed_remove(speed);
  }

  dlist = global.pings;
  while (dlist) {
    ping = (ping_t *) (dlist->data);
    dlist = dlist->next;
    ping->cnt++;
    if (ping->cnt > PING_LIFE)
      ping_destroy(ping);
  }

  dlist = global.sockets;
  while (dlist) {
    socket = (socket_t *) (dlist->data);
    dlist = dlist->next;
    switch (socket->type) {
    case S_SHARE:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt)
	socket_destroy(socket, 0);
      break;
    case S_BROWSE:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt) {
	socket_destroy(socket, 0);
      }
      break;
    case S_HTTP:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt)
	socket_destroy(socket, 0);
      break;
    case S_SERVER:
      if (socket->fd >= 0) {
	socket->cnt++;
	if (socket->cnt >= socket->max_cnt) {
	  socket->cnt = 0;
	  napster_disconnect("Server connection timed out");
	}
      }
      break;
    case S_DATA:
      break;
    case S_TRANSFER:
      transfer = (transfer_t *) (socket->data);
      if (!transfer)
	break;
      if (transfer_in_progress(transfer)) {
	socket->cnt++;
	if (socket->cnt >= socket->max_cnt)
	  socket_end(socket, &(SocketStatus[S_TIMEOUT]));
	if ((cnt % global.network.transfer_delay) == 0) {
	  transfer_update(socket, 0);
	}
      }
      break;
    case S_UNKNOWN:
      socket->cnt++;
      if (socket->cnt >= socket->max_cnt)
	socket_destroy(socket, 0);
      break;
    default:
      break;
    }
  }

  statistic_update(&global.statistic);
  if (notebook_page_visible(7))
    statistic_output(&global.statistic);

  // updating stats
  temp = lookup_widget(global.win, "label534");
  sprintf(comm, "%d/%d", global.limit.cur_downloads,
	  global.limit.cur_real_downloads);
  gtk_label_set_text(GTK_LABEL(temp), comm);
  temp = lookup_widget(global.win, "label547");
  sprintf(comm, "%d/%d/%d", global.limit.cur_uploads,
	  global.limit.cur_real_uploads, global.limit.cur_large);
  gtk_label_set_text(GTK_LABEL(temp), comm);

  global.down_width.current =
      (global.down_width.current * (cnt - 1) +
       global.down_width.bytes) /cnt;
  global.up_width.current =
      (global.up_width.current * (cnt - 1) + global.up_width.bytes) /cnt;

  if (cnt % global.network.transfer_delay == 0) {
    draw_band_width(&global.down_width, 0);
    draw_band_width(&global.up_width, 0);
    cnt = 0;
  }

  downloads_enable();
  uploads_enable();

  resume_remove_outdated();

  if (global.status.connection < 2)
    return 1;
  if (!global.napster)
    return 1;
  if (SERVER->network == N_UNKNOWN)
    return 1;

  transfer_start_queued();
  
  temp = lookup_widget(global.win, "progressbar1");
  if ((global.napster->cnt > 1) &&GTK_WIDGET_VISIBLE(temp)) {
    gtk_progress_bar_update(GTK_PROGRESS_BAR(temp),
			    (double) (global.napster->cnt) /(double) global.napster->max_cnt);
  }

  search_next();

  return 1;
}

gint await_conn_ack(gpointer data, gint source,
		    GdkInputCondition condition)
{
  int res;
  char c;
  socket_t *socket = (socket_t *) data;

  if (condition != GDK_INPUT_READ) {
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  }

  gdk_input_remove(socket->input);

  switch (res = recv(source, &c, 1, 0)) {
  case -1:
#ifdef TRANSFER_DEBUG
    printf("rec1 error\n");
#endif
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  case 0:
#ifdef TRANSFER_DEBUG
    printf("received nothing\n");
#endif
    socket_end(socket, &(SocketStatus[S_CONERROR]));
    return 1;
  default:
    break;
  }

#ifdef TRANSFER_DEBUG
  printf("got [%c]\n", c);
#endif
  if (c != '1') {
    socket_end(socket, &(SocketStatus[S_UNAVAILABLE]));
    return 1;
  }

  socket->input =
      gdk_input_add(socket->fd, GDK_INPUT_WRITE,
		    GTK_SIGNAL_FUNC(transfer_send_info), socket);

  return 1;
}


void on_button_band_cancel_clicked(GtkButton * button ATTR_UNUSED, gpointer user_data)
{
  GtkWidget *win;

  win = GTK_WIDGET(user_data);
  gtk_widget_destroy(win);
}

void on_button_band_ok_clicked(GtkButton * button, gpointer user_data)
{
  user_info_t* userinfo;
  GtkWidget *temp;
  GtkObject *adj;
  int upload;

  userinfo = user_data;
  if (!userinfo) return;
  adj = gtk_object_get_data(GTK_OBJECT(button), "adj");
  upload = !((int)gtk_object_get_data(GTK_OBJECT(button), "type"));
  if (!adj) return;
  userinfo->limit[upload] = (int)(GTK_ADJUSTMENT(adj)->value);

  temp = lookup_widget(GTK_WIDGET(button), "checkbutton");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(temp)))
    userinfo->ignore_global[upload] = 1;
  else userinfo->ignore_global[upload] = 0;

  temp = lookup_widget(GTK_WIDGET(button), "win");
  on_button_band_cancel_clicked(button, temp);
}

void on_adj_value_changed(GtkAdjustment *adjustment,
			  gpointer user_data) {
  char str[1024];
  GtkWidget* temp = user_data;

  print_speed(str, adjustment->value, 1);
  gtk_label_set_text(GTK_LABEL(temp), str);
}

void create_bandwidth_win(char* user, int download)
{
  GtkWidget *win;
  GtkWidget *frame;
  GtkWidget *vbox175;
  GtkWidget *vbox176;
  GtkWidget *hbox635;
  GtkWidget *label796;
  GtkWidget *label797;
  GtkWidget *hseparator34;
  GtkWidget *frame296;
  GtkWidget *hbox634;
  GtkWidget *button275;
  GtkWidget *button276;
  GtkWidget *hscale1;
  GtkWidget *checkbutton;
  GtkObject *adj;
  user_info_t* userinfo;
  char str[1024];

  userinfo = user_info_search(user, download);
  if (!userinfo) return;

  win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_object_set_data(GTK_OBJECT(win), "win", win);
  gtk_widget_set_usize(win, 500, -2);
  gtk_window_set_title(GTK_WINDOW(win), _("User Bandwidth"));
  gtk_window_set_default_size(GTK_WINDOW(win), 170, -1);
  gtk_window_set_policy(GTK_WINDOW(win), FALSE, FALSE, FALSE);

  frame = gtk_frame_new(NULL);
  gtk_widget_show(frame);
  gtk_container_add(GTK_CONTAINER(win), frame);
  gtk_container_set_border_width(GTK_CONTAINER(frame), 5);

  vbox175 = gtk_vbox_new(FALSE, 5);
  gtk_widget_show(vbox175);
  gtk_container_add(GTK_CONTAINER(frame), vbox175);
  gtk_container_set_border_width(GTK_CONTAINER(vbox175), 5);

  vbox176 = gtk_vbox_new(FALSE, 5);
  gtk_widget_show(vbox176);
  gtk_box_pack_start(GTK_BOX(vbox175), vbox176, FALSE, FALSE, 0);

  hbox635 = gtk_hbox_new(FALSE, 5);
  gtk_widget_show(hbox635);
  gtk_box_pack_start(GTK_BOX(vbox176), hbox635, FALSE, FALSE, 0);

  if (download)
    label796 = gtk_label_new(_("Specify download bandwidth for"));
  else
    label796 = gtk_label_new(_("Specify upload bandwidth for"));
  gtk_widget_show(label796);
  gtk_box_pack_start(GTK_BOX(hbox635), label796, FALSE, FALSE, 0);

  label797 = gtk_label_new(user);
  gtk_widget_show(label797);
  gtk_box_pack_start(GTK_BOX(hbox635), label797, FALSE, FALSE, 0);
  gtk_widget_set_style(label797, global.style[5]);

  hbox635 = gtk_hbox_new(FALSE, 5);
  gtk_widget_show(hbox635);
  gtk_box_pack_start(GTK_BOX(vbox176), hbox635, FALSE, FALSE, 0);

  if (download)
    adj = gtk_adjustment_new (userinfo->limit[0], 0, 
			      global.down_width.limit, 0, 0, 0);
  else
    adj = gtk_adjustment_new (userinfo->limit[1], 0,
			      global.up_width.limit, 0, 0, 0);
  hscale1 = gtk_hscale_new (GTK_ADJUSTMENT (adj));
  gtk_widget_show (hscale1);
  gtk_box_pack_start (GTK_BOX (hbox635), hscale1, TRUE, TRUE, 0);
  gtk_scale_set_draw_value (GTK_SCALE (hscale1), FALSE);

  label796 = gtk_label_new(print_speed(str, userinfo->limit[(!download)], 1));
  gtk_widget_show(label796);
  gtk_box_pack_start(GTK_BOX(hbox635), label796, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(adj), "value-changed",
		     on_adj_value_changed, (gpointer)label796);
  
  checkbutton = gtk_check_button_new_with_label (_("Ignore global bandwidth limit for this user"));
  gtk_object_set_data (GTK_OBJECT (win), "checkbutton", checkbutton);
  gtk_widget_show (checkbutton);
  gtk_box_pack_start(GTK_BOX(vbox176), checkbutton, FALSE, FALSE, 0);
  if (userinfo->ignore_global[(!download)])
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton), TRUE);
  
  hseparator34 = gtk_hseparator_new();
  gtk_widget_show(hseparator34);
  gtk_box_pack_start(GTK_BOX(vbox175), hseparator34, FALSE, FALSE, 0);

  frame296 = gtk_frame_new(NULL);
  gtk_widget_show(frame296);
  gtk_box_pack_start(GTK_BOX(vbox175), frame296, FALSE, FALSE, 0);
  gtk_widget_set_usize(frame296, -2, 41);
  gtk_frame_set_shadow_type(GTK_FRAME(frame296), GTK_SHADOW_IN);

  hbox634 = gtk_hbox_new(TRUE, 5);
  gtk_widget_show(hbox634);
  gtk_container_add(GTK_CONTAINER(frame296), hbox634);
  gtk_container_set_border_width(GTK_CONTAINER(hbox634), 5);

  button275 = gtk_button_new_with_label(_("Ok"));
  gtk_object_set_data(GTK_OBJECT(button275), "adj", adj);
  gtk_object_set_data(GTK_OBJECT(button275), "type", (gpointer)(download));
  gtk_widget_show(button275);
  gtk_box_pack_start(GTK_BOX(hbox634), button275, TRUE, TRUE, 0);
  gtk_signal_connect(GTK_OBJECT(button275), "clicked",
		     on_button_band_ok_clicked, (gpointer)userinfo);

  button276 = gtk_button_new_with_label(_("Cancel"));
  gtk_widget_show(button276);
  gtk_box_pack_start(GTK_BOX(hbox634), button276, TRUE, TRUE, 0);
  gtk_signal_connect(GTK_OBJECT(button276), "clicked",
		     on_button_band_cancel_clicked, win);
  gtk_signal_connect(GTK_OBJECT(win), "destroy",
		     GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
  
  gtk_widget_show(win);

  return;
}

GtkWidget* create_bandwidth_popup(int mode) {
  user_info_t* userinfo1;
  GtkWidget* item = NULL;
  GtkWidget* item2;
  GtkWidget* popup;
  char* user;

  user = get_popup_user(mode);
  if (!user) return NULL;
  userinfo1 = user_info_search(user, 1);

  if (userinfo1) {
    item = gtk_menu_item_new_with_label(_("Set Bandwidth"));
    gtk_widget_show(item);
    
    popup = gtk_menu_new();
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), popup);
    
    item2 = gtk_menu_item_new_with_label(_("Download..."));
    gtk_widget_show(item2);
    gtk_container_add(GTK_CONTAINER(popup), item2);
    gtk_signal_connect(GTK_OBJECT(item2), "activate",
		       GTK_SIGNAL_FUNC(on_enter_bandwidth1_activate),
		       (gpointer)mode);
    item2 = gtk_menu_item_new_with_label(_("Upload..."));
    gtk_widget_show(item2);
    gtk_container_add(GTK_CONTAINER(popup), item2);
    gtk_signal_connect(GTK_OBJECT(item2), "activate",
		       GTK_SIGNAL_FUNC(on_enter_bandwidth2_activate),
		       (gpointer)mode);
  } else {
    item = NULL;
  }
  return item;
}

gint user_info_comp_func(gconstpointer a, gconstpointer b) {
  const user_info_t* userinfo1 = a;
  const user_info_t* userinfo2 = b;

  if (!userinfo1) return (userinfo2 != NULL);
  if (!userinfo2) return -1;

  return l_strcasecmp(userinfo1->user, userinfo2->user);
}

void on_button_band_cancel2_clicked(GtkButton * button ATTR_UNUSED, gpointer user_data)
{
  GtkWidget *win;
  GList* adj_list;
  GList* info_list;

  adj_list = gtk_object_get_data(GTK_OBJECT(user_data), "adj_list");
  info_list = gtk_object_get_data(GTK_OBJECT(user_data), "info_list");
  g_list_free(adj_list);
  g_list_free(info_list);

  win = GTK_WIDGET(user_data);
  gtk_widget_destroy(win);
}

void on_button_band_ok2_clicked(GtkButton * button, gpointer user_data)
{
  GtkObject *adj;
  GList* dlist1;
  GList* dlist2;
  GList* adj_list;
  GList* info_list;
  user_info_t* userinfo;
  int upload;

  adj_list = gtk_object_get_data(GTK_OBJECT(user_data), "adj_list");
  info_list = gtk_object_get_data(GTK_OBJECT(user_data), "info_list");
  upload = !((int)gtk_object_get_data(GTK_OBJECT(user_data), "type"));

  for (dlist1 = adj_list, dlist2 = info_list; dlist1 && dlist2; dlist1 = dlist1->next, dlist2 = dlist2->next) {
    userinfo = dlist2->data;
    adj = dlist1->data;
    userinfo->limit[upload] = (int)(GTK_ADJUSTMENT(adj)->value);
  }
  on_button_band_cancel2_clicked(button, user_data);
}

int user_info_involved(user_info_t* userinfo, char* string) {
  int friend;

  if (!strcmp(_("All Users"), string)) return 1;
  
  friend = (string_list_search(LIST_FRIEND, userinfo->user) != NULL);
  if (!strcmp(_("Users, friend"), string) && friend) return 1;
  if (!strcmp(_("Users, not friend"), string) && !friend) return 1;
  
  return 0;
}

void on_button_set_clicked(GtkButton * button ATTR_UNUSED, gpointer user_data)
{
  GtkObject *adj;
  GtkObject *adj2;
  GtkEntry* entry;
  GList* dlist1;
  GList* dlist2;
  GList* adj_list;
  GList* info_list;
  user_info_t* userinfo;
  char* text;

  adj = GTK_OBJECT(gtk_object_get_data(GTK_OBJECT(user_data), "adj"));
  entry = GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(user_data), "entry"));
  text = gtk_entry_get_text(entry);
  adj_list = gtk_object_get_data(GTK_OBJECT(user_data), "adj_list");
  info_list = gtk_object_get_data(GTK_OBJECT(user_data), "info_list");
  
  for (dlist1 = adj_list, dlist2 = info_list; dlist1 && dlist2; dlist1 = dlist1->next, dlist2 = dlist2->next) {
    userinfo = dlist2->data;
    adj2 = dlist1->data;
    if (!user_info_involved(userinfo, text)) continue;

    //    userinfo->limit = (int)(GTK_ADJUSTMENT(adj)->value);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(adj2), GTK_ADJUSTMENT(adj)->value);
  }
}


void create_bandman_win (int download)
{
  GtkWidget *bandman_win;
  GtkWidget *frame;
  GtkWidget *vbox1;
  GtkWidget *vbox2;
  GtkWidget *hbox;
  GtkWidget *scrolledwindow;
  GtkWidget *label;
  GtkWidget *hscale1;
  GtkWidget *viewport1;
  GtkWidget *table;
  GtkWidget *separator;
  GtkWidget *button;
  GtkWidget *combo;
  GtkWidget *entry;
  GList* combo_items = NULL;

  GtkObject* adj;
  char str[1024];
  GList* user_list;
  GList* dlist;
  user_info_t* userinfo;
  long limit;
  int no;
  GList* info_list;
  GList* adj_list;
  int upload = !download;

  global.userinfo = 
    g_list_sort(global.userinfo, (GCompareFunc)user_info_comp_func);
  user_list = global.userinfo;
  if (download) {
    limit = global.down_width.limit;
  } else {
    limit = global.up_width.limit;
  }
  no = g_list_length(user_list);

  bandman_win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (bandman_win), _("Bandwidth Limit Manager"));
  gtk_window_set_default_size (GTK_WINDOW (bandman_win), 400, 450);
  gtk_signal_connect(GTK_OBJECT(bandman_win), "destroy",
		     GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

  frame = gtk_frame_new (NULL);
  gtk_widget_show (frame);
  gtk_container_add (GTK_CONTAINER (bandman_win), frame);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 3);

  vbox1 = gtk_vbox_new (FALSE, 5);
  gtk_widget_show (vbox1);
  gtk_container_add (GTK_CONTAINER (frame), vbox1);
  gtk_container_set_border_width (GTK_CONTAINER (vbox1), 5);

  ///////////

  scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scrolledwindow);
  gtk_box_pack_start (GTK_BOX (vbox1), scrolledwindow, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

  viewport1 = gtk_viewport_new (NULL, NULL);
  gtk_widget_show (viewport1);
  gtk_container_add (GTK_CONTAINER (scrolledwindow), viewport1);

  table = gtk_table_new (no, 3, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (viewport1), table);
  gtk_container_set_border_width (GTK_CONTAINER (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 2);

  no = 0;
  info_list = adj_list = NULL;
  for (dlist = user_list; dlist; dlist = dlist->next) {
    userinfo = dlist->data;

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox);
    gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, no, no+1,
		      (GtkAttachOptions) (GTK_FILL),
		      (GtkAttachOptions) (0), 0, 0);

    label = gtk_label_new (userinfo->user);
    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

    adj = gtk_adjustment_new (userinfo->limit[upload], 0, limit, 0, 0, 0);
    hscale1 = gtk_hscale_new (GTK_ADJUSTMENT (adj));
    gtk_widget_show (hscale1);
    gtk_table_attach (GTK_TABLE (table), hscale1, 1, 2, no, no+1,
		      (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		      (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_scale_set_draw_value (GTK_SCALE (hscale1), FALSE);

    adj_list = g_list_prepend(adj_list, adj);
    info_list = g_list_prepend(info_list, userinfo);

    label = gtk_label_new (print_speed(str, userinfo->limit[upload], 1));
    gtk_widget_show (label);
    gtk_table_attach (GTK_TABLE (table), label, 2, 3, no, no+1,
		      (GtkAttachOptions) (GTK_FILL),
		      (GtkAttachOptions) (0), 0, 0);
    gtk_signal_connect(GTK_OBJECT(adj), "value-changed",
		       on_adj_value_changed, (gpointer)label);
    no++;
  }
  gtk_object_set_data(GTK_OBJECT(bandman_win), "info_list", info_list);
  gtk_object_set_data(GTK_OBJECT(bandman_win), "adj_list", adj_list);
  gtk_object_set_data(GTK_OBJECT(bandman_win), "type", (gpointer)(download));

  /////////////////////////

  vbox2 = gtk_vbox_new (FALSE, 5);
  gtk_widget_show (vbox2);
  gtk_box_pack_start (GTK_BOX (vbox1), vbox2, FALSE, FALSE, 0);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);

  adj = gtk_adjustment_new (0, 0, limit, 0, 0, 0);
  hscale1 = gtk_hscale_new (GTK_ADJUSTMENT (adj));
  gtk_widget_show (hscale1);
  gtk_box_pack_start (GTK_BOX (hbox), hscale1, TRUE, TRUE, 0);
  gtk_scale_set_draw_value (GTK_SCALE (hscale1), FALSE);
  gtk_object_set_data(GTK_OBJECT(bandman_win), "adj", adj);

  label = gtk_label_new (print_speed(str, 0, 1));
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(adj), "value-changed",
		     on_adj_value_changed, (gpointer)label);

  hbox = gtk_hbox_new (FALSE, 5);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);

  button = gtk_button_new_with_label (_("Set limit for"));
  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     on_button_set_clicked, bandman_win);

  combo = gtk_combo_new ();
  gtk_widget_show (combo);
  gtk_box_pack_start (GTK_BOX (hbox), combo, TRUE, TRUE, 0);
  combo_items = g_list_append (combo_items, _("All Users"));
  //  combo_items = g_list_append (combo_items, _("Users, uploading to"));
  //  combo_items = g_list_append (combo_items, _("Users, not uploading to"));
  //  combo_items = g_list_append (combo_items, _("Users, downloading from"));
  //  combo_items = g_list_append (combo_items, _("Users, not downloading from"));
  combo_items = g_list_append (combo_items, _("Users, friend"));
  combo_items = g_list_append (combo_items, _("Users, not friend"));
  gtk_combo_set_popdown_strings (GTK_COMBO (combo), combo_items);
  g_list_free (combo_items);

  entry = GTK_COMBO (combo)->entry;
  gtk_widget_show (entry);
  gtk_entry_set_text (GTK_ENTRY (entry), _("All Users"));
  gtk_object_set_data(GTK_OBJECT(bandman_win), "entry", entry);

  /////////////////////////

  separator = gtk_hseparator_new ();
  gtk_widget_show (separator);
  gtk_box_pack_start (GTK_BOX (vbox1), separator, FALSE, FALSE, 0);

  /////////////////////////

  frame = gtk_frame_new (NULL);
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (vbox1), frame, FALSE, FALSE, 0);
  gtk_widget_set_usize (frame, -2, 41);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);

  hbox = gtk_hbox_new (TRUE, 5);
  gtk_widget_show (hbox);
  gtk_container_add (GTK_CONTAINER (frame), hbox);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);

  button = gtk_button_new_with_label (_("Ok"));
  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     on_button_band_ok2_clicked, bandman_win);

  button = gtk_button_new_with_label (_("Cancel"));
  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     on_button_band_cancel2_clicked, bandman_win);

  gtk_widget_show(bandman_win);
}

void on_button_percent_ok_clicked(GtkButton * button, gpointer user_data)
{
  GtkWidget *temp;
  GtkObject *adj;

  adj = user_data;
  if (!adj) return;

  global.limit.download_percent = (int)(GTK_ADJUSTMENT(adj)->value);

  temp = lookup_widget(GTK_WIDGET(button), "win");
  on_button_band_cancel_clicked(button, temp);
}

void create_percent_win() {
  GtkWidget *win;
  GtkWidget *frame;
  GtkWidget *vbox175;
  GtkWidget *vbox176;
  GtkWidget *hbox635;
  GtkWidget *label796;
  GtkWidget *hseparator34;
  GtkWidget *frame296;
  GtkWidget *hbox634;
  GtkWidget *button275;
  GtkWidget *button276;
  GtkWidget *hscale1;
  GtkObject *adj;

  win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_object_set_data(GTK_OBJECT(win), "win", win);
  gtk_widget_set_usize(win, 500, -2);
  gtk_window_set_title(GTK_WINDOW(win), _("Dynamic Upload limit"));
  gtk_window_set_default_size(GTK_WINDOW(win), 170, -1);
  gtk_window_set_policy(GTK_WINDOW(win), FALSE, FALSE, FALSE);

  frame = gtk_frame_new(NULL);
  gtk_widget_show(frame);
  gtk_container_add(GTK_CONTAINER(win), frame);
  gtk_container_set_border_width(GTK_CONTAINER(frame), 5);

  vbox175 = gtk_vbox_new(FALSE, 5);
  gtk_widget_show(vbox175);
  gtk_container_add(GTK_CONTAINER(frame), vbox175);
  gtk_container_set_border_width(GTK_CONTAINER(vbox175), 5);

  vbox176 = gtk_vbox_new(FALSE, 5);
  gtk_widget_show(vbox176);
  gtk_box_pack_start(GTK_BOX(vbox175), vbox176, FALSE, FALSE, 0);

  hbox635 = gtk_hbox_new(FALSE, 5);
  gtk_widget_show(hbox635);
  gtk_box_pack_start(GTK_BOX(vbox176), hbox635, FALSE, FALSE, 0);

  label796 = gtk_label_new(_("Setup value the upload limit should be reduced with the current download bandwidth"));
  gtk_widget_show(label796);
  gtk_box_pack_start(GTK_BOX(hbox635), label796, FALSE, FALSE, 0);

  hbox635 = gtk_hbox_new(FALSE, 5);
  gtk_widget_show(hbox635);
  gtk_box_pack_start(GTK_BOX(vbox176), hbox635, FALSE, FALSE, 0);

  adj = gtk_adjustment_new (global.limit.download_percent, 0, 
			    100, 1, 1, 0);
  hscale1 = gtk_hscale_new (GTK_ADJUSTMENT (adj));
  gtk_widget_show (hscale1);
  gtk_box_pack_start (GTK_BOX (hbox635), hscale1, TRUE, TRUE, 0);
  gtk_scale_set_draw_value (GTK_SCALE (hscale1), TRUE);
  gtk_scale_set_digits (GTK_SCALE (hscale1), 0);

  hseparator34 = gtk_hseparator_new();
  gtk_widget_show(hseparator34);
  gtk_box_pack_start(GTK_BOX(vbox175), hseparator34, FALSE, FALSE, 0);

  frame296 = gtk_frame_new(NULL);
  gtk_widget_show(frame296);
  gtk_box_pack_start(GTK_BOX(vbox175), frame296, FALSE, FALSE, 0);
  gtk_widget_set_usize(frame296, -2, 41);
  gtk_frame_set_shadow_type(GTK_FRAME(frame296), GTK_SHADOW_IN);

  hbox634 = gtk_hbox_new(TRUE, 5);
  gtk_widget_show(hbox634);
  gtk_container_add(GTK_CONTAINER(frame296), hbox634);
  gtk_container_set_border_width(GTK_CONTAINER(hbox634), 5);

  button275 = gtk_button_new_with_label(_("Ok"));
  gtk_widget_show(button275);
  gtk_box_pack_start(GTK_BOX(hbox634), button275, TRUE, TRUE, 0);
  gtk_signal_connect(GTK_OBJECT(button275), "clicked",
		     on_button_percent_ok_clicked, adj);

  button276 = gtk_button_new_with_label(_("Cancel"));
  gtk_widget_show(button276);
  gtk_box_pack_start(GTK_BOX(hbox634), button276, TRUE, TRUE, 0);
  gtk_signal_connect(GTK_OBJECT(button276), "clicked",
		     on_button_band_cancel_clicked, win);
  gtk_signal_connect(GTK_OBJECT(win), "destroy",
		     GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
  
  gtk_widget_show(win);

  return;
}

void upload_update_speed(user_info_t* userinfo) {
  GtkCList *clist;
  socket_t *socket;
  transfer_t *transfer;
  int i1;

  if (!userinfo) return;
  clist = GTK_CLIST(lookup_widget(global.win, "transfer_up"));

  gtk_clist_freeze(clist);
  for (i1 = 0; i1 < clist->rows; i1++) {
    socket = (socket_t *) gtk_clist_get_row_data(clist, i1);
    transfer = (transfer_t *) (socket->data);
    if (transfer->user_info != userinfo) continue;

    transfer_update(socket, 1);
  }
  gtk_clist_thaw(clist);
}

