/* Copyright (C) 2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <time.h>
#include <string.h>
#include <sys/time.h>
#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>

#include "lopster.h"
#include "global.h"
#include "userinfo.h"
#include "support.h"
#include "callbacks.h"
#include "string_list.h"
#include "subscription.h"
#include "whois.h"

#define INFO_LIFE        10    // days

user_info_t* user_info_new(char* user) {
  user_info_t* user_info;

  user_info = l_malloc(sizeof(user_info_t));
  user_info->last_server = NULL;
  user_info->user = l_strdup(user);
  user_info->cur[0] = 0;
  user_info->cur[1] = 0;
  user_info->real_cur[0] = 0;
  user_info->real_cur[1] = 0;
  user_info->max[0] = -1;
  user_info->max[1] = -1;
  user_info->limit[0] = 0;
  user_info->limit[1] = 0;
  user_info->bytes[0] = 0;
  user_info->bytes[1] = 0;
  user_info->ignore_global[0] = 0;
  user_info->ignore_global[1] = 0;
  user_info->linespeed = -1;
  user_info->timestamp = 0;
  user_info->last_seen = global.current_time;
  user_info->shares = 0;
  user_info->remotes = 0;
  user_info->client = C_UNKNOWN;
  global.userinfo = g_list_prepend(global.userinfo, user_info);

  return user_info;
}

void user_info_destroy(user_info_t* userinfo) {
  if (!userinfo) return;
  global.userinfo = g_list_remove(global.userinfo, userinfo);
  if (userinfo->user) l_free(userinfo->user);
  if (userinfo->last_server) l_free(userinfo->last_server);
  l_free(userinfo);
}

void user_info_save() {
  GList *dlist;
  char *fname;
  char *fname_new;
  FILE *fd;
  user_info_t *info;

  fname = l_strdup_printf("%s/userinfo.list", global.options.config_dir);
  fname_new = l_strdup_printf("%s/access.list.new", global.options.config_dir);

  if ((fd = fopen(fname_new, "w")) == NULL) {
    g_warning(_("Could not write [%s]\n"), fname);
    l_free(fname);
    l_free(fname_new);
    return;
  }

  for (dlist = global.userinfo; dlist; dlist = dlist->next) {
    info = dlist->data;

    if (global.current_time - info->last_seen > 60*60*24*INFO_LIFE) continue;
    fprintf(fd, "%s %d %d %ld %ld %d %d %lu \"%s\"\n",
	    info->user, info->max[0], info->max[1], info->limit[0], info->limit[1],
	    info->ignore_global[0], info->ignore_global[1], info->last_seen,
	    info->last_server?info->last_server:"");
  }
  
  if (!ferror(fd)) rename(fname_new, fname);
  else {
    g_warning(_("Could not write [%s]\n"), fname);
  }

  fclose(fd);
  l_free(fname);
  l_free(fname_new);
}

void user_info_load() {
  FILE *fd;
  char *fname;
  char line[2048];
  char* user;
  char* max1, *max2;
  char* limit1, *limit2;
  char* ignore1, *ignore2;
  user_info_t* userinfo;
  char* seen;
  char* server;
  time_t last_seen;

  //  printf("user_info_load(): %ld\n", time(NULL));
  fname = l_strdup_printf("%s/userinfo.list", global.options.config_dir);
  if ((fd = fopen(fname, "r")) == NULL) {
    l_free(fname);
    return;
  }
  l_free(fname);

  while (fgets(line, 2048 - 1, fd)) {
    user = arg(line, 0);
    max1 = arg(NULL, 0);
    max2 = arg(NULL, 0);
    limit1 = arg(NULL, 0);
    limit2 = arg(NULL, 0);
    ignore1 = arg(NULL, 0);
    ignore2 = arg(NULL, 0);
    seen = arg(NULL, 0);
    server = arg(NULL, 0);
    
    if (!user || !max1 || !max2 || !limit1 || !limit2 || !ignore1 || !ignore2) continue;

    if (seen) last_seen = strtol(seen, NULL, 10);
    else last_seen = global.current_time;
    if (global.current_time - last_seen > 60*60*24*INFO_LIFE) continue;

    userinfo = user_info_new(user);
    userinfo->timestamp = 1;    // make lopster to whois the user on first appearance
    userinfo->max[0] = atoi(max1);
    userinfo->max[1] = atoi(max2);
    userinfo->limit[0] = strtol(limit1, NULL, 10);
    userinfo->limit[1] = strtol(limit2, NULL, 10);
    userinfo->ignore_global[0] = atoi(ignore1);
    userinfo->ignore_global[0] = atoi(ignore2);
    userinfo->last_seen = last_seen;
    if (server) userinfo->last_server = strdup(server);
  }

  fclose(fd);

  return;
}

user_info_t *user_info_search(char *user, int down ATTR_UNUSED)
{
  GList *source;
  GList *dlist;
  user_info_t *user_info;

  source = global.userinfo;

  for (dlist = source; dlist; dlist = dlist->next) {
    user_info = dlist->data;
    if (!l_strcasecmp(user_info->user, user))
      return user_info;
  }

  return NULL;
}

int user_info_priority(user_info_t* userinfo) {
  int pri = 0;
  subscription_t* sub;

  if (string_list_search(LIST_FRIEND, userinfo->user)) {
    if (pri < global.options.upload_priority[PRI_FRIEND])
      pri = global.options.upload_priority[PRI_FRIEND];
  }

  sub = subscription_user_search(userinfo->user);
  if (sub && sub->friend_while_subs) {
    if (pri < global.options.upload_priority[PRI_FRIEND])
      pri = global.options.upload_priority[PRI_FRIEND];
  }

  if (userinfo->real_cur[0] > 0) {
    if (pri < global.options.upload_priority[PRI_DOWNLOAD])
      pri = global.options.upload_priority[PRI_DOWNLOAD];
  }

  if (!userinfo->timestamp) {
    return pri;
  }

  if (userinfo->shares > 0) {
    if (pri < global.options.upload_priority[PRI_SHARE])
      pri = global.options.upload_priority[PRI_SHARE];
  } else {
    if (pri < global.options.upload_priority[PRI_NONE])
      pri = global.options.upload_priority[PRI_NONE];
  }

  return pri;
}

void user_info_get(user_info_t* userinfo) {
  // check timestamp.
  if (global.current_time - userinfo->timestamp > 60*60) {
    whois_request(userinfo->user, WHOIS_NONE);
  }
}

void user_info_show(user_info_t* userinfo, int download, int active) {
  GtkWidget* area;
  GtkWidget* spin;
  int posx;
  int posy;
  long limit;
  char str[1024];
  char* ss;
  int width, height;
  int upload = !download;
  
  if (!userinfo) return;
  if (download) {
    limit = global.down_width.limit;
    area = lookup_widget(global.win, "drawingarea12");
    spin = lookup_widget(global.win, "spinbutton15");
  } else {
    limit = global.up_width.limit;
    area = lookup_widget(global.win, "drawingarea11");
    spin = lookup_widget(global.win, "spinbutton49");
  }

  gtk_object_set_data(GTK_OBJECT(area), "userinfo", userinfo);
  gtk_object_set_data(GTK_OBJECT(spin), "userinfo", userinfo);

  if (!active) 
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), (gfloat) (userinfo->max[upload]));

  if (!area->window) return;
  width = area->allocation.width;
  height = area->allocation.height;
  
  posx = width * userinfo->limit[upload] / limit;
  if (posx > width) posx = width;
  if (posx > 0)
    gtk_paint_box(area->style, area->window,
		  GTK_STATE_PRELIGHT, GTK_SHADOW_OUT,
		  NULL, area, "bar", 0, 0, posx, height);
  if (posx < width)
    gtk_paint_flat_box(area->style, area->window,
		       GTK_STATE_NORMAL, GTK_SHADOW_NONE,
		       NULL, area, "entry_bg",
		       posx, 0, width - posx, height);
  if (!active) {
    ss = userinfo->user;
  } else if (userinfo->limit[upload] == 0) {
    strcpy(str, _("No bandwidth limit"));
    ss = str;
  } else {
    ss = print_speed(str, userinfo->limit[upload], 1);
  }
  posx = (width - gdk_string_width(area->style->font, ss))/2;
  posy = (height + gdk_string_height(area->style->font, "A")) / 2;
  gtk_paint_string(area->style, area->window,
		   GTK_STATE_PRELIGHT, NULL, area, "label",
		   posx, posy, ss);
}

// this function could be dangerous, as it does delete user infos that
// might be still in use somewhere else
// deactivated at the moment.

int user_info_cleanup(gpointer data ATTR_UNUSED) {
  GList* dlist;
  user_info_t* info;
  GList* res = NULL;

  return 0;

  for (dlist = global.userinfo; dlist; dlist = dlist->next) {
    info = dlist->data;
    if (global.current_time - info->last_seen > 60*60*24*INFO_LIFE)
      res = g_list_prepend(res, dlist);
  }
  // not finished yet.
  return 1;
}
