/*
 *  Linux snipes, a text-based maze-oriented game for linux.
 *  Copyright (C) 1997 Jeremy Boulton.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Jeremy Boulton is reachable via electronic mail at
 *  boultonj@ugcs.caltech.edu.
 */


#include <curses.h>
#include <assert.h>

#include "snipes.h"
#include "screen.h"


int  (*scrn_ptr_init)( void ) = NULL;
void (*scrn_ptr_close)( void ) = NULL;
void (*scrn_ptr_addch)( unsigned char ) = NULL;
void (*scrn_ptr_addstr)( char* ) = NULL;
void (*scrn_ptr_move)( int, int ) = NULL;
void (*scrn_ptr_refresh)( void ) = NULL;
void (*scrn_ptr_setcolorpair)( int ) = NULL;
void (*scrn_ptr_setnormalattr)( void ) = NULL;
void (*scrn_ptr_getyx)( int*, int* ) = NULL;
void (*scrn_ptr_getmaxyx)( int*, int* ) = NULL;
void (*scrn_ptr_scrolldown)( void ) = NULL;
void (*scrn_ptr_scroll_off)( void ) = NULL;
void (*scrn_ptr_scroll_on)( void ) = NULL;
void (*scrn_ptr_cursor_off)( void ) = NULL;
void (*scrn_ptr_cursor_on)( void ) = NULL;
void (*scrn_ptr_clear)( void ) = NULL;



#ifdef DOUBLEWIDE_FONT_HACK

#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/vt.h>		/* linux specific terminal ioctls */

#include "font_table.h"

#endif



#ifdef USE_SVGALIB

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <vga.h>
#include <vgakeyboard.h>
#include <vga.h>
#include <vgagl.h>



/*
 * The format of a .PSF file is as follows:
 *  
 * Byte    Size of field           Contents
 * 0       Word                    File ID, always 0436h
 * 2       Byte                    File mode
 * 
 * === FILE MODE 0:  256 character byte-aligned monofont (as used by EGA/VGA)
 * 
 * 3       Byte                    Height of characters in scan lines
 * 4...    256x[byte 3] bytes      Font data; each character contains one byte
 *                                     per scan line, characters continuous
 * 
 * === FILE MODE 1:  512 character byte-aligned monofont
 * 
 * 3       Byte                    Height of characters in scan lines
 * 4...    512x[byte 3] bytes      Font data; each character contains one byte
 * 
 *                                     per scan line, characters continuous
 */

int fdi;
char *font_buffer;
char *font_start;
char *expanded_font;
int font_buffer_size;
int char_height;

int map_font( char *fontname )
{
  int num_chars;
  struct stat stat_buf;

  if( (fdi = open( fontname, O_RDONLY )) == -1 ) {
    fprintf( stderr, "Could not open file \"%s\" for read.\n", fontname );
    return 1;
  }

  fstat( fdi, &stat_buf );
  
  if( stat_buf.st_size == 0 ) {
    close( fdi );
    fprintf( stderr, "Font file is too short.\n" );
    return 1;
  }

  font_buffer_size = stat_buf.st_size;

  if( (font_buffer = mmap( 0, font_buffer_size, PROT_READ, MAP_SHARED, fdi, 0 )) == MAP_FAILED ) {
    close( fdi );
    perror( "Could not mmap font file" );
    return 1;
  }
  
  if( font_buffer[1] != 0x04 || font_buffer[0] != 0x36 ) {
    munmap( font_buffer, font_buffer_size );
    close( fdi );
    fprintf( stderr, "Invalid magic number in font file.\n" );
    return 1;
  }

  if( font_buffer[2] >= 2 ) {
    munmap( font_buffer, font_buffer_size );
    close( fdi );
    fprintf( stderr, "Invalid mode number in font file.\n" );
    return 1;
  }
  
  char_height = (int)font_buffer[3];
  num_chars = (font_buffer_size-4)/char_height;

  font_start = font_buffer+4;
  
  return 0;
}


void unmap_font( void )
{
  munmap( font_buffer, font_buffer_size );
  close( fdi );
}

#endif




#if defined( USE_SVGALIB ) || defined ( USE_XWIN )

int screen_max_x, screen_max_y;
int screen_max_x_cell, screen_max_y_cell;
int screen_cur_x_cell, screen_cur_y_cell;
int scrolling_enabled;
int fgcolor, bgcolor;



void _vidutil_null( void )
{
}


void _vidutil_move( int y, int x )
{
  screen_cur_x_cell = x;
  screen_cur_y_cell = y;

  if( screen_cur_x_cell > screen_max_x_cell )
    screen_cur_x_cell = 0;
  if( screen_cur_y_cell > screen_max_y_cell )
    screen_cur_y_cell = 0;
}


void _vidutil_newline( void )
{
  screen_cur_x_cell = 0;
  screen_cur_y_cell++;
  while( screen_cur_y_cell >= screen_max_y_cell ) {
    (*scrn_ptr_scrolldown)();
    screen_cur_y_cell--;
  }
}


void _vidutil_advance_cursor( void )
{
  screen_cur_x_cell++;
  if( screen_cur_x_cell >= screen_max_x_cell ) {
    screen_cur_x_cell = 0;
    screen_cur_y_cell++;
    while( screen_cur_y_cell > screen_max_y_cell ) {
      (*scrn_ptr_scrolldown)();
      screen_cur_y_cell--;
    }
  }
}
#endif


#ifdef USE_XWIN

/* Modified with permission from the "mxt" X toolkit which is:
 *  Copyright (C) 1997 John Meacham 
 *  john@foo.net
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>


/* Global variables use for talking to the X server */

int okay_to_draw;

static int screen_num;
static Screen *screen_ptr;
Atom WM_DELETE_WINDOW;

#define NUM_X_COLORS 5

GC gc, rev_gc;
GC color_gc[NUM_X_COLORS];
char *X_color_names[NUM_X_COLORS] = { "white", "red", "blue", "white", "green" };
GC *current_gc;

Display *disp;
Window win;
Pixmap pixmap;
Colormap cmap;

XFontStruct *font_info;

void set_okay_to_draw_X( void )
{
  okay_to_draw = 1;
}

#endif




#ifdef USE_SVGALIB
int _target_svgalib_init( void )
{
  screen_max_x = 640;
  screen_max_y = 480;
  screen_max_x_cell = screen_max_x/8;
  screen_max_y_cell = screen_max_y/char_height;
  screen_cur_x_cell = 0;
  screen_cur_y_cell = 0;
  fgcolor = 7;
  bgcolor = 0;
  
  if( map_font( "/home/boultonj/test/default8x16.psf" ) )
    return 0;

  vga_init();

  if( !vga_hasmode( G640x480x256 ) ) {
    fprintf( stderr, "Graphics mode 640x480x16 not available.\n" );
    return 0;
  }

  vga_setmode( G640x480x16 );
  return 1;
}


void _target_svgalib_close( void )
{
  vga_setmode(TEXT);
  unmap_font();
}


void _target_svgalib_getmaxyx( int *maxy, int *maxx )
{
  *maxy = screen_max_y_cell;
  *maxx = screen_max_x_cell;
}


void _target_svgalib_getyx( int *my_y, int *my_x )
{
  *my_y = screen_cur_y_cell;
  *my_x = screen_cur_x_cell;
}


void _target_svgalib_refresh( void )
{
}


void _target_svgalib_setcolorpair( int n )
{
  switch( n )
  {
    case 0:
      fgcolor = 7;
      break;
    case 1:
      fgcolor = 4;
      break;
    case 2:
      fgcolor = 3;
      break;
    case 3:
      fgcolor = 2;
      break;
  }
}


void _target_svgalib_setnormalattr( void )
{
  fgcolor = 7;
}


void _target_svgalib_addch( unsigned char ch )
{
  int i, j;
  assert( screen_cur_y_cell < screen_max_y_cell );
  assert( screen_cur_x_cell < screen_max_x_cell );

  if( ch == '\n' ) {
    _vidutil_newline();
    return;
  }

  vga_setegacolor( fgcolor );
  for( i=0; i<char_height; i++ )
    for( j=0; j<8; j++ )
      if( font_start[(char_height*(int)ch)+i] & (128>>j) ) {
	vga_drawpixel( 8*screen_cur_x_cell+j, char_height*screen_cur_y_cell+i );
      }
#if 0
      else {
	vga_setegacolor( bgcolor );
	vga_drawpixel( 8*screen_cur_x_cell+j, char_height*screen_cur_y_cell+i );
      }
#endif

  _vidutil_advance_cursor();
}


void _target_svgalib_move( int y, int x )
{
  _vidutil_move( y, x );
}


void _target_svgalib_scrolldown( void )
{
}


void _target_svgalib_scroll_off( void )
{
  scrolling_enabled = 0;
}


void _target_svgalib_scroll_on( void )
{
  scrolling_enabled = 1;
}


void _target_svgalib_clear( void )
{
  vga_clear();
}
#endif


#ifdef USE_XWIN
int _target_X_init( void )
{
  XClassHint xch;
  XWMHints *hints;
  XSizeHints xsh;
  unsigned long black;
  unsigned long white;
  int default_depth;
  
  int i;

  XColor c;

  Window root;
  
  /* Get DISPLAY from environment variable */
  if((disp=XOpenDisplay(NULL)) == NULL) {
    fprintf( stderr, "Unable to open display.\n" );
    return 0;
  }

  screen_num = DefaultScreen(disp);
  screen_ptr = DefaultScreenOfDisplay(disp);

  cmap = DefaultColormap(disp,DefaultScreen(disp));
  root = RootWindow(disp,screen_num);
  WM_DELETE_WINDOW = XInternAtom(disp, "WM_DELETE_WINDOW", False);
  
  black = BlackPixel(disp,screen_num);
  white = WhitePixel(disp,screen_num);

  win = XCreateSimpleWindow(disp,root,0,0,640,400,0, black,black);

  xch.res_name = "snipes";
  xch.res_class = "Snipes";
  XSetClassHint(disp, win, &xch);

  hints = XAllocWMHints();
  hints->window_group = win;
  hints->flags = WindowGroupHint;
  XSetWMHints(disp, win, hints);

  XStoreName(disp,win,"Snipes");

  XMapWindow(disp, win);

  gc = XCreateGC( disp, root, 0, NULL );
  XSetForeground( disp, (GC)gc, white );
  XSetBackground( disp, (GC)gc, black );

  rev_gc = XCreateGC( disp, root, 0, NULL);
  XSetForeground( disp, (GC)rev_gc, black);
  XSetBackground( disp, (GC)rev_gc, white);

  XSelectInput( disp, win, ExposureMask | KeyPressMask | KeyReleaseMask );
  XSetWMProtocols( disp, win, &WM_DELETE_WINDOW, 1 );

  xsh.flags = ( PMaxSize | PSize | PMinSize);
  xsh.width = 640;
  xsh.height = 400;
  xsh.min_width = 640;
  xsh.min_height = 400;
  xsh.max_width = 640;
  xsh.max_height = 400;
  xsh.x = 0;
  xsh.y = 0;

  //XSetSizeHints(disp, win, &xsh);

  default_depth = DefaultDepth( disp, screen_num );
  pixmap = XCreatePixmap( disp, win, 640, 400, default_depth );
  
  if( (font_info = XLoadQueryFont( disp, "vga" )) == NULL ) {
    fprintf( stderr, "Cannot open vga font.\n" );
    XCloseDisplay(disp);
    return 0;
  }
  
  XSetFont( disp, gc, font_info->fid );
  
  for( i=0; i<NUM_X_COLORS; i++ ) {
    color_gc[i] = XCreateGC( disp, root, 0, NULL );
    XParseColor( disp, cmap, X_color_names[i], &c );
    assert( XAllocColor( disp, cmap, &c ) ); 
    XSetForeground( disp, (GC)color_gc[i], c.pixel );
    XSetBackground( disp, (GC)color_gc[i], black );
    XSetFont( disp, (GC)color_gc[i], font_info->fid );
  }
  
  current_gc = &color_gc[0];
  
  screen_max_x = 640;
  screen_max_y = 400;
  screen_max_x_cell = 80;
  screen_max_y_cell = 25;
  screen_cur_x_cell = 0;
  screen_cur_y_cell = 0;
  
  scrolling_enabled = 1;
  okay_to_draw = 0;
  
  return 1;
}


void _target_X_close( void )
{
  XCloseDisplay(disp);
}


void _target_X_getmaxyx( int *maxy, int *maxx )
{
  *maxy = screen_max_y_cell;
  *maxx = screen_max_x_cell;
}


void _target_X_getyx( int *my_y, int *my_x )
{
  *my_y = screen_cur_y_cell;
  *my_x = screen_cur_x_cell;
}


void _target_X_refresh( void )
{
  XCopyArea( disp, pixmap, win, gc, 0, 0, 640, 400, 0, 0 );
  XFlush( disp );
}


void _target_X_setcolorpair( int n )
{
  current_gc = &color_gc[n%NUM_X_COLORS];
}


void _target_X_setnormalattr( void )
{
  current_gc = &gc;
}


void _target_X_addch( unsigned char ch )
{
  assert( screen_cur_y_cell < screen_max_y_cell );
  assert( screen_cur_x_cell < screen_max_x_cell );

  if( ch == '\n' ) {
    _vidutil_newline();
    return;
  }
  
#if 0
  XFillRectangle( disp, pixmap, rev_gc, 8*screen_cur_x_cell,
		  16*screen_cur_y_cell, 8, 16 );
#endif
  XDrawString( disp, pixmap, *current_gc, 8*screen_cur_x_cell,
	       16*screen_cur_y_cell + font_info->ascent, (char *)&ch, 1 );

  _vidutil_advance_cursor();
}


void _target_X_move( int y, int x )
{
  _vidutil_move( y, x );
}


void _target_X_scrolldown( void )
{
  int height = 16;
  if( scrolling_enabled ) {
    XCopyArea( disp, pixmap, pixmap, gc, 0, height, 640, 400-height, 0, 0 );
    XFillRectangle( disp, pixmap, rev_gc, 0, 24*height, 640, height );
  }
}


void _target_X_scroll_off( void )
{
  scrolling_enabled = 0;
}


void _target_X_scroll_on( void )
{
  scrolling_enabled = 1;
}


void _target_X_clear( void )
{
  XFillRectangle( disp, pixmap, rev_gc, 0, 0, 640, 400 );
}
#endif


#ifdef USE_CURSES
int _target_curses_init( void )
{
  initscr();		/* curses call */
  start_color();	/* curses call */
  cbreak();		/* curses call */
  noecho();		/* curses call */
  
  nonl();			/* optional curses call */
  intrflush(stdscr, FALSE);	/* optional curses call */
  keypad(stdscr, TRUE);		/* optional curses call */

  init_pair( 1, COLOR_RED,   COLOR_BLACK );
  init_pair( 2, COLOR_BLUE,  COLOR_BLACK );
  init_pair( 3, COLOR_WHITE, COLOR_BLACK );
  init_pair( 4, COLOR_GREEN, COLOR_BLACK );
  
  return 1;
}


void _target_curses_close( void )
{
  endwin();				/* curses call */
}


void _target_curses_getmaxyx( int *maxy, int *maxx )
{
  getmaxyx( stdscr, *maxy, *maxx );
}


void _target_curses_getyx( int *my_y, int *my_x )
{
  getyx( stdscr, *my_y, *my_x );
}


void _target_curses_refresh( void )
{
  refresh();
}


void _target_curses_setcolorpair( int n )
{
  attron( COLOR_PAIR(n) );
}


void _target_curses_setnormalattr( void )
{
  attrset( A_NORMAL );
}


void _target_curses_addch( unsigned char ch )
{
  addch( A_ALTCHARSET|ch );
}


void _target_curses_addstr( char *str )
{
  addstr( str );
}


void _target_curses_move( int y, int x )
{
  move( y, x );
}


void _target_curses_scrolldown( void )
{
}


void _target_curses_scroll_off( void )
{
  scrollok( stdscr, FALSE );
}


void _target_curses_scroll_on( void )
{
  scrollok( stdscr, TRUE );
}


void _target_curses_cursor_off( void )
{
  curs_set(0);			/* invisible cursor */
}


void _target_curses_cursor_on( void )
{
  curs_set(1);			/* visible cursor */
}


void _target_curses_clear( void )
{
  erase();
}
#endif



#ifdef DOUBLEWIDE_FONT_HACK
int _target_dwcurses_init( void )
{
  int i;
  int handle;
  struct vt_consize vc;

  if( (i=fork()) == 0 ) {
    system("setfont /home/boultonj/test/newfont");
    exit(0);
  }
  else if( i != -1 ) {
    waitpid( i, &i, 0 );
  }

  if( (handle = open( "/dev/tty", O_WRONLY )) == -1 ) {
    perror( "open(\"/dev/tty\")" );
    return 0;
  }
  
  vc.v_rows = 25;
  vc.v_cols = 80;

  vc.v_clin = 16;
  vc.v_ccol = 8;

  vc.v_vlin = vc.v_rows * vc.v_clin;
  vc.v_vcol = vc.v_cols * vc.v_ccol;

  if( ioctl( handle, VT_RESIZEX, &vc ) == -1 ) {
    perror( "VT_RESIZEX" );
    return 0;
  }

  initscr();		/* curses call */
  start_color();	/* curses call */
  cbreak();		/* curses call */
  noecho();		/* curses call */
  
  nonl();			/* optional curses call */
  intrflush(stdscr, FALSE);	/* optional curses call */
  keypad(stdscr, TRUE);		/* optional curses call */

  init_pair( 1, COLOR_RED,   COLOR_BLACK );
  init_pair( 2, COLOR_BLUE,  COLOR_BLACK );
  init_pair( 3, COLOR_WHITE, COLOR_BLACK );
  init_pair( 4, COLOR_GREEN, COLOR_BLACK );

  return 1;
}


void _target_dwcurses_close( void )
{
  int i;
  if( (i=fork()) == 0 ) {
    system("setfont default8x16");
    exit(0);
  }
  else if( i != -1 ) {
    waitpid( i, &i, 0 );
  }
  endwin();				/* curses call */
}


void _target_dwcurses_getmaxyx( int *maxy, int *maxx )
{
  getmaxyx( stdscr, *maxy, *maxx );
  *maxx /= 2;
}


void _target_dwcurses_getyx( int *my_y, int *my_x )
{
  getyx( stdscr, *my_y, *my_x );
  *my_x /= 2;
}


void _target_dwcurses_refresh( void )
{
  refresh();
}


void _target_dwcurses_setcolorpair( int n )
{
  attron( COLOR_PAIR(n) );
}


void _target_dwcurses_setnormalattr( void )
{
  attrset( A_NORMAL );
}


void _target_dwcurses_addch( unsigned char ch )
{
  unsigned char c = first_char_table[ch];
  addch( A_ALTCHARSET|c );
  if( c>=128 )
    addch( A_ALTCHARSET|(c+64) );
  else
    addch( A_ALTCHARSET|(c+1) );
}


void _target_dwcurses_move( int y, int x )
{
  move( y, 2*x );
}


void _target_dwcurses_scrolldown( void )
{
}


void _target_dwcurses_scroll_off( void )
{
  scrollok( stdscr, FALSE );
}


void _target_dwcurses_scroll_on( void )
{
  scrollok( stdscr, TRUE );
}


void _target_dwcurses_cursor_off( void )
{
  curs_set(0);			/* invisible cursor */
}


void _target_dwcurses_cursor_on( void )
{
  curs_set(1);			/* visible cursor */
}


void _target_dwcurses_clear( void )
{
  erase();
}
#endif



/* ---------------------------------------------------- */
/*                                                      */
/*                    GENERIC                           */
/*                                                      */
/* ---------------------------------------------------- */


void screen_refresh( void )
{
  (*scrn_ptr_refresh)();
}


void screen_addch( unsigned char ch )
{
  (*scrn_ptr_addch)( ch );
}


void screen_move( int y, int x )
{
  (*scrn_ptr_move)( y, x );
}


void screen_mvaddch( int y, int x, unsigned char ch )
{
  (*scrn_ptr_move)( y, x );
  (*scrn_ptr_addch)( ch );
}


void screen_addstr( char *str )
{
  if( scrn_ptr_addstr )
    (*scrn_ptr_addstr)( str );
  else 
    while( *str )
      screen_addch( *str++ );
}


void screen_mvaddstr( int y, int x, char *str )
{
  screen_move( y, x );
  screen_addstr( str );
}


void screen_setcolorpair( int n )
{
  (*scrn_ptr_setcolorpair)( n );
}


void screen_setnormalattr( void )
{
  (*scrn_ptr_setnormalattr)();
}


void screen_getmaxyx( int *maxy, int *maxx )
{
  (*scrn_ptr_getmaxyx)( maxy, maxx );
}


void screen_getyx( int *my_y, int *my_x )
{
  (*scrn_ptr_getyx)( my_y, my_x );
}


void screen_scroll_off( void )
{
  (*scrn_ptr_scroll_off)();
}


void screen_scroll_on( void )
{
  (*scrn_ptr_scroll_on)();
}


void screen_cursor_off( void )
{
  (*scrn_ptr_cursor_off)();
}


void screen_cursor_on( void )
{
  (*scrn_ptr_cursor_on)();
}


void screen_clear( void )
{
  (*scrn_ptr_clear)();
}


void screen_close( void )
{
  if( scrn_ptr_close ) {
    (*scrn_ptr_close)();
    scrn_ptr_init = NULL;
    scrn_ptr_close = NULL;
    scrn_ptr_addch = NULL;
    scrn_ptr_addstr = NULL;
    scrn_ptr_move = NULL;
    scrn_ptr_refresh = NULL;
    scrn_ptr_setcolorpair = NULL;
    scrn_ptr_setnormalattr = NULL;
    scrn_ptr_getyx = NULL;
    scrn_ptr_getmaxyx = NULL;
    scrn_ptr_scrolldown = NULL;
    scrn_ptr_scroll_off = NULL;
    scrn_ptr_scroll_on = NULL;
    scrn_ptr_cursor_off = NULL;
    scrn_ptr_cursor_on = NULL;
    scrn_ptr_clear = NULL;
  }
}



int screen_init( enum DisplayType mode )
{
  screen_close();

  switch( mode ) {
#ifdef USE_XWIN
    case DT_XWIN:
      scrn_ptr_init = _target_X_init;
      scrn_ptr_close = _target_X_close;
      scrn_ptr_addch = _target_X_addch;
      scrn_ptr_addstr = NULL;
      scrn_ptr_move = _target_X_move;
      scrn_ptr_refresh = _target_X_refresh;
      scrn_ptr_setcolorpair = _target_X_setcolorpair;
      scrn_ptr_setnormalattr = _target_X_setnormalattr;
      scrn_ptr_getyx = _target_X_getyx;
      scrn_ptr_getmaxyx = _target_X_getmaxyx;
      scrn_ptr_scrolldown = _target_X_scrolldown;
      scrn_ptr_scroll_off = _target_X_scroll_off;
      scrn_ptr_scroll_on = _target_X_scroll_on;
      scrn_ptr_cursor_off = _vidutil_null;
      scrn_ptr_cursor_on = _vidutil_null;
      scrn_ptr_clear = _target_X_clear;
      break;
#endif
#ifdef USE_SVGALIB
    case DT_SVGALIB:
      scrn_ptr_init = _target_svgalib_init;
      scrn_ptr_close = _target_svgalib_close;
      scrn_ptr_addch = _target_svgalib_addch;
      scrn_ptr_addstr = NULL;
      scrn_ptr_move = _target_svgalib_move;
      scrn_ptr_refresh = _target_svgalib_refresh;
      scrn_ptr_setcolorpair = _target_svgalib_setcolorpair;
      scrn_ptr_setnormalattr = _target_svgalib_setnormalattr;
      scrn_ptr_getyx = _target_svgalib_getyx;
      scrn_ptr_getmaxyx = _target_svgalib_getmaxyx;
      scrn_ptr_scrolldown = _target_svgalib_scrolldown;
      scrn_ptr_scroll_off = _target_svgalib_scroll_off;
      scrn_ptr_scroll_on = _target_svgalib_scroll_on;
      scrn_ptr_cursor_off = _vidutil_null;
      scrn_ptr_cursor_on = _vidutil_null;
      scrn_ptr_clear = _target_svgalib_clear;
      break;
#endif
#ifdef USE_CURSES
    case DT_CURSES:
      scrn_ptr_init = _target_curses_init;
      scrn_ptr_close = _target_curses_close;
      scrn_ptr_addch = _target_curses_addch;
      scrn_ptr_addstr = _target_curses_addstr;
      scrn_ptr_move = _target_curses_move;
      scrn_ptr_refresh = _target_curses_refresh;
      scrn_ptr_setcolorpair = _target_curses_setcolorpair;
      scrn_ptr_setnormalattr = _target_curses_setnormalattr;
      scrn_ptr_getyx = _target_curses_getyx;
      scrn_ptr_getmaxyx = _target_curses_getmaxyx;
      scrn_ptr_scrolldown = _target_curses_scrolldown;
      scrn_ptr_scroll_off = _target_curses_scroll_off;
      scrn_ptr_scroll_on = _target_curses_scroll_on;
      scrn_ptr_cursor_off = _target_curses_cursor_off;
      scrn_ptr_cursor_on = _target_curses_cursor_on;
      scrn_ptr_clear = _target_curses_clear;
      break;
#endif
#ifdef DOUBLEWIDE_FONT_HACK
    case DT_CURSES_DOUBLEWIDE:
      scrn_ptr_init = _target_dwcurses_init;
      scrn_ptr_close = _target_dwcurses_close;
      scrn_ptr_addch = _target_dwcurses_addch;
      scrn_ptr_addstr = NULL;
      scrn_ptr_move = _target_dwcurses_move;
      scrn_ptr_refresh = _target_dwcurses_refresh;
      scrn_ptr_setcolorpair = _target_dwcurses_setcolorpair;
      scrn_ptr_setnormalattr = _target_dwcurses_setnormalattr;
      scrn_ptr_getyx = _target_dwcurses_getyx;
      scrn_ptr_getmaxyx = _target_dwcurses_getmaxyx;
      scrn_ptr_scrolldown = _target_dwcurses_scrolldown;
      scrn_ptr_scroll_off = _target_dwcurses_scroll_off;
      scrn_ptr_scroll_on = _target_dwcurses_scroll_on;
      scrn_ptr_cursor_off = _target_dwcurses_cursor_off;
      scrn_ptr_cursor_on = _target_dwcurses_cursor_on;
      scrn_ptr_clear = _target_dwcurses_clear;
      break;
#endif
    default:
      fprintf( stderr, "That display mode is not available.\n" );
      return 0;
  }
  
  if( scrn_ptr_init )
    return (*scrn_ptr_init)();

  return 1;
}
