/* vim:sw=4:sts=4
 * Lua Gtk2 binding.
 */

#include "luagtk.h"
#include <lauxlib.h>	    // luaL_error
#include <string.h>	    // strcmp
#include <errno.h>	    // errno

/**
 * Overrides for existing Gtk/Gdk functions.
 *
 * @class module
 * @name gtk.override
 */

#define WIDGET_ARG(name, cls, ptr, idx) cls ptr name = \
    (cls ptr) get_widget(L, idx, #cls)->p

/**
 * Get a widget from the Lua stack, checking its type (class).  It should at
 * least be some kind of widget, not an unrelated userdata - might crash
 * otherwise.
 *
 * Note: a more thorough implementation could get the metatable and check that
 * it has a _classname attribute etc.
 */
static struct widget *get_widget(lua_State *L, int index, const char *name)
{
    luaL_checktype(L, index, LUA_TUSERDATA);
    struct widget *w = (struct widget*) lua_topointer(L, index);
    if (!strcmp(name, WIDGET_NAME(w)))
	return w;
    {
	char msg[100];
	snprintf(msg, sizeof(msg), "expected %s, is %s", name, WIDGET_NAME(w));
	luaL_argerror(L, index, msg);
	return NULL;
    }
}


/**
 * g_type_from_name fails unless the type has been initialized before.  Use
 * my wrapper that handles the initialization when required.
 *
 * @name g_type_from_name
 * @luaparam name  The type name to look up.
 * @luareturn  The type number, or nil on error.
 */
static int l_g_type_from_name(lua_State *L)
{
    const char *s = luaL_checkstring(L, 1);
    GTK_INITIALIZE();
    int type_nr = luagtk_g_type_from_name(s);
    if (!type_nr)
	return 0;
    lua_pushnumber(L, type_nr);
    return 1;
}


/**
 * Perform the G_OBJECT_GET_CLASS on an object.
 *
 * @name g_object_get_class
 * @luaparam object
 * @luareturn The class object (GObjectClass) of the object, or nil on error.
 */
static int l_g_object_get_class(lua_State *L)
{
    GObject *parent = (GObject*) lua_topointer(L, 1);
    GObjectClass *c = G_OBJECT_GET_CLASS(parent);
    const struct type_info *ti = find_struct("GObjectClass", 1);
    if (!ti)
	luaL_error(L, "%s type GObjectClass unknown.\n", msgprefix);

    // flags is 0 - use the default widget type, and increase refcount for this
    // non-new object.
    luagtk_get_widget(L, c, ti - type_list, FLAG_NOT_NEW_OBJECT);
    return 1;
}

/* XXX an ad-hoc created function */
static int l_g_object_get_class_name(lua_State *L)
{
    luaL_checktype(L, 1, LUA_TUSERDATA);
    struct widget *w = (struct widget*) lua_topointer(L, 1);
    lua_pushstring(L, WIDGET_NAME(w));
    return 1;
}

// determine the required GValue type
static int l_gtk_list_store_set_value(lua_State *L)
{
    WIDGET_ARG(store, GtkListStore, *, 1);
    WIDGET_ARG(iter, GtkTreeIter, *, 2);
    int column = luaL_checkinteger(L, 3);

    GType t = gtk_tree_model_get_column_type((GtkTreeModel*) store, column);
    GValue gvalue = {0};

    if (!luagtk_fill_gvalue(L, &gvalue, t, 4))
	return luaL_error(L, "%s can't set GValue of type %s",
	    msgprefix, g_type_name(t));

    gtk_list_store_set_value(store, iter, column, &gvalue);
    return 0;
}

    

/**
 * Call this function and return the result as a Lua string.
 * 
 * @name gdk_pixbuf_save_to_buffer
 * @luaparam pixbuf  The pixbuf to convert
 * @luaparam type  The output format, e.g. "jpeg"
 * @luareturn  The converted pixbuf as string, or nil on error
 */
static int l_gdk_pixbuf_save_to_buffer(lua_State *L)
{
    struct widget *w = (struct widget*) lua_topointer(L, 1);
    GdkPixbuf *pixbuf = (GdkPixbuf*) w->p;
    gchar *buffer = NULL;
    gsize buffer_size = 0;
    const char *type = lua_tostring(L, 2);
    GError *error = NULL;
    gboolean rc;

    rc = gdk_pixbuf_save_to_buffer(pixbuf, &buffer, &buffer_size, type, &error,
	NULL);
    if (buffer) {
	lua_pushlstring(L, buffer, buffer_size);
	g_free(buffer);
	return 1;
    }

    return 0;
}

/**
 * Set a property of an object.  The difficulty is to first convert the value
 * to a GValue of the correct type, because that is what the GLib function
 * expects.
 *
 * @name g_object_set_property
 * @luaparam object   Object derived from GObject
 * @luaparam property_name
 * @luaparam value    The value to set the property to
 */
static int l_g_object_set_property(lua_State *L)
{
    struct widget *w = (struct widget*) lua_topointer(L, 1);
    struct widget_type *wt = luagtk_get_widget_type(w);

    if (!wt) {
	printf("%s invalid object in l_g_object_set_property.\n", msgprefix);
	return 0;
    }

    // this widget must be one derived from gobject.
    if (strcmp(wt->name, "gobject")) {
	printf("%s g_object_set_property on a %s object\n", msgprefix,
	    wt->name);
	return 0;
    }

    lua_getmetatable(L, 1);
    lua_getfield(L, -1, "_gtktype");
    GType type = lua_tointeger(L, -1);
    GObjectClass *oclass = (GObjectClass*) g_type_class_ref(type);
    const gchar *prop_name = luaL_checkstring(L, 2);

    // find the property; this searches all parent classes, too.
    GParamSpec *pspec = g_object_class_find_property(oclass, prop_name);
    if (!pspec) {
	printf("g_object_set_property: no property %s.%s\n", WIDGET_NAME(w),
	    prop_name);
	return 0;
    }

    GValue gvalue = {0};
    if (luagtk_fill_gvalue(L, &gvalue, pspec->value_type, 3))
	g_object_set_property((GObject*) w->p, prop_name, &gvalue);
    g_type_class_unref(oclass);
    return 0;
}


/* The following code for timeout and idle is not absolutely necessary.  The
 * generic callback code with closures and handling of conversion of any Lua
 * data type to void* can handle this, but this is cleaner and doesn't leak
 * memory.  Maybe the generic code will improve, so this one can be
 * removed eventually. */

#define TIMEOUT_OVERRIDE
#ifdef TIMEOUT_OVERRIDE

struct timeout_data {
    lua_State *L;
    int function_ref;
    int arg_ref;	// ref to an extra argument, or 0
};

/**
 * Callback functions for timer events.
 */
static int _timeout_func(void *data)
{
    struct timeout_data *td = (struct timeout_data*) data;
    lua_State *L = td->L;
    int args_cnt = 0;

    lua_rawgeti(L, LUA_REGISTRYINDEX, td->function_ref);

    // push extra argument, if any
    if (td->arg_ref) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, td->arg_ref);
	args_cnt ++;
    }

    lua_call(L, args_cnt, 1);

    // if it returns 0 (false), the timer will be deleted -> clean up.
    // for timeouts, always stop the timer.
    int rc = lua_toboolean(L, -1);
    if (rc == 0) {
	luaL_unref(L, LUA_REGISTRYINDEX, td->function_ref);
	if (td->arg_ref)
	    luaL_unref(L, LUA_REGISTRYINDEX, td->arg_ref);
	g_free(data);
	rc = 0;
    }
    return rc;
}

/**
 * Allocate a struct timeout_data for certain callbacks.
 *
 * The stack contains a function and zero or more arguments starting at
 * the given index.
 */
static struct timeout_data *_build_timeout_callback(lua_State *L, int index)
{
    luaL_checktype(L, index, LUA_TFUNCTION);
    luaL_checkany(L, index + 1);
    struct timeout_data *td = g_malloc(sizeof(*td));
    td->L = L;
    lua_pushvalue(L, index);
    td->function_ref = luaL_ref(L, LUA_REGISTRYINDEX);

    // collect extra argument if not NIL
    if (!lua_isnil(L, index + 1)) {
	lua_pushvalue(L, index + 1);
	td->arg_ref = luaL_ref(L, LUA_REGISTRYINDEX);
    } else {
	td->arg_ref = 0;
    }

    return td;
}


/**
 * Add a timeout handler.  The handler will receive the "data" argument
 * as argument.
 *
 * @name g_timeout_add
 * @luaparam interval  sleep time in ms
 * @luaparam function  the function to call
 * @luaparam data  extra argument of any data type to pass to the callback.
 */
static int l_g_timeout_add(lua_State *L)
{
    int interval = luaL_checkint(L, 1);
    struct timeout_data *td = _build_timeout_callback(L, 2);
    guint id = g_timeout_add(interval, _timeout_func, td);
    lua_pushnumber(L, id);
    return 1;
}


/**
 * Add an idle handler.
 *
 * @name g_idle_add
 * @luaparam callback
 * @luaparam data  An extra parameter of any data type for the callback
 * @luareturn  An ID of the idle handler.
 */
static int l_g_idle_add(lua_State *L)
{
    luaL_checktype(L, 1, LUA_TFUNCTION);
    struct timeout_data *td = _build_timeout_callback(L, 1);
    guint id = g_idle_add(_timeout_func, td);
    lua_pushnumber(L, id);
    return 1;
}

#endif

/**
 * gtk_text_view_get_buffer doesn't increase the refcount.
 */
static int l_gtk_text_view_get_buffer(lua_State *L)
{
    luaL_checktype(L, 1, LUA_TUSERDATA);
    struct widget *w = (struct widget*) lua_topointer(L, 1);
    GtkTextBuffer *buf = gtk_text_view_get_buffer(w->p);
    luagtk_get_widget(L, buf, 0, FLAG_NOT_NEW_OBJECT);
    return 1;
}


/**
 * Find a table element, possibly in a subtable by considering "." elements
 * of the name.  This is similar to lauxlib.c:luaL_findtable.
 *
 * @param L  Lua State
 * @param idx  Stack position of the table to look in
 * @param fname  Name of the field to look up
 * @return  0 if OK, 1 on error.  If OK, the Lua stack contains the found
 *   item.
 */
static int resolve_dotted_name(lua_State *L, int idx, const char *fname)
{
    const char *e;
    lua_pushvalue(L, idx);

    do {
	e = strchr(fname, '.');
	if (!e)
	    e = fname + strlen(fname);
	lua_pushlstring(L, fname, e - fname);
	lua_rawget(L, -2);

	// if not found, return an error.
	if (lua_isnil(L, -1)) {
	    lua_pop(L, 2);
	    return 1;
	}

	lua_remove(L, -2);	// remove previous table
	fname = e + 1;
    } while (*e == '.');

    return 0;
}


/**
 * If a function is not always available in Gtk, retrieve it with
 * this helper; it throws an error if the function is not available.
 *
 * This ensures compatibility with older Gtk versions.
 */
static void *optional_func(lua_State *L, const char *name,
    const char *min_version)
{
    struct func_info fi;
    if (!find_func(name, &fi))
	luaL_error(L, "%s function %s not defined.  Please use at least %s",
	    msgprefix, name, min_version);
    return fi.func;
}



/**
 * Connect one signal.
 *
 * The connect_object and flags are ignored.  The handlers are looked up
 * in the provided table.
 *
 * Lua stack: [1]=builder, [2]=handler table
 */
static void _connect_func(GtkBuilder *builder, GObject *object,
    const gchar *signal_name, const gchar *handler_name,
    GObject *connect_object, GConnectFlags flags, gpointer user_data)
{
    lua_State *L = (lua_State*) user_data;
    int stack_top = lua_gettop(L);

    // first, the function to call
    lua_pushcfunction(L, luagtk_connect);

    // arg 1: widget
    luagtk_get_widget(L, object, 0, FLAG_NOT_NEW_OBJECT);
    if (lua_isnil(L, -1)) {
	printf("_connect_func: failed to find the widget\n");
	goto ex;
    }

    // arg 2: signal name
    lua_pushstring(L, signal_name);

    if (G_UNLIKELY(connect_object)) {
	struct widget *w = (struct widget *) lua_topointer(L, -2);
	const gchar *(*func)(GObject*) = optional_func(L,
	    "gtk_buildable_get_name", "Gtk 2.12");
	printf("%s object given for signal %s %s.%s - not supported.\n",
	    msgprefix, WIDGET_NAME(w), func(object), signal_name);
    }

    // arg 3: handler function.  Look it up in the given handler table;
    // subtables can be accessed, too.
    if (resolve_dotted_name(L, 2, handler_name)) {
	printf("%s signal handler %s not found.\n", msgprefix, handler_name);
	goto ex;
    }

    // got it.
    lua_call(L, 3, 1);

ex:
    lua_settop(L, stack_top);
}


/**
 * Autoconnect all signals.  The _full variant is used because it has a 
 * callback for the connection of the functions, so we can look for an
 * appropriate Lua function here.
 *
 * @luaparam builder  A builder object
 * @luaparam tbl  A table with signal handlers (optional, default is _G)
 */
static int l_gtk_builder_connect_signals_full(lua_State *L)
{
    WIDGET_ARG(builder, GtkBuilder, *, 1);

    void (*func)(GtkBuilder*, GtkBuilderConnectFunc, gpointer)
	= optional_func(L, "gtk_builder_connect_signals_full", "Gtk 2.12");

    switch (lua_gettop(L)) {
	case 1:
	    lua_pushvalue(L, LUA_GLOBALSINDEX);
	    break;
	
	case 2:
	    luaL_checktype(L, 2, LUA_TTABLE);
	    break;
	
	default:
	    return luaL_error(L, "too many arguments");
    }

    func(builder, _connect_func, L);
    return 0;
}


// I've run into this problem before; just use that routine.
static GType _my_get_type_from_name(GtkBuilder *builder, const gchar *name)
{
    return luagtk_g_type_from_name(name);
}


/**
 * Plug in a better get_type_from_name.  The default routine,
 * gtk_builder_real_get_type_from_name, using _gtk_builder_resolve_type_lazily,
 * doesn't work when the executable is not linked with libgtk2.0, as is the
 * case here.
 */
static int l_gtk_builder_new(lua_State *L)
{
    GTK_INITIALIZE();

    GtkBuilder *(*func)(void) = optional_func(L, "gtk_builder_new", "Gtk 2.12");
    GtkBuilder *builder = func();
    GtkBuilderClass *c = (GtkBuilderClass*) G_OBJECT_GET_CLASS(builder);
    c->get_type_from_name = _my_get_type_from_name;
    luagtk_get_widget(L, builder, 0, FLAG_NEW_OBJECT);
    return 1;
}


/*-
 * args: model, iter, column
 * returns: GValue
 */
static int l_gtk_tree_model_get_value(lua_State *L)
{
    struct widget *model = (struct widget*) lua_topointer(L, 1);
    struct widget *iter = (struct widget*) lua_topointer(L, 2);
    int column = lua_tonumber(L, 3);
    GValue gvalue = { 0 };

    gtk_tree_model_get_value(model->p, iter->p, column, &gvalue);
    luagtk_push_gvalue(L, &gvalue);
    return 1;
}

// avoid the warning about superfluous arguments if given as callback
static int l_gtk_main_quit(lua_State *L)
{
    lua_settop(L, 0);
    return luagtk_call_byname(L, "gtk_main_quit");
}

// the output array size isn't easily determined.
static int l_pango_tab_array_get_tabs(lua_State *L)
{
    WIDGET_ARG(tab_array, PangoTabArray, *, 1);
    int t2, t3, i, n;

    PangoTabAlign *alignments = NULL;
    gint *locations = NULL;

    t2 = lua_type(L, 2);
    luaL_argcheck(L, t2 == LUA_TNIL || t2 == LUA_TTABLE, 2, "nil or table");

    t3 = lua_type(L, 3);
    luaL_argcheck(L, t3 == LUA_TNIL || t3 == LUA_TTABLE, 3, "nil or table");

    n = pango_tab_array_get_size(tab_array);

    pango_tab_array_get_tabs(tab_array, t2 == LUA_TNIL ? NULL : &alignments,
	t3 == LUA_TNIL ? NULL : &locations);

    // copy the values to the table
    if (alignments) {
	luagtk_empty_table(L, 2);
	const struct type_info *ti = find_struct("PangoTabAlign", 0);
	for (i=0; i<n; i++) {
	    luagtk_enum_push(L, alignments[i], ti - type_list);
	    lua_rawseti(L, 2, i + 1);
	}
	g_free(alignments);
    }

    if (locations) {
	luagtk_empty_table(L, 3);
	for (i=0; i<n; i++) {
	    lua_pushnumber(L, locations[i]);
	    lua_rawseti(L, 3, i + 1);
	}
	g_free(locations);
    }

    return 0;
}

/**
 * Streaming capable converter.  It will convert as much as possible from
 * the input buffer; it may leave some bytes unused, which you should prepend
 * to any data read later.
 *
 * @name g_iconv
 * @luaparam converter  GIConv as returned from g_iconv_open
 * @luaparam inbuf  The string to convert
 * @return  status (0=ok, <1=error)
 * @return  The converted string
 * @return  Remaining unconverted string
 *
 */
static int l_g_iconv(lua_State *L)
{
    WIDGET_ARG(converter, GIConv, , 1);
    gsize ilen, olen, olen2;
    gchar *inbuf = (gchar*) luaL_checklstring(L, 2, &ilen);
    char *obuf, *obuf2;
    int rc, result=0;

    if (lua_gettop(L) != 2)
	return luaL_error(L, "%s g_iconv(converter, inbuf) not called "
	    "properly.  Note that the luagtk API differs from the C API here.",
	    msgprefix);

    // happens on windows sometimes.
    if (converter == (GIConv) -1) {
	lua_pushnumber(L, 0);
	lua_pushvalue(L, 2);
	lua_pushliteral(L, "");
	return 3;
    }

    luaL_Buffer buf;
    luaL_buffinit(L, &buf);
    olen = ilen;
    obuf = g_malloc(olen);

    while (ilen > 0) {
	obuf2 = obuf;
	olen2 = olen;
	rc = g_iconv(converter, &inbuf, &ilen, &obuf2, &olen2);

	// push whatever has been converted in this run
	luaL_addlstring(&buf, obuf, obuf2 - obuf);

	if (rc < 0) {
	    // illegal sequence
	    if (errno == EILSEQ) {
		result = -errno;
		break;
	    }

	    // ends with an incomplete sequence - is ok
	    if (errno == EINVAL)
		break;

	    // full, don't worry continue
	    if (errno != E2BIG) {
		result = -errno;
		break;
	    }
		
	} else
	    result += rc;
    }

    lua_pushinteger(L, result);
    luaL_pushresult(&buf);
    lua_pushlstring(L, inbuf, ilen);
    g_free(obuf);

    return 3;
}

#define WIDGET_FLAGS_GFREE 1
#define WIDGET_FLAGS_ATTR_FREE 2
#define WIDGET_FLAGS_GLYPH_FREE 3

// Call a function that returns a GSList.  Set the widget flags.
static int _set_flag(lua_State *L, const char *funcname, int flag)
{
    int rc = luagtk_call_byname(L, funcname);
    struct widget *w = (struct widget*) lua_topointer(L, -1);
    w->flags = flag;
    return rc;
}

/* need to free strings: FSO=free string override */
#define FSO(x, f) static int l_##x(lua_State *L) { return _set_flag(L, #x, f); }

FSO(gtk_stock_list_ids, WIDGET_FLAGS_GFREE)
FSO(gtk_file_chooser_get_filenames, WIDGET_FLAGS_GFREE)
FSO(gtk_file_chooser_get_uris, WIDGET_FLAGS_GFREE)
FSO(gtk_file_chooser_list_shortcut_folders, WIDGET_FLAGS_GFREE);
FSO(gtk_file_chooser_list_shortcut_folder_uris, WIDGET_FLAGS_GFREE);
FSO(pango_attr_iterator_get_attrs, WIDGET_FLAGS_ATTR_FREE);
FSO(pango_glyph_item_apply_attrs, WIDGET_FLAGS_GLYPH_FREE);


#if 0
static int l_pango_attr_iterator_get_font(lua_State *L)
{
    int rc = luagtk_call_byname(L, "pango_attr_iterator_get_font");
    // XXX this is an optional return value... where is it on the stack?
    struct widget *w = (struct widget*) lua_topointer(L, WHATEVER);
    w->flags = WIDGET_FLAGS_ATTR_FREE;
    return rc;
}
#endif

/**
 * Override for g_slist_free: if the "data" part of each list element
 * is to be freed, do it automatically.
 *
 * After calling g_slist_free, the list doesn't exist anymore.  This is shown
 * by setting w->p to NULL and w->is_deleted to 1.
 */
static int l_g_slist_free(lua_State *L)
{
    struct widget *w = get_widget(L, 1, "GSList");
    void (*func)(void*) = NULL;

    switch (w->flags) {
	case WIDGET_FLAGS_GFREE:
	func = g_free;
	break;

	case WIDGET_FLAGS_ATTR_FREE:
	func = optional_func(L, "pango_attribute_destroy", "Pango 1.0");
	break;

	case WIDGET_FLAGS_GLYPH_FREE:
	func = optional_func(L, "pango_glyph_item_free", "Pango 1.6");
	break;
    }

    if (func) {
	GSList *l = (GSList*) w->p;
	while (l) {
	    func(l->data);
	    l->data = NULL;
	    l = l->next;
	}
    }

    int rc = luagtk_call_byname(L, "g_slist_free");
    w->p = NULL;
    w->is_deleted = 1;
    return rc;
}


/**
 * This function actually frees the widget.  Don't do that again when
 * collecting garbage.
 */
static int l_g_tree_destroy(lua_State *L)
{
    struct widget *w = get_widget(L, 1, "GTree");
    int rc = luagtk_call_byname(L, "g_tree_destroy");
    w->p = NULL;
    w->is_deleted = 1;
    return rc;
}


/**
 * The function g_atexit cannot be used.  The reason is that before such
 * atexit routines are called by libc, shared libraries may already have been
 * unloaded, or the memory containing the closure may have been freed.  This
 * all can lead to SEGV on program exit.  Disallow usage of atexit.
 */
static int l_g_atexit(lua_State *L)
{
    return luaL_error(L, "g_atexit called. This is not allowed; see "
	"documentation.");
}

/* overrides */
#define OVERRIDE(x) { #x, l_##x }
static const luaL_reg luagtk_overrides[] = {
    OVERRIDE(g_type_from_name),
    OVERRIDE(g_object_get_class),
    OVERRIDE(g_object_set_property),
    OVERRIDE(gdk_pixbuf_save_to_buffer),
    OVERRIDE(gtk_tree_model_get_value),
    OVERRIDE(gtk_text_view_get_buffer),
    OVERRIDE(gtk_main_quit),
    OVERRIDE(g_atexit),
    OVERRIDE(pango_tab_array_get_tabs),
    OVERRIDE(g_iconv),

    /* SList freeing */
    OVERRIDE(g_slist_free),
    OVERRIDE(gtk_stock_list_ids),
    OVERRIDE(gtk_file_chooser_get_filenames),
    OVERRIDE(gtk_file_chooser_get_uris),
    OVERRIDE(gtk_file_chooser_list_shortcut_folders),
    OVERRIDE(gtk_file_chooser_list_shortcut_folder_uris),
    OVERRIDE(pango_attr_iterator_get_attrs),
    OVERRIDE(pango_glyph_item_apply_attrs),

    OVERRIDE(g_tree_destroy),

#ifdef TIMEOUT_OVERRIDE
    OVERRIDE(g_timeout_add),
    OVERRIDE(g_idle_add),
#endif

    OVERRIDE(g_object_get_class_name),
    OVERRIDE(gtk_list_store_set_value),
    OVERRIDE(gtk_builder_new),
    OVERRIDE(gtk_builder_connect_signals_full),

    // in callback.c
    {"g_object_connect", luagtk_connect },
    {"g_object_disconnect", luagtk_disconnect },

    { NULL, NULL }
};

void luagtk_init_overrides(lua_State *L)
{
    luaL_register(L, NULL, luagtk_overrides);
}

