/*
 * cmd-list.c
 *	Functions for listing things.
 *
 * Copyright (C) 1993	Lysator computer Club,
 *			Linkoping University,  Sweden
 *
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 * Author:	Linus Tolke
 *		Lysator Computer Club
 *		Linkoping University
 *		Sweden
 *
 * email: 	linus@lysator.liu.se
 *
 *
 *  Any opinions expressed in this code are the author's PERSONAL
 *  opinions, and does NOT represent any official standpoint of
 *  Lysator, even if so stated.
 */

#include <config.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <kom-types.h>
#include <zmalloc.h>
#include <s-string.h>

#include <libintl.h>

#include <kom-errno.h>
#include <services.h>

#include "xoutput.h"

#include "copy.h"
#include "error.h"
#include "cache.h"
#include "quit.h"

#include "internal.h"
#include "offline.h"

#include "cmds.h"

/* Comparing function. */
static int time_before_time (struct tm a, struct tm b)
{	/* Returns > 0 if a is a time before b */
        /* Returns < 0 if a is a time after b */
    /* The granularity is 1 second. */
    if (a.tm_year > b.tm_year)
        return 10;
    if (a.tm_year < b.tm_year)
        return -10;
    if (a.tm_yday > b.tm_yday)
        return 5;
    if (a.tm_yday < b.tm_yday)
        return -5;
    if (a.tm_hour > b.tm_hour)
        return 3;
    if (a.tm_hour < b.tm_hour)
        return -3;
    if (a.tm_min > b.tm_min)
        return 2;
    if (a.tm_min < b.tm_min)
        return -2;
    if (a.tm_sec > b.tm_sec)
        return 1;
    if (a.tm_sec < b.tm_sec)
        return -1;
    return 0;
}

struct cle {
    Conf_no 	cno;
    Conference * info;
};

static int most_recent_change (const void *va, const void *vb)
{
    const struct cle * a = (const struct cle *)va;
    const struct cle * b = (const struct cle *)vb;
    return time_before_time(b->info->last_written,a->info->last_written);
}

/* if (when) end is set to -1 we are cleaning up. */
static void show_sign_of_life(int done, int end)
{
    static int last = -1;

    if (end >= 0)
    {
	int new;
	new = 100 * done / end;

	if (new == last)
	{
	    /* Do nothing. */
	}
	else
	{
	    xprintf("%d%%\r", new);
	    xflush(kom_outstream);
	    last = new;
	}
    }
    else
    {
	xprintf("       \r");
	xflush(kom_outstream);
	last = -1;
    }
}

/* 
 * List confs
 * Confs are listed in "most recently recieved article"-order.
 * 
 * The problem is thus: 
 *	- Retrieve all confs.
 *	- Sort in "most recently recieved article"-order.
 * 	- Print the confs
 */

Success
cmd_list_confs (String	argument)
{
#define FUNCTION "cmd_list_confs()"

    Conf_z_info_list	  all_confs	= EMPTY_CONF_Z_INFO_LIST;
    Success		  retval;
    struct cle * volatile conf_list = NULL; /* May be clobbered by longjmp */
    volatile int	  count = 0;	    /* May be clobbered by longjmp */
    int 		  i;
    Kom_err		  error;

#define CLEAN_UP()					\
    do {						\
	release_conf_z_info_list ( &all_confs );	\
	while (count > 0)				\
	{						\
	    count --;					\
	    release_conf(conf_list[count].info);	\
	}						\
        if (conf_list) 					\
	{ 						\
	    zfree(conf_list);				\
	}						\
    } while (0)


    TOPLOOP_SETJMP();

    OFFLINE();

    CHECK_NO_ARGUMENTS();

    retval = kom_lookup_z_name (s_fcrea_str(""), FALSE, TRUE, &all_confs);
    if (retval == FAILURE)
    {
	switch (kom_errno)
	{
	case  KOM_OUT_OF_MEMORY:
	    fatal1 (CLIENT_OUT_OF_MEMORY, "lookup_name() failed");
	    break;

	case  KOM_NO_CONNECT:
	    fatal1 (CLIENT_CONNECTION_LOST, "lookup_name() failed");
	    break;

	default:
	    fatal3(CLIENT_UNEXPECTED_SERVER_ERROR,
		   "lookup_name() failed: %d %s",
		   kom_errno, kom_errno_string());
	    break;
	}
    }

    newline();

    /* all_confs does now contain a list of all confs. */
    /* Lets fetch them all in order to sort them. */
    /* We will here simultaneously map all confs to just preserve the confs. */
    
    conf_list = (struct cle *) zmalloc(all_confs.no_of_confs 
				      * sizeof(struct cle));
    for (i=0; i < all_confs.no_of_confs; i++)
    {
	show_sign_of_life(i + 1, all_confs.no_of_confs);

	if (!all_confs.confs[i].type.letter_box)
	{
	    /* BUG: doesn't look for error */
	    conf_list[count].info = conf_stat(all_confs.confs[i].conf_no, &error);
	    conf_list[count].cno = all_confs.confs[i].conf_no;
	    /* secret confs are represented by NULL - pointer. */
	    /* Since we cannot sort them (we don't know when they last
	       recieved an article we leave them here. */
	    if (conf_list[count].info) /* Skipping secret confs. */
	    {
		count++;
	    }
	}
    }
    show_sign_of_life(-1, -1);	/* This clears the area */

    newline();

    /* Sorting. */
    qsort (conf_list, count, sizeof (struct cle), &most_recent_change);

    /* Print header line */
    xprintf(gettext("Senaste inlgg   Medl. Tot Inl   Namn (typ)\n"));
    /* Lets print what we have got. */
    for (i = 0; i < count; i++)
    {
	/* Print last written time. */
	print_time(&conf_list[i].info->last_written);

	/* Print number of members. */
	xprintf(gettext("%5d"), conf_list[i].info->no_of_members);

	/* Print number of articles. */
	xprintf(gettext("%8d"), conf_list[i].info->first_local_no + 
		conf_list[i].info->no_of_texts - 1);

	/* Print optional * for not member */
	if (locate_membership (conf_list[i].cno, cpn, NULL) == -1)
	{
	    xprintf(gettext(" *"));
	} else {
	    xprintf(gettext("  "));
	}

	/* Print optional O for is supervisor. */
	/* Should really search the conf for membership. */
	if (cpn == conf_list[i].info->supervisor)
	{
	    xprintf(gettext("O "));
	} else {
	    xprintf(gettext("  "));
	}

	/* Print conf name. */
	xprintf("%S",conf_list[i].info->name);

	/* Is the conf closed? */
	if(conf_list[i].info->type.rd_prot)
	    xprintf(gettext(" (slutet)"));
	/* Original? */
	if(conf_list[i].info->type.original)
	    xprintf(gettext(" (original)"));
	newline();
    }
    RETURN (OK);
#undef FUNCTION
#undef CLEAN_UP
}



/*  Show a list of texts in conf  */
Success
cmd_list_texts (String   argument)
{
#define FUNCTION "cmd_list_texts()"

    Kom_err		  error;
    Conference * volatile conf = NULL;
    int			  pos;
    Text_list * volatile  map = NULL;
    Text_stat * volatile  textstat = NULL;
    String                text = EMPTY_STRING;
    String_size           end_of_line_pos;


#define CLEAN_UP()	do {	release_conf (conf); \
				if (conf) zfree(conf); \
			        if (map) zfree(map); \
				if (textstat) zfree(textstat); \
			} while (0)

    TOPLOOP_SETJMP();

    OFFLINE();

    CHECK_NO_ARGUMENTS();

    if(!ccn)
    {
        xprintf(gettext("\nDu mste vara i ngot mte.\n"));
	return (FAILURE);
    }

    if (((conf=conf_stat(ccn, &error)) == NULL)
	|| (((map=(Text_list *) zcalloc(sizeof(Text_list),conf->no_of_texts))
	     == NULL)
	    && (error != KOM_NO_ERROR)))
    {
        xprintf(gettext("Minnet har tagit slut..\n"));
        RETURN (FAILURE);
    }

    kom_get_map(ccn, conf->first_local_no, conf->no_of_texts, map);

    pos=conf->no_of_texts-1;

    zfree(conf);
    conf=NULL;

    xprintf(gettext("\nInlgg  Datum     Frfattare           rende\n"));

    while(pos>=0)
    {
        if(map->texts[pos])
	{

	    if ((textstat = text_stat (map->texts[pos])) == NULL
		|| ((conf=conf_stat(textstat->author, &error)) == NULL
                || (s_empty (text = get_text (map->texts[pos], 0,
					      END_OF_STRING, &error))
		&& (error != KOM_NO_ERROR) )))
	    {
		xprintf (gettext("Det finns inget sdant inlgg.\n"));
		RETURN (FAILURE);
	    }
	    end_of_line_pos = s_strchr (text, '\n', 0);
	    xprintf("%-7d %04d%02d%02d  %-20S %S",
		    map->texts[pos],
		    textstat->creation_time.tm_year+1900,
		    textstat->creation_time.tm_mon+1,
		    textstat->creation_time.tm_mday,
		    conf->name,
		    s_fsubstr(text,0,(end_of_line_pos == -1
				      ? END_OF_STRING
				      : end_of_line_pos)));
	    zfree(textstat);
	    textstat=NULL;
	    zfree(conf);
	    conf=NULL;
	}
	pos--;
    }

    RETURN (OK);

#undef CLEAN_UP
#undef FUNCTION
}
