/**
 * \file insetgraphics.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Baruch Even
 * \author Herbert Voss
 *
 * Full author contact details are available in file CREDITS
 */

/*
TODO

    * What advanced features the users want to do?
      Implement them in a non latex dependent way, but a logical way.
      LyX should translate it to latex or any other fitting format.
    * Add a way to roll the image file into the file format.
    * When loading, if the image is not found in the expected place, try
      to find it in the clipart, or in the same directory with the image.
    * The image choosing dialog could show thumbnails of the image formats
      it knows of, thus selection based on the image instead of based on
      filename.
    * Add support for the 'picins' package.
    * Add support for the 'picinpar' package.
    * Improve support for 'subfigure' - Allow to set the various options
      that are possible.
*/

/* NOTES:
 * Fileformat:
 * Current version is 1 (inset file format version), when changing it
 * it should be changed in the Write() function when writing in one place
 * and when reading one should change the version check and the error message.
 * The filename is kept in  the lyx file in a relative way, so as to allow
 * moving the document file and its images with no problem.
 *
 *
 * Conversions:
 *   Postscript output means EPS figures.
 *
 *   PDF output is best done with PDF figures if it's a direct conversion
 *   or PNG figures otherwise.
 *	Image format
 *	from        to
 *	EPS         epstopdf
 *	PS          ps2pdf
 *	JPG/PNG     direct
 *	PDF         direct
 *	others      PNG
 */

#include <config.h>

#ifdef __GNUG__
#pragma implementation
#endif

#include "insets/insetgraphics.h"
#include "insets/insetgraphicsParams.h"

#include "graphics/GraphicsLoader.h"
#include "graphics/GraphicsImage.h"
#include "graphics/GraphicsParams.h"

#include "lyxtext.h"
#include "buffer.h"
#include "BufferView.h"
#include "converter.h"
#include "frontends/Painter.h"
#include "lyxrc.h"
#include "frontends/font_metrics.h"
#include "debug.h"
#include "gettext.h"
#include "LaTeXFeatures.h"
#include "lyxlex.h"

#include "frontends/Alert.h"
#include "frontends/Dialogs.h"
#include "frontends/LyXView.h"

#include "frontends/controllers/helper_funcs.h" // getVectorFromString

#include "support/LAssert.h"
#include "support/filetools.h"
#include "support/lyxalgo.h" // lyx::count
#include "support/lyxlib.h" // float_equal
#include "support/path.h"
#include "support/systemcall.h"
#include "support/os.h"

#include <boost/weak_ptr.hpp>
#include <boost/bind.hpp>
#include <boost/signals/trackable.hpp>
#include "BoostFormat.h"

#include <algorithm> // For the std::max

extern string system_tempdir;

using std::ostream;
using std::endl;


namespace {

///////////////////////////////////////////////////////////////////////////
int const VersionNumber = 1;
///////////////////////////////////////////////////////////////////////////

// This function is a utility function
// ... that should be with ChangeExtension ...
inline
string const RemoveExtension(string const & filename)
{
	return ChangeExtension(filename, string());
}


string const uniqueID()
{
	static unsigned int seed = 1000;

	ostringstream ost;
	ost << "graph" << ++seed;

	// Needed if we use lyxstring.
	return STRCONV(ost.str());
}


string findTargetFormat(string const & suffix)
{
	// lyxrc.pdf_mode means:
	// Are we creating a PDF or a PS file?
	// (Should actually mean, are we using latex or pdflatex).
	if (lyxrc.pdf_mode) {
		lyxerr[Debug::GRAPHICS] << "findTargetFormat: PDF mode\n";
		if (contains(suffix, "ps") || suffix == "pdf")
			return "pdf";
		else if (suffix == "jpg")	// pdflatex can use jpeg
			return suffix;
		else
			return "png";		// and also png
	}
	// If it's postscript, we always do eps.
	lyxerr[Debug::GRAPHICS] << "findTargetFormat: PostScript mode\n";
	if (suffix != "ps")			// any other than ps
	    return "eps";			// is changed to eps
	else
	    return suffix;			// let ps untouched
}

} // namespace anon


struct InsetGraphics::Cache : boost::signals::trackable
{
	///
	Cache(InsetGraphics &);
	///
	void update(string const & file_with_path);

	///
	int old_ascent;
	///
	grfx::Loader loader;
	///
	unsigned long checksum;
	///
	boost::weak_ptr<BufferView> view;

private:
	///
	InsetGraphics & parent_;
};


InsetGraphics::Cache::Cache(InsetGraphics & p)
	: old_ascent(0), checksum(0), parent_(p)
{
	loader.connect(boost::bind(&InsetGraphics::statusChanged, &parent_));
}


void InsetGraphics::Cache::update(string const & file_with_path)
{
	lyx::Assert(!file_with_path.empty());

	string const path = OnlyPath(file_with_path);
	loader.reset(file_with_path, parent_.params().as_grfxParams(path));
}


InsetGraphics::InsetGraphics()
	: graphic_label(uniqueID()),
	  cache_(new Cache(*this))
{}


InsetGraphics::InsetGraphics(InsetGraphics const & ig,
			     string const & filepath,
			     bool same_id)
	: Inset(ig, same_id),
	  graphic_label(uniqueID()),
	  cache_(new Cache(*this))
{
	setParams(ig.params(), filepath);
}


InsetGraphics::~InsetGraphics()
{
	// Emits the hide signal to the dialog connected (if any)
	hideDialog();
}


string const InsetGraphics::statusMessage() const
{
	string msg;

	switch (cache_->loader.status()) {
	case grfx::WaitingToLoad:
		msg = _("Not shown.");
		break;
	case grfx::Loading:
		msg = _("Loading...");
		break;
	case grfx::Converting:
		msg = _("Converting to loadable format...");
		break;
	case grfx::Loaded:
		msg = _("Loaded into memory. Must now generate pixmap.");
		break;
	case grfx::ScalingEtc:
		msg = _("Scaling etc...");
		break;
	case grfx::Ready:
		msg = _("Ready to display");
		break;
	case grfx::ErrorNoFile:
		msg = _("No file found!");
		break;
	case grfx::ErrorConverting:
		msg = _("Error converting to loadable format");
		break;
	case grfx::ErrorLoading:
		msg = _("Error loading file into memory");
		break;
	case grfx::ErrorGeneratingPixmap:
		msg = _("Error generating the pixmap");
		break;
	case grfx::ErrorUnknown:
		msg = _("No image");
		break;
	}

	return msg;
}


bool InsetGraphics::imageIsDrawable() const
{
	if (!cache_->loader.image() || cache_->loader.status() != grfx::Ready)
		return false;

	return cache_->loader.image()->isDrawable();
}


int InsetGraphics::ascent(BufferView *, LyXFont const &) const
{
	cache_->old_ascent = 50;
	if (imageIsDrawable())
		cache_->old_ascent = cache_->loader.image()->getHeight();
	return cache_->old_ascent;
}


int InsetGraphics::descent(BufferView *, LyXFont const &) const
{
	return 0;
}


int InsetGraphics::width(BufferView *, LyXFont const & font) const
{
	if (imageIsDrawable())
		return cache_->loader.image()->getWidth() + 2 * TEXT_TO_INSET_OFFSET;
	else {
		int font_width = 0;

		LyXFont msgFont(font);
		msgFont.setFamily(LyXFont::SANS_FAMILY);

		string const justname = OnlyFilename (params().filename);
		if (!justname.empty()) {
			msgFont.setSize(LyXFont::SIZE_FOOTNOTE);
			font_width = font_metrics::width(justname, msgFont);
		}

		string const msg = statusMessage();
		if (!msg.empty()) {
			msgFont.setSize(LyXFont::SIZE_TINY);
			int const msg_width = font_metrics::width(msg, msgFont);
			font_width = std::max(font_width, msg_width);
		}

		return std::max(50, font_width + 15);
	}
}


void InsetGraphics::draw(BufferView * bv, LyXFont const & font,
			 int baseline, float & x, bool) const
{
	// MakeAbsPath returns params().filename unchanged if it absolute
	// already.
	string const file_with_path =
		MakeAbsPath(params().filename, bv->buffer()->filePath());

	// A 'paste' operation creates a new inset with the correct filepath,
	// but then the 'old' inset stored in the 'copy' operation is actually
	// added to the buffer.
	// Thus, we should ensure that the filepath is correct.
	if (file_with_path != cache_->loader.filename())
		cache_->update(file_with_path);

	cache_->view = bv->owner()->view();
	int oasc = cache_->old_ascent;

	int ldescent = descent(bv, font);
	int lascent  = ascent(bv, font);
	int lwidth   = width(bv, font);

	// we may have changed while someone other was drawing us so better
	// to not draw anything as we surely call to redraw ourself soon.
	// This is not a nice thing to do and should be fixed properly somehow.
	// But I still don't know the best way to go. So let's do this like this
	// for now (Jug 20020311)
	if (lascent != oasc) {
		return;
	}

	// Make sure now that x is updated upon exit from this routine
	int old_x = int(x);
	x += lwidth;

	grfx::Params const & gparams = params().as_grfxParams();

	if (gparams.display != grfx::NoDisplay
		&& cache_->loader.status() == grfx::WaitingToLoad)
		cache_->loader.startLoading(*this, *bv);

	if (!cache_->loader.monitoring())
		cache_->loader.startMonitoring();

	// This will draw the graphics. If the graphics has not been loaded yet,
	// we draw just a rectangle.
	Painter & paint = bv->painter();

	if (imageIsDrawable()) {
		paint.image(old_x + TEXT_TO_INSET_OFFSET, baseline - lascent,
			    lwidth - 2 * TEXT_TO_INSET_OFFSET, lascent + ldescent,
			    *cache_->loader.image());

	} else {

		paint.rectangle(old_x + TEXT_TO_INSET_OFFSET, baseline - lascent,
				lwidth - 2 * TEXT_TO_INSET_OFFSET, lascent + ldescent);

		// Print the file name.
		LyXFont msgFont(font);
		msgFont.setFamily(LyXFont::SANS_FAMILY);
		string const justname = OnlyFilename (params().filename);
		if (!justname.empty()) {
			msgFont.setSize(LyXFont::SIZE_FOOTNOTE);
			paint.text(old_x + TEXT_TO_INSET_OFFSET + 6,
				   baseline - font_metrics::maxAscent(msgFont) - 4,
				   justname, msgFont);
		}

		// Print the message.
		string const msg = statusMessage();
		if (!msg.empty()) {
			msgFont.setSize(LyXFont::SIZE_TINY);
			paint.text(old_x + TEXT_TO_INSET_OFFSET + 6, baseline - 4, msg, msgFont);
		}
	}

	// the status message may mean we changed size, so indicate
	// we need a row redraw
#if 0
	if (old_status_ != grfx::ErrorUnknown && old_status_ != cached_status_) {
		bv->getLyXText()->status(bv, LyXText::CHANGED_IN_DRAW);
	}
#endif

	// Reset the cache, ready for the next draw request
#if 0
	cached_status_ = grfx::ErrorUnknown;
	cached_image_.reset();
	cache_filled_ = false;
#endif
}


void InsetGraphics::edit(BufferView *bv, int, int, mouse_button::state)
{
	bv->owner()->getDialogs().showGraphics(this);
}


void InsetGraphics::edit(BufferView * bv, bool)
{
	edit(bv, 0, 0, mouse_button::none);
}


Inset::EDITABLE InsetGraphics::editable() const
{
	return IS_EDITABLE;
}


void InsetGraphics::write(Buffer const *, ostream & os) const
{
	os << "Graphics\n";
	params().Write(os);
}


void InsetGraphics::read(Buffer const * buf, LyXLex & lex)
{
	string const token = lex.getString();

	if (token == "Graphics")
		readInsetGraphics(lex);
	else
		lyxerr[Debug::GRAPHICS] << "Not a Graphics inset!\n";

	cache_->update(MakeAbsPath(params().filename, buf->filePath()));
}


void InsetGraphics::readInsetGraphics(LyXLex & lex)
{
	bool finished = false;

	while (lex.isOK() && !finished) {
		lex.next();

		string const token = lex.getString();
		lyxerr[Debug::GRAPHICS] << "Token: '" << token << '\''
				    << std::endl;

		if (token.empty()) {
			continue;
		} else if (token == "\\end_inset") {
			finished = true;
		} else if (token == "FormatVersion") {
			lex.next();
			int version = lex.getInteger();
			if (version > VersionNumber)
				lyxerr
				<< "This document was created with a newer Graphics widget"
				", You should use a newer version of LyX to read this"
				" file."
				<< std::endl;
			// TODO: Possibly open up a dialog?
		}
		else {
			if (! params_.Read(lex, token))
				lyxerr << "Unknown token, " << token << ", skipping."
					<< std::endl;
		}
	}
}


string const InsetGraphics::createLatexOptions() const
{
	// Calculate the options part of the command, we must do it to a string
	// stream since we might have a trailing comma that we would like to remove
	// before writing it to the output stream.
	ostringstream options;
	if (!params().bb.empty())
	    options << "  bb=" << rtrim(params().bb) << ",\n";
	if (params().draft)
	    options << "  draft,\n";
	if (params().clip)
	    options << "  clip,\n";
	if (!lyx::float_equal(params().scale, 0.0, 0.05)) {
		if (!lyx::float_equal(params().scale, 100.0, 0.05))
			options << "  scale=" << params().scale / 100.0
				<< ",\n";
	} else {
		if (!params().width.zero())
			options << "  width=" << params().width.asLatexString() << ",\n";
		if (!params().height.zero())
			options << "  height=" << params().height.asLatexString() << ",\n";
		if (params().keepAspectRatio)
			options << "  keepaspectratio,\n";
	}

	// Make sure rotation angle is not very close to zero;
	// a float can be effectively zero but not exactly zero.
	if (!lyx::float_equal(params().rotateAngle, 0, 0.001)) {
	    options << "  angle=" << params().rotateAngle << ",\n";
	    if (!params().rotateOrigin.empty()) {
		options << "  origin=" << params().rotateOrigin[0];
		if (contains(params().rotateOrigin,"Top"))
		    options << 't';
		else if (contains(params().rotateOrigin,"Bottom"))
		    options << 'b';
		else if (contains(params().rotateOrigin,"Baseline"))
		    options << 'B';
		options << ",\n";
	    }
	}

	if (!params().special.empty())
	    options << params().special << ",\n";

	string opts = STRCONV(options.str());
	// delete last ",\n"
	return opts.substr(0, opts.size() - 2);
}


string const InsetGraphics::prepareFile(Buffer const * buf) const
{
	// LaTeX can cope if the graphics file doesn't exist, so just return the
	// filename.
	string const orig_file = params().filename;
	string orig_file_with_path =
		MakeAbsPath(orig_file, buf->filePath());
	lyxerr[Debug::GRAPHICS] << "[InsetGraphics::prepareFile] orig_file = "
		    << orig_file << "\n\twith path: "
		    << orig_file_with_path << endl;

	if (!IsFileReadable(orig_file_with_path))
		return orig_file;

	bool const zipped = zippedFile(orig_file_with_path);

	// If the file is compressed and we have specified that it
	// should not be uncompressed, then just return its name and
	// let LaTeX do the rest!
	if (zipped && params().noUnzip) {
		lyxerr[Debug::GRAPHICS]
			<< "\tpass zipped file to LaTeX but with full path.\n";
		// LaTeX needs an absolue path, otherwise the
		// coresponding *.eps.bb file isn't found
		return orig_file_with_path;
	}

	// Ascertain whether the file has changed.
	unsigned long const new_checksum = cache_->loader.checksum();
	bool const file_has_changed = cache_->checksum != new_checksum;
	if (file_has_changed)
		cache_->checksum = new_checksum;

	// temp_file will contain the file for LaTeX to act on if, for example,
	// we move it to a temp dir or uncompress it.
	string temp_file(orig_file);

	if (zipped) {
		// Uncompress the file if necessary.
		// If it has been uncompressed in a previous call to
		// prepareFile, do nothing.
		temp_file = MakeAbsPath(OnlyFilename(temp_file), buf->tmppath);
		lyxerr[Debug::GRAPHICS]
			<< "\ttemp_file: " << temp_file << endl;
		if (file_has_changed || !IsFileReadable(temp_file)) {
			bool const success = lyx::copy(orig_file_with_path,
						       temp_file);
			lyxerr[Debug::GRAPHICS]
				<< "\tCopying zipped file from "
				<< orig_file_with_path << " to " << temp_file
				<< (success ? " succeeded\n" : " failed\n");
		} else
			lyxerr[Debug::GRAPHICS]
				<< "\tzipped file " << temp_file
				<< " exists! Maybe no tempdir ...\n";
		orig_file_with_path = unzipFile(temp_file);
		lyxerr[Debug::GRAPHICS]
			<< "\tunzipped to " << orig_file_with_path << endl;
	}

	string const from = getExtFromContents(orig_file_with_path);
	string const to   = findTargetFormat(from);
	lyxerr[Debug::GRAPHICS]
		<< "\t we have: from " << from << " to " << to << '\n';

	if (from == to && !lyxrc.use_tempdir) {
		// No conversion is needed. LaTeX can handle the
		// graphic file as is.
		// This is true even if the orig_file is compressed.
		if (formats.getFormat(to)->extension() == GetExtension(orig_file)) {
			return RemoveExtension(orig_file_with_path);
		} else {
			return orig_file_with_path;
		}
	} 

	// We're going to be running the exported buffer through the LaTeX
	// compiler, so must ensure that LaTeX can cope with the graphics
	// file format.

	// Perform all these manipulations on a temporary file if possible.
	// If we are not using a temp dir, then temp_file contains the
	// original file.
	// to allow files with the same name in different dirs
	// we manipulate the original file "any.dir/file.ext"
	// to "any_dir_file.ext"! changing the dots in the
	// dirname is important for the use of ChangeExtension
	lyxerr[Debug::GRAPHICS]
		<< "\tthe orig file is: " << orig_file_with_path << endl;

	if (lyxrc.use_tempdir) {
		string const ext_tmp = GetExtension(orig_file_with_path);
		// without ext and /
		temp_file = subst(
			ChangeExtension(orig_file_with_path, string()), "/", "_");
		// without dots and again with ext
		temp_file = ChangeExtension(
			subst(temp_file, ".", "_"), ext_tmp);
		// now we have any_dir_file.ext
		temp_file = MakeAbsPath(temp_file, buf->tmppath);
		lyxerr[Debug::GRAPHICS]
			<< "\tchanged to: " << temp_file << endl;

		// if the file doen't exists, copy it into the tempdir
		if (file_has_changed || !IsFileReadable(temp_file)) {
			bool const success = lyx::copy(orig_file_with_path, temp_file);
			lyxerr[Debug::GRAPHICS]
				<< "\tcopying from " << orig_file_with_path << " to "
				<< temp_file
				<< (success ? " succeeded\n" : " failed\n");
			if (!success) {
				Alert::alert(_("Cannot copy file"), orig_file_with_path,
					_("into tempdir"));
				return orig_file;
			}
		}

		if (from == to) {
			// No conversion is needed. LaTeX can handle the
			// graphic file as is.
			if (formats.getFormat(to)->extension() == GetExtension(orig_file)) 
				return RemoveExtension(temp_file);
			else 
				return temp_file;
		}
	}
	
	string const outfile_base = RemoveExtension(temp_file);
	lyxerr[Debug::GRAPHICS]
		<< "\tThe original file is " << orig_file << "\n"
		<< "\tA copy has been made and convert is to be called with:\n"
		<< "\tfile to convert = " << temp_file << '\n'
		<< "\toutfile_base = " << outfile_base << '\n'
		<< "\t from " << from << " to " << to << '\n';

	// if no special converter defined, than we take the default one
	// from ImageMagic: convert from:inname.from to:outname.to
	if (!converters.convert(buf, temp_file, outfile_base, from, to)) {
		string const command =
			"sh " + LibFileSearch("scripts", "convertDefault.sh") +
				' ' + from + ':' + temp_file + ' ' +
				to + ':' + outfile_base + '.' + to;
		lyxerr[Debug::GRAPHICS]
			<< "No converter defined! I use convertDefault.sh:\n\t"
			<< command << endl;
		Systemcall one;
		one.startscript(Systemcall::Wait, command);
		if (!IsFileReadable(ChangeExtension(outfile_base, to)))
#if USE_BOOST_FORMAT
			Alert::alert(_("Cannot convert Image (not existing file?)"),
				     boost::io::str(boost::format(_("No information for converting from %1$s to %2$s"))
				% from % to));
#else
			Alert::alert(_("Cannot convert Image (not existing file?)"),
				     _("No information for converting from ") + from + " to " + to);
#endif
	}

	return RemoveExtension(temp_file);
}


int InsetGraphics::latex(Buffer const *buf, ostream & os,
			 bool /*fragile*/, bool/*fs*/) const
{
	// If there is no file specified or not existing,
	// just output a message about it in the latex output.
	lyxerr[Debug::GRAPHICS]
		<< "insetgraphics::latex: Filename = "
		<< params().filename << endl;

	// A missing (e)ps-extension is no problem for LaTeX, so
	// we have to test three different cases
	string const file_(MakeAbsPath(params().filename, buf->filePath()));
	bool const file_exists =
		!file_.empty() &&
		(IsFileReadable(file_) ||		// original
		 IsFileReadable(file_ + ".eps") ||	// original.eps
		 IsFileReadable(file_ + ".ps"));	// original.ps
	string const message = file_exists ?
		string() : string("bb = 0 0 200 100, draft, type=eps");
	// if !message.empty() than there was no existing file
	// "filename(.(e)ps)" found. In this case LaTeX
	// draws only a rectangle with the above bb and the
	// not found filename in it.
	lyxerr[Debug::GRAPHICS]
		<< "\tMessage = \"" << message << '\"' << endl;

	// These variables collect all the latex code that should be before and
	// after the actual includegraphics command.
	string before;
	string after;
	// Do we want subcaptions?
	if (params().subcaption) {
		before += "\\subfigure[" + params().subcaptionText + "]{";
		after = '}';
	}
	// We never use the starred form, we use the "clip" option instead.
	before += "\\includegraphics";

	// Write the options if there are any.
	string const opts = createLatexOptions();
	lyxerr[Debug::GRAPHICS] << "\tOpts = " << opts << endl;

	if (!opts.empty() && !message.empty())
		before += ("[%\n" + opts + ',' + message + ']');
	else if (!opts.empty() || !message.empty())
		before += ("[%\n" + opts + message + ']');

	lyxerr[Debug::GRAPHICS]
		<< "\tBefore = " << before
		<< "\n\tafter = " << after << endl;


	// "nice" means that the buffer is exported to LaTeX format but not
	//        run through the LaTeX compiler.
	if (buf->niceFile) {
		os << before <<'{' << params().filename << '}' << after;
		return 1;
	}

	// Make the filename relative to the lyx file
	// and remove the extension so the LaTeX will use whatever is
	// appropriate (when there are several versions in different formats)
	string const latex_str = message.empty() ?
		(before + '{' + os::external_path(prepareFile(buf)) + '}' + after) :
		(before + '{' + params().filename + " not found!}" + after);
	os << latex_str;

	// Return how many newlines we issued.
	int const newlines =
		int(lyx::count(latex_str.begin(), latex_str.end(),'\n') + 1);

	return newlines;
}


int InsetGraphics::ascii(Buffer const *, ostream & os, int) const
{
	// No graphics in ascii output. Possible to use gifscii to convert
	// images to ascii approximation.
	// 1. Convert file to ascii using gifscii
	// 2. Read ascii output file and add it to the output stream.
	// at least we send the filename
#if USE_BOOST_FORMAT
	os << '<'
	   << boost::format(_("Graphics file: %1$s")) % params().filename
	   << ">\n";
#else
	os << '<'
	   << _("Graphics file: ") << params().filename
	   << ">\n";
#endif
	return 0;
}


int InsetGraphics::linuxdoc(Buffer const *, ostream &) const
{
	// No graphics in LinuxDoc output. Should check how/what to add.
	return 0;
}


// For explanation on inserting graphics into DocBook checkout:
// http://linuxdoc.org/LDP/LDP-Author-Guide/inserting-pictures.html
// See also the docbook guide at http://www.docbook.org/
int InsetGraphics::docbook(Buffer const *, ostream & os,
			   bool /*mixcont*/) const
{
	// In DocBook v5.0, the graphic tag will be eliminated from DocBook, will
	// need to switch to MediaObject. However, for now this is sufficient and
	// easier to use.
	os << "<graphic fileref=\"&" << graphic_label << ";\">";
	return 0;
}


void InsetGraphics::validate(LaTeXFeatures & features) const
{
	// If we have no image, we should not require anything.
	if (params().filename.empty())
		return ;

	features.includeFile(graphic_label, RemoveExtension(params().filename));

	features.require("graphicx");

	if (params().subcaption)
		features.require("subfigure");
}


void InsetGraphics::statusChanged()
{
	if (cache_->view.get())
		cache_->view.get()->updateInset(this, false);
}


bool InsetGraphics::setParams(InsetGraphicsParams const & p,
			      string const & filepath)
{
	// If nothing is changed, just return and say so.
	if (params() == p && !p.filename.empty()) {
		return false;
	}

	// Copy the new parameters.
	params_ = p;

	// Update the inset with the new parameters.
	cache_->update(MakeAbsPath(params().filename, filepath));

	// We have changed data, report it.
	return true;
}


InsetGraphicsParams const & InsetGraphics::params() const
{
	return params_;
}


Inset * InsetGraphics::clone(Buffer const & buffer, bool same_id) const
{
	return new InsetGraphics(*this, buffer.filePath(), same_id);
}
