#include "mod_xmlrpc_rb.h"
#include "mod_xmlrpc_server.h"
#include "mod_xmlrpc.h"
#include "http_log.h"
#include "apr.h"
#include "apr_strings.h"
#include "ruby.h"
#include "st.h"
#include <stdarg.h>
#include <ctype.h>
#include <assert.h>
#include "xmlrpc.h"

extern module AP_MODULE_DECLARE_DATA xmlrpc_module;

static void print_exception(void)
{
	VALUE res;
	int i;

	res = rb_funcall(ruby_errinfo, rb_intern("message"), 0);
	DPRINTF(DEBUG_ERR, "Exception raised (%s): %s\n",
			rb_obj_classname(ruby_errinfo), RSTRING(res)->ptr);
	res = rb_funcall(ruby_errinfo, rb_intern("backtrace"), 0);
	for (i = 0; i < RARRAY(res)->len; i++)
		DPRINTF(DEBUG_ERR, "\t%s\n", RSTRING(RARRAY(res)->ptr[i])->ptr);
}

static char *exception_info(void)
{
	VALUE res;
	static char err[512];

	res = rb_funcall(ruby_errinfo, rb_intern("message"), 0);
	apr_snprintf(err, sizeof(err), "Exception raised (%s): %s",
			rb_obj_classname(ruby_errinfo), RSTRING(res)->ptr);
	return err;
}

static VALUE find_const(VALUE parent, const char *name)
{
	VALUE rb_obj = Qnil;
	ID id_obj;

	switch (TYPE(parent)) {
		case T_OBJECT:
		case T_CLASS:
		case T_ICLASS:
		case T_MODULE:
			/* look for object/module symbols */
			id_obj = rb_intern(name);
			if (rb_const_defined(parent, id_obj))
				rb_obj = rb_const_get(parent, id_obj);
			break;

		/* refuse to handle any other types */
	}

	return rb_obj;
}

static VALUE do_funcall(VALUE args)
{
	VALUE obj = rb_ary_shift(args);
	ID method = SYM2ID(rb_ary_shift(args));
	long i, arg_nr = RARRAY(args)->len;
	VALUE vargs[arg_nr];
	VALUE result;

	for (i = 0; i < arg_nr; i++)
		vargs[i] = RARRAY(args)->ptr[i];
	result = rb_funcall2(obj, method, arg_nr, vargs);

	return result;
}

static VALUE protected_funcall2(VALUE obj, ID method, int *err, int argc,
		VALUE *args)
{
	VALUE tmp;
	int i;
	
	tmp = rb_ary_new2(argc + 2);
	rb_ary_store(tmp, 0, obj);
	rb_ary_store(tmp, 1, ID2SYM(method));

	for (i = 0; i < argc; i++)
		rb_ary_store(tmp, i+2, args[i]);

	tmp = rb_protect(do_funcall, tmp, err);
	if (*err)
		print_exception();

	return tmp;
}

static VALUE protected_funcall(VALUE obj, ID method, int *err, int argc, ...)
{
	va_list args;
	VALUE tmp[argc];
	int i;
	
	va_start(args, argc);
	for (i = 0; i < argc; i++)
		tmp[i] = va_arg(args, VALUE);
	va_end(args);

	return protected_funcall2(obj, method, err, argc, tmp);
}

struct _xmlrpc_info
{
	xmlrpc_env *env;
	xmlrpc_value *result;
};

static xmlrpc_value *do_serialize(xmlrpc_env *env, VALUE v);
static int hash_each(VALUE key, VALUE val, struct _xmlrpc_info *inf)
{
	xmlrpc_value *x_key = do_serialize(inf->env, key);
	char *c_key;

	if (xmlrpc_value_type(x_key) != XMLRPC_TYPE_STRING) {
		DPRINTF(DEBUG_ERR, "Error: hash key wasn't a string!\n");
		return ST_STOP;
	}
	if (inf->env->fault_occurred)
		return ST_STOP;

	xmlrpc_parse_value(inf->env, x_key, "s", &c_key);
	xmlrpc_struct_set_value(inf->env, inf->result, c_key,
			do_serialize(inf->env, val));
	return ST_CONTINUE;
}

static xmlrpc_value *do_serialize(xmlrpc_env *env, VALUE v)
{
	xmlrpc_value *result;
	int i;

	if (env->fault_occurred)
		return NULL;

	switch (TYPE(v)) {
		case T_FLOAT:
			result = xmlrpc_build_value(env, "d", NUM2DBL(v));
			break;

		case T_FIXNUM:
			result = xmlrpc_build_value(env, "i", FIX2LONG(v));
			break;

		case T_TRUE:
			result = xmlrpc_build_value(env, "b", 1);
			break;

		case T_FALSE:
			result = xmlrpc_build_value(env, "b", 0);
			break;

		case T_STRUCT:
			/* TODO (check marshal.c for an example) */
			/*
			result = xmlrpc_build_value(env, "{}");
			for (i = 0; i < RSTRUCT(v)->len; i++)
				xmlrpc_struct_set_value(env, result, key,
					do_serialize(env, RSTRUCT(v)->ptr[i]));
			*/
			assert(0);
			break;
			
		case T_HASH:
			{
				struct _xmlrpc_info inf;
				
				result = xmlrpc_build_value(env, "{}");
				inf.env = env;
				inf.result = result;
				st_foreach(RHASH(v)->tbl, hash_each,
						(st_data_t) &inf);
			}
			break;

		case T_ARRAY:
			result = xmlrpc_build_value(env, "()");
			for (i = 0; i < RARRAY(v)->len; i++)
				xmlrpc_array_append_item(env, result,
					do_serialize(env, RARRAY(v)->ptr[i]));
			break;

		case T_STRING:
			result = xmlrpc_build_value(env, "s", RSTRING(v)->ptr);
			break;

		case T_NIL:
			// We don't support nil just yet; see
			// http://www.ontosys.com/xml-rpc/extensions.html
			// Use an empty string instead.
			result = xmlrpc_build_value(env, "s", "");
			break;

		default:
			DPRINTF(DEBUG_ERR, "Error: unknown type %d\n", TYPE(v));
			assert(0);
	}

	return result;
}

static VALUE do_unserialize(xmlrpc_env *env, xmlrpc_value *param)
{
	VALUE result;
	int i;
	long l;
	double d;
	char *s;
	xmlrpc_value *key, *val;

	if (env->fault_occurred)
		 return Qnil;

	switch (xmlrpc_value_type(param)) {
		case XMLRPC_TYPE_INT:
			DPRINTF(DEBUG_TRACE, "param type XMLRPC_TYPE_INT\n");
			xmlrpc_parse_value(env, param, "i", &l);
			result = INT2NUM(l);
			break;
			
		case XMLRPC_TYPE_BOOL:
			DPRINTF(DEBUG_TRACE, "param type XMLRPC_TYPE_BOOL\n");
			xmlrpc_parse_value(env, param, "b", &i);
			result = i ? Qtrue : Qfalse;
			break;

		case XMLRPC_TYPE_DOUBLE:
			DPRINTF(DEBUG_TRACE, "param type XMLRPC_TYPE_DOUBLE\n");
			xmlrpc_parse_value(env, param, "d", &d);
			result = rb_float_new(d);
			break;

		case XMLRPC_TYPE_STRING:
			DPRINTF(DEBUG_TRACE, "param type XMLRPC_TYPE_STRING\n");
			xmlrpc_parse_value(env, param, "s", &s);
			result = rb_str_new2(s);
			break;

		case XMLRPC_TYPE_ARRAY:
			DPRINTF(DEBUG_TRACE, "param type XMLRPC_TYPE_ARRAY\n");
			result = rb_ary_new();
			for (i = 0; i < xmlrpc_array_size(env, param); i++) {
				rb_ary_push(result, do_unserialize(env,
							xmlrpc_array_get_item(env, param, i)));
			}
			break;
			
		case XMLRPC_TYPE_STRUCT:
			DPRINTF(DEBUG_TRACE, "param type XMLRPC_TYPE_STRUCT\n");
			result = rb_hash_new();
			for (i = 0; i < xmlrpc_struct_size(env, param); i++) {
				xmlrpc_struct_get_key_and_value(env, param, i,
						&key, &val);
				rb_hash_aset(result, do_unserialize(env, key),
						do_unserialize(env, val));
			}
			break;

		case XMLRPC_TYPE_C_PTR:
		case XMLRPC_TYPE_DATETIME:
			/* TODO */
			DPRINTF(DEBUG_ERR, "param type TODO (%d)\n",
					xmlrpc_value_type(param));
			assert(0);

		default:
			DPRINTF(DEBUG_ERR, "Error: unknown type %d\n",
					xmlrpc_value_type(param));
			assert(0);
	}

	return result;
}

static xmlrpc_value *do_callback(xmlrpc_env *env, xmlrpc_value *param, void **n)
{
	xmlrpc_value *result;
	size_t param_nr = xmlrpc_array_size(env, param);
	VALUE args[param_nr], res, arity, meth;
	int i, err;

	/* sanity check; make sure args match */
	meth = protected_funcall((VALUE) n[0], rb_intern("method"), &err, 1,
			rb_str_new2(rb_id2name((ID) n[1])));
	arity = protected_funcall(meth, rb_intern("arity"), &err, 0);
	i = FIX2LONG(arity);
	if ((i >= 0 && i != param_nr) ||
			(i < 0 && param_nr < abs(i+1))) {
		xmlrpc_env_set_fault(env, 1, "Incorrect number of arguments!");
		return NULL;
	}

	for (i = 0; i < param_nr; i++)
		args[i] = do_unserialize(env, xmlrpc_array_get_item(env, param, i));
	if (env->fault_occurred)
		return NULL;

	res = protected_funcall2((VALUE) n[0], (ID) n[1], &err, param_nr, args);
	if (err) {
		xmlrpc_env_set_fault(env, 2, exception_info());
		return NULL;
	}
	result = do_serialize(env, res);

	if (env->fault_occurred)
		return NULL;
	DPRINTF(DEBUG_TRACE, "serialized the result\n");

	return result;
}

static void concat_and_up(char *buf, const char *one, const char *two)
{
	strcpy(buf, one);
	strcat(buf, two);
	while (*buf) {
		*buf = toupper(*buf);
		buf++;
	}
}

static VALUE each_method(VALUE *method, void **args)
{
	size_t len = RSTRING(method)->len + 1;
	char sig[len + sizeof("_SIG")];
	char help[len + sizeof("_HELP")];
	char meth[len + sizeof("ModXMLRPC::") + strlen((char *) args[0])];
	char name[len + sizeof(".") + strlen((char *) args[0])];
	VALUE rb_sig, rb_help;
	ID id_method;
	char *real_sig = "?", *real_help = "";
	void **extra_arg;

	cmd_parms *parms = (cmd_parms *) args[3];
	concat_and_up(help, RSTRING(method)->ptr, "_HELP");
	concat_and_up(sig, RSTRING(method)->ptr, "_SIG");
	strcpy(meth, RSTRING(method)->ptr);
	sprintf(name, "%s.%s", (char *) args[0], meth);

	rb_sig = find_const((VALUE) args[1], sig);
	if (TYPE(rb_sig) == T_STRING)
		real_sig = RSTRING(rb_sig)->ptr;
	else
		DPRINTF(DEBUG_WARN, "couldn't find %s\n", sig);

	rb_help = find_const((VALUE) args[1], help);
	if (TYPE(rb_help) == T_STRING)
		real_help = RSTRING(rb_help)->ptr;
	else
		DPRINTF(DEBUG_WARN, "couldn't find %s\n", help);

	id_method = rb_intern(meth);
	extra_arg = (void **) apr_pcalloc(parms->pool, 2 * sizeof(void *));
	extra_arg[0] = args[1]; /* k_type */
	extra_arg[1] = (void *) id_method;
	//printf("method: %s, help: %s, sig: %s\n", meth, real_help, real_sig);
	//rb_funcall((VALUE) args[1], id_method, 0);
	if (!mod_xmlrpc_server_register(args[2], name, &do_callback,
				(void *) extra_arg, real_sig, real_help)) {
		ap_log_error(APLOG_MARK, APLOG_WARNING, 0, parms->server,
				"Could not register callback: %s", name);
	}

	return Qnil;
}

static VALUE do_register(void *registry, cmd_parms *parms)
{
	VALUE rb_ModXMLRPC, klasses, k_type, methods;
	const char *klass;
	void *arg[4];
	int i;

	rb_ModXMLRPC = find_const(rb_cObject, "ModXMLRPC");
	if (rb_ModXMLRPC != Qnil) {
		klasses = rb_mod_constants(rb_ModXMLRPC);
		for (i = 0; i < RARRAY(klasses)->len; i++) {
			klass = RSTRING(RARRAY(klasses)->ptr[i])->ptr;

			/* Weed out the classes from non-classes */
			k_type = find_const(rb_ModXMLRPC, klass);
			if (TYPE(k_type) != T_CLASS)
				continue;

			arg[0] = (void *) klass;
			arg[1] = (void *) k_type;
			arg[2] = registry;
			arg[3] = (void *) parms;

			/* We only export singleton methods */
			methods = rb_obj_singleton_methods(0, NULL, k_type);
			rb_iterate(rb_each, methods, each_method, (VALUE) arg);
		}
	}

	return Qnil;
}

static void do_init(void)
{
	static int initted = 0;

	if (!initted) {
		ruby_init();
		ruby_init_loadpath();
		initted = 1;
	}
}

static VALUE do_require(const char *fname)
{
	VALUE kern, result = Qfalse;
	int err;
	
	kern = find_const(rb_cObject, "Kernel");
	if (kern != Qnil) {
		result = protected_funcall(kern, rb_intern("require"),
				&err, 1, rb_str_new2(fname));
		if (err)
			result = Qfalse;
	}

	return result;
}

const char *rb_set_xmlrpc_dir(cmd_parms *parms, void *ign, char *dir)
{
	void *registry;
	apr_status_t stat;
	apr_finfo_t info;
	apr_int32_t wanted;
	apr_dir_t *d;
	char fname[APR_PATH_MAX];

	do_init();
	registry = ap_get_module_config(parms->server->module_config, &xmlrpc_module);
	stat = apr_dir_open(&d, dir, parms->temp_pool);
	if (stat != APR_SUCCESS)
		return "Error opening directory!";

	wanted = APR_FINFO_NAME | APR_FINFO_TYPE;
	while ((stat = apr_dir_read(&info, wanted, d)) == APR_SUCCESS) {
		if ((info.filetype != APR_REG && info.filetype != APR_LNK) ||
				info.name[0] == '.' || info.name[0] == '\0')
			continue;

		apr_snprintf(fname, sizeof(fname), "%s/%s", dir, info.name);
		if (do_require(fname) != Qtrue)
			ap_log_error(APLOG_MARK, APLOG_WARNING, 0,
					parms->server, "Could not load file: %s", fname);
	}
	do_register(registry, parms);

	apr_dir_close(d);
	return NULL;
}

