/*
 * Copyright (C) 2005,2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <apt-front/cache/entity/tag.h>
#include <apt-front/cache/component/tags.h>

using namespace std;
using namespace aptFront;
using namespace cache;
using namespace entity;

static inline std::string constget(const map<string, string>& m, const std::string& key)
{
	map<string, string>::const_iterator i = m.find(key);
	if (i == m.end())
		return string();
	else
		return i->second;
}

std::string Facet::name() const
{
	if (valid())
		return m_tags->facetName(m_id);
	throw std::out_of_range( "No name for this facet" );
}
std::string Facet::name(const std::string& d) const
{
	return valid() ? m_tags->facetName(m_id) : d;
}

std::string Facet::shortDescription() const
{
	if (valid())
		return constget(m_tags->facetData(m_id), "_SD_");
	throw std::out_of_range( "No short description for this facet" );
}
std::string Facet::shortDescription(const std::string& d) const
{
	return valid() ? constget(m_tags->facetData(m_id), "_SD_") : d;
}

std::string Facet::longDescription() const
{
	if (valid())
		return constget(m_tags->facetData(m_id), "Description");
	throw std::out_of_range( "No long description for this facet" );
}
std::string Facet::longDescription(const std::string& d) const
{
	return valid() ? constget(m_tags->facetData(m_id), "Description") : d;
}

bool Facet::hasTag(const std::string& name) const
{
	if (!valid())
		throw std::out_of_range( "hasTag() called on an invalid facet" );
	return m_tags->hasTag(this->name() + "::" + name);
}

Tagcoll::OpSet<Tag> Facet::tags() const
{
	if (!valid())
		throw std::out_of_range( "tags() called on an invalid facet" );
	return m_tags->tags(m_id);
}


Facet Tag::facet() const
{
	if (valid())
		return m_tags->facetByTag(m_id);
	throw std::out_of_range( "No facet for this tag" );
}

std::string Tag::name() const
{
	if (valid())
		return m_tags->tagShortName(m_id);
	throw std::out_of_range( "No name for this tag" );
}
std::string Tag::name(const std::string& d) const
{
	return valid() ? m_tags->tagShortName(m_id) : d;
}

std::string Tag::fullname() const
{
	if (valid())
		return m_tags->tagName(m_id);
	throw std::out_of_range( "No full name for this facet" );
}
std::string Tag::fullname(const std::string& d) const
{
	return valid() ? m_tags->tagName(m_id) : d;
}

std::string Tag::shortDescription() const
{
	if (valid())
		return constget(m_tags->tagData(m_id), "_SD_");
	throw std::out_of_range( "No short description for this tag" );
}
std::string Tag::shortDescription(const std::string& d) const
{
	return valid() ? constget(m_tags->tagData(m_id), "_SD_") : d;
}

std::string Tag::longDescription() const
{
	if (valid())
		return constget(m_tags->tagData(m_id), "Description");
	throw std::out_of_range( "No long description for this tag" );
}
std::string Tag::longDescription(const std::string& d) const
{
	return valid() ? constget(m_tags->tagData(m_id), "Description") : d;
}


// hmmm bah this is pita, maybe we can come up with better solution?
// maybe raname COMPILE_TESTSUITE to APTFRONT_COMPILE_TESTSUITE?
/* Instantiate templates */
#ifdef COMPILE_TESTSUITE
#define SAVED_COMPILE_TESTSUITE
#undef COMPILE_TESTSUITE
#endif

#define INSTANTIATING_TEMPLATES

#include <tagcoll/OpSet.cc>

template class OpSet<entity::Facet>;
template class OpSet<entity::Tag>;

#ifdef SAVED_COMPILE_TESTSUITE
#define COMPILE_TESTSUITE
#endif


#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>

namespace tut {

struct cache_entity_tag_shar {
    cache_entity_tag_shar() {
        aptInit();
        c.open( Cache::OpenDefault | Cache::OpenTags | Cache::OpenReadOnly );
    }
    Cache c;
};
TESTGRP(cache_entity_tag);

template<> template<>
void to::test<1>()
{
    entity::Tag a, b;
    ensure( a == b );
    ensure( !a.valid() );
    ensure( !b.valid() );
}

template<> template<>
void to::test<2>()
{
    entity::Tag a;
    int x = 1;
    try {
        a.shortDescription();
        x = 2;
    } catch (...) {
        x = 3;
    }
    ensure_equals( x, 3 );
}

template<> template<>
void to::test< 3 >()
{
    entity::Facet f = c.tags().facetByName( "works-with" );
    entity::Tag t = c.tags().tagByName( "works-with::people" );
    ensure( t.valid() );
    ensure( f.valid() );
    ensure( t.facet() == f );
    ensure( f.tags().contains( t ) );
}

template<> template<>
void to::test< 4 >()
{
    entity::Facet f = c.tags().facetByName( "works-with" );
    entity::Tag t = c.tags().tagByName( "works-with::people" );
    ensure( t.valid() );
    ensure( f.valid() );
    ensure( f.hasTag( t.name() ) );
}

template<> template<>
void to::test< 5 >()
{
    entity::Tag t = c.tags().tagByName( "works-with::people" );
    ensure( t.valid() );
    ensure( t.facet().hasTag( t.name() ) );
    ensure( t.facet().tags().contains( t ) );
}

}

#endif

// vim:set ts=3 sw=3:
