/***************************************************************************
 $RCSfile: ctdriver_ctapi.c,v $
                             -------------------
    cvs         : $Id: ctdriver_ctapi.c,v 1.17 2003/04/21 03:01:38 aquamaniac Exp $
    begin       : Mon Dec 09 2002
    copyright   : (C) 2002 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_CHIPCARD_DLL
#  define CHIPCARD_API __declspec (dllexport)
# else /* Not BUILDING_CHIPCARD_DLL */
#  define CHIPCARD_API __declspec (dllimport)
# endif /* Not BUILDING_CHIPCARD_DLL */
#else
# define CHIPCARD_API
#endif


#include "ctdriver_ctapi.h"
#include "chameleon/libloader.h"
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <chameleon/debug.h>
#include <unistd.h> /* debug */


CTDRIVERCTAPI_DRIVERDATA *CTDriver_CTAPI_DriverData_new() {
  CTDRIVERCTAPI_DRIVERDATA *dd;

  DBG_ENTER;
  dd=(CTDRIVERCTAPI_DRIVERDATA *)malloc(sizeof(CTDRIVERCTAPI_DRIVERDATA));
  assert(dd);
  memset(dd,0,sizeof(CTDRIVERCTAPI_DRIVERDATA));
  DBG_LEAVE;
  return dd;
}


void CTDriver_CTAPI_DriverData_free(CTDRIVERCTAPI_DRIVERDATA *dd) {
  DBG_ENTER;
  if (dd) {
    LibLoader_free(dd->libHandle);
    free(dd);
  }
  DBG_LEAVE;
}


CTDRIVERCTAPI_READERDATA *CTDriver_CTAPI_ReaderData_new() {
  CTDRIVERCTAPI_READERDATA *rd;

  DBG_ENTER;
  rd=(CTDRIVERCTAPI_READERDATA *)malloc(sizeof(CTDRIVERCTAPI_READERDATA));
  assert(rd);
  memset(rd,0,sizeof(CTDRIVERCTAPI_READERDATA));
  DBG_LEAVE;
  return rd;
}


void CTDriver_CTAPI_ReaderData_free(CTDRIVERCTAPI_READERDATA *rd) {
  DBG_ENTER;
  if (rd)
    free(rd);
  DBG_LEAVE;
}




ERRORCODE CTDriver_CTAPI__SendAPDU(CTREADERTABLE *rt,
				   const unsigned char *apdu,
				   int apdulen,
				   unsigned char *buffer,
				   int *bufferlen){
  CTDRIVERCTAPI_DRIVERDATA *dd;
  CTDRIVERCTAPI_READERDATA *rd;
  char retval;
  unsigned char  dad;
  unsigned char  sad;
  unsigned short lr;

  DBG_ENTER;
  assert(rt);
  assert(rt->driver);
  assert(rt->driver->driverData);
  assert(rt->driverData);
  assert(apdu);
  assert(apdulen>3);
  assert(buffer);

  dd=(CTDRIVERCTAPI_DRIVERDATA *)rt->driver->driverData;
  rd=(CTDRIVERCTAPI_READERDATA *)rt->driverData;

  sad=CTDRIVER_CTAPI_SAD_HOST;
  /* fix for Cyberjack */
  if (apdu[0]==0x20 || (apdu[0]==0x80 && apdu[1]==0x60))
    dad=CTDRIVER_CTAPI_DAD_CT;
  else
    dad=0;

  DBG_DEBUG("CTN=%d, SAD=%d, DAD=%d, CLA=%02x, INS=%02x, P1=%02x, P2=%02x",
	    rd->ctn, sad, dad, apdu[0], apdu[1], apdu[2], apdu[3]);
  lr=*bufferlen;
  assert(dd->data);
  retval=dd->data(rd->ctn,
		  &dad,
		  &sad,
		  apdulen,
		  (char*)apdu,
		  &lr,
		  buffer);
  if (retval!=0) {
    DBG_ERROR("CTAPI error on \"CT_data\": %d",retval);
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_DRIVER_IO);
  }
  if (lr<2 || lr>258) {
    DBG_ERROR("Bad response size (%d)",lr);
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_DRIVER_IO);
  }

  if ((unsigned char)buffer[lr-2]!=0x90) {
    DBG_INFO("CTAPI: Error: SW1=%02x, SW2=%02x "
	     "(CLA=%02x, INS=%02x, P1=%02x, P2=%02x)",
	     (unsigned char)buffer[lr-2],
	     (unsigned char)buffer[lr-1],
	     apdu[0], apdu[1], apdu[2], apdu[3]);
  }

  *bufferlen=lr;
  return 0;
}


ERRORCODE CTDriver_CTAPI__Connect(CTREADERTABLE *rt) {
  CTDRIVERCTAPI_DRIVERDATA *dd;
  CTDRIVERCTAPI_READERDATA *rd;
  ERRORCODE err;
  unsigned char apdu[]={0x20, 0x12, 0x01, 0x01, 0x00};
  unsigned char responseBuffer[300];
  int lr;

  DBG_ENTER;
  assert(rt);
  assert(rt->driver);
  assert(rt->driver->driverData);
  assert(rt->driverData);

  dd=(CTDRIVERCTAPI_DRIVERDATA *)rt->driver->driverData;
  rd=(CTDRIVERCTAPI_READERDATA *)rt->driverData;

  lr=sizeof(responseBuffer);
  err=CTDriver_CTAPI__SendAPDU(rt,
			       apdu,
			       sizeof(apdu),
			       responseBuffer,
			       &lr);
  if (!Error_IsOk(err))
    return err;

  /* copy ATR */
  if (lr>2) {
    assert(lr<=sizeof(rt->atr));
    memmove(rt->atr, responseBuffer, lr-2);
    rt->atrlen=lr-2;
  }
  if (responseBuffer[lr-2]==0x90) {
    rd->isProcessorCard=responseBuffer[lr-1];
  }
  else {
    DBG_INFO("CTAPI: Soft error SW1=%02x, SW2=%02x",
	     (unsigned char)responseBuffer[lr-2],
	     (unsigned char)responseBuffer[lr-1]);
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_DRIVER_SOFT);
  }
  return 0;
};


ERRORCODE CTDriver_CTAPI__Status(CTREADERTABLE *rt, unsigned int *status) {
  CTDRIVERCTAPI_DRIVERDATA *dd;
  CTDRIVERCTAPI_READERDATA *rd;
  ERRORCODE err;
  unsigned char apdu[]={0x20, 0x13, 0x00, 0x80, 0x00};
  unsigned char responseBuffer[300];
  int lr;
  unsigned int lstatus;

  DBG_ENTER;
  assert(rt);
  assert(rt->driver);
  assert(rt->driver->driverData);
  assert(rt->driverData);

  dd=(CTDRIVERCTAPI_DRIVERDATA *)rt->driver->driverData;
  rd=(CTDRIVERCTAPI_READERDATA *)rt->driverData;

  lr=sizeof(responseBuffer);
  err=CTDriver_CTAPI__SendAPDU(rt,
			       apdu,
			       sizeof(apdu),
			       responseBuffer,
			       &lr);
  if (!Error_IsOk(err))
    return err;

  if (0>=(lr-2)) { /* FIXME: Use slot number */
    DBG_ERROR("CTAPI: Too few bytes in response (%d)",lr);
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_DRIVER_SOFT);
  }

  /* check whether a complete tag is returned */
  if (responseBuffer[0]!=0x80) {
    /* no, it's a simple response */
    lstatus=responseBuffer[0]; /* FIXME: use slot number */
  }
  else {
    /* otherwise a full TLV is returned, skip T and L */
    lstatus=responseBuffer[0+2]; /* FIXME: use slot number */
  }

  /* now translate the status */
  *status=0;
  if (lstatus & 0x01) /* inserted */
    *status|=CTREADERSTATUS_INSERTED;
  if (lstatus & 0x05) /* connected */
    *status|=CTREADERSTATUS_CONNECTED;
  if (rd->isProcessorCard)
    *status|=CTREADERSTATUS_PROCESSOR;

  /* Merlin would say: "That's it" (-> "Excalibur", I love this movie ;-) */
  return 0;
};


ERRORCODE CTDriver_CTAPI__Disconnect(CTREADERTABLE *rt) {
  CTDRIVERCTAPI_DRIVERDATA *dd;
  CTDRIVERCTAPI_READERDATA *rd;
  ERRORCODE err;
  unsigned char apdu[]={0x20, 0x15, 0x01, 0x00}; /* FIXME: P1=slotnumber+1 */
  unsigned char responseBuffer[300];
  int lr;

  DBG_ENTER;
  assert(rt);
  assert(rt->driver);
  assert(rt->driver->driverData);
  assert(rt->driverData);

  dd=(CTDRIVERCTAPI_DRIVERDATA *)rt->driver->driverData;
  rd=(CTDRIVERCTAPI_READERDATA *)rt->driverData;

  lr=sizeof(responseBuffer);
  err=CTDriver_CTAPI__SendAPDU(rt,
			       apdu,
			       sizeof(apdu),
			       responseBuffer,
			       &lr);
  if (!Error_IsOk(err))
    return err;

  if (responseBuffer[lr-2]==0x90) {
    rd->isProcessorCard=responseBuffer[lr-1];
  }
  else if (responseBuffer[lr-2]!=0x62) {
    DBG_NOTICE("CTAPI: Soft error SW1=%02x, SW2=%02x",
	       (unsigned char)responseBuffer[lr-2],
	       (unsigned char)responseBuffer[lr-1]);
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_DRIVER_SOFT);
  }
  return 0;
};


int CTDriver__CTAPI_GetPortNumber(CTREADERTABLE *rt){
  int pn;
  CTDRIVERCTAPI_DRIVERDATA *dd;

  assert(rt);
  assert(rt->descr);
  assert(rt->driver);
  assert(rt->driver->driverData);
  dd=(CTDRIVERCTAPI_DRIVERDATA*)(rt->driver->driverData);

  if (dd->ports) {
    /* ports section found, so lookup the port name there */
    pn=Config_GetIntValue(dd->ports, rt->descr->portName,-1,0);
    if (pn!=-1) {
      DBG_INFO("Found portname \"%s\"=%d in configuration.",
	       rt->descr->portName, pn);
      return pn;
    }
  }
  if (sscanf(rt->descr->portName,"%i",&pn)!=1) {
    DBG_ERROR("Invalid port \"%s\"",rt->descr->portName);
    return -1;
  }

  DBG_INFO("Portnumber given directly, returning %d",pn);
  return pn;
}


/**
 * Checks a bit field for a free (unset) bit
 */
int CTDriver__CTAPI_AllocBit(unsigned int *bf){
  int i;
  int j;
  unsigned int mask;

  DBG_ENTER;
  mask=0x01;
  j=0;
  for (i=0; i<16; i++) {
    if (!(*bf & mask)) {
      *bf|=mask;
      DBG_LEAVE;
      return j;
    }
    j++;
    mask=mask<<1;
  } /* for */

  /* no free id found */
  DBG_LEAVE;
  return -1;
}


/**
 * Checks a bit field for a free (unset) bit
 */
int CTDriver__CTAPI_ReleaseBit(unsigned int *bf, int b){
  int i;
  unsigned int mask;

  DBG_ENTER;
  if (b>15 || b<0) {
    DBG_ERROR("CTAPI: Bad bit number (%d)", b);
    DBG_LEAVE;
    return 1;
  }
  mask=0x01;
  for (i=0; i<b; i++)
    mask=mask<<1;

  if (*bf & mask)
    *bf&=~mask;
  else {
    DBG_NOTICE("CTAPI: Bit %d not allocated in %04x",b, *bf);
    DBG_LEAVE;
    return 1;
  }

  DBG_LEAVE;
  return 0;
}


ERRORCODE CTDriver_CTAPI_AllocTerm(CTREADERTABLE *rt){
  unsigned short ctn;
  unsigned short pn;
  char rv;
  CTDRIVERCTAPI_DRIVERDATA *dd;
  CTDRIVERCTAPI_READERDATA *rd;
  int i;

  DBG_ENTER;
  assert(rt);
  assert(rt->descr);
  assert(rt->driver);
  assert(rt->driver->driverData);
  dd=(CTDRIVERCTAPI_DRIVERDATA *)rt->driver->driverData;
  DBG_INFO("CTAPI allocating reader (%08x)",(unsigned int)dd);

  /* get a free CTN (first argument of CTAPI_init) */
  i=CTDriver__CTAPI_AllocBit(&(dd->terminalIds));
  if (i==-1) {
    DBG_NOTICE("CTAPI: All terminals in use");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_DRIVER_BUSY);
  }
  /* Kobil drivers for FreeBSD do not work with a value of 0 */
  ctn=(unsigned short)i;

  /* get the port number (2nd argument of CTAPI_init) */
  i=CTDriver__CTAPI_GetPortNumber(rt);
  if (i==-1) {
    DBG_NOTICE("CTAPI: Bad port number");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_INVALID);
  }
  pn=(unsigned short)i;

  /* now alloc private reader data */
  rd=CTDriver_CTAPI_ReaderData_new();
  assert(rd);
  rt->driverData=rd;
  rd->ctn=ctn;

  /* now call init for this reader */
  assert(dd->init);
  rv=dd->init(ctn, pn);
  if (rv!=0) {
    DBG_ERROR("CTAPI: Error on init (ctn=%d, pn=%d, err=%d)",
	      ctn, pn, rv);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_DRIVER_IO);
  }

  DBG_INFO("Allocated reader (ctn=%d, pn=%d)",
	   ctn, pn);
  DBG_LEAVE;
  return 0;
}


ERRORCODE CTDriver_CTAPI_ReleaseTerm(CTREADERTABLE *rt){
  int rv;
  CTDRIVERCTAPI_DRIVERDATA *dd;
  CTDRIVERCTAPI_READERDATA *rd;

  DBG_ENTER;
  assert(rt);
  assert(rt->driver);
  assert(rt->driver->driverData);
  assert(rt->driverData);
  dd=(CTDRIVERCTAPI_DRIVERDATA *)rt->driver->driverData;
  rd=(CTDRIVERCTAPI_READERDATA *)rt->driverData;

  DBG_INFO("Releasing reader (ctn=%d)",
	   rd->ctn);

  if (CTDriver__CTAPI_ReleaseBit(&(dd->terminalIds),
				 rd->ctn)) {
    DBG_WARN("CTAPI: Reader %d not allocated",rd->ctn);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }
  assert(dd->close);
  rv=dd->close(rd->ctn);
  rd->ctn=-1;
  CTDriver_CTAPI_ReaderData_free(rt->driverData);
  rt->driverData=0;
  if (rv!=0) {
    DBG_WARN("CTAPI: Error on close (%d)",rd->ctn);
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }
  DBG_LEAVE;
  return 0;
}


ERRORCODE CTDriver_CTAPI_ConnectTerm(CTREADERTABLE *rt,
				     char *atrbuffer,
				     int *atrbufferlen){
  CTDRIVERCTAPI_DRIVERDATA *dd;
  CTDRIVERCTAPI_READERDATA *rd;
  ERRORCODE err;
  unsigned int status;
  int s;

  DBG_ENTER;
  assert(rt);
  assert(rt->driver);
  assert(rt->driver->driverData);
  assert(rt->driverData);

  dd=(CTDRIVERCTAPI_DRIVERDATA *)rt->driver->driverData;
  rd=(CTDRIVERCTAPI_READERDATA *)rt->driverData;

  DBG_INFO("Connecting reader (ctn=%d)",
	   rd->ctn);

  /* already connected ? */
  if (rt->openCount<1) {
    /* not connected, so do it now */
    err=CTDriver_CTAPI__Connect(rt);
    if (!Error_IsOk(err)) {
      DBG_DEBUG_ERR(err);
      rt->openCount=0;
      return err;
    }
  }
  rt->openCount++;

  /* get status */
  err=CTDriver_CTAPI__Status(rt, &status);
  if (!Error_IsOk(err)) {
    DBG_DEBUG_ERR(err);
    rt->openCount=0;
    return err;
  }

  /* check if still connected */
  if (!(status & CTREADERSTATUS_CONNECTED) ||
      !(status & CTREADERSTATUS_INSERTED)) {
    /* not connected */
    DBG_NOTICE("CTAPI: Connection lost (status=%0x4)",status);
    rt->cardChanged++;
    rt->openCount=0;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_CARD_REMOVED);
  }
  rt->deltaStatus=(rt->lastStatus^status);
  rt->lastStatus=status;
  if (rt->deltaStatus) {
    DBG_NOTICE("Status changed");
  }

  /* copy ATR */
  s=(rt->atrlen>*atrbufferlen)?*atrbufferlen:rt->atrlen;
  if (s<rt->atrlen) {
    DBG_WARN("ATR buffer too short, copying partially");
  }
  if (s)
    memmove(atrbuffer, rt->atr, s);
  *atrbufferlen=s;

  /* finished */
  rt->cardChanged=0;
  return 0;
}


ERRORCODE CTDriver_CTAPI_DisconnectTerm(CTREADERTABLE *rt){
  CTDRIVERCTAPI_DRIVERDATA *dd;
  CTDRIVERCTAPI_READERDATA *rd;
  int oc;
  ERRORCODE err;

  DBG_ENTER;
  assert(rt);
  assert(rt->driver);
  assert(rt->driver->driverData);
  assert(rt->driverData);
  dd=(CTDRIVERCTAPI_DRIVERDATA *)rt->driver->driverData;
  rd=(CTDRIVERCTAPI_READERDATA *)rt->driverData;

  DBG_INFO("Disconnecting reader (ctn=%d)",
	   rd->ctn);

  oc=rt->openCount--;
  if (rt->openCount<0)
    rt->openCount=0;

  rt->openCount=0;
  if (oc>0) {
    err=CTDriver_CTAPI__Disconnect(rt);
    if (!Error_IsOk(err)) {
      DBG_DEBUG_ERR(err);
      DBG_LEAVE;
      return err;
    }
  }

  DBG_LEAVE;
  return 0;

}


ERRORCODE CTDriver_CTAPI_Status(CTREADERTABLE *rt,
				char *atrbuffer,
				int *atrbufferlen){
  CTDRIVERCTAPI_DRIVERDATA *dd;
  CTDRIVERCTAPI_READERDATA *rd;
  ERRORCODE err;
  unsigned int status;
  int s;

  DBG_ENTER;
  assert(rt);
  assert(rt->driver);
  assert(rt->driver->driverData);
  assert(rt->driverData);
  dd=(CTDRIVERCTAPI_DRIVERDATA *)rt->driver->driverData;
  rd=(CTDRIVERCTAPI_READERDATA *)rt->driverData;
  DBG_INFO("Statting reader (ctn=%d)",
	   rd->ctn);

  if (rt->openCount<1) {
    /* open driver, so that we get the ATR and some other info */
    err=CTDriver_CTAPI__Connect(rt);
    if (!Error_IsOk(err)) {
      DBG_INFO_ERR(err);
    }
    else {
      rt->openCount++;
    }
  }
  err=CTDriver_CTAPI__Status(rt, &status);
  if (!Error_IsOk(err)) {
    DBG_INFO_ERR(err);
    rt->openCount=0;
    rt->cardChanged++;
    DBG_LEAVE;
    return err;
  }

  rt->deltaStatus=(rt->lastStatus^status);
  if (rt->deltaStatus&CTREADERSTATUS_CONNECTED &&
      !(status&CTREADERSTATUS_CONNECTED) && rt->openCount>0) {
    DBG_DEBUG("Status changed to disconnected, closing");
    rt->openCount=0;
    rt->cardChanged++;
  }
  rt->lastStatus=status;
  if (rt->deltaStatus) {
    DBG_NOTICE("Status changed");
  }

  /* copy ATR */
  s=rt->atrlen;
  if (s>*atrbufferlen) {
    s=*atrbufferlen;
    DBG_WARN("ATR buffer too short, copying partially (%d of %d bytes)",
	     s,rt->atrlen);
  }
  if (s)
    memmove(atrbuffer, rt->atr, s);
  *atrbufferlen=s;

  return 0;
}


ERRORCODE CTDriver_CTAPI_Command(CTREADERTABLE *rt,
				 const char *sendBuffer,
				 int sendBufferLength,
				 char *recvBuffer,
				 int *recvBufferLength){
  CTDRIVERCTAPI_DRIVERDATA *dd;
  CTDRIVERCTAPI_READERDATA *rd;
  ERRORCODE err;

  DBG_ENTER;
  assert(rt);
  assert(rt->driver);
  assert(rt->driver->driverData);
  assert(rt->driverData);
  dd=(CTDRIVERCTAPI_DRIVERDATA *)rt->driver->driverData;
  rd=(CTDRIVERCTAPI_READERDATA *)rt->driverData;
  DBG_INFO("Commanding reader (ctn=%d)",
	   rd->ctn);

  if (rt->cardChanged!=0) {
    DBG_NOTICE("Card removed");
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_CARD_REMOVED);
  }

  /* execute command */
  err=CTDriver_CTAPI__SendAPDU(rt,
			       (const unsigned char*)sendBuffer,
			       sendBufferLength,
			       (unsigned char*)recvBuffer,
			       recvBufferLength);
  if (!Error_IsOk(err)) {
    DBG_DEBUG_ERR(err);
    DBG_LEAVE;
    return err;
  }

  return 0;
}


ERRORCODE CTDriver_CTAPI_Open(CTDRIVERTABLE *dt,
			      CTREADERDESCRIPTION *rd,
			      CONFIGGROUP *ddescr){
  CTDRIVERCTAPI_DRIVERDATA *dd;
  ERRORCODE err;
  CONFIGGROUP *g;

  assert(dt);
  assert(rd);
  DBG_INFO("Opening CTAPI driver \"%s\"",rd->driverName);
  dd=CTDriver_CTAPI_DriverData_new();
  dt->driverData=dd;
  dd->libHandle=LibLoader_new();

  /* get ports section */
  g=Config_GetGroup(ddescr,
		    rd->typeName,
		    CONFIGMODE_NAMEMUSTEXIST);
  if (g) {
    /* ok, we have a section for this readertype */
    g=Config_GetGroup(g,
		      "ports",
		      CONFIGMODE_NAMEMUSTEXIST);
    if (g) {
      dd->ports=g;
      DBG_NOTICE("Valid ports section for type \"%s\" found.",
		 rd->typeName);
    }
    else {
      DBG_WARN("No ports section, using defaults upon name resolution");
    }
  }
  else {
    DBG_WARN("No section for type \"%s\"",
	     rd->typeName);
  }

  /* load library */
  err=LibLoader_OpenLibrary(dd->libHandle, rd->driverName);
  if (!Error_IsOk(err)) {
    DBG_WARN_ERR(err);
    CTDriver_CTAPI_DriverData_free(dd);
    dt->driverData=0;
    return err;
  }

  /* resolve symbols */
  err=LibLoader_Resolve(dd->libHandle,
			"CT_init",
			(void*)&dd->init);
  if (!Error_IsOk(err))
    err=LibLoader_Resolve(dd->libHandle,
			  "_CT_init",
			  (void*)&dd->init);

  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    LibLoader_CloseLibrary(dd->libHandle);
    CTDriver_CTAPI_DriverData_free(dd);
    dt->driverData=0;
    return err;
  }

  err=LibLoader_Resolve(dd->libHandle,
			"CT_data",
			(void*)&dd->data);
  if (!Error_IsOk(err))
    err=LibLoader_Resolve(dd->libHandle,
			  "_CT_data",
			  (void*)&dd->data);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    LibLoader_CloseLibrary(dd->libHandle);
    CTDriver_CTAPI_DriverData_free(dd);
    dt->driverData=0;
    return err;
  }

  err=LibLoader_Resolve(dd->libHandle,
			"CT_close",
			(void*)&dd->close);
  if (!Error_IsOk(err))
    err=LibLoader_Resolve(dd->libHandle,
			  "_CT_close",
			  (void*)&dd->close);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    LibLoader_CloseLibrary(dd->libHandle);
    CTDriver_CTAPI_DriverData_free(dd);
    dt->driverData=0;
    return err;
  }

  /* set function pointers */
  dt->closeDriver=CTDriver_CTAPI_Close;
  dt->statTerm=CTDriver_CTAPI_Status;
  dt->allocTerm=CTDriver_CTAPI_AllocTerm;
  dt->releaseTerm=CTDriver_CTAPI_ReleaseTerm;
  dt->connectTerm=CTDriver_CTAPI_ConnectTerm;
  dt->disconnectTerm=CTDriver_CTAPI_DisconnectTerm;
  dt->commandTerm=CTDriver_CTAPI_Command;
  dt->enumTerms=0; /* no enumeration possible */

  DBG_INFO("CTAPI driver open (%08x)",(unsigned int)dd);
  return 0;
}


ERRORCODE CTDriver_CTAPI_Close(CTDRIVERTABLE *dt){
  ERRORCODE err;
  CTDRIVERCTAPI_DRIVERDATA *dd;

  assert(dt);
  assert(dt->driverData);
  dd=(CTDRIVERCTAPI_DRIVERDATA *)(dt->driverData);
  err=LibLoader_CloseLibrary(dd->libHandle);
  CTDriver_CTAPI_DriverData_free(dd);
  dt->driverData=0;

  dt->closeDriver=0;
  dt->statTerm=0;
  dt->allocTerm=0;
  dt->releaseTerm=0;
  dt->connectTerm=0;
  dt->disconnectTerm=0;
  dt->commandTerm=0;
  dt->enumTerms=0;

  if (!Error_IsOk(err))
    return err;

  return 0;
}



