/*
  CLAW - a C++ Library Absolutely Wonderful

  CLAW is a free library without any particular aim but being useful to 
  anyone.

  Copyright (C) 2005-2008 Julien Jorge

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: julien_jorge@yahoo.fr
*/
/**
 * \file game_ai.hpp
 * \brief Regroupement de fonctions d'intelligence artificielle.
 * \author Julien Jorge & Sbastien Angibaud
 */
#ifndef __CLAW_GAME_AI_HPP__
#define __CLAW_GAME_AI_HPP__

#include <list>

namespace claw
{
  namespace ai
  {
    namespace game
    {
      //**************************** game_state ********************************

      /**
       * \brief Reprsentation de l'tat du jeu.
       * \param Action Type des actions. Doit hriter de game_state.
       * \param Numeric Un type numrique pour l'valuation des score.
       * \author Julien Jorge
       */
      template <class Action, class Numeric = int> class game_state
      {
      public:
        /** \brief Le type de l'valuation. */
        typedef Numeric score;
        typedef Action action;

        virtual ~game_state();

        /** \brief Rcupre l'valuation de l'tat. */
        virtual score evaluate() const = 0;

        static score min_score();
        static score max_score();

        /** 
         * \brief Rcupre la listes des actions possibles. 
         * \param l (sortie) La liste des actions jouables.
         */
        virtual void nexts_actions( std::list<action>& l ) const = 0;

        /** 
         * \brief Excute une action. 
         * \param a Action a effectuer.
         * \return L'tat rsultant de cette action.
         */
        virtual game_state* do_action( const action& a ) const = 0;

        /** \brief Indique s'il s'agit de l'tat d'une partie termine. */
        virtual bool final() const = 0;

      protected :

        score fit( score score_val ) const;

      protected :
        /** \brief Score minimal d'un tat. */
        static const score s_min_score;
        /** \brief Score maximal d'un tat. */
        static const score s_max_score;
      }; // class game_state

      //**************************** action_eval ******************************

      /**
       * \brief Couple (action, score) pour associer une action  son score.
       */
      template <class Action, class Numeric> class action_eval
      {
      public:
        action_eval( const Action& a,
                     const Numeric& e);

        bool operator< ( const action_eval& ae ) const;
        bool operator==( const action_eval& ae ) const;
                
        /** \brief Action. */
        Action action;
        /** \brief Evaluation. */
        Numeric eval;
      }; // class action_eval

      //*************************** min_max ***********************************

      /**
       * \brief Slection d'une action avec la mthode MinMax.
       * \author Julien Jorge
       */
      template <class State> class min_max
      {
      public:
        typedef State state;
        typedef typename State::action action;
        typedef typename State::score score;

        score operator()( int depth, const state& current_state, 
                          bool computer_turn ) const;
      }; // class min_max

      //*************************** alpha_beta ********************************

      /**
       * \brief Slection d'une action avec la mthode Alpha Beta.
       * \author Julien Jorge & Sbastien Angibaud
       */
      template <class State> class alpha_beta
      {
      public:
        typedef State state;
        typedef typename State::action action;
        typedef typename State::score score;

        score operator()( int depth, const state& current_state, 
                          bool computer_turn ) const;
      private:
        score calcul( int depth, const state& current_state, 
                      bool computer_turn, score alpha, score beta ) const;
      }; // class alpha_beta

      //*************************** select_action *****************************

      /**
       * \brief Slection d'une action avec une mthode quelconque.
       * \author Julien Jorge & Sbastien Angibaud
       */
      template<class Method> class select_action
      {
      public:
        typedef typename Method::state state;
        typedef typename Method::action action;
        typedef typename Method::score score;

        void operator()( int depth, const state& current_state,
                         action& new_action, bool computer_turn ) const;
      }; // class select_action

      //************************ select_random_action *************************

      /**
       * \brief Slection d'une action parmis les meilleures avec une mthode
       *        quelconque.
       * \author Julien Jorge & Sbastien Angibaud
       */
      template<class Method>
      class select_random_action
      {
      public:
        typedef typename Method::state state;
        typedef typename Method::action action;
        typedef typename Method::score score;

        void operator()( int depth, const state& current_state,
                         action& new_action, bool computer_turn ) const;
      }; // class select_random_action

    } // namespace game
  } // namespace it
} // namespace claw

#include <claw/impl/game_ai.tpp>

#endif // __CLAW_IA_JEUX_HPP__
