/***************************************************************************
                          cdownloadmanager.cpp  -  description
                             -------------------
    begin                : Don Mai 16 2002
    copyright            : (C) 2002-2005 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <math.h>
#include <list>

#include <dclib/dcos.h>
#include <dclib/dcobject.h>
#include <dclib/cconfig.h>
#include <dclib/core/cdir.h>
#include <dclib/cconnectionmanager.h>
#include <dclib/core/cmd5.h>
#include <dclib/core/clisten.h>
#include <dclib/cdownloadqueue.h>
#include <dclib/core/clogfile.h>
#include <dclib/core/cmanager.h>
#include <dclib/core/filecopy.h>
#include <dclib/cfilemanager.h>
#include <dclib/clistenmanager.h>
#include <dclib/dclib.h>
#include <dclib/core/cxml.h>
#include <dclib/core/platform.h>
#include <dclib/csearchmanager.h>
#include <dclib/cutils.h>
#include <dclib/core/cbase64.h>

#ifndef WIN32
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#endif

#define CHUNK_SIZE	(1024*1024)
#define TEST_CHUNK_SIZE	(10*1024)

#include "cdownloadmanager.h"

/** */
CDownloadManager::CDownloadManager()
{
	m_nID            = 0;
	m_eShutdownState = essNONE;

	m_tDownloadQueueTimeout  = time(0);
	m_tUpdateTransferTimeout = time(0);
	m_tHubSearchTimeout	 = time(0);
	m_tSearchTimeout	 = 0;

	m_pTransferList      = new CThreadStringList();
	m_pTransferBanList   = new CThreadStringList();
	m_pExtraUserSlotList = new CThreadList<CExtraUserSlot>();
	m_pTransferWaitList  = new CThreadList<DCTransferWait>();
	m_pSearchList	     = new CThreadList<CMessageSearchResult>();
	m_pSearchQueryList   = new CList<CObject>();

	m_pDownloadQueue = new CDownloadQueue();

	m_pCallback = 0;

	SetInstance(this);
}

/** */
CDownloadManager::~CDownloadManager()
{
	CManager::Instance()->Remove( m_pCallback );
	if ( m_pCallback )
	{
		delete m_pCallback;
		m_pCallback = 0;
	}

	if (m_pExtraUserSlotList)
	{
		delete m_pExtraUserSlotList;
		m_pExtraUserSlotList = 0;
	}

	if(m_pTransferList)
	{
		delete m_pTransferList;
		m_pTransferList = 0;
	}

	if(m_pTransferWaitList)
	{
		delete m_pTransferWaitList;
		m_pTransferWaitList = 0;
	}

	if(m_pTransferBanList)
	{
		delete m_pTransferBanList;
		m_pTransferBanList = 0;
	}

	if (m_pSearchList)
	{
		delete m_pSearchList;
		m_pSearchList = 0;
	}

	if ( m_pSearchQueryList )
	{
		delete m_pSearchQueryList;
		m_pSearchQueryList = 0;
	}
	
	if ( m_pDownloadQueue )
	{
		delete m_pDownloadQueue;
		m_pDownloadQueue = 0;
	}
}

/** load the queue */
int CDownloadManager::DLM_LoadQueue()
{
	int err = -1;
	CStringList * StringList = 0;

	m_pDownloadQueue->pQueue->Lock();
	m_pDownloadQueue->pQueue->Clear();
	m_pDownloadQueue->pChunkList->Lock();
	m_pDownloadQueue->pChunkList->Clear();

	if ( CConfig::Instance() )
		err = CConfig::Instance()->LoadDCTra( m_pDownloadQueue->pQueue, m_pDownloadQueue->pChunkList );

	// send all files over the callback function
	if ( err == 0 )
	{
		while( m_pDownloadQueue->pQueue->Next( (CObject *&) StringList) )
		{
			DCTransferQueueObject * TransferObject = 0;

			while( StringList->Next( (CObject *&) TransferObject) )
			{
				DCTransferFileObject * TransferFileObject = 0;

				while( TransferObject->pTransferFileList.Next( (CObject *&)TransferFileObject) )
				{
					SendFileInfo( TransferObject, TransferFileObject );
				}
			}
		}
	}

	m_pCallback = new CCallback<CDownloadManager>( this, &CDownloadManager::Callback );
	
	if ( CManager::Instance() )
		CManager::Instance()->Add( m_pCallback );

	m_pDownloadQueue->pChunkList->UnLock();
	m_pDownloadQueue->pQueue->UnLock();

	return err;
}

/** save the queue  */
int CDownloadManager::DLM_SaveQueue()
{
	int err = -1;

	m_pDownloadQueue->pQueue->Lock();
	m_pDownloadQueue->pChunkList->Lock();

	if ( CConfig::Instance() )
		err = CConfig::Instance()->SaveDCTra( m_pDownloadQueue->pQueue, m_pDownloadQueue->pChunkList );

	m_pDownloadQueue->pChunkList->UnLock();
	m_pDownloadQueue->pQueue->UnLock();

	return err;
}

/** disconnect all transfers for a clean shutdown */
void CDownloadManager::DLM_Shutdown()
{
	CObject *obj;
	CTransferObject * TransferObject;

	// set shutdown state
	m_eShutdownState = essSHUTDOWN;

	SendLogInfo("Shutdown download manager ...\n");

	m_pTransferList->Lock();

	DPRINTF("Running Transfers: %ld\n",m_pTransferList->Count());

	obj = 0;
	while( m_pTransferList->Next( (CObject*&)obj ) )
	{
		TransferObject = (CTransferObject*)obj;
		TransferObject->m_pTransfer->Disconnect(TRUE);
	}

	m_pTransferList->UnLock();
}

/** add a extra slot to the user */
void CDownloadManager::DLM_AddUserSlot( CString nick, CString hubname, int slot, bool permanent )
{
	CExtraUserSlot * ExtraUserSlot;

	m_pExtraUserSlotList->Lock();

	ExtraUserSlot = 0;

	while( (ExtraUserSlot=m_pExtraUserSlotList->Next(ExtraUserSlot)) != 0 )
	{
		if ( (ExtraUserSlot->sNick == nick) && (ExtraUserSlot->sHubName == hubname) )
		{
			if ( (slot == 0) && (!permanent) )
			{
				ExtraUserSlot->iSlots     = slot;
				ExtraUserSlot->bPermanent = FALSE;
			}
			else
			{
				if (ExtraUserSlot->bPermanent)
					ExtraUserSlot->iSlots = 0;
				ExtraUserSlot->iSlots     += slot;
				ExtraUserSlot->bPermanent  = permanent;
			}
			break;
		}
	}

	if ( ExtraUserSlot == 0 )
	{
		ExtraUserSlot = new CExtraUserSlot();
		ExtraUserSlot->sNick      = nick;
		ExtraUserSlot->sHubName   = hubname;
		ExtraUserSlot->iSlots     = slot;
		ExtraUserSlot->bPermanent = permanent;

		m_pExtraUserSlotList->Add(ExtraUserSlot);
	}

	SendSlotInfo(ExtraUserSlot);

	if ( (ExtraUserSlot->iSlots == 0) && (!ExtraUserSlot->bPermanent) )
	{
		m_pExtraUserSlotList->Del(ExtraUserSlot);
	}

	m_pExtraUserSlotList->UnLock();
}

/** */
bool CDownloadManager::DLM_TransferConnect( CString nick, CString hubname )
{
	bool res = FALSE;

	m_pDownloadQueue->pQueue->Lock();

	DCTransferQueueObject * TransferObject;

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( nick, hubname, "" )) != 0 )
	{
		TransferObject->tTimeout = 0;
		res = TRUE;
	}

	m_pDownloadQueue->pQueue->UnLock();

	return res;
}

/** */
bool CDownloadManager::DLM_TransferClose( ulonglong transferid )
{
	CObject *obj;
	CTransferObject * TransferObject;
	bool res = FALSE;

	m_pTransferList->Lock();

	if ( m_pTransferList->Get( CString().setNum(transferid), (CObject*&)obj) == 0 )		
	{
		TransferObject = (CTransferObject*)obj;
		TransferObject->m_pTransfer->Disconnect(TRUE);
		res = TRUE;
	}

	m_pTransferList->UnLock();

	return res;
}

/** */
bool CDownloadManager::DLM_TransferSetRate( ulonglong transferid, ulonglong rate )
{
	CObject * obj;
	CTransferObject * TransferObject;
	bool res = FALSE;

	m_pTransferList->Lock();

	if ( m_pTransferList->Get( CString().setNum(transferid), (CObject*&)obj ) == 0 )
	{
		TransferObject = (CTransferObject*)obj;
		TransferObject->m_pTransfer->SetRate(rate);
		res = TRUE;
	}

	m_pTransferList->UnLock();

	return res;
}

/** */
bool CDownloadManager::DLM_TransferGetRate( ulonglong transferid, ulonglong & rate )
{
	CObject * obj;
	CTransferObject * TransferObject;
	bool res = FALSE;

	m_pTransferList->Lock();

	if ( m_pTransferList->Get( CString().setNum(transferid), (CObject*&)obj) == 0 )
	{
		TransferObject = (CTransferObject*)obj;
		rate = TransferObject->m_pTransfer->GetRate();
		res = TRUE;
	}

	m_pTransferList->UnLock();

	return res;
}

/** */
eDirection CDownloadManager::DLM_TransferDirection( ulonglong transferid )
{
	CObject * obj;
	CTransferObject * TransferObject;
	eDirection direction=edNONE;

	m_pTransferList->Lock();

	if ( m_pTransferList->Get( CString().setNum(transferid), (CObject*&)obj ) == 0 )
	{
		TransferObject = (CTransferObject*)obj;
		direction = TransferObject->m_pTransfer->GetSrcDirection();
	}

	m_pTransferList->UnLock();

	return direction;
}

/** */
void CDownloadManager::DLM_QueueAdd( CString nick, CString hubname, CString hubhost,
				CString remotename, CString localname,
				CString localpath, CString localrootpath,
				eltMedium medium, ulonglong size,
				ulonglong startposition,
				ulonglong endposition,
				CString hash,
				bool multi )
{
	CDir dir;
	CString sfile;
	DCTransferQueueObject * TransferObject;
	DCTransferFileObject * TransferFileObject = 0;
	DCHubObject * HubObject;
	CString fixedHash = hash;
	
	if ((fixedHash.Left(4)).ToUpper() == "TTH:")
	{
		printf("DLM_QueueAdd: Warning! Removing TTH: prefix from TTH\n");
		fixedHash = fixedHash.Mid(4, fixedHash.Length() - 4);
	}
	
/*	DPRINTF("add queue entry: %s %s %s %s %d %s %s %s %llu %llu\n",
		nick.Data(),
		remotename.Data(),
		hubname.Data(),
		hubhost.Data(),
		medium,
		localname.Data(),
		localpath.Data(),
		localrootpath.Data(),
		startposition,
		size );*/

	m_pDownloadQueue->pQueue->Lock();

	CStringList * StringList = m_pDownloadQueue->GetUserHubList( nick );

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( nick, hubname, hubhost )) == 0 )
	{
		TransferObject = new DCTransferQueueObject();

		TransferObject->sNick    = nick;
		TransferObject->sHubHost = hubhost;
		TransferObject->sHubName = hubname;
		TransferObject->eState   = etwsIDLE;
		TransferObject->iConnections = 0;
		TransferObject->tTimeout = 0;

		HubObject = new DCHubObject();

		HubObject->m_sHubName = hubname;
		HubObject->m_sHubHost = hubhost;
		HubObject->m_bActive  = TRUE;

		TransferObject->pHubList.Add(HubObject);

		if ( StringList == 0 )
		{
			StringList = new CStringList();
			m_pDownloadQueue->pQueue->Add( nick, StringList );
		}

		StringList->Add( hubname, TransferObject );

		//DPRINTF("transfer add\n");
	}
	else
	{
		TransferFileObject = m_pDownloadQueue->GetUserFileObject( nick, hubname, hubhost, remotename );
	}

	// file not found for the nick/hub
	if ( TransferFileObject == 0 )
	{
		TransferFileObject = new DCTransferFileObject();

		TransferFileObject->m_eState      = etfsNONE;
		TransferFileObject->m_nSize       = size;
		TransferFileObject->m_bMulti      = multi;
		TransferFileObject->m_eMedium     = medium;
		TransferFileObject->m_sRemoteFile = remotename;
		TransferFileObject->m_sHash       = fixedHash;

		if ( remotename == CString(DC_USER_FILELIST) )
		{
			TransferFileObject->m_nPriority = 0;
		}
		else
		{
			TransferFileObject->m_nPriority = 2;
		}
			
		if ( localrootpath == "" )
		{
			sfile = CConfig::Instance()->GetDownloadFolder();
		}
		else
		{
			sfile = localrootpath;
		}

		localpath = localpath.Replace(':',"");
		localname = localname.Replace(':',"");

		sfile = sfile + "/" + localpath + "/" + localname;
		sfile = dir.SimplePath(sfile);

		TransferFileObject->m_sLocalFile     = sfile;
		TransferFileObject->m_sLocalPath     = localpath;
		TransferFileObject->m_sLocalFileName = localname;

		TransferObject->pTransferFileList.Add( remotename, TransferFileObject );

		//DPRINTF("file add '%s' %ld\n",sfile.Data(),TransferObject->pTransferFileList.Count());

		m_pDownloadQueue->pChunkList->Lock();

		DCFileChunkObject * FileChunkObject;

		if ( remotename == CString(DC_USER_FILELIST) )
		{
			DPRINTF("no chunk for userlists\n");
		}
		else if ( m_pDownloadQueue->pChunkList->Get( sfile, (CObject *&)FileChunkObject ) != 0 )
		{
			FileChunkObject = new DCFileChunkObject();

			FileChunkObject->m_sLocalFile      = sfile;
			FileChunkObject->m_stHash          = "";
			FileChunkObject->m_sHash           = fixedHash;
			FileChunkObject->m_bMulti          = multi;
			FileChunkObject->m_nSize           = size;
			FileChunkObject->m_nSizeDone       = startposition;
			FileChunkObject->m_nReferenceCount = 1;

			// create the first unfinished chunk
			DCChunkObject * ChunkObject = new DCChunkObject();
			ChunkObject->m_nStart = startposition;
			ChunkObject->m_nEnd   = size;

			if( endposition )
			{
				endposition++;
				FileChunkObject->m_nSizeDone = size - (endposition - startposition);
				ChunkObject->m_nEnd = endposition;
			}

			FileChunkObject->m_Chunks.Add(ChunkObject);
			m_pDownloadQueue->pChunkList->Add( sfile, FileChunkObject );

			//DPRINTF("add file chunk object\n");
		}
		else
		{
			FileChunkObject->m_nReferenceCount++;
			DPRINTF("file chunk object found\n");
		}

		m_pDownloadQueue->pChunkList->UnLock();
	}
	else
	{
		DPRINTF("file found ...\n");
	}

	SendFileInfo( TransferObject, TransferFileObject );

	m_pDownloadQueue->pQueue->UnLock();
}

/**	0: file not in the queue
	1: file allready in the queue by same nick und hub
	2: local file with same size allready in the queue (multi download)
	3: local file with same size allready in the queue without multi (no multi download)
	4: local file with not the same size allready in the queue (multi download error)
*/
int CDownloadManager::DLM_QueueCheck( CString nick, CString hubname, CString hubhost,
				CString remotename, CString localname,
				CString localpath, CString localrootpath,
				eltMedium, ulonglong size )
{
	m_pDownloadQueue->pQueue->Lock();

	CDir dir;
	CString sfile;
	int res = 0;
	DCTransferQueueObject * TransferObject;
	DCTransferFileObject * TransferFileObject = 0;

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( nick, hubname, hubhost )) != 0 )
	{
		if ( (TransferFileObject = m_pDownloadQueue->GetUserFileObject( nick, hubname, hubhost, remotename )) != 0 )
		{
			res = 1;
		}
	}

	// check if a local file allready exist
	if ( (res == 0) && (remotename != CString(DC_USER_FILELIST)) )
	{
		if ( localrootpath == "" )
		{
			sfile = CConfig::Instance()->GetDownloadFolder();
		}
		else
		{
			sfile = localrootpath;
		}

		localpath = localpath.Replace(':',"");
		localname = localname.Replace(':',"");

		sfile = sfile + "/" + localpath + "/" + localname;
		sfile = dir.SimplePath(sfile);

		DCFileChunkObject * FileChunkObject;

		m_pDownloadQueue->pChunkList->Lock();

		if ( m_pDownloadQueue->pChunkList->Get( sfile, (CObject *&)FileChunkObject ) == 0 )
		{
			if ( FileChunkObject->m_nSize == size )
			{
				if ( FileChunkObject->m_bMulti == TRUE )
				{
					res = 2;
				}
				else
				{
					res = 3;
				}
			}
			else
			{
				res = 4;
			}
		}

		m_pDownloadQueue->pChunkList->UnLock();
	}

	m_pDownloadQueue->pQueue->UnLock();

	return res;
}

/** change the src-nick/hubname to dst-nick/hubname in the queue */
bool CDownloadManager::DLM_QueueEdit( CString srcnick, CString srchubname, CString dstnick, CString dsthubname, CString dsthubhost )
{
	bool res;
	DCTransferQueueObject * TransferObject;
	DCTransferFileObject * TransferFileObject;

	m_pDownloadQueue->pQueue->Lock();

	res = FALSE;

	// check if transfer run
	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( srcnick, srchubname, "" )) != 0 )
	{
		if ( (TransferObject->eState != etwsWAIT) &&
		     (TransferObject->eState != etwsRUN) )
		{
			// only rename if the new not exists
			if ( m_pDownloadQueue->GetUserTransferObject( dstnick, dsthubname, dsthubhost ) == 0 )
			{
				// send remove all files
				TransferFileObject = 0;

				while( TransferObject->pTransferFileList.Next( (CObject *&)TransferFileObject) )
				{
					SendFileInfo( TransferObject, TransferFileObject, TRUE );
				}

				// rename old entry
				m_pDownloadQueue->RenameNick( srcnick, dstnick, srchubname, dsthubname );

				// get hubhost from new entry
				if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( dstnick, dsthubname, dsthubhost )) != 0 )
				{
					TransferObject->sHubHost = dsthubhost;

					// send new files
					TransferFileObject = 0;

					while( TransferObject->pTransferFileList.Next( (CObject *&)TransferFileObject) )
					{
						SendFileInfo( TransferObject, TransferFileObject );
					}

					res = TRUE;
				}
			}
			else
			{
				// update hubhost
				if ( TransferObject->sHubHost != dsthubhost )
				{
					TransferObject->sHubHost = dsthubhost;
					SendFileInfo( TransferObject );
				}
				
			}
		}
	}

	m_pDownloadQueue->pQueue->UnLock();

	return res;
}

/** */
bool CDownloadManager::DLM_QueuePause( CString nick, CString hubname, CString remotefile, bool pause )
{
	bool res = FALSE;
	DCTransferFileObject * TransferFileObject = 0;
	DCTransferQueueObject * TransferObject = 0;

	m_pDownloadQueue->pQueue->Lock();

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( nick, hubname, "" )) != 0 )
	{
		if ( remotefile != "" )
		{
			if ( (TransferFileObject = m_pDownloadQueue->GetUserFileObject( nick, hubname, "", remotefile )) != 0 )
			{
				if ( TransferFileObject->m_eState != etfsTRANSFER )
				{
					if ( pause )
						TransferFileObject->m_eState = etfsPAUSE;
					else
						TransferFileObject->m_eState = etfsNONE;

					SendFileInfo( TransferObject, TransferFileObject );
					res = TRUE;
				}
			}
		}
		else
		{
			TransferFileObject = 0;

			while ( TransferObject->pTransferFileList.Next( (CObject *&) TransferFileObject ) )
			{
				if ( TransferFileObject->m_eState != etfsTRANSFER )
				{
					if ( pause )
						TransferFileObject->m_eState = etfsPAUSE;
					else
						TransferFileObject->m_eState = etfsNONE;

					SendFileInfo( TransferObject, TransferFileObject );
					res = TRUE;
				}
			}
		}
	}

	m_pDownloadQueue->pQueue->UnLock();

	return res;
}

/** */
bool CDownloadManager::DLM_QueueRemove( CString nick, CString hubname, CString remotefile )
{
	bool res;
	
	m_pDownloadQueue->pQueue->Lock();

	res = RemoveQueueFile( nick, hubname, remotefile );

	m_pDownloadQueue->pQueue->UnLock();
	
	return res;
}

/** */
bool CDownloadManager::DLM_QueueRemove( CString localfile )
{
	bool res;

	m_pDownloadQueue->pQueue->Lock();

	res = RemoveQueueFile(localfile);

	m_pDownloadQueue->pQueue->UnLock();

	return res;
}

/** */
bool CDownloadManager::DLM_QueueSetFilePriority( CString nick, CString hubname, CString remotefile, int priority )
{
	m_pDownloadQueue->pQueue->Lock();

	bool res = FALSE;
	DCTransferQueueObject * TransferObject = 0;
	DCTransferFileObject * TransferFileObject = 0;

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( nick, hubname, "" )) != 0 )
	{
		if ( (remotefile != "") && (priority <= MAX_FILE_PRIORITY) )
		{
			if ( (TransferFileObject = m_pDownloadQueue->GetUserFileObject( nick, hubname, "", remotefile )) != 0 )
			{
				// we don't need check if transfer is running
				TransferFileObject->m_nPriority = priority;

				SendFileInfo( TransferObject, TransferFileObject );

				res = TRUE;
			}
		}
	}

	m_pDownloadQueue->pQueue->UnLock();
	
	return res;
}

/** */
bool CDownloadManager::DLM_QueueGetFileInfo( CString nick, CString hubname, CString hubhost, CString remotefile, CUserFileInfo * UserFileInfo )
{
	bool res = FALSE;

	if (!UserFileInfo)
	{
		return res;
	}

	m_pDownloadQueue->pQueue->Lock();

	DCTransferQueueObject * TransferObject;
	DCTransferFileObject * TransferFileObject;

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( nick, hubname, hubhost )) != 0 )
	{
		UserFileInfo->eWaitState    = TransferObject->eState;

		if ( remotefile != "" )
		{
			if ( TransferObject->pTransferFileList.Get( remotefile, (CObject *&) TransferFileObject ) == 0 )
			{
				UserFileInfo->eFileState = TransferFileObject->m_eState;
				UserFileInfo->sLocalFile = TransferFileObject->m_sLocalFile;
				UserFileInfo->bMulti     = TransferFileObject->m_bMulti;
				res = TRUE;
			}
		}
		else
		{
			res = TRUE;
		}
	}

	m_pDownloadQueue->pQueue->UnLock();

	return res;
}

/** */
DCFileChunkObject * CDownloadManager::DLM_QueueGetFileChunk( CString file )
{
	m_pDownloadQueue->pQueue->Lock();

	DCFileChunkObject * FileChunkObject1, * FileChunkObject2=0;

	if ( (FileChunkObject1 = m_pDownloadQueue->GetFileChunkObject(file)) != 0 )
	{
		FileChunkObject2 = new DCFileChunkObject(FileChunkObject1);
	}

	m_pDownloadQueue->pQueue->UnLock();

	return FileChunkObject2;
}

/** */
bool CDownloadManager::DLM_QueueUpdateHub( CString nick, CString hubname )
{
	m_pDownloadQueue->pQueue->Lock();

	bool res = FALSE;
	DCConfigHubItem hubitem;
	DCTransferQueueObject * TransferObject;

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( nick, hubname, "" )) != 0 )
	{
		if ( (res = CConfig::Instance()->GetPublicHub( hubname, &hubitem )) == TRUE )
			TransferObject->sHubHost = hubitem.m_sHost;
		else if ( (res = CConfig::Instance()->GetBookmarkHub( hubname, &hubitem )) == TRUE )
			TransferObject->sHubHost = hubitem.m_sHost;

		if ( res == TRUE )
			SendFileInfo( TransferObject );
	}

	m_pDownloadQueue->pQueue->UnLock();

	return res;
}

/** */
void CDownloadManager::DLM_QueueGetHub( CString nick, CString hubname, CList<DCHubObject> * list )
{
	DCTransferQueueObject * TransferObject;
	DCHubObject * HubObject1;

	if (!list)
	{
		return;
	}

	list->Clear();

	m_pDownloadQueue->pQueue->Lock();

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( nick, hubname, "" )) != 0 )
	{
		HubObject1=0;

		while( (HubObject1=TransferObject->pHubList.Next(HubObject1)) != 0 )
			list->Add( new DCHubObject(HubObject1) );
	}

	m_pDownloadQueue->pQueue->UnLock();

	return;
}



/** */
bool CDownloadManager::DLM_AddTransferRequest( CString nick, CString userhost, CString hubname, CString hubhost )
{
	bool res;
	DCTransferWait * TransferWait;

	if ( m_eShutdownState != essNONE )
	{
		return FALSE;
	}
	
	m_pTransferWaitList->Lock();

	res = FALSE;
	TransferWait = 0;

	DPRINTF("ATR: '%s' '%s' '%s' '%s'\n",nick.Data(),userhost.Data(),hubname.Data(),hubhost.Data());

	DPRINTF("ATR COUNT: %ld\n",m_pTransferWaitList->Count());

	if ( m_pTransferWaitList->Count() < 250 )
	{
		// check if the user allready in the waitlist
		while ( (TransferWait = m_pTransferWaitList->Next(TransferWait)) != 0 )
		{
			if ( TransferWait->sHubName == hubname )
			{
				if ( ((TransferWait->sNick == nick) && (TransferWait->sNick != "")) ||
				     ((TransferWait->sUserHost == userhost) && (TransferWait->sUserHost != "")))
				{
					DPRINTF("ATR FOUND\n");

					// reset timeout, ignore it on to fast connections
					if ( (time(0)-TransferWait->tTimeout) > 2 )
					{
						// count in requests from a client
						TransferWait->m_nCount++;

						TransferWait->tTimeout = time(0);
						res = TRUE;
					}
					else
					{
						DPRINTF("ATR to fast connections\n");
					}

					break;
				}
			}
		}

		if ( TransferWait == 0 )
		{
			// user not found, add it
			DPRINTF("ATR ADD\n");

			TransferWait = new DCTransferWait();
			TransferWait->sNick     = nick;
			TransferWait->sUserHost = userhost;
			TransferWait->sHubName  = hubname;
			TransferWait->sHubHost  = hubhost;
			TransferWait->tTimeout  = time(0);
			TransferWait->m_nCount  = 1;

			m_pTransferWaitList->Add(TransferWait);
			
			res = TRUE;
		}
		
		if ( res == TRUE )
			if ( CListenManager::Instance() )
				CListenManager::Instance()->AddConnectionWait();
	}
	else
	{
		// TODO: warn message ...
	}

	m_pTransferWaitList->UnLock();

	return res;
}

/** */
void CDownloadManager::DLM_AddTransferRequest( CString host, int port, CString hubname, CString hubhost )
{
	if ( m_eShutdownState != essNONE )
	{
		return;
	}

	DPRINTF("ATR: '%s:%d' '%s' '%s'\n",host.Data(),port,hubname.Data(),hubhost.Data());

	// check private address space
	if ( !((CConfig::Instance()->GetCheckPrivateAddressSpace() == TRUE) &&
	     (CSocket::IsPrivateAddressSpace(host.Data()) == TRUE )) )
	{
		// TODO: check if client allready in the transfer wait list
		// create a new transfer
		CTransferObject * TransferObject = new CTransferObject();
		TransferObject->m_pTransfer = new CTransfer();

		TransferObject->m_pTransfer->SetTransferID( GetNewID() );
		TransferObject->m_pTransfer->SetNick( CConfig::Instance()->GetNick( hubname, hubhost ) );

		TransferObject->m_pTransfer->SetHubName(hubname);
		TransferObject->m_pTransfer->SetHubHost(hubhost);

		TransferObject->m_pTransfer->SetHost( host, port );
		TransferObject->m_pTransfer->SetRate( CConfig::Instance()->GetMaxUploadRate() );

		// add this transfer to the wait list
		if ( DLM_AddTransferRequest( "", TransferObject->m_pTransfer->GetHost(), hubname, hubhost ) == FALSE )
		{
			// request allready in the list
			delete TransferObject->m_pTransfer;
			TransferObject->m_pTransfer = 0;
			delete TransferObject;
		}
		else
		{
			m_pTransferList->Lock();

			m_pTransferList->Add( CString().setNum(TransferObject->m_pTransfer->GetTransferID()), TransferObject );
			TransferObject->m_pTransfer->SetCallBackFunction( new CCallback<CDownloadManager>( this, &CDownloadManager::DM_TransferCallBack ) );

			DPRINTF("ATR CONNECT: %s:%d %s %s\n",host.Data(),port,hubname.Data(),hubhost.Data());

			// connect to the client
			TransferObject->m_pTransfer->Connect();

			m_pTransferList->UnLock();
		}
	}
	else
	{
		// send warning
		SendLogInfo( "Warning: Detect private address space: " + host + ":" + CString().setNum(port) + " at hub '" + hubname + "' ("+hubhost+")" );
	}
}

/** */
bool CDownloadManager::DLM_GetDownloadManagerInfo( CDownloadManagerInfo * pinfo )
{
	*pinfo = DownloadManagerInfo;

	return TRUE;
}

/** handle search */
bool CDownloadManager::DLM_HandleSearch( CMessageSearchResult * MessageSearchResult )
{
	bool res = FALSE;
	CMessageSearchResult * msg;
	DCTransferFileObject * tfo1=0, * tfo2=0;

	msg = 0;
	
	while ( (msg = m_pSearchList->Next(msg)) != 0 )
	{
		//MessageSearchResult->sNick += "+";
		if ( msg->m_nSize == MessageSearchResult->m_nSize )
		{
			// handle this file
			m_pDownloadQueue->pQueue->Lock();
			
			// check if this file allready in the queue
			if ( m_pDownloadQueue->GetUserFileObject( MessageSearchResult->m_sNick, MessageSearchResult->m_sHubName, MessageSearchResult->m_sHubHost, MessageSearchResult->m_sFile ) == 0 )
			{
				// get original file
				if ( (tfo1=m_pDownloadQueue->GetUserFileObject( msg->m_sNick, msg->m_sHubName, msg->m_sHubHost, msg->m_sFile )) != 0 )
				{
					tfo2 = new DCTransferFileObject(*tfo1);
				}
			}

			m_pDownloadQueue->pQueue->UnLock();
			
			if ( tfo2 )
				break;
		}
	}

	if ( tfo2 )
	{
		CDir dir;
		CString path,file;
		int adj = 0;

		dir.SplitPathFile( tfo2->m_sLocalFile, path, file );
		
		// remove tfo2->m_sLocalPath from path
		if ( (path.Right(1) == DIRSEPARATOR) && (tfo2->m_sLocalPath.Right(1) != DIRSEPARATOR) )
		{
			adj = 1;
		}
		else if ( (path.Right(1) != DIRSEPARATOR) && (tfo2->m_sLocalPath.Right(1) == DIRSEPARATOR) )
		{
			adj = -1;
		}
		
		path = path.Left( path.Length() - (tfo2->m_sLocalPath.Length() + adj) );

		DLM_QueueAdd( MessageSearchResult->m_sNick,
			MessageSearchResult->m_sHubName,
			MessageSearchResult->m_sHubHost,
			MessageSearchResult->m_sFile,
			tfo2->m_sLocalFileName,
			tfo2->m_sLocalPath,
			path,
			tfo2->m_eMedium,
			tfo2->m_nSize,
			0, 0,
			MessageSearchResult->m_sHash,
			TRUE );
		delete tfo2;
	}

	return res;
}

/** */
void CDownloadManager::SendFileInfo( DCTransferQueueObject * TransferObject, DCTransferFileObject * TransferFileObject, bool bRemoveFile )
{
	if ( m_eShutdownState != essNONE )
		return;

	LogThread.Lock();

	CMessageDMFileObject * fo = new CMessageDMFileObject();

	fo->m_sNick              = TransferObject->sNick;
	fo->m_sHubName           = TransferObject->sHubName;
	fo->m_sHubHost           = TransferObject->sHubHost;
	fo->m_eTransferWaitState = TransferObject->eState;
	fo->m_tTimeout           = TransferObject->tTimeout;
	fo->m_bRemoveFile        = bRemoveFile;
	fo->m_nConnections       = TransferObject->iConnections;

	if ( TransferFileObject )
	{
		fo->m_sRemoteFile        = TransferFileObject->m_sRemoteFile;
		fo->m_sLocalFile         = TransferFileObject->m_sLocalFile;
		fo->m_nSize              = TransferFileObject->m_nSize;
		fo->m_eTransferFileState = TransferFileObject->m_eState;
		fo->m_bMulti             = TransferFileObject->m_bMulti;
		fo->m_nPriority          = TransferFileObject->m_nPriority;
	}

	if ( DC_DownloadManagerCallBack(fo) == -1 )
		delete fo;

	LogThread.UnLock();
}

/** */
CMessageDMTransferObject * CDownloadManager::CreateDMTransferObject( CTransfer * Transfer )
{
	DCFileChunkObject * FileChunkObject;
	CMessageDMTransferObject * to = new CMessageDMTransferObject();

	to->m_nTransferID    = Transfer->GetTransferID();
	to->m_sSrcNick       = Transfer->GetNick();
	to->m_sDstNick       = Transfer->GetDstNick();
	to->sHost            = Transfer->GetHost();
	to->m_sHubHost       = Transfer->GetHubHost();
	to->sHubName         = Transfer->GetHubName();
	to->eState           = Transfer->GetMode();
	to->m_sDstFile       = Transfer->GetDstFilename();
	to->m_sSrcFile       = Transfer->GetSrcFilename();
	to->lSize            = Transfer->GetLength();
	to->lStartPosition   = Transfer->GetStartPosition();
	to->lEndPosition     = Transfer->GetEndPosition();
	to->lRate            = Transfer->GetTransferrate();
	to->lTransfered      = Transfer->GetTransfered();
	to->m_bEncrypted     = Transfer->GetEncrypted();

	// get size done from the chunk list
	if ( Transfer->GetSrcDirection() == edUPLOAD )
	{
		to->lSizeDone = to->lStartPosition + to->lTransfered;
	}
	else
	{
		m_pDownloadQueue->pChunkList->Lock();

		if ( Transfer->GetMedium() == eltBUFFER )
		{
			to->lSizeDone = to->lStartPosition + to->lTransfered;
		}
		else if ( (FileChunkObject = m_pDownloadQueue->GetFileChunkObject(Transfer->GetSrcFilename())) != 0 )
		{
			to->lSizeDone = FileChunkObject->m_nSizeDone+to->lTransfered;
		}
		else
		{
			to->lSizeDone = to->lSize;
		}

		m_pDownloadQueue->pChunkList->UnLock();	
	}

	return to;
}

/** */
CList<CObject> * CDownloadManager::DLM_TransferGetList()
{
	CTransferObject * TransferObject;
	CObject *obj;
	CList<CObject> * list;
	CMessageDMTransferObject * to;

	m_pTransferList->Lock();

	obj = 0;

	list = new CList<CObject>();

	while( m_pTransferList->Next( (CObject*&)obj) )
	{
		TransferObject = (CTransferObject*)obj;
 		to = CreateDMTransferObject(TransferObject->m_pTransfer);
		list->Add(to);
	}

	m_pTransferList->UnLock();

	return list;
}

/** */
void CDownloadManager::SendTransferInfo( CTransfer * Transfer, bool remove )
{
	LogThread.Lock();

	CMessageDMTransferObject * to = CreateDMTransferObject(Transfer);

	to->bRemoveTransfer  = remove;

	if ( DC_DownloadManagerCallBack(to) == -1 )
		delete to;

	LogThread.UnLock();
}

/** */
void CDownloadManager::SendSlotInfo( CExtraUserSlot * Object )
{
	LogThread.Lock();

	CMessageDMSlotObject * so = new CMessageDMSlotObject();

	so->sNick      = Object->sNick;
	so->sHubName   = Object->sHubName;
	so->iSlots     = Object->iSlots;
	so->bPermanent = Object->bPermanent;

	so->m_eType = DC_MESSAGE_SLOT_OBJECT;

	if ( DC_DownloadManagerCallBack(so) == -1 )
		delete so;

	LogThread.UnLock();
}

/** */
void CDownloadManager::SendLogInfo( CString message, CTransfer * Transfer )
{
	LogThread.Lock();

	CString s = "";

	CMessageLog * log = new CMessageLog();

	if ( Transfer != 0 )
	{
		s = "[";
		if ( Transfer->GetDstNick() == "" )
			s += "???";
		else
			s += Transfer->GetDstNick();
		s += "] ";
	}
	
	s += message;
	
	log->sMessage = s;
	log->m_eType  = DC_MESSAGE_LOG;

	// write message in log tab in transfer window to logfile
	if ( (CConfig::Instance()->GetLogFile() == TRUE) && (CConfig::Instance()->GetLogDetails() == TRUE) )
	{
		CLogFile::Write(CConfig::Instance()->GetLogFileName(),eltINFO,s);
	}

	if ( DC_DownloadManagerCallBack(log) == -1 )
		delete log;

	LogThread.UnLock();
}

/** */
void CDownloadManager::SendDownloadManagerInfo( CDownloadManagerInfo * dminfo )
{
	if ( !dminfo )
	{
		return;
	}

	LogThread.Lock();

	CDownloadManagerInfo * dmi = new CDownloadManagerInfo;

	if ( dmi )
	{
		*dmi = *dminfo;

		if ( DC_DownloadManagerCallBack(dmi) == -1 )
			delete dmi;
	}

	LogThread.UnLock();
}

/** */
void CDownloadManager::SendTrafficInfo()
{
	LogThread.Lock();

	DCMessageTraffic * to = new DCMessageTraffic();

	to->m_nRx        = CSocket::m_Traffic.GetTraffic(ettRX);
	to->m_nTx        = CSocket::m_Traffic.GetTraffic(ettTX);
	to->m_nDataRx    = CSocket::m_Traffic.GetTraffic(ettDATARX);
	to->m_nDataTx    = CSocket::m_Traffic.GetTraffic(ettDATATX);
	to->m_nControlRx = CSocket::m_Traffic.GetTraffic(ettCONTROLRX);
	to->m_nControlTx = CSocket::m_Traffic.GetTraffic(ettCONTROLTX);

	if ( DC_DownloadManagerCallBack(to) == -1 )
		delete to;

	LogThread.UnLock();
}

/** */
void CDownloadManager::SendFileManagerInfo( CFileManagerInfo * info )
{
	if ( m_eShutdownState != essNONE )
	{
		return;
	}
	
	if ( !info )
	{
		return;
	}

	LogThread.Lock();

	CFileManagerInfo * fmi = new CFileManagerInfo();

	if ( fmi )
	{
		*fmi = *info;

		if ( DC_DownloadManagerCallBack(fmi) == -1 )
			delete fmi;
	}

	LogThread.UnLock();
}

/** */
bool CDownloadManager::CheckUserSlot( CString nick, CString )
{
	bool res;
	CExtraUserSlot * ExtraUserSlot;

	m_pExtraUserSlotList->Lock();

	res = FALSE;
	ExtraUserSlot = 0;

	while( (ExtraUserSlot=m_pExtraUserSlotList->Next(ExtraUserSlot)) != 0 )
	{
		if ( ExtraUserSlot->sNick == nick )
		{
			if ( ExtraUserSlot->iSlots > 0 )
			{
				ExtraUserSlot->iSlots--;

				SendSlotInfo(ExtraUserSlot);

				if ( ExtraUserSlot->iSlots == 0 )
				{
					m_pExtraUserSlotList->Del(ExtraUserSlot);
				}

				res = TRUE;
			}
			else if ( ExtraUserSlot->bPermanent )
			{
				res = TRUE;
			}

			break;
		}
	}

	m_pExtraUserSlotList->UnLock();

	return res;
}

/** */
void CDownloadManager::FileListDone( CTransfer * Transfer )
{
	DCTransferQueueObject * TransferObject;
	CByteArray ba;
	CString s,fn;
	CXml xml;
	CDir dir;
	CFile file;
	int mode = 0;
	
	if ( Transfer->GetBuffer(&ba) != 0 )
	{
		if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( Transfer->GetDstNick(), Transfer->GetHubName(), Transfer->GetHubHost() )) != 0 )
		{
			dir.SetPath( CConfig::Instance()->GetFileListPath() );
			
			// printf("Nick=%s, HubHost=%s\n", Transfer->GetDstNick().Data(), Transfer->GetHubHost().Data() );
			
			fn = Transfer->GetDstNick() + "-" + Transfer->GetHubHost() + ".filelist";
			
			// convert invalid characters - from dcchat.cpp line 1156
			fn = fn.Replace('/', "_");
			fn = fn.Replace('\\', "_");
			fn = fn.Replace(':', "_");
			
			// printf("Filelist name = %s\n", fn.Data());
			
			if ( dir.IsFile(fn) == FALSE )
			{
				mode |= IO_CREAT;
			}
			else
			{
				mode |= IO_TRUNCATE;
			}
			
			if ( fn != "" )
			{
				mode |= IO_RAW | IO_WRITEONLY;

				s  = "---HEADER START---\n";
				
				if ( Transfer->GetSupport().m_bXMLBZList == TRUE )
				{
					s += "TYPE=XMLFILELIST\n";
				}
				else
				{
					s += "TYPE=FILELIST\n";
				}

				// since dclib now does all encoding to/from utf8 in
				// cdcproto and cmessagehandler, these should all be
				// already in UTF-8
            			s += "NICK=" + TransferObject->sNick + "\n";
            			s += "HUBNAME=" + TransferObject->sHubName + "\n";
            			s += "HUBHOST=" + TransferObject->sHubHost + "\n";
            			s += "---HEADER END---\n";
				
				if ( file.Open( CConfig::Instance()->GetFileListPath()+fn, mode, MO_IRUSR|MO_IWUSR|MO_IRGRP|MO_IROTH ) == TRUE )
				{
					file.Write(s.Data(),s.Length());
					
					// move list to string
					s.Set((char*)ba.Data(),ba.Size());
					
					if ( Transfer->GetSupport().m_bXMLBZList == FALSE )
					{
						s = xml.ToUTF8( s.Data(), CConfig::Instance()->GetRemoteEncoding() );
					}
					else
					{
						// sanity check for empty xml lists
						if ( s == "" )
						{
							s += "<?xml version=\"1.0\" encoding=\"utf-8\" standalone=\"yes\"?>";
							s += "<FileListing Version=\"1\" Generator=\"\">";
							s += "</FileListing>";
						}
					}

					file.Write((char*)s.Data(),s.Length());
					file.Close();
				}
			}

			CMessageDMFileListObject * flo = new CMessageDMFileListObject();

			flo->sNick         = TransferObject->sNick;
			flo->sHubName      = TransferObject->sHubName;
			flo->sHubHost      = TransferObject->sHubHost;
			flo->sLocalFile	   = CConfig::Instance()->GetFileListPath()+fn;
			
			flo->m_eType = DC_MESSAGE_FILELIST_OBJECT;

			LogThread.Lock();

			if ( DC_DownloadManagerCallBack(flo) == -1 )
				delete flo;

			LogThread.UnLock();
		}
	}
}

/** */
bool CDownloadManager::CheckHash( CTransfer * Transfer )
{
	bool res = FALSE;
	CByteArray ba;
	DCTransferFileObject * TransferFileObject;
	DCFileChunkObject * FileChunkObject;
	md5_ctx * context = 0;
	unsigned char resbuf[16];
	char c[3];
	int i = 0;
	CString result = "";

	if ( (TransferFileObject = m_pDownloadQueue->GetUserFileObject( Transfer->GetDstNick(), Transfer->GetHubName(), Transfer->GetHubHost(), Transfer->GetDstFilename() )) != 0 )
	{
		if ( Transfer->GetBuffer(&ba) != 0 )
		{
			if ( (TransferFileObject->m_stHash == "") && (TransferFileObject->m_bMulti == TRUE) )
			{
				// calc hash
				context = new md5_ctx();
				
				md5_init_ctx( context );
				md5_process_bytes( ba.Data(), ba.Size(), context );
				md5_finish_ctx( context, &resbuf );
				
				delete context;
				
				// convert digest to hexadecimal
				for ( i = 0; i < 16; i++ )
				{
					sprintf( c, "%02x", resbuf[i] );
					c[2] = 0;
					result += c;
				}

				TransferFileObject->m_stHash = result;

				DPRINTF("hash is :'%s'\n",TransferFileObject->m_stHash.Data());

				m_pDownloadQueue->pChunkList->Lock();

				if ( m_pDownloadQueue->pChunkList->Get( Transfer->GetSrcFilename(), (CObject *&)FileChunkObject ) == 0 )
				{
					if ( FileChunkObject->m_stHash == "" )
					{
						DPRINTF("Set hash ...\n");
						FileChunkObject->m_stHash = TransferFileObject->m_stHash;

						res = TRUE;
					}
					else if ( FileChunkObject->m_stHash == TransferFileObject->m_stHash )
					{
						DPRINTF("Hash ok...\n");

						res = TRUE;
					}
					else
					{
						DPRINTF("Wrong hash !!!\n");
						TransferFileObject->m_eState = etfsERROR;
					}
				}
				else
				{
					DPRINTF("warning file chunk object not found\n");
				}

				m_pDownloadQueue->pChunkList->UnLock();
			}
			else
			{
				DPRINTF("warning hash not empty or no multi download\n");
			}
		}
		else
		{
			DPRINTF("warning file object not found\n");
		}
	}
	else
	{
		DPRINTF("warning get buffer error\n");
	}

	return res;
}

/** 
bool CDownloadManager::SetMultiDownload( CString nick, CString hubname, CString remotefile )
{
	Lock();
	bool res = FALSE;
	DCTransferQueueObject * TransferObject;
	DCTransferFileObject * TransferFileObject;
	DCFileChunkObject * FileChunkObject;

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( nick, hubname )) != 0 )
	{
		UserFileInfo->eWaitState    = TransferObject->eState;
		UserFileInfo->sUserFileList = TransferObject->sUserFileList;

		if ( TransferObject->pTransferFileList.Get( remotefile, (CObject *&) TransferFileObject ) == 0 )
		{
			if ( (TransferFileObject->eTransferFileState != etfsTRANSFER) &&
			     (TransferFileObject->bMulti == FALSE) )
			{
				m_pDownloadQueue->pChunkList->Lock();

				if ( (FileChunkObject = m_pDownloadQueue->GetFileChunkObject(TransferFileObject->sLocalFile)) != 0 )
				{
					DCChunkObject * ChunkObject = 0;
					while( (ChunkObject=FileChunkObject->pChunks.Next(ChunkObject)) != 0 )
					{
						if ( ChunkObject->lStart <= TEST_CHUNK_SIZE )
						{
							// error
							break;
						}
					}

					if ( ChunkObject == 0 )
				}

				m_pDownloadQueue->pChunkList->UnLock();
			}

			UserFileInfo->eFileState = TransferFileObject->eState;
			UserFileInfo->sLocalFile = TransferFileObject->sLocalFile;
			UserFileInfo->bMulti     = TransferFileObject->bMulti;
			res = TRUE;
		}
	}

	UnLock();

	return res;
}
*/

/** */
ulonglong CDownloadManager::GetNewID()
{
	return ((++m_nID)==0) ? (++m_nID) : (m_nID);
}

/** */
bool CDownloadManager::RemoveQueueFile( CString localfile )
{
	bool res = FALSE;
	CStringList * StringList;
	DCTransferQueueObject * TransferObject;
	DCTransferFileObject * TransferFileObject;

	// remove file from the chunk list
	m_pDownloadQueue->pChunkList->Lock();
	m_pDownloadQueue->pChunkList->Del( localfile );
	m_pDownloadQueue->pChunkList->UnLock();

	// remove file from the queue
	StringList = 0;

	while( m_pDownloadQueue->pQueue->Next( (CObject *&) StringList) )
	{
		TransferObject = 0;

		while( StringList->Next( (CObject *&) TransferObject) )
		{
			TransferFileObject = 0;

			while( TransferObject->pTransferFileList.Next( (CObject *&) TransferFileObject) )
			{
				if ( TransferFileObject->m_sLocalFile == localfile )
				{
					// on file transfer don't remove the file
					if ( TransferFileObject->m_eState != etfsTRANSFER )
					{
						SendFileInfo( TransferObject, TransferFileObject, TRUE );
						TransferObject->pTransferFileList.Del(TransferFileObject->m_sRemoteFile);
						TransferFileObject = 0;
						res = TRUE;
					}
					else
					{
						DPRINTF("WARNING: RemoveQueueFile: file transfer is running\n");
					}
				}
			}
		}
	}

	return res;
}

/** */
bool CDownloadManager::RemoveQueueFile( CString nick, CString hubname, CString remotefile )
{
	bool res = FALSE;
	DCTransferFileObject * TransferFileObject = 0;
	DCTransferQueueObject * TransferObject = 0;

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( nick, hubname, "" )) != 0 )
	{
		if ( remotefile != "" )
		{
			if ( (TransferFileObject = m_pDownloadQueue->GetUserFileObject( nick, hubname, "", remotefile )) != 0 )
			{
				if ( TransferFileObject->m_eState != etfsTRANSFER )
				{
					res = TRUE;
				}
				else
				{
					DPRINTF("WARNING: RemoveQueueFile: file transfer is running\n");
				}
			}
		}
		else
		{
			if ( TransferObject->eState != etwsRUN )
			{
				res = TRUE;
			}
			else
			{
				DPRINTF("WARNING: RemoveQueueFile: transfer is running\n");
			}
		}

		if ( res == TRUE )
		{
			SendFileInfo( TransferObject, TransferFileObject, TRUE );
			res = m_pDownloadQueue->DelUserFileObject( nick, hubname, "", remotefile );
		}
	}

	return res;
}

/** */
bool CDownloadManager::UpdateWaitTransfer( CTransfer * Transfer, bool rem )
{
	bool res = FALSE;
	DCTransferWait * TransferWait = 0;

	// search in the wait list ...
	m_pTransferWaitList->Lock();

	DPRINTF("UWT: Search user in the waitlist\n");

	while ( (TransferWait = m_pTransferWaitList->Next(TransferWait)) != 0 )
	{
		// search for the nick ...
		if ( ((TransferWait->sNick == Transfer->GetDstNick()) && (TransferWait->sNick != "")) ||
		     ((TransferWait->sUserHost == Transfer->GetHost()) && (TransferWait->sUserHost != "")) )
		{
			Transfer->SetHubName(TransferWait->sHubName);
			Transfer->SetHubHost(TransferWait->sHubHost);

			DPRINTF("UWT: User found\n");

			res = TRUE;
			break;
		}
	}

	if ( !TransferWait )
	{
		DPRINTF("UWT: User not found\n");
	}
	else
	{
		CListenManager::Instance()->RemoveConnectionWait();

		if ( rem )
		{
			TransferWait->m_nCount--;
		
			DPRINTF("UWT: Remove user %lld\n",TransferWait->m_nCount);

			if ( TransferWait->m_nCount == 0 )
				m_pTransferWaitList->Del(TransferWait);
		}
	}

	m_pTransferWaitList->UnLock();

	return res;
}

/** */
eDirection CDownloadManager::CheckWaitTransfer( CTransfer * Transfer )
{
	m_pDownloadQueue->pQueue->Lock();

	eDirection res = edNONE;
	DCTransferQueueObject * TransferObject;
	DCTransferBanObject * TransferBanObject;
	CString host;
	bool waittransfer;
	int port;
	bool ban = FALSE;
	long int i;
	
	DPRINTF("CWT: '%s' on '%s'\n",
			Transfer->GetDstNick().Data(),
			Transfer->GetHubName().Data() );

	// allow only 1 request in 30 seconds
	m_pTransferBanList->Lock();

	TransferBanObject = 0;

	if ( Transfer->GetPeerName( &host, &port ) == FALSE )
	{
		// error
		DPRINTF("CWT: Error: Can't get peername\n");
	}
	else if ( m_pTransferBanList->Get( Transfer->GetDstNick(), (CObject*&)TransferBanObject ) != 0 )
	{
		DPRINTF("CWT: Create new TransferBanObject '%s'\n",host.Data());

		TransferBanObject = new DCTransferBanObject();

		TransferBanObject->m_sIP   = host;
		TransferBanObject->m_tTime = time(0);

		m_pTransferBanList->Add( Transfer->GetDstNick(), TransferBanObject );

		DPRINTF("CWT: Banlist count %ld objects\n",m_pTransferBanList->Count());
	}

	if ( (TransferBanObject != 0) && (TransferBanObject->m_nRequestCount > 0) )
	{
		i = lrint(ceil((time(0)-TransferBanObject->m_tTime)/60.0)*4);

		if ( i < TransferBanObject->m_nRequestCount )
		{
			ban = TRUE;
		}
	}

	waittransfer = UpdateWaitTransfer( Transfer );
	
	DPRINTF("CWT: CheckWaitTransfer II: %s on %s\n",
			Transfer->GetDstNick().Data(),
			Transfer->GetHubName().Data() );

	// now we check if we need to set the correct nick for the hub
	if ( Transfer->GetNick() == "" )
	{
		Transfer->SetNick( CConfig::Instance()->GetNick( Transfer->GetHubName(), Transfer->GetHubHost() ) );

		DPRINTF("CWT: Set transfer NICK: '%s'\n",Transfer->GetNick().Data() );

		// send the correct nick
		Transfer->SendMyNick(Transfer->GetNick());
	}

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( Transfer->GetDstNick(), Transfer->GetHubName(), Transfer->GetHubHost() )) != 0 )
	{
		DPRINTF("CWT: Waiting: %s on %s %s\n",
				TransferObject->sNick.Data(),
				TransferObject->sHubName.Data(),
				TransferObject->sHubHost.Data() );

		if ( TransferObject->eState == etwsIDLE )
		{
			SendLogInfo( "WARNING: Increase the response timeout." );
		}

		if ( (TransferObject->eState == etwsWAIT) || (TransferObject->eState == etwsIDLE) )
		{
			DPRINTF("CWT: wait found ...\n");

			TransferObject->eState = etwsRUN;
			TransferObject->iConnections++;

			// waiting transfer found ...
			res = edDOWNLOAD;

			SendFileInfo( TransferObject );
		}
		else
		{
			DPRINTF("CWT: ERROR: wait in wrong state (please report!) (%d/%d)\n",TransferObject->eState,TransferObject->iConnections);
		}
	}

	if ( (res == edNONE) && (waittransfer == FALSE) )
	{
		DPRINTF("CWT: Warning: no wait transfer found for '%s'\n",Transfer->GetDstNick().Data());
		Transfer->Disconnect(TRUE);
	}
	else if ( (res == edNONE) && (waittransfer == TRUE) )
	{
		TransferBanObject->m_nRequestCount++;

		DPRINTF("CWT: Requestcount is now %d\n",TransferBanObject->m_nRequestCount);

		if ( ban == TRUE )
		{
			Transfer->Disconnect(TRUE);
			SendLogInfo( "WARNING: Disconnect aggressive client " + host );

			DPRINTF("CWT: Host banned\n");
		}
		else
		{
			res = edUPLOAD;
		}
	}

	m_pTransferBanList->UnLock();

	m_pDownloadQueue->pQueue->UnLock();

	return res;
}

/** */
bool CDownloadManager::ChangeDirection( CTransfer * Transfer )
{
	bool res = FALSE;
	DCTransferQueueObject * TransferObject;

	m_pDownloadQueue->pQueue->Lock();

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( Transfer->GetDstNick(), Transfer->GetHubName(), Transfer->GetHubHost() )) != 0 )
	{
		DPRINTF("Waiting: %s on %s %s\n",
			TransferObject->sNick.Data(),
			TransferObject->sHubName.Data(),
			TransferObject->sHubHost.Data() );

		// change to upload ...
		if ( Transfer->GetSrcDirection() == edDOWNLOAD )
		{
			if ( TransferObject->eState == etwsRUN )
			{
				if ( TransferObject->iConnections > 0 )
					TransferObject->iConnections--;
				else
					DPRINTF("WARNING: ChangeDirection: RUN:0\n");
				if ( TransferObject->iConnections == 0 )
					TransferObject->eState = etwsIDLE;
				SendFileInfo( TransferObject );

				DPRINTF("change transfer -> upload ...\n");

				res = TRUE;
			}
			else
			{
				DPRINTF("can't change transfer upload ...\n");
			}
		}
	}

	m_pDownloadQueue->pQueue->UnLock();

	return res;
}

/** */
bool CDownloadManager::SetNextFile( CTransfer * Transfer )
{
	bool res;

	m_pDownloadQueue->pQueue->Lock();
	res = SetFile(Transfer);
	m_pDownloadQueue->pQueue->UnLock();

	return res;
}

/** */
bool CDownloadManager::SetDirection( CTransfer * Transfer )
{
	CTransferObject * TransferObject;
	CObject *obj;
	int i;
	bool res = FALSE;

	// check if transfer mode correct
	if ( (Transfer->GetSrcDirection() == edNONE) ||
	     (Transfer->GetDstDirection() == edNONE) )
	{
		// never use slots for wrong transfer modes
		return res;
	}

	// check max uploads for a user
	if ( Transfer->GetSrcDirection() == edUPLOAD )
	{
		// check for max user upload slots
		obj = 0;
		i = 0;

		// the list is allready locked by the thread
		while( m_pTransferList->Next( (CObject*&)obj) )
		{
			TransferObject = (CTransferObject*)obj;

			if ( TransferObject->m_pTransfer->GetDstDirection() == edDOWNLOAD )
			{
				if ( Transfer->GetDstNick() == TransferObject->m_pTransfer->GetDstNick() )
				{
					i++;
				}
			}
		}

		// the current transfer is allready in the list
		if ( (CConfig::Instance()->GetUserUploadSlots() == 0) ||
		     (CConfig::Instance()->GetUserUploadSlots() >= i) )
		{
			// check for extra user slots
			if ( (res = CheckUserSlot( Transfer->GetDstNick(), Transfer->GetHubName() )) == TRUE )
			{
				DownloadManagerInfo.slot_use_user++;
				Transfer->SetTransferType(ettUSER);
				res = TRUE;
			}
			else
			{
				// check slot limit
				i = CConfig::Instance()->GetMaxUpload();

				// unlimit
				if ( i == 0 )
				{
					res = TRUE;
				}
				else if ( i > DownloadManagerInfo.slot_use_settings )
				{
					res = TRUE;
				}

				if (res)
				{
					DownloadManagerInfo.slot_use_settings++;
					Transfer->SetTransferType(ettSETTINGS);
				}
				else // no more free slots ... a op can download the filelist ....
        			{
					if ( Transfer->GetDstNick() == "" )
					{
						DPRINTF("WARNING: get a free slot -> remote nick is empty\n");
					}
					else
					{
						// check if user operator
						if ( CConnectionManager::Instance()->IsAdmin(Transfer->GetHubName(),Transfer->GetDstNick()) == TRUE )
						{
							// set operator slot if available
							if ( 3 >= DownloadManagerInfo.slot_use_operator )
							{
								DownloadManagerInfo.slot_use_operator++;
								Transfer->SetTransferType(ettOPERATOR);
								res = TRUE;
							}
						}

						// no operator or no free operator slot ... set to special
						if ( res == FALSE )
						{
							// set special slot if available
							if ( 3 >= DownloadManagerInfo.slot_use_special )
							{
								DownloadManagerInfo.slot_use_special++;
								Transfer->SetTransferType(ettSPECIAL);
								res = TRUE;
							}
						}
					}
				}
			}
		}
	}
	// allow all downloads
	else
	{
		res = TRUE;
	}

	return res;
}

/** */
void CDownloadManager::OptimizeChunks( DCFileChunkObject * FileChunkObject )
{
	DCChunkObject *co, *co1;

	// get first not locked chunk
	co = 0;
	while ( (co=FileChunkObject->m_Chunks.Next(co)) != 0 )
	{
		if ( co->m_eChunkState == ecsFREE )
		{
			co1 = co;
			while ( (co1=FileChunkObject->m_Chunks.Next(co1)) != 0 )
			{
				if ( co1 == co )
				{
					continue;
				}

				if ( co1->m_eChunkState == ecsFREE )
				{
					if ( co->m_nEnd == co1->m_nStart )
					{
						co->m_nEnd = co1->m_nEnd;
						FileChunkObject->m_Chunks.Del(co1);
						co1 = co;
					}
					else if ( co->m_nStart == co1->m_nEnd )
					{
						co->m_nStart = co1->m_nStart;
						FileChunkObject->m_Chunks.Del(co1);
						co1 = co;
					}
				}
			}
		}
	}

}

/** */
bool CDownloadManager::GetNextChunk( CString file, ulonglong * lstart, ulonglong * lend )
{
	bool res = FALSE;
	DCFileChunkObject * FileChunkObject;
	DCChunkObject * ChunkObject, *co, *co1, *bigco, *firstco;
	ulonglong size;
	bool split, standalone;

	DPRINTF("get the next chunk for '%s'\n",file.Data());

	if ( (FileChunkObject = m_pDownloadQueue->GetFileChunkObject(file)) != 0 )
	{
		OptimizeChunks(FileChunkObject);

		co = 0;
		ChunkObject = 0;
		bigco = 0;
		firstco = 0;
		size = 0;

		// get a free chunk
		while ( (ChunkObject=FileChunkObject->m_Chunks.Next(ChunkObject)) != 0 )
		{
			if ( ChunkObject->m_eChunkState == ecsFREE )
			{
				// find the biggest chunk
				if ( (ChunkObject->m_nEnd-ChunkObject->m_nStart) > size )
				{
					bigco = ChunkObject;
					size = ChunkObject->m_nEnd-ChunkObject->m_nStart;
				}
				// find the first chunk with nothing before it
				standalone = TRUE;
				while ( (co=FileChunkObject->m_Chunks.Next(co)) != 0 )
				{
					if ( co->m_nEnd == ChunkObject->m_nStart )
					{
						standalone = FALSE;
					}
				}
				if (standalone) {
					if ( firstco == 0 || firstco->m_nStart > ChunkObject->m_nStart ) {
						firstco = ChunkObject;
					}
				}
			}
		}

		// prefer the first free chunk to the biggest
		if ( firstco != 0 ) {
			co = firstco;
		} else {
			co = bigco;
		}

		// found a free chunk
		if ( co != 0 )
		{
			split = FALSE;

			// split if the half chunk size > max. chunk size
			if ( ((co->m_nEnd-co->m_nStart)/2) > CHUNK_SIZE )
			{
				ChunkObject = 0;

				// check if a running chunk for this chunk
				while ( (ChunkObject=FileChunkObject->m_Chunks.Next(ChunkObject)) != 0 )
				{
					if ( ChunkObject->m_eChunkState == ecsLOCKED )
					{	
						if ( ChunkObject->m_nEnd == co->m_nStart )
						{
							split = TRUE;
						}
					}
				}
			}

			ChunkObject = co;

			if ( ( (ChunkObject->m_nEnd - ChunkObject->m_nStart) > CHUNK_SIZE ) && (FileChunkObject->m_bMulti == TRUE) )
			{
				// create a new chunk
				co = new DCChunkObject();
	
				// split chunk
				if ( (FileChunkObject->m_bMulti == FALSE) || (ChunkObject->m_nStart == 0) || (split == FALSE) )
				{
					co->m_nStart  = ChunkObject->m_nStart;
					co->m_nEnd    = ChunkObject->m_nStart+CHUNK_SIZE;
					ChunkObject->m_nStart += CHUNK_SIZE;
				}
				else
				{
					DPRINTF("CHUNK SET 1: %llu %llu\n",ChunkObject->m_nStart,ChunkObject->m_nEnd);

					co->m_nStart = ChunkObject->m_nStart+((ChunkObject->m_nEnd - ChunkObject->m_nStart)/2);

					if ( (ChunkObject->m_nEnd-co->m_nStart) > CHUNK_SIZE )
					{
						co->m_nEnd = co->m_nStart+CHUNK_SIZE;
						co1 = new DCChunkObject();
						co1->m_nStart = co->m_nEnd;
						co1->m_nEnd   = ChunkObject->m_nEnd;
						FileChunkObject->m_Chunks.Add(co1);
						DPRINTF("CHUNK SET 2: %llu %llu\n",co1->m_nStart,co1->m_nEnd);
					}
					else
					{
						co->m_nEnd = ChunkObject->m_nEnd;
					}

					ChunkObject->m_nEnd = co->m_nStart;

					DPRINTF("CHUNK SET 3: %llu %llu\n",ChunkObject->m_nStart,ChunkObject->m_nEnd);
					DPRINTF("CHUNK SET 4: %llu %llu\n",co->m_nStart,co->m_nEnd);
				}

				co->m_eChunkState = ecsLOCKED;

				FileChunkObject->m_Chunks.Add(co);

				*lstart = co->m_nStart;
				*lend   = co->m_nEnd;

				DPRINTF("NEW CHUNK SPLIT/LOCKED: %llu %llu\n",*lstart,*lend);

				res = TRUE;
			}
			else
			{
				ChunkObject->m_eChunkState = ecsLOCKED;

				*lstart = ChunkObject->m_nStart;
				*lend   = ChunkObject->m_nEnd;

				DPRINTF("NEW CHUNK LOCKED: %llu %llu\n",*lstart,*lend);

				res = TRUE;
			}
		}
	}
	else
	{
		DPRINTF("warning file not found in the chunk list\n");
	}

	return res;
}

/** 0: not found; 1: found and update; 2: no more chunks (file download ok) */
int CDownloadManager::UpdateChunk( CString file, ulonglong lstart, ulonglong lend, ulonglong lcurrent )
{
	int res = 0;
	DCFileChunkObject * FileChunkObject;
	DCChunkObject * ChunkObject, *co;

	DPRINTF("update chunk for '%s'\n",file.Data());

	m_pDownloadQueue->pChunkList->Lock();

	if ( (FileChunkObject = m_pDownloadQueue->GetFileChunkObject(file)) != 0 )
	{
		ChunkObject = 0;

		// search the chunk
		while ( (ChunkObject=FileChunkObject->m_Chunks.Next(ChunkObject)) != 0 )
		{
			if ( (ChunkObject->m_nStart == lstart) && (ChunkObject->m_nEnd == lend) )
			{
				// chunk found
				res = 1;

				if ( ChunkObject->m_eChunkState == ecsFREE )
				{
					DPRINTF("warning wrong chunk state\n");
				}

				ChunkObject->m_eChunkState = ecsFREE;

				if ( lstart != lcurrent )
				{
					// update size done
					FileChunkObject->m_nSizeDone += (lcurrent-lstart);

					DPRINTF("FILESTATE: %llu %llu\n",FileChunkObject->m_nSizeDone,FileChunkObject->m_nSize);

					if ( lcurrent == lend )
					{
						FileChunkObject->m_Chunks.Del(ChunkObject);
						ChunkObject = 0;
					}
					else
					{
						ChunkObject->m_nStart = lcurrent;
					}

					if ( FileChunkObject->m_nSizeDone == FileChunkObject->m_nSize )
					{
						// remove the complete chunk from the list
						m_pDownloadQueue->pChunkList->Del(file);
						res = 2;
						break;
					}
				}

				if ( ChunkObject != 0 )
				{
					co = 0;
					while ( (co=FileChunkObject->m_Chunks.Next(co)) != 0 )
					{
						if ( co == ChunkObject )
							continue;

						if ( ChunkObject->m_nEnd == co->m_nStart )
						{
							if ( co->m_eChunkState == ecsFREE )
							{
								co->m_nStart = ChunkObject->m_nStart;
								FileChunkObject->m_Chunks.Del(ChunkObject);
								ChunkObject = co;
								DPRINTF("CHUNK FIX1: %llu %llu\n",ChunkObject->m_nStart,ChunkObject->m_nEnd);
							}
							break;
						}
					}

					co = 0;
					while ( (co=FileChunkObject->m_Chunks.Next(co)) != 0 )
					{
						if ( co == ChunkObject )
							continue;

						if ( ChunkObject->m_nStart == co->m_nEnd )
						{
							if ( co->m_eChunkState == ecsFREE )
							{
								co->m_nEnd = ChunkObject->m_nEnd;
								FileChunkObject->m_Chunks.Del(ChunkObject);
								ChunkObject = 0;
								DPRINTF("CHUNK FIX2: %llu %llu\n",co->m_nStart,co->m_nEnd);
							}
							break;
						}
					}
				}

				break;
			}
		}
	}
	else
	{
		DPRINTF("warning file not found in the chunk list\n");
	}

	m_pDownloadQueue->pChunkList->UnLock();

	return res;
}

/** */
bool CDownloadManager::GetNewChunkEnd( CString file, ulonglong lstart, ulonglong lend, ulonglong lcurrent, ulonglong * lnstart, ulonglong *lnend )
{
	m_pDownloadQueue->pChunkList->Lock();

	bool res = FALSE;
	DCFileChunkObject * FileChunkObject;
	DCChunkObject * ChunkObject, *co1=0,*co2=0;
	ulonglong chunk_size;

	if ( (FileChunkObject = m_pDownloadQueue->GetFileChunkObject(file)) != 0 )
	{
		ChunkObject = 0;

		while ( (ChunkObject=FileChunkObject->m_Chunks.Next(ChunkObject)) != 0 )
		{
			if ( (ChunkObject->m_nStart == lstart) && (ChunkObject->m_nEnd == lend) )
			{
				// start chunk found
				co1 = ChunkObject;

				if ( co2 )
				{
					// next chunk is set -> stop search
					break;
				}
			}
			else if ( ChunkObject->m_nStart == lend )
			{
				// next chunk found
				if ( ChunkObject->m_eChunkState == ecsLOCKED )
				{
					// the chunk after the original chunk is locked -> stop search
					break;
				}
				else
				{
					// set next chunk
					co2 = ChunkObject;
       	
					if ( co1 )
					{
						// start chunk is set -> stop search
						break;
					}
				}
			}
		}

		// found start and next chunk
		if ( (co1 != 0) && (co2 != 0) )
		{
			DPRINTF("set new chunk end for '%s'\n",file.Data());

			// calc chunk size
			if ( (lend-lcurrent) > CHUNK_SIZE )
				chunk_size = CHUNK_SIZE;
			else
				chunk_size = CHUNK_SIZE-(lend-lcurrent);

			// if chunk > max chunk split chunk to chunk size
			if ( (co2->m_nEnd-co2->m_nStart) > chunk_size )
			{
				// set new chunk end
				co1->m_nEnd   += chunk_size;
				// set correct new chunk start
				co2->m_nStart += chunk_size;
			}
			else
			{
				// set new chunk end
				co1->m_nEnd = co2->m_nEnd;
				// remote the chunk
				FileChunkObject->m_Chunks.Del(co2);
			}

			if ( (lcurrent-lstart) > 0 )
			{
				FileChunkObject->m_nSizeDone += (lcurrent-lstart);
				co1->m_nStart = lcurrent;
			}

			*lnstart = co1->m_nStart;
			*lnend   = co1->m_nEnd;

			DPRINTF("new chunk end set %llu -> %llu [%llu/%llu]\n",lend,*lnend,(*lnend)-(*lnstart),chunk_size);

			res = TRUE;
		}
	}
	else
	{
		DPRINTF("warning file not found in the chunk list\n");
	}

	m_pDownloadQueue->pChunkList->UnLock();

	return res;
}

/** */
bool CDownloadManager::SetFile( CTransfer * Transfer )
{
	bool res;
	DCTransferQueueObject * TransferObject;
	ulonglong lstart,lend;
	int priority = 0;
	res = FALSE;

	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( Transfer->GetDstNick(), Transfer->GetHubName(), Transfer->GetHubHost() )) != 0 )
	{
		if ( TransferObject->pTransferFileList.Count() > 0 )
		{
			DCTransferFileObject * TransferFileObject = 0;

			while( (priority <= MAX_FILE_PRIORITY) && (TransferFileObject == 0) )
			{
			while( TransferObject->pTransferFileList.Next((CObject*&)TransferFileObject) )
			{
				// check the priority
				if ( priority != TransferFileObject->m_nPriority )
					continue;

				if ( TransferFileObject->m_eState == etfsNONE )
				{
					DPRINTF("set file: '%s'\n",TransferFileObject->m_sRemoteFile.Data());

					CString sPath="",sLocalPath="",sFile="";
					CDir dir;

					if ( TransferFileObject->m_eMedium == eltFILE )
					{
						sFile = TransferFileObject->m_sLocalFile;

						DPRINTF("DEBUG: file: '%s'\n",sFile.Data());

						// extract file
						int pos = sFile.FindRev(DIRSEPARATOR);
						if( pos != -1 )
						{
							sPath = sFile.Left(pos);
						}

						DPRINTF("DEBUG: path: '%s'\n", sPath.Data());

						// create the path
						if ( dir.CreatePath(sPath) == FALSE )
						{
							TransferFileObject->m_eState = etfsERROR;
							SendFileInfo( TransferObject, TransferFileObject );
							SendLogInfo( "Create path failed: " + sPath, Transfer );
							DPRINTF("DEBUG: create path failed: '%s'\n", sPath.Data());
						}
						else
						{
							DPRINTF("DOWNLOAD: '%s' %llu '%s'\n",
								TransferFileObject->m_sRemoteFile.Data(),
								TransferFileObject->m_nSize,
								sFile.Data());

								res = TRUE;
						}
					}
					else
					{
						res = TRUE;
					}

					if ( res == TRUE )
					{
						// first we create the hash over the first 1MB of a multi download file in a buffer
						if ( (TransferFileObject->m_bMulti == TRUE) &&
						     (TransferFileObject->m_stHash == ""))
						{
							DPRINTF("create the hash for the file\n");

							Transfer->SetMedium(eltBUFFER);

							lstart = 0;
							lend   = TEST_CHUNK_SIZE;
						}
						else // get the next chunk of the file
						{
							if ( TransferFileObject->m_eMedium == eltCLIENTVERSION )
							{
								DPRINTF("DEBUG: resolve client version ...\n");
								lstart = 0;
								lend   = 0;
							}
							else if ( TransferFileObject->m_sRemoteFile == CString(DC_USER_FILELIST) )
							{
								lstart = 0;
								lend   = 0;
							}
							else if ( GetNextChunk( TransferFileObject->m_sLocalFile, &lstart, &lend ) == FALSE )
							{
								// no more chunks
								DPRINTF("no more chunks ...\n");
								continue;
							}

							Transfer->SetMedium(TransferFileObject->m_eMedium);
						}

						Transfer->SetDone(etsNONE);
						
						CString TTH = TransferFileObject->m_sHash;
						
						if ((TTH.Left(4)).ToUpper() == "TTH:")
						{
							printf("CDownloadManager::SetFile Warning! TTH has prefix \"TTH:\", removing it\n");
							TTH = TTH.Mid(4, TTH.Length() - 4);
						}

						if ( Transfer->StartDownload(TransferFileObject->m_sRemoteFile,lstart,lend,TransferFileObject->m_nSize,lend-lstart,sFile,TTH) == -1 )
						{
							Transfer->Disconnect();
						}
						else
						{
							// mark file as transfered
							TransferFileObject->m_eState = etfsTRANSFER;
						}
					}
					else
					{
						continue;
					}

					SendFileInfo( TransferObject, TransferFileObject );

					break;
				}
				else
				{
//					DPRINTF("state file: '%s' %d\n",TransferFileItem->sRemoteName.Data(),TransferFileItem->eState);
				}
			}

			priority++;
			}
		}
	}

	if ( res == FALSE )
	{
		// set disconnect timeout ... save remote slot ;-)
		if ( Transfer->GetDone() != etsIDLE )
		{
			Transfer->SetStartTime(time(0));
			Transfer->SetDone(etsIDLE);
		}
	}

	return res;
}

/** */
void CDownloadManager::UpdateFileState( CTransfer * Transfer, eTransferFileState eState )
{
	DCTransferQueueObject * TransferObject;
	DCTransferFileObject * TransferFileObject;

	m_pDownloadQueue->pQueue->Lock();

	DPRINTF("updatefile\n");
	if ( (TransferObject = m_pDownloadQueue->GetUserTransferObject( Transfer->GetDstNick(), Transfer->GetHubName(), Transfer->GetHubHost() )) != 0 )
	{
		TransferFileObject = m_pDownloadQueue->GetUserFileObject( Transfer->GetDstNick(), Transfer->GetHubName(), Transfer->GetHubHost(), Transfer->GetDstFilename() );

		if ( TransferFileObject != 0 )
		{
			if ( TransferFileObject->m_eState != etfsTRANSFER )
			{
				DPRINTF("warning, wrong state in updatefile\n");
			}
			// if this a check hash dl ?
			else if ( (Transfer->GetMedium() == eltBUFFER) && (TransferFileObject->m_bMulti == TRUE) )
			{
				DPRINTF("updatefile hash\n");
				if ( eState == etfsNONE )
				{
					if ( (Transfer->GetStartPosition()+Transfer->GetTransfered()) == Transfer->GetEndPosition() )
					{
						if ( CheckHash(Transfer) == TRUE )
						{
							SendLogInfo( "Hash ok '" + TransferFileObject->m_sRemoteFile + "'", Transfer );

							TransferFileObject->m_eState = etfsNONE;

							// reconnect if remote dont support chunk dl
							if ( (Transfer->GetSupport().m_bChunk == FALSE) &&
							     (Transfer->GetSupport().m_bZBlock == FALSE) &&
							     (Transfer->GetSupport().m_bXMLBZList == FALSE) &&
							     (Transfer->GetSupport().m_bADCGet == FALSE) )
							{
								// wan't a reconnect
								TransferObject->bReconnect = TRUE;
							}
						}
						else
						{
							SendLogInfo( "Hash failed '" + TransferFileObject->m_sRemoteFile + "'", Transfer );
							TransferFileObject->m_eState = etfsERROR;
						}
					}
					else
					{
						TransferFileObject->m_eState = eState;
					}
				}
				else
				{
					TransferFileObject->m_eState = eState;
				}

				SendFileInfo( TransferObject, TransferFileObject );
			}
			else
			{
				DPRINTF("updatefile normal\n");

				int state=0;

				TransferFileObject->m_eState = eState;

				if ( Transfer->GetMedium() == eltCLIENTVERSION )
				{
					state = 2;
				}
				// check if mylist.dclist done
				else if ( TransferFileObject->m_sRemoteFile == CString(DC_USER_FILELIST) )
				{
					if ( (Transfer->GetLength() != 0) &&
					     (Transfer->GetLength() == Transfer->GetTransfered()) )
					{
						state = 2;
					}
				}
				else
				{
					state = UpdateChunk( TransferFileObject->m_sLocalFile, Transfer->GetStartPosition(),
						Transfer->GetEndPosition(), Transfer->GetStartPosition()+Transfer->GetTransfered() );
				}

				TransferFileObject->m_nSize = Transfer->GetLength();

				// download finished
				if ( state == 2 )
				{
					if ( Transfer->GetMedium() != eltCLIENTVERSION )
						SendLogInfo( "Transfer done '" + TransferFileObject->m_sRemoteFile + "'", Transfer );
					SendFileInfo( TransferObject, TransferFileObject, TRUE );
					SendTransferInfo( Transfer, FALSE );

					// finished downloads are now a log option
					if ( (TransferFileObject->m_eMedium == eltFILE) &&
					(CConfig::Instance()->GetLogFile() == TRUE) &&
					(CConfig::Instance()->GetLogFinishedDownloads() == TRUE) &&
					(TransferFileObject->m_sRemoteFile != CString(DC_USER_FILELIST)) )
					{
						CLogFile::Write(CConfig::Instance()->GetLogFileName(),eltINFO,"Transfer done '" + TransferFileObject->m_sLocalFile + "'");
					}
					
					// move the file to the finished path
					if ( (TransferFileObject->m_eMedium == eltFILE) && // only files
					     (CConfig::Instance()->GetDownloadFinishedFolder() != "") && // only if dlfolder set
					     (TransferFileObject->m_sRemoteFile != CString(DC_USER_FILELIST)) && // only if this not a filelist
					     (CDir::ConvertSeparators(TransferFileObject->m_sLocalFile).Find(CDir::ConvertSeparators(CConfig::Instance()->GetDownloadFolder())) == 0) ) // only if the download in the download folder
					{
						CString s;

						// close file
						Transfer->CloseFile();
						
						CDir dir(CConfig::Instance()->GetDownloadFinishedFolder());

						if ( dir.CreatePath(TransferFileObject->m_sLocalPath) == TRUE )
						{
							dir.SetPath(CConfig::Instance()->GetDownloadFinishedFolder()+DIRSEPARATOR+TransferFileObject->m_sLocalPath);
#ifdef WIN32
							s  = "move ";
							s += "\"" + TransferFileObject->m_sLocalFile + "\" ";
							s += "\"" + dir.Path() + "\"";

							DPRINTF("move file '%s'\n",s.Data());

							if ( system(s.Data()) != 0 )
							{
								DPRINTF("move failed !\n");
							}
#else
							s = dir.Path() +DIRSEPARATOR+ TransferFileObject->m_sLocalFileName;
							DPRINTF("move file: '%s' ---> '%s'\n",TransferFileObject->m_sLocalFile.Data(), s.Data());
							if ( rename(TransferFileObject->m_sLocalFile.Data(), s.Data()) != 0 )
							{
								if (errno == EXDEV)
								{
									// we have to copy it manually
									if (fastcopy(TransferFileObject->m_sLocalFile.Data(), s.Data()) == FALSE)
									{
										DPRINTF("move failed !\n");
									}
									else
									{
										unlink(TransferFileObject->m_sLocalFile.Data());
									}
								}
							}
#endif
						}
						else
						{
							DPRINTF("move failed (create path)!\n");
						}
					}

					if ( TransferFileObject->m_bMulti == TRUE )
					{
						// remove all files from the wait queue
						RemoveQueueFile(TransferFileObject->m_sLocalFile);
					}
					else
					{
						// remove a userlist only from this user/hub
						RemoveQueueFile( TransferObject->sNick, TransferObject->sHubName, TransferFileObject->m_sRemoteFile );
					}
				}
				else
				{
					SendFileInfo( TransferObject, TransferFileObject );
				}
			}
		}
	}
	else
	{
		DPRINTF("updatefile no GetUserTransferObject\n");
	}

	m_pDownloadQueue->pQueue->UnLock();
}

/** */
void CDownloadManager::UpdateBanList( time_t ttimeout )
{
	DCTransferBanObject * TransferBanObject = 0, * tbo = 0;

	// remove old entrys
	m_pTransferBanList->Lock();

	if ( m_pTransferBanList->Count() > 0 )
	{
		CString s;

		while ( m_pTransferBanList->Next( s, (CObject*&)TransferBanObject ) == 1 )
		{
			if ( (ttimeout-TransferBanObject->m_tTime) > 180 )
			{
				m_pTransferBanList->Del(s);
				TransferBanObject = tbo;
			}
			else
			{
				tbo = TransferBanObject;
			}
		}
	}

	m_pTransferBanList->UnLock();
}

/** */
void CDownloadManager::UpdateTransferList( time_t ttimeout )
{
	long count,rate,tmprate;
	CTransferObject * TransferObject;
	CObject *obj,*oobj;
	CStringList * ratelist=0;
	CMessageFileTransferRate * filetransferrate;
	CList<CTransfer> * transferlist=0;

	m_pTransferList->Lock();

	count = m_pTransferList->Count();

	// check if shutdown ready
	if ( (m_eShutdownState == essSHUTDOWN) && (count == 0) )
	{
		m_eShutdownState = essSHUTDOWNREADY;
	}

	if ( (ttimeout-m_tUpdateTransferTimeout) >= 1 )
	{
		// update slots
		DownloadManagerInfo.slot_max = CConfig::Instance()->GetMaxUpload();
		DownloadManagerInfo.rate_ul_settings = 0;
		DownloadManagerInfo.rate_ul_operator = 0;
		DownloadManagerInfo.rate_ul_user     = 0;
		DownloadManagerInfo.rate_ul_special  = 0;
		DownloadManagerInfo.rate_dl          = 0;

		UpdateBanList( ttimeout );
	}

	if ( count > 0 )
	{
		obj = oobj = 0;

		ratelist = new CStringList();
		transferlist = new CList<CTransfer>();

		while( m_pTransferList->Next( (CObject*&)obj) )
		{
			TransferObject = (CTransferObject*)obj;

			// check if user offline and disconnect
			if ( (ttimeout-TransferObject->m_UserDisconnectTimeout) > 180 )
			{
				eCloseType closetype = CConfig::Instance()->GetHubOfflineTransferClose();

				TransferObject->m_UserDisconnectTimeout = ttimeout;

				if ( CConnectionManager::Instance()->IsHubOnline( TransferObject->m_pTransfer->GetHubName(), TransferObject->m_pTransfer->GetHubHost() ) == ehsONLINE )
				{
					if ( CConnectionManager::Instance()->IsUserOnline( TransferObject->m_pTransfer->GetDstNick(), TransferObject->m_pTransfer->GetHubName(), TransferObject->m_pTransfer->GetHubHost(), 0 ) == FALSE )
					{
						if ( (TransferObject->m_pTransfer->GetSrcDirection()==edDOWNLOAD) &&
					   	( (closetype==ectBOTH) || (closetype==ectDLD) ) )
						{
							TransferObject->m_pTransfer->Disconnect(TRUE);
							SendLogInfo("Disconnect offline user: "+TransferObject->m_pTransfer->GetDstNick()+"@"+TransferObject->m_pTransfer->GetHubName());
						}
						else if ( (TransferObject->m_pTransfer->GetSrcDirection()==edUPLOAD) &&
					        	( (closetype==ectBOTH) || (closetype==ectUPLD) ) )
						{
							TransferObject->m_pTransfer->Disconnect(TRUE);
							SendLogInfo("Disconnect offline user: "+TransferObject->m_pTransfer->GetDstNick()+"@"+TransferObject->m_pTransfer->GetHubName());
						}
					}
				}
				else
				{
					//DPRINTF("Not disconnecting user because hub is offline\n");
				}
			}

			// refresh transfer
			TransferObject->m_pTransfer->Thread(0);

			if ( (ttimeout-m_tUpdateTransferTimeout) >= 1 )
			{
				// done==2 set new file or disconnect
				if ( (TransferObject->m_pTransfer->GetDone() == etsIDLE) &&
				     (m_eShutdownState == essNONE) )
				{
					bool newfile = FALSE;

					// set new file on download mode
					if ( TransferObject->m_pTransfer->GetSrcDirection() == edDOWNLOAD )
					{
						while(1)
						{
							// stop if we cant set a new file
							if ( SetFile(TransferObject->m_pTransfer) == FALSE )
							{
								break;
							}

							// check for clientcheck
							if ( TransferObject->m_pTransfer->GetMedium() == eltCLIENTVERSION )
							{
								// remove clientcheck from queue (we have the client info at connect)
								CDownloadManager::Instance()->UpdateFileState(TransferObject->m_pTransfer,etfsNONE);
							}
							else
							{
								// new file set successful
								newfile = TRUE;
								break;
							}
						}
						
					}

					// no new file or another transfer mode
					if ( newfile == FALSE )
					{
						// check timeout for ul/dl
						if ( TransferObject->m_pTransfer->GetStartTime() != 0 )
						{
							if ( (ttimeout-TransferObject->m_pTransfer->GetStartTime()) >= 60 )
							{
								TransferObject->m_pTransfer->SetStartTime(0);
								TransferObject->m_pTransfer->Disconnect(TRUE);
							}
						}
					}

					SendTransferInfo(TransferObject->m_pTransfer);

					oobj = obj;
				}
				else if ( TransferObject->m_pTransfer->GetDone() == etsREADY )
				{
					if ( TransferObject->m_pTransfer->GetDstDirection() == edDOWNLOAD )
					{
						// TODO: update slot infos etc. in downloadmanager and gui
						if ( TransferObject->m_pTransfer->GetTransferType() == ettSETTINGS )
						{
							if ( DownloadManagerInfo.slot_use_settings > 0 )
								DownloadManagerInfo.slot_use_settings--;
						}
						else if ( TransferObject->m_pTransfer->GetTransferType() == ettOPERATOR )
						{
							if ( DownloadManagerInfo.slot_use_operator > 0 )
								DownloadManagerInfo.slot_use_operator--;
						}
						else if ( TransferObject->m_pTransfer->GetTransferType() == ettUSER )
						{
							if ( DownloadManagerInfo.slot_use_user > 0 )
								DownloadManagerInfo.slot_use_user--;
						}
						else if ( TransferObject->m_pTransfer->GetTransferType() == ettSPECIAL )
						{
							if ( DownloadManagerInfo.slot_use_special > 0 )
								DownloadManagerInfo.slot_use_special--;
						}

						// update used slots
						CConnectionManager::Instance()->SendMyInfoToConnectedServers();
					}

					SendTransferInfo( TransferObject->m_pTransfer, TRUE );

					// update internal transfer state for this user/hubname
					// set to idle (this work only with 1 transfer per user/hubname ...)
					DCTransferQueueObject * TransferQueueObject;

					if ( TransferObject->m_pTransfer->GetSrcDirection() == edDOWNLOAD )
					{
						if ( (TransferQueueObject = m_pDownloadQueue->GetUserTransferObject( TransferObject->m_pTransfer->GetDstNick(), TransferObject->m_pTransfer->GetHubName(), TransferObject->m_pTransfer->GetHubHost() )) != 0 )
						{
							if ( TransferQueueObject->eState == etwsRUN )
							{
								if ( TransferQueueObject->iConnections > 0 )
									TransferQueueObject->iConnections--;
								else
									DPRINTF("WARNING: UpdateTransferList: RUN:0\n");
								if ( TransferQueueObject->iConnections == 0 )
									TransferQueueObject->eState = etwsIDLE;
							}
							else
							{
								DPRINTF("WARNING: UpdateTransferList: wrong queue state\n");
							}

							// wan't reconnect ?
							if ( TransferQueueObject->bReconnect == TRUE )
							{
								TransferQueueObject->tTimeout = 0;
								TransferQueueObject->bReconnect = FALSE;
							}
							else
							{
								TransferQueueObject->tTimeout = ttimeout;
							}

							SendFileInfo( TransferQueueObject );
						}
					}

					m_pTransferList->Del( CString().setNum(TransferObject->m_pTransfer->GetTransferID()) );

					obj = oobj;
				}
				else if ( m_eShutdownState == essNONE )
				{
					if ( TransferObject->m_pTransfer->GetSrcDirection() == edUPLOAD )
					{
						switch(TransferObject->m_pTransfer->GetTransferType())
						{
							case ettSETTINGS:
								DownloadManagerInfo.rate_ul_settings += TransferObject->m_pTransfer->GetTransferrate();
								break;
							case ettOPERATOR:
								DownloadManagerInfo.rate_ul_operator += TransferObject->m_pTransfer->GetTransferrate();
								break;
							case ettUSER:
								DownloadManagerInfo.rate_ul_user += TransferObject->m_pTransfer->GetTransferrate();
								break;
							case ettSPECIAL:
								DownloadManagerInfo.rate_ul_special += TransferObject->m_pTransfer->GetTransferrate();
								break;
							default:
								break;
						}

						SendTransferInfo(TransferObject->m_pTransfer);
					}
					else if ( TransferObject->m_pTransfer->GetSrcDirection() == edDOWNLOAD )
					{
						DownloadManagerInfo.rate_dl += TransferObject->m_pTransfer->GetTransferrate();

						// create the filetransferratelist
						if ( TransferObject->m_pTransfer->GetDstFilename() != CString(DC_USER_FILELIST_HE3) )
						{
							if ( ratelist->Get( TransferObject->m_pTransfer->GetSrcFilename(), (CObject*&)filetransferrate ) == 0 )
							{
								filetransferrate->m_nRate += TransferObject->m_pTransfer->GetTransferrate();
							}
							else
							{
								filetransferrate = new CMessageFileTransferRate();
								filetransferrate->m_sLocalFile = TransferObject->m_pTransfer->GetSrcFilename();
								filetransferrate->m_nRate = TransferObject->m_pTransfer->GetTransferrate();

								ratelist->Add( TransferObject->m_pTransfer->GetSrcFilename(), filetransferrate );
							}
						}

						transferlist->Add(TransferObject->m_pTransfer);
					}
					else
					{
						SendTransferInfo(TransferObject->m_pTransfer);
					}

					oobj = obj;
				}
				else
				{
					oobj = obj;
				}
			}
		}
	}


	// send all download with global filetransferrate
	if ( (ratelist != 0) && (transferlist != 0) )
	{
		CTransfer *tr=0;
		CString s;
		filetransferrate = 0;

		while ( (tr=transferlist->Next(0)) != 0 )
		{
			if ( ratelist->Get( tr->GetSrcFilename(), (CObject*&)filetransferrate ) == 0 )
			{
				LogThread.Lock();

				CMessageDMTransferObject * to = CreateDMTransferObject(tr);

				to->m_nMultiRate = filetransferrate->m_nRate;

				if ( DC_DownloadManagerCallBack(to) == -1 )
					delete to;

				LogThread.UnLock();
			}
			else
			{
				SendTransferInfo(tr);
			}

			transferlist->Remove(tr);
		}

		delete ratelist;
		delete transferlist;
	}

	if ( ((ttimeout-m_tUpdateTransferTimeout) >= 1) && (count > 0) )
	{
		// dynamic upload rate
		if ( (CConfig::Instance()->GetDynamicUploadRate() == TRUE) &&
		     (CConfig::Instance()->GetMaxUploadRate() > 0) &&
		     (DownloadManagerInfo.slot_use_settings > 0) &&
		     (DownloadManagerInfo.slot_max > 0) &&
		     (DownloadManagerInfo.rate_ul_settings<(CConfig::Instance()->GetMaxUploadRate()*DownloadManagerInfo.slot_max)) )
		{
			// calculate max rate for every slot
			rate = CConfig::Instance()->GetMaxUploadRate()*DownloadManagerInfo.slot_max;

			// this is the max rate for every slot but if a slot use less than
			// we give the other slots the "restrate"
			rate /= DownloadManagerInfo.slot_use_settings;

			if ( rate > 0 )
			{
				obj     = 0;
				count   = 0;
				tmprate = 0;

				while( m_pTransferList->Next( (CObject*&)obj) )
				{
					TransferObject = (CTransferObject*)obj;

					if ( (TransferObject->m_pTransfer->GetDone() == etsNONE) &&
					     (TransferObject->m_pTransfer->GetSrcDirection() == edUPLOAD) &&
					     (TransferObject->m_pTransfer->GetTransferType() == ettSETTINGS) )
					{
						// check if the slot use less than 1/2 of the max rate
						if ( (TransferObject->m_pTransfer->GetTransferrate() < (TransferObject->m_pTransfer->GetRate()/2)) && ((ulonglong)rate >= TransferObject->m_pTransfer->GetRate()) )
						{
							// set the new rate for this transfer (3/4)
							TransferObject->m_pTransfer->SetRate((TransferObject->m_pTransfer->GetRate()*3/4));
							// give other slots the restrate
							tmprate += rate-TransferObject->m_pTransfer->GetRate();
						}
						// check if the slot use less than 3/4 but more as 1/2 of the current rate we made no changes
						else if ( (TransferObject->m_pTransfer->GetTransferrate() < (TransferObject->m_pTransfer->GetRate()*3/4)) && ((ulonglong)rate >= TransferObject->m_pTransfer->GetRate()) )
						{
							// give other slots the restrate
							tmprate += rate-TransferObject->m_pTransfer->GetRate();
						}
						// all other slots use more as 3/4 or the max rate for a slot is less than the current rate for the transfer
						else
						{
							count++;
							tmprate += rate;
						}
					}
				}

				// calc new rate
				if ( (tmprate > 0) && (count > 0) )
				{
					tmprate /= count;
					obj     = 0;

					while( m_pTransferList->Next( (CObject*&)obj) )
					{
						TransferObject = (CTransferObject*)obj;

						if ( (TransferObject->m_pTransfer->GetDone() == etsNONE) &&
						     (TransferObject->m_pTransfer->GetSrcDirection() == edUPLOAD) )
						{
							if ( (TransferObject->m_pTransfer->GetTransferrate() >= (TransferObject->m_pTransfer->GetRate()*3/4)) || ((ulonglong)rate < TransferObject->m_pTransfer->GetRate()) )
							{
								TransferObject->m_pTransfer->SetRate(tmprate);
							}
						}
					}
				}
			}
		}

		// send dl manager info
		SendDownloadManagerInfo(&DownloadManagerInfo);
	}

	if ( (ttimeout-m_tUpdateTransferTimeout) >= 1 )
	{
		// send traffic info
		SendTrafficInfo();
	}

	m_pTransferList->UnLock();
}

/** */
void CDownloadManager::UpdateQueueList( time_t ttimeout )
{
	int i;
	CString nick;
	CStringList * StringList = 0, * OldStringList = 0;
	CList<DCHubObject> hublist;
	DCHubObject * HubObject1,* HubObject2;
	CString hubname;

	while( m_pDownloadQueue->pQueue->Next( nick, (CObject *&) StringList) )
	{
		DCTransferQueueObject * TransferObject = 0;

		while( StringList->Next( (CObject *&) TransferObject) )
		{
			// check for empty filelist
			if ( TransferObject->pTransferFileList.Count() == 0 )
			{
				// empty filelist and no connections
				if ( TransferObject->iConnections == 0 )
				{
					// remove from queue
					SendFileInfo( TransferObject, 0, TRUE );
					StringList->Del(TransferObject->sHubName);
				}
				break;
			}
			// handle wait state
			else if ( (TransferObject->eState == etwsWAIT) && (m_eShutdownState == essNONE) )
			{
				// check response timeout
				if ( (ttimeout-TransferObject->tTimeout) >= CConfig::Instance()->GetTransferResponseTimeout() )
				{
					// set timeout
					TransferObject->eState = etwsIDLE;
					TransferObject->tTimeout = ttimeout;
					SendFileInfo( TransferObject );
				}
			}
			// handle none wait state
			else if ( (m_eShutdownState == essNONE) &&
				( (TransferObject->eState == etwsIDLE) ||
				  (TransferObject->eState == etwsHUBOFFLINE) ||
				  (TransferObject->eState == etwsUSEROFFLINE) ||
				  (TransferObject->eState == etwsUSERBUSY) ||
				  (TransferObject->eState == etwsSENDERROR) ) )
			{

				// idle state ... check for timeout to reconnect
				if ( TransferObject->tTimeout == 0 )
				{
					// check for max download rate before new connection start
					if ( !((CConfig::Instance()->GetMaxDownloadRate() != 0)  &&
					     (CConfig::Instance()->GetMaxDownloadRate() < DownloadManagerInfo.rate_dl)) )
					{
						// reset hubname
						hubname = TransferObject->sHubName;

						// update internal hublist
						if ( CConnectionManager::Instance()->IsUserOnline( TransferObject->sNick, "", "", &hublist ) == TRUE )
						{
							DPRINTF("user is online on:\n");

							HubObject1 = 0;
							while( (HubObject1=hublist.Next(HubObject1)) != 0 )
							{
								DPRINTF("'%s' '%s'\n",HubObject1->m_sHubName.Data(),HubObject1->m_sHubHost.Data());

								HubObject2 = 0;
								while( (HubObject2=TransferObject->pHubList.Next(HubObject2)) != 0 )
								{
									if ( HubObject1->m_sHubName == HubObject2->m_sHubName )
										break;
								}

								// add new entry
								if ( HubObject2 == 0 )
								{
									DPRINTF("NEW '%s' '%s'\n",HubObject1->m_sHubName.Data(),HubObject1->m_sHubHost.Data());
									HubObject2 = new DCHubObject();
									HubObject2->m_sHubName = HubObject1->m_sHubName;
									HubObject2->m_sHubHost = HubObject1->m_sHubHost;
									HubObject2->m_bActive = TRUE;

									TransferObject->pHubList.Add(HubObject2);
								}
								else // set entry to connect
								{
									if ( (hubname == "") && (HubObject2->m_bActive == TRUE) )
									{
										DPRINTF("USE '%s'\n",HubObject2->m_sHubName.Data());
										hubname = HubObject2->m_sHubName;
									}
								}
							}

							hublist.Clear();
						}

						// check for available files
						DCTransferFileObject * TransferFileObject = 0;

						while( TransferObject->pTransferFileList.Next( (CObject *&)TransferFileObject) )
						{
							if ( TransferFileObject->m_eState == etfsNONE )
							{
								break;
							}
						}

						if ( TransferFileObject != 0 )
						{
							// send connection request to the hub
							i = CConnectionManager::Instance()->SendConnectionRequest( TransferObject->sNick, hubname, TransferObject->sHubHost );

							switch (i)
							{
								case 0:
									TransferObject->eState = etwsWAIT;
									break;
								case -1:
									TransferObject->eState = etwsUSEROFFLINE;
									break;
								case -2:
								case -3:
									TransferObject->eState = etwsHUBOFFLINE;
									break;
								case -4:
									TransferObject->eState = etwsSENDERROR;
									break;
								default:
									break;
							}
						}
					}

					// update timeout
					TransferObject->tTimeout = ttimeout;
					// send info
					SendFileInfo( TransferObject );
				}
				// check resend timeout
				else if ( (ttimeout-TransferObject->tTimeout) >= CConfig::Instance()->GetTransferResendTimeout() )
				{
					// reset timeout
					TransferObject->tTimeout = 0;
					// send info
					SendFileInfo( TransferObject );
				}
			}
		}

		// remove nick on empty list
		if ( StringList->Count() == 0 )
		{
			m_pDownloadQueue->pQueue->Del(nick);
			StringList = OldStringList;
		}
		else
		{
			OldStringList = StringList;
		}
	}
}

/** */
bool CDownloadManager::InitSearch( time_t /*ttimeout*/ )
{
	CStringList * StringList = 0;

	std::list<CString> *tthlist = new std::list<CString>; // added_by_frg to temp store unique TTHs

	// clear searchlist
	// TODO: we can leave it and search again on the last break, 
	//       but we need a check against the current queue (traffic)
	m_pSearchList->Clear();
	m_pSearchQueryList->Clear();

	if ( !CSearchManager::Instance() )
	{
		return FALSE;
	}

	m_pDownloadQueue->pQueue->Lock();

	while( m_pDownloadQueue->pQueue->Next( (CObject *&) StringList) )
	{
		DCTransferQueueObject * TransferObject = 0;

		while( StringList->Next( (CObject *&) TransferObject) )
		{
			DCTransferFileObject * TransferFileObject = 0;

			while( TransferObject->pTransferFileList.Next( (CObject *&)TransferFileObject) )
			{
				if ( (TransferFileObject->m_bMulti == TRUE) && // only if md enabled
				     (TransferFileObject->m_sHash != "") && // only if we have a TTH (previously also allowed m_stHash != "")
				     (TransferFileObject->m_eMedium == eltFILE) ) // only if medium file
				{
					CMessageSearchResult * msg = new CMessageSearchResult();

					msg->m_nSize    = TransferFileObject->m_nSize;
					msg->m_sFile    = TransferFileObject->m_sRemoteFile;
					msg->m_sNick    = TransferObject->sNick;
					msg->m_sHubName = TransferObject->sHubName;

					CMessageSearchFile * smsg = new CMessageSearchFile();
					
					smsg->m_eType = DC_MESSAGE_SEARCH_FILE;
					
					// printf("Debug TransferFileObject->m_sHash: %s\n", TransferFileObject->m_sHash.Data());
					
					if ( TransferFileObject->m_sHash != "" )
					{
						smsg->m_sString   = "TTH:" + TransferFileObject->m_sHash;
						smsg->m_eFileType = eftHASH;
						msg->m_sHash = smsg->m_sString; // set hash in search result
					}
					else
					{
						CDir dir;
						dir.SetPath(TransferFileObject->m_sRemoteFile);
						
						smsg->m_sString    = dir.DirName();
						smsg->m_eFileType  = eftALL;
						smsg->m_nSize      = TransferFileObject->m_nSize;
						smsg->m_bSizeLimit = TRUE;
						smsg->m_eSizeType  = esstEXACT;
					}
					
					// frg: now we search to see if a file with the same TTH is not already in the list
					bool isduplicate = FALSE;
					
					for (std::list<CString>::const_iterator i = tthlist->begin(); i != tthlist->end(); i++)
					{
						if ( *i == smsg->m_sString )
						{
							isduplicate = TRUE;
							break;
						}
					}
					
					if (isduplicate)
					{
						// a tth duplicate was found - skip this entry
						delete msg;
						delete smsg;
						continue;
					}
					
					// adding this tth to the list
					tthlist->push_back(smsg->m_sString);
					
					if ( CConfig::Instance()->GetMode() == ecmPASSIVE )
					{
						smsg->m_bLocal = TRUE;
						smsg->m_sSource = "";
					}
					else
					{
						smsg->m_bLocal  = FALSE;
						smsg->m_sSource = CConfig::Instance()->GetUDPHostString();
					}
					
					m_pSearchList->Add(msg);
					m_pSearchQueryList->Add(smsg);
				}
			}
		}
	}

	delete tthlist;

	m_pDownloadQueue->pQueue->UnLock();

	if ( m_pSearchList->Count() <= 0 )
	{
		return FALSE;
	}

	if ( CSearchManager::Instance()->StartSearch(esmCONNECTEDALL,estyEXTERNAL,m_pSearchQueryList,0) != eseNONE )
	{
		return FALSE;
	}

	return TRUE;
}

/** thread callbackfunction */
int CDownloadManager::Callback( CObject *, CObject * )
{
	int i;
	time_t ttimeout;

	if ( m_eShutdownState == essSHUTDOWNREADY )
	{
		return 0;
	}

	ttimeout = time(0);

	if ( m_eShutdownState == essNONE )
	{
		// save queue
		i = CConfig::Instance()->GetDownloadQueueTime();

		if ( i > 0 )
		{
			i *= 60;

			if ( (ttimeout-m_tDownloadQueueTimeout) > i )
			{
				DLM_SaveQueue();
				m_tDownloadQueueTimeout = ttimeout;
			}
		}
	}

	// update the transfer list
	UpdateTransferList(ttimeout);

	// update the queue list
	if ( (ttimeout-m_tUpdateTransferTimeout) >= 1 )
	{
		m_pDownloadQueue->pQueue->Lock();

		if ( m_pDownloadQueue->pQueue->Count() > 0 )
		{
			UpdateQueueList(ttimeout);
		}

		m_pDownloadQueue->pQueue->UnLock();
	}

	if ( CConfig::Instance()->GetTransferAutoSearch() &&
	     CSearchManager::Instance() )
	{
		// check if the user break the search or search is finished ;-)
		if ( (m_tHubSearchTimeout == 0) && (CSearchManager::Instance()->SearchType() != estyEXTERNAL) )
		{
			// reset search
			m_tHubSearchTimeout = ttimeout;
		}
		
		// search for new sources
		if ( (m_tHubSearchTimeout != 0) && (ttimeout-m_tHubSearchTimeout) >= CConfig::Instance()->GetAutoSearchInterval() )
		{
			DPRINTF("init search\n");
			
			if ( InitSearch(ttimeout) == FALSE )
			{
				DPRINTF("failed\n");
				m_tHubSearchTimeout = ttimeout;
			}
			else
			{
				m_tHubSearchTimeout = 0;
			}
		}
	}

	// update timeout
	m_tUpdateTransferTimeout = ttimeout;

	return 0;
}

/** callback function */
int CDownloadManager::DM_ListenCallBack( CObject * /*Sender*/, CObject * Object )
{
	bool disc = FALSE;

	int handle = *((int*)Object);

	if ( m_eShutdownState == essNONE )
	{
		CDownloadManager::Instance()->m_pTransferWaitList->Lock();

		if ( CDownloadManager::Instance()->m_pTransferWaitList->Count() == 0 )
		{
			// no waiting transfers, disconnect incoming connection
			disc = TRUE;
		}

		CDownloadManager::Instance()->m_pTransferWaitList->UnLock();
	}
	else
	{
		// dont accept connections on shutdown state
		disc = TRUE;
	}

	if ( disc )
	{
#ifdef WIN32
		_close(handle);
#else
		close(handle);
#endif
		handle = -1;
	}

	if ( handle == -1 )
	{
		return -1;
	}

	CTransferObject * TransferObject = new CTransferObject();
	TransferObject->m_pTransfer = new CTransfer(TRUE);

	TransferObject->m_pTransfer->SetTransferID( CDownloadManager::Instance()->GetNewID() );
	TransferObject->m_pTransfer->SetRate( CConfig::Instance()->GetMaxUploadRate() );
	TransferObject->m_pTransfer->SetCallBackFunction( new CCallback<CDownloadManager>( CDownloadManager::Instance(), &CDownloadManager::DM_TransferCallBack ) );

	if ( TransferObject->m_pTransfer->SetSocket(handle) == 0 )
	{
		CDownloadManager::Instance()->SendLogInfo("Incoming connection from '"+TransferObject->m_pTransfer->GetHost()+"'");

		CDownloadManager::Instance()->m_pTransferList->Lock();
		CDownloadManager::Instance()->m_pTransferList->Add( CString().setNum(TransferObject->m_pTransfer->GetTransferID()), TransferObject );
		CDownloadManager::Instance()->m_pTransferList->UnLock();
	}
	else
	{
		delete TransferObject;
#ifdef WIN32
		_close(handle);
#else
		close(handle);
#endif
	}

	return 0;
}

/** */
int CDownloadManager::DM_TransferCallBack( CObject * transfer, CObject * Object )
{
	CDownloadManager::Instance()->TransferCallBackThread.Lock();

	CDCMessage *DCMsg;
	CByteArray ba;
	CString s,r;
	ulonglong  len;
	CDir dir;
	bool remove,bdirec;
	eShareBufferType stype;
	eDirection direction;

	CTransfer * Transfer = (CTransfer*)transfer;

	DCMsg = (CDCMessage*)Object;

	switch ( DCMsg->m_eType )
	{
		case DC_MESSAGE_MYNICK:
		{
			// check with the nick for download or upload
			direction = CDownloadManager::Instance()->CheckWaitTransfer( Transfer );
			Transfer->SetSrcDirection(direction);

			DCMessageConnectClient mcc;					
			mcc.m_sHubHost = Transfer->GetHost();
			CConnectionManager::Instance()->SetUserTransferInfo( Transfer->GetHubName(), Transfer->GetHubHost(), Transfer->GetDstNick(), &mcc );
			
			break;
		}

		case DC_MESSAGE_GET:
		{
			// remote want a file ...
			CMessageGet * msg = (CMessageGet*)Object;

			if ( (Transfer->GetSrcDirection() == edUPLOAD) &&
			     (Transfer->GetDstDirection() == edDOWNLOAD) )
			{
				// reset done flags
				Transfer->SetDone(etsNONE);

				// send a log info
				CDownloadManager::Instance()->SendLogInfo( "Upload: " + msg->m_sFilename +
					" (" + CString().setNum(msg->m_nPos) + "/" +
					CString().setNum(msg->m_nSize) + ")", Transfer );

				// check for operator/special upload (only the filelist)
				if ( (Transfer->GetTransferType() == ettOPERATOR) &&
				     (msg->m_sFilename != CString(DC_USER_FILELIST_HE3)) &&
				     (msg->m_sFilename != CString(DC_USER_FILELIST_BZ))  &&
				     (msg->m_sFilename != CString(DC_USER_FILELIST_XMLBZ)) )
				{
					// disconnect ...
					CDownloadManager::Instance()->SendLogInfo( "Operator Transfer not for the filelist", Transfer );
					Transfer->SendMaxedOut();
					Transfer->Disconnect(TRUE);
					break;
				}

				stype = esbtNONE;

				if ( msg->m_sFilename == CString(DC_USER_FILELIST_HE3) )
				{
					stype = esbtHE3;
				}
				else if ( msg->m_sFilename == CString(DC_USER_FILELIST_BZ) )
				{
					stype = esbtBZ;
				}
				else if ( msg->m_sFilename == CString(DC_USER_FILELIST_XMLBZ) )
				{
					stype = esbtXMLBZ;
				}

				// check special transfer file size
				if ( stype != esbtNONE )
				{
					// remote want the filelist
					if ( CFileManager::Instance()->GetShareSize() == 0 )
					{
						Transfer->SendError("File Not Available");

						CDownloadManager::Instance()->SendLogInfo( "Upload (sharesize is 0): " + msg->m_sFilename, Transfer );
					}
					else
					{
						if ( CFileManager::Instance()->GetShareBuffer( stype, &ba ) == 0 )
						{
							// for special transfers only filelists <= 16K
							/* if ( (Transfer->GetTransferType() == ettSPECIAL) &&
							     (ba.Size() > (1024*16)) )
							{
								// disconnect ...
								CDownloadManager::Instance()->SendLogInfo( "Special Transfer not for files > 16k", Transfer );
								Transfer->SendMaxedOut();
								Transfer->Disconnect(TRUE);
							}
							else */
							{
								// set send buffer
								Transfer->SetBuffer(&ba);
								// set local transfer medium to buffer
								Transfer->SetMedium(eltBUFFER);
								// start upload
								if ( Transfer->StartUpload( msg->m_sFilename, CFileManager::Instance()->GetShareBufferSize(stype), msg->m_nPos-1, 0, "", msg->m_bUGet, FALSE, "", msg->m_bZLib ) == -1 )
								{
									Transfer->Disconnect(TRUE);
								}
							}
						}
						else
						{
							Transfer->SendError("File Not Available");

							CDownloadManager::Instance()->SendLogInfo( "Upload (no sharebuffer): " + msg->m_sFilename, Transfer );
						}
					}
				}
				else // remote download from the share ...
				{
					// search the wanted file in the share
					if ( (s = CConfig::Instance()->AliasToPath(msg->m_sFilename.Data())) == "" )
					{
						Transfer->SendError("File Not Available");

						CDownloadManager::Instance()->SendLogInfo( "Upload (File Not Available): " + msg->m_sFilename, Transfer );
					}
					else
					{
						// file found, check filesize
						len = dir.getFileSize(s,FALSE);

						if ( msg->m_nPos > len ) // if the wanted filepos > local length send an error
						{
							Transfer->SendError("File allready download.");
						}
						else if ( msg->m_nPos == 0 ) // we begin at 1 (see dc protocoll)
						{
							Transfer->SendError("Wrong file position.");
						}
						else // upload the wanted file ...
						{
							// special transfer only for files <= Configured small file size
							if ( (Transfer->GetTransferType() == ettSPECIAL) &&
							     (len > (CConfig::Instance()->GetSmallFileSize())) )
							{
								// disconnect ...
								CDownloadManager::Instance()->SendLogInfo( "Special Transfer not for files > " + CUtils::GetSizeString( CConfig::Instance()->GetSmallFileSize(), euAUTO ), Transfer );
								Transfer->SendMaxedOut();
								Transfer->Disconnect(TRUE);
							}
							else
							{
								// set local transfer medium to buffer
								Transfer->SetMedium(eltFILE);
								// start upload
								if ( Transfer->StartUpload( msg->m_sFilename, len, msg->m_nPos-1, msg->m_nSize, s, msg->m_bUGet, FALSE, "", msg->m_bZLib ) == -1 )
								{
									Transfer->Disconnect(TRUE);
								}
							}
						}
					}
				}
			}
			else
			{
				CDownloadManager::Instance()->SendLogInfo( "Warning wrong mode", Transfer );
				Transfer->Disconnect(TRUE);
			}

			break;
		}

		case DC_MESSAGE_ADCGET:
		{
			// remote want a file by TTH (or files.xml.bz2)
			CMessageADCGet * msg = (CMessageADCGet*)Object;

			if (msg->m_sType == "list")
			{
				CDownloadManager::Instance()->SendLogInfo( "ADCGet list type not supported", Transfer );
				Transfer->SendError("ADCGet list type not supported");
				Transfer->Disconnect(TRUE);
				break;
			}

			if ( (Transfer->GetSrcDirection() == edUPLOAD) &&
			     (Transfer->GetDstDirection() == edDOWNLOAD) )
			{
				// reset done flags
				Transfer->SetDone(etsNONE);

				// send a log info
				if (msg->m_sType == "file")
				{
					CDownloadManager::Instance()->SendLogInfo( "Upload: " + msg->m_sTTH +
						" (" + CString().setNum(msg->m_nPos) + "/" +
						CString().setNum(msg->m_nSize) + ")", Transfer );
				}

				// check for operator/special upload (only the filelist)
				// only files.xml.bz2 will be requested by ADCGet
				if ( (Transfer->GetTransferType() == ettOPERATOR) && (msg->m_sTTH != CString(DC_USER_FILELIST_XMLBZ)) )
				{
					// disconnect ...
					CDownloadManager::Instance()->SendLogInfo( "Operator Transfer not for the filelist", Transfer );
					Transfer->SendMaxedOut();
					Transfer->Disconnect(TRUE);
					break;
				}

				stype = esbtNONE;

				if ( msg->m_sTTH == CString(DC_USER_FILELIST_XMLBZ) )
				{
					stype = esbtXMLBZ;
				}

				// check special transfer file size
				if ( stype != esbtNONE )
				{
					// remote want the filelist
					if ( CFileManager::Instance()->GetShareSize() == 0 )
					{
						Transfer->SendError("File Not Available");

						CDownloadManager::Instance()->SendLogInfo( "Upload (sharesize is 0): " + msg->m_sTTH, Transfer );
					}
					else
					{
						if ( CFileManager::Instance()->GetShareBuffer( stype, &ba ) == 0 )
						{
							// for special transfers only filelists <= 16K
							/* if ( (Transfer->GetTransferType() == ettSPECIAL) &&
							     (ba.Size() > (1024*16)) )
							{
								// disconnect ...
								CDownloadManager::Instance()->SendLogInfo( "Special Transfer not for files > 16k", Transfer );
								Transfer->SendMaxedOut();
								Transfer->Disconnect(TRUE);
							}
							else */
							{
								// set send buffer
								Transfer->SetBuffer(&ba);
								// set local transfer medium to buffer
								Transfer->SetMedium(eltBUFFER);
								// start upload
								if ( Transfer->StartUpload( msg->m_sTTH, CFileManager::Instance()->GetShareBufferSize(stype), msg->m_nPos, 0, "", FALSE, TRUE, DC_USER_FILELIST_XMLBZ, msg->m_bZlib ) == -1 )
								{
									Transfer->Disconnect(TRUE);
								}
							}
						}
						else
						{
							Transfer->SendError("File Not Available");

							CDownloadManager::Instance()->SendLogInfo( "Upload (no sharebuffer): " + msg->m_sTTH, Transfer );
						}
					}
				}
				else // remote download from the share ...
				{
					CString fileName, justTTH = "";
					
					if ( msg->m_sTTH.Left(4).ToUpper() != "TTH/" )
					{
						DPRINTF("Warning! ADCGet without TTH is undocumented behaviour!\n");
						if ( msg->m_sTTH.Left(1) == "/" )
						{
							DPRINTF("Warning! Removing leading / from filename for ADCGet without TTH\n");
							fileName = msg->m_sTTH.Mid(1, msg->m_sTTH.Length() - 1);
						}
						else
						{
							fileName = msg->m_sTTH;
						}
						
						// unescape spaces in the filename
						fileName = fileName.Replace("\\ ", " ");
					}
					else
					{
						// search the share for the file matching the TTH
						// ADCGet has prefix "tth/"
						justTTH = msg->m_sTTH.Mid(4, msg->m_sTTH.Length()-4);
						CStringList * resultsList = 0;
						CString *ps = 0;
						
						if ( CFileManager::Instance()->IsCreateShareListRunning() == TRUE )
						{
							Transfer->SendError("Filelist refresh in progress, please try again later.");
							CDownloadManager::Instance()->SendLogInfo( "Cannot start upload while filelist refresh in progress", Transfer );
							break;
						}
						else
						{
							// printf("About to search filelist for file matching TTH=%s\n", justTTH.Data());
							resultsList = CFileManager::Instance()->SearchHash( "TTH:" + justTTH );
						}
					
						// this line causes a crash!
						//printf("Found %ld files for TTH %s\n", resultsList->Count(), justTTH.Data());
						
						if ( (resultsList == 0) || (resultsList->Count() == 0) )
						{
							Transfer->SendError("File Not Available");
							CDownloadManager::Instance()->SendLogInfo( "Upload: No file found for TTH:" + justTTH, Transfer );
							break;
						}
						else
						{
							if ( resultsList->Count() > 1 )
							{
								CDownloadManager::Instance()->SendLogInfo( "Upload: Warning: Multiple files match TTH:" + justTTH, Transfer );
							}
						
							resultsList->Next( (CObject*&)ps );
							fileName = CFileManager::Instance()->GetFileName(*ps);
							delete resultsList;
							
							if (msg->m_sType == "file")
							{
								CDownloadManager::Instance()->SendLogInfo("Upload: File for TTH:" + justTTH + " is " + fileName, Transfer );
							}
						}
					}
					
					s = CConfig::Instance()->AliasToPath(fileName.Data());
					if ( s == "" )
					{
						DPRINTF("Error: didn't find path to file from share alias!\n");
						Transfer->SendError("File Not Available");
						CDownloadManager::Instance()->SendLogInfo( "Upload: File Not Available: " + fileName, Transfer );
					}
					else
					{
						// file found, check filesize
						len = dir.getFileSize(s,FALSE);
						if ( msg->m_nPos > len ) // if the wanted filepos > local length send an error
						{
							Transfer->SendError("File already downloaded.");
						}
						else // upload the wanted file ...
						{
							// special transfer only for files <= Configured small file size
							// and for TTH leaves
							if ( (Transfer->GetTransferType() == ettSPECIAL) &&
					     		(len > (CConfig::Instance()->GetSmallFileSize())) &&
							(msg->m_sType == "file") )
							{
								// disconnect ...
								CDownloadManager::Instance()->SendLogInfo( "Special Transfer not for files > " + CUtils::GetSizeString( CConfig::Instance()->GetSmallFileSize(), euAUTO ), Transfer );
								Transfer->SendMaxedOut();
								Transfer->Disconnect(TRUE);
							}
							else
							{
								if (msg->m_sType == "tthl")
								{
									Transfer->SetMedium(eltTTHL);
									CDownloadManager::Instance()->SendLogInfo( "Send TTH leaves for " + fileName, Transfer);
									fileName = "TTHL " + fileName;
									s = "TTHL " + s;
								}
								else // fixme add handling of list
								{
									Transfer->SetMedium(eltFILE);
								}
								
								// start upload
								if ( Transfer->StartUpload( fileName, len, msg->m_nPos, msg->m_nSize, s, FALSE, TRUE, justTTH, msg->m_bZlib ) == -1 )
								{
									Transfer->Disconnect(TRUE);
								}
							}
						}
					}
				}
			}
			else
			{
				CDownloadManager::Instance()->SendLogInfo( "Warning wrong mode", Transfer );
				Transfer->Disconnect(TRUE);
			}

			break;
		}

		case DC_MESSAGE_DIRECTION:
		{
			break;
		}

		case DC_MESSAGE_KEY:
		{
			// direction message ...
			//CMessageDirection * msg = (CMessageDirection*)Object;
			bdirec = FALSE;

			DPRINTF("DIRECTION: level: LOCAL: %d REMOTE: %d\n",Transfer->GetSrcLevel(), Transfer->GetDstLevel() );
			DPRINTF("DIRECTION: direc: LOCAL: %d REMOTE: %d\n",Transfer->GetSrcDirection(), Transfer->GetDstDirection() );

			// equal direction ...
			if ( Transfer->GetDstDirection() == Transfer->GetSrcDirection() )
			{
				// check the level ...
				if ( Transfer->GetDstLevel() < Transfer->GetSrcLevel() )
				{
					// now we must change the dst direction
					if ( Transfer->GetSrcDirection() == edDOWNLOAD )
					{
						Transfer->SetDstDirection(edUPLOAD);
						bdirec = TRUE;
					}
					else if ( Transfer->GetSrcDirection() == edUPLOAD )
					{
						Transfer->SetDstDirection(edDOWNLOAD);
						bdirec = TRUE;
					}
				}
				else if ( Transfer->GetDstLevel() > Transfer->GetSrcLevel() )
				{
					if ( Transfer->GetSrcDirection() == edDOWNLOAD )
					{
						// change direction from download to upload and update the queue state
						if ( CDownloadManager::Instance()->ChangeDirection(Transfer) == TRUE )
						{
							Transfer->SetSrcDirection(edUPLOAD);
							bdirec = TRUE;
						}
					}
					else if ( (Transfer->GetSrcDirection() == edUPLOAD) && (Transfer->GetDstDirection() == edUPLOAD) )
					{
						CDownloadManager::Instance()->SendLogInfo( "Warning: remote want to upload a file !", Transfer );
					}
					else
					{
						CDownloadManager::Instance()->SendLogInfo( "Warning: change direction not supported !", Transfer );
					}
				}
			}
			else
			{
				bdirec = TRUE;
			}

			if ( bdirec == TRUE )
			{
				// now we try to set direction with a slot check (update slots)
				if ( CDownloadManager::Instance()->SetDirection(Transfer) == FALSE )
				{
					bdirec = FALSE;
				}
			}

			if ( bdirec == FALSE )
			{
				CDownloadManager::Instance()->SendLogInfo( "Warning no more free slots", Transfer );
				Transfer->SetDstDirection(edNONE);
				Transfer->SendMaxedOut();
				Transfer->Disconnect(TRUE);
			}
			// check if both modes set correct
			else if ( (Transfer->GetSrcDirection() == edNONE) ||
				  (Transfer->GetDstDirection() == edNONE))
			{
				DPRINTF("DIRECTION: wrong mode ...\n");
				CDownloadManager::Instance()->SendLogInfo( "Warning wrong transfer mode", Transfer );
				Transfer->SetDstDirection(edNONE);
				Transfer->Disconnect(TRUE);
			}
			else
			{
				if ( Transfer->GetSrcDirection() == edDOWNLOAD )
				{
					DPRINTF("DIRECTION: download mode ...\n");

					while (1)
					{
						if ( CDownloadManager::Instance()->SetNextFile(Transfer) == FALSE )
						{
							DPRINTF("DIRECTION: download mode without destination file -> disconnecting!\n");
							Transfer->Disconnect(TRUE);
							break;
						}
						else if ( Transfer->GetMedium() == eltCLIENTVERSION )
						{
							CDownloadManager::Instance()->UpdateFileState(Transfer,etfsNONE);
						}
						else
						{
							break;
						}
					}
				}
				else if ( Transfer->GetSrcDirection() == edUPLOAD )
				{
					DPRINTF("DIRECTION: we are in upload mode ...\n");
				}
			}

			break;
		}

		case DC_MESSAGE_CONNECTION_STATE:
		{
			// connection-state message, generated by the ctransfer-class
			remove = TRUE;
			CMessageConnectionState *msg = (CMessageConnectionState*)Object;

			switch(msg->m_eState)
			{
				case estDISCONNECTED:
					CDownloadManager::Instance()->SendLogInfo( "Disconnected from "+Transfer->GetHost(), Transfer );
					break;

//				case estCONNECTED:
//				case estSOCKETERROR:
//				case estCONNECTIONTIMEOUT:
				default:
					remove = FALSE;
					break;
			}

			if ( (remove == TRUE) && (Transfer->GetSrcDirection() == edDOWNLOAD) )
			{
				if ( Transfer->GetMedium() == eltBUFFER )
					Transfer->SetStartPosition(0);
				CDownloadManager::Instance()->UpdateFileState(Transfer,etfsNONE);
			}

			if ( remove == TRUE )
			{
				CDownloadManager::Instance()->UpdateWaitTransfer(Transfer,TRUE);

				Transfer->SetDone(etsREADY);
			}

			break;
		}

		case DC_MESSAGE_TRANSFER:
		{
			// transfer message, generated by the ctransfer-class
			// CMessageTransfer *msg = (CMessageTransfer*)Object;

			// filetransfer done
			if ( Transfer->GetChunkSize() == Transfer->GetTransfered() )
			{
				// download ready
				if ( Transfer->GetSrcDirection() == edDOWNLOAD )
				{
					bool b = FALSE;

					// set new chunk-end if available
					if ( (Transfer->GetMedium() == eltFILE) &&
				             (Transfer->GetSupport().m_bChunk == FALSE) &&
					     (Transfer->GetSupport().m_bZBlock == FALSE) &&
					     (Transfer->GetSupport().m_bXMLBZList == FALSE) &&
					     (Transfer->GetSupport().m_bADCGet == FALSE) )
					{
						ulonglong lstart,lend;

						// get new chunk end
						if ( CDownloadManager::Instance()->GetNewChunkEnd( Transfer->GetSrcFilename(), Transfer->GetStartPosition(), Transfer->GetEndPosition(), Transfer->GetStartPosition()+Transfer->GetTransfered(), &lstart, &lend ) == TRUE )
						{
							Transfer->SetStartPosition(lstart);
							Transfer->SetEndPosition(lend);
							Transfer->SetChunkSize(lend-lstart);
							Transfer->SetTransfered(0);
							// reset stattime
							Transfer->SetStartTime(time(0));

							b = TRUE;
						}

					}

					// no new chunkend set
					if ( b == FALSE )
					{
						CDownloadManager::Instance()->UpdateFileState(Transfer,etfsNONE);

						// show user file list
						if ( Transfer->GetDstFilename() == CString(DC_USER_FILELIST) )
						{
							CDownloadManager::Instance()->FileListDone(Transfer);
						}

						// check if transfer in idle state and set next file
						if ( Transfer->IsIdle() )
						{
							while ( CDownloadManager::Instance()->SetNextFile(Transfer) == TRUE )
							{
								if ( Transfer->GetMedium() == eltCLIENTVERSION )
								{
									CDownloadManager::Instance()->UpdateFileState(Transfer,etfsNONE);
								}
								else
								{
									break;
								}
							}
						}
					}
				}
				else // upload ready
				{
					// set timeout to save local slots
					if ( Transfer->GetDone() != etsIDLE )
					{
						Transfer->SetStartTime(time(0));
						Transfer->SetDone(etsIDLE);
					}
				}
			}

			break;
		}

		case DC_MESSAGE_FILELENGTH:
		{
			// remote send a filelength
			CMessageFileLength * msg = (CMessageFileLength*)Object;

			if ( msg->m_nFileLength == 0 )
			{
				CDownloadManager::Instance()->SendLogInfo( "Warning: Filelength is NULL.", Transfer );

				if ( Transfer->GetSrcDirection() == edDOWNLOAD )
				{
					CDownloadManager::Instance()->UpdateFileState(Transfer,etfsERROR);
				}

				Transfer->Disconnect(TRUE);
			}
			else
			{
				CDownloadManager::Instance()->SendTransferInfo(Transfer);
			}

			break;
		}

		case DC_MESSAGE_SENDING:
		{
			/* CMessageSending * msg = (CMessageSending*)Object;
			
			if ( msg->m_nLength == 0)
			{
				CDownloadManager::Instance()->SendLogInfo( "Warning: $Sending length is 0", Transfer );
			}
			
			CDownloadManager::Instance()->SendTransferInfo(Transfer); */
			
			break;
		}

		case DC_MESSAGE_ADCSND:
		{
			/* CMessageADCSnd * msg = (CMessageADCSnd*)Object;
			
			if ( msg->m_nSize == 0 )
			{
				CDownloadManager::Instance()->SendLogInfo( "Warning: $ADCSnd size is 0", Transfer );
			}
			
			CDownloadManager::Instance()->SendTransferInfo(Transfer); */
			
			break;
		}

		case DC_MESSAGE_MAXEDOUT:
		{
			// no free slots on the remote side
			CDownloadManager::Instance()->SendLogInfo( "Busy", Transfer );
			Transfer->Disconnect(TRUE);
			break;
		}

		case DC_MESSAGE_GETLISTLEN:
		{
			// remote want the listlen
			if ( Transfer->GetSupport().m_bXMLBZList == TRUE )
				Transfer->SendListLen(CFileManager::Instance()->GetShareBufferSize(esbtXMLBZ));
			else if ( Transfer->GetSupport().m_bBZList == FALSE )
				Transfer->SendListLen(CFileManager::Instance()->GetShareBufferSize(esbtHE3));
			else
				Transfer->SendListLen(CFileManager::Instance()->GetShareBufferSize(esbtBZ));
			break;
		}

		case DC_MESSAGE_ERROR:
		{
			// remote send a error message
			CMessageError * msg = (CMessageError*)Object;

			CDownloadManager::Instance()->SendLogInfo( "Error: " + msg->m_sError, Transfer );

			if ( Transfer->GetSrcDirection() == edDOWNLOAD )
			{
				CDownloadManager::Instance()->UpdateFileState(Transfer,etfsERROR);

				while ( CDownloadManager::Instance()->SetNextFile(Transfer) == TRUE )
				{
					if ( Transfer->GetMedium() == eltCLIENTVERSION )
					{
						CDownloadManager::Instance()->UpdateFileState(Transfer,etfsNONE);
					}
					else
					{
						break;
					}
				}
			}
			break;
		}

		case DC_MESSAGE_LOCK:
		{
			CMessageLock * msg = (CMessageLock*)Object;
			CString s;

			CConnectionManager::Instance()->SetUserTransferInfo( Transfer->GetHubName(), Transfer->GetHubHost(), Transfer->GetDstNick(), msg );
			
			switch(msg->m_eClientVersion)
			{
				case eucvDCPP:
					s = "DC++";
					break;
				case eucvDCGUI:
					s = "DCGUI-QT";
					break;
				case eucvMICRODC:
					s = "microdc";
					break;
				case eucvSHAKESPEER:
					s = "ShakesPeer";
					break;
				default:
					s = "Unknown";
					break;
			}

			if ( msg->m_sVersionString != "" )
			{
				s += " (" + msg->m_sVersionString + ")";
			}

			CDownloadManager::Instance()->SendLogInfo( "Client: " + s, Transfer );

			break;
		}

		case DC_MESSAGE_SUPPORTS:
		{
			CMessageSupports * msg = (CMessageSupports*)Object;
			
			CConnectionManager::Instance()->SetUserTransferInfo( Transfer->GetHubName(), Transfer->GetHubHost(), Transfer->GetDstNick(), msg );
			
			break;
		}
						
		default:
		{
			DPRINTF("dctransfer unknown message: %d\n",DCMsg->m_eType);
			break;
		}
	}

	delete Object;

	CDownloadManager::Instance()->TransferCallBackThread.UnLock();

	return 0;
}
