/** -*- C++ -*-
    @file cache/apt/packages.h
    @author Peter Rockai <me@mornfall.net>
*/
#include <wibble/range.h>

#include <ept/cache/component.h>
#include <ept/cache/package.h>
#include <ept/cache/version.h>
#include <ept/cache/relation.h>
#include <ept/cache/apt/depcache.h>

#include <sigc++/sigc++.h>
#include <ept/cache/apt/index.h>
#include <ept/cache/apt/state.h>
#include <ept/cache/apt/records.h>

#include <string>
#include <vector>

#ifndef EPT_CACHE_APTCACKAGES_H
#define EPT_CACHE_APTCACKAGES_H

class pkgPolicy;

namespace ept {
namespace t {
namespace cache {
namespace apt {

template< typename C >
struct Aggregator {
    typedef typename C::PackagePointer PackagePointer;
    typedef typename C::Index Index;
    typedef typename C::State State;
    typedef typename C::Records Records;
    typedef typename C::Vocabulary Vocabulary;
    typedef typename C::TagMap TagMap;
    typedef typename C::Package Package;

    void reload();

    const TagMap &tagMap() const { return delayedCreate( m_tagMap ); }
    const Records &records() const { return delayedCreate( m_records ); }
    const Index &index() const { return delayedCreate( m_index ); }
    const State &state() const { return delayedCreate( m_state ); }
    const Vocabulary &vocabulary() const { return delayedCreate( m_vocabulary ); }

    TagMap &tagMap() { return delayedCreate( m_tagMap ); }
    Records &records() { return delayedCreate( m_records ); }
    Index &index() { return delayedCreate( m_index ); }
    State &state() { return delayedCreate( m_state ); }
    Vocabulary &vocabulary() { return delayedCreate( m_vocabulary ); }

    template< typename T >
    T &delayedCreate( T *&var ) {
        if ( !var ) {
            /* mornfall | well, new T(...) calls T's ctor
               mornfall | T's ctor calls something that uses Aggregator
               mornfall | and references var
               mornfall | oops
               mornfall | var is uninitialized
               mornfall | because in var = new T(), new T() needs to be performed before
                          assigning anything to var
                 enrico | aah, ok, because the assignment to var hasn't taken place yet
                 enrico | smart
                 enrico | it could use a comment, though :-) */
            var = (T *) new char[ sizeof( T ) ];
            var = new (var) T( *this );
        }
        return *var;
    }

    template< typename T >
    const T &delayedCreate( T *&var ) const {
        return const_cast< Aggregator * >( this )->delayedCreate( var );
    }

    void invalidate();

    Aggregator();

    sigc::signal< void > worldChanged;
    sigc::signal< void, Package > packageChanged;

protected:
    mutable Index *m_index;
    mutable State *m_state;
    mutable Records *m_records;
    mutable Vocabulary *m_vocabulary;
    mutable TagMap *m_tagMap;

};

}

}
}
}

#endif
