/** -*- C++ -*-
	@file manager.h
	@author Peter Rockai <me@mornfall.net>
*/

#include <apt-pkg/packagemanager.h>
#include <apt-pkg/acquire.h>

#include <ept/error.h>
#include <ept/progresscallback.h>
#include <ept/cache/cache.h>

#ifndef EPT_PACKAGEMANAGER_H
#define EPT_PACKAGEMANAGER_H

namespace ept {
namespace t {

/**
   @brief Sync cache changes to system, update cache (apt-get update).

   This class encapsulates all actions libapt-front could be doing and that
   will have an impact on the state of the system. It can update the cache
   or commit the changes done in cache to the system (downloading required
   archives in the process). The ::update interface should work with all
   available cache components (including debtags), so all you need to do
   is call manager.update(). You can construct and throw away
   PackageManager objects at will.
*/
template< typename C >
struct Manager {
    struct OperationCancelled : wibble::exception::Generic {};
    Manager( typename C::Aggregator &c );
    ~Manager();
    /**
       @brief Commit changes done in the associated cache to system.

       You can only call this when Cache is in consistent state, exception
       will be thrown otherwise. commit will use the associated
       ProgressCallback object to report download and install progress.
    */
    void commit();

    void download();

    /**
       @brief Update the cache ala apt-get update.

       This will do a normal cache update, possibly clearing the changed
       cache states. It will use the associated ProgressCallback object
       to report download and parsing progress.

       Returns true if the operation was successfull (false if not)
    */
    void update();

    /// set a ProgressCallback to be used to report progress in update and commit
    void setProgressCallback( ProgressCallback *progress );

    void setUpdateInterval( unsigned interval ) {
        m_updateInterval = interval;
    }

protected:
    void getArchives();
    void fetch();
    void createFetcher();

    typename C::Aggregator &m_packages;
    ProgressCallback *m_progress;
    unsigned m_updateInterval;
    pkgPackageManager *m_pm;
    pkgAcquire *m_fetcher;
};

}
}

#endif
