/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  




-----------------------------------------------------------------------
FILENAME:               eval.h
LIBRARY:                libgpiv
EXTERNAL FUNCTIONS:
SOURCE:                 eval_par.c
                        gpiv_piv_default_parameters
                        gpiv_piv_parameters_logic
                        gpiv_piv_read_parameters
                        gpiv_piv_check_parameters_read
                        gpiv_piv_test_parameters
                        gpiv_piv_print_parameters
                        gpiv_piv_fprint_parameters
                        gpiv_piv_cp_parameters
                        gpiv_piv_fread_hdf_parameters
                        gpiv_piv_fwrite_hdf_parameters

SOURCE:                 eval.c
                        gpiv_piv_count_pivdata_fromimage
                        gpiv_piv_select_int_point
                        gpiv_piv_interr_reg
                        gpiv_piv_isizadapt
                        gpiv_eval_write_deformed_image
                        gpiv_piv_bounds_cov
                        gpiv_piv_weight_kernel_lin
                        gpiv_fread_fftw_wisdom
                        gpiv_fwrite_fftw_wisdom

LAST MODIFICATION DATE: $Id: eval.h,v 1.20 2006/01/31 15:16:37 gerber Exp $
---------------------------------------------------------------------
 Global variables and parameters of piv_eval */


#ifndef __LIBGPIV_EVAL_H__
#define __LIBGPIV_EVAL_H__

#define GPIV_ZEROPAD_FACT 2     /* magnification factor of zero-padded int. area */
#define GPIV_DIFF_ISI  0        /* difference between interrogation sizes 
                                   of first and second image if zero 
                                   offset has been used */
#define GPIV_CUM_RESIDU_MIN 0.25 /* minimum cumulative residu for convergence */
#define GPIV_MAX_EVAL_SWEEP 10   /* maximum number of evaluation sweeps,
                                   starting from zero */
#define GPIV_SNR_DISABLE 88.0    /* snr value for manually disabled estimator */
#define GPIV_DEFORMED_IMG_NAME "gpiv_defimg" /* Deformed image to be stored in TMP_DIR */ 

/*
 * Default values of GpivEvalPar
 */
#define GPIV_EVALPAR_DEFAULT__AUTOKEY 1
#define GPIV_EVALPAR_DEFAULT__INT_GEO 0
#define GPIV_EVALPAR_DEFAULT__COL_START 0
#define GPIV_EVALPAR_DEFAULT__COL_END 255
#define GPIV_EVALPAR_DEFAULT__ROW_START 0
#define GPIV_EVALPAR_DEFAULT__ROW_END 255
#define GPIV_EVALPAR_DEFAULT__INT_LINE_ROW 64
#define GPIV_EVALPAR_DEFAULT__INT_LINE_ROW_START 0
#define GPIV_EVALPAR_DEFAULT__INT_LINE_ROW_END 127
#define GPIV_EVALPAR_DEFAULT__INT_LINE_COL 64
#define GPIV_EVALPAR_DEFAULT__INT_LINE_COL_START 0
#define GPIV_EVALPAR_DEFAULT__INT_LINE_COL_END 127
#define GPIV_EVALPAR_DEFAULT__INT_POINT_COL 64
#define GPIV_EVALPAR_DEFAULT__INT_POINT_ROW 64
#define GPIV_EVALPAR_DEFAULT__INT_SIZE_1 64
#define GPIV_EVALPAR_DEFAULT__INT_SIZE_2 32
#define GPIV_EVALPAR_DEFAULT__INT_SHIFT 16
#define GPIV_EVALPAR_DEFAULT__PRE_SHIFT_COL 0
#define GPIV_EVALPAR_DEFAULT__PRE_SHIFT_ROW 0
#define GPIV_EVALPAR_DEFAULT__AD_INT 0
#define GPIV_EVALPAR_DEFAULT__IFIT 1
#define GPIV_EVALPAR_DEFAULT__PEAK 1
#define GPIV_EVALPAR_DEFAULT__INT_SCHEME 4
#define GPIV_EVALPAR_DEFAULT__OLD_PIV 0

/*-----------------------------------------------------------------------------
 * Global variables and parameters of Piv evaluation
 */



/*
 * Type of sub-pixel estimation
 */
enum GpivIFit {
    GPIV_NONE = 0,
    GPIV_GAUSS = 1,
    GPIV_POWER = 2,
    GPIV_GRAVITY = 3
} IFIT;



/*
 * Interrogation geometry: within an area of interest, along a line or at a 
 * single point
 */
enum GpivIntGeo {
    GPIV_AOI = 0,
    GPIV_LINE_C = 1,
    GPIV_LINE_R = 2,
    GPIV_POINT = 3
} GPIV_INT_GEO;



/*
 * Interrogation scheme
 */
enum GpivIntScheme {
    GPIV_NO_CORR,
    GPIV_LK_WEIGHT,              /* Linear kernel weighting */
    GPIV_ZERO_OFF_FORWARD,       /* classical zero ofsetting */
    GPIV_ZERO_OFF_CENTRAL,       /* zero ofsetting using central differential scheme */
    GPIV_IMG_DEFORM              /* image deformation */
} GPIV_INT_SCHEME;


/*
 * PIV evaluation parameters needed for interrogation
 */
typedef struct __GpivEvalPar GpivEvalPar; 
struct __GpivEvalPar {
    int ad_int;                 /* adaptive interrogation area for iterative steps */
    int autokey;                /* automatic setting of related parameters */
    int col_start;              /* first column of image to use for interrogation */
    int col_end;                /* last column in image to use for interrogation */
    int ifit;                   /* Type of sub-pixel estimation */
    int int_geo;                /* Interrogation along a line or within an area of interest */
    int int_line_col;           /* interrogation along a column */
    int int_line_col_start;     /* first column of image to use for interrogation along a line */
    int int_line_col_end;       /* last column of image to use for interrogation along a line */
    int int_line_row;           /* interrogation along a row */
    int int_line_row_start;     /* first row of image to use for interrogation along a column */
    int int_line_row_end;       /* first row of image to use for interrogation along a column */
    int int_point_col;          /* column of a interrogation area for single point */
    int int_point_row;          /* row of a interrogation area for single point */
    int int_size_1;             /* first interrogation area size; to be ended with */
    int int_size_2;             /* second interrogation area size; to start with */
    int int_shift;              /* interrogation area shift */
    int int_deform;             /* deform image for improved accuracy */    
    int old_piv;                /* use old estimators from a previous stored data set */
    int peak;                   /* peak number of correlation function to be used as estimator */
    int pre_shift_col;          /* general pre-shift in horizontal direction (columns) */
    int pre_shift_row;          /* general pre-shift in vertical direction (rows) */
    int print_cov;              /* print covariance data */
    int print_par;              /* print parameters */
    int print_piv;              /* print piv data */
    int row_end;                /* first row of image to use for interrogation */
    int row_start;              /* last row in image to use for interrogation */
/*     int img_intar;  */             /* Way to correct on biasing effects with value of GpivImgIntAr */
    int spline_degree;          /* degree of b-splines to be used for image data interpolation */
    enum GpivIntScheme int_scheme;              /* Way to correct on biasing effects with value of GpivIntScheme */

/*
 * all parameter logics to check whether they have been read or defined
 */
    gboolean ad_int_logic;
    gboolean autokey_logic;
    gboolean col_start_logic;
    gboolean col_end_logic;
    gboolean ifit_logic;
    gboolean int_size; 
    gboolean int_size_1_logic;
    gboolean int_size_2_logic;
    gboolean int_shift_logic;
    gboolean int_deform_logic;
    gboolean int_line_col_logic;
    gboolean int_line_col_start_logic;
    gboolean int_line_col_end_logic;
    gboolean int_line_row_logic;
    gboolean int_line_row_start_logic;
    gboolean int_line_row_end_logic;
    gboolean int_point_col_logic;
    gboolean int_point_row_logic;
    gboolean int_geo_logic;
    gboolean old_piv_logic;
    gboolean pre_shift_col_logic;
    gboolean peak_logic;
    gboolean pre_shift_row_logic; 
    gboolean print_par_logic;
    gboolean print_piv_logic;
    gboolean print_cov_logic;
    gboolean row_start_logic;
    gboolean row_end_logic;
/*     gboolean img_intar_logic; */
    gboolean spline_degree_logic;
    gboolean int_scheme_logic;
};


typedef struct __GpivCovariance Covariance;
struct __GpivCovariance {	/* Data structure of covariance data */
    float **z;			/* 2-dim array containing covariance data */
    float min;			/* minimum covariance value */
    float max;			/* maximum covariance value */

    long top_x;                 /* x-position of maximum at pixel level */
    long top_y;                 /* y-position of maximum at pixel level */
    float subtop_x;             /* x-position of maximum at sub-pixel level */
    float subtop_y;             /* y-position of maximum at sub-pixel level */
    float snr;                  /* Signal to Noice defined by quotient of first 
                                   and second highest peaks */

    int z_rnl;			/* lowest negative row index */
    int z_rnh;			/* highest negative row index */
    int z_rpl;			/* lowest positive row index */
    int z_rph;			/* highest positive row index */

    int z_cnl;			/* lowest negative column index */
    int z_cnh;			/* highest negative column index */
    int z_cpl;			/* lowest positive column index */
    int z_cph;			/* highest positive index */

    int z_rl;			/* general lowest row index */
    int z_rh;			/* general highest row index */
    int z_cl;			/* general lowest column index */
    int z_ch;			/* general highest column index */
};



/*-----------------------------------------------------------------------------
  Function prototypes */

void
gpiv_piv_default_parameters(GpivEvalPar *piv_eval_par_default,
                            gboolean flag
                            );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Sets default parameter values
 *
 * INPUTS:
 *     force:                 flag to enforce parameters set to defaults
 *
 * OUTPUTS:
 *     piv_eval_par_default:   structure of piv evaluation parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_piv_parameters_logic(GpivEvalPar * piv_eval_par,
                           gboolean flag
                           );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Set flags for all piv_eval_par_logic
 *
 * INPUTS:
 *      flag:           true (1) or false (0)
 *
 * OUTPUTS:
 *      piv_eval_par:   PIV evaluation parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_piv_read_parameters(FILE * fp_par, 
                         GpivEvalPar * piv_eval_par, 
                         int print_par
                         );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Read all parameters for PIV evaluation
 *
 * INPUTS:
 *      fp_par:        file pointer to parameter file
 *      print_par:     flag to print parametrs to stdout
 *
 * OUTPUTS:
 *     piv_eval_par
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_piv_check_parameters_read(GpivEvalPar *piv_eval_par,
                               GpivEvalPar piv_eval_par_default
                               );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Check out if all parameters have been read
 *
 * INPUTS:
 *      piv_eval_par_default:
 *
 * OUTPUTS:
 *      piv_eval_par:   PIV evaluation parameters
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_piv_test_parameter(GpivImagePar * image_par, 
                        GpivEvalPar * piv_eval_par
                        );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Testing parameters on valid values and initializing derived variables 
 *
 * INPUTS:
 *      image_par:      image parameters
 *      piv_eval_par:   PIV evaluation parameters
 *      fname_logic:    flag if file name has been used for loading image 
 *                      data
 *
 * OUTPUTS:
 *     piv_eval_par:   
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_piv_print_parameters(GpivEvalPar piv_eval_par
                          );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Prints parameters to stdout
 *
 * INPUTS:
 *      piv_eval_par:   PIV evaluation parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/



void 
gpiv_piv_fprint_parameters(FILE * fp_par_out,
                           GpivEvalPar piv_eval_par
                           );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Prints parameters to fp_par_out
 *
 * INPUTS:
 *      fp_par_out:     
 *      piv_eval_par:   PIV evaluation parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/



void 
gpiv_piv_cp_parameters(GpivEvalPar piv_eval_par_src, 
                       GpivEvalPar * piv_eval_par_dest,
                       gboolean force,
                       gboolean print_par);
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Copies piv parameters from src to dest
 *
 * INPUTS:
 *      piv_eval_par_src:       source piv parameters
 *      force:                  flag to force the copying, even if destination
 *                              already exists
 *      print_par:              verbose output. Not implemented, yet.
 *
 * OUTPUTS:
 *      piv_eval_par_dest:     destination piv parameters
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/



char *
gpiv_piv_fread_hdf5_parameters(char *fname,
                               GpivEvalPar * piv_eval_par 
                               );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *   Reads piv parameters from hdf5 data file
 *
 * INPUTS:
 *      fname:
 *
 * OUTPUTS:
 *      piv_eval_par
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_piv_fwrite_hdf5_parameters(char *fname,
                                GpivEvalPar * piv_eval_par 
                                );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes piv parameters from hdf5 data file
 *
 * INPUTS:
 *      fname:          
 *
 * OUTPUTS:
 *      piv_eval_par:   
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_piv_count_pivdata_fromimage(GpivPivData * out_data,
                                 GpivImagePar image_par,
                                 GpivEvalPar piv_eval_par
                                 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculates the number of interrogation areas from the image sizes, 
 *     pre-shift and area of interest
 *
 * INPUTS:
 *     image_par:      structure of image parameters
 *     piv_eval_par:   structure of piv evaluation parameters
 *
 * OUTPUTS:
 *     out_data:       output piv data from image analysis
 *
 *  RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_piv_select_int_point(GpivPivData * out_data,
                          GpivImagePar image_par,
                          GpivEvalPar piv_eval_par
                          );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Old interface; substituted by gpiv_piv_gridgen
 *     Calculates the positions of interrogation areas to be analysed
 *
 * INPUTS:
 *     image_par:      structure of image parameters
 *     piv_eval_par:   structure of piv evaluation parameters
 *
 * OUTPUTS:
 *     out_data:       output piv data from image analysis
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_piv_interr_reg(int index_y,
                    int index_x,
                    guint16 **img_1,
                    guint16 **img_2,
                    float **int_area_1,
                    float **int_area_2,
                    Covariance * cov,
                    Covariance * w_k,
                    GpivPivData * piv_data,
                    int sweep,
                    int last_sweep,
                    GpivImagePar image_par,
                    GpivEvalPar piv_eval_par
                    );
/**----------------------------------------------------------------------------
 * DESCRIPTION:
 *     gpiv_piv_interr_reg: 
 *     Interrogates an individual region 
 *
 * INPUTS:
 *     @index_y:        y-index of interrogation area position  
 *     @index_x:        x-index of interrogation area position
 *     @img_1:          (binary) image data of first piv image
 *     @img_2:          (binary) image data of second piv image
 *     @int_area_1:     first interrogation area
 *     @int_area_2:     second interrogation area
 *     @cov:            structure containing covariance data
 *     @w_k:            structure containing weight kerel data
 *     @sweep:	       sweep number of iterative process 
 *     @last_sweep:     flag for last sweep
 *     @image_par:      structure of image parameters
 *     @piv_eval_par:   structure of piv evaluation parameters
 *
 * OUTPUTS:
 *     @piv_data:       output piv data from image analysis
 *
 * RETURNS:
 *     %char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void
gpiv_piv_isizadapt(GpivEvalPar piv_eval_par_src,
                   GpivEvalPar *piv_eval_par_dest,
                   gboolean *isiz_last
                   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Adjusts interrogation area sizes. For each interrogation sweep,
 *     (dest) int_size2 is halved, until it reaches (src)
 *     int_size_1. Then, isiz_last is set TRUE, which will avoid
 *     changing the interrogation sizes in next calls.
 *
 * INPUTS:
 *     piv_eval_par_src:       original parameters
 *
 * OUTPUTS:
 *     piv_eval_par_dest:      actual parameters, to be modified during sweeps
 *     isiz_last:              flag for last interrogation sweep
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_eval_write_deformed_image(guint16 **img1, 
                               guint16 **img2, 
                               GpivImagePar image_par
                               );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Stores deformed image to file system with pre defined name to TMPDIR
 *     and prints message to stdout
 *
 * INPUTS:
 *     img1:                   first image of PIV image pair
 *     img2:                   second image of PIV image pair
 *     image_par:              image parameters to be stored in header
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/




void 
gpiv_piv_bounds_cov(Covariance * cov,
                    int int_size_0,
                    GpivImagePar image_par
                    );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculate boundaries of covariance data array
 *
 * INPUTS:
 *     int_size_0:	 zero-padded interrogation size
 *     image_par:        structure of image parameters
 *
 * OUTPUTS:
 *     cov:	       Structure containing covariance data and boundaries
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/



void 
gpiv_piv_weight_kernel_1(Covariance * w_k
                         );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Sets weight values for covariance data to one
 *
 * INPUTS:
 *
 * OUTPUTS:
 *      w_k:	        zero-padded interrogation size
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/


void 
gpiv_piv_weight_kernel_lin(Covariance * w_k,
                           int int_size_0
                           );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculate linear weight kernel values for covariance data
 *
 * INPUTS:
 *
 * OUTPUTS:
 *	w_k:            Structure containing weighting data
 *      int_size_0:     zero-padded interrogation size
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/



void
gpiv_fread_fftw_wisdom(int dir
                       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads fftw wisdoms from file and stores into a string
 *
 * INPUTS:
 *     dir:            direction of fft; forward (+1) or inverse (-1)
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     fftw_wisdom
 *---------------------------------------------------------------------------*/



void 
gpiv_fwrite_fftw_wisdom(int dir
                        );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Writes fftw wisdoms to a file
 *
 * INPUTS:
 *      dir:            direction of fft; forward (+1) or inverse (-1)
 *
 * OUTPUTS:
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/




#endif /* __LIBGPIV_EVAL_H__ */
