
/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2004, 2005 
   Gerber van der Graaf <gerber_graaf@users.sourceforge.net>
   Julio Soria <julio.soria@eng.monash.edu.au>

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



------------------------------------------------------------------------------
FILENAME:                dac_par.c
LIBRARY:                 libgpiv
EXTERNAL FUNCTIONS:      
                         gpiv_dac_default_parameters
                         gpiv_dac_parameters_logic
                         gpiv_dac_read_parameters
                         gpiv_dac_check_parameters_read
		         gpiv_dac_test_parameter
                         gpiv_dac_print_parameters
                         gpiv_dac_fprint_parameters
                         gpiv_dac_fread_hdf5_parameters

LAST MODIFICATION DATE:  $Id: dac_par.c,v 1.2 2006/01/31 13:30:13 gerber Exp $
--------------------------------------------------------------------------- */


#ifndef DISABLE_DAC
#include <gpiv.h>



void
gpiv_dac_default_parameters(GpivDacPar *dac_par_default,
                            gboolean force
                            )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Sets default GpivDacPar parameter values
 *
 * INPUTS:
 *
 * OUTPUTS:
 *     piv_dac_par_default:   structure of data aquisition parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    if (!dac_par_default->ttime__cam_acq_period_logic || force)
        dac_par_default->ttime.cam_acq_period = 
            (RTIME) (GPIV_DACPAR_DEFAULT__TTIME_CAP * GPIV_MILI2NANO);
    if (!dac_par_default->ttime__laser_trig_pw_logic || force)
        dac_par_default->ttime.laser_trig_pw = 
            (RTIME) (GPIV_DACPAR_DEFAULT__TTIME_LPW * GPIV_MILI2NANO);
    if (!dac_par_default->ttime__time2laser_logic || force)
        dac_par_default->ttime.time2laser = 
            (RTIME) (GPIV_DACPAR_DEFAULT__TTIME_T2L * GPIV_MILI2NANO);
    if (!dac_par_default->ttime__dt_logic || force)
        dac_par_default->ttime.dt = 
            (RTIME) (GPIV_DACPAR_DEFAULT__TTIME_DT * GPIV_MILI2NANO);
    if (!dac_par_default->ttime__mode_logic || force)
        dac_par_default->ttime.mode = GPIV_DACPAR_DEFAULT__TTIME_MODE;
    if (!dac_par_default->ttime__cycles_logic || force)
        dac_par_default->ttime.cycles = GPIV_DACPAR_DEFAULT__TTIME_CYCLES;
    if (!dac_par_default->ttime__increment_logic || force)
        dac_par_default->ttime.increment = 
            (RTIME) (GPIV_DACPAR_DEFAULT__TTIME_INCR_DT * GPIV_MILI2NANO);
    
    
    if (!dac_par_default->fname_logic || force)
        snprintf(dac_par_default->fname, GPIV_MAX_CHARS, "%s", 
                 GPIV_DACPAR_DEFAULT__FNAME);


    gpiv_dac_parameters_logic(dac_par_default, TRUE);
}



void 
gpiv_dac_read_parameters(FILE * fp_par, 
                         GpivDacPar * dac_par, 
                         int print_par
                         )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Read all GpivDacPar parameters
 *
 * INPUTS:
 *      fp_par:         file pointer to parameter file
 *      print_par:      flag to print parametrs to stdout
 *
 * OUTPUTS:
 *     piv_dac_par:     Data Aquisition parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char line[GPIV_MAX_CHARS], par_name[GPIV_MAX_CHARS];
    gfloat val;

    while (fgets(line, GPIV_MAX_CHARS, fp_par) != NULL) {
        if (line[0] != '#' && line[0] != '\n' && line[0] != ' ' 
            && line[0] != '\t') {
            sscanf(line,"%s", par_name);

/*
 * Parameters used to acquire images
 */
            if (dac_par->ttime__cam_acq_period_logic == FALSE) {
                dac_par->ttime__cam_acq_period_logic =
                    gpiv_scan_fph(GPIV_DAC_PAR_KEY, line, par_name, 
                                  "Ttime_Cap:", &val, print_par, 1);
                dac_par->ttime.cam_acq_period = (RTIME) GPIV_NANO2MILI * val;
            }

            if (dac_par->ttime__laser_trig_pw_logic == FALSE) {
                dac_par->ttime__laser_trig_pw_logic =
                    gpiv_scan_fph(GPIV_DAC_PAR_KEY, line, par_name, 
                                  "Ttime_Lpw:", &val, print_par, 1);
                dac_par->ttime.laser_trig_pw = (RTIME) GPIV_NANO2MILI * val;
            }

            if (dac_par->ttime__time2laser_logic == FALSE) {
                dac_par->ttime__time2laser_logic =
                    gpiv_scan_fph(GPIV_DAC_PAR_KEY, line, par_name, 
                                  "Ttime_T2l:", &val, print_par, 1);
                dac_par->ttime.time2laser = (RTIME) GPIV_NANO2MILI * val;
            }
            
            if (dac_par->ttime__dt_logic == FALSE) {
                dac_par->ttime__dt_logic =
                    gpiv_scan_fph(GPIV_DAC_PAR_KEY, line, par_name, 
                                  "Ttime_Dt:", &val, print_par, 1);
                dac_par->ttime.dt = (RTIME) GPIV_NANO2MILI * val;
            }
            
            if (dac_par->ttime__mode_logic == FALSE) {
                dac_par->ttime__mode_logic =
                    gpiv_scan_iph(GPIV_DAC_PAR_KEY, line, par_name, 
                                  "Ttime_Mode:", &dac_par->ttime.mode, 
                                  print_par, 1);
            }
            
            if (dac_par->ttime__cycles_logic == FALSE) {
                dac_par->ttime__cycles_logic =
                    gpiv_scan_iph(GPIV_DAC_PAR_KEY, line, par_name, 
                                  "Ttime_Cycles:", &dac_par->ttime.cycles, 
                                  print_par, 1);
            }
            
            if (dac_par->ttime__increment_logic == FALSE) {
                dac_par->ttime__increment_logic =
                    gpiv_scan_fph(GPIV_DAC_PAR_KEY, line, par_name, 
                                  "Ttime_Incr_Dt:", &val, print_par, 1);
                dac_par->ttime.increment = (RTIME) GPIV_NANO2MILI * val;
	    }
	    
            if (dac_par->fname_logic == FALSE) {
                dac_par->fname_logic =
                    gpiv_scan_sph(GPIV_DAC_PAR_KEY, line, par_name, 
                                  "Fname:", &dac_par->fname, print_par, 1);
	    }
	    
	    
	}
    }
    
}



char *
gpiv_dac_check_parameters_read(GpivDacPar *dac_par,
                               GpivDacPar dac_par_default
                               )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Check out if all GpivDacPar parameters have been read
 *
 * INPUTS:
 *     piv_dac_par_default:
 *
 * OUTPUTS:
 *     piv_dac_par:     Data Aquisition parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    gchar *err_msg = NULL;

    if (dac_par->ttime__cam_acq_period_logic == FALSE) {
        err_msg = "Using default:";
        dac_par->ttime.cam_acq_period = dac_par_default.ttime.cam_acq_period;
        dac_par->ttime__cam_acq_period_logic = 
            dac_par_default.ttime__cam_acq_period_logic;
        gpiv_warning("%s\nDAC.Ttime_Cap: %lld", err_msg, 
                     dac_par_default.ttime.cam_acq_period /* * GPIV_NANO2MILI */);
    }
 
    if (dac_par->ttime__laser_trig_pw_logic == FALSE) {
        err_msg = "Using default:";
       dac_par->ttime.laser_trig_pw = dac_par_default.ttime.laser_trig_pw;
       dac_par->ttime__laser_trig_pw_logic = 
           dac_par_default.ttime__laser_trig_pw_logic;
        gpiv_warning("%s\nDAC.Ttime_Lpw: %lld", err_msg, 
                     dac_par_default.ttime.laser_trig_pw/*  * GPIV_NANO2MILI */);
    }

    if (dac_par->ttime__time2laser_logic == FALSE) {
        err_msg = "Using default:";
        dac_par->ttime.time2laser = dac_par_default.ttime.time2laser;
        dac_par->ttime__time2laser_logic = 
            dac_par_default.ttime__time2laser_logic;
        gpiv_warning("%s\nDAC.Ttime_T2l: %lld", err_msg, 
                     dac_par_default.ttime.time2laser/*  * GPIV_NANO2MILI */);
    }

    if (dac_par->ttime__dt_logic == FALSE) {
        err_msg = "Using default:";
        dac_par->ttime.dt = dac_par_default.ttime.dt;
        dac_par->ttime__dt_logic = dac_par_default.ttime__dt_logic;
        gpiv_warning("%s\nDAC.Ttime_Dt: %lld", err_msg, 
                     dac_par_default.ttime.dt/*  * GPIV_NANO2MILI */);
    }

    if (dac_par->ttime__mode_logic == FALSE) {
        err_msg = "Using default:";
        dac_par->ttime.mode = dac_par_default.ttime.mode;
        dac_par->ttime__mode_logic =
            dac_par_default.ttime__mode_logic;
        gpiv_warning("%s\nDAC.Ttime_Mode: %d", err_msg, 
                     dac_par_default.ttime.mode);
    }

    if (dac_par->ttime__cycles_logic == FALSE) {
        err_msg = "Using default:";
        dac_par->ttime.cycles= dac_par_default.ttime.cycles;
        dac_par->ttime__cycles_logic = 
            dac_par_default.ttime__cycles_logic;
        gpiv_warning("%s\nDAC.Ttime_Cycles: %d", err_msg, 
                     dac_par_default.ttime.cycles);
    }

    if (dac_par->ttime__increment_logic == FALSE) {
        err_msg = "Using default:";
        dac_par->ttime.increment = dac_par_default.ttime.increment;
        dac_par->ttime__increment_logic = 
            dac_par_default.ttime__increment_logic;
        gpiv_warning("%s\nDAC.Ttime_Incr: %lld", err_msg, 
                     dac_par_default.ttime.increment/*  * GPIV_NANO2MILI */);
    }

    if (dac_par->fname_logic == FALSE) {
        err_msg = "Using default:";
        snprintf(dac_par->fname, GPIV_MAX_CHARS, "%s", 
             dac_par_default.fname);
        dac_par->fname_logic = dac_par_default.fname_logic;
        gpiv_warning("%s\nDAC.Fname: %s", err_msg, 
                     dac_par_default.fname);
    }


    return (err_msg);
}



void
gpiv_dac_parameters_logic(GpivDacPar * dac_par,
                          gboolean flag
                          )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Sets flags for _logic variables of GpivDacPar 
 *
 * INPUTS:
 *     dac_par:         Data Aquisition parameters
 *
 * OUTPUTS:
 *     dac_par:         Data Aquisition parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    dac_par->ttime__cam_acq_period_logic = flag;
    dac_par->ttime__laser_trig_pw_logic = flag;
    dac_par->ttime__time2laser_logic = flag;
    dac_par->ttime__dt_logic = flag;
    dac_par->ttime__mode_logic = flag;
    dac_par->ttime__cycles_logic = flag;
    dac_par->ttime__increment_logic = flag;

    dac_par->fname_logic = flag;

}



char *
gpiv_dac_test_parameter(GpivDacPar * dac_par
                        )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Testing GpivDacPar parameters on valid values and initializing derived 
 *     variables 
 *
 * INPUTS:
 *     dac_par:      
 *
 * OUTPUTS:
 *     dac_par:         Data Aquisition parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 * --------------------------------------------------------------------------*/
{
    gchar * err_msg = NULL;
    if (dac_par->ttime.cam_acq_period < GPIV_DACPAR_TTIME_CAP_MIN
        ||dac_par->ttime.cam_acq_period > GPIV_DACPAR_TTIME_CAP_MAX) {
        err_msg = "gpiv_dac_test_parameter:ac_par->ttime.cam_acq_period out of range";
        return (err_msg);
    }

    if (dac_par->ttime.laser_trig_pw < GPIV_DACPAR_TTIME_LPW_MIN
        ||dac_par->ttime.laser_trig_pw > GPIV_DACPAR_TTIME_LPW_MAX) {
        err_msg = "gpiv_dac_test_parameter:ac_par->ttime.laser_trig_pw out of range";
        return (err_msg);
    }

    if (dac_par->ttime.time2laser < GPIV_DACPAR_TTIME_T2L_MIN
        ||dac_par->ttime.time2laser > GPIV_DACPAR_TTIME_T2L_MAX) {
        err_msg = "gpiv_dac_test_parameter:ac_par->ttime.time2laser out of range";
        return (err_msg);
    }

    if (dac_par->ttime.dt < GPIV_DACPAR_TTIME_DT_MIN
        ||dac_par->ttime.dt > GPIV_DACPAR_TTIME_DT_MAX) {
        err_msg = "gpiv_dac_test_parameter:ac_par->ttime.dt out of range";
        return (err_msg);
    }
    if (dac_par->ttime.mode < GPIV_DACPAR_TTIME_MODE_MIN
        ||dac_par->ttime.mode > GPIV_DACPAR_TTIME_MODE_MAX) {
        err_msg = "gpiv_dac_test_parameter:ac_par->ttime.mode out of range";
        return (err_msg);
    }

    if (dac_par->ttime.cycles < GPIV_DACPAR_TTIME_CAP_MIN
        ||dac_par->ttime.cycles > GPIV_DACPAR_TTIME_CAP_MAX) {
        err_msg = "gpiv_dac_test_parameter:ac_par->ttime.cycles out of range";
        return (err_msg);
    }

    if (dac_par->ttime.increment < GPIV_DACPAR_TTIME_INCR_DT_MIN
        ||dac_par->ttime.increment > GPIV_DACPAR_TTIME_INCR_DT_MAX) {
        err_msg = "gpiv_dac_test_parameter:ac_par->ttime.increment out of range";
        return (err_msg);
    }

/*
 * Adjusting parameter settings for selected timing mode
 */
/* BUGFIX: OLD
/*     if ((dac_par.ttime.mode != GPIV_TIMER_MODE__PERIODIC)  */
/*         && (dac_par.ttime.mode != GPIV_TIMER_MODE__DURATION)  */
/*         && (dac_par.ttime.mode != GPIV_TIMER_MODE__TRIGGER_IRQ)  */
/*         && (dac_par.ttime.mode != GPIV_TIMER_MODE__DOUBLE)) { */


/* BUGFIX: NEW */
    if ((dac_par->ttime.mode == GPIV_TIMER_MODE__ONE_SHOT_IRQ)
        || (dac_par->ttime.mode == GPIV_TIMER_MODE__INCREMENT)) {
        dac_par->ttime.cam_acq_period  = (dac_par->ttime.dt) * 2;
    }
        
/* BUGFIX: OLD
/*     if ((dac_par.ttime.mode != GPIV_TIMER_MODE__DURATION)  */
/*         && (dac_par.ttime.mode != GPIV_TIMER_MODE__ONE_SHOT_IRQ )  */
/*         && (dac_par.ttime.mode != GPIV_TIMER_MODE__TRIGGER_IRQ)  */
/*         && (dac_par.ttime.mode != GPIV_TIMER_MODE__INCREMENT)) { */

/* BUGFIX: NEW */
    if ((dac_par->ttime.mode == GPIV_TIMER_MODE__PERIODIC)
        || (dac_par->ttime.mode == GPIV_TIMER_MODE__DOUBLE)) {
        dac_par->ttime.cycles = 1;
    }
    
    if (dac_par->ttime.mode != GPIV_TIMER_MODE__INCREMENT) {
        dac_par->ttime.increment = 1;
    }


    return (err_msg);
}



void 
gpiv_dac_print_parameters(GpivDacPar dac_par
                          )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Prints GpivDacPar parameters to stdout
 *
 * INPUTS:
 *      dac_par:        Data Aquisition parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/
{
    gfloat val;

    if (dac_par.ttime__cam_acq_period_logic) {
        val = dac_par.ttime.cam_acq_period * GPIV_MILI2NANO;
	printf("%s%s %f\n", GPIV_DAC_PAR_KEY, "Ttime_Cap:", 
		val);
    }

    if (dac_par.ttime__laser_trig_pw_logic) {
        val = dac_par.ttime.laser_trig_pw * GPIV_MILI2NANO;
	printf("%s%s %f\n", GPIV_DAC_PAR_KEY, "Ttime_Lpw:", 
		val);
    }

    if (dac_par.ttime__time2laser_logic) {
        val = dac_par.ttime.time2laser * GPIV_MILI2NANO;
	printf("%s%s %f\n", GPIV_DAC_PAR_KEY, "Ttime_T2l:", 
		val);
    }

    if (dac_par.ttime__dt_logic) {
        val = dac_par.ttime.dt * GPIV_MILI2NANO;
	printf("%s%s %f\n", GPIV_DAC_PAR_KEY, "Ttime_Dt:", 
		val);
    }

    if (dac_par.ttime__mode_logic) {
	printf("%s%s %d\n", GPIV_DAC_PAR_KEY, "Ttime_Mode:", 
		dac_par.ttime.mode);
    }

    if (dac_par.ttime__cycles_logic) {
	printf("%s%s %d\n", GPIV_DAC_PAR_KEY, "Ttime_Cycles:", 
		dac_par.ttime.cycles);
    }

    if (dac_par.ttime__increment_logic) {
        val = dac_par.ttime.increment * GPIV_MILI2NANO;
 	printf("%s%s %f\n", GPIV_DAC_PAR_KEY, "Ttime_Incr_Dt:", 
               val);
    }

    if (dac_par.fname_logic) {
	printf("%s%s %s\n", GPIV_DAC_PAR_KEY, "Fname:", 
		dac_par.fname);
    }

}



void 
gpiv_dac_fprint_parameters(FILE * fp_par_out, 
                           GpivDacPar dac_par
                           )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Prints GpivDacPar parameters to fp_par_out
 *
 * INPUTS:
 *      fp_par_out:     
 *      dac_par:        Data Aquisition parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/
{
    gfloat val;

    if (dac_par.ttime__cam_acq_period_logic) {
        val = dac_par.ttime.cam_acq_period * GPIV_MILI2NANO;
	fprintf(fp_par_out, "%s%s %f\n", GPIV_DAC_PAR_KEY, "Ttime_Cap:", 
		val);
    }

    if (dac_par.ttime__laser_trig_pw_logic) {
        val = dac_par.ttime.laser_trig_pw * GPIV_MILI2NANO;
	fprintf(fp_par_out, "%s%s %f\n", GPIV_DAC_PAR_KEY, "Ttime_Lpw:", 
		val);
    }

    if (dac_par.ttime__time2laser_logic) {
        val = dac_par.ttime.time2laser * GPIV_MILI2NANO;
	fprintf(fp_par_out, "%s%s %f\n", GPIV_DAC_PAR_KEY, "Ttime_T2l:", 
		val);
    }

    if (dac_par.ttime__dt_logic) {
        val = dac_par.ttime.dt * GPIV_MILI2NANO;
	fprintf(fp_par_out, "%s%s %f\n", GPIV_DAC_PAR_KEY, "Ttime_Dt:", 
		val);
    }

    if (dac_par.ttime__mode_logic) {
	fprintf(fp_par_out, "%s%s %d\n", GPIV_DAC_PAR_KEY, "Ttime_Mode:", 
		dac_par.ttime.mode);
    }

    if (dac_par.ttime__cycles_logic) {
	fprintf(fp_par_out, "%s%s %d\n", GPIV_DAC_PAR_KEY, "Ttime_Cycles:", 
		dac_par.ttime.cycles);
    }

    if (dac_par.ttime__increment_logic) {
        val = dac_par.ttime.increment * GPIV_MILI2NANO;
	fprintf(fp_par_out, "%s%s %f\n", GPIV_DAC_PAR_KEY, "Ttime_Incr_Dt:", 
		val);
    }

    if (dac_par.fname_logic) {
	fprintf(fp_par_out, "%s%s %s\n", GPIV_DAC_PAR_KEY, "Fname:", 
		dac_par.fname);
    }

}



#endif /* DISABLE_DAC */
