/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  pixbuf-loader.cc - GdkPixbufLoader C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "pixbuf-loader.h"
#include "private/pixbuf-loader_p.h"
#include "pixbuf.h"
#include "pixbuf-animation.h"
#include "../glib/error.h"

using namespace Inti;

/*  Gdk::PixbufLoader
 */

Gdk::PixbufLoader::PixbufLoader(GdkPixbufLoader *pixbuf_loader, bool reference)
: G::Object((GObject*)pixbuf_loader, reference)
{
}
	
Gdk::PixbufLoader::PixbufLoader() 
: G::Object((GObject*)PixbufLoaderClass::create())
{
}

Gdk::PixbufLoader::PixbufLoader(const char *image_type, G::Error *error)
: G::Object((GObject*)gdk_pixbuf_loader_new_with_type(image_type, *error))
{
}

Gdk::PixbufLoader::~PixbufLoader()
{
}
	
GdkPixbufLoaderClass*
Gdk::PixbufLoader::gdk_pixbuf_loader_class() const 
{ 
	return get_class<GdkPixbufLoaderClass>(); 
}
	
Gdk::PixbufLoader::operator GdkPixbufLoader* () const 
{ 
	return this ? gdk_pixbuf_loader() : 0; 
}
	
Pointer<Gdk::Pixbuf>
Gdk::PixbufLoader::get_pixbuf() const
{
	GdkPixbuf *pixbuf = gdk_pixbuf_loader_get_pixbuf(gdk_pixbuf_loader());
	return pixbuf ? G::Object::wrap_new<Pixbuf>(pixbuf) : 0;
}

Pointer<Gdk::PixbufAnimation>
Gdk::PixbufLoader::get_animation() const
{
	GdkPixbufAnimation *animation = gdk_pixbuf_loader_get_animation(gdk_pixbuf_loader());
	return animation ? G::Object::wrap_new<PixbufAnimation>(animation) : 0;
}

Pointer<Gdk::PixbufFormat>
Gdk::PixbufLoader::get_format() const
{
	return new PixbufFormat(gdk_pixbuf_loader_get_format(gdk_pixbuf_loader()));
}

void
Gdk::PixbufLoader::set_size(int width, int height)
{
	gdk_pixbuf_loader_set_size(gdk_pixbuf_loader(), width, height);
}

bool
Gdk::PixbufLoader::write(const unsigned char *buffer, size_t count, G::Error *error)
{
	return gdk_pixbuf_loader_write(gdk_pixbuf_loader(), buffer, count, *error);
}

bool 
Gdk::PixbufLoader::close(G::Error *error)
{
	return gdk_pixbuf_loader_close(gdk_pixbuf_loader(), *error);
}

/*  Gdk::PixbufLoaderClass
 */

void
Gdk::PixbufLoaderClass::init(GdkPixbufLoaderClass *g_class)
{
	G::ObjectClass::init((GObjectClass*)g_class);
	g_class->size_prepared = &size_prepared_proxy;
	g_class->area_prepared = &area_prepared_proxy;
	g_class->area_updated = &area_updated_proxy;
	g_class->closed = &closed_proxy;
}

GType
Gdk::PixbufLoaderClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GDK_TYPE_PIXBUF_LOADER, (GClassInitFunc)init);
	}
	return type;
}

void*
Gdk::PixbufLoaderClass::create()
{
	return g_object_new(get_type(), 0);
}

void 
Gdk::PixbufLoaderClass::size_prepared_proxy(GdkPixbufLoader *loader, int width, int height)
{
	PixbufLoader *tmp_loader = G::Object::pointer<PixbufLoader>(loader);
	if (tmp_loader)
		tmp_loader->on_size_prepared(width, height);
	else
	{
		GdkPixbufLoaderClass *tmp_class = GDK_PIXBUF_LOADER_GET_CLASS(loader);
		GdkPixbufLoaderClass *g_class = G::TypeInstance::class_peek_parent<GdkPixbufLoaderClass>(tmp_class);
		if (g_class->size_prepared)
			g_class->size_prepared(loader, width, height);
	}
}

void
Gdk::PixbufLoaderClass::area_prepared_proxy(GdkPixbufLoader *loader)
{
	PixbufLoader *tmp_loader = G::Object::pointer<PixbufLoader>(loader);
	if (tmp_loader)
		tmp_loader->on_area_prepared();
	else
	{
		GdkPixbufLoaderClass *tmp_class = GDK_PIXBUF_LOADER_GET_CLASS(loader);
		GdkPixbufLoaderClass *g_class = G::TypeInstance::class_peek_parent<GdkPixbufLoaderClass>(tmp_class);
		if (g_class->area_prepared)
			g_class->area_prepared(loader);
	}
}

void
Gdk::PixbufLoaderClass::area_updated_proxy(GdkPixbufLoader *loader, int x, int y, int width, int height)
{
	PixbufLoader *tmp_loader = G::Object::pointer<PixbufLoader>(loader);
	if (tmp_loader)
		tmp_loader->on_area_updated(x, y, width, height);
	else
	{
		GdkPixbufLoaderClass *tmp_class = GDK_PIXBUF_LOADER_GET_CLASS(loader);
		GdkPixbufLoaderClass *g_class = G::TypeInstance::class_peek_parent<GdkPixbufLoaderClass>(tmp_class);
		if (g_class->area_updated)
			g_class->area_updated(loader, x, y, width, height);
	}
}

void
Gdk::PixbufLoaderClass::closed_proxy(GdkPixbufLoader *loader)
{
	PixbufLoader *tmp_loader = G::Object::pointer<PixbufLoader>(loader);
	if (tmp_loader)
		tmp_loader->on_closed();
	else
	{
		GdkPixbufLoaderClass *tmp_class = GDK_PIXBUF_LOADER_GET_CLASS(loader);
		GdkPixbufLoaderClass *g_class = G::TypeInstance::class_peek_parent<GdkPixbufLoaderClass>(tmp_class);
		if (g_class->closed)
			g_class->closed(loader);
	}
}

/*  Signal handlers
 */

void
Gdk::PixbufLoader::on_size_prepared(int width, int height)
{
	GdkPixbufLoaderClass *g_class = class_peek_parent<GdkPixbufLoaderClass>(gdk_pixbuf_loader_class());
	if (g_class->size_prepared)
		g_class->size_prepared(gdk_pixbuf_loader(), width, height);
}

void
Gdk::PixbufLoader::on_area_prepared()
{
	GdkPixbufLoaderClass *g_class = class_peek_parent<GdkPixbufLoaderClass>(gdk_pixbuf_loader_class());
	if (g_class->area_prepared)
		g_class->area_prepared(gdk_pixbuf_loader());
}

void
Gdk::PixbufLoader::on_area_updated(int x, int y, int width, int height)
{
	GdkPixbufLoaderClass *g_class = class_peek_parent<GdkPixbufLoaderClass>(gdk_pixbuf_loader_class());
	if (g_class->area_updated)
		g_class->area_updated(gdk_pixbuf_loader(), x, y, width, height);
}

void
Gdk::PixbufLoader::on_closed()
{
	GdkPixbufLoaderClass *g_class = class_peek_parent<GdkPixbufLoaderClass>(gdk_pixbuf_loader_class());
	if (g_class->closed)
		g_class->closed(gdk_pixbuf_loader());
}

/*  Signals
 */

const Gdk::PixbufLoader::SizePreparedSignalType Gdk::PixbufLoader::size_prepared_signal("size_prepared");

const Gdk::PixbufLoader::AreaPreparedSignalType Gdk::PixbufLoader::area_prepared_signal("area_prepared");

const Gdk::PixbufLoader::AreaUpdatedSignalType Gdk::PixbufLoader::area_updated_signal("area_updated");

const Gdk::PixbufLoader::ClosedSignalType Gdk::PixbufLoader::closed_signal("closed");

