/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  calendar.cc - GtkCalendar C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "calendar.h"
#include "private/calendar_p.h"

using namespace Inti;

/*  Gtk::Calendar
 */

Gtk::Calendar::Calendar(GtkCalendar *calendar, bool reference)
: Widget((GtkWidget*)calendar, reference)
{
}

Gtk::Calendar::Calendar()
: Widget((GtkWidget*)CalendarClass::create())
{
}

Gtk::Calendar::Calendar(CalendarDisplayOptionsField flags)
: Widget((GtkWidget*)CalendarClass::create())
{
	display_options(flags);
}

Gtk::Calendar::~Calendar()
{
}

GtkCalendarClass*
Gtk::Calendar::gtk_calendar_class() const 
{ 
	return get_class<GtkCalendarClass>(); 
}
	
Gtk::Calendar::operator GtkCalendar* () const 
{ 
	return this ? gtk_calendar() : 0; 
}
	
int
Gtk::Calendar::year() const
{
	return gtk_calendar()->year;
}

int
Gtk::Calendar::month() const
{
	return gtk_calendar()->month;
}

int
Gtk::Calendar::day() const
{
	return gtk_calendar()->selected_day;
}

Gtk::CalendarDisplayOptionsField
Gtk::Calendar::display_flags() const
{
	return (CalendarDisplayOptionsField)gtk_calendar()->display_flags;
}

int
Gtk::Calendar::num_marked_dates() const
{
	return gtk_calendar()->num_marked_dates;
}

std::vector<bool>
Gtk::Calendar::marked_date() const
{
	std::vector<bool> dates(31);

	for (int i = 0; i < 31; i++)
	{
		dates[i] = gtk_calendar()->marked_date[i];
	}

	return dates;
}

void
Gtk::Calendar::get_date(int *year, int *month, int *day) const
{
	gtk_calendar_get_date(gtk_calendar(), (unsigned int*)year, (unsigned int*)month, (unsigned int*)day);
}

bool
Gtk::Calendar::select_month(int month, int year)
{
	return gtk_calendar_select_month(gtk_calendar(), month, year);
}

void
Gtk::Calendar::select_day(int day)
{
	gtk_calendar_select_day(gtk_calendar(), day);
}

bool
Gtk::Calendar::mark_day(int day)
{
	return gtk_calendar_mark_day(gtk_calendar(), day);
}

bool
Gtk::Calendar::unmark_day(int day)
{
	return gtk_calendar_unmark_day(gtk_calendar(), day);
}

void 
Gtk::Calendar::clear_marks()
{
	gtk_calendar_clear_marks(gtk_calendar());
}

void
Gtk::Calendar::display_options(CalendarDisplayOptionsField flags)
{
	gtk_calendar_display_options(gtk_calendar(), (GtkCalendarDisplayOptions)flags);
}

void 
Gtk::Calendar::freeze()
{
	gtk_calendar_freeze(gtk_calendar());
}

void 
Gtk::Calendar::thaw()
{
	gtk_calendar_thaw(gtk_calendar());
}

/*  Gtk::CalendarClass
 */

void
Gtk::CalendarClass::init(GtkCalendarClass *g_class)
{
	WidgetClass::init((GtkWidgetClass*)g_class);
	g_class->month_changed = &month_changed_proxy;
	g_class->day_selected = &day_selected_proxy;
	g_class->day_selected_double_click = &day_selected_double_click_proxy;
	g_class->prev_month = &prev_month_proxy;
	g_class->next_month = &next_month_proxy;
	g_class->prev_year = &prev_year_proxy;
	g_class->next_year = &next_year_proxy;
}

GType
Gtk::CalendarClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_CALENDAR, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::CalendarClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::CalendarClass::month_changed_proxy(GtkCalendar *calendar)
{
	Calendar *tmp_calendar = G::Object::pointer<Calendar>(calendar);
	if (tmp_calendar)
		tmp_calendar->on_month_changed();
	else
	{
		GtkCalendarClass *g_class = G::TypeInstance::class_peek_parent<GtkCalendarClass>(GTK_CALENDAR_GET_CLASS(calendar));
		if (g_class->month_changed)
			g_class->month_changed(calendar);
	}
}

void
Gtk::CalendarClass::day_selected_proxy(GtkCalendar *calendar)
{
	Calendar *tmp_calendar = G::Object::pointer<Calendar>(calendar);
	if (tmp_calendar)
		tmp_calendar->on_day_selected();
	else
	{
		GtkCalendarClass *g_class = G::TypeInstance::class_peek_parent<GtkCalendarClass>(GTK_CALENDAR_GET_CLASS(calendar));
		if (g_class->day_selected)
			g_class->day_selected(calendar);
	}
}

void
Gtk::CalendarClass::day_selected_double_click_proxy(GtkCalendar *calendar)
{
	Calendar *tmp_calendar = G::Object::pointer<Calendar>(calendar);
	if (tmp_calendar)
		tmp_calendar->on_day_selected_double_click();
	else
	{
		GtkCalendarClass *g_class = G::TypeInstance::class_peek_parent<GtkCalendarClass>(GTK_CALENDAR_GET_CLASS(calendar));
		if (g_class->day_selected_double_click)
			g_class->day_selected_double_click(calendar);
	}
}

void
Gtk::CalendarClass::prev_month_proxy(GtkCalendar *calendar)
{
	Calendar *tmp_calendar = G::Object::pointer<Calendar>(calendar);
	if (tmp_calendar)
		tmp_calendar->on_prev_month();
	else
	{
		GtkCalendarClass *g_class = G::TypeInstance::class_peek_parent<GtkCalendarClass>(GTK_CALENDAR_GET_CLASS(calendar));
		if (g_class->prev_month)
			g_class->prev_month(calendar);
	}
}

void
Gtk::CalendarClass::next_month_proxy(GtkCalendar *calendar)
{
	Calendar *tmp_calendar = G::Object::pointer<Calendar>(calendar);
	if (tmp_calendar)
		tmp_calendar->on_next_month();
	else
	{
		GtkCalendarClass *g_class = G::TypeInstance::class_peek_parent<GtkCalendarClass>(GTK_CALENDAR_GET_CLASS(calendar));
		if (g_class->next_month)
			g_class->next_month(calendar);
	}
}

void
Gtk::CalendarClass::prev_year_proxy(GtkCalendar *calendar)
{
	Calendar *tmp_calendar = G::Object::pointer<Calendar>(calendar);
	if (tmp_calendar)
		tmp_calendar->on_prev_year();
	else
	{
		GtkCalendarClass *g_class = G::TypeInstance::class_peek_parent<GtkCalendarClass>(GTK_CALENDAR_GET_CLASS(calendar));
		if (g_class->prev_year)
			g_class->prev_year(calendar);
	}
}

void
Gtk::CalendarClass::next_year_proxy(GtkCalendar *calendar)
{
	Calendar *tmp_calendar = G::Object::pointer<Calendar>(calendar);
	if (tmp_calendar)
		tmp_calendar->on_next_year();
	else
	{
		GtkCalendarClass *g_class = G::TypeInstance::class_peek_parent<GtkCalendarClass>(GTK_CALENDAR_GET_CLASS(calendar));
		if (g_class->next_year)
			g_class->next_year(calendar);
	}
}

/*  Signal handlers
 */

void
Gtk::Calendar::on_month_changed()
{
	GtkCalendarClass *g_class = class_peek_parent<GtkCalendarClass>(gtk_calendar_class());
	if (g_class->month_changed)
		g_class->month_changed(gtk_calendar());
}

void
Gtk::Calendar::on_day_selected()
{
	GtkCalendarClass *g_class = class_peek_parent<GtkCalendarClass>(gtk_calendar_class());
	if (g_class->day_selected)
		g_class->day_selected(gtk_calendar());
}

void
Gtk::Calendar::on_day_selected_double_click()
{
	GtkCalendarClass *g_class = class_peek_parent<GtkCalendarClass>(gtk_calendar_class());
	if (g_class->day_selected_double_click)
		g_class->day_selected_double_click(gtk_calendar());
}

void
Gtk::Calendar::on_prev_month()
{
	GtkCalendarClass *g_class = class_peek_parent<GtkCalendarClass>(gtk_calendar_class());
	if (g_class->prev_month)
		g_class->prev_month(gtk_calendar());
}

void
Gtk::Calendar::on_next_month()
{
	GtkCalendarClass *g_class = class_peek_parent<GtkCalendarClass>(gtk_calendar_class());
	if (g_class->next_month)
		g_class->next_month(gtk_calendar());
}

void
Gtk::Calendar::on_prev_year()
{
	GtkCalendarClass *g_class = class_peek_parent<GtkCalendarClass>(gtk_calendar_class());
	if (g_class->prev_year)
		g_class->prev_year(gtk_calendar());
}

void
Gtk::Calendar::on_next_year()
{
	GtkCalendarClass *g_class = class_peek_parent<GtkCalendarClass>(gtk_calendar_class());
	if (g_class->next_year)
		g_class->next_year(gtk_calendar());
}

/*  Signals
 */

const Gtk::Calendar::MonthChangedSignalType Gtk::Calendar::month_changed_signal("month_changed");

const Gtk::Calendar::DaySelectedSignalType Gtk::Calendar::day_selected_signal("day_selected");

const Gtk::Calendar::DaySelectedDoubleClickSignalType Gtk::Calendar::day_selected_double_click_signal("day_selected_double_click");

const Gtk::Calendar::PrevMonthSignalType Gtk::Calendar::prev_month_signal("prev_month");

const Gtk::Calendar::NextMonthSignalType Gtk::Calendar::next_month_signal("next_month");

const Gtk::Calendar::PrevYearSignalType Gtk::Calendar::prev_year_signal("prev_year");

const Gtk::Calendar::NextYearSignalType Gtk::Calendar::next_year_signal("next_year");


