/**************************************************************************

    begin                : June 30 2004
	version		 : 1.2 
    copyright            : (C) 2004 by Gleb Beliakov
    email                : gleb@deakin.edu.au


  SLipInt.cpp: declaration of the Simple Lipschitz interpolant class.

  SLipInt class implements the method of Lipschitz interpolation when
  the distance d(x,y) is l_inf or l_2 norm. The interpolant is computed as
  g(x)= 0.5(H_upper(x)  + H_lower(x))

  with H_upper(x)= min_k (y^k + LipConst d(x,x^k))
  H_lower(x)= max_k (y^k - LipConst d(x,x^k))

  where the input data is (x^k,y^k), k=1,...npts.

  This is the best interpolant in the worst case scenario, if the interpolated
  function is known to be Lipschitz with the Lipschitz constant LipConst.

  There are no restrictions on the distribution of data x^k in R^dim


 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at your  *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License       *
 * along with this program; if not, write to the Free Software Foundation, *
 * Inc., 59 Temple Place Suite 330, Boston, MA 02111-1307 USA.             *
 ***************************************************************************/
#if !defined(SLIPINTERPOL)
#define SLIPINTERPOL

#include <cstdlib>
#include <cmath>


class SLipInt {
public:
	double*   LipConst, MaxLipConst;  // an array of Lipschitz constants and the largest Lipschitz constant
// computed automatically in ComputeLipschitzInf or ComputeLipschitz2

	SLipInt() {
		LipConst=NULL;
		MaxLipConst=0;
		Dim=0;
	}

	~SLipInt() { free(LipConst);	}

// In the methods below, XData contain the abscissae of data points x^k (arranged
// in rows (C-convention)) and YData contain y^k. x is the point at which g(x) is needed.

// Computes the smallest Lipschitz constant in l_inf norm, compatible with the data
	void	ComputeLipschitzInf(int dim, int npts, double* XData, double* YData);
// Computes the smallest Lipschitz constant in l_2 norm, compatible with the data
	void	ComputeLipschitz2(int dim, int npts, double* XData, double* YData);

// Returns the value of the interpolant in l_inf norm
	double	ValueInf(int dim, int npts, double* x, double* XData, double* YData,   double LipConst);

// Same, but uses an array of Lipschitz constants wrt each variable
	double	ValueInf(int dim, int npts, double* x, double* XData, double* YData,   double* LipConst);

// Returns the value of the interpolant in l_2 norm
	double	Value2(int dim, int npts,   double* x, double* XData, double* YData,   double LipConst);

// Returns the value of the interpolant in l_inf norm, with the Lipschitz constant
// computed from the data. Can be used after ComputeLipschitzInf
	double	ValueInf(int dim, int npts, double* x, double* XData, double* YData);

// Returns the value of the interpolant in l_2 norm, with the Lipschitz constant
// computed from the data. Can be used after ComputeLipschitz2
	double	Value2(int dim, int npts,   double* x, double* XData, double* YData);



// same as above, but XData is arranged in columns (Fortran-like)
	void	ComputeLipschitzInfCol(int dim, int npts, double* XData, double* YData);
	void	ComputeLipschitz2Col(int dim, int npts, double* XData, double* YData);

	double	ValueInfCol(int dim, int npts, double* x, double* XData, double* YData,   double LipConst);
	double	ValueInfCol(int dim, int npts, double* x, double* XData, double* YData,   double* LipConst);
	double	Value2Col(int dim, int npts,   double* x, double* XData, double* YData,   double LipConst);

	double	ValueInfCol(int dim, int npts, double* x, double* XData, double* YData);
	double	Value2Col(int dim, int npts,   double* x, double* XData, double* YData);


private:
	double g1,g2,d1,d2;
	int i,j;
	int Dim; 
};

#endif

