/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package test.mx4j.tools.heartbeat;

import java.util.Properties;

import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.MBeanServerInvocationHandler;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectName;
import javax.naming.Context;
import junit.framework.TestCase;

import mx4j.tools.adaptor.rmi.jrmp.JRMPAdaptor;
import mx4j.tools.adaptor.rmi.jrmp.JRMPAdaptorMBean;
import mx4j.tools.connector.rmi.RMIConnector;
import mx4j.tools.connector.rmi.jrmp.JRMPConnector;
import mx4j.log.Log;
import mx4j.log.Logger;
import mx4j.tools.heartbeat.HeartBeat;
import mx4j.tools.heartbeat.HeartBeatListener;
import mx4j.tools.heartbeat.HeartBeatListenerMBean;
import mx4j.tools.heartbeat.HeartBeatMBean;
import mx4j.tools.naming.NamingServiceMBean;

/**
 * @see
 * @author <a href="mailto:mgore@users.sourceforge.net">Michael Gorelik</a>
 * @version $Revision: 1.4 $
 */
public class HeartBeatTest extends TestCase
{
   public static final String JNDI_SERVER_NAME = "HeartBeatSource";
   public static final String JNDI_CLIENT_NAME = "HeartBeatListener";

   private int          m_defaultPriority;
   private Logger       m_logger =  Log.getLogger(getClass().getName());
   private MBeanServer  m_server;
   private ObjectName   m_naming;

   public HeartBeatTest(String s)
   {
      super(s);
   }

   protected void setUp() throws Exception
   {
      m_defaultPriority = Log.getDefaultPriority();
      Log.setDefaultPriority(Logger.WARN);
      m_server = MBeanServerFactory.createMBeanServer(JNDI_SERVER_NAME);
      // create and start RMIregistry
      m_naming = startRMIRegistry(m_server);
   }

   protected void tearDown() throws Exception
   {
      Log.setDefaultPriority(m_defaultPriority);
      // stop rmiregistry
      stopRMIRegistry(m_server, m_naming);

   }

   public void testCreateHeartBeatSource() throws Exception
   {
      String canonicalName;

      m_logger.trace("HeartBeatTest.testCreateHeartBeatSource");
      // FIXME: should it be variable number of HeartBeat objects?
      int n = 2;
      int nSuccess = 0;
      HeartBeat[] hb = new HeartBeat[n];
      for (int i=0; i<n; i++)
      {
         canonicalName = JNDI_SERVER_NAME + ":" +
                         HeartBeatMBean.HEARTBEAT_OBJECT_NAME + i;
         if ((hb[i] = mkHeartBeat(m_server, canonicalName)) != null)
         {
            nSuccess++;
         }
      }
      String msg = getClass().getName() + ".testCreateHeartBeatSource: ";
      if (nSuccess == n)
      {
         // success
         String successMsg = "SUCCESS: created " + nSuccess + " HeartBean object(s)";
         m_logger.info(msg + successMsg);
      }
      else
      {
         // total or partial failure
         String failMsg = "FAIL: created " + nSuccess + " HeartBean object(s) out of " + n;
         m_logger.error(msg + failMsg);
      }

      // cleanup
      for (int i=0; i<n; i++)
      {
         hb[i].stop();
      }
   }

   public void testCreateHeartBeatListener() throws Exception
   {
      m_logger.trace("HeartBeatTest.testCreateHeartBeatListener");

      // create server
      MBeanServer server = MBeanServerFactory.createMBeanServer(JNDI_CLIENT_NAME);
      // HeartBeatListener object
      String canonName = JNDI_CLIENT_NAME + ":" + HeartBeatListenerMBean.DEFAULT_LISTENER_NAME;
      HeartBeatListener hbl = new HeartBeatListener(canonName);
      try
      {
         server.registerMBean(hbl, new ObjectName(canonName));
      }
      catch (Exception ex)
      {
         m_logger.error(getClass().getName() + ".testCreateHeartBeatListener:  ex=" + ex.toString());
         return;
      }
      m_logger.info(getClass().getName() + ".testCreateHeartBeatListener: SUCCESS");
      hbl.stop();
   }

   public void testLifeHeartBeat() throws Exception
   {
      m_logger.trace("HeartBeatTest.testLifeHeartBeat");

      // start HeartBeat objects
      Source src = new Source();

      // start HeartBeatListener object
      Listener listener = new Listener();

      // wait for 2 heartbeats and termination
      Thread.sleep(Math.round(2.5*HeartBeatMBean.DEFAULT_PERIOD*1000));

      // stop HeartBeat objects
      m_logger.trace(getClass().getName()+".testLifeHeartBeat: stopping source");
      src.stop();

      // wait for listener to declare connections down
      long t = (HeartBeatMBean.DEFAULT_RETRIES+1)*HeartBeatMBean.DEFAULT_PERIOD*1000;
      Thread.sleep(t);

      m_logger.trace(getClass().getName()+".testLifeHeartBeat: stopping listener");
      listener.stop();
      // check that listener lost 2 heartbeats
      if (listener.m_lossCounter == src.NHEARTBEANS)
      {
         m_logger.info(getClass().getName()+".testLifeHeartBeat: SUCCESS");
      }
      else
      {
         m_logger.error(getClass().getName()+".testLifeHeartBeat: FAIL, " +
                        listener.m_lossCounter + " lost heartbeats");
      }
      System.gc();
   }

   // create and start rmiregistry
   private ObjectName startRMIRegistry(MBeanServer server)
   {
      NamingServiceMBean namingMB = null;
      ObjectName namingName = null;
      try
      {
         namingName = new ObjectName("Naming:type=registry");
         server.createMBean("mx4j.tools.naming.NamingService", namingName, null);
         namingMB = (NamingServiceMBean)MBeanServerInvocationHandler.newProxyInstance(server, namingName, NamingServiceMBean.class, false);
         Properties env = System.getProperties();
         String p = (String)env.get(Context.PROVIDER_URL);
         if (p == null)
         {
            namingMB.setPort(1099);
         }
         namingMB.start();
      }
      catch (Exception e)
      {
         String msg = this.getClass().getName() + ".startRMIRegistry: " + e.toString();
         m_logger.fatal(msg, e);
         return null;
      }
      return namingName;
   }

   // stop and remove rmiregistry from server
   private void stopRMIRegistry(MBeanServer server, ObjectName name)
   {
      try
      {
         server.invoke(name, "stop", null, null);
         server.unregisterMBean(name);
      }
      catch (Exception e)
      {
         String msg = this.getClass().getName() + ".stopRMIRegistry: " + e.toString();
         m_logger.fatal(msg, e);
      }
   }

   // return HeartBeat object if successful
   private HeartBeat mkHeartBeat(MBeanServer server, String canonName)
   {
      HeartBeat hb = new HeartBeat(canonName);
      hb.start();
      try
      {
         server.registerMBean(hb, new ObjectName(canonName));
      }
      catch (Exception ex)
      {
         String msg = getClass().getName() + ".mkHeartBeat: " + ex.toString();
         m_logger.fatal(msg, ex);
         return null;
      }
      String msg = this.getClass().getName() + ".mkHeartBeat: created HeartBeatMBean=" + canonName;
      m_logger.warn(msg);
      return hb;
   }

   private JRMPAdaptor mkJRMPAdaptor(String jndiName, String adaptName, MBeanServer server)
   {
      JRMPAdaptor adaptor = new JRMPAdaptor();

      try
      {
         ObjectName adaptorName = new ObjectName(adaptName);
         server.registerMBean(adaptor, adaptorName);
         adaptor.setJNDIName(jndiName);
         adaptor.start();
      }
      catch (Exception ex)
      {
         String msg = getClass().getName() + ".mkJRMPAdaptor: " + ex.toString();
         m_logger.fatal(msg, ex);
      }
      return adaptor;
   }

   // create 2 HeartBeat objects
   private class Source
   {
      public int NHEARTBEANS = 2;
      JRMPAdaptorMBean m_jrmpAdaptorMBean;
      private HeartBeat[] m_hbeats = new HeartBeat[NHEARTBEANS];

      public Source()
      {
         // create JRMP adaptor for server
         String adaptName = JNDI_SERVER_NAME + ":type=jrmpAdaptor";
         JRMPAdaptorMBean jrmpAdaptorMBean = mkJRMPAdaptor(JNDI_SERVER_NAME, adaptName, m_server);
         for (int i=0; i<NHEARTBEANS; i++)
         {
            String canonicalName = JNDI_SERVER_NAME + ":" +
                         HeartBeatMBean.HEARTBEAT_OBJECT_NAME + i;
            m_hbeats[i] = mkHeartBeat(m_server, canonicalName);
         }
         synchronized(this)
         {
            notifyAll();
         }
      }

      public synchronized void stop()
      {
         m_logger.trace(getClass().getName() + ".stop");
         for (int i=0; i<m_hbeats.length; i++)
         {
            m_hbeats[i].stop();
         }
         try
         {
            m_jrmpAdaptorMBean.stop();
         }
         catch (Exception e)
         {}
      }
   } // Source

   /**
    * Create one HeartBeatListener object, register with two HeartBeat objects and
    * wait for notification.
    */
   private class Listener implements NotificationListener
   {
      public int m_lossCounter = 0;

      private MBeanServer m_mbserver = MBeanServerFactory.createMBeanServer(JNDI_CLIENT_NAME);
      private HeartBeatListener m_hbl;
      private JRMPAdaptorMBean m_jrmpAdaptorMBean;

      public Listener()
      {
         // create JRMP adaptor for client
         String adaptName = JNDI_CLIENT_NAME + ":type=jrmpAdaptor";
         JRMPAdaptorMBean jrmpAdaptorMBean = mkJRMPAdaptor(JNDI_CLIENT_NAME, adaptName, m_mbserver);
         String canonName = JNDI_CLIENT_NAME + ":" +
                            HeartBeatListenerMBean.DEFAULT_LISTENER_NAME;
         m_hbl = new HeartBeatListener(canonName);
         RMIConnector conn = null;
         try
         {
            m_mbserver.registerMBean(m_hbl, new ObjectName(canonName));
            conn = new JRMPConnector();
            conn.connect(JNDI_SERVER_NAME, System.getProperties());
         }
         catch (Exception ex)
         {
            m_logger.error("Listener.run: ex=" + ex.toString());
         }

         String heartBeatCanonName;
         for (int i=0; i<2; i++)
         {
            heartBeatCanonName = JNDI_SERVER_NAME + ":" +
                              HeartBeatMBean.HEARTBEAT_OBJECT_NAME + i;
            try
            {
               m_hbl.registerObserver(conn.getRemoteMBeanServer(),
                                      HeartBeatMBean.RMI_TYPE, JNDI_CLIENT_NAME,
                                      heartBeatCanonName, this);
            }
            catch (Exception e)
            {
               m_logger.debug("Listener.run: ex=" + e.toString());
            }
         } // for
      } // Listener

      public void handleNotification(Notification not, Object handback)
      {
         m_logger.debug(getClass().getName()+".handleNotification: lost heartbeat from " +
                         not.getSource());
         m_lossCounter++;
      }

      public void stop()
      {
         m_hbl.stop();
         try
         {
            m_jrmpAdaptorMBean.stop();
         }
         catch (Exception e)
         {}
      }

   }
}
