(**
  This module is responsible for the handling of the display and all
  its attributes like color, font and that sort. It also exports the
  class @code{DrawInfo} which implements an abstract, that means
  OS-independend, drawing engine.

  TODO
  * Busy function for *all* windows

  * Rethink pens
*)

MODULE VO:Base:Display [OOC_EXTENSIONS];

(*
    Classes for visualisation.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D    := VO:Base:DragDrop,
       E    := VO:Base:Event,
       O    := VO:Base:Object,
       P    := VO:Prefs:Base,
       U    := VO:Base:Util,

       ch   := Channel,
       str  := Strings;

CONST
  (* Different messages the display can listen to *)

  exitMsg*    = 0;
  timeOutMsg* = 1;

  (* Different modes of display color (Display.colorMode) *)

  monochromeMode * = 0; (** The display runs in monochrome mode,
                            that means only black and white are available *)
  greyScaleMode  * = 1; (** The display runs in greyscale mode *)
  colorMode      * = 2; (** The display runs in color mode *)

  backgroundColorIndex       * =  0;
  tableBackgroundColorIndex  * =  1;
  tableTextColorIndex        * =  2;
  textBackgroundColorIndex   * =  3;
  buttonBackgroundColorIndex * =  4;
  textColorIndex             * =  5;
  shineColorIndex            * =  6;
  halfShineColorIndex        * =  7;
  halfShadowColorIndex       * =  8;
  shadowColorIndex           * =  9;
  fillColorIndex             * = 10;
  fillTextColorIndex         * = 11;
  warnColorIndex             * = 12;
  disabledColorIndex         * = 13;
  focusColorIndex            * = 14;
  blackColorIndex            * = 15;
  whiteColorIndex            * = 16;
  helpBackgroundColorIndex   * = 17;

  colorCount   * = 18;

  (* The different supported fonts  *)

  tinyFontIndex       * =  0;
  scriptFontIndex     * =  1;
  footnoteFontIndex   * =  2;
  smallFontIndex      * =  3;
  normalFontIndex     * =  4;
  largeFontIndex      * =  5;
  LargeFontIndex      * =  6;
  LARGEFontIndex      * =  7;
  hugeFontIndex       * =  8;
  HUGEFontIndex       * =  9;

  smallFixedFontIndex * = 10;
  fixedFontIndex      * = 11;
  hugeFixedFontIndex  * = 12;

  fontCount      * = 13;

  (*
    The different styles a font can have.
    NOTE
    Not all styles can used together. italic and slanted f.e. are
    exclusive.
  *)

  bold        * = 0;
  italic      * = 1;
  underlined  * = 2;
  slanted     * = 3;

  maxStyle    * = 3;
  maxStyleNum * = 2*2*2*2;
  styleMask   * = {bold,italic,underlined,slanted};

  (*
    List of features a font can have. A Font structure has a number of
    attributes. The feature list tells which features are known to this font.
    (A font selection must only that parameter ofa font that are within its
    feature list.

    The other way round, when loading a font you fill in a number of features
    about that font to specify it. VisualOberon is free to ignore some features.
  *)

  fontName*        =  0; (** This should always be set *)
  fontHeight*      =  1;
  fontAscent*      =  2; (** I'm not sure if these two are really  feature *)
  fontDescent*     =  3;
  fontPointHeight* =  4;
  fontHorizRes*    =  5;
  fontVertRes*     =  6;
  fontAvWidth*     =  7;
  fontSetWidth*    =  8;
  fontSpacing*     =  9;
  fontCharSet*     = 10;


  (*
    font width for the above fontSetWidth feature
  *)

  fontNormal *   = 0;
  fontCondensed* = 1;
  fontNarrow*    = 2;
  fontDouble*    = 3;

  (*
    font spacing for the above fontSpacing feature
  *)

  fontFixed*        = 0;
  fontProportional* = 1;
  (* there is also "c", whats that? *)



  (*
    Different DrawInfo.mode
    Not all modes other than "normal" need to be implemented
  *)

  selected*  = 0; (** Not allways supported *)
  hidden*    = 1; (** Can be used interaly, need not to be supported *)
  activated* = 2; (** Active but not selected *)
  disabled*  = 3; (** Draw yourself disabled *)

  (* pen styles *)

  normalPen * = 0;
  roundPen  * = 1;

  (* draw mode *)

  copy   * = 0;
  invert * = 1;

  (* dash modes *)

  fMode * = 0; (** Draw dashes in foreground color *)
  fbMode* = 1; (** Draw dashes in foreground color, gaps in background color *)

  (* pattern styles *)

  fgPattern* = 0; (** set bits are draw in foreground color, unset bits arn't used *)
  fbPattern* = 1; (** set bits are draw in foreground color, unset in background   *)

  (* various types of data *)

  none * = 0;
  text * = 1;

  (* flags for window positioning *)

  manualPos*      = 0;
  centerOnParent* = 1;
  centerOnScreen* = 2;
  osPos*          = 3;

  disableWidth*  = 8;
  disableHeight* = 2;

TYPE
  ColorName*    = ARRAY 30 OF CHAR;
  FontName*     = ARRAY 100 OF CHAR;

  Window*       = POINTER TO WindowDesc;
  Display*      = POINTER TO DisplayDesc;
  DrawInfo*     = POINTER TO DrawInfoDesc;
  Font*         = POINTER TO FontDesc;


  PrefsCallback*     = POINTER TO PrefsCallbackDesc;
  PrefsCallbackDesc* = RECORD [ABSTRACT]
                       END;

  Color*        = LONGINT;


  Object*     = POINTER TO ObjectDesc;


  ObjectDesc* = RECORD (O.MsgObjectDesc)
  (**
    The @otype{Display} needs to know Objects within a window.
   *)
                END;


  FreeEntry     = POINTER TO FreeEntryDesc;
  FreeEntryDesc = RECORD
                    next   : FreeEntry;
                    object : O.Object;
                  END;


  WindowDesc*   = RECORD [ABSTRACT] (O.MsgObjectDesc)
  (**
    The abstract baseclass for all windows. All windows should derive
    from this class and call the apropiate methods where stated.

    Window inherits from O.MsgObject, you can send messages to it.
  *)
                    x*,y*,       (* TODO: Fix export *)
                    width*,
                    height*,
                    minWidth*,
                    minHeight*,
                    maxWidth*,
                    horizontalPos*,
                    verticalPos*,
                    maxHeight*     : LONGINT;
                    parent-        : Window; (* TODO: Fix export *)
                    title-         : U.Text; (* TODO: Fix export *)
                    background*    : Color;  (* TODO: Fix export *)
                    focus,
                    maped,
                    borderless*,             (* TODO: Fix export *)
                    open           : BOOLEAN;
                    freeList       : FreeEntry;
                  END;

  Bitmap*       = POINTER TO BitmapDesc;
  BitmapDesc*   = RECORD
                    draw*   : DrawInfo; (* TODO: Fix export *)
                    width*,             (* TODO: Fix export *)
                    height* : LONGINT;  (* TODO: Fix export *)
                  END;

  FontDesc*     = RECORD [ABSTRACT]
  (**
    Abstract fontdefinition.
  *)
                    next*,                 (* TODO: Fix export *)
                    last*        : Font;   (* TODO: Fix export *)
                    features*    : SET;
                    name*,
                    charSet*     : U.Text;
                    ascent*,
                    descent*,
                    height*,
                    pointHeight*,
                    horizRes*,
                    vertRes*,
                    avWidth*,
                    setWidth*,
                    spacing*     : LONGINT;
                  END;

  FontExtentDesc* = RECORD
  (**
    Returned by Font.FontExtent.
  *)
                      lbearing*,             (* TODO: Fix export? *)
                      rbearing*,
                      width*,
                      height*,
                      ascent*,
                      descent*  : LONGINT;
                    END;

  TimeOut*        = POINTER TO TimeOutDesc;
  TimeOutDesc*    = RECORD
                    END;

  Sleep*          = POINTER TO SleepDesc;
  SleepDesc*      = RECORD
                    END;

  Channel*        = POINTER TO ChannelDesc;
  ChannelDesc*    = RECORD
                    END;

  DisplayDesc * = RECORD [ABSTRACT] (O.MsgObjectDesc)
  (**
    Class that abstracts the properties of a display.
   *)
                    appName-     : POINTER TO ARRAY OF CHAR;

                    name*        : ARRAY 100 OF CHAR;  (* fix export *)

                    scrWidth*,                         (* fix export *)
                    scrHeight*   : LONGINT;            (* fix export *)

                    (* colors *)
                    colorMode*   : LONGINT;            (* fix export *)
                    colorDepth*  : LONGINT;            (* fix export *)

                    currentWin*  : Window;  (* will become obsolete in future *) (* fix export *)
                    spaceHeight*,           (* will become obsolete in future *) (* fix export *)
                    spaceWidth*  : LONGINT; (* will become obsolete in future *) (* fix export *)
                  END;

  DisplayPrefs*     = POINTER TO DisplayPrefsDesc;
  DisplayPrefsDesc* = RECORD (P.PrefsDesc)
  (**
    In this class all preferences stuff of the display is stored.
   *)
                        colors* : ARRAY colorCount OF ColorName;
                        fonts*  : ARRAY fontCount  OF FontName;
                      END;

  ExitMsg*        = POINTER TO ExitMsgDesc;
  ExitMsgDesc*    = RECORD (O.MessageDesc)
  (**
    Send this message to Display if you want to leave the application.
   *)
                    END;

  TimeOutMsg*     = POINTER TO TimeOutMsgDesc;
  TimeOutMsgDesc* = RECORD (O.MessageDesc)
  (**
    The message an objects gets, when the given timeout runs out.
   *)
                      timeOut* : TimeOut;
                    END;

  SleepMsg*       = POINTER TO SleepMsgDesc;
  SleepMsgDesc*   = RECORD (O.MessageDesc)
  (**
    The message an objects gets, when a registered sleep event get called.
   *)
                      sleep* : Sleep;
                    END;

  ChannelMsg*     = POINTER TO ChannelMsgDesc;
  ChannelMsgDesc* = RECORD (O.MessageDesc)
  (**
    The message an objects gets, when a registered fd event get called.
   *)
                      channel* : Channel;
                    END;

  Msg2Exit*       = POINTER TO Msg2ExitDesc;
  Msg2ExitDesc*   = RECORD (O.HandlerDesc)
  (**
    A converter that throuws the incomming message away and generates a
    ExitMsg for Display.
  *)
                    END;

  PointDesc* = RECORD
  (**
    A point has an x and an y member with size INTEGER.
    However the X11-function wants a XPoint as parameter,
    so we simply alias from it. Ports may implement this
    differently but they must have two members x and y.
  *)             x*,y* : LONGINT;
               END;


  DrawInfoDesc* = RECORD [ABSTRACT]
  (**
    The class for all abstract drawings.
   *)
                    mode* : SET;
                  END;

(*  Data*     = POINTER TO DataDesc;
  DataDesc* = RECORD
                type*   : LONGINT;
                length* : LONGINT;

                string* : U.Text;
                ints*   : POINTER TO ARRAY OF INTEGER;
                longs*  : POINTER TO ARRAY OF LONGINT;

                xData   : C.charPtr1d;
              END;*)

  Factory*      = POINTER TO FactoryDesc;
  FactoryDesc*  = RECORD [ABSTRACT]
  (**
    Factory for generating instances of certain classes.
   *)
                  END;

VAR

  (*
    This function will be called when all inportant display filed are filled
    and before the corresponding preferences settings are evaluated.
  *)

  prefsCallback* : PrefsCallback;

  (* Pattern for disabling of gadgets *)

  disablePattern- : ARRAY 2 OF CHAR;

  (* Diverse patterns for dashlines *)

  sPointLine- : ARRAY 2 OF CHAR; (** small  points *)
  mPointLine- : ARRAY 2 OF CHAR; (** medium points *)
  lPointLine- : ARRAY 2 OF CHAR; (** large points  *)

  colorNames- : ARRAY colorCount OF ColorName;
  fontNames-  : ARRAY fontCount OF FontName;

  prefs*      : DisplayPrefs;

  display-    : Display;

  factory     : Factory;
VAR
  (* Pens *)

  backgroundColor       *, (** background color of windows *)
  tableBackgroundColor  *, (** background of table like object *)
  tableTextColor        *,
  textBackgroundColor   *, (** backgrond of text pinput objects *)
  buttonBackgroundColor *, (** background color of button-like objects, should be close to backgroundColor *)
  textColor             *, (** normal text background *)
  shineColor            *,
  halfShineColor        *,
  halfShadowColor       *,
  shadowColor           *,
  fillColor             *, (** color for visualizing selection of filling *)
  fillTextColor         *, (** textcolor while selecting *)
  warnColor             *, (** warn color *)
  disabledColor         *, (** color of "disable"grid *)
  focusColor            *, (** color of focus frame *)
  blackColor            *,
  whiteColor            *,
  helpBackgroundColor   * : Color; (** background color of tooltips *)
  
  tinyFont*,
  scriptFont*,
  footnoteFont*,
  smallFont*,
  normalFont*,
  largeFont*,
  LargeFont*,
  LARGEFont*,
  hugeFont*,
  HUGEFont*,

  smallFixedFont*,
  fixedFont*,
  hugeFixedFont*   : Font;
  

  PROCEDURE (p : PrefsCallback) [ABSTRACT] LoadPrefs*(appName : ARRAY OF CHAR);
  END LoadPrefs;

  PROCEDURE (p : PrefsCallback) [ABSTRACT] ReadDisplayPrefs*;
  END ReadDisplayPrefs;

  PROCEDURE (p : PrefsCallback) [ABSTRACT] ReadOtherPrefs*;
  END ReadOtherPrefs;

  PROCEDURE (p : PrefsCallback) [ABSTRACT] FreePrefs*;
  END FreePrefs;

  (* ------------ object stuff --------------- *)

  PROCEDURE (object : Object) GetDragInfo*(VAR dragInfo : D.DnDDataInfo);
  (**
    Before the data drop actually occurs the drag object will be asked
    for a list of supported datatypes. This will then be handed to the
    drop object which than can select a apropiate datatype. This type will
    the requested from the drag object and after that will be handed to the
    drop object.

    NOTE
    The is no way to cancel the action at that point the object has already
    made itself valid by answering VO:Object.Object.GetDnDObject.

    This is only true for internal drag and drop. External drag and drop may
    leave some of the described steps. An object must be prepared of that.
  *)

  BEGIN
  END GetDragInfo;

  PROCEDURE (object : Object) GetDragData*(group, type, action : LONGINT):D.DnDData;
  (**
    All objects support drag actions. Return a derived class of
    VODragDrop.DnDData if you can offer drag data of the given type
    or NIL.

    NOTE
    data should really be of type DataDesc, but there seems to be
    compiler bugs with it. We will change this later.
  *)

  BEGIN
    RETURN NIL;
  END GetDragData;

  PROCEDURE (object : Object) GetDropDataType*(VAR dragInfo : D.DnDDataInfo;
                                               VAR group,type,action : LONGINT):BOOLEAN;
  (**
    The object gets a list of supported datatypes of the drag object and
    has to choose one of them. If there is no fitting datatype it must return
    FALSE.
  *)

  BEGIN
    RETURN FALSE;
  END GetDropDataType;

  PROCEDURE (object : Object) HandleDrop*(data : D.DnDData; action : LONGINT):BOOLEAN;
  (**
    All object can handle drop actions. Return TRUE if you have
    handled the drop event.
  *)

  BEGIN
    RETURN FALSE;
  END HandleDrop;

  PROCEDURE (object : Object) Deselect*;
  (**
    Gets called, when the object has registered a selection using
    Display.RegisterSelection and now should deselect itself because
    somebody else has registered an selection.
  *)

  BEGIN
  END Deselect;

  PROCEDURE (d : DrawInfo) [ABSTRACT] InstallClip*(x,y,w,h : LONGINT);
  (**
    Start a new clipping. Calling this function does not start
    any cliping (in fact it copies the cliping rectangles of the
    current cliping) you must add cliping rectangles using
    DrawInfo.AddRegion.

    Clips are stacked and must be freed in order.
  *)
  END InstallClip;

  PROCEDURE (d : DrawInfo) [ABSTRACT] AddRegion*(x,y,width,height : LONGINT);
  (**
    Add a new cliping rectangle to the current clip.
  *)
  END AddRegion;

  PROCEDURE (d : DrawInfo) [ABSTRACT] SubRegion*(x,y,width,height : LONGINT);
  (**
    Removes a rectangle from the allowed area.
  *)
  END SubRegion;

  PROCEDURE (d : DrawInfo) [ABSTRACT] GetClipRegion*(VAR x,y,w,h : LONGINT);
  (**
    Returns the outer box of the current clip region.
  *)
  END GetClipRegion;

  PROCEDURE (d : DrawInfo) [ABSTRACT] FreeLastClip*;
  (**
    Free the current clip and reset clipping to the last stacked clip.
  *)
  END FreeLastClip;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PushFont*(handle : Font; style : SET);
  (**
    Push the given font on the stack and make it the current one.
  *)
  END PushFont;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PopFont*;
  (**
    Pops the last stacked font and reactivates the last stacked.
  *)
  END PopFont;

  PROCEDURE (d : DrawInfo) [ABSTRACT] DrawString*(x,y : LONGINT; text : ARRAY OF CHAR; length : LONGINT);
  (**
    Draws the given string in the current font at the given position.
  *)
  END DrawString;

  PROCEDURE (d : DrawInfo) [ABSTRACT] DrawLongString*(x,y : LONGINT; text : ARRAY OF LONGCHAR; length : LONGINT);
  END DrawLongString;

  PROCEDURE (d : DrawInfo) [ABSTRACT] DrawFillString*(x,y : LONGINT; text : ARRAY OF CHAR; length : LONGINT);
  (**
    Draws the given string in the current font at the given position.
    The background will be filled with the current background color.
  *)
  END DrawFillString;

  PROCEDURE (d : DrawInfo) [ABSTRACT] DrawFillLongString*(x,y : LONGINT; text : ARRAY OF LONGCHAR; length : LONGINT);
  (**
    Similar to DrawFillString but for unicode strings.
  *)
  END DrawFillLongString;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PushForeground*(color : Color);
  (**
    Push the given color on the stack and make it the current foreground
    color.
  *)
  END PushForeground;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PopForeground*;
  (**
    Pop the last pushed foreground color from the stack and thus reinstalls
    the previous color.
  *)
  END PopForeground;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PushDrawMode*(mode : LONGINT);
  (**
    Push the given draw mode on the stack and make it the current draw mode.
  *)
  END PushDrawMode;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PopDrawMode*;
  (**
    Push the given draw mode on the stack and make it the current draw mode.
  *)
  END PopDrawMode;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PushBackground*(color : Color);
  (**
    Push the given draw mode on the stack and make it the current draw mode.
  *)
  END PushBackground;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PopBackground*;
  (**
    Push the given draw mode on the stack and make it the current draw mode.
  *)
  END PopBackground;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PushStyle*(size, mode : LONGINT);
  (**
    Push the given draw mode on the stack and make it the current draw mode.
  *)
  END PushStyle;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PopStyle*;
  (**
    Push the given draw mode on the stack and make it the current draw mode.
  *)
  END PopStyle;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PushDash*(dashList : ARRAY OF CHAR; mode : LONGINT);
  (**
    Pushes the a new style for drawing lines on the stack.
  *)
  END PushDash;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PopDash*;
  (**
    Pushes the a new style for drawing lines on the stack.
  *)
  END PopDash;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PushPattern*(pattern : ARRAY OF CHAR; width, height : LONGINT; mode : LONGINT);
  (**
    Push a new pattern for filling onto the stack.
  *)
  END PushPattern;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PopPattern*;
  (**
    Pop the last pattern from the stack.
  *)
  END PopPattern;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PushUniqueFillPattern*(pos, maximum : LONGINT);
  (**
    This methods tries to create a unique fill pattern on the stack. VisualOberon tries
    to select the optimal fill pattern for the given color mode.VisualOberon cannot offer
    an unlimited number of different fill patterns. After a not specified amount of patterns
    (at three) VisualOberon will reuse the patterns. E.g. when VisualOberon offers three
    patterns the fourth one will be equal to the first one. VisualOberon will also garantee
    that the the last pattern will not be equal to the first pattern. Garanteeing this, you
    need not take special care when filling f.e. a pie chart.

    PARAMETER
      - @oparam{pos} is the running number of the pattern.
      - @oparam{maximum} is the maximum number of patterns you want.
        Only when this value is correctly set VisualOberon will garantee the above fact.
        If you don't want VisualOberon to take special care just hand -1.
  *)
  END PushUniqueFillPattern;

  PROCEDURE (d : DrawInfo) [ABSTRACT] PopUniqueFillPattern*(pos, maximum : LONGINT);
  (**
    Pop the pushed pattern from the stack.
  *)
  END PopUniqueFillPattern;

  (* --- Drawing functions --- *)

  PROCEDURE (d : DrawInfo) [ABSTRACT] DrawPoint*(x,y : LONGINT);
  (**
    Draws a point at the given position.
  *)
  END DrawPoint;

  PROCEDURE (d : DrawInfo) [ABSTRACT] DrawLine*(x1,y1,x2,y2 : LONGINT);
  (**
    Draws a line from x1,y1 to x2,y2, including.
  *)
  END DrawLine;

  PROCEDURE (d : DrawInfo) [ABSTRACT] FillRectangle*(x,y,width,height : LONGINT);
  END FillRectangle;

  PROCEDURE (d : DrawInfo) [ABSTRACT] DrawArc*(x,y,width,height,angle1,angle2 : LONGINT);
  END DrawArc;

  PROCEDURE (d : DrawInfo) [ABSTRACT] FillArc*(x,y,width,height,angle1,angle2 : LONGINT);
  END FillArc;

  PROCEDURE (d : DrawInfo) [ABSTRACT] FillPolygon*(points : ARRAY OF PointDesc; count : LONGINT);
  END FillPolygon;

  PROCEDURE (d : DrawInfo) [ABSTRACT] FillBackground*(x,y,width,height : LONGINT);
  (**
    Fill the given rectangle with the background color defined in Display.
    This is a high level function. You should use it whenever you want to clear
    a area and give it the background color.
  *)
  END FillBackground;

  PROCEDURE (d : DrawInfo) [ABSTRACT] CopyArea*(sX,sY,width,height,dX,dY : LONGINT);
  (**
    Fill the given rectangle with the background color defined in Display.
    This is a high level function. You should use it whenever you want to clear
    a area and give it the background color.
  *)
  END CopyArea;

  PROCEDURE (d : DrawInfo) [ABSTRACT] CopyFromBitmap*(bitmap : Bitmap; sX,sY,width,height,dX,dY : LONGINT);
  END CopyFromBitmap;

  PROCEDURE (d : DrawInfo) [ABSTRACT] CopyToBitmap*(sX,sY,width,height,dX,dY : LONGINT; bitmap : Bitmap);
  END CopyToBitmap;

  (* ------------ Font ----------------- *)

  PROCEDURE (font : Font) [ABSTRACT] TextWidth*(text : ARRAY OF CHAR; length : LONGINT; style : SET):LONGINT;
  (**
    Returns the width of the given text in pixels in respect to the font.
  *)
  END TextWidth;

  PROCEDURE (font : Font) [ABSTRACT] LongTextWidth*(text : ARRAY OF LONGCHAR; length : LONGINT; style : SET):LONGINT;
  END LongTextWidth;

  PROCEDURE (font : Font) [ABSTRACT] TextExtent*(text : ARRAY OF CHAR; length : LONGINT; style : SET; VAR extent : FontExtentDesc);
  (**
    Returns the width of the given text in pixels in respect to the font.
  *)
  END TextExtent;

  PROCEDURE (font : Font) [ABSTRACT] LongTextExtent*(text : ARRAY OF LONGCHAR; length : LONGINT; style : SET; VAR extent : FontExtentDesc);
  END LongTextExtent;

  PROCEDURE (f : Factory) [ABSTRACT] CreateFont*():Font;
  END CreateFont;

  PROCEDURE (f : Factory) [ABSTRACT] CreateDrawInfo*():DrawInfo;
  END CreateDrawInfo;

  PROCEDURE (f : Factory) [ABSTRACT] CreateDisplay*():Display;
  END CreateDisplay;

  PROCEDURE (f : Factory) [ABSTRACT] CreateBitmap*():Bitmap;
  END CreateBitmap;

(* ============== Implementation classes ============== *)

  PROCEDURE (d : Display) SetAppName*(name : ARRAY OF CHAR);

  BEGIN
    NEW(d.appName,str.Length(name)+1);
    COPY(name,d.appName^);
  END SetAppName;

  PROCEDURE (d : Display) [ABSTRACT] Open*():BOOLEAN;
  (**
    Initialize a instance of the Display class.
  *)
  END Open;

  PROCEDURE (d : Display) [ABSTRACT] LoadFont*(font : Font):BOOLEAN;
  (**
    Loads the font. The OS specific font will be matched by evaluating the
    handed features. Every font loaded must be free using Free.

    RETURNS
    If the font can be loaded the method returns a new instance of font that must be used
    for future uses exspecially when calling Free.
  *)
  END LoadFont;

  PROCEDURE (d : Display) [ABSTRACT] FreeFont*(font : Font);
  (**
    Free a font loaded using LoadFont.
  *)
  END FreeFont;

  PROCEDURE (d : Display) [ABSTRACT] GetFontList*():Font;
  END GetFontList;

  (* -------- color stuff of display ------------- *)

  PROCEDURE (d : Display) [ABSTRACT] AllocateColor*(r,g,b,default : LONGINT):Color;
  (**
    Allocate a color with the given rgb values. Since colors cannot be
    garbage collected you must store the color id and free the color later.

    RESULT
    A color id. You need to store this to free the color later.

    NOTE
    You cannot be sure that all bits of the color values will be used.
    X11 f.e. only uses 16 bit for color description.
  *)
  END AllocateColor;

  PROCEDURE (d : Display) [ABSTRACT] AllocateNamedColor*(name : ARRAY OF CHAR; default : Color):Color;
  END AllocateNamedColor;

  PROCEDURE (d : Display) [ABSTRACT] IsAllocatedColor*(color : Color):BOOLEAN;
  END IsAllocatedColor;

  PROCEDURE (d : Display) [ABSTRACT] FreeColor*(color : Color);
  END FreeColor;

  PROCEDURE (d : Display) [ABSTRACT] AddTimeOut*(sec, msec : LONGINT; object : O.MsgObject):TimeOut;
  (**
    Add a timeout. The object will be send a TimeOutMsg directly after
    the given (relative) time has been elapsed.

    NOTE
    It is garanteed that the time has been elasped when the object
    get notified. However, since there is no real multitasking involved,
    it may get called later.

    If the timeout has been elasped and the message has been send, there is
    no need to remove the TimeOut. Dipslay does this for you.
  *)
  END AddTimeOut;

  PROCEDURE (d : Display) [ABSTRACT] RemoveTimeOut*(timeOut : TimeOut);
  (**
    Remove a TimeOut before it has been elasped. It is save to call this function with
    a TimeOut-instance of an already elapsed time event.
  *)
  END RemoveTimeOut;

  PROCEDURE (d : Display) [ABSTRACT] AddSleep*(object : O.MsgObject):Sleep;
  (**
    Using this method, you can add an object to be called whenever the application is
    not busy. This way you can write applications that take as much processor time as
    possible while still listening to windowing events.

    Note, that you must design the action to be taken when the object is called to be
    rather short, since all event handling will be blocked during this time. If you
    want do handle long time actions you must split them into short partial actions.
  *)
  END AddSleep;

  PROCEDURE (d : Display) [ABSTRACT] RemoveSleep*(sleep : Sleep);
  (**
    Removes the given sleep notifier.
  *)
  END RemoveSleep;

  PROCEDURE (d : Display) [ABSTRACT] AddChannel*(channel : ch.Channel; object : O.MsgObject):Channel;
  (**
    Using this method, you can add an object to be called whenever the handed
    channel gets available.

    Note, that you must design the action to be taken when the object is
    called to be rather short, since the complete event handling will be
    blocked during this time. If you want do handle long time actions you
    must split them into short partial actions.
  *)
  END AddChannel;

  PROCEDURE (d : Display) [ABSTRACT] RemoveChannel*(channel : Channel);
  (**
    Removes the given sleep notifier.
  *)
  END RemoveChannel;

  PROCEDURE (d : Display) [ABSTRACT] CreateBitmap*(width, height : LONGINT):Bitmap;
  END CreateBitmap;

  PROCEDURE (d : Display) [ABSTRACT] FreeBitmap*(bitmap : Bitmap);
  END FreeBitmap;

  PROCEDURE (d : Display) [ABSTRACT] IsDoubleClicked*():BOOLEAN;
  END IsDoubleClicked;

  PROCEDURE (d : Display) [ABSTRACT] SetMultiClickTime*(t : LONGINT);
  END SetMultiClickTime;

  PROCEDURE (d : Display) [ABSTRACT] Beep*;
  (**
    Do a display beep.
  *)
  END Beep;

  PROCEDURE (d : Display) [ABSTRACT] RegisterSelection*(object : Object; window : Window):BOOLEAN;
  (**
    using this method an object can register itself a holder of the current
    selection.
  *)
  END RegisterSelection;

  PROCEDURE (d : Display) [ABSTRACT] CancelSelection*;
  (**
    Using this method an object can cancel the before registered selection.
    This can f.e happen, when the object hides.

    NOTE
    The current object holding the selection gets notified via calling
    its Object.Deselect method.
  *)
  END CancelSelection;

  PROCEDURE (d : Display) [ABSTRACT] QuerySelection*(window: Window; object: Object; type: LONGINT):BOOLEAN;
  (**
    Call this method if you want to get the value of the global selection. If
    VisualOberon can get the selection the handleDropData method of the
    querying object will be called with a VODragDrop.insert action. Note that there
    may be some undeterminate amount of time between the query and the drop, it is
    also possible that the HandleDrop method will never will called.
  *)
  END QuerySelection;

  PROCEDURE (d : Display) [ABSTRACT] EventLoop*;
  (**
    This is the main event loop of your application. Call this the get
    things started. If you want to leave the event loop and fiish the
    application, send a ExitMsg to the display.
  *)
  END EventLoop;

  PROCEDURE (d : Display) [ABSTRACT] Receive*(message : O.Message);
  (**
    The message receive function of the Display. Currently only ExitMsg and
    TimeOutMsgs, which get generated by Display when a contextHelp should
    occure, are supported.
  *)
  END Receive;

  PROCEDURE (d : Display) [ABSTRACT] PutBackEvent*(event : E.Event; destWin : Window);
  (**
    Put an event back into the message handling queue.
  *)
  END PutBackEvent;

  PROCEDURE (d : Display)[ABSTRACT] ReinitWindows*;
  END ReinitWindows;

  PROCEDURE (d : Display) [ABSTRACT] Exit*;
  (**
    This sets the exit flag to true. The handler-method will be left after
    this flag has been set.
  *)
  END Exit;

  PROCEDURE (d : Display) [ABSTRACT] Close*;
  (**
    This closes the display. Call this method at the end of your program.
  *)
  END Close;

  (* ------------ Window ----------------- *)

  PROCEDURE (w : Window) IsOpen*():BOOLEAN;

  BEGIN
    RETURN w.open;
  END IsOpen;

  PROCEDURE (w : Window) SetParent*(parent : Window);
  (**
    Sets the parent window of the current window. VisualOberon will try to
    make use of the supplied information about the hierachical structure of
    the window tree.

    NOTE
    Parent will only be evaluated before first opening.

  *)

  BEGIN
    IF ~w.IsOpen() THEN
      w.parent:=parent;
    END;
  END SetParent;

  PROCEDURE (w : Window) GetParent*():Window;

  BEGIN
    RETURN w.parent;
  END GetParent;

  PROCEDURE (w : Window) GetTopWindow*():Window;

  VAR
    window : Window;

  BEGIN
    IF w.parent=NIL THEN
      RETURN NIL;
    ELSE
      window:=w;
      WHILE window.parent#NIL DO
        window:=window.parent;
      END;
      RETURN window;
    END;
  END GetTopWindow;

  PROCEDURE (w : Window) GetTitle*():U.Text;

  BEGIN
    RETURN w.title;
  END GetTitle;

  PROCEDURE (w : Window) SetBackground*(color : Color);
  (**
    Sets the background color.
  *)

  BEGIN
    w.background:=color;
  END SetBackground;

  PROCEDURE (w : Window) Borderless*(borderless : BOOLEAN);
  (**
    Make the window borderless.

    NOTE
    Only evaluated before first opening.
  *)

  BEGIN
    w.borderless:=borderless;
  END Borderless;

  PROCEDURE (w : Window) SetSize*(width,height : LONGINT);
  (**
    Sets the width and height of the window.

    NOTE
    Does only work, if the windows is not open. If the window  is open,
    use Window.Resize instead.
  *)

  BEGIN
    IF ~w.IsOpen() THEN
      w.width:=width;
      w.height:=height;
    END;
  END SetSize;

  PROCEDURE (w : Window) SetMinSize*(width,height : LONGINT);
  (**
    Sets the minimal width and height of the window.

    NOTE
    Does only work, if the windows is not open. If the window  is open,
    use Window.Resize instead.
  *)

  BEGIN
    IF ~w.IsOpen() THEN
      IF width>=0 THEN
        w.minWidth:=width;
      END;
      IF height>=0 THEN
        w.minHeight:=height;
      END;
    END;
  END SetMinSize;

  PROCEDURE (w : Window) SetMaxSize*(width,height : LONGINT);
  (**
    Sets the maximal width and height of the window.

    NOTE
    Does only work, if the windows is not open. If the window  is open,
    use Window.Resize instead.
  *)

  BEGIN
    IF ~w.IsOpen() THEN
      IF width>=0 THEN
        w.maxWidth:=width;
      END;
      IF height>=0 THEN
        w.maxHeight:=height;
      END;
    END;
  END SetMaxSize;

  PROCEDURE (w : Window) SetPos*(x,y : LONGINT);
  (**
    Sets the top left of the window.

    NOTE
    Does only work, if the windows is not open. If the window  is open,
    use Window.Resize instead.
  *)

  BEGIN
    IF ~w.IsOpen() THEN
      w.x:=x;
      w.y:=y;
    END;
  END SetPos;

  PROCEDURE (w : Window) SetPosition*(horiz, vert : LONGINT);
  (**
    Set the position modes for vertical and horizotal positioning of the window
    on the display. If no modes are explicitly set, the x,y position of the
    window will be used. This position is the position set with SetPos
    or defaults to zero. In this case the windowmanager might position the
    window (it might, too, if you set a mode, since a windowmanager may ignore
    these values).
  *)

  BEGIN
    w.horizontalPos:=horiz;
    w.verticalPos:=vert;
  END SetPosition;

  PROCEDURE (w : Window) AddFreeList*(object : O.Object);
  (**
    Using this function you can link object to the window. If the window gets
    deleted (you must call Window.Delete when the window is not used anymore!)
    all Object.Free gets called for all objects in the freelist. This way you
    can asure that objects that allocate OS ressources (like bitmaps, colors,
    fonts etc...) free this ressources.

    The order of calls to Free for the registered objects is not determinated.
    Also can cannot be sure that Free gets called only ones.Likely Free gets
    called every time after the window gets closed. So you should allocate
    your ressources and register yourself in GObject.CalcSize to asure reentrace.
    Object. Free is currently not called on program exit! You can call
    AddFreeList more than once.

    You can get a pointer to the window via Display.currentWin while in
    Object.CalcSize.
  *)

  VAR
    entry : FreeEntry;

  BEGIN
    entry:=w.freeList;
    WHILE (entry#NIL) & (entry.object#object) DO
      entry:=entry.next;
    END;

    IF (entry#NIL) & (entry.object=object) THEN
      RETURN;
    END;

    NEW(entry);
    entry.object:=object;
    entry.next:=w.freeList;
    w.freeList:=entry;
  END AddFreeList;

  PROCEDURE (w : Window) PreInit*;
  (**
    This method will be called on every call to Open().

    This allows delayed creation of the object-hierachie within the window.

    NOTE
    You must call the method of the baseclass if you overload this method.
  *)

  BEGIN
  END PreInit;

  (* ------------ Handlers for window-events -------------- *)

  PROCEDURE (w : Window) ClosePressed*;
  (**
    Called when the the closing gadget of the window got pressed.
  *)

  BEGIN
  END ClosePressed;

  PROCEDURE (w : Window) Unmaped*;
  (**
    Called, when window has been unmaped.
    Call baseclass method, when you inherit.
  *)

  BEGIN
    w.maped:=FALSE;
  END Unmaped;

  PROCEDURE (w : Window) IsMaped*():BOOLEAN;

  BEGIN
    RETURN w.maped;
  END IsMaped;

  PROCEDURE (w : Window) ReinitWindow*;
  (**
    Called, when the display want the window to reinit itself, f.e
    when the preferences of some or all of the objects has been changed.
  *)
  BEGIN
  END ReinitWindow;

  PROCEDURE (w : Window) Hidden*;
  (**
    This method get calls, when the window becomes (partial) hidden, e.g.
    gets covered by another window. This is likely to not be supported for all
    platforms.
  *)

  BEGIN
  END Hidden;

  PROCEDURE (w : Window) Left*;
  (**
    Gets called, when the mouse (not the focus!) has left the window.
  *)

  BEGIN
  END Left;

  PROCEDURE (w : Window) Redraw*(x,y,width,height : LONGINT);
  (**
    Will be called, if you have to redraw yourself.
    Overload it as aproximate. The window can restrict the redrawing to the given
    area.
  *)
  BEGIN
  END Redraw;

  PROCEDURE (w : Window) GetDrawInfo*():DrawInfo;

  BEGIN
    RETURN NIL;
  END GetDrawInfo;

  PROCEDURE (w : Window) Resized*(width,height : LONGINT);
  (**
    This method gets called, when the window has been resized.
    The given size is the size of the window-inner, that means,
    the size of the window without its borders.

    Note, that window#decorationwindow!
  *)

  BEGIN
  END Resized;

  PROCEDURE (w : Window) ContextHelp*;
  (**
    This method gets called, when the display thinks you should
    open a tooltip help.
  *)

  BEGIN
  END ContextHelp;

  PROCEDURE (w : Window) ContextMenu*():BOOLEAN;
  (**
    This method gets called, when the display things you should
    open a context sensitiv menu.

    RESULT
    Return TRUE if you have opened a context menu, else FALSE.

    NOTE
    If this method returns FALSE, the display will propagate
    the corresponding event that started contextsensitive help
    to the window.
  *)

  BEGIN
    RETURN FALSE;
  END ContextMenu;

  PROCEDURE (w : Window) FocusNext*;
  (**
    This one gets called, when the display thinks the window should
    change the KeyboardFocus.
  *)

  BEGIN
  END FocusNext;

  PROCEDURE (w : Window) FocusObjectHides*;
  (**
    This one gets called by the focus object, if it hides.
  *)

  BEGIN
  END FocusObjectHides;

  PROCEDURE (w : Window) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):Object;
  (**
    Returns the object that coveres the given point and that supports
    dragging of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.

    NOTE
    A window can be sure to be called only, if dragging or dropping makes sense.
    For example, you cannot drop data onto a window that is block due to its
    modal count. However, you can always drag data from a window.
  *)

  BEGIN
    RETURN NIL;
  END GetDnDObject;

  PROCEDURE (w : Window) Exit*;
  (**
    This sets the exit flag to true. The handler-method will be left after
    this flag has been set.
  *)

  BEGIN
  END Exit;

  PROCEDURE (w : Window) IsInEventLoop*():BOOLEAN;

  BEGIN
    RETURN FALSE;
  END IsInEventLoop;

  PROCEDURE (w : Window) Init*;
  (**
    Initialize an instance of the window class.

    NOTE
    You must call this before using a window.
  *)

  BEGIN
    w.Init^;

    NEW(w.title,1);
    COPY("",w.title^);

    w.parent:=NIL;

    w.x:=0;
    w.y:=0;

    w.width:=0;
    w.height:=0;

    w.minWidth:=0;
    w.minHeight:=0;

    w.maxWidth:=MAX(LONGINT);
    w.maxHeight:=MAX(LONGINT);

    w.horizontalPos:=osPos;
    w.verticalPos:=osPos;

    w.background:=backgroundColor;

    w.freeList:=NIL;

    w.maped:=FALSE;
    w.focus:=FALSE;
    w.borderless:=FALSE;
    w.open:=FALSE;
  END Init;

  PROCEDURE (w : Window) SetTitle*(name : ARRAY OF CHAR);
  (**
    Sets the title of the window.

    NOTE
    Value will only be used before first opening.
  *)

  BEGIN
    NEW(w.title,str.Length(name)+1);
    COPY(name,w.title^);
  END SetTitle;

  PROCEDURE (w : Window) Resize*(width,height : LONGINT);
  (**
    Resize the window to the given size. Be carefull to not resize the window
    below the minimal bounds of the top object. Best is to leave resize handling
    completely to the derived window class.

    NOTE
    Derived class should size against minimal and maximal size of its top object.
  *)

  BEGIN
    w.width:=width;
    w.height:=height;
  END Resize;

  PROCEDURE (w : Window)[ABSTRACT] Grab*(grab : BOOLEAN);
  (**
    Do mouse and keyboard-grabing.

    NOTE
    Changing the value does only work, when the window is not visible.
  *)
  END Grab;

  PROCEDURE (w : Window) Open*;
  (**
    Opens the window.

    NOTE
    Derived classes must call th e baseclass method.
  *)

  BEGIN
    w.PreInit;

    w.open:=TRUE;
    w.maped:=FALSE;
  END Open;

  PROCEDURE (w : Window) Close*;
  (**
    Removes the window from the list of windows known by the
    Display.

    NOTE
    You must call this method before closing the window.
  *)

  VAR
    entry : FreeEntry;

  BEGIN
    w.open:=FALSE;

    entry:=w.freeList;
    WHILE entry#NIL DO
      entry.object.Free;
      entry:=entry.next;
    END;
    w.freeList:=NIL;
  END Close;

  PROCEDURE (w : Window)[ABSTRACT] GetXY*(VAR x,y : LONGINT);
  (**
    Convert the window relative x,y coords to absolute coords.
  *)
  END GetXY;

  PROCEDURE (w : Window) Maped*;
  (**
    Called, when window has been maped.
    Call baseclass method, when you inherit.
  *)

  BEGIN
    w.maped:=TRUE;
  END Maped;

  PROCEDURE (w : Window)[ABSTRACT] GetMousePos*(VAR rx, ry, wx, wy : LONGINT);
  (**
    Returns the mouse position in window relative and abslute coords.
  *)
  END GetMousePos;

  PROCEDURE (w : Window) FocusIn*;
  (**
    Called, when the window gets the keyboard focus.
  *)

  BEGIN
    w.focus:=TRUE;
  END FocusIn;

  PROCEDURE (w : Window) FocusOut*;
  (**
    Called, when the window looses the keyboard focus.
  *)

  BEGIN
    w.focus:=FALSE;
  END FocusOut;

  PROCEDURE (w : Window)[ABSTRACT] HandleEvent*(event : E.Event):BOOLEAN;
  (**
    If you derive yourself from window and want to overload the
    defaulthandler, call the baseclass first in your handler and check the
    result, if its is TRUE the defaulthandler has allready handled it.
  *)
  END HandleEvent;

  PROCEDURE (w : Window)[ABSTRACT] EventLoop*;
  (**
    This implements a local event loop for a modal window. The method will
    return, when the window has been closed.
  *)
  END EventLoop;

  PROCEDURE (h : Msg2Exit) Convert*(message : O.Message):O.Message;
  (**
    Converts any incomming message to a ExitMsg.
  *)

  VAR
    new : ExitMsg;

  BEGIN
    NEW(new);
    RETURN new;
  END Convert;

  PROCEDURE GetColorByIndex*(index : LONGINT):LONGINT;

  BEGIN
    CASE index OF
      backgroundColorIndex:
      RETURN backgroundColor;
    | tableBackgroundColorIndex:
      RETURN tableBackgroundColor;
    | tableTextColorIndex:
      RETURN tableTextColor;
    | textBackgroundColorIndex:
      RETURN textBackgroundColor;
    | buttonBackgroundColorIndex:
      RETURN buttonBackgroundColor;
    | textColorIndex:
      RETURN textColor;
    | shineColorIndex:
      RETURN shineColor;
    | halfShineColorIndex:
      RETURN halfShineColor;
    | halfShadowColorIndex:
      RETURN halfShadowColor;
    | shadowColorIndex:
      RETURN shadowColor;
    | fillColorIndex:
      RETURN fillColor;
    | fillTextColorIndex:
      RETURN fillTextColor;
    | warnColorIndex:
      RETURN warnColor;
    | disabledColorIndex:
      RETURN disabledColor;
    | focusColorIndex:
      RETURN focusColor;
    | blackColorIndex:
      RETURN blackColor;
    | whiteColorIndex:
      RETURN whiteColor;
    | helpBackgroundColorIndex:
      RETURN helpBackgroundColor;
    END;
  END GetColorByIndex;

  PROCEDURE GetColorByName*(name : ARRAY OF CHAR):LONGINT;

  VAR
    x,pos : LONGINT;

  BEGIN
    x:=0;
    pos:=-1;
    WHILE (x<=colorCount-1) & (pos=-1) DO
      IF name=colorNames[x] THEN
        pos:=x;
      END;

      INC(x);
    END;

    IF pos>=0 THEN
      RETURN GetColorByIndex(pos);
    END;

    RETURN -1;
  END GetColorByName;

  PROCEDURE SetFactory*(f : Factory);

  BEGIN
    ASSERT((factory=NIL) & (display=NIL));

    factory:=f;
    display:=factory.CreateDisplay();
    display.Init;
  END SetFactory;

BEGIN
  factory:=NIL;

  disablePattern[0]:=CHR(170);  (* 10101010 *)
  disablePattern[1]:=CHR(85);   (* 01010101 *)

  sPointLine[0]:=CHR(1);
  sPointLine[1]:=CHR(1);

  mPointLine[0]:=CHR(2);
  mPointLine[1]:=CHR(2);

  lPointLine[0]:=CHR(3);
  lPointLine[1]:=CHR(3);

  backgroundColor       := MIN(Color);
  tableBackgroundColor  := MIN(Color);
  tableTextColor        := MIN(Color);
  textBackgroundColor   := MIN(Color);
  buttonBackgroundColor := MIN(Color);
  textColor             := MIN(Color);
  shineColor            := MIN(Color);
  halfShineColor        := MIN(Color);
  halfShadowColor       := MIN(Color);
  shadowColor           := MIN(Color);
  fillColor             := MIN(Color);
  fillTextColor         := MIN(Color);
  warnColor             := MIN(Color);
  disabledColor         := MIN(Color);
  focusColor            := MIN(Color);
  blackColor            := MIN(Color);
  whiteColor            := MIN(Color);
  helpBackgroundColor   := MIN(Color);
                
  (* we should assign the constants, but since it is only a temporary solution... *)

  tinyFont:=NIL;
  scriptFont:=NIL;
  footnoteFont:=NIL;
  smallFont:=NIL;
  normalFont:=NIL;
  largeFont:=NIL;
  LargeFont:=NIL;
  LARGEFont:=NIL;
  hugeFont:=NIL;
  HUGEFont:=NIL;

  smallFixedFont:=NIL;
  fixedFont:=NIL;
  hugeFixedFont:=NIL;

  colorNames[backgroundColorIndex]      :="background";
  colorNames[tableBackgroundColorIndex] :="tableBackground";
  colorNames[tableTextColorIndex]       :="tableText";
  colorNames[textBackgroundColorIndex]  :="textBackground";
  colorNames[buttonBackgroundColorIndex]:="buttonBackground";
  colorNames[textColorIndex]            :="text";
  colorNames[shineColorIndex]           :="shine";
  colorNames[shadowColorIndex]          :="shadow";
  colorNames[fillColorIndex]            :="fill";
  colorNames[fillTextColorIndex]        :="fillText";
  colorNames[halfShineColorIndex]       :="halfShine";
  colorNames[halfShadowColorIndex]      :="halfShadow";
  colorNames[warnColorIndex]            :="warn";
  colorNames[disabledColorIndex]        :="disabled";
  colorNames[focusColorIndex]           :="focus";
  colorNames[blackColorIndex]           :="black";
  colorNames[whiteColorIndex]           :="white";
  colorNames[helpBackgroundColorIndex]  :="helpBackground";

  fontNames[tinyFontIndex]      :="tiny";
  fontNames[scriptFontIndex]    :="script";
  fontNames[footnoteFontIndex]  :="footnote";
  fontNames[smallFontIndex]     :="small";
  fontNames[normalFontIndex]    :="normal";
  fontNames[largeFontIndex]     :="large";
  fontNames[LargeFontIndex]     :="Large";
  fontNames[LARGEFontIndex]     :="LARGE";
  fontNames[hugeFontIndex]      :="huge";
  fontNames[HUGEFontIndex]      :="HUGE";
  fontNames[smallFixedFontIndex]:="smallFixed";
  fontNames[fixedFontIndex]     :="fixed";
  fontNames[hugeFixedFontIndex] :="hugeFixed";

  prefsCallback:=NIL;
END VO:Base:Display.