MODULE VO:EditText;

IMPORT O   := VO:Base:Object,
       U   := VO:Base:Util,

       R   := VO:EditRun,
       S   := VO:EditSyntaxParser,


              Ascii,
              Err,
       F   := Filenames,  (* depreciated! *)
       str := Strings;

CONST
  cursorName     * = "Cursor";
  blockStartName * = "Block start";
  blockEndName   * = "Block end";

TYPE
  (* refreshing messages *)

  RedrawMsg*     = POINTER TO RedrawMsgDesc;
  RedrawMsgDesc* = RECORD (O.ResyncMsgDesc)
                     from*, to* : LONGINT;
                   END;

  MarkMsg*     = POINTER TO MarkMsgDesc;
  MarkMsgDesc* = RECORD (O.ResyncMsgDesc)
                   mark*  : R.Mark;
                   oldX*,
                   oldY*,
                   newX*,
                   newY*  : LONGINT;
                 END;

  DeleteLinesMsg*      = POINTER TO DeleteLinesMsgDesc;
  DeleteLinesMsgDesc * = RECORD (O.ResyncMsgDesc)
                           from*,
                           count* : LONGINT;
                         END;

  InsertLinesMsg*      = POINTER TO InsertLinesMsgDesc;
  InsertLinesMsgDesc * = RECORD (O.ResyncMsgDesc)
                           from*,
                           count* : LONGINT;
                         END;

  (* buffer *)

  Buffer*     = POINTER TO BufferDesc;
  BufferDesc* = RECORD (O.ModelDesc)
                  first*   : R.LineRun;
                  lines*   : LONGINT;
                  name-    : POINTER TO ARRAY OF CHAR;
                END;

  MarkEntry     = POINTER TO MarkEntryDesc;
  MarkEntryDesc = RECORD
                    last,
                    next  : MarkEntry;
                    mark  : R.Mark;
                  END;

  (* text *)

  Text*     = POINTER TO TextDesc;
  TextDesc* = RECORD (BufferDesc)
               cursor-,
               markA-,
               markB-    : R.Mark; (* Cursor, block start and block end marks *)
               parser-   : S.SyntaxParser;
               markList  : MarkEntry; (* list of marks *)
               fileName- : POINTER TO ARRAY OF CHAR;
             END;

VAR
  buffer* : Text;

  (**
    Initialize the buffer
  **)

  PROCEDURE (b : Buffer) Init*;

  BEGIN
    NEW(b.first);
    b.first.Init;
    b.lines:=1;
    b.first.next:=NIL;
    b.first.last:=NIL;

    NEW(b.name,1);
    COPY("",b.name^);
  END Init;

  (**
    Set the name of the buffer.
  **)

  PROCEDURE (b : Buffer) SetName*(name : ARRAY OF CHAR);

  VAR
    length : LONGINT;

  BEGIN
    length:=str.Length(name);
    NEW(b.name,length+1);
    COPY(name,b.name^);
  END SetName;

  (**
    Create a (temporary) copy of the text within the buffer.

    NOTE
    The text will be copied without the first leading LineRun.
  **)

  PROCEDURE (b : Buffer) Copy*():R.Run;

  VAR
    first,new,run,last : R.Run;

  BEGIN
    run:=b.first.next;
    IF run#NIL THEN
      first:=run.Copy();
      last:=first;
      last.last:=NIL;
      last.next:=NIL;

      run:=run.next;
      WHILE run#NIL DO
        new:=run.Copy();
        new.last:=last;
        last.next:=new;
        last:=new;
        run:=run.next;
      END;
      last.next:=NIL;

      RETURN first;
    ELSE
      RETURN NIL;
    END;
  END Copy;

  (**
    Assign the given text to the buffer. Old text will be deleted.
  **)

  PROCEDURE (b : Buffer) Assign*(text : R.LineRun);

  VAR
    next : R.LineRun;

  BEGIN
    b.first:=text;
    b.lines:=1;

    next:=b.first.NextLine();
    WHILE next#NIL DO
      INC(b.lines);
      next:=next.NextLine();
    END;
  END Assign;

  (**
    Reinitialize the buffer and delete existing text.
  **)

  PROCEDURE (b : Buffer) New*;

  BEGIN
    b.Init;
  END New;

  (**

  **)

  PROCEDURE (b : Buffer) GetText*():U.Text;

  VAR
    text   : U.Text;
    start  : R.Run;
    length,
    x,y    : LONGINT;

  BEGIN
    (* calculate length of text for correct allocation of buffer *)

    length:=0;
    start:=b.first.next; (* We do not save the starting newline *)
    WHILE start#NIL DO
      WITH
        start : R.TextRun DO
          INC(length,start.length);
      | start : R.LineRun DO
          INC(length,1);
      ELSE
      END;
      start:=start.next;
    END;

    NEW(text,length+1);
    text[length]:=0X;

    x:=0;
    start:=b.first.next; (* We do not save the starting newline *)
    WHILE start#NIL DO
      WITH
        start : R.TextRun DO
          FOR y:=start.pos TO start.pos+start.length-1 DO
            text[x]:=start.block.text[y];
            INC(x);
          END;
      | start : R.LineRun DO
          text[x]:=Ascii.lf;
          INC(x);
      ELSE
      END;
      start:=start.next;
    END;

    RETURN text;
  END GetText;

  (**
    Set the filename of the text.
  **)

  PROCEDURE (t : Text) SetFileName*(name : ARRAY OF CHAR);

  VAR
    length : LONGINT;

  BEGIN
    length:=str.Length(name);
    NEW(t.fileName,length+1);
    COPY(name,t.fileName^);
  END SetFileName;

  (**
    Automatically generate a buffer name from the file name.
    The buffer name will be the basename of the file.

    NOTE
    This method uses the depreciated "Filenames" module to cut away
    the filename part.
  **)

  PROCEDURE (t : Text) FileToBufferName*;

  VAR
    path,file : POINTER TO ARRAY OF CHAR;

  BEGIN
    NEW(path,LEN(t.fileName^));
    NEW(file,LEN(t.fileName^));

    F.GetPath(t.fileName^,path^,file^);

    t.SetName(file^);
  END FileToBufferName;

  (**
    Add the given mark to the list of marks stored within the text.
  **)

  PROCEDURE (t : Text) AddMark(mark : R.Mark);

  VAR
    entry : MarkEntry;

  BEGIN
    NEW(entry);
    entry.mark:=mark;
    entry.last:=NIL;
    entry.next:=t.markList;
    IF t.markList#NIL THEN
      t.markList.last:=entry;
    END;
    t.markList:=entry;
  END AddMark;

  (**
    Remove a previous added mark.
  **)

  PROCEDURE (t : Text) RemoveMark(mark : R.Mark);

  VAR
    entry : MarkEntry;

  BEGIN
    IF t.markList.mark=mark THEN
      t.markList:=t.markList.next;
      IF t.markList#NIL THEN
        t.markList.last:=NIL;
      END;
    ELSE
      entry:=t.markList.next;
      WHILE entry.mark#mark DO
        entry:=entry.next;
      END;

      entry.last.next:=entry.next;

      IF entry.next#NIL THEN
        entry.next.last:=entry.last;
      END;
    END;
  END RemoveMark;

  (**
    Correct y position of all marks in list that are positioned behind
    the reference mark
  **)

  PROCEDURE (t : Text) CorrectMarkY(afterX, afterY, offset : LONGINT);

  VAR
    entry : MarkEntry;

  BEGIN
    entry:=t.markList;
    WHILE entry#NIL DO
      IF ((entry.mark.y>afterY) OR
         ((entry.mark.y=afterY) & (entry.mark.x>afterX))) THEN
        INC(entry.mark.y,offset);
      END;
      entry:=entry.next;
    END;
  END CorrectMarkY;

  (**
    Set a mark at the given position with the handed name and type.
    The mark wil be automatically added to the list of marks of the corresponding
    textobject.
  **)

  PROCEDURE (t : Text) SetMark*(x,y : LONGINT; name : ARRAY OF CHAR; type : LONGINT):R.Mark;

  VAR
    redrawMsg : RedrawMsg;
    mark      : R.Mark;

  BEGIN
    mark:=t.cursor.SetMark(x,y);
    IF mark#NIL THEN
      mark.SetName(name);
      mark.SetType(type);
      t.AddMark(mark);
      NEW(redrawMsg);
      redrawMsg.from:=y;
      redrawMsg.to:=redrawMsg.from;
      RETURN mark;
    ELSE
      RETURN NIL;
    END;
  END SetMark;

  (**
    Move the given mark within the text.
  **)

  PROCEDURE (t : Text) MoveMark*(mark : R.Mark; x,y : LONGINT):BOOLEAN;

  VAR
    markMsg : MarkMsg;
    oldX,
    oldY    : LONGINT;

  BEGIN
    ASSERT(mark#NIL);

    oldX:=mark.x;
    oldY:=mark.y;
    IF mark.Move(x,y) THEN
      NEW(markMsg);
      markMsg.mark:=mark;
      markMsg.oldX:=oldX;
      markMsg.oldY:=oldY;
      markMsg.newX:=mark.x;
      markMsg.newY:=mark.y;
      t.Notify(markMsg);
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END MoveMark;

  (**
    Move the cursor to the given position.
  **)

  PROCEDURE (t : Text) MoveCursor*(x,y : LONGINT):BOOLEAN;

  BEGIN
    RETURN t.MoveMark(t.cursor,x,y);
  END MoveCursor;

  (**
    (Re)initalizes cursor.
  **)

  PROCEDURE (t : Text) InitCursor*;

  BEGIN
    IF t.cursor#NIL THEN
      t.cursor.Remove;
      t.RemoveMark(t.cursor);
    ELSE
      NEW(t.cursor);
    END;
    t.cursor.Init;
    t.cursor.x:=1;
    t.cursor.y:=1;
    t.cursor.SetName(cursorName);
    t.cursor.SetType(R.cursor);
    t.AddMark(t.cursor);
    t.first.InsertEntry(t.cursor);
  END InitCursor;

  (**
    Initialize the text.
  **)

  PROCEDURE (t : Text) Init*;

  BEGIN
    t.Init^;

    t.SetFileName("");

    t.markList:=NIL;
    t.cursor:=NIL;
    t.markA:=NIL;
    t.markB:=NIL;
    t.InitCursor;

    t.parser:=S.GetParser(t.name^,t.fileName^,t.first);
  END Init;

  PROCEDURE (t : Text) EvaluateParser*;

  VAR
    parser : S.SyntaxParser;

  BEGIN
    parser:=S.GetParser(t.name^,t.fileName^,t.first);

    IF parser#t.parser THEN
      t.parser:=parser;
      (* Refresh??? *)
    END;
  END EvaluateParser;

  (**
    Assign the given text to the textobject.
  **)

  PROCEDURE (t : Text) Assign*(text : R.LineRun);

  BEGIN
    t.Assign^(text);

    t.InitCursor;
  END Assign;

  (**
    Delete the text stored within the textobject.
  **)

  PROCEDURE (t : Text) New*;

  BEGIN
    t.Init;
  END  New;

  (**
    Utility method. Appends the given ammount of spaces at the end of the line.

    This method does not send an update message nor does it update the position
    of marks in the line because it makes no sense.
  **)

  PROCEDURE (t : Text) AppendSpaces(line : R.LineRun; count : LONGINT);

  VAR
    current,
    text      : R.TextRun;
    pos,
    length    : LONGINT;

  BEGIN
    text:=R.CreateTextRunChars(" ",count);

    length:=line.Length();

    line:=t.cursor.line;
    line.GetPos(length+1,current,pos);

    IF current#NIL THEN
      current:=current.Split(pos);
      current.InsertAfter(text);
    ELSE
      line.InsertAfter(text);
    END;

    line.Nice;
  END AppendSpaces;

  (**
    Insert a character at the position given by the mark.
  **)

  PROCEDURE (t : Text) InsertCharAtMark*(a : R.Mark; char : CHAR);

  VAR
    current,
    text      : R.TextRun;
    line      : R.LineRun;
    pos       : LONGINT;
    redrawMsg : RedrawMsg;
    entry     : R.LineEntry;

  BEGIN
    IF a.x>a.line.Length()+1 THEN
      t.AppendSpaces(a.line,a.x-a.line.Length()-1);
    END;

    text:=R.CreateTextRunChar(char);

    line:=a.line;
    line.GetPos(a.x,current,pos);

    IF current#NIL THEN
      current:=current.Split(pos);
      current.InsertAfter(text);
    ELSE
      line.InsertAfter(text);
    END;

    line.Nice;

    pos:=a.x;
    entry:=line.first;
    WHILE entry#NIL DO
      IF entry.x>=pos THEN
        INC(entry.x);
      END;
      entry:=entry.next;
    END;

    NEW(redrawMsg);
    redrawMsg.from:=a.y;
    redrawMsg.to:=redrawMsg.from;

    t.Notify(redrawMsg);
  END InsertCharAtMark;

  (**
    Insert a string at the position given by the mark.
  **)

  PROCEDURE (t : Text) InsertStringAtMark*(a : R.Mark; string : ARRAY OF CHAR);

  VAR
    current,
    text      : R.TextRun;
    line      : R.LineRun;
    pos       : LONGINT;
    redrawMsg : RedrawMsg;
    entry     : R.LineEntry;

  BEGIN
    IF a.x>a.line.Length()+1 THEN
      t.AppendSpaces(a.line,a.x-a.line.Length()-1);
    END;

    text:=R.CreateTextRunString(string,str.Length(string));

    line:=a.line;
    line.GetPos(a.x,current,pos);

    IF current#NIL THEN
      current:=current.Split(pos);
      current.InsertAfter(text);
    ELSE
      line.InsertAfter(text);
    END;

    line.Nice;

    pos:=a.x;
    entry:=line.first;
    WHILE entry#NIL DO
      IF entry.x>=pos THEN
        INC(entry.x);
      END;
      entry:=entry.next;
    END;

    NEW(redrawMsg);
    redrawMsg.from:=a.y;
    redrawMsg.to:=redrawMsg.from;

    t.Notify(redrawMsg);
  END InsertStringAtMark;

  (**
    Insert the character given at the cursor position.
  **)

  PROCEDURE (t : Text) InsertCharAtCursor*(char : CHAR);

  BEGIN
    t.InsertCharAtMark(t.cursor,char);
  END InsertCharAtCursor;

  (**
    Split the line at the position given by the mark.
  **)

  PROCEDURE (t : Text) SplitLine*(mark : R.Mark);

  VAR
    line      : R.LineRun;
    redrawMsg : RedrawMsg;
    insertMsg : InsertLinesMsg;
    y         : LONGINT;

  BEGIN
    line:=mark.line;
    y:=mark.y;

    (*
      Correct y position of all marks in list that are positioned behind
      the reference mark
    *)
    t.CorrectMarkY(mark.x-1,mark.y,1);

    line.Split(mark.x);
    INC(t.lines);

    NEW(insertMsg);
    insertMsg.from:=mark.y;
    insertMsg.count:=1;
    t.Notify(insertMsg);

    NEW(redrawMsg);
    redrawMsg.from:=mark.y-1;
    redrawMsg.to:=redrawMsg.from;
    t.Notify(redrawMsg);
  END SplitLine;

  (* -------- blocks --------------- *)

  (**
    Switch block, if block mark two comes before block mark one.
  **)

  PROCEDURE (t : Text) CheckBlockMarkOrder;

  VAR
    help : R.Mark;

  BEGIN
    IF (t.markA#NIL) & (t.markB#NIL) THEN
      IF (t.markA.y>t.markB.y) OR ((t.markA.y=t.markB.y) & (t.markA.x>t.markB.x)) THEN
        help:=t.markB;
        t.markB:=t.markA;
        t.markA:=help;
      END;
    END;
  END CheckBlockMarkOrder;

  (**
    Update the block information for the syntax parser.
  **)

  PROCEDURE (t : Text) UpdateBlockInfo;

  BEGIN
    t.CheckBlockMarkOrder;
    t.parser.markA:=t.markA;
    t.parser.markB:=t.markB;
  END UpdateBlockInfo;

  (**
    Return information about the block mode.

    0 - no block.
    1 - first mark set.
    2 - block set.
  **)

  PROCEDURE (t : Text) IsBlockSet*():INTEGER;

  BEGIN
    IF (t.markA#NIL) & (t.markB#NIL) THEN
      RETURN 2;
    ELSIF (t.markA#NIL) THEN
      RETURN 1;
    ELSE
      RETURN 0;
    END;
  END IsBlockSet;

  (**
    (Re)set the start of the block.
  **)

  PROCEDURE (t : Text) SetBlockStart*(x,y : LONGINT);

  VAR
    redrawMsg : RedrawMsg;

  BEGIN
    NEW(redrawMsg);
    IF (t.markA#NIL) & (t.markB#NIL) THEN
      redrawMsg.from:=t.markA.y;
      redrawMsg.to:=t.markB.y;
    ELSIF t.markA#NIL THEN
      redrawMsg.from:=t.markA.y;
      redrawMsg.to:=redrawMsg.from;
    END;

    IF t.markA#NIL THEN
      t.markA.Remove;
    END;

    t.markA:=t.SetMark(x,y,blockStartName,R.block);
    IF t.markA.x>t.markA.line.Length()+1 THEN
      IF t.markA.Move(t.markA.line.Length()+1,t.markA.y) THEN END;
    END;

    t.UpdateBlockInfo;

    IF t.markA.y<redrawMsg.from THEN
      redrawMsg.from:=t.markA.y;
    END;
    IF t.markB#NIL THEN
      IF t.markB.y>redrawMsg.to THEN
        redrawMsg.to:=t.markB.y;
      END;
    END;

    t.Notify(redrawMsg);
  END SetBlockStart;

  (**
    (Re)set the end of the block.
  **)

  PROCEDURE (t : Text) SetBlockEnd*(x,y : LONGINT);

  VAR
    redrawMsg : RedrawMsg;

  BEGIN
    ASSERT(t.markA#NIL);

    NEW(redrawMsg);
    IF t.markB#NIL THEN
      redrawMsg.from:=t.markA.y;
      redrawMsg.to:=t.markB.y;
    ELSE
      redrawMsg.from:=t.markA.y;
      redrawMsg.to:=redrawMsg.from;
    END;

    IF t.markB#NIL THEN
      t.markB.Remove;
    END;

    t.markB:=t.SetMark(x,y,blockEndName,R.block);
    IF t.markB.x>t.markB.line.Length()+1 THEN
      IF t.markB.Move(t.markB.line.Length()+1,t.markB.y) THEN END;
    END;

    t.UpdateBlockInfo;

    IF t.markA.y<redrawMsg.from THEN
      redrawMsg.from:=t.markA.y;
    END;
    IF t.markB.y>redrawMsg.to THEN
      redrawMsg.to:=t.markB.y;
    END;

    t.Notify(redrawMsg);
  END SetBlockEnd;

  (**
    Remove all block marks.
  **)

  PROCEDURE (t : Text) ClearBlock*;

  VAR
    redrawMsg : RedrawMsg;

  BEGIN
    NEW(redrawMsg);
    IF t.markB#NIL THEN
      t.markB.Remove();
      t.RemoveMark(t.markB);
    END;
    IF t.markA#NIL THEN
      t.markA.Remove();
      t.RemoveMark(t.markA);
    END;

    IF (t.markA#NIL) & (t.markB#NIL) THEN
      redrawMsg.from:=t.markA.y;
      redrawMsg.to:=t.markB.y;
    ELSIF t.markA#NIL THEN
      redrawMsg.from:=t.markA.y;
      redrawMsg.to:=redrawMsg.from;
    END;

    t.markA:=NIL;
    t.markB:=NIL;

    t.UpdateBlockInfo;

    t.Notify(redrawMsg);
  END ClearBlock;

  (**
    Clear the text area marked by the given marks. Assign cutout text to the buffer
    if buffer is !=NULL.
  **)

  PROCEDURE (t : Text) DeleteArea*(a,b : R.Mark):BOOLEAN;

  VAR
    line,
    next      : R.LineRun;
    block,
    end       : R.Run;
    redrawMsg : RedrawMsg;
    deleteMsg : DeleteLinesMsg;
    lines     : LONGINT;
    mark      : MarkEntry;

    c1,c2     : R.TextRun;
    t1,t2     : R.Run;
    p1,p2     : LONGINT;

  BEGIN
    ASSERT((a#NIL) & (b#NIL));

    IF a.Equal(b) & (a.x>a.line.Length()) THEN
      (* delete line *)
      line:=a.line;
      next:=a.line.NextLine();
      IF next#NIL THEN
        next.Remove;
        DEC(t.lines);

        next.next:=NIL;

        IF buffer#NIL THEN
          buffer.Assign(next);
        END;

        mark:=t.markList;
        WHILE mark#NIL DO
          IF (mark.mark.y>a.y) & (mark.mark.line#a.line) THEN
            DEC(mark.mark.y);
          END;
          mark:=mark.next;
        END;

        NEW(deleteMsg);
        deleteMsg.from:=a.y+1;
        deleteMsg.count:=1;
        t.Notify(deleteMsg);

        NEW(redrawMsg);
        redrawMsg.from:=a.y;
        redrawMsg.to:=redrawMsg.from;
        t.Notify(redrawMsg);

        RETURN TRUE;
      ELSE
        RETURN FALSE;
      END;
    ELSE
     (* delete block *)
      lines:=0;

      IF a.x<=a.line.Length() THEN
        a.line.GetPos(a.x,c1,p1);
        t1:=c1.Split(p1);
      ELSE
        t1:=a.line.NextLine();
        t1:=t1.last;
        INC(lines);
      END;

      IF b.x<=b.line.Length() THEN
        b.line.GetPos(b.x+1,c2,p2);
        t2:=c2.Split(p2);
      ELSE
        t2:=b.line.NextLine();
        IF t2#NIL THEN
          INC(lines);
        ELSE
          b.line.GetPos(b.line.Length(),c2,p2);
          t2:=c2.Split(p2);
        END;
      END;

      INC(lines,b.y-a.y);

      mark:=t.markList;
      WHILE mark#NIL DO
        IF (mark.mark#a) & (mark.mark#b) THEN
          IF mark.mark.AfterEqual(a) & mark.mark.BeforeEqual(b) THEN
            (* Marks inthe deleted area are place at the start of the area *)
            mark.mark.RepositionEntry(a.line,a.x);
            mark.mark.y:=a.y;
          ELSIF mark.mark.SameLine(b) & (mark.mark.x< b.x) THEN
            (* After end of block but same line *)
            mark.mark.RepositionEntry(a.line,mark.mark.x-b.x);
            mark.mark.y:=a.y;
          ELSIF mark.mark.y>b.y THEN
            (* After line containing end of block *)
            DEC(mark.mark.y,lines);
          END;
        END;
        mark:=mark.next;
      END;

      (* Correct position of mark b *)
      b.RepositionEntry(a.line,a.x);
      b.y:=a.y;

      block:=t1.next;
      end:=t2;

      t1.Join(t2.next);

      end.next:=NIL;

      a.line.Nice;

      NEW(redrawMsg);
      redrawMsg.from:=a.y;
      redrawMsg.to:=redrawMsg.from;
      t.Notify(redrawMsg);

      IF lines>0 THEN
        NEW(deleteMsg);
        deleteMsg.from:=a.y+1;
        deleteMsg.count:=lines;
        t.Notify(deleteMsg);
      END;

      RETURN TRUE;
    END;
  END DeleteArea;

  (**
    Clear the text area marked by the given marks. Assign cutout text to the buffer
    if buffer is !=NULL.
  **)

  PROCEDURE (t : Text) CopyAreaToBuffer*(a,b : R.Mark; buffer : Buffer):BOOLEAN;

  VAR
    start    : R.LineRun;
    block,
    end,
    new,new2 : R.Run;
    c1,c2    : R.TextRun;
    t1,t2    : R.Run;
    p1,p2    : LONGINT;

  BEGIN
    ASSERT((a#NIL) & (b#NIL));
    ASSERT(buffer#NIL);

    IF ~(a.Equal(b) & (a.x>a.line.Length())) THEN (* cannot copy "nothing" *)

      IF a.x<=a.line.Length() THEN
        a.line.GetPos(a.x,c1,p1);
        t1:=c1.Split(p1);
      ELSE
        t1:=a.line.NextLine();
        t1:=t1.last;
      END;

      IF b.x<=b.line.Length() THEN
        b.line.GetPos(b.x+1,c2,p2);
        t2:=c2.Split(p2);
      ELSE
        t2:=b.line.NextLine();
        IF t2=NIL THEN
          b.line.GetPos(b.line.Length(),c2,p2);
          t2:=c2.Split(p2);
        END;
      END;

      block:=t1.next;
      end:=t2;

      NEW(start);
      start.Init;
      new:=start;
      WHILE block#end DO
        new2:=block.Copy();
        new2.next:=NIL;
        new.InsertAfter(new2);
        new:=new2;

        block:=block.next;
      END;
      new2:=block.Copy();
      new2.next:=NIL;
      new.InsertAfter(new2);

      buffer.Assign(start);

      (*
        We have slited the line to cut out buffer.
        We can join the cuts together again now.
      *)
      a.line.Nice;
      IF a.line#b.line THEN
        b.line.Nice;
      END;

      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END CopyAreaToBuffer;

  (**
    Delete the character at the mark position.
  **)

  PROCEDURE (t : Text) DeleteCharAtMark*(mark : R.Mark):BOOLEAN;

  BEGIN
    RETURN t.DeleteArea(mark,mark);
  END DeleteCharAtMark;

  (**
    Delete the character at the cursor position.
  **)

  PROCEDURE (t : Text) DeleteCharAtCursor*():BOOLEAN;

  BEGIN
    RETURN t.DeleteArea(t.cursor,t.cursor);
  END DeleteCharAtCursor;

  (**
    Insert block at mark position.
  **)

  PROCEDURE (t : Text) InsertBuffer*(mark : R.Mark; buffer : Buffer):BOOLEAN;

  VAR
    text      : R.TextRun;
    line      : R.LineRun;
    run,old,
    new       : R.Run;
    pos,lines : LONGINT;
    insertMsg : InsertLinesMsg;
    redrawMsg : RedrawMsg;

  BEGIN
    IF buffer.first.next=NIL THEN (* pasting empty text *)
      RETURN FALSE;
    END;

    IF mark.x>mark.line.Length() THEN
      mark.line.GetPos(mark.line.Length(),text,pos);
    ELSE
      mark.line.GetPos(mark.x,text,pos);
      text:=text.Split(pos);
    END;

    (* copy buffer *)
    new:=buffer.Copy();

    old:=text.next;
    text.Join(new);
    run:=text;
    lines:=0;
    line:=NIL;
    WHILE run.next#NIL DO
      (* TODO: clear all information *)
      IF run IS R.LineRun THEN
        line:=run(R.LineRun);
        INC(lines);
      END;
      run:=run.next;
    END;

    run.Join(old);

    mark.line.Nice;
    IF line#NIL THEN
      line.Nice;
    END;

    NEW(insertMsg);
    insertMsg.from:=mark.y+1;
    insertMsg.count:=lines;
    t.Notify(insertMsg);

    NEW(redrawMsg);
    redrawMsg.from:=mark.y;
    redrawMsg.to:=redrawMsg.from;
    t.Notify(redrawMsg);

    RETURN TRUE;
  END InsertBuffer;

BEGIN
  NEW(buffer);
  buffer.Init;
  buffer.SetName("*buffer*");
END VO:EditText.