/***************************************************************************
 $RCSfile: adminsegs.cpp,v $
                             -------------------
    cvs         : $Id: adminsegs.cpp,v 1.39 2003/06/10 17:48:41 aquamaniac Exp $
    begin       : Sun Nov 18 2001
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif

#include <stdio.h>

#include "adminsegs.h"
#include "bankimpl.h"
#include "hbcistring.h"
#include "error.h"
#include "mediumrdhbase.h"


namespace HBCI {

/****************************************************************************
 * SEGMessageHead
 ****************************************************************************/

SEGMessageHead::SEGMessageHead(Pointer<Customer> cust) : Seg(cust){
    _msgnumber=0;
    _dialogid="";
    _referenceid=0;
}


SEGMessageHead::~SEGMessageHead(){
}


string SEGMessageHead::toString(int segnr){
    string result;

    _segnumber=segnr;
    // create segment head
    result="HNHBK:1:";
    if (_hbciversion < HBCI_VERSION_220)
        result += "2+";
    else
        result += "3+";
    // message-size (just a dummy, yet we don't know the size)
    result+=String::num2string(0, true, 12) + "+";
    //hbci-version
    result += String::num2string(_hbciversion) + "+";
    //dialog-id
    result += _dialogid + "+";
    // message-number
    result += String::num2string(_msgnumber) + "'";
    // return the result
    return result;
}


bool SEGMessageHead::parse(const string& segment, unsigned int pos){
    // skip segment head
    pos = String::nextDE(segment, 0).length() + 1;
    // skip message size
    pos += String::nextDE(segment, pos).length() + 1;
    // skip HBCI version
    pos += String::nextDE(segment, pos).length() + 1;
    // read dialog id
    _dialogid = String::nextDE(segment, pos).c_str();
    pos += String::nextDE(segment, pos).length() + 1;
    // read message number
    _msgnumber = atoi(String::nextDE(segment, pos).c_str());
    pos += String::nextDE(segment, pos).length() + 1;
    // read reference id if any
    if (pos<segment.length()) {
        _referenceid = atoi(String::nextDE(segment, pos).c_str());
        pos += String::nextDE(segment, pos).length() + 1;
    }
    else
        _referenceid=0;
    return true;
}


void SEGMessageHead::setData(int msgnumber, string dialogid){
    _msgnumber=msgnumber;
    _dialogid=dialogid;
}


void SEGMessageHead::setSize(string &message){
    string s;
    int i;

    i=message.find('+');
    s=String::num2string(message.length(), true, 12);
    message.replace(i+1,12,s);
}



/****************************************************************************
 * SEGMessageTail
 ****************************************************************************/


SEGMessageTail::SEGMessageTail(Pointer<Customer> cust) : Seg(cust){
    _msgnumber=0;
    _segnumber=0;
}


SEGMessageTail::~SEGMessageTail(){
}


string SEGMessageTail::toString(int segnr){
    string result;

    _segnumber=segnr;
    // seg-head
    result="HNHBS:"+String::num2string(_segnumber) + ":1+";
    // message-number
    result+=String::num2string(_msgnumber) + "'";
    return result;
}


bool SEGMessageTail::parse(const string& segment, unsigned int pos){
    // skip segment head
    pos += String::nextDE(segment, pos).length() + 1;
    // read message number
    _msgnumber = atoi(String::nextDE(segment, pos).c_str());
    pos += String::nextDE(segment, pos).length() + 1;
    return true;
}


void SEGMessageTail::setData(int msgnumber){
    _msgnumber=msgnumber;
}



/****************************************************************************
 * SEGSignatureHead
 ****************************************************************************/


SEGSignatureHead::SEGSignatureHead(Pointer<Customer> cust, bool sync)
: Seg(cust)
,_sync(sync)
{
}


SEGSignatureHead::~SEGSignatureHead(){
}


string SEGSignatureHead::toString(int segnr){
    string result;
    Pointer<Medium> medium;
    int secmode;

    _segnumber=segnr;
    // first sample some info we need
    medium=_customer.ref().user().ref().medium();
    secmode=medium.ref().securityMode();
    if (secmode!=HBCI_SECURITY_DDV &&
        secmode!=HBCI_SECURITY_RDH)
        throw Error("SEGSignatureHead::toString()",
                        "Bad security mode.",
                        0);
    _signkeynumber=medium.ref().signKeyNumber();
    _signkeyversion=medium.ref().signKeyVersion();

    // now create the segment
    result="HNSHK:";                             // segment ID
    result+=String::num2string(_segnumber);  // segment number
    result+=":3+";                               // segment version
    /* add security function
     *  1=NRO (Non Repudiation of Origin), use with RDH
     *  2=AUT (Message Origin Authentication), use with DDV
     */
    if (secmode==HBCI_SECURITY_DDV)
        result += "2+"; //2=ddv 1=rdh
    else
        result += "1+"; //2=ddv 1=rdh
    // add sicherheitskontrollreferenz
    result+=_controlref + "+";
    /* add area of encryption
     * 1=SHM : signature head and hbci payload
     * 2=SHT : signature head to signature tail
     * currently according to HBCI ducumentation only the first is allowed
     */
    result+="1+";
    /* add role of deliverer
     * 1=ISS : we are the author of this message
     * 3=CON : signer supports this message (but is not the author)
     * 4=WIT : signer is witness but is not responsible for this message
     * currently HBCI advises not to use this field (simply set it to 1)
     */
    result+="1+";
    /* add security id.*/
    result += "1:"; // 1=we are the sender
    if (secmode==HBCI_SECURITY_RDH)
        // next field left out for RDH
        result+=":";
    // medium id is CID for DDV, system id for RDH
    if (medium.ref().mediumId().empty() || _sync)
      result+="0+";
    else
      result+=medium.ref().mediumId() + "+";

    /* add security reference id. This is used to protect against doubling of
     * message. After each signature this number has to be incremented by one.
     * add the next signatureID.
     */
    result+=String::num2string(medium.ref().nextSEQ())+"+";

    /* add security date and time. the leading one indicates that this is a
     security stamp.*/
    result += "1:";	//datum+uhrzeit
    result += String::date2string() + ":";
    result += String::time2string() + "+";
    /* add description of the used hash algorithm */
    result += "1:999:1+";

    /* add desciption about signature algorithm */
    result += "6:";
    if (secmode==HBCI_SECURITY_RDH)
        result+="10:16+";
    else
        result+="1:999+";

    /* add the keyname (including institute id etc) */
    result+=String::num2string(_bank.ref().countryCode()) + ":"; // country
    result+=_bank.ref().bankCode() + ":"; //bankdaten
    result+=String::escape(_customer.ref().user().ref().userId()) + ":";
    //result+=_customer.ref().custId() + ":";
    result+="S:"; // its the crypt key

    result+=String::num2string(medium.ref().signKeyNumber()) + ":";
    result+=String::num2string(medium.ref().signKeyVersion());

    result+="'";
    return result;
}


bool SEGSignatureHead::parse(const string& segment, unsigned int pos){
    // skip segment head
    pos += String::nextDE(segment, pos).length() + 1;
    // skip security fn
    pos += String::nextDE(segment, pos).length() + 1;
    // skip controll reference
    pos += String::nextDE(segment, pos).length() + 1;
    // skip security area
    pos += String::nextDE(segment, pos).length() + 1;
    // skip role of deliverer
    pos += String::nextDE(segment, pos).length() + 1;
    // skip security id
    pos += String::nextDE(segment, pos).length() + 1;
    // skip seq
    pos += String::nextDE(segment, pos).length() + 1;
    // skip date and time
    pos += String::nextDE(segment, pos).length() + 1;
    // skip hash descr
    pos += String::nextDE(segment, pos).length() + 1;
    // skip algo descr
    pos += String::nextDE(segment, pos).length() + 1;

    // now we have the key name
    // skip country
    pos += String::nextDEG(segment, pos).length() + 1;
    // skip user id
    pos += String::nextDEG(segment, pos).length() + 1;
    // skip "S"
    pos += String::nextDEG(segment, pos).length() + 1;
    // get the key number
    _signkeynumber = atoi(String::nextDEG(segment, pos).c_str());
    pos += String::nextDEG(segment, pos).length() + 1;
    // get the key version
    _signkeyversion = atoi(String::nextDEG(segment, pos).c_str());
    return true;
}


int SEGSignatureHead::signKeyNumber(){
    return _signkeynumber;
}


int SEGSignatureHead::signKeyVersion(){
    return _signkeyversion;
}


void SEGSignatureHead::setData(string &ctrlref){
    _controlref=ctrlref;
}



/****************************************************************************
 * SEGSignatureTail
 ****************************************************************************/


SEGSignatureTail::SEGSignatureTail(Pointer<Customer> cust):Seg(cust){
    _controlref="";
    _signature="";
}


SEGSignatureTail::~SEGSignatureTail(){
}


string SEGSignatureTail::toString(int segnr){
    string result;

    _segnumber=segnr;
    result= "HNSHA:" + String::num2string(_segnumber) + ":1+";
    result += _controlref + "+";
    result += "@";
    result += String::num2string(_signature.length());
    result += "@";
    result += _signature;
    result += "'";

    return result;
}


bool SEGSignatureTail::parse(const string& segment, unsigned int pos){
    // skip segment head
    pos+= String::nextDE(segment, pos).length() + 1;
    // read control reference
    _controlref=String::nextDE(segment, pos);
    pos += String::nextDE(segment, pos).length() + 1;
    // read the signature
    _signature=segment.substr(segment.find("@", pos + 1) + 1);
    return true;
}


void SEGSignatureTail::setData(string &ctrlref,
                               string &signature){
    _controlref=ctrlref;
    _signature=signature;
}


string SEGSignatureTail::signature(){
    return _signature;
}


string SEGSignatureTail::controlReference(){
    return _controlref;
}


/****************************************************************************
 * SEGSignatureTail
 ****************************************************************************/


SEGIdentification::SEGIdentification(Pointer<Customer> cust, bool sync)
:Seg(cust)
,_sync(sync)
{
}


SEGIdentification::~SEGIdentification(){
}


string SEGIdentification::toString(int segnr){
  string result;
  Pointer<Medium> medium;

  _segnumber=segnr;
  medium=_customer.ref().user().ref().medium();
  // segment head
  result="HKIDN:"+ String::num2string(_segnumber) + ":2+";
  // institute data
  result+=String::num2string(_bank.ref().countryCode()) + ":";
  result+=_bank.ref().bankCode() + "+";
  // user data
  if (_specialid.empty()) {
    result+=_customer.ref().custId() + "+";
  }
  else
    result+=_specialid + "+";
  // system id (only in RDH, and only if !_anonymous
  if (medium.ref().securityMode()==HBCI_SECURITY_DDV ||
      medium.ref().mediumId().empty() || _anonymous || _sync) {
    result+="0";
    result+="+";
  }
  else
    result+=medium.ref().mediumId() + "+";
  // system status
  if (_anonymous)
    result+="0";
  else
    result+=(medium.ref().securityMode()==HBCI_SECURITY_DDV)?"0":"1";
  result+="'";
  return result;
}


bool SEGIdentification::parse(const string& segment, unsigned int pos){
  return true;
}


void SEGIdentification::setData(bool anon, string specialid){
    _anonymous=anon;
    _specialid=specialid;
}


/****************************************************************************
 * SEGPreProcessing
 ****************************************************************************/


SEGPreProcessing::SEGPreProcessing(Pointer<Customer> cust):Seg(cust){
}


SEGPreProcessing::~SEGPreProcessing(){
}


string SEGPreProcessing::toString(int segnr){
    string result;
    Pointer<Medium> medium;
    const Hbci *hbci;

    // preparation
    _segnumber=segnr;
    medium=_customer.ref().user().ref().medium();
    BankImpl &bank = 
        dynamic_cast<BankImpl&> (_bank.ref());
    hbci=bank.hbci();

    // now do it
    result="HKVVB:" + String::num2string(_segnumber) +":2+";
    // BPD-Version
    result+=String::num2string(bank.version());
    result+="+";
    // UPD-Version
    result+=String::num2string(_customer.ref().user().ref().version()) + "+";
    // dialog-language
    result+=String::num2string(bank.language()) + "+";
    result+=hbci->systemName();
    result+="+";
    result+=hbci->systemVersion();
    result+="'";
    return result;
}


SEGGetInstKey::SEGGetInstKey(Pointer<Customer> cust):Seg(cust){
    _oldkn=0;
    _oldkv=0;
    _signkey=true;
    _specialuserid="";
}


SEGGetInstKey::~SEGGetInstKey(){
}


string SEGGetInstKey::toString(int segnr){
    string result;
    Pointer<Medium> medium;

    _segnumber=segnr;
    medium=_customer.ref().user().ref().medium();

    result="HKISA:" + String::num2string(_segnumber) +":2+";
    result+="2+124+";

    result+=String::num2string(_bank.ref().countryCode())+ ":";
    result+=_bank.ref().bankCode() + ":";
    result+=(_specialuserid.empty())?_customer.ref().custId():_specialuserid;
    result+=":";
    result+=(_signkey)?"S":"V";
    result+=":";
    result+=String::num2string(_oldkn) + ":";
    result+=String::num2string(_oldkv) + "'";

    return result;
}


bool SEGGetInstKey::parse(const string& segment, unsigned int pos){
    return true;
}


void SEGGetInstKey::setData(bool signkey,
                            int oldkeynumber,
                            int oldkeyversion,
                            string specialuserid){
    _oldkn=oldkeynumber;
    _oldkv=oldkeyversion;
    _signkey=signkey;
    _specialuserid=specialuserid;
}



/****************************************************************************
 * SEGCryptedHead
 ****************************************************************************/


SEGCryptedHead::SEGCryptedHead(Pointer<Customer> cust, bool sync)
:Seg(cust)
,_sync(sync)
{
}


SEGCryptedHead::~SEGCryptedHead(){
}


string SEGCryptedHead::toString(int segnr){
  string result;
  Pointer<Medium> medium;
  int secmode;
  string tmp;

  _segnumber=segnr; // not used
  // first sample some info we need
  medium=_customer.ref().user().ref().medium();
  secmode=medium.ref().securityMode();
  if (secmode!=HBCI_SECURITY_DDV &&
      secmode!=HBCI_SECURITY_RDH)
    throw Error("SEGCryptedHead::toString()",
		"Bad security mode.",
		0);

  // segment-head
  result="HNVSK:998:2+";

  // always has to be "4+1+"
  result+="4+1+";

  // security identification (DEBUG:check this if we don't have a system id)
  result += "1:"; // 1=we are the sender
  if (secmode==HBCI_SECURITY_RDH)
    // next field left out for RDH
    result+=":";
  // medium id is CID for DDV, system id for RDH
  if (medium.ref().mediumId().empty() || _sync)
    result+="0+";
  else
    result+=medium.ref().mediumId() + "+";

  // date and time
  result += "1:";
  result += String::date2string() + ":";
  result += String::time2string() + "+";

  // encryption-algorithm
  result += "2:2:13:";
  result += "@" + String::num2string(_cryptedkey.length()) + "@";
  result += _cryptedkey + ":";
  result += (secmode==HBCI_SECURITY_RDH)?"6":"5";
  result += ":1+";

  /* add the keyname (including institute id etc) */
  result+=String::num2string(_bank.ref().countryCode()) + ":"; // country
  result+=_bank.ref().bankCode() + ":"; //bankdaten
  /* FIXME: Here we should add the keyname of the institute, not
   * the customer ID !*/
  if (secmode==HBCI_SECURITY_RDH) {
    // RDH mode
    Pointer<MediumRDHBase> mrdh=medium.cast<MediumRDHBase>();
    tmp=mrdh.ref().cryptKeyOwner();
    if (Hbci::debugLevel()>2)
      fprintf(stderr,"Bank-Keyname unescaped: %s\n",tmp.c_str());
    tmp=String::escape(tmp);
    if (Hbci::debugLevel()>2)
      fprintf(stderr,"Bank-Keyname escaped: %s\n",tmp.c_str());
    result+=tmp+ ":";
  }
  else
    // DDV mode
    result+=_customer.ref().user().ref().userId() + ":";

  result+="V:"; // its the crypt key

  result+=String::num2string(medium.ref().cryptKeyNumber()) + ":";
  result+=String::num2string(medium.ref().cryptKeyVersion()) + "+";
  // compression method (for now: none)
  result += "0";

  result+="'";
  return result;
}


bool SEGCryptedHead::parse(const string& segment, unsigned int pos){
    pos+=String::nextDE(segment, pos).length() + 1;
    pos+=String::nextDE(segment, pos).length() + 1;
    pos+=String::nextDE(segment, pos).length() + 1;
    pos+=String::nextDE(segment, pos).length() + 1;
    pos+=String::nextDE(segment, pos).length() + 1;

    pos+=String::nextDEG(segment, pos).length() + 1;
    pos+=String::nextDEG(segment, pos).length() + 1;
    pos+=String::nextDEG(segment, pos).length() + 1;

    _cryptedkey=String::nextDEG(segment, pos);
    _cryptedkey=_cryptedkey.substr(_cryptedkey.find("@", 1) + 1);
    return true;
}


void SEGCryptedHead::setData(string cryptedkey){
    _cryptedkey=cryptedkey;
}



/****************************************************************************
 * SEGCryptedData
 ****************************************************************************/


SEGCryptedData::SEGCryptedData(Pointer<Customer> cust):Seg(cust) {
    _data.erase();
}


SEGCryptedData::~SEGCryptedData(){
}


string SEGCryptedData::toString(int segnr){
    string result;

    _segnumber=segnr; // not used
    result="HNVSD:999:1+";
    result+="@" + String::num2string(_data.length()) + "@";
    result+=_data + "'";
    return result;
}


bool SEGCryptedData::parse(const string& segment, unsigned int pos){
    unsigned int i;
    string tmp;

    i=String::nextDE(segment, 0).length() + 1;
    tmp=String::nextDE(segment, i);
    // tmp = @xxx@datadatadata
    i = tmp.find("@", 1);
    _data = tmp.substr(i+1);
    return true;
}


void SEGCryptedData::setData(string data){
    _data=data;
}



/****************************************************************************
 * SEGSynchronize
 ****************************************************************************/


SEGSynchronize::SEGSynchronize(Pointer<Customer> cust):Seg(cust){
    _mode=HBCI_SYNC_SYSTEMID;
}


SEGSynchronize::~SEGSynchronize(){
}


string SEGSynchronize::toString(int segnr){
    string result;

    _segnumber=segnr;
    result="HKSYN:"+String::num2string(segnr) + ":2+";
    result+=String::num2string(_mode) + "'";
    return result;
}


bool SEGSynchronize::parse(const string& segment, unsigned int pos){
    pos+=String::nextDE(segment, pos).length() + 1;
    _mode=atoi(String::nextDE(segment, pos).c_str());
    return true;
}


void SEGSynchronize::setData(int mode){
    _mode=mode;
}



/****************************************************************************
 * SEGPublicKeyChange
 ****************************************************************************/


SEGPublicKeyChange::SEGPublicKeyChange(Pointer<Customer> cust):Seg(cust){
}


SEGPublicKeyChange::~SEGPublicKeyChange(){
}


string SEGPublicKeyChange::toString(int segnr){
    string result;
    Pointer<Medium> medium;

    _segnumber=segnr;

    medium=_customer.ref().user().ref().medium();
    if (!_key.isValid())
        throw Error("SEGPublicKeyChange::toString",
                        "no key given.",0);

    result="HKSAK:"+String::num2string(segnr) + ":2+";

    // key-management-message and certificate replacement
    result += "2+112+";
    // institute-data and userid
    result += String::num2string(_bank.ref().countryCode()) + ":";
    result += _bank.ref().bankCode() + ":";

    /* According to HBCI-Specs the USER id should be used
     * here. Eventually we do this now. Previously we used to use the
     * Customer-ID here, since many programs did that too, and also
     * for those banks with User-ID == Customer-ID it worked since it
     * didn't make a difference. Hopefully now with this
     * implementation *every* bank will be satisfied.
     */
    result += _customer.ref().user().ref().userId() +":";

    result += _key.ref().isCryptoKey()?"V":"S";
    result += ":";
    // number and version of key
    result += String::num2string(_key.ref().number()) + ":";
    result += String::num2string(_key.ref().version()) + "+";
    // keydata
    result += _key.ref().isCryptoKey()?"5":"6";
    result += ":16:10:";
    result += "@" + String::num2string(_key.ref().getModulusData().length()) + "@";
    result += _key.ref().getModulusData() +":12:";
    result += "@" + String::num2string(_key.ref().getExpData().length()) + "@";
    result += _key.ref().getExpData() +":13'";

    return result;
}


bool SEGPublicKeyChange::parse(const string& segment, unsigned int pos){
    if (_customer.ref().user().ref().medium().ref().securityMode()==HBCI_SECURITY_RDH){
        unsigned int pos = 0;
        unsigned int i = 0;
        string tmp;
        string tmp2;
        RSAKey::keyData kd;
        kd.isPublic = true;

        // head
        pos += String::nextDE(segment, pos).length() + 1;
        // management/certificate replacement
        pos += String::nextDE(segment, pos).length() + 1;
        pos += String::nextDE(segment, pos).length() + 1;

        // keyname
        tmp = String::nextDE(segment, pos);
        pos += tmp.length() + 1;
        // kredinst-data
        i = String::nextDEG(tmp, 0).length() + 1;
        i += String::nextDEG(tmp, i).length() + 1;
        // user-id
	kd.owner=String::unEscape(String::nextDEG(tmp, i));
	i += (String::nextDEG(tmp, i)).length() + 1;
	// crypto or signature-key?
        kd.isCrypt=(tmp.at(i) == 'V');
        i += 2;
        // key-number
        tmp2 = String::nextDEG(tmp, i);
        i += tmp2.length() + 1;
        kd.number = atoi(tmp2.c_str());
        // key-version
        tmp2 = String::nextDEG(tmp, i);
        i += tmp2.length() + 1;
        kd.version = atoi(tmp2.c_str());

        // key
        tmp = String::nextDE(segment, pos);
        pos += tmp.length() + 1;
        // 3x dummy
        i = String::nextDEG(tmp, 0).length() + 1;
        i += String::nextDEG(tmp, i).length() + 1;
        i += String::nextDEG(tmp, i).length() + 1;
        // modulus
        kd.modulus = String::nextDEG(tmp, i);
        kd.modulus = kd.modulus.substr(kd.modulus.find("@", 1) + 1);

        // modulus-length should be about 96 bytes. if it differs very much,
        // this is no modulus -> parse-error
        if (kd.modulus.length() < 20 || kd.modulus.length() > 200)
            throw Error("SEGPublicKeyChange::parseResponse()",
                            "HKSAK: Could not read the RSA-key",0);
        kd.exponent = DEFAULT_EXPONENT;
        _parsedkey=new RSAKey(&kd);
        return true;
    }
    return false;
}




/****************************************************************************
 * SEGDialogEnd
 ****************************************************************************/


SEGDialogEnd::SEGDialogEnd(Pointer<Customer> cust):Seg(cust){
}


SEGDialogEnd::~SEGDialogEnd(){
}


string SEGDialogEnd::toString(int segnr){
    string result;

    _segnumber=segnr;
    result="HKEND:"+String::num2string(segnr)+":1+";
    result+=_dialogid;
    result+="'";

    return result;
}


bool SEGDialogEnd::parse(const string& segment, unsigned int pos){
    return true;
}


void SEGDialogEnd::setData(string dialogid){
    _dialogid=dialogid;
}



/****************************************************************************
 * SEGInstituteParameter
 ****************************************************************************/


SEGInstituteParameter::SEGInstituteParameter(Pointer<Customer> cust):Seg(cust){
}


SEGInstituteParameter::~SEGInstituteParameter(){
}


string SEGInstituteParameter::toString(int segnr){
    return "";
}


bool SEGInstituteParameter::parse(const string& segment, unsigned int pos){
    string deg;
    string tmp;
    int i;
    unsigned int pos2;

    // skip segment head
    pos+=String::nextDE(segment, pos).length() + 1;
    // bpd version
    _bpd._version=atoi(String::nextDE(segment, pos).c_str());
    pos+=String::nextDE(segment, pos).length() + 1;
    // country code (280 for Germany)
    _bpd._country=atoi(String::nextDEG(segment, pos).c_str());
    pos+=String::nextDEG(segment, pos).length() + 1;
    // institute code (BLZ in Germany)
    _bpd._instituteCode=String::nextDEG(segment, pos);
    pos+=String::nextDE(segment, pos).length() + 1;
    // instite name
    _bpd._name=String::nextDE(segment, pos);
    pos+=String::nextDE(segment, pos).length() + 1;
    // maximum of different transaction types per message
    _bpd._maxdifferentactions=atoi(String::nextDE(segment, pos).c_str());
    pos+=String::nextDE(segment, pos).length() + 1;
    // supported languages
    deg=String::nextDE(segment, pos);
    pos2=0;
    while(pos2<deg.length()) {
        tmp=String::nextDEG(deg, pos2);
        _bpd._languages.push_back(atoi(tmp.c_str()));
        pos2+=tmp.length()+1;
    } // while
    pos+=String::nextDE(segment, pos).length() + 1;
    // supported versions
    deg=String::nextDE(segment, pos);
    pos2=0;
    while(pos2<deg.length()) {
        tmp=String::nextDEG(deg, pos2);
        i=atoi(String::nextDEG(deg, pos2).c_str());
        _bpd._supportedVersions.push_back(i);
        pos2+=tmp.length()+1;
    } // while
    pos+=String::nextDE(segment, pos).length() + 1;
    // maybe maximum size of message follows
    if (pos<segment.length())
        _bpd._maxmsgsize=atoi(String::nextDE(segment, pos).c_str());
    else
        _bpd._maxmsgsize=0;
    return true;
}



/****************************************************************************
 * SEGComParameter
 ****************************************************************************/


SEGComParameter::SEGComParameter(Pointer<Customer> cust):Seg(cust){
}


SEGComParameter::~SEGComParameter(){
}


bool SEGComParameter::parse(const string& segment, unsigned int pos){
    // skip segment head
    pos+=String::nextDE(segment, pos).length() + 1;
    // skip institute ( we already know it)
    pos+=String::nextDE(segment, pos).length() + 1;
    // get default language
    _bpdcom._language=atoi(String::nextDE(segment, pos).c_str());
    pos+=String::nextDE(segment, pos).length() + 1;
    // communication type
    _bpdcom._type=atoi(String::nextDEG(segment, pos).c_str());
    pos+=String::nextDEG(segment, pos).length() + 1;
    // read address
    _bpdcom._addr=String::nextDEG(segment, pos);
    pos+=String::nextDEG(segment, pos).length() + 1;
    if (_bpdcom._type==1) {
        // read next deg only for T-Online
        _bpdcom._suffix=String::nextDEG(segment, pos).c_str();
        pos+=String::nextDEG(segment, pos).length() + 1;
    }
    // read filter, if any
    if (pos<segment.length()) {
        // read filter type
        _bpdcom._filter=String::nextDEG(segment, pos);
        pos+=String::nextDEG(segment, pos).length() + 1;
        // read filter version
        _bpdcom._filterVersion=atoi(String::nextDEG(segment, pos).c_str());
    }
    return true;
}



/****************************************************************************
 * SEGSupportedJob
 ****************************************************************************/


SEGSupportedJob::SEGSupportedJob(Pointer<Customer> cust):Seg(cust){
}


SEGSupportedJob::~SEGSupportedJob(){
}


bool SEGSupportedJob::parse(const string& segment, unsigned int pos){
    // segment head
    _bpdjob._segmentCode=String::nextDEG(segment, pos);
    pos+=String::nextDEG(segment, pos).length() + 1;
    // skip segment number
    pos+=String::nextDEG(segment, pos).length() + 1;
    // read segment version
    _bpdjob._segmentVersion=atoi(String::nextDEG(segment, pos).c_str());
    pos+=String::nextDE(segment, pos).length() + 1;
    // jobs per message
    _bpdjob._jobsPerMessage=atoi(String::nextDE(segment, pos).c_str());
    pos+=String::nextDE(segment, pos).length() + 1;
    // minimum of signatures needed
    _bpdjob._minSigCount=atoi(String::nextDE(segment, pos).c_str());
    pos+=String::nextDE(segment, pos).length() + 1;
    // store params without parsing them
    _bpdjob._parameter=String::nextDE(segment, pos);
    return true;
}



/*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 SEGUserParameter
 AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 */


SEGUserParameter::SEGUserParameter(Pointer<Customer> cust):Seg(cust){
}


SEGUserParameter::~SEGUserParameter(){
}


bool SEGUserParameter::parse(const string& segment, unsigned int pos){
    string deg;
    string el;
    unsigned int spos;
    updJob job;
    //bool hasSuffix;

    _acc._countryCode=280;
    _acc._limitType=0;
    _acc._limitDays=0;
    pos=0;

    // skip segment head
    pos+=String::nextDE(segment, pos).length() + 1;

    // read KTV
    deg=String::nextDE(segment, pos);
    pos+=deg.length() + 1;
    parse_ktv(deg, _acc._accountNumber, _acc._accountSuffix, 
	      _acc._instituteCode, _acc._countryCode);

    // read customer id
    _acc._userId=String::nextDE(segment, pos);
    pos+=String::nextDE(segment, pos).length() + 1;
    // read currency
    _acc._currency=String::nextDE(segment, pos);
    pos+=String::nextDE(segment, pos).length() + 1;
    // read name 1
    _acc._name1=String::nextDE(segment, pos);
    pos+=String::nextDE(segment, pos).length() + 1;
    // read name 2
    _acc._name2=String::nextDE(segment, pos);
    pos+=String::nextDE(segment, pos).length() + 1;
    // read account name
    _acc._accountName=String::nextDE(segment, pos);
    pos+=String::nextDE(segment, pos).length() + 1;
    // read account limit, if any
    deg=String::nextDE(segment, pos);
    spos=0;
    el=String::nextDEG(deg, spos);
    spos+=String::nextDEG(deg, spos).length() + 1;
    if (el=="E" || el=="T" || el=="W" || el=="M" || el=="Z") {
        _acc._limitType=el.at(0);
        _acc._limitValue=Value(String::nextDEG(deg, spos));
        spos+=String::nextDEG(deg, spos).length() + 1;
        if (_acc._limitType=='Z')
            _acc._limitDays=atoi(String::nextDEG(deg, spos).c_str());
    }

    pos+=String::nextDE(segment, pos).length() + 1;
    // now read the jobs
    while(pos<segment.length()) {
        deg=String::nextDE(segment, pos);
        // TODO: Create a segment for this
        job=updJob(deg);
        pos+=String::nextDE(segment, pos).length() + 1;
        _acc._allowedJobs.push_back(job);
    } // while

    return true;
}



/****************************************************************************
 * SEGPublicKeyDisable
 ****************************************************************************/


SEGPublicKeyDisable::SEGPublicKeyDisable(Pointer<Customer> cust) : Seg(cust){
  _keyVersion = 0;
  _keyNumber = 0;  
}


SEGPublicKeyDisable::~SEGPublicKeyDisable(){
}


string SEGPublicKeyDisable::toString(int segnr) {
  string result;
  Pointer<Medium> medium;

  _segnumber=segnr;
  medium=_customer.ref().user().ref().medium();
  //    if (!_key.isValid())
  if ( (0 == _keyNumber) && (0 == _keyVersion))
    throw Error("SEGPublicKeyDisable::toString()",
		"No Key.",0);
  if (medium.ref().securityMode()!=HBCI_SECURITY_RDH)
    throw Error("SEGPublicKeyDisable::toString",
		"not in RDH mode",0);

  result = "HKSSP:" + String::num2string(segnr) + ":2+";
  result += "2+130+";

  // institute-data and userid
  result += String::num2string(_bank.ref().countryCode()) + ":";
  result += _bank.ref().bankCode() + ":";
  result += _customer.ref().user().ref().userId() +":";
  result += "S:";// always the signkey! _key.ref().isCryptoKey()?"V":"S";
  //    result += ":";
  // number and version of key
  result += String::num2string(_keyNumber) + ":";
  result += String::num2string(_keyVersion) + "+";
  // cancel key because of undefined reason
  result += "999'";

  return result;
}


void SEGPublicKeyDisable::setData(Pointer<RSAKey> key) {
  if (!key.isValid())
	throw Error("SEGPublicKeyDisable::setData()",
				"no Key.", 0);

  _keyNumber = key.ref().number();
  _keyVersion = key.ref().version();
}



SEGPublicKeyReturn::SEGPublicKeyReturn(Pointer<Customer> cust)
:Seg(cust)
{
}


SEGPublicKeyReturn::~SEGPublicKeyReturn(){
}



bool SEGPublicKeyReturn::parse(const string& segment, unsigned int pos){
    string tmp;
    string tmp2;
    RSAKey::keyData kd;
    int i;
    string descr;

    kd.isPublic = true;

    // head
    pos += String::nextDE(segment, 0).length() + 1;
    // message-type
    pos += String::nextDE(segment, pos).length() + 1;
    // ref1
    pos += String::nextDE(segment, pos).length() + 1;
        // ref2
    pos += String::nextDE(segment, pos).length() + 1;
    // function-type
    pos += String::nextDE(segment, pos).length() + 1;

    // keyname
    tmp = String::nextDE(segment, pos);
    pos += tmp.length() + 1;
    // kredinst-data
    i = String::nextDEG(tmp, 0).length() + 1;
    i += String::nextDEG(tmp, i).length() + 1;
    // user-id
    kd.owner=String::unEscape(String::nextDEG(tmp, i));
    //kd.owner=String::nextDEG(tmp, i);   Store the name unescaped !
    i += (String::nextDEG(tmp, i)).length() + 1;
    // crypto or signature-key?
    kd.isCrypt=(tmp.at(i) == 'V');
    i += 2;
    // key-number
    tmp2 = String::nextDEG(tmp, i);
    i += tmp2.length() + 1;
    kd.number = atoi(tmp2.c_str());
    // key-version
    tmp2 = String::nextDEG(tmp, i);
    i += tmp2.length() + 1;
    kd.version = atoi(tmp2.c_str());

    // key
    tmp = String::nextDE(segment, pos);
    pos += tmp.length() + 1;
    // 3x dummy
    i = String::nextDEG(tmp, 0).length() + 1;
    i += String::nextDEG(tmp, i).length() + 1;
    i += String::nextDEG(tmp, i).length() + 1;
    // modulus
    kd.modulus = String::nextDEG(tmp, i);
    kd.modulus = kd.modulus.substr(kd.modulus.find("@", 1) + 1);

    // modulus-length should be about 96 bytes. if it differs very mucht,
    // this is no modulus -> parse-error
    if (kd.modulus.length() < 20 || kd.modulus.length() > 200)
        throw Error("JOBDialogInit::parseResponse",
                        "HIISA: Could not read the RSA-key",0);
    kd.exponent = DEFAULT_EXPONENT;

    // create key
    _key=new RSAKey(&kd);
    // create description for pointer object (debugging purpose)
    descr="RSAKey from SEGPublicKeyReturn (";
    descr+=kd.isCrypt?"crypt":"sign";
    descr+=" ";
    descr+=kd.isPublic?"public":"private";
    descr+=")";
    _key.setObjectDescription(descr);

    // done
    return true;
}


/*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 * SEGGetStatusReport
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 */


SEGGetStatusReport::SEGGetStatusReport(Pointer<Customer> cust):Seg(cust) {

}


SEGGetStatusReport::~SEGGetStatusReport(){
}


string SEGGetStatusReport::toString(int segnr) {
  int versionMin = -1;
  int versionMax = -1;
  string result;
  const bpdJob *jp;

  BankImpl &bank =
    dynamic_cast<BankImpl&> (_bank.ref());

  _segnumber=segnr;
  Seg::segment_number(versionMin, versionMax,
		      bank.hbciVersion(),
		      2,2,
		      3,3,
		      3,3);

  jp=bank.findJob("HIPROS",versionMin,versionMax);
  if (jp==0)
    throw Error("SEGGetStatusReport::toString()",
		"job not supported",0);

  // create segment
  // segment head
  result="HKPRO:";
  // segment number
  result+=String::num2string(segnr) + ":";
  // segment version
  result+=String::num2string(jp->segmentVersion());

  if (jp->segmentVersion()>2) {
    // only since HBCI 2.1 there are date fields
    if (_fromdate.isValid() ||
	_todate.isValid() ||
	_maxEntries!=-1 ||
	!_attachPoint.empty())
      result+="+";
    if (_fromdate.isValid()) {
      result+=_fromdate.toString();
    }

    if (_todate.isValid() || _maxEntries!=-1 || !_attachPoint.empty())
      result+="+";
    if (_todate.isValid()) {
      result+=_todate.toString();
    }
  }

  if (_maxEntries!=-1 || !_attachPoint.empty())
    result+="+";
  if (_maxEntries!=-1) {
    result+=String::num2string(_maxEntries);
  }

  // check the attachpoint
  if (!_attachPoint.empty()) {
    result+="+";
    result += _attachPoint;
  }

  result += "'";
  return result;

}


void SEGGetStatusReport::setData(Date fromDate,
				 Date toDate,
				 int maxEntries,
				 string attachPoint){
  _fromdate=fromDate;
  _todate=toDate;
  _maxEntries=maxEntries;
  _attachPoint = attachPoint;
}





SEGStatusReport::SEGStatusReport(Pointer<Customer> cust)
:Seg(cust)
{
}


SEGStatusReport::~SEGStatusReport(){
}


bool SEGStatusReport::parse(const string& segment, unsigned int pos){
  string tmp;
  string group;
  string dep;
  unsigned int gpos;

  /* The job of this method is very easy, it just takes
   * a segment received from the server as input and tries to parse all
   * informative out of it. In this case the interesting thing is
   * the account balance, as the name of this class implies ;-)
   */

  // skip segment head
  pos += String::nextDE(segment, pos).length() + 1;

  // get MessageReference
  tmp=String::nextDE(segment, pos);
  _rep.setMessageReference(MessageReference(tmp));
  pos += String::nextDE(segment, pos).length() + 1;

  // get segment ref (if given)
  tmp=String::nextDE(segment, pos);
  if (!tmp.empty())
    _rep.setSegment(atoi(tmp.c_str()));
  pos += String::nextDE(segment, pos).length() + 1;

  // get date
  tmp=String::nextDE(segment, pos);
  _rep.setDate(Date(tmp));
  pos += String::nextDE(segment, pos).length() + 1;

  // get time
  tmp=String::nextDE(segment, pos);
  _rep.setTime(Time(tmp));
  pos += String::nextDE(segment, pos).length() + 1;

  // get result code group
  group=String::nextDE(segment, pos);
  pos += String::nextDE(segment, pos).length() + 1;
  gpos=0;

  // DEG: Result code
  tmp=String::nextDEG(group, gpos);
  _rep.setResult(atoi(tmp.c_str()));
  gpos += String::nextDEG(group, gpos).length() + 1;

  // DEG: "Bezugssegment"
  _rep.setGroupReference(String::nextDEG(group, gpos));
  gpos += String::nextDEG(group, gpos).length() + 1;

  // DEG: Text
  _rep.setResultText(String::nextDEG(group, gpos));
  gpos += String::nextDEG(group, gpos).length() + 1;

  return true;
}




} // namespace HBCI
