/* outboxaccjobs.cpp

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/



#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "assert.h"

#include "outboxaccjobs.h"
#include <accountimpl.h>
#include <accountjobs.h>
#include <bankimpl.h>
#include <hbcistring.h>

namespace HBCI {

// This const_cast is used frequently in this file since the
// Pointer to the customer won't be changed (thus doesn't violate
// his const declaration) but it is impossible to express this with
// the Pointer, unfortunately.
static Pointer<Customer> custPointer_const_cast(const Customer *c){
  Pointer<Customer> cp = const_cast<Customer*>(c);
  cp.setAutoDelete(false);
  return cp;
}


static Pointer<Account> accPointer(Account *a) {
  Pointer<Account> ap = a;
  ap.setAutoDelete(false);
  return ap;
}


OutboxAccountJob::OutboxAccountJob(Pointer<Customer> c,
				   Pointer<Account> a)
:OutboxJob(c)
,_acc(a){
  if (!a.isValid())
    fprintf(stderr,"OutboxAccountJob: invalid account pointer.\n");
  _acc.setDescription("OutboxAccountJob::_acc");
}


OutboxAccountJob::~OutboxAccountJob(){
}


string OutboxAccountJob::_makeDescription(const string &dscr) const {
  Pointer<Bank> bank;
  string result;

  if (!_acc.isValid())
    return dscr;
  bank=_acc.ref().bank();
  result=dscr;
  result+=" for ";
  result+=_acc.ref().accountId();
  result+=" (";
  if (bank.ref().name().empty())
    result+=bank.ref().bankCode();
  else
    result+=bank.ref().name();
  result+=")";
  return result;
}



OutboxJobGetBalance::OutboxJobGetBalance(Pointer<Customer> c,
					 Pointer<Account> a)
:OutboxAccountJob(c,a)
{
}


OutboxJobGetBalance::~OutboxJobGetBalance(){
}


bool OutboxJobGetBalance::createHBCIJobs(Pointer<MessageQueue> mbox,
					 int){

  _job=new JOBGetBalance(_cust,_acc);
  mbox.ref().addJob(_job);
  addSignersToQueue(mbox);
  return true;
}


bool OutboxJobGetBalance::evaluate(){
  if (_job.ref().hasErrors())
    _result=HBCI_JOB_RESULT_FAILED;
  else
    _result=HBCI_JOB_RESULT_SUCCESS;

  return _result==HBCI_JOB_RESULT_SUCCESS;
}


bool OutboxJobGetBalance::commit(int msgNumber){
  // only commit changes for the whole job
  if (HBCI_COMMIT_WHOLE_JOB != msgNumber)
    return true;

  // only commit if there was no error
  if (_result!=HBCI_JOB_RESULT_SUCCESS)
    return false;

  // in retrieval-only mode?
  if (_bank.ref().hbci()->isRetrievalOnly())
    return true;

  dynamic_cast<AccountImpl&>(_acc.ref()).
    setBalance(dynamic_cast<JOBGetBalance&>(_job.ref()).getBalance());
  return true;
}


const AccountBalance &OutboxJobGetBalance::getBalance() const {
  return dynamic_cast<JOBGetBalance&>(_job.ref()).getBalance();
}

string OutboxJobGetBalance::description() const {
  return _makeDescription("Get balance");
}


list<int> OutboxJobGetBalance::resultCodes() const {
  list<int> res;
  if (_job.isValid())
    res = resultCodesFromJob(_job.ref());
  return res;
}


bool OutboxJobGetBalance::isSupported(Pointer<Account> forAccount) {
  AccountImpl &accountImpl = forAccount.cast<AccountImpl>().ref();

  return (NULL != accountImpl.updForJob("HKSAL"));
}

} // namespaec HBCI


HBCI_OutboxJobGetBalance *
HBCI_OutboxJobGetBalance_new(const HBCI_Customer *c, HBCI_Account *a){
  assert(c);
  assert(a);
  return new HBCI_OutboxJobGetBalance(custPointer_const_cast(c),
				      accPointer(a));
}


HBCI_OutboxJob
*HBCI_OutboxJobGetBalance_OutboxJob(HBCI_OutboxJobGetBalance *j){
  return j;
}


const HBCI_AccountBalance *
HBCI_OutboxJobGetBalance_getBalance(const HBCI_OutboxJobGetBalance *j){
  assert(j);
  return &(j->getBalance());
}



namespace HBCI {

OutboxJobGetTransactions::OutboxJobGetTransactions(Pointer<Customer> c,
						   Pointer<Account> a,
						   Date fromDate,
						   Date toDate)
    :OutboxAccountJob(c,a)
,_fromdate(fromDate)
    ,_todate(toDate)
{
}


OutboxJobGetTransactions::~OutboxJobGetTransactions(){
}


bool OutboxJobGetTransactions::createHBCIJobs(Pointer<MessageQueue> mbox,
					      int n){
  Date newf;
  DateTime ti;
  Transaction lastt;
  list<Transaction> tlist;

  newf=_fromdate;
  if (!_fromdate.isValid() &&
      !_todate.isValid()) {
    // take the date of the last transaction data we have -1 day
    tlist=_acc.ref().transactions();
    if (!tlist.empty()) {
      lastt=tlist.back();
      if (lastt.date().isValid()) {
	ti=DateTime(lastt.date().year(),
		    lastt.date().month(),
		    lastt.date().day(),
		    0,0,0);

	// add one day to last one
	ti=ti.addSeconds(-60*60*24);
	newf=Date(ti.day(),
		  ti.month(),
		  ti.year());
      }
      else if (lastt.valutaDate().isValid()) {
	ti=DateTime(lastt.valutaDate().year(),
		    lastt.valutaDate().month(),
		    lastt.valutaDate().day(),
		    0,0,0);
	// add one day to last one
	ti=ti.addSeconds(60*60*24);
	newf=Date(ti.day(),
		  ti.month(),
		  ti.year());
      }
    }
  } // if valid date

  if (0 == n)
    _job=new JOBGetTurnover(_cust, _acc, newf, _todate);
  else {
    fprintf(stderr,"Will use jump point.\n");
    _job=new JOBGetTurnover(_cust, _acc, newf, _todate, _job);
  }

  mbox.ref().addJob(_job.cast<Job>());
  addSignersToQueue(mbox);
  return true;
}


bool OutboxJobGetTransactions::stillMessagesToSend(int nextMsg) const{
  // zero? of course we have a message to send!
  if (0 == nextMsg)
    return true;

  // any more?
  return dynamic_cast<JOBGetTurnover&>(_job.ref()).attachMore();
}


bool OutboxJobGetTransactions::evaluate(){
  if (_job.ref().hasErrors())
    _result=HBCI_JOB_RESULT_FAILED;
  else
    _result=HBCI_JOB_RESULT_SUCCESS;

  return _result==HBCI_JOB_RESULT_SUCCESS;
}


bool OutboxJobGetTransactions::commit(int msgNumber){
  list<Transaction> newTransactions;
  list<Transaction>::iterator it1;
  list<Transaction>::const_iterator it2;
  Pointer<AccountImpl> ai;
  AccountBalance ab;
  Balance jb;

  ai=_acc.cast<AccountImpl>();

  // don't do anything when commit is called for the whole
  // job. otherwise, we would add the result of the last subjob twice
  if (HBCI_COMMIT_WHOLE_JOB == msgNumber)
    return true;

  // only commit if there was no error
  evaluate();
  if (_result!=HBCI_JOB_RESULT_SUCCESS)
    return false;

  // in retrieval-only mode?
  if (_bank.ref().hbci()->isRetrievalOnly())
    return true;

  const list<Transaction> &xa =
    dynamic_cast<JOBGetTurnover&> (_job.ref()).transactions();

  // in case we already received some or all transaction statements
  // from this session during previous sessions, remove the duplicate
  // entries in the account's list prior to adding the newly received
  // ones.
  for (it2=xa.begin();
       it2!=xa.end();
       it2++) {
    Transaction newT;
    const Transaction *oldT;

    oldT=ai.ref().findTransaction(*it2);
    if (oldT) {
      if (Hbci::debugLevel()>0)
	fprintf(stderr, "Using old transaction id (%d):\n",
		oldT->id());
      newT=*it2;
      newT.setId(oldT->id());
      newTransactions.push_back(newT);
      ai.ref().removeTransaction(*it2);
    }
    else {
      // new transaction, assign a new id
      if (Hbci::debugLevel()>0)
	fprintf(stderr, "Assigning new transaction id\n");
      newT=*it2;
      newT.setId(API::nextTransactionId());
      newTransactions.push_back(newT);
    }
  } // for

  // copy eventually received turnovers into account
  for (it1=newTransactions.begin();
       it1!=newTransactions.end();
       it1++)
    ai.ref().addTransaction(*it1);

  // check whether the balance recieved is newer then the current balance
  ab=ai.ref().balance();
  jb = dynamic_cast<JOBGetTurnover&> (_job.ref()).lastBalance();

  if (Hbci::debugLevel()>1) {
    fprintf(stderr,"LastBalance was:\n");
    jb.dump();
  }

  if (jb.date().isValid()) {
      if (ab.notedBalance().date().isValid()) {
	  if (jb.date()>ab.notedBalance().date() ||
	      (jb.date()==ab.notedBalance().date() &&
	       jb.time()>ab.notedBalance().time())) {
	      if (Hbci::debugLevel()>1) {
		  fprintf(stderr,
			  "Setting new noted balance to:\n");
		  jb.dump();
	      }
	      ab.setNotedBalance(jb);
	      ai.ref().setBalance(ab);
	  }
      }
      else {
	  if (Hbci::debugLevel()>1) {
	      fprintf(stderr,
		      "Setting new noted balance to:\n");
	      jb.dump();
	  }
	  ab.setNotedBalance(jb);
	  ai.ref().setBalance(ab);
      }
  }

  return true;
}

const Balance &OutboxJobGetTransactions::lastBalance() const 
{
  if (_job.isValid())
    return dynamic_cast<JOBGetTurnover&> (_job.ref()).lastBalance();
  else 
    throw Error("OutboxJobGetTransactions::lastBalance()",
		ERROR_LEVEL_NORMAL,
		0,
		ERROR_ADVISE_ABORT,
		"No HBCI JOB created yet.");
}
    
string OutboxJobGetTransactions::description() const{
  return _makeDescription("Get transaction statements");
}


const list<Transaction> &
OutboxJobGetTransactions::transactions() const{
  return dynamic_cast<JOBGetTurnover&>(_job.ref()).transactions();
}


list<int> OutboxJobGetTransactions::resultCodes() const{
  list<int> res;
  if (_job.isValid())
    res = resultCodesFromJob(_job.ref());
  return res;
}

bool OutboxJobGetTransactions::isSupported(Pointer<Account> forAccount) {
  AccountImpl &accountImpl = forAccount.cast<AccountImpl>().ref();

  return ((NULL != accountImpl.updForJob("HKKAZ")) ||
	  (NULL != accountImpl.updForJob("HKKAN")));
}


} // namespace HBCI

HBCI_OutboxJobGetTransactions *
HBCI_OutboxJobGetTransactions_new(const HBCI_Customer *c, HBCI_Account *a,
				  const HBCI_Date *fromdate,
				  const HBCI_Date *todate){
  assert(c);
  assert(a);
  assert(fromdate);
  assert(todate);
  return new HBCI_OutboxJobGetTransactions(custPointer_const_cast(c),
					   accPointer(a),
					   *fromdate,
					   *todate);
}


HBCI_OutboxJob *
HBCI_OutboxJobGetTransactions_OutboxJob(HBCI_OutboxJobGetTransactions *j){
  return j;
}


const list_HBCI_Transaction *
HBCI_OutboxJobGetTransactions_transactions(const
					   HBCI_OutboxJobGetTransactions *j){
  assert(j);
  return &(j->transactions());
}

const HBCI_Balance *
HBCI_OutboxJobGetTransactions_lastBalance(const 
					  HBCI_OutboxJobGetTransactions *j)
{
  assert(j);
  return &(j->lastBalance());
}

const HBCI_Date * 
HBCI_OutboxJobGetTransactions_fromDate(const HBCI_OutboxJobGetTransactions *j)
{
  assert(j);
  return &(j->fromDate());
}

const HBCI_Date * 
HBCI_OutboxJobGetTransactions_toDate(const HBCI_OutboxJobGetTransactions *j)
{
  assert(j);
  return &(j->toDate());
}



namespace HBCI {

OutboxJobTransfer::OutboxJobTransfer(Pointer<Customer> c,
				     Pointer<Account> a,
				     Transaction xa)
    :OutboxAccountJob(c,a)
,_xaction(xa)
{
}


OutboxJobTransfer::~OutboxJobTransfer(){
}


bool OutboxJobTransfer::createHBCIJobs(Pointer<MessageQueue> mbox,
				       int){
  // check for readonly mode
  if (_bank.ref().hbci()->isReadOnly()) {
#if DEBUGMODE>0
    fprintf(stderr,"Readonly mode, job not allowed.\n");
#endif
    return false;
  }
  _job=new JOBSingleTransfer(_cust,_xaction);

  mbox.ref().addJob(_job);
  addSignersToQueue(mbox);
  return true;
}


bool OutboxJobTransfer::evaluate(){
  if (_job.ref().hasErrors())
    _result=HBCI_JOB_RESULT_FAILED;
  else
    _result=HBCI_JOB_RESULT_SUCCESS;

  return _result==HBCI_JOB_RESULT_SUCCESS;
}


bool OutboxJobTransfer::commit(int msgNumber){
  // maybe later we have a list of already taken transfers
  return true;
}


string OutboxJobTransfer::description() const{
  return _makeDescription("Transfer money");
}


list<int> OutboxJobTransfer::resultCodes() const{
  list<int> res;
  if (_job.isValid())
    res = resultCodesFromJob(_job.ref());
  return res;
}

// all the bpd-information-methods
int
OutboxJobTransfer::maxDescriptionLines(const Bank &forBank)
{
  const BankImpl &bank =
    dynamic_cast<const BankImpl&>(forBank);
  const bpdJob *jobData = bank.findJob("HIUEBS");

  if (NULL == jobData)
    return 0;

  string param = jobData->parameter();

  // the first entry is for purpose lines
  return atoi(String::nextDEG(param, 0).c_str());
}


list<int>
OutboxJobTransfer::transactionCodes(const Bank &forBank) {
  list<int> keys;
  const BankImpl &bank =
    dynamic_cast<const BankImpl&>(forBank);
  const bpdJob *jobData = bank.findJob("HIUEBS");

  if (NULL == jobData)
    return keys;

  string param = jobData->parameter();

  // the second entry
  unsigned pos = 0;
  pos += String::nextDEG(param, pos).length() + 1;

  while (pos < param.length()) {
    string tmp = String::nextDEG(param, pos);
    pos += 3;
    keys.push_back(atoi(tmp.c_str()));
  }

  return keys;
}


bool OutboxJobTransfer::isSupported(Pointer<Account> forAccount) {
  AccountImpl &accountImpl = forAccount.cast<AccountImpl>().ref();

  return (NULL != accountImpl.updForJob("HKUEB"));
}


const Limit OutboxJobTransfer::limit(Pointer<Account> forAccount) {
  AccountImpl &accountImpl = forAccount.cast<AccountImpl>().ref();
  const updJob *upd = accountImpl.updForJob("HKUEB");

  if (NULL == upd)
    return Limit();
  else
    return upd->limit();
}


int OutboxJobTransfer::segmentForStatusReport() const {
  if (_job.isValid())
    return _job.ref().startSegment();
  else
    return -1;
}


} // namespace HBCI

HBCI_OutboxJobTransfer *
HBCI_OutboxJobTransfer_new(const HBCI_Customer *c, HBCI_Account *a,
			   const HBCI_Transaction *trans)
{
    assert(c);
    assert(a);
    assert(trans);
    return new HBCI_OutboxJobTransfer(custPointer_const_cast(c), 
				      accPointer(a), *trans);
}

/** Upcast */
HBCI_OutboxJob *
HBCI_OutboxJobTransfer_OutboxJob(HBCI_OutboxJobTransfer *j)
{
    return j;
}




namespace HBCI {
OutboxJobDebitNote::OutboxJobDebitNote(Pointer<Customer> c,
				       Pointer<Account> a,
				       Transaction xa)
    :OutboxAccountJob(c,a)
,_xaction(xa)
{
}


OutboxJobDebitNote::~OutboxJobDebitNote(){
}


bool OutboxJobDebitNote::createHBCIJobs(Pointer<MessageQueue> mbox,
					int){
  // check for readonly mode
  if (_bank.ref().hbci()->isReadOnly()) {
#if DEBUGMODE>0
    fprintf(stderr,"Readonly mode, job not allowed.\n");
#endif
    return false;
  }
  _job=new JOBDebitNote(_cust,_xaction);

  mbox.ref().addJob(_job);
  addSignersToQueue(mbox);
  return true;
}


bool OutboxJobDebitNote::evaluate(){
  if (_job.ref().hasErrors())
    _result=HBCI_JOB_RESULT_FAILED;
  else
    _result=HBCI_JOB_RESULT_SUCCESS;

  return _result==HBCI_JOB_RESULT_SUCCESS;
}


bool OutboxJobDebitNote::commit(int msgNumber){
  // maybe later we have a list of already taken transfers
  return true;
}


string OutboxJobDebitNote::description() const {
  return _makeDescription("Debit note");
}


list<int> OutboxJobDebitNote::resultCodes() const{
  list<int> res;
  if (_job.isValid())
    res = resultCodesFromJob(_job.ref());
  return res;
}


bool OutboxJobDebitNote::isSupported(Pointer<Account> forAccount) {
  AccountImpl &accountImpl = forAccount.cast<AccountImpl>().ref();

  return (NULL != accountImpl.updForJob("HKLAS"));
}


const Limit OutboxJobDebitNote::limit(Pointer<Account> forAccount) {
  AccountImpl &accountImpl = forAccount.cast<AccountImpl>().ref();
  const updJob *upd = accountImpl.updForJob("HKLAS");

  if (NULL == upd)
    return Limit();
  else
    return upd->limit();
}

int OutboxJobDebitNote::segmentForStatusReport() const {
  if (_job.isValid())
    return _job.ref().startSegment();
  else
    return -1;
}

} // namespace HBCI
HBCI_OutboxJobDebitNote *
HBCI_OutboxJobDebitNote_new(const HBCI_Customer *c, HBCI_Account *a,
			    const HBCI_Transaction *trans)
{
    assert(c);
    assert(a);
    assert(trans);
    return new HBCI_OutboxJobDebitNote(custPointer_const_cast(c), 
				       accPointer(a), *trans);
}

/** Upcast */
HBCI_OutboxJob *
HBCI_OutboxJobDebitNote_OutboxJob(HBCI_OutboxJobDebitNote *j)
{
    return j;
}
