/***************************************************************************
 $RCSfile: mediumkeyfile.cpp,v $
                             -------------------
    cvs         : $Id: mediumkeyfile.cpp,v 1.11 2004/01/13 21:19:39 aquamaniac Exp $
    begin       : Thu Aug 22 2002
    copyright   : (C) 2002 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#define MEDIUMKEYFILE_VERSION_MAJOR 1
#define MEDIUMKEYFILE_VERSION_MINOR 1

#include "deskey.h"
#include "hbcistring.h"
#include "file.h"


#define MEDIUMKEYFILE_TAG_VERSION_MAJOR      (unsigned char)0x02
#define MEDIUMKEYFILE_TAG_VERSION_MINOR      (unsigned char)0x03
#define MEDIUMKEYFILE_TAG_SEQ                (unsigned char)0x04

#define MEDIUMKEYFILE_TAG_USER_PUBSIGNKEY    (unsigned char)0xc5
#define MEDIUMKEYFILE_TAG_USER_PRIVSIGNKEY   (unsigned char)0xc6
#define MEDIUMKEYFILE_TAG_USER_PUBCRYPTKEY   (unsigned char)0xc7
#define MEDIUMKEYFILE_TAG_USER_PRIVCRYPTKEY  (unsigned char)0xc8
#define MEDIUMKEYFILE_TAG_USER_ID            (unsigned char)0x09

#define MEDIUMKEYFILE_TAG_INST_PUBSIGNKEY    (unsigned char)0xca
#define MEDIUMKEYFILE_TAG_INST_PUBCRYPTKEY   (unsigned char)0xcb
#define MEDIUMKEYFILE_TAG_INST_COUNTRY       (unsigned char)0x0c
#define MEDIUMKEYFILE_TAG_INST_CODE          (unsigned char)0x0d
#define MEDIUMKEYFILE_TAG_INST_SYSTEMID      (unsigned char)0x0e

// temporary keys (new in version 1.1)
#define MEDIUMKEYFILE_TAG_TEMP_PUBSIGNKEY    (unsigned char)0xcf
#define MEDIUMKEYFILE_TAG_TEMP_PRIVSIGNKEY   (unsigned char)0xd0
#define MEDIUMKEYFILE_TAG_TEMP_PUBCRYPTKEY   (unsigned char)0xd1
#define MEDIUMKEYFILE_TAG_TEMP_PRIVCRYPTKEY  (unsigned char)0xd2


// keydata
#define MEDIUM_RDH_TAG_KEY_ISPUBLIC      (unsigned char) 0x01
#define MEDIUM_RDH_TAG_KEY_ISCRYPT       (unsigned char) 0x02
#define MEDIUM_RDH_TAG_KEY_OWNER         (unsigned char) 0x03
#define MEDIUM_RDH_TAG_KEY_VERSION       (unsigned char) 0x04
#define MEDIUM_RDH_TAG_KEY_NUMBER        (unsigned char) 0x05
#define MEDIUM_RDH_TAG_KEY_MODULUS       (unsigned char) 0x06
#define MEDIUM_RDH_TAG_KEY_EXP           (unsigned char) 0x07
#define MEDIUM_RDH_TAG_KEY_N             (unsigned char) 0x08
#define MEDIUM_RDH_TAG_KEY_P             (unsigned char) 0x09
#define MEDIUM_RDH_TAG_KEY_Q             (unsigned char) 0x0a
#define MEDIUM_RDH_TAG_KEY_DMP1          (unsigned char) 0x0b
#define MEDIUM_RDH_TAG_KEY_DMQ1          (unsigned char) 0x0c
#define MEDIUM_RDH_TAG_KEY_IQMP          (unsigned char) 0x0d
#define MEDIUM_RDH_TAG_KEY_D             (unsigned char) 0x0e


#include "mediumkeyfile.h"
#include "file.h"


namespace HBCI {


const string MediumKeyfileBase::str_empty="";


MediumKeyfileBase::MediumKeyfileBase(const Hbci *hbci)
    :MediumRDHBase(hbci)
    ,_country(280)
    ,_seq(0)
{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::MediumKeyfileBase\n";
}



MediumKeyfileBase::~MediumKeyfileBase(){
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::~MediumKeyfileBase\n";
}


void MediumKeyfileBase::_clearMedium(){
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::_clearMedium\n";

    _userPubSignatureKey=0;
    _userPrivateSignatureKey=0;
    _userPubCryptKey=0;
    _userPrivateCryptKey=0;
    _userId.erase();

    _country=280;
    _instcode.erase();
    _systemid.erase();
    _instPubSignKey=0;
    _instPubCryptKey=0;

}


unsigned int MediumKeyfileBase::nextSEQ() {
    _seq++;
    return _seq;
}


void MediumKeyfileBase::resetSEQ() {
  _seq=0;
}


void MediumKeyfileBase::setSEQ(int seq){
  _seq=seq;
}


Error MediumKeyfileBase::selectContext(int country,
				       const string &instcode,
				       const string &userid){
  if (Hbci::debugLevel()>15) {
    cerr<<"MediumKeyfileBase::selectContext\n";
    cerr<<" Country="<<country<<" BankCode="<<instcode
      <<" UserId="<<userid<<"\n";
  }

  if (_country==country &&
      _instcode==instcode &&
      _userId==userid)
    return Error();

  return Error("MediumKeyfileBase::selectContext",
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_INVALID,
	       ERROR_ADVISE_DONTKNOW,
	       "no matching entry found");
}


Error MediumKeyfileBase::createMedium(int country,
				      const string &instcode,
				      const string &userid){
  if (Hbci::debugLevel()>15)
    cerr<<"MediumKeyfileBase::createMedium\n";
  if (Hbci::debugLevel()>3)
    cerr<<" Country="<<country<<" BankCode="<<instcode
      <<" UserId="<<userid<<"\n";

  _clearMedium();
  _country=country;
  _instcode=instcode;
  _userId=userid;

  return Error();
}


int MediumKeyfileBase::signKeyNumber() const{
  if (Hbci::debugLevel()>15)
    cerr<<"MediumKeyfileBase::signKeyNumber\n";
  return _userPrivateSignatureKey.ref().number();
}


int MediumKeyfileBase::signKeyVersion() const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::signKeyVersion\n";
    return _userPrivateSignatureKey.ref().version();
}


int MediumKeyfileBase::cryptKeyNumber() const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::cryptKeyNumber\n";
    if (!_instPubCryptKey.isValid())
        return 0;
    return _instPubCryptKey.ref().number();
}


int MediumKeyfileBase::cryptKeyVersion() const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::cryptKeyVersion\n";
    if (!_instPubCryptKey.isValid())
        return 0;
    return _instPubCryptKey.ref().version();
}


const string &MediumKeyfileBase::cryptKeyOwner() const{
    if (Hbci::debugLevel()>15)
	cerr<<"MediumKeyfileBase::cryptKeyOwner\n";
    if (!_instPubCryptKey.isValid())
	return str_empty;
    return _instPubCryptKey.ref().userId();
}


Error MediumKeyfileBase::getContext(int num,
				    int &countrycode,
				    string &instcode,
				    string &userid,
				    string &server) const{
  if (Hbci::debugLevel()>15)
    cerr<<"MediumKeyfileBase::getContext "<<num<<"\n";
  if (num!=1)
    return Error("MediumKeyfileBase::getContext",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_DONTKNOW,
		 "bad context number");

  countrycode=_country;
  instcode=_instcode;
  userid=_userId;
  server.erase();
  return Error();
}


string MediumKeyfileBase::createMessageKey() const{
    DESKey desKey;

    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::createMessageKey\n";
    desKey=DESKey::createKey();
    return desKey.getKeyString();
}


string MediumKeyfileBase::encryptKey(const string &srckey){
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::encryptKey\n";
    if (!_instPubCryptKey.isValid())
        throw Error("MediumKeyFile::encryptKey",
                    "No Key.",0);
    _instPubCryptKey.ref().setData(srckey);
    _instPubCryptKey.ref().encrypt();
    return _instPubCryptKey.ref().getData();
}


string MediumKeyfileBase::decryptKey(const string &srckey){
    string result;

    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::decryptKey\n";
    _userPrivateCryptKey.ref().setData(srckey);
    _userPrivateCryptKey.ref().decrypt();
    result=_userPrivateCryptKey.ref().getData();
    // unpadd
    result=result.substr(result.length() - 16);
    return result;
}


Error MediumKeyfileBase::verify(const string &data, const string &signature){
  if (Hbci::debugLevel()>15)
    cerr<<"MediumKeyfileBase::verify\n";
  _instPubSignKey.ref().setData(data);

  if (_instPubSignKey.ref().verify(signature))
    return Error();

  return Error("MediumKeyfileBase::verify",
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_INVALID,
	       ERROR_ADVISE_DONTKNOW,
	       "bad signature");
}


string MediumKeyfileBase::sign(const string &data){
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::sign\n";
    _userPrivateSignatureKey.ref().setData(data);
    _userPrivateSignatureKey.ref().sign();
    return _userPrivateSignatureKey.ref().getData();
}





Error MediumKeyfileBase::createUserKeys(bool activate){
  RSAKey *k1;
  RSAKey *k2;

  if (Hbci::debugLevel()>15)
    cerr<<"MediumKeyfileBase::createUserKeys\n";
  RSAKey::generateKeyPair(DEFAULT_KEY_LENGTH,
			  &k1,
			  &k2);
  _tempPrivateSignatureKey=k1;
  _tempPubSignatureKey=k2;

  _tempPrivateSignatureKey.setObjectDescription("userPrivateSignatureKey");
  _tempPrivateSignatureKey.ref().setToSignKey(true);
  _tempPrivateSignatureKey.ref().setUserId(_userId);
  _tempPubSignatureKey.setObjectDescription("userPubSignatureKey");
  _tempPubSignatureKey.ref().setToSignKey(true);
  _tempPubSignatureKey.ref().setUserId(_userId);

  RSAKey::generateKeyPair(DEFAULT_KEY_LENGTH,
			  &k1,
			  &k2);
  _tempPrivateCryptKey=k1;
  _tempPubCryptKey=k2;
  _tempPrivateCryptKey.setObjectDescription("userPrivateCryptKey");
  _tempPrivateCryptKey.ref().setToSignKey(false);
  _tempPrivateCryptKey.ref().setUserId(_userId);
  _tempPubCryptKey.setObjectDescription("userPubCryptKey");
  _tempPubCryptKey.ref().setToSignKey(false);
  _tempPubCryptKey.ref().setUserId(_userId);
  //writeFile(_path,_pin);

  if (activate)
    return activateKeys();
  return Error();
}


Error MediumKeyfileBase::activateKeys(){
  if (!_tempPrivateSignatureKey.isValid() ||
      !_tempPubSignatureKey.isValid() ||
      !_tempPrivateCryptKey.isValid() ||
      !_tempPubCryptKey.isValid())
    return Error("MediumKeyfileBase::activateKeys",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_DONTKNOW,
		 "no temporary keys created");

  _userPrivateSignatureKey=_tempPrivateSignatureKey;
  _userPubSignatureKey=_tempPubSignatureKey;

  _userPrivateCryptKey=_tempPrivateCryptKey;
  _userPubCryptKey=_tempPubCryptKey;

  return Error();
}


Pointer<RSAKey> MediumKeyfileBase::getTempSignKey(){
  return _tempPubSignatureKey;
}


Pointer<RSAKey> MediumKeyfileBase::getTempCryptKey(){
  return _tempPubCryptKey;
}


Error MediumKeyfileBase::setInstituteCryptKey(Pointer<RSAKey> cryptkey){
  if (Hbci::debugLevel()>15)
    cerr<<"MediumKeyfileBase::setInstituteCryptKey\n";
  _instPubCryptKey=cryptkey;
  return Error();
}


Error MediumKeyfileBase::setInstituteSignKey(Pointer<RSAKey> signkey){
  if (Hbci::debugLevel()>15)
    cerr<<"MediumKeyfileBase::setInstituteSignKey\n";
  _instPubSignKey=signkey;
  return Error();
}


void MediumKeyfileBase::setSystemId(const string &newid){
  if (Hbci::debugLevel()>15)
    cerr<<"MediumKeyfileBase::setSystemId\n";
  _systemid=newid;
}


bool MediumKeyfileBase::hasInstSignKey() const{
  if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::hasInstSignKey="<<_instPubSignKey.isValid()<<"\n";
    return _instPubSignKey.isValid();
}


Pointer<RSAKey> MediumKeyfileBase::userPubCryptKey() const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::userPubCryptKey\n";
    return _userPubCryptKey;
}


Pointer<RSAKey> MediumKeyfileBase::userPubSignKey() const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::userPubSignKey\n";
    return _userPubSignatureKey;
}


string MediumKeyfileBase::getInstIniLetterModulus(bool crypt) const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::getInstIniletterModulus\n";
    if (!crypt)
        return _instPubSignKey.ref().getIniLetterModulus();
    else
        return _instPubCryptKey.ref().getIniLetterModulus();
}


string MediumKeyfileBase::getInstIniLetterExponent(bool crypt) const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::getInstIniLetterExponent\n";
    if (!crypt)
        return _instPubSignKey.ref().getIniLetterExponent();
    else
        return _instPubCryptKey.ref().getIniLetterExponent();
}


string MediumKeyfileBase::getInstIniLetterHash(bool crypt) const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::getInstIniLetterHash\n";
    if (!crypt)
        return _instPubSignKey.ref().getIniLetterHash();
    else
        return _instPubCryptKey.ref().getIniLetterHash();
}


int MediumKeyfileBase::getInstKeyNumber(bool usecrypt) const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::getInstKeyNumber\n";
    return (usecrypt ?
            _instPubCryptKey.ref().number() :
            _instPubSignKey.ref().number());
}


int MediumKeyfileBase::getInstKeyVersion(bool usecrypt) const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::getInstKeyVersion\n";
    return (usecrypt ?
            _instPubCryptKey.ref().version() :
            _instPubSignKey.ref().version());
}


string MediumKeyfileBase::getUserIniLetterModulus() const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::getUserIniLetterModulus\n";
    return _userPubSignatureKey.ref().getIniLetterModulus();
}


string MediumKeyfileBase::getUserIniLetterExponent() const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::getUserIniLetterExponent\n";
    return _userPubSignatureKey.ref().getIniLetterExponent();
}


string MediumKeyfileBase::getUserIniLetterHash() const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::getUserIniLetterHash\n";
    return _userPubSignatureKey.ref().getIniLetterHash();
}


int MediumKeyfileBase::getUserKeyNumber() const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::getUserKeyNumber\n";
    return _userPubCryptKey.ref().number();
}


int MediumKeyfileBase::getUserKeyVersion() const{
    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::getUserKeyVersion\n";
    return _userPubCryptKey.ref().version();
}



Pointer<RSAKey> MediumKeyfileBase::_readKey(const string &data){
    Pointer<RSAKey> key;
    RSAKey::keyData kd;
    string v;
    string tag;
    unsigned int t;
    unsigned int pos;

    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::_readKey\n";
    pos=0;
    while(pos<data.length()) {
        tag=String::nextTLV(data,pos);
        v=String::dataTLV(tag);
        t=String::typeTLV(tag);
        switch(t) {
        case MEDIUM_RDH_TAG_KEY_ISPUBLIC:
            if (v=="yes" || v=="YES")
                kd.isPublic=true;
            else
                kd.isPublic=false;
            break;
        case MEDIUM_RDH_TAG_KEY_ISCRYPT:
            if (v=="yes" || v=="YES")
                kd.isCrypt=true;
            else
                kd.isCrypt=false;
            break;
        case MEDIUM_RDH_TAG_KEY_OWNER:
            kd.owner=v;
            break;
        case MEDIUM_RDH_TAG_KEY_VERSION:
            kd.version=atoi(v.c_str());
            break;
        case MEDIUM_RDH_TAG_KEY_NUMBER:
            kd.number=atoi(v.c_str());
            break;
        case MEDIUM_RDH_TAG_KEY_MODULUS:
            kd.modulus=v;
            break;
        case MEDIUM_RDH_TAG_KEY_EXP:
            kd.exponent=atoi(v.c_str());
            break;
        case MEDIUM_RDH_TAG_KEY_N:
            kd.n=v;
            break;

        case MEDIUM_RDH_TAG_KEY_P:
            kd.p=v;
            break;
        case MEDIUM_RDH_TAG_KEY_Q:
            kd.q=v;
            break;
        case MEDIUM_RDH_TAG_KEY_D:
            kd.d=v;
            break;
        case MEDIUM_RDH_TAG_KEY_DMP1:
            kd.dmp1=v;
            break;
        case MEDIUM_RDH_TAG_KEY_DMQ1:
            kd.dmq1=v;
            break;
        case MEDIUM_RDH_TAG_KEY_IQMP:
            kd.iqmp=v;
            break;
        default:
            break;
        } // switch
        pos+=tag.length();;
    } // while
    key=new RSAKey(&kd);
    return key;
}


Error MediumKeyfileBase::readContext(const string &data) {
    string v;
    string tag;
    unsigned int t;
    unsigned int pos;
    int i;

    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::readContext\n";
    pos=0;
    while(pos<data.length()) {
        tag=String::nextTLV(data,pos);
        v=String::dataTLV(tag);
        t=String::typeTLV(tag);

	if (Hbci::debugLevel()>10) {
	  cerr<<"checking tag type "<<t<<"\n";
	  //String::simpleDump(v);
	}
	switch(t) {
        case MEDIUMKEYFILE_TAG_VERSION_MAJOR:
            i=atoi(v.c_str());
            // is that version supported ?
            if (i!=MEDIUMKEYFILE_VERSION_MAJOR)
                return Error("MediumKeyFile::readContext()",
                             ERROR_LEVEL_CRITICAL,
                             0,
                             ERROR_ADVISE_ABORT,
                             "Unsupported Keyfile version.");
            break;
        case MEDIUMKEYFILE_TAG_VERSION_MINOR:
            // minor version is ignored for now
            break;

        case MEDIUMKEYFILE_TAG_SEQ:
            _seq=atoi(v.c_str());
            break;
        case MEDIUMKEYFILE_TAG_USER_PUBSIGNKEY:
            _userPubSignatureKey=_readKey(v);
            break;
        case MEDIUMKEYFILE_TAG_USER_PRIVSIGNKEY:
            _userPrivateSignatureKey=_readKey(v);
            break;
        case MEDIUMKEYFILE_TAG_USER_PUBCRYPTKEY:
            _userPubCryptKey=_readKey(v);
            break;
        case MEDIUMKEYFILE_TAG_USER_PRIVCRYPTKEY:
	  _userPrivateCryptKey=_readKey(v);
	  break;
	  // read temporary keys
	case MEDIUMKEYFILE_TAG_TEMP_PUBSIGNKEY:
	  _tempPubSignatureKey=_readKey(v);
	  break;
	case MEDIUMKEYFILE_TAG_TEMP_PRIVSIGNKEY:
	  _tempPrivateSignatureKey=_readKey(v);
	  break;
	case MEDIUMKEYFILE_TAG_TEMP_PUBCRYPTKEY:
	  _tempPubCryptKey=_readKey(v);
	  break;
	case MEDIUMKEYFILE_TAG_TEMP_PRIVCRYPTKEY:
	  _tempPrivateCryptKey=_readKey(v);
	  break;
	case MEDIUMKEYFILE_TAG_USER_ID:
            _userId=v;
            break;
        case MEDIUMKEYFILE_TAG_INST_PUBSIGNKEY:
            _instPubSignKey=_readKey(v);
            break;
        case MEDIUMKEYFILE_TAG_INST_PUBCRYPTKEY:
            _instPubCryptKey=_readKey(v);
            break;
        case MEDIUMKEYFILE_TAG_INST_COUNTRY:
            _country=atoi(v.c_str());
            break;
        case MEDIUMKEYFILE_TAG_INST_SYSTEMID:
            _systemid=v;
            break;
        case MEDIUMKEYFILE_TAG_INST_CODE:
            _instcode=v;
            break;
        default:
            return Error("MediumKeyFile::readContext()",
                         ERROR_LEVEL_CRITICAL,
                         0,
                         ERROR_ADVISE_ABORT,
                         "Unknown tag found.");
            break;
        } // switch
        pos+=tag.length();
    } // while
    return Error();
}


string MediumKeyfileBase::_writeKey(Pointer<RSAKey> key) const {
    RSAKey::keyData kd;
    string result;
    char buffer[32];

    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::_writeKey\n";
    if (!key.isValid())
        return "";
    if (!key.ref().getKeyData(&kd))
        return "";

    result+=String::newTLV(MEDIUM_RDH_TAG_KEY_ISPUBLIC,
                           (kd.isPublic)?"YES":"NO");
    result+=String::newTLV(MEDIUM_RDH_TAG_KEY_ISCRYPT,
                           (kd.isCrypt)?"YES":"NO");
    if (!kd.owner.empty())
        result+=String::newTLV(MEDIUM_RDH_TAG_KEY_OWNER,
                               kd.owner);

    sprintf(buffer,"%d",kd.number);
    result+=String::newTLV(MEDIUM_RDH_TAG_KEY_NUMBER,
                           buffer);
    sprintf(buffer,"%d",kd.version);
    result+=String::newTLV(MEDIUM_RDH_TAG_KEY_VERSION,
                           buffer);
    sprintf(buffer,"%d",kd.exponent);
    result+=String::newTLV(MEDIUM_RDH_TAG_KEY_EXP,
                           buffer);

    if (!kd.modulus.empty())
        result+=String::newTLV(MEDIUM_RDH_TAG_KEY_MODULUS,
                               kd.modulus);
    if (!kd.n.empty())
        result+=String::newTLV(MEDIUM_RDH_TAG_KEY_N,
                               kd.n);
    if (!kd.p.empty())
        result+=String::newTLV(MEDIUM_RDH_TAG_KEY_P,
                               kd.p);
    if (!kd.q.empty())
        result+=String::newTLV(MEDIUM_RDH_TAG_KEY_Q,
                                   kd.q);
    if (!kd.d.empty())
        result+=String::newTLV(MEDIUM_RDH_TAG_KEY_D,
                               kd.d);
    if (!kd.dmp1.empty())
        result+=String::newTLV(MEDIUM_RDH_TAG_KEY_DMP1,
                               kd.dmp1);
    if (!kd.dmq1.empty())
        result+=String::newTLV(MEDIUM_RDH_TAG_KEY_DMQ1,
                                   kd.dmq1);
    if (!kd.iqmp.empty())
        result+=String::newTLV(MEDIUM_RDH_TAG_KEY_IQMP,
                                   kd.iqmp);
    return result;
}


string MediumKeyfileBase::writeContext() const {
    string result;
    char buffer[32];
    string tag;

    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfileBase::writeContext\n";
    sprintf(buffer,"%d",MEDIUMKEYFILE_VERSION_MAJOR);
    result+=String::newTLV(MEDIUMKEYFILE_TAG_VERSION_MAJOR,
                           buffer);
    sprintf(buffer,"%d",MEDIUMKEYFILE_VERSION_MINOR);
    result+=String::newTLV(MEDIUMKEYFILE_TAG_VERSION_MINOR,
                           buffer);
    sprintf(buffer,"%d",_seq);
    result+=String::newTLV(MEDIUMKEYFILE_TAG_SEQ,
                           buffer);
    if (Hbci::debugLevel()>15)
	cerr<<"Writing SEQ of "<<_seq<<"\n";
    tag=_writeKey(_userPubSignatureKey);
    if (!tag.empty())
        result+=String::newTLV(MEDIUMKEYFILE_TAG_USER_PUBSIGNKEY,tag);

    tag=_writeKey(_userPrivateSignatureKey);
    if (!tag.empty())
        result+=String::newTLV(MEDIUMKEYFILE_TAG_USER_PRIVSIGNKEY,tag);

    tag=_writeKey(_userPubCryptKey);
    if (!tag.empty())
        result+=String::newTLV(MEDIUMKEYFILE_TAG_USER_PUBCRYPTKEY,tag);

    tag=_writeKey(_userPrivateCryptKey);
    if (!tag.empty())
        result+=String::newTLV(MEDIUMKEYFILE_TAG_USER_PRIVCRYPTKEY,tag);


    // write temporary keys
    if (_tempPubSignatureKey.isValid() &&
	_tempPrivateSignatureKey.isValid() &&
	_tempPubCryptKey.isValid() &&
	_tempPrivateCryptKey.isValid()) {
      tag=_writeKey(_tempPubSignatureKey);
      if (!tag.empty())
	result+=String::newTLV(MEDIUMKEYFILE_TAG_TEMP_PUBSIGNKEY,tag);

      tag=_writeKey(_tempPrivateSignatureKey);
      if (!tag.empty())
	result+=String::newTLV(MEDIUMKEYFILE_TAG_TEMP_PRIVSIGNKEY,tag);

      tag=_writeKey(_tempPubCryptKey);
      if (!tag.empty())
	result+=String::newTLV(MEDIUMKEYFILE_TAG_TEMP_PUBCRYPTKEY,tag);

      tag=_writeKey(_tempPrivateCryptKey);
      if (!tag.empty())
	result+=String::newTLV(MEDIUMKEYFILE_TAG_TEMP_PRIVCRYPTKEY,tag);
    }

    if (!_userId.empty()) {
      result+=String::newTLV(MEDIUMKEYFILE_TAG_USER_ID,
			     _userId);
    }


    tag=_writeKey(_instPubSignKey);
    if (!tag.empty())
        result+=String::newTLV(MEDIUMKEYFILE_TAG_INST_PUBSIGNKEY,tag);
    tag=_writeKey(_instPubCryptKey);
    if (!tag.empty())
        result+=String::newTLV(MEDIUMKEYFILE_TAG_INST_PUBCRYPTKEY,tag);

    sprintf(buffer,"%d",_country);
    result+=String::newTLV(MEDIUMKEYFILE_TAG_INST_COUNTRY,
                           buffer);

    if (!_instcode.empty())
        result+=String::newTLV(MEDIUMKEYFILE_TAG_INST_CODE,
                               _instcode);
    if (!_systemid.empty())
        result+=String::newTLV(MEDIUMKEYFILE_TAG_INST_SYSTEMID,
                               _systemid);

    return result;
}









MediumKeyfile::MediumKeyfile(const Hbci *hbci, const string &path)
:MediumKeyfileBase(hbci)
,_mountCount(0)
,_path(path)
{
  if (_fileExists(path.c_str())) {
      File f(path);
      s_filestat s;
      Error err = f.statFile(s);
      if (!err.isOk()) {
	  //fprintf(stderr, "MediumKeyfile::MediumKeyfile: could not stat file  %s because of %s\n", path.c_str(), err.errorString().c_str());
	  //throw err;
      }
      keyfile_mode = s.mode | FILE_CM_WRITE | FILE_CM_READ &
	  (FILE_CM_WRITE | FILE_CM_READ
	   | FILE_CM_GROUP_WRITE | FILE_CM_GROUP_READ 
	   | FILE_CM_OTHER_WRITE | FILE_CM_OTHER_READ);
      //fprintf(stderr, "MediumKeyfile::MediumKeyfile: gott s.mode 0%o and keyfile_mode 0%o for file %s\n", s.mode, keyfile_mode, path.c_str());
      if (keyfile_mode & (FILE_CM_OTHER_WRITE | FILE_CM_OTHER_READ))
	  fprintf(stderr,"MediumKeyfile::_writeFile(): Your key file %s is created with world-readable and/or -writable permissions. Please change the file acces permissions so that it is no longer world-readable and -writable.\n", path.c_str());
  }
  //else
  //  fprintf(stderr, "MediumKeyfile::MediumKeyfile: file %s not existing", path.c_str());
}


MediumKeyfile::~MediumKeyfile(){
  //fprintf(stderr,"Destroying mediumKeyfile %s.\n", _path.c_str());
}

/*
void MediumKeyfile::clear(){
  _lastMounter=0;
  Medium::clear();
}
*/

Error MediumKeyfile::mountMedium(const string &pin){
  try {
    Error err;
    string localpin;
    File f(_path);
    Pointer<Interactor> ia;

    ia=hbci()->interactor();

    if (Hbci::debugLevel()>15)
      cerr<<"MediumKeyfile::mountMedium\n";

    _hbci->interactor().ref().abort(false);

    _lastMounter=owner();
    if (_mountCount<1) {
      // check if file exists
      while (1) {
	err=f.accessFile(FILE_ACCESS_EXIST);
	if (!err.isOk()) {
	  if (!ia.ref().msgInsertMediumOrAbort(_lastMounter,
					       MediumTypeFile))
	    return Error("MediumKeyFile::mountMedium()",
			 ERROR_LEVEL_NORMAL,
			 HBCI_ERROR_CODE_FILE_NOT_FOUND,
			 ERROR_ADVISE_ABORT,
			 "File not found and user aborted PIN dialog.");
	}
	else
	  break;
      } // while

      // check/ask for pin
      localpin=pin;
      while (localpin.length()<minPinSize) {
	err=hbci()->authentificator().ref().getSecret(owner(),
						      _path,
						      localpin);
	if (!err.isOk())
	  return Error("MediumKeyFile::mountMedium()",
		       ERROR_LEVEL_NORMAL,
		       HBCI_ERROR_CODE_PIN_ABORTED,
		       ERROR_ADVISE_ABORT,
		       "Bad PIN and user aborted PIN dialog.");
	if (localpin.length()<minPinSize)
	  return Error("MediumKeyFile::mountMedium()",
		       ERROR_LEVEL_CRITICAL,
		       HBCI_ERROR_CODE_PIN_TOO_SHORT,
		       ERROR_ADVISE_ABORT,
		       "BAD PROGRAM: Your program returns a pin that is\n"
		       "shorter than the given minimum length.\n"
		       "This is a severe internal error of your "
		       "application,\n"
		       "please report to the author of this application.");
      } // while
      err=_readFile(_path,localpin);
      if (!err.isOk()) {
	if (Hbci::debugLevel()>2)
	  cerr<<"MediumKeyfile::mountMedium: "<<err.errorString()<<"\n";
	return err;
      }
      _mountCount=1;
      _pin=localpin;
    }
    else {
      _mountCount++;
    }
    return Error();
  } // try
  catch (Error xerr) {
    return Error("MediumKeyfile::mountMedium()",xerr);
  }
}


Error MediumKeyfile::unmountMedium(const string &pin){
  Error err;
  string localpin;

  if (Hbci::debugLevel()>15)
    cerr<<"MediumKeyfile::unmountMedium\n";

  if (_mountCount==1) {
    if (!pin.empty())
      localpin=pin;
    else
      localpin=_pin;
    if (_backupFiles(_path.c_str(), KEYFILE_BACKUPS)) {
      cerr<<"MediumKeyfile::unmountMedium: Error creating backups\n";
    }
    err=_writeFile(_path,localpin);
    if (!err.isOk()) {
      if (Hbci::debugLevel()>2)
	cerr<<"MediumKeyfile::unmountMedium: "<<err.errorString()<<"\n";
    }
    _pin.erase();
    _mountCount=0;
  }
  else if (_mountCount)
    _mountCount--;
  if (Hbci::debugLevel()>3)
    cerr<<"MediumKeyfile::unmountMedium done.\n";
  return err;
}


Error MediumKeyfile::createMedium(int country,
				  const string &instcode,
				  const string &userid,
				  const string &pin){
  Error err;
  string localpin;
  Pointer<Interactor> ia;


  if (Hbci::debugLevel()>15)
    cerr<<"MediumKeyfile::createMedium";
  if (Hbci::debugLevel()>15)
    cerr<<" Country="<<country<<" BankCode="<<instcode
      <<" UserId="<<userid<<"\n";

  if (_mountCount) {
    if (Hbci::debugLevel()>0)
      cerr<<"MediumKeyfile::createMedium: already mounted !\n";
    return Error("MediumKeyfile::createMedium",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_DONTKNOW,
		 "medium already mounted");
  }

  ia=hbci()->interactor();
  err=MediumKeyfileBase::createMedium(country,
				      instcode,
				      userid);
  if (!err.isOk()) {
    if (Hbci::debugLevel()>0)
      cerr<<"MediumKeyfile::createMedium: Could not create medium.\n";
    return Error("MediumKeyfile::createMedium", err);
  }

  // let the user insert and physically mount the medium
  if (!ia.ref().msgInsertMediumOrAbort(owner(),
				       MediumTypeFile)) {
    ia.ref().msgStateResponse("No medium, user aborted");
    return Error("MediumKeyfile::createMedium",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_DONTKNOW,
		 "no medium, user aborted");
  }

  // check/ask for pin, create secret if necessary
  localpin=pin;
  while (localpin.length()<minPinSize) {
    err=hbci()->authentificator().ref().getSecret(0,
						  _path,
						  localpin,
						  true);
    if (!err.isOk()) {
      ia.ref().msgStateResponse("Bad pin, user aborted.");
      return Error("MediumKeyfile::createMedium",
		   ERROR_LEVEL_NORMAL,
		   HBCI_ERROR_CODE_PIN_ABORTED,
		   ERROR_ADVISE_DONTKNOW,
		   "bad pin, user aborted");
    }
    if (localpin.length()<minPinSize) {
      fprintf(stderr,
	      "BAD PROGRAM: Your program returns a pin that is\n"
	      "shorter than the given minimum length.\n"
	      "This is a severe internal error of your "
	      "application,\n"
	      "please report to the author of this application.");
      ia.ref().msgStateResponse("Pin too short, aborting.");
      return Error("MediumKeyfile::createMedium",
		   ERROR_LEVEL_NORMAL,
		   HBCI_ERROR_CODE_PIN_TOO_SHORT,
		   ERROR_ADVISE_DONTKNOW,
		   "PIN too short");
    }
  } // while
  _pin=localpin;

  // now write the file
  err=_writeFile(_path,localpin);
  if (!err.isOk()) {
    if (Hbci::debugLevel()>0)
      cerr<<"MediumKeyfile::createMedium: "<<err.errorString()<<"\n";
    return Error("MediumKeyfile::createMedium",err);
  }
  return Error();
}


unsigned int MediumKeyfile::nextSEQ(){
    unsigned int s;
    Error err;

    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfile::nextSEQ\n";

    s=MediumKeyfileBase::nextSEQ();

    err=_writeFile(_path,_pin);
    if (!err.isOk()) {
        if (Hbci::debugLevel()>0)
            cerr<<"MediumKeyfile::nextSEQ: "<<err.errorString()<<"\n";
    }
    if (Hbci::debugLevel()>3)
        cerr<<"Will return SEQ= "<<s<<"\n";
    return s;
}


void MediumKeyfile::resetSEQ(){
    Error err;

    if (Hbci::debugLevel()>15)
	cerr<<"MediumKeyfile::resetSEQ\n";

    MediumKeyfileBase::resetSEQ();

    err=_writeFile(_path,_pin);
    if (!err.isOk()) {
	if (Hbci::debugLevel()>0)
	    cerr<<"MediumKeyfile::resetSEQ: "<<err.errorString()<<"\n";
    }
}


void MediumKeyfile::setSEQ(int seq){
  Error err;

  if (Hbci::debugLevel()>15)
    cerr<<"MediumKeyfile::setSEQ\n";

  MediumKeyfileBase::setSEQ(seq);

  err=_writeFile(_path,_pin);
  if (!err.isOk()) {
    if (Hbci::debugLevel()>0)
      cerr<<"MediumKeyfile::resetSEQ: "<<err.errorString()<<"\n";
  }
}


Error MediumKeyfile::_reallyReadFile(File f, string &result) {
    string tmp;
    int size;
    int rest;
    Error err;

    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfile::_reallyReadFile\n";

    // read tag type and size
    err=f.readData(tmp,3);
    if (!err.isOk()) {
      return Error("MediumKeyfile::_reallyReadFile", err);
    }
    if (tmp.length()!=3)
        return Error("MediumKeyFile::_reallyReadFile()",
                     ERROR_LEVEL_NORMAL,
                     0,
                     ERROR_ADVISE_ABORT,
                     "This seems not to be a key file.");
    if (String::typeTLV(tmp)!=MEDIUMKEYFILE_TAG_CRYPT)
        return Error("MediumKeyFile::_reallyReadFile()",
                     ERROR_LEVEL_NORMAL,
                     0,
                     ERROR_ADVISE_ABORT,
                     "This seems not to be a key file (bad type).");
    size=String::sizeTLV(tmp);
    if (size % 8)
        return Error("MediumKeyFile::_reallyReadFile()",
                     ERROR_LEVEL_NORMAL,
                     0,
                     ERROR_ADVISE_ABORT,
                     "This seems not to be a key file (bad size).");
    result+=tmp;
    while (size) {
        tmp.erase();
        // read 1k blocks
        if (size<1024)
            rest=size;
        else
            rest=1024;
        err=f.readData(tmp,rest);
        if (!err.isOk() || tmp.length()<1)
            return Error("MediumKeyFile::_reallyReadFile()",
                         ERROR_LEVEL_NORMAL,
                         0,
                         ERROR_ADVISE_ABORT,
                         "Error reading key file.");
        result+=tmp;
        size-=tmp.length();
    } // while
    // done
    return Error();
}


Error MediumKeyfile::_readFile(const string &path,
			       const string &pin){
    Error err,err2;
    File f(path);
    Pointer<Interactor> ia;
    string rawcontent;
    unsigned int pos;
    string tag;
    int t;
    string result;

    if (Hbci::debugLevel()>15)
	cerr<<"MediumKeyfile::_readFile\n";
    if (pin.length()<minPinSize)
      return Error("MediumKeyFile::_readFile()",
                   ERROR_LEVEL_INTERNAL,
                   0,
                   ERROR_ADVISE_SHUTDOWN,
                   "There is no pin. This is absolutely unexpected and\n"
                   "indicates a severe error either in OpenHBCI or your\n"
                   "application. Please report this bug to the author of\n"
                   "your homebanking application or, if he is unresponsive\n"
                   "report it to martin@libchipcard.de\n");

    // read file
    err=f.openFile(FILE_AM_RDONLY);
    if (!err.isOk())
      return Error("MediumKeyfile::_readFile",  err);
    err=_reallyReadFile(f,rawcontent);
    err2=f.closeFile();
    // first err takes precedence
    if (!err.isOk())
	return err;
    if (!err2.isOk())
	return err2;

    // now decrypt the medium
    DESKey dk(pin);
    dk.setData(String::dataTLV(rawcontent));
    dk.decrypt();
    result=dk.getData(); // decrypted tag data

    if (Hbci::debugLevel()>30) {
      fprintf(stderr,
              "\n"
	      "================================================\n"
	      "VERY SERIOUS SECURITY WARNING:\n"
	      "The following part possibly includes you private keys !\n"
	      "Please remove the following output before submitting this\n"
	      "data to anyone !!!\n"
	      "The end of the sensitive output is marked below.\n"
	      "================================================\n"
              "\n");
      String::simpleDump(result);
      fprintf(stderr,
	      "\n"
	      "End of sensitive data.\n"
	      "================================================\n"
	      "\n");
    }

    // check for correct decryption
    pos=0;
    tag=String::nextTLV(result,pos);
    t=String::typeTLV(tag);
    // first tag must be VERSION_MAJOR, otherwise the pin was bad
    if (t!=MEDIUMKEYFILE_TAG_VERSION_MAJOR)
	throw Error("MediumKeyFile::_readFile()",
                    ERROR_LEVEL_NORMAL,
                    HBCI_ERROR_CODE_PIN_WRONG,
                    ERROR_ADVISE_ABORT,
                    "The PIN is bad and does not match the given key file.");
    err=readContext(result);
    if (!err.isOk())
        return err;
    return Error();
}


Error MediumKeyfile::_writeFile(const string &path,
                                const string &pin){
    string data;
    string encdata;
    string outdata;
    string tmppath;
    Error err, err2;

    if (Hbci::debugLevel()>15)
        cerr<<"MediumKeyfile::_writeFile\n";

    if (pin.length()<minPinSize)
	throw Error("MediumKeyFile::_writeFile()",
		    ERROR_LEVEL_INTERNAL,
		    0,
		    ERROR_ADVISE_SHUTDOWN,
		    "There is no pin. This is absolutely unexpected and\n"
		    "indicates a severe error either in OpenHBCI or your\n"
		    "application. Please report this bug to the author of\n"
		    "your homebanking application or, if he is unresponsive\n"
		    "report it to martin@libchipcard.de\n");

    // create context data
    data=writeContext();

    // check/ask for pin
    /*
    if (!pin.empty())
        localpin=pin;
    else
        localpin=_pin;
    while (localpin.length()<minPinSize) {
	if (!ia.ref().msgInputPin(_lastMounter,
				  localpin,
				  minPinSize,
				  true))
	    return Error("MediumKeyFile::_writeFile()",
			 ERROR_LEVEL_NORMAL,
			 HBCI_ERROR_CODE_PIN_ABORTED,
			 ERROR_ADVISE_ABORT,
			 "Bad pin, user aborted.");
	if (localpin.length()<minPinSize)
	    return Error("MediumKeyFile::_writeFile()",
			 ERROR_LEVEL_CRITICAL,
			 HBCI_ERROR_CODE_PIN_TOO_SHORT,
			 ERROR_ADVISE_ABORT,
			 "BAD PROGRAM: Your program returns a pin that is\n"
			 "shorter than the given minimum length.\n"
			 "This is a severe internal error of your "
			 "application,\n"
			 "please report to the author of this application.");
			 } // while
			 */
    if (pin.length()<minPinSize)
	return Error("MediumKeyFile::_writeFile()",
		     ERROR_LEVEL_CRITICAL,
		     0,
		     ERROR_ADVISE_ABORT,
		     "There is a pin that is too short. This is absolutely\n"
		     "unexpected and indicates a severe error either in \n"
		     "OpenHBCI or your application. Please file a bug report.");

    // encrypt data
    DESKey dk(pin);
    dk.setData(data);
    dk.encrypt();
    encdata=dk.getData(); // decrypted tag data

    // make TLV of it
    outdata=String::newTLV(MEDIUMKEYFILE_TAG_CRYPT,
                           encdata);

    // now create temp file
    tmppath=path+".tmp";
    File f(tmppath);
    //fprintf(stderr, "MediumKeyfile::_writeFile: mode is 0%o for file %s\n", keyfile_mode, path.c_str());

    keyfile_mode = keyfile_mode | FILE_CM_WRITE | FILE_CM_READ;
//#if OS_LINUX
    // modify umask so that the desired file permissions are forced to be accepted
    //mode_t previous_umask = umask(~keyfile_mode);
    // huh? doesn't work
//#endif
    err=f.createFile(FILE_AM_RDWR | FILE_AM_OPEN_ALWAYS,
                     keyfile_mode);
//#if OS_LINUX
    //umask(previous_umask);
//#endif
    if (!err.isOk())
        return err;

    // write data to temp file and close it
    err=f.writeData(outdata);
    err2=f.closeFile();
    if (!err.isOk())
        return err;
    if (!err2.isOk())
        return err2;

    // rename temp file to target file
    err=f.renameFile(path);
    if (!err.isOk())
        return err;

    if (Hbci::debugLevel()>2)
        cerr<<"MediumKeyfile::_writeFile done.\n";
    return Error();
}


Error MediumKeyfile::createUserKeys(bool overwrite, bool activate){
  Error err;

  if (Hbci::debugLevel()>15)
    cerr<<"MediumKeyfile::createUserKeys ("<<overwrite<<")\n";

  if (_mountCount<1) {
    if (Hbci::debugLevel()>0)
      cerr<<"MediumKeyfile::createMedium: not mounted !\n";
    return Error("MediumKeyfile::createUserKeys",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_DONTKNOW,
		 "medium not mounted");
  }

  if (userPubCryptKey().isValid() && !overwrite) {
    if (Hbci::debugLevel()>0)
      cerr<<"MediumKeyfile::createMedium: keys already existing !\n";
    return Error("MediumKeyfile::createUserKeys",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_EXISTS,
		 ERROR_ADVISE_DONTKNOW,
		 "keys already exist");
  }
  err=MediumKeyfileBase::createUserKeys(activate);
  if (!err.isOk()) {
    if (Hbci::debugLevel()>0)
      cerr<<"MediumKeyfile::createUserkeys: Could not create keys.\n";
    return Error("MediumKeyfile::createMedium", err);
  }

  err=_writeFile(_path,_pin);
  if (!err.isOk()) {
    if (Hbci::debugLevel()>0)
      cerr<<"MediumKeyfile::createUserKeys: "<<err.errorString()<<"\n";
    return Error("MediumKeyfile::createMedium", err);
  }
  return Error();
}


Error MediumKeyfile::activateKeys(){
  return MediumKeyfileBase::activateKeys();
}


Pointer<RSAKey> MediumKeyfile::getTempSignKey(){
  return MediumKeyfileBase::getTempSignKey();
}


Pointer<RSAKey> MediumKeyfile::getTempCryptKey(){
  return MediumKeyfileBase::getTempCryptKey();
}


Error MediumKeyfileBase::changePIN() {
  // make sure, the medium is not mounted
  if (isMounted())
    unmountMedium();

  Pointer<Interactor> ia = hbci()->interactor();

  string oldPin = "";
  string newPin = "";
  bool changed = false;

  if (ia.ref().msgInputPin(owner(), oldPin, minPinSize, false) &&
      mountMedium(oldPin).isOk()) {

    if (ia.ref().msgInputPin(owner(), newPin, minPinSize, true))
      changed = true;
    else
      newPin = oldPin;

    unmountMedium(newPin);
  }
  if (changed)
    return Error();
  return Error("MediumKeyfileBase::changePIN",
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_INVALID,
	       ERROR_ADVISE_DONTKNOW,
	       "PIN not changed");
}


Error MediumKeyfileBase::changeContext(int context, int country,
				       const string instcode,
				       const string userid,
				       const string custid,
				       const string server) {
  Error err;

  err=mountMedium();
  if (!err.isOk())
    return Error("MediumKeyfileBase::changeContext",err);

  // server and customerid are not used in keyfiles

  if (0 != country)
    _country = country;
  if (! instcode.empty())
    _instcode = instcode;
  if (! userid.empty())
    _userId = userid;

  return unmountMedium();
}


int MediumKeyfile::_fileExists(const char *path) {
  FILE *f;

  f=fopen(path,"r");
  if (f!=0) {
    fclose(f);
    return 1;
  }
  return 0;
}


int MediumKeyfile::_backupFiles(const char *path, int steps) {
  char buffer1[256];
  char buffer2[256];
  int i;

  for (i=steps; i>=0; i--) {
    // create filenames
    if (i==0) {
      snprintf(buffer1, sizeof(buffer1),
	       "%s.bak", path);
      snprintf(buffer2, sizeof(buffer2),
	       "%s", path);
    }
    else {
      snprintf(buffer1, sizeof(buffer1),
	       "%s.bak.%d", path, i);
      if (i==1)
	snprintf(buffer2, sizeof(buffer2),
		 "%s.bak", path);
      else
	snprintf(buffer2, sizeof(buffer2),
		 "%s.bak.%d", path, i-1);
    }

    // remove old file, if it is the last one
    if (i==steps)
      unlink(buffer1);

    // rename previous file to current ("x.bak.1"->"x.bak.2")
    if (_fileExists(buffer2)) {
      if (rename(buffer2, buffer1)) {
	fprintf(stderr,"Error: rename(%s, %s): %s\n",
		buffer2, buffer1, strerror(errno));
	return 1;
      }
    }

  } // for

  return 0;
}



} // namespace







