/* otfdrive.c -- OpenType font driver.

Copyright (C) 2003, 2004
  National Institute of Advanced Industrial Science and Technology (AIST)
  Registration Number H15PRO167

This file is part of libotf.

Libotf is free software; you can redistribute it and/or modify it
under the terms of the GNU Lesser General Public License as published
by the Free Software Foundation; either version 2.1 of the License, or
(at your option) any later version.

Libotf is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library, in a file named COPYING; if not,
write to the Free Software Foundation, Inc., 59 Temple Place, Suite
330, Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <config.h>

#include "otf.h"
#include "otferror.h"

/* Return nonzero if OTF_Glyph *G should be ignored according to
   LookupFlag FLAG.  */
#define IGNORED_GLYPH(g, flag)			\
  ((g)->glyph_id == 0				\
   || ((flag) & (1 << (g)->GlyphClass))		\
   || (((flag) & OTF_MarkAttachmentType)	\
       && (g)->GlyphClass == OTF_GlyphClassMark	\
       && ((flag) >> 8) != (g)->MarkAttachClass))

#define GSTRING_DELETE(gstring, from, len)				\
  do {									\
    memmove (gstring->glyphs + from, gstring->glyphs + from + len,	\
	     sizeof (OTF_Glyph) * (gstring->used - from - len));	\
    gstring->used -= len;						\
  } while (0)


#define GSTRING_INSERT(gstring, pos, len)				\
  do {									\
    if (gstring->used + len > gstring->size)				\
      {									\
	char *errfmt = "GSTRING%s";					\
									\
	gstring->size = gstring->used + len;				\
	gstring->glyphs							\
	  = (OTF_Glyph *) realloc (gstring->glyphs,			\
				   sizeof (OTF_Glyph) * gstring->size);	\
	if (! gstring->glyphs)						\
	  OTF_ERROR (OTF_ERROR_MEMORY, "");				\
      }									\
    memmove (gstring->glyphs + pos + len, gstring->glyphs + pos,	\
	     sizeof (OTF_Glyph) * (gstring->used - pos));		\
    gstring->used += len;						\
  } while (0)


static int
gstring_subst (OTF *otf, OTF_GlyphString *gstring, int from, int to, int flag,
	       OTF_GlyphID *ids, int num)
{
  int errret = -1;
  int len = to - from;
  int i;
  int from_idx = gstring->glyphs[from].f.index.from;
  int to_idx = gstring->glyphs[to - 1].f.index.to;
  int non_ignored_idx;

  for (i = non_ignored_idx = to - 1; i >= from; i--)
    {
      OTF_Glyph *g = gstring->glyphs + i;
      if (IGNORED_GLYPH (g, flag))
	{
	  OTF_Glyph temp = *g;

	  memmove (g, g + 1, sizeof (OTF_Glyph) * (non_ignored_idx - i));
	  temp.f.index.from = from_idx;
	  temp.f.index.to = to_idx;
	  gstring->glyphs[non_ignored_idx--] = temp;
	  len--;
	}
    }

  if (len < num)
    GSTRING_INSERT (gstring, from, (num - len));
  else if (len > num)
    GSTRING_DELETE (gstring, from, (len - num));
  for (i = 0; i < num; i++)
    {
      gstring->glyphs[from + i].c = otf->cmap->decode_table[ids[i]];
      gstring->glyphs[from + i].glyph_id = ids[i];
      gstring->glyphs[from + i].f.index.from = from_idx;
      gstring->glyphs[from + i].f.index.to = to_idx;
    }
  return 0;
}


static int
get_coverage_index (OTF_Coverage *coverage, OTF_GlyphID id)
{
  int i;

  if (coverage->CoverageFormat == 1)
    {
      for (i = 0; i < coverage->Count; i++)
	if (coverage->table.GlyphArray[i] == id)
	  return i;
    }
  else
    {
      for (i = 0; i < coverage->Count; i++)
	if (coverage->table.RangeRecord[i].Start <= id
	    && coverage->table.RangeRecord[i].End >= id)
	  return (coverage->table.RangeRecord[i].StartCoverageIndex
		  + (id - coverage->table.RangeRecord[i].Start));
    }
  return -1;
}

static unsigned
get_class_def (OTF_ClassDef *class_def, OTF_GlyphID glyph_id)
{
  if (class_def->ClassFormat == 1)
    {
      int idx = (int) glyph_id - (int) class_def->f.f1.StartGlyph;

      if (idx >= 0 && idx < class_def->f.f1.GlyphCount)
	return class_def->f.f1.ClassValueArray[idx];
    }
  else
    {
      int i;

      for (i = 0; i < class_def->f.f2.ClassRangeCount; i++)
	if (glyph_id >= class_def->f.f2.ClassRangeRecord[i].Start
	    && glyph_id <= class_def->f.f2.ClassRangeRecord[i].End)
	  return class_def->f.f2.ClassRangeRecord[i].Class;
    }
  return 0;
}

static OTF_LangSys *
get_langsys (OTF_ScriptList *script_list, char *script, char *language)
{

  OTF_Tag script_tag = OTF_tag (script);
  OTF_Tag langsys_tag = OTF_tag (language);
  int i, j;
  OTF_Tag dflt_tag = OTF_tag ("DFLT");
  OTF_Script *dflt = NULL;

  for (i = 0; i < script_list->ScriptCount; i++)
    {
      OTF_Script *script = script_list->Script + i;

      if (script_list->Script[i].ScriptTag == dflt_tag)
	dflt = script;
      if (script_list->Script[i].ScriptTag == script_tag)
	{
	  if (! langsys_tag)
	    return &script->DefaultLangSys;
	  for (j = 0; j < script->LangSysCount; j++)
	    if (script->LangSysRecord[j].LangSysTag == langsys_tag)
	      return script->LangSys + j;
	  return &script->DefaultLangSys;	
	}
    }

  if (! dflt)
    dflt = script_list->Script;
  if (! langsys_tag)
    return &dflt->DefaultLangSys;
  for (j = 0; j < dflt->LangSysCount; j++)
    if (dflt->LangSysRecord[j].LangSysTag == langsys_tag)
      return dflt->LangSys + j;
  return &dflt->DefaultLangSys;	
}

static int
setup_lookup_indices (OTF_LookupList *LookupList, OTF_FeatureList *FeatureList,
		      char *features, int *lookup_indices)
{
  int i, j, n = 0;
  OTF_Feature *feature;
  int *feature_table = alloca (sizeof (int) * FeatureList->FeatureCount);

  for (i = 0; i < FeatureList->FeatureCount; i++)
    feature_table[i] = 0;

  while (*features)
    {
      char tagname[4];
      OTF_Tag tag;
      int use_it = 1;

      if (*features == '*')
	{
	  /* Consume all remaining features.  */
	  for (i = 0; i < FeatureList->FeatureCount; i++)
	    if (! feature_table[i])
	      {
		feature = FeatureList->Feature + i;
		for (j = 0; j < feature->LookupCount; j++)
		  lookup_indices[n++] = feature->LookupListIndex[j];
	      }
	  break;
	}

      if (*features == '~')
	use_it = -1, features++;
      for (i = 0; *features && *features != ','; i++, features++)
	tagname[i] = *features;
      if (*features)
	/* Skip ',' */
	features++;
      for (; i < 4; i++)
	tagname[i] = '\0';
      tag = OTF_tag (tagname);
      for (i = 0; i < FeatureList->FeatureCount; i++)
	{
	  feature = FeatureList->Feature + i;
	  if (tag == feature->FeatureTag)
	    {
	      if (feature_table[i])
		break;
	      if (use_it > 0)
		for (j = 0; j < feature->LookupCount; j++)
		  lookup_indices[n++] = feature->LookupListIndex[j];
	      feature_table[i] = use_it;
	      break;
	    }
	}
    }

  return n;
}

static int
match_ids (OTF_GlyphString *gstring, int gidx, int flag,
	   int count, OTF_GlyphID *ids)
{
  OTF_Glyph *gbeg = gstring->glyphs + gidx;
  OTF_Glyph *gend = gstring->glyphs + gstring->used;
  OTF_Glyph *g;
  int i;

  for (g = gbeg, i = 0; g < gend && i < count; g++)
    if (! IGNORED_GLYPH (g, flag) && g->glyph_id != ids[i++])
      return -1;
  return (i < count ? -1 : g - gbeg);
}

static int
match_chain_ids (OTF_GlyphString *gstring, int gidx, int flag,
		 OTF_ChainRule *rule)
{
  int i = rule->BacktrackGlyphCount;

  if (i > 0)
    {
      int j;
      OTF_Glyph *g;

      for (j = gidx - 1, g = gstring->glyphs + j; j >= 0; j--, g--)
	if (! IGNORED_GLYPH (g, flag) && --i == 0)
	  break;
      if (i > 0)
	return -1;
      if (match_ids (gstring, j, flag,
		     rule->BacktrackGlyphCount, rule->Backtrack)
	  < 0)
	return -1;
    }
  gidx++;
  i = match_ids (gstring, gidx, flag,
		 rule->InputGlyphCount - 1, rule->Input);
  if (i < 0)
    return -1;
  gidx += i;
  i = match_ids (gstring, gidx, flag,
		 rule->LookaheadGlyphCount, rule->LookAhead);
  if (i < 0)
    return -1;
  return 0;
}

static int
match_classes (OTF_ClassDef *class_def, OTF_GlyphString *gstring, int gidx,
	       int flag, int count, unsigned *classes)
{
  OTF_Glyph *gbeg = gstring->glyphs + gidx;
  OTF_Glyph *gend = gstring->glyphs + gstring->used;
  OTF_Glyph *g;
  int i;

  for (g = gbeg, i = 0; g < gend && i < count; g++)
    if (! IGNORED_GLYPH (g, flag)
	&& get_class_def (class_def, g->glyph_id) != classes[i++])
      return -1;
  return (i < count ? -1 : g - gbeg);
}

static int
match_chain_classes (OTF_GlyphString *gstring, int gidx, int flag,
		     OTF_ClassDef *BacktrackClassDef,
		     OTF_ClassDef *InputClassDef,
		     OTF_ClassDef *LookaheadClassDef,
		     OTF_ChainClassRule *rule)
{
  int i = rule->BacktrackGlyphCount;

  if (i > 0)
    {
      int j;
      OTF_Glyph *g;

      for (j = gidx - 1, g = gstring->glyphs + j; j >= 0; j--, g--)
	if (! IGNORED_GLYPH (g, flag) && i-- == 0)
	  break;
      if (i > 0)
	return -1;
      if (match_classes (BacktrackClassDef, gstring, j, flag,
			 rule->BacktrackGlyphCount, rule->Backtrack) < 0);
      return -1;
    }
  gidx++;
  i = match_classes (InputClassDef, gstring, gidx, flag,
		     rule->InputGlyphCount - 1, rule->Input);
  if (i < 0)
    return -1;
  gidx += i;
  i = match_classes (LookaheadClassDef, gstring, gidx, flag,
		     rule->LookaheadGlyphCount, rule->LookAhead);
  if (i < 0)
    return -1;
  return 0;
}


static int
match_coverages (OTF_GlyphString *gstring, int gidx, int flag, int count,
		 OTF_Coverage *coverages)
{
  OTF_Glyph *gbeg = gstring->glyphs + gidx;
  OTF_Glyph *gend = gstring->glyphs + gstring->used;
  OTF_Glyph *g;
  int i;

  for (g = gbeg, i = 0; g < gend && i < count; g++)
    if (! IGNORED_GLYPH (g, flag)
	&& get_coverage_index (coverages + i++, g->glyph_id) < 0)
      return -1;
  return (i < count ? -1 : g - gbeg);
}

static int
match_chain_coverages (OTF_GlyphString *gstring, int gidx, int flag,
		       OTF_GSUB_ChainContext3 *context3)
{
  int i = context3->BacktrackGlyphCount;

  if (i > 0)
    {
      int j;
      OTF_Glyph *g;

      for (j = gidx - 1, g= gstring->glyphs +j; j >= 0; j--, g--)
	if (! IGNORED_GLYPH (g, flag) && --i == 0)
	  break;
      if (i > 0)
	return -1;
      if (match_coverages (gstring, j, flag, context3->BacktrackGlyphCount,
			   context3->Backtrack) < 0)
	return -1;
    }
  gidx++;
  if (context3->InputGlyphCount > 1)
    {
      i = match_coverages (gstring, gidx, flag, context3->InputGlyphCount - 1,
			   context3->Input + 1);
      if (i < 0)
	return -1;
      gidx += i;
    }
  if (match_coverages (gstring, gidx, flag, context3->LookaheadGlyphCount,
		       context3->LookAhead) < 0)
    return -1;
  return 0;
}

static int
lookup_gsub (OTF *otf, OTF_LookupList *lookup_list, unsigned lookup_list_index,
	     OTF_GlyphString *gstring, int gidx, int alternate_subst)
{
  char *errfmt = "GSUB Looking up%s";
  int errret = -1;
  OTF_Lookup *lookup = lookup_list->Lookup + lookup_list_index;
  unsigned int flag = (lookup->LookupFlag
		       & (OTF_LookupFlagIgnoreMask | OTF_MarkAttachmentType));
  int orig_gidx = gidx;
  OTF_Glyph *g = gstring->glyphs + gidx;
  int i;

  if (IGNORED_GLYPH (g, flag))
    return (gidx + 1);

  /* Try all subtables until one of them handles the current glyph.  */
  for (i = 0; i < lookup->SubTableCount && gidx == orig_gidx; i++)
    {
      unsigned lookup_type = lookup->LookupType;
      OTF_LookupSubTableGSUB *subtable = lookup->SubTable.gsub + i;
      int coverage_idx;

      if (lookup_type == 7)
	{
	  OTF_GSUB_Extension1 *extension1 = &subtable->u.extension1;

	  lookup_type = extension1->ExtensionLookupType;
	  subtable = extension1->ExtensionSubtable;
	}

      if (alternate_subst
	  ? (lookup_type != 3 && lookup_type != 5 && lookup_type != 6)
	  : (lookup_type == 3))
	continue;

      if (subtable->Coverage.offset)
	{
	  coverage_idx = get_coverage_index (&subtable->Coverage,
					     g->glyph_id);
	  if (coverage_idx < 0)
	    continue;
	}

      switch (lookup_type)
	{
	case 1:
	  if (subtable->Format == 1)
	    g->glyph_id += subtable->u.single1.DeltaGlyphID;
	  else
	    g->glyph_id = subtable->u.single2.Substitute[coverage_idx];
	  gidx++;
	  break;

	case 2:
	  if (subtable->Format == 1)
	    {
	      OTF_GSUB_Multiple1 *multiple1 = &subtable->u.multiple1;
	      OTF_Sequence *seq = multiple1->Sequence + coverage_idx;

	      gstring_subst (otf, gstring, gidx, gidx + 1, flag,
			     seq->Substitute, seq->GlyphCount);
	      gidx += seq->GlyphCount;
	    }
	  else
	    OTF_ERROR (OTF_ERROR_GSUB_DRIVE, " (invalid SubFormat)");
	  break;

	case 3:
	  if (subtable->Format == 1)
	    {
	      OTF_GSUB_Alternate1 *alt1 = &subtable->u.alternate1;
	      OTF_AlternateSet *altset = alt1->AlternateSet + coverage_idx;

	      gstring_subst (otf, gstring, gidx + 1, gidx + 1, flag,
			     altset->Alternate, altset->GlyphCount);
	      gidx += altset->GlyphCount;;
	    }
	  else
	    OTF_ERROR (OTF_ERROR_GSUB_DRIVE, " (invalid SubFormat)");
	  break;

	case 4:
	  if (subtable->Format == 1)
	    {
	      OTF_GSUB_Ligature1 *lig1 = &subtable->u.ligature1;
	      OTF_LigatureSet *ligset = lig1->LigatureSet + coverage_idx;
	      OTF_Ligature *lig;
	      int j;

	      for (j = 0; j < ligset->LigatureCount; j++)
		{
		  int n;

		  lig = ligset->Ligature + j;
		  n = match_ids (gstring, gidx + 1, flag,
				 lig->CompCount - 1, lig->Component);
		  if (n < 0)
		    continue;
		  gstring_subst (otf, gstring, gidx, gidx + 1 + n, flag,
				 &lig->LigGlyph, 1);
		  gidx++;
		  break;
		}
	    }
	  else
	    OTF_ERROR (OTF_ERROR_GSUB_DRIVE, " (invalid SubFormat)");
	  break;

	case 5:
	  if (subtable->Format == 1)
	    {
	      OTF_GSUB_Context1 *context1 = &subtable->u.context1;
	      OTF_RuleSet *set = context1->RuleSet + coverage_idx;
	      OTF_Rule *rule;
	      int orig_used;
	      int j, k;

	      for (j = 0; j < set->RuleCount; j++)
		{
		  rule = set->Rule + j;
		  if (match_ids (gstring, gidx + 1, flag,
				 rule->GlyphCount - 1, rule->Input) < 0)
		    continue;
		  orig_used = gstring->used;
		  for (k = 0; k < rule->LookupCount; k++)
		    lookup_gsub (otf, lookup_list,
				 rule->LookupRecord[k].LookupListIndex,
				 gstring,
				 gidx + rule->LookupRecord[k].SequenceIndex,
				 alternate_subst);
		  gidx += rule->GlyphCount + (gstring->used - orig_used);
		  break;
		}
	    }
	  else if (subtable->Format == 2)
	    {
	      OTF_GSUB_Context2 *context2 = &subtable->u.context2;
	      OTF_ClassSet *set;
	      OTF_ClassRule *rule;
	      unsigned class;
	      int orig_used;
	      int j, k;

	      class = get_class_def (&context2->ClassDef, g->glyph_id);
	      set = context2->ClassSet + class;
	      if (set)
		for (j = 0; j < set->ClassRuleCnt; j++)
		  {
		    rule = set->ClassRule + j;
		    if (match_classes (&context2->ClassDef,
				       gstring, gidx + 1, flag,
				       rule->GlyphCount - 1, rule->Class)
			< 0)
		      continue;
		    orig_used = gstring->used;
		    for (k = 0; k < rule->LookupCount; k++)
		      lookup_gsub (otf, lookup_list,
				   rule->LookupRecord[k].LookupListIndex,
				   gstring,
				   gidx + rule->LookupRecord[k].SequenceIndex,
				   alternate_subst);
		    gidx += rule->GlyphCount + (gstring->used - orig_used);
		    break;
		  }
	    }
	  else 			/* subtable->Format == 3 */
	    {
	      OTF_GSUB_Context3 *context3 = &subtable->u.context3;
	      int orig_used;
	      int j;

	      if (match_coverages (gstring, gidx + 1, flag,
				   context3->GlyphCount - 1,
				   context3->Coverage + 1) < 0)
		continue;
	      orig_used = gstring->used;
	      for (j = 0; j < context3->LookupCount; j++)
		lookup_gsub (otf, lookup_list,
			     context3->LookupRecord[j].LookupListIndex,
			     gstring,
			     gidx + context3->LookupRecord[j].SequenceIndex,
			     alternate_subst);
	      gidx += context3->GlyphCount + (gstring->used - orig_used);
	    }
	  break;

	case 6:
	  if (subtable->Format == 1)
	    {
	      OTF_GSUB_ChainContext1 *context1 = &subtable->u.chain_context1;
	      OTF_ChainRuleSet *set = context1->ChainRuleSet + coverage_idx;
	      int orig_used;
	      int j, k;
	      
	      for (j = 0; j < set->ChainRuleCount; j++)
		{
		  OTF_ChainRule *rule = set->ChainRule + j;

		  if (gidx < rule->BacktrackGlyphCount
		      || (gidx + rule->InputGlyphCount
			  + rule->LookaheadGlyphCount) > gstring->used)
		    continue;
		  if (match_chain_ids (gstring, gidx, flag, rule) < 0)
		    continue;
		  orig_used = gstring->used;
		  for (k = 0; k < rule->LookupCount; k++)
		    lookup_gsub (otf, lookup_list,
				 rule->LookupRecord[k].LookupListIndex,
				 gstring,
				 gidx + rule->LookupRecord[k].SequenceIndex,
				 alternate_subst);
		  gidx += rule->InputGlyphCount + (gstring->used - orig_used);
		  break;
		}
	    }
	  else if (subtable->Format == 2)
	    {
	      OTF_GSUB_ChainContext2 *context2 = &subtable->u.chain_context2;
	      OTF_ChainClassSet *set;
	      unsigned class;
	      int j;
	      int orig_used;

	      class = get_class_def (&context2->InputClassDef, g->glyph_id);
	      set = context2->ChainClassSet + class;
	      for (j = 0; j < set->ChainClassRuleCnt; j++)
		{
		  OTF_ChainClassRule *rule = set->ChainClassRule + j;
		  int k;

		  if (gidx < rule->BacktrackGlyphCount
		      || (gidx + rule->InputGlyphCount
			  + rule->LookaheadGlyphCount) > gstring->used)
		    continue;
		  if (match_chain_classes (gstring, gidx, flag,
					   &context2->BacktrackClassDef,
					   &context2->InputClassDef,
					   &context2->LookaheadClassDef,
					   rule) < 0)
		    continue;
		  orig_used = gstring->used;
		  for (k = 0; k < rule->LookupCount; k++)
		    lookup_gsub (otf, lookup_list,
				 rule->LookupRecord[k].LookupListIndex,
				 gstring,
				 gidx + rule->LookupRecord[k].SequenceIndex,
				 alternate_subst);
		  gidx += rule->InputGlyphCount + (gstring->used - orig_used);
		  break;
		}
	    }
	  else
	    {
	      OTF_GSUB_ChainContext3 *context3 = &subtable->u.chain_context3;
	      int orig_used;
	      int j;

	      if (gidx < context3->BacktrackGlyphCount
		  || (gidx + context3->InputGlyphCount
		      + context3->LookaheadGlyphCount) > gstring->used)
		continue;
	      if (match_chain_coverages (gstring, gidx, flag, context3) < 0)
		continue;
	      orig_used = gstring->used;
	      for (j = 0; j < context3->LookupCount; j++)
		lookup_gsub (otf, lookup_list,
			     context3->LookupRecord[j].LookupListIndex,
			     gstring,
			     gidx + context3->LookupRecord[j].SequenceIndex,
			     alternate_subst);
	      gidx += context3->InputGlyphCount + (gstring->used - orig_used);
	    }
	  break;

	case 8:
	  {
	    OTF_GSUB_ReverseChain1 *reverse = &subtable->u.reverse_chain1;
	    int back_gidx = gidx + 1 + reverse->BacktrackGlyphCount;
	    int ahead_gidx = gidx - reverse->LookaheadGlyphCount;
	    int j;

	    if (back_gidx > gstring->used || ahead_gidx < 0)
	      break;

	    for (j = 0; j < reverse->BacktrackGlyphCount; j++)
	      if (get_coverage_index (reverse->Backtrack + j,
				      gstring->glyphs[gidx + 1 + j].glyph_id)
		  < 0)
		break;
	    if (j < reverse->BacktrackGlyphCount)
	      continue;
	    for (j = 0; j < reverse->LookaheadGlyphCount; j++)
	      if (get_coverage_index (reverse->LookAhead + j,
				      gstring->glyphs[gidx - 1 - j].glyph_id)
		  < 0)
		break;
	    if (j < reverse->LookaheadGlyphCount)
	      continue;
	    g->glyph_id = reverse->Substitute[coverage_idx];
	    gidx--;
	  }

	default:
	  continue;
	}
    }
  if (gidx == orig_gidx)
    gidx++;
  return gidx;
}



/* GPOS */
unsigned
get_anchor (OTF_Anchor *anchor, OTF_ValueRecord *rec)
{
  unsigned value_format = OTF_XPlacement | OTF_YPlacement;

  rec->XPlacement = anchor->XCoordinate;
  rec->YPlacement = anchor->YCoordinate;
  if (anchor->AnchorFormat == 1)
    /* Nothing to do */
    ;
  else if (anchor->AnchorFormat == 2)
    /* Not yet implemented */
    ;
  else if (anchor->AnchorFormat == 3)
    /* Not yet implemented */
    ;
  return value_format;
}


static int
lookup_gpos (OTF_LookupList *lookup_list, unsigned lookup_list_index,
	     OTF_GlyphString *gstring, int gidx)
{
  char *errfmt = "GPOS Looking up%s";
  int errret = -1;
  OTF_Lookup *lookup = lookup_list->Lookup + lookup_list_index;
  unsigned int flag = (lookup->LookupFlag
		       & (OTF_LookupFlagIgnoreMask | OTF_MarkAttachmentType));
  int orig_gidx = gidx;
  OTF_Glyph *g = gstring->glyphs + gidx;
  int i;

  if (IGNORED_GLYPH (g, flag)
      || g->positioning_type)
    return (gidx + 1);

  /* Try all subtables until one of them handles the current glyph.  */
  for (i = 0; i < lookup->SubTableCount && gidx == orig_gidx; i++)
    {
      unsigned lookup_type = lookup->LookupType;
      OTF_LookupSubTableGPOS *subtable = lookup->SubTable.gpos + i;
      int coverage_idx;

      if (lookup_type == 9)
	{
	  OTF_GPOS_Extension1 *extension1 = &subtable->u.extension1;

	  lookup_type = extension1->ExtensionLookupType;
	  subtable = extension1->ExtensionSubtable;
	}

      if (subtable->Coverage.offset)
	{
	  coverage_idx = get_coverage_index (&subtable->Coverage,
					     g->glyph_id);
	  if (coverage_idx < 0)
	    continue;
	}

      switch (lookup_type)
	{
	case 1:
	  g->positioning_type = lookup_type;
	  if (subtable->Format == 1)
	    {
	      OTF_GPOS_Single1 *single1 = &subtable->u.single1;

	      g->f.f1.format = single1->ValueFormat;
	      g->f.f1.value = &single1->Value;
	    }
	  else if (subtable->Format == 2)
	    {
	      OTF_GPOS_Single2 *single2 = &subtable->u.single2;

	      g->f.f1.format = single2->ValueFormat;
	      g->f.f1.value = single2->Value + coverage_idx;
	    }
	  break;

	case 2:
	  {
	    int next_gidx;
	    OTF_Glyph *nextg;

	    for (next_gidx = gidx + 1, nextg = gstring->glyphs + next_gidx;
		 next_gidx < gstring->used && ! IGNORED_GLYPH (nextg, flag);
		 next_gidx++, nextg++);

	    if (next_gidx >= gstring->used
		|| nextg->positioning_type)
	      continue;
	    if (subtable->Format == 1)
	      {
		OTF_GPOS_Pair1 *pair1 = &subtable->u.pair1;
		OTF_PairSet *set = pair1->PairSet + coverage_idx;
		int j;

		for (j = 0; j < set->PairValueCount; j++)
		  if (set->PairValueRecord[j].SecondGlyph == nextg->glyph_id)
		    {
		      if (pair1->ValueFormat1)
			{
			  g->positioning_type = lookup_type;
			  g->f.f2.format = pair1->ValueFormat1;
			  g->f.f2.value = &set->PairValueRecord[j].Value1;
			}
		      gidx = next_gidx;
		      if (pair1->ValueFormat2)
			{
			  nextg->positioning_type = lookup_type;
			  nextg->f.f2.format = pair1->ValueFormat2;
			  nextg->f.f2.value = &set->PairValueRecord[j].Value2;
			  gidx++;
			}
		      break;
		    }
	      }
	    else if (subtable->Format == 2)
	      {
		OTF_GPOS_Pair2 *pair2 = &subtable->u.pair2;
		unsigned class1, class2;

		class1 = get_class_def (&pair2->ClassDef1, g->glyph_id);
		class2 = get_class_def (&pair2->ClassDef2, nextg->glyph_id);
		if (pair2->ValueFormat1)
		  {
		    g->positioning_type = lookup_type;
		    g->f.f2.format = pair2->ValueFormat1;
		    g->f.f2.value
		      = &pair2->Class1Record[class1].Class2Record[class2].Value1;
		  }
		gidx = next_gidx;
		if (pair2->ValueFormat2)
		  {
		    nextg->positioning_type = lookup_type;
		    nextg->f.f2.format = pair2->ValueFormat2;
		    nextg->f.f2.value
		      = &pair2->Class1Record[class1].Class2Record[class2].Value2;
		    gidx++;
		  }
	      }
	  }
	  break;

	case 3:
	  {
	    OTF_GPOS_Cursive1 *cursive1 = &subtable->u.cursive1;
	  
	    g->positioning_type = lookup_type;
	    g->f.f3.entry_anchor
	      = &cursive1->EntryExitRecord[coverage_idx].EntryAnchor;
	    g->f.f3.exit_anchor
	      = &cursive1->EntryExitRecord[coverage_idx].ExitAnchor;
	  }
	  break;

	case 4:
	  if (gidx < 1)
	    continue;
	  if (subtable->Format == 1)
	    {
	      OTF_GPOS_MarkBase1 *mark_base1 = &subtable->u.mark_base1;
	      OTF_MarkRecord *mark_record;
	      OTF_AnchorRecord *base_record;
	      OTF_Glyph *baseg;
	      int coverage_idx_base;

	      for (baseg = g - 1;
		   baseg >= gstring->glyphs && IGNORED_GLYPH (baseg, flag);
		   baseg--);
	      if (baseg < gstring->glyphs)
		continue;
	      coverage_idx_base
		= get_coverage_index (&mark_base1->BaseCoverage,
				      baseg->glyph_id);
	      if (coverage_idx_base < 0)
		continue;
	      mark_record = mark_base1->MarkArray.MarkRecord + coverage_idx;
	      base_record
		= mark_base1->BaseArray.AnchorRecord + coverage_idx_base;
	      g->f.f4.mark_anchor = &mark_record->MarkAnchor;
	      g->f.f4.base_anchor
		= &base_record->Anchor[mark_record->Class];
	      g->positioning_type = lookup_type;
	    }
	  break;

	case 5:
	  if (gidx < 1)
	    continue;
	  if (subtable->Format == 1)
	    {
	      OTF_GPOS_MarkLig1 *mark_lig1 = &subtable->u.mark_lig1;
	      OTF_Glyph *ligg;
	      int coverage_idx_lig;
	      OTF_MarkRecord *mark_record;
	      OTF_LigatureAttach *attach;
	      int *num_class = alloca (sizeof (int) * mark_lig1->ClassCount);
	      int j;
				       
	      for (j = 0; j < mark_lig1->ClassCount; j++)
		num_class[j] = 0;

	      for (ligg = g - 1;
		   (ligg >= gstring->glyphs
		    && (IGNORED_GLYPH (ligg, flag)
			|| ligg->GlyphClass > OTF_GlyphClassLigature));
		   ligg--)
		if (ligg->positioning_type == 5
		    && ligg->MarkAttachClass < mark_lig1->ClassCount)
		  num_class[ligg->MarkAttachClass]++;
	      if (ligg < gstring->glyphs)
		continue;
	      coverage_idx_lig
		= get_coverage_index (&mark_lig1->LigatureCoverage,
				      ligg->glyph_id);
	      if (coverage_idx_lig < 0)
		continue;
	      mark_record = mark_lig1->MarkArray.MarkRecord + coverage_idx;
	      g->MarkAttachClass = mark_record->Class;
	      attach = (mark_lig1->LigatureArray.LigatureAttach
			+ coverage_idx_lig);
	      for (j = 0; j < attach->ComponentCount; j++)
		{
		  OTF_Anchor *lig_anchor
		    = attach->ComponentRecord[j].LigatureAnchor;

		  if (lig_anchor[mark_record->Class].AnchorFormat
		      && num_class[mark_record->Class]-- == 0)
		    {
		      g->positioning_type = lookup_type;
		      g->f.f5.mark_anchor = &mark_record->MarkAnchor;
		      g->f.f5.ligature_anchor = lig_anchor + mark_record->Class;
		      break;
		    }
		}
	    }
	  break;

	case 6:
	  if (gidx < 1)
	    continue;
	  if (subtable->Format == 1)
	    {
	      OTF_GPOS_MarkMark1 *mark_mark1 = &subtable->u.mark_mark1;
	      OTF_MarkRecord *mark1_record;
	      OTF_AnchorRecord *mark2_record;
	      OTF_Glyph *prevg;
	      int coverage_idx_base;

	      for (prevg = g - 1;
		   prevg >= gstring->glyphs && IGNORED_GLYPH (prevg, flag);
		   prevg--);
	      if (prevg < gstring->glyphs)
		continue;
	      coverage_idx_base
		= get_coverage_index (&mark_mark1->Mark2Coverage,
				      prevg->glyph_id);
	      if (coverage_idx_base < 0)
		continue;
	      mark1_record = mark_mark1->Mark1Array.MarkRecord + coverage_idx;
	      mark2_record
		= mark_mark1->Mark2Array.AnchorRecord + coverage_idx_base;
	      g->f.f6.mark1_anchor = &mark1_record->MarkAnchor;
	      g->f.f6.mark2_anchor
		= &mark2_record->Anchor[mark1_record->Class];
	      g->positioning_type = lookup_type;
	      break;
	    }
	  break;

	case 7:
	  if (subtable->Format == 1)
	    {
	      OTF_GPOS_Context1 *context1 = &subtable->u.context1;
	      OTF_RuleSet *set = context1->RuleSet + coverage_idx;
	      OTF_Rule *rule;
	      int orig_used;
	      int j, k;

	      for (j = 0; j < set->RuleCount; j++)
		{
		  rule = set->Rule + j;
		  if (match_ids (gstring, gidx + 1, flag,
				 rule->GlyphCount - 1, rule->Input) < 0)
		    continue;
		  orig_used = gstring->used;
		  for (k = 0; k < rule->LookupCount; k++)
		    lookup_gpos (lookup_list,
				 rule->LookupRecord[k].LookupListIndex,
				 gstring,
				 gidx + rule->LookupRecord[k].SequenceIndex);
		  gidx += rule->GlyphCount + (gstring->used - orig_used);
		  break;
		}
	    }
	  else if (subtable->Format == 2)
	    {
	      OTF_GPOS_Context2 *context2 = &subtable->u.context2;
	      OTF_ClassSet *set;
	      OTF_ClassRule *rule;
	      unsigned class;
	      int orig_used;
	      int j, k;

	      class = get_class_def (&context2->ClassDef, g->glyph_id);
	      set = context2->ClassSet + class;
	      if (set)
		for (j = 0; j < set->ClassRuleCnt; j++)
		  {
		    rule = set->ClassRule + j;
		    if (match_classes (&context2->ClassDef,
				       gstring, gidx + 1, flag,
				       rule->GlyphCount - 1, rule->Class)
			< 0)
		      continue;
		    orig_used = gstring->used;
		    for (k = 0; k < rule->LookupCount; k++)
		      lookup_gpos (lookup_list,
				   rule->LookupRecord[k].LookupListIndex,
				   gstring,
				   gidx + rule->LookupRecord[k].SequenceIndex);
		    gidx += rule->GlyphCount + (gstring->used - orig_used);
		    break;
		  }
	    }
	  else 			/* subtable->Format == 3 */
	    {
	      OTF_GPOS_Context3 *context3 = &subtable->u.context3;
	      int orig_used;
	      int j;

	      if (match_coverages (gstring, gidx + 1, flag,
				   context3->GlyphCount - 1,
				   context3->Coverage + 1) < 0)
		continue;
	      orig_used = gstring->used;
	      for (j = 0; j < context3->LookupCount; j++)
		lookup_gpos (lookup_list,
			     context3->LookupRecord[j].LookupListIndex,
			     gstring,
			     gidx + context3->LookupRecord[j].SequenceIndex);
	      gidx += context3->GlyphCount + (gstring->used - orig_used);
	    }
	  break;

	case 8:
	  if (subtable->Format == 1)
	    {
	      OTF_GPOS_ChainContext1 *context1 = &subtable->u.chain_context1;
	      OTF_ChainRuleSet *set = context1->ChainRuleSet + coverage_idx;
	      int orig_used;
	      int j, k;
	      
	      for (j = 0; j < set->ChainRuleCount; j++)
		{
		  OTF_ChainRule *rule = set->ChainRule + j;

		  if (gidx < rule->BacktrackGlyphCount
		      || (gidx + rule->InputGlyphCount
			  + rule->LookaheadGlyphCount) > gstring->used)
		    continue;
		  if (match_chain_ids (gstring, gidx, flag, rule) < 0)
		    continue;
		  orig_used = gstring->used;
		  for (k = 0; k < rule->LookupCount; k++)
		    lookup_gpos (lookup_list,
				 rule->LookupRecord[k].LookupListIndex,
				 gstring,
				 gidx + rule->LookupRecord[k].SequenceIndex);
		  gidx += rule->InputGlyphCount + (gstring->used - orig_used);
		  break;
		}
	    }
	  else if (subtable->Format == 2)
	    {
	      OTF_GPOS_ChainContext2 *context2 = &subtable->u.chain_context2;
	      OTF_ChainClassSet *set;
	      unsigned class;
	      int j;
	      int orig_used;

	      class = get_class_def (&context2->InputClassDef, g->glyph_id);
	      set = context2->ChainClassSet + class;
	      for (j = 0; j < set->ChainClassRuleCnt; j++)
		{
		  OTF_ChainClassRule *rule = set->ChainClassRule + j;
		  int k;

		  if (gidx < rule->BacktrackGlyphCount
		      || (gidx + rule->InputGlyphCount
			  + rule->LookaheadGlyphCount) > gstring->used)
		    continue;
		  if (match_chain_classes (gstring, gidx, flag,
					   &context2->BacktrackClassDef,
					   &context2->InputClassDef,
					   &context2->LookaheadClassDef,
					   rule) < 0)
		    continue;
		  orig_used = gstring->used;
		  for (k = 0; k < rule->LookupCount; k++)
		    lookup_gpos (lookup_list,
				 rule->LookupRecord[k].LookupListIndex,
				 gstring,
				 gidx + rule->LookupRecord[k].SequenceIndex);
		  gidx += rule->InputGlyphCount + (gstring->used - orig_used);
		  break;
		}
	    }
	  else if (subtable->Format == 3)
	    {
	      OTF_GPOS_ChainContext3 *context3 = &subtable->u.chain_context3;
	      int orig_used;
	      int j;

	      if (gidx < context3->BacktrackGlyphCount
		  || (gidx + context3->InputGlyphCount
		      + context3->LookaheadGlyphCount) > gstring->used)
		continue;
	      if (match_chain_coverages (gstring, gidx, flag, context3) < 0)
		continue;
	      orig_used = gstring->used;
	      for (j = 0; j < context3->LookupCount; j++)
		lookup_gpos (lookup_list,
			     context3->LookupRecord[j].LookupListIndex,
			     gstring,
			     gidx + context3->LookupRecord[j].SequenceIndex);
	      gidx += context3->InputGlyphCount + (gstring->used - orig_used);
	    }
	  else
	    OTF_ERROR (OTF_ERROR_GPOS_DRIVE, " (invalid subformat)");
	  break;

	default:
	  continue;
	}
    }
  if (gidx == orig_gidx)
    gidx++;
  return gidx;
}

static int
lookup_encoding_0 (OTF_EncodingSubtable0 *sub0, OTF_GlyphString *gstring)
{
  int i, c;

  for (i = 0; i < gstring->used; i++)
    {
      c = gstring->glyphs[i].c;
      if (c < 0 || c >= 256)
	gstring->glyphs[i].glyph_id = 0;
      else
	gstring->glyphs[i].glyph_id = sub0->glyphIdArray[c];
    }
  return 0;
}

static int
lookup_encoding_2 (OTF_EncodingSubtable2 *sub2, OTF_GlyphString *gstring)
{
  return 0;
}

static int
lookup_encoding_4 (OTF_EncodingSubtable4 *sub4, OTF_GlyphString *gstring)
{
  int i, j, c;
  int segCount = sub4->segCountX2 / 2;

  for (i = 0; i < gstring->used; i++)
    {
      c = gstring->glyphs[i].c;
      if (c < 0)
	gstring->glyphs[i].glyph_id = 0;
      for (j = 0; j < segCount; j++)
	{
	  OTF_cmapSegument *seg = sub4->segments + i;

	  if (c >= seg->startCount && c <= seg->endCount)
	    {
	      if (seg->idRangeOffset == 0xFFFF)
		gstring->glyphs[i].glyph_id = c + seg->idDelta;
	      else
		gstring->glyphs[i].glyph_id
		  = sub4->glyphIdArray[seg->idRangeOffset
				       + (c - seg->startCount)];
	      break;
	    }
	}
    }

  return 0;
}

static int
lookup_encoding_6 (OTF_EncodingSubtable6 *sub6, OTF_GlyphString *gstring)
{
  return 0;
}

static int
lookup_encoding_8 (OTF_EncodingSubtable8 *sub8, OTF_GlyphString *gstring)
{
  return 0;
}

static int
lookup_encoding_10 (OTF_EncodingSubtable10 *sub10, OTF_GlyphString *gstring)
{
  return 0;
}

static int
lookup_encoding_12 (OTF_EncodingSubtable12 *sub12, OTF_GlyphString *gstring)
{
  return 0;
}



/* API */

int
OTF_drive_cmap (OTF *otf, OTF_GlyphString *gstring)
{
  OTF_cmap *cmap;
  int i;

  if (! otf->cmap
      && OTF_get_table (otf, "cmap") < 0)
    return -1;

  cmap = otf->cmap;
  for (i = 0; i < gstring->used; i++)
    if (! gstring->glyphs[i].glyph_id)
      {
	int c = gstring->glyphs[i].c;
	if (c < 32 || ! cmap->unicode_table)
	  gstring->glyphs[i].glyph_id = 0;
	else
	  gstring->glyphs[i].glyph_id = cmap->unicode_table[c];
      }
  return 0;
}


int
OTF_drive_cmap2 (OTF *otf, OTF_GlyphString *gstring,
		 int platform_id, int encoding_id)
{
  OTF_cmap *cmap;
  int i;
  char *errfmt = "CMAP Looking up%s";
  int errret = -1;
  OTF_EncodingRecord *enc;

  if (! otf->cmap
      && OTF_get_table (otf, "cmap") < 0)
    return -1;

  cmap = otf->cmap;
  for (i = 0; i < cmap->numTables; i++)
    if (cmap->EncodingRecord[i].platformID == platform_id
	&& cmap->EncodingRecord[i].encodingID == encoding_id)
      break;
  if (i == cmap->numTables)
    OTF_ERROR (OTF_ERROR_CMAP_DRIVE, " (unknown platformID/encodingID)");
  enc = cmap->EncodingRecord + i;
  switch (enc->subtable.format)
    {
    case 0: return lookup_encoding_0 (enc->subtable.f.f0, gstring);
    case 2: return lookup_encoding_2 (enc->subtable.f.f2, gstring);
    case 4: return lookup_encoding_4 (enc->subtable.f.f4, gstring);
    case 6: return lookup_encoding_6 (enc->subtable.f.f6, gstring);
    case 8: return lookup_encoding_8 (enc->subtable.f.f8, gstring);
    case 10: return lookup_encoding_10 (enc->subtable.f.f10, gstring);
    case 12: return lookup_encoding_12 (enc->subtable.f.f12, gstring);
    }
  OTF_ERROR (OTF_ERROR_CMAP_DRIVE, " (invalid format)");
}


int
OTF_get_unicode (OTF *otf, OTF_GlyphID code)
{
  if (! otf->cmap
      && OTF_get_table (otf, "cmap") < 0)
    return 0;
  if (code == 0
      || code > otf->cmap->max_glyph_id
      || ! otf->cmap->decode_table)
    return 0;
  return otf->cmap->decode_table[code];
}

int
OTF_drive_gdef (OTF *otf, OTF_GlyphString *gstring)
{
  OTF_GDEF *gdef;
  int i;

  if (! otf->gdef
      && OTF_get_table (otf, "GDEF") < 0)
    return -1;
  gdef = otf->gdef;

  if (gdef->glyph_class_def.offset)
    for (i = 0; i < gstring->used; i++)
      gstring->glyphs[i].GlyphClass
	= get_class_def (&gdef->glyph_class_def,
			 gstring->glyphs[i].glyph_id);

  if (gdef->mark_attach_class_def.offset)
    for (i = 0; i < gstring->used; i++)
      gstring->glyphs[i].MarkAttachClass
	= get_class_def (&gdef->mark_attach_class_def,
			 gstring->glyphs[i].glyph_id);

  return 0;
}

static int
OTF_drive_gsub_internal (OTF *otf, OTF_GlyphString *gstring,
			 char *script, char *language, char *features,
			 int alternate_subst)
{
  char *errfmt = "GSUB driving%s";
  int errret = -1;
  OTF_GSUB *gsub;
  OTF_LangSys *LangSys;
  int *lookup_indices;
  int i, n;

  for (i = 0; i < gstring->used; i++)
    {
      gstring->glyphs[i].positioning_type = 0;
      gstring->glyphs[i].f.index.from = gstring->glyphs[i].f.index.to = i;
    }

  if (OTF_get_table (otf, "GSUB") < 0)
    return errret;
  gsub = otf->gsub;
  if (gsub->FeatureList.FeatureCount == 0
      || gsub->LookupList.LookupCount == 0)
    return 0;

  LangSys = get_langsys (&gsub->ScriptList, script, language);
  if (! LangSys)
    return errret;

  /* One lookup may be used by multiple features.  */
  lookup_indices = alloca (sizeof (int)
			   * gsub->LookupList.LookupCount
			   * (gsub->FeatureList.FeatureCount + 1));
  if (! lookup_indices)
    OTF_ERROR (OTF_ERROR_MEMORY, " feature list");
  n = setup_lookup_indices (&gsub->LookupList, &gsub->FeatureList,
			    features, lookup_indices);
  if (n < 0)
    return errret;

  for (i = 0; i < n; i++)
    {
      int index = lookup_indices[i];
      int gidx;

      if (gsub->LookupList.Lookup[index].LookupType != 8)
	{
	  gidx = 0;
	  while (gidx < gstring->used)
	    {
	      gidx = lookup_gsub (otf, &gsub->LookupList, index, gstring, gidx,
				  alternate_subst);
	      if (gidx < 0)
		return errret;
	    }
	}
      else
	{
	  gidx = gstring->used - 1;
	  while (gidx >= 0)
	    {
	      gidx = lookup_gsub (otf, &gsub->LookupList, index, gstring, gidx,
				  alternate_subst);
	      if (gidx < 0)
		return errret;
	    }
	}
    }

  return 0;
}

int
OTF_drive_gsub (OTF *otf, OTF_GlyphString *gstring,
		char *script, char *language, char *features)
{
  return OTF_drive_gsub_internal (otf, gstring, script, language, features, 0);
}

int
OTF_drive_gpos (OTF *otf, OTF_GlyphString *gstring,
		char *script, char *language, char *features)
{
  char *errfmt = "GPOS driving%s";
  int errret = -1;
  OTF_GPOS *gpos;
  OTF_LangSys *LangSys;
  int *lookup_indices;
  int i, n;

  for (i = 0; i < gstring->used; i++)
    gstring->glyphs[i].positioning_type = 0;

  if (OTF_get_table (otf, "GPOS") < 0)
    return errret;
  gpos = otf->gpos;
  if (gpos->FeatureList.FeatureCount == 0
      || gpos->LookupList.LookupCount == 0)
    return 0;

  LangSys = get_langsys (&gpos->ScriptList, script, language);
  if (! LangSys)
    return errret;

  /* One lookup may be used by multiple features.  */
  lookup_indices = alloca (sizeof (int)
			   * gpos->LookupList.LookupCount
			   * (gpos->FeatureList.FeatureCount + 1));
  if (! lookup_indices)
    OTF_ERROR (OTF_ERROR_MEMORY, " feature list");
  n = setup_lookup_indices (&gpos->LookupList, &gpos->FeatureList,
			    features, lookup_indices);
  if (n < 0)
    return errret;

  for (i = 0; i < n; i++)
    {
      int index = lookup_indices[i];
      int gidx = 0;

      while (gidx < gstring->used)
	{
	  gidx = lookup_gpos (&gpos->LookupList, index, gstring, gidx);
	  if (gidx < 0)
	    return errret;
	}
    }

  return 0;
}

int
OTF_drive_tables (OTF *otf, OTF_GlyphString *gstring,
		  char *script, char *language,
		  char *gsub_features, char *gpos_features)
{
  if (OTF_drive_cmap (otf, gstring) < 0)
    return -1;
  if (OTF_drive_gdef (otf, gstring) < 0)
    return -1;
  if (gsub_features
      && OTF_drive_gsub (otf, gstring, script, language, gsub_features) < 0)
    return -1;
  if (gpos_features
      && OTF_drive_gpos (otf, gstring, script, language, gpos_features) < 0)
    return -1;
  return 0;
}

int
OTF_drive_gsub_alternate (OTF *otf, OTF_GlyphString *gstring,
			  char *script, char *language, char *features)
{
  return OTF_drive_gsub_internal (otf, gstring, script, language, features, 1);
}
