/* 
 * pam_rsa PAM-module for local authentication with RSA keypairs
 * copyright (c) 2006 Vesa-Matti Kari <hyperllama@laamanaama.helsinki.fi>
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 * 
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include <ctype.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "util.h"
#include "pam_rsa.h"


/* For mapping strings (yes|true|on|no|false|off) to corresponding boolean values */
struct s2b {
	char *s;
	int b;
};


/* For mapping syslog(3) levels to human readable strings */
struct lev2str {
	const int level;
	const char *str;
};


/* For mapping characters to stat(2) modes and human readable strings */
struct c2mode {
	const char c;
	const mode_t mode;
	const char *txt;
};

void pamrsa_log(int level, const char *fmt, ...)
{
	static const struct lev2str lev2str[] = {
		{ LOG_ALERT, "alert" },
		{ LOG_CRIT, "critical" },
		{ LOG_EMERG, "emergency" },
		{ LOG_ERR, "error" },
		{ LOG_WARNING, "warning" },
		{ LOG_NOTICE, "notice" },
		{ LOG_INFO, "info" },
		{ LOG_DEBUG, "debug" }
	};
	const char *str;
	char buf[LOG_MSGLEN];
	va_list args;
	int i;

	va_start(args, fmt);

	(void)vsnprintf(buf, sizeof(buf), fmt, args);

	va_end(args);

	str = NULL;
	for (i = 0; i < (int)(sizeof(lev2str)/sizeof(struct lev2str)); i++) {
		if (level == lev2str[i].level) {
				str = lev2str[i].str;	
				break;
		}
	}

	if (str == NULL) {
		str = "[unknown_sysloglevel]";
	}
#ifdef DEBUG
	syslog(PAM_RSA_FACILITY|level, "%s-DEBUG-BUILD: %s: %s", pam_rsa_name, str, buf);
#else
	syslog(PAM_RSA_FACILITY|level, "%s: %s: %s", pam_rsa_name, str, buf);
#endif
}



/* Note: The return value of set_bool() is not the product 
 * of the string-to-boolean conversion! It indicates whether
 * the conversion succeeded or not. 
 */

int set_bool(int *b, const char *s)
{
	int i;
	static struct s2b s2b[] = { 
		 { "true", 1 },
		 { "yes", 1 },
		 { "on", 1 },
		 { "false", 0 },
		 { "no", 0 },
		 { "off", 0 },
	};

	for (i = 0; i < (int)(sizeof(s2b)/sizeof(struct s2b)); i++) {
		if (!strcasecmp(s, s2b[i].s)) {
			*b = s2b[i].b;	
			return 1;
		}
	}

	return 0;
}


static int is_safechar(char c)
{
	static const char safechars[] = "_-,.;:!@#$&/()[]=?+'*|<>";
	int i;

	if (isalnum(c) || isspace(c)) {
		return 1;
	}

	for (i = 0; i < (int)sizeof(safechars); i++) {
		if (c == safechars[i]) {
			return 1;
		}
	}
	return 0;
}


int is_safestr(const char *s)
{
	while (*s != '\0') {
		if (!is_safechar(*s)) {
			return 0;
		}
		++s;
	}
	return 1;
}


int is_safefile(const char *file, const char *fty, const char *fpe, unsigned int *errs)
{
	static const struct c2mode c2m[] =
	{
		{ 'b', S_IFBLK, "block device" },
		{ 'c', S_IFCHR, "character device" },
		{ 'd', S_IFDIR, "directory" },
		{ 'f', S_IFIFO, "fifo" },
		{ 'l', S_IFLNK, "symbolic link" },
		{ 'r', S_IFREG, "regular file" },
		{ 's', S_IFSOCK, "socket" },
	};
	mode_t mode;
	const char *filetype;
	struct stat st;
	int ftyok = 0;

	int gwdir = 0;		/* G */
	int gwfile = 0;	/* g */
	int wwdir = 0;		/* O */
	int wwfile = 0;	/* o */
	int grdir = 0;		/* R */
	int grfile = 0;	/* r */
	int wrdir = 0;		/* F */
	int wrfile = 0;	/* f */

	int i;

	if (fty == NULL || *fty == '\0') {
		++*errs;
		return -1;	
	} 

	memset(&st, 0, sizeof(st));

	if (lstat(file, &st) != 0) {
		return -1;
	}

	mode = st.st_mode & S_IFMT;

	while (*fty != '\0') {

		int found = 0;
		for (i = 0; i < (int)(sizeof(c2m)/sizeof(struct c2mode)); i++) {
			if (*fty == c2m[i].c) {
				found = 1;
				break;
			}
		}

		if (!found) {
			pamrsa_log(LOG_ALERT, "invalid filetype specifier '%c'\n", *fty);
			return -1;	
		}

		if (mode == (c2m[i].mode & S_IFMT)) {
			ftyok = 1;
			break;
		}
		++fty;
	}

	filetype = "[unknown filetype]";
	for (i = 0; i < (int)(sizeof(c2m)/sizeof(struct c2mode)); i++) {
		if (mode == (c2m[i].mode & S_IFMT)) {
			filetype = c2m[i].txt;	
			break;
		}
	}

	if (!ftyok) {
		++*errs;
		pamrsa_log(LOG_ERR, "%s: filetype '%s' is not safe", file, filetype);
	} 

	if (fpe != NULL) {
		while (*fpe != '\0') {
			if (*fpe == 'G') {	
				gwdir = 1;
			} else if (*fpe == 'g') {
				gwfile = 1;
			} else if (*fpe == 'O') {
				wwdir = 1;
			} else if (*fpe == 'o') {
				wwfile = 1;
			} else if (*fpe == 'R') {
				grdir = 1;
			} else if (*fpe == 'r') {
				grfile = 1;
			} else if (*fpe == 'F') {
				wrdir = 1;
			} else if (*fpe == 'f') {
				wrfile = 1;
			} else {
				pamrsa_log(LOG_ALERT, "invalid permission specifier '%c'", *fpe);
				return -1;	
			}
			++fpe;
		}
	}
	
	if (mode == S_IFDIR) {
		if (gwdir == 0 && (st.st_mode & S_IWGRP)) {
			pamrsa_log(LOG_ERR, "group writable directory %s unsafe", file);
			++*errs;
		}
		if (wwdir == 0 && (st.st_mode & S_IWOTH)) {
			pamrsa_log(LOG_ERR, "world writable directory %s unsafe", file);
			++*errs;
		}
		if (grdir == 0 && (st.st_mode & S_IRGRP)) {
			pamrsa_log(LOG_ERR, "group readable directory %s unsafe", file);
			++*errs;
		}
		if (wrdir == 0 && (st.st_mode & S_IROTH)) {
			pamrsa_log(LOG_ERR, "world readable directory %s unsafe", file);
			++*errs;
		}
	} else if (mode == S_IFREG) {
		if (gwfile == 0 && (st.st_mode & S_IWGRP)) {
			pamrsa_log(LOG_ERR, "group writable file %s unsafe", file);
			++*errs;
		}
		if (wwfile == 0 && (st.st_mode & S_IWOTH)) {
			pamrsa_log(LOG_ERR, "world writable file %s unsafe", file);
			++*errs;
		}
		if (grfile == 0 && (st.st_mode & S_IRGRP)) {
			pamrsa_log(LOG_ERR, "group readable file %s unsafe", file);
			++*errs;
		}
		if (wrfile == 0 && (st.st_mode & S_IROTH)) {
			pamrsa_log(LOG_ERR, "world readable file %s unsafe", file);
			++*errs;
		}
	}

	if (*errs > 0) {
		return -2;
	}

	return 0;
}


int is_safepath(const char *path, const char *fty, const char *fpe)
{
	char **res = NULL;
	int nelem;
	int i;
	int r;
	unsigned int errs;

	if (parse_path(path, &res, &nelem) != PARSEPATH_SUCCESS) {
		r = -1;
		goto endfree;
	}

	for (i = 0; i < nelem; i++) {
		errs = 0;
		if ((r = is_safefile(res[i], fty, fpe, &errs)) == 0) {
			continue;
		} else if (r == -1) {
			pamrsa_log(LOG_ERR, "could not stat %s", res[i]);
			break;	
		} else if (r == -2){
			pamrsa_log(LOG_ALERT, "path %s is not safe", res[i]);
			break;	
		} else {
			pamrsa_log(LOG_ALERT, "impossible error: is_safefile() code %d unknown", r);
			r = -1;
			break;
		}
	}

endfree:
	for (i = 0; i < nelem; i++) {
		free(res[i]);	
	}
	free(res);

	return (r == 0)?1:0;
}


int parse_path(const char *p, char ***parsed, int *nelem)
{
	char **ret = NULL;
	char *buf;
	char *s;
	int len;
	int i;
	int j;


	if (p == NULL || p[0] == '\0') {
		return PARSEPATH_EMPTY;
	}

	if (p[0] != '/') {
		return PARSEPATH_NOTABSOLUTE;
	}

	if ((len = strlen(p)) >= MAXIMUM_PATH) {
		return PARSEPATH_TOOLONG;
	}

	/* No consecutive slashes allowed */

	for (i = 1; p[i] != '\0'; i++) {
		if (p[i] == '/' && p[i-1] == '/') {
			return PARSEPATH_TWOSLASHES;
		}
	}

	if ((buf = malloc(len + 2)) == NULL) {
		return PARSEPATH_OUTOFMEMORY;
	}
	strcpy(buf, p);

	s = strchr(buf, '\0');
	--s;
	if (*s != '/')  {
		++s;
		*s = '/';
		++s;
		*s = '\0';
	}

	/* buf now contains a canonical pathname that:
     * 1) has no consecutive slashes 
     * 2) starts with a slash
     * 2) ends with a slash
     * 
     * NOTE: canonical form for "/" is "/" 
     */

	s = buf;
	*nelem = 0;
	while (*s != '\0') {
		if (*s == '/') {
			++*nelem;
		}
		++s;
	}

#ifdef DEBUG
	pamrsa_log(LOG_DEBUG, "path='%s' len=%d nelem=%d canonical='%s'\n",
		p, len, *nelem, buf);
#endif

	if ((ret = malloc(sizeof(char *) * *nelem)) == NULL) {
		free(buf);
		return PARSEPATH_OUTOFMEMORY;
	}
	for (i = 0; i < *nelem; i++) {
		ret[i] = NULL; /* so that caller can safely free() these */
	}

	if ((ret[0] = strdup("/")) == NULL) {
		free(buf);
		return PARSEPATH_OUTOFMEMORY;
	}

	for (i = 1; i < *nelem; i++) {
		int slashes = 0;
		for (j = 0; buf[j] != '\0'; j++) {	
			if (buf[j] == '/') {
				 if (++slashes > i) {
					buf[j] = '\0';
					if ((ret[i] = strdup(buf)) == NULL) {
						free(buf);
						return PARSEPATH_OUTOFMEMORY;
					}
					buf[j] = '/';
					break;
				} 
			}
		}
	}
	free(buf);
	*parsed = ret;

	return PARSEPATH_SUCCESS;
}


const char *parsepath_error(int e)
{
	switch(e) {
	case PARSEPATH_EMPTY:
		return "path was NULL or empty";
	case PARSEPATH_NOTABSOLUTE:
		return "path not absolute";
	case PARSEPATH_OUTOFMEMORY:
		return "memory allocation failure";
	case PARSEPATH_TOOLONG:
		return "path was too long";
	case PARSEPATH_TWOSLASHES:
		return "path contained two consecutive slashes";
	case PARSEPATH_SUCCESS:
		return "path parsed successfully";
	default:
		return "[unrecognized parsepath error]";
	}
}
