/* GenericSelectionGroup.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.HashMap;
import java.util.Map;
import javax.swing.AbstractButton;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * Acts like a button group with associated generic selection model for 
 * type {@code E}. Similar
 * to {@code SelectionGroup} but backed by a {@link GenericSelectionModel} 
 * instead of a {@code SingleSelectionModel}. 
 * 
 * <p>Of all toggle buttons
 * added to a group of this kind, at most one can be selected at the same time.
 * Every time a new selection is made, this is reported to the selection model.
 * Each button added to the group is given an associated object
 * of type {@code E}.
 * When a new object is selected in the model, the corresponding button will
 * be selected and the old button will be deselected.
 * <p>Provides convenience methods to add listeners to the associated selection model
 * and to determine what is the currently selected button.
 * <p><b>Note:</b>
 * In typical applications of this class {@code E} will denote an enumeration type. 
 * It is always required that {@code E} has implementations of
 * {@code hashCode} and {@code equals} that are compatible.
 * @see SelectionGroup
 */
public class GenericSelectionGroup<E>  {
    
    //
    private GenericSelectionModel<E> model;
    
    /**
     * The selection model used by this group.
     */
    public GenericSelectionModel<E> getModel() {
        return model;
    }
    
    //
    private boolean clearable;
    
    /**
     * Create a selection group with a newly created selection model and no
     * buttons. Short for {@code GenericSelectiongroup(true)}.
     */
    public GenericSelectionGroup() {
        this(new DefaultGenericSelectionModel<E>(), true);
    }
    
    /**
     * Create a selection group with a newly created selection model and no
     * buttons.
     * @param clearable indicates whether buttons can be cleared by clicking
     * on them. If false the group behaves like a button group.
     */
    public GenericSelectionGroup(boolean clearable) {
        this(new DefaultGenericSelectionModel<E>(), clearable);
    }
    
    /**
     * Create a selection group with the given model.
     * @param clearable indicates whether buttons can be cleared by clicking
     * on them. If false the group behaves like a button group.
     */
    public GenericSelectionGroup(GenericSelectionModel<E> model, boolean clearable) {
        super();
        this.model = model;
        this.clearable = clearable;
        buttons = new HashMap<E,AbstractButton> ();
        lastButton = null; // NOPMD
        model.addChangeListener(new ChangeListener() {
            public void stateChanged(ChangeEvent e) {
                selectButton(getSelectedButton());
            }
        });
    }
    
    /**
     * Register the listener with the model.
     */
    public void addChangeListener(ChangeListener listener) {
        if (model != null)
            model.addChangeListener(listener);
    }
    
    /**
     * Unregister the listener from the model.
     */
    public void removeChangeListener(ChangeListener listener) {
        if (model != null)
            model.removeChangeListener(listener);
    }
    
    // Button which was selected last
    private AbstractButton lastButton;
    
    private Map<E,AbstractButton> buttons;
    
    /**
     * Select the given button.
     */
    private void selectButton(AbstractButton button) {
        if (lastButton != null)
            lastButton.setSelected(false);
        lastButton = button;
        if (button != null)
            button.setSelected(true);
    }
    
    /**
     * Add the given button to the group.
     * @param button Button to be added
     * @param element Element which correspond to this button
     */
    public void add(AbstractButton button, E element) {
        if (element == null)
             throw new IllegalArgumentException ("Enumeration constant cannot be null");
        buttons.put(element, button);
        if (model != null && model.getSelection() == element)
            selectButton(button);
        button.addActionListener(new ToggleAction(element));
    }
    
    /**
     * Return the button which is currently selected.
     * @return the currently selected button or null if none is selected
     */
    public AbstractButton getSelectedButton() {
        E element = model.getSelection();
        if (element == null)
            return null;
        else
            return buttons.get(element);
    }
    
    /**
     * Return the current selection.
     * @return the current selection index or -1 if none.
     */
    public E getSelection() {
        return model.getSelection();
    }
    
    /**
     * Return the action command of the currently selected button,
     * or null if none is selected.
     */
    public String getActionCommand() {
        E element = model.getSelection();
        if (element == null)
            return null;
        else
            return buttons.get(element).getActionCommand();
    }
    
    /**
     * Set the current selection.
     */
    public void setSelection (E element){
        model.setSelection (element);
    }
    
    
    // listener for buttons
    private class ToggleAction implements ActionListener {
        
        private final E element;
        
        public ToggleAction(E element) {
            this.element = element;
        }
        
        // implements ActionListener
        public void actionPerformed(ActionEvent e) {
            AbstractButton button = (AbstractButton)e.getSource();
            if (button.isSelected())
                model.setSelection (element);
            else if (model.getSelection() == element) {
                if  (clearable)
                    model.clearSelection(); // clear if deselected
                else
                    button.setSelected(true); // override deselection
            }
        }
        
    }
    
}
