                                 /*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "syncml.h"

#include "syncml_internals.h"
#include "sml_queue_internals.h"
#include "sml_session_internals.h"
#include "sml_command_internals.h"
#include "sml_elements_internals.h"
#include "sml_parse_internals.h"
#include "sml_transport_internals.h"

/**
 * @defgroup SessionPrivate Session Private API
 * @ingroup PrivateAPI
 * @brief Private Interfaces to create, manipulate and delete sessions
 * 
 */
/*@{*/

typedef enum SmlSessionCommandType {
	SML_SESSION_COMMAND_START,
	SML_SESSION_COMMAND_END,
	SML_SESSION_STATUS,
	SML_SESSION_FLUSH
} SmlSessionCommandType;

typedef struct SmlSessionCommand {
	SmlSessionCommandType type;
	SmlCommand *cmd;
	SmlCommand *parent;
	SmlStatus *status;
	SmlStatusReplyCb callback;
	void *callbackUserdata;
	SmlBool final;
	SmlBool end;
} SmlSessionCommand;

void smlSessionDispatchEvent(SmlSession *session, SmlSessionEventType type, SmlCommand *cmd, SmlCommand *parent, SmlStatus *headerreply, SmlError *error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p, %p, %p, %p)", __func__, session, type, cmd, parent, headerreply, error);
	smlAssert(session);
	smlAssert(session->eventCallback);
	
	session->eventCallback(session, type, cmd, parent, headerreply, error, session->eventCallbackUserdata);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

static void _smlSessionFreePendingReplies(SmlSession *session)
{
	while (session->pendingReplies) {
		g_free((SmlPendingStatus *)session->pendingReplies->data);
		session->pendingReplies = g_list_delete_link(session->pendingReplies, session->pendingReplies);
	}
}

/** @brief Flushes a session
 * 
 * This flushes the already added statuses and/or commands. Note that if the session
 * has a size limitation, it will get flushed automatically if the size of the already
 * added commands exceeds the limit. Use this function if you want for some reason
 * to flush manually.
 * 
 * @param session The session to flush
 * @param final Set to TRUE if you want to have the final flag added. Otherwise it will just flush
 * the already added statuses and commands
 * @param error A error struct
 * @return TRUE if successful, FALSE otherwise
 * 
 */
static SmlBool _smlSessionFlushInternal(SmlSession *session, SmlBool final, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p)", __func__, session, final, error);
	smlAssert(session);
	smlAssert(session->dataCallback);
	
	char *data = NULL;
	unsigned int size = 0;
	SmlBool end = FALSE;
	
	/* Add the latest header */
	if (!smlAssemblerAddHeader(session->assembler, session, error))
		goto error;
	
	/* Get the data that is waiting in the assembler */
	if (!smlAssemblerRun(session->assembler, &data, &size, &end, final, error))
		goto error;
	
	/* Flush and reset the command ID counter */
	session->lastCommandID = smlAssemblerFlush(session->assembler) - 1;
	
	SmlTransportData *tspdata = smlTransportDataNew(data, size, session->type, TRUE, error);
	if (!tspdata)
		goto error_free_data;
	data = NULL;
	
	if (end && session->sessionType == SML_SESSION_TYPE_SERVER)
		tspdata->needsAnswer = FALSE;
	
	/* Send the flush event */
	smlSessionDispatchEvent(session, SML_SESSION_EVENT_FLUSH, NULL, NULL, NULL, NULL);
	
	/* Send the data over the transport */
	session->dataCallback(session, tspdata, session->dataCallbackUserdata);
		
	smlTransportDataDeref(tspdata);
	
	session->waiting = TRUE;
	
	if (final)
		session->sending = FALSE;
		
	if (end && session->sessionType == SML_SESSION_TYPE_SERVER) {
		smlTrace(TRACE_INTERNAL, "Ending session now");
		session->end = TRUE;
		smlSessionDispatchEvent(session, SML_SESSION_EVENT_END, NULL, NULL, NULL, NULL);
	}
	
	session->lastMessageID++;
	
	session->hasCommand = FALSE;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_data:
	g_free(data);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static void _check_reply(SmlSession *session, SmlStatus *status, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, status, userdata);

	smlTrace(TRACE_EXIT, "%s", __func__);
}

static void _smlSessionFragmentStatus(SmlSession *session, SmlStatus *status, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, status, userdata);
	//SmlError *error = NULL;
	
	smlTrace(TRACE_INTERNAL, "Got a status reply %i", status->type);
	
	if (smlStatusGetCode(status) != SML_CHUNK_ACCEPTED) {
		/* We need to send a error reply for the original command */
		
		smlCommandUnref(session->frag_command);
		session->frag_command = NULL;
	}

	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
/*error:
	smlSessionDispatchEvent(session, SML_SESSION_EVENT_ERROR, NULL, NULL, NULL, error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
	return;*/
}

static SmlCommand *_smlSessionFragmentSend(SmlSession *session, SmlCommand *orig_cmd, SmlCommand *parent, unsigned int space, unsigned int start, unsigned int complete_size, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %i, %i, %i, %p)", __func__, session, orig_cmd, parent, space, start, complete_size, error);
	SmlCommand *frag_cmd = NULL;
	
	char *data = NULL;
	unsigned int size = 0;
	if (!smlItemGetData(orig_cmd->private.change.item, &data, &size, error))
		goto error;
	
	if (space < size - start) {
		/* We need to create a new command. But we only send as much data as space
		 * is available */
		frag_cmd = smlCommandNew(orig_cmd->type, error);
		if (!frag_cmd)
			goto error;
		
		frag_cmd->private.change.item = smlItemNewForData(data + start, space, error);
		if (!frag_cmd->private.change.item)
			goto error_free_cmd;
		
		frag_cmd->private.change.item->moreData = TRUE;
		
		frag_cmd->private.change.item->target = orig_cmd->private.change.item->target;
		if (frag_cmd->private.change.item->target)
			smlLocationRef(frag_cmd->private.change.item->target);
		
		frag_cmd->private.change.item->source = orig_cmd->private.change.item->source;
		if (frag_cmd->private.change.item->source)
			smlLocationRef(frag_cmd->private.change.item->source);
		
		frag_cmd->private.change.item->contenttype = g_strdup(orig_cmd->private.change.item->contenttype);
		
		if (start == 0)
			frag_cmd->size = complete_size;
			
		session->frag_size += space;
	} else {
		/* We have enough room to send the original command. But we have to alter to item
		 * to only hold the rest of the data */
		SmlItem *orig_item = orig_cmd->private.change.item;
		
		orig_cmd->private.change.item = smlItemNewForData(data + start, size - start, error);
		if (!orig_cmd->private.change.item)
			goto error;
		orig_cmd->private.change.item->contenttype = g_strdup(orig_item->contenttype);
		
		orig_cmd->private.change.item->target = orig_item->target;
		if (orig_cmd->private.change.item->target)
			smlLocationRef(orig_cmd->private.change.item->target);
		
		orig_cmd->private.change.item->source = orig_item->source;
		if (orig_cmd->private.change.item->source)
			smlLocationRef(orig_cmd->private.change.item->source);
		
		smlItemUnref(orig_item);
		
		frag_cmd = orig_cmd;
		
		session->frag_size += size - start;
	}
	
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, frag_cmd);
	return frag_cmd;
	
error_free_cmd:
	smlCommandUnref(frag_cmd);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

static SmlAssemblerResult _smlSessionStartCommandInternal(SmlSession *session, SmlCommand *cmd, SmlCommand *parent, SmlStatusReplyCb callback, void *userdata, SmlBool addToCommandStack, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p, %p, %i, %p)", __func__, session, cmd, parent, callback, userdata, addToCommandStack, error);
	smlAssert(session);
	smlAssert(cmd);
	SmlPendingStatus *pending = NULL;
	SmlBool fragmented = FALSE;
	
	if (parent && !session->hasCommand) {
		GList *c = NULL;
		SmlCommand *oldcmd = NULL;
		SmlCommand *oldparent = NULL;
		GList * cmds = g_list_copy(session->commandStack);
		for (c = cmds; c; c = c->next) {
			oldcmd = c->data;
		
			if (_smlSessionStartCommandInternal(session, oldcmd, oldparent, _check_reply, NULL, FALSE, error) != SML_ASSEMBLER_RESULT_OK)
				goto error;
			oldparent = oldcmd;
		}
		g_list_free(cmds);
	}
	
	/* Large object handling. Only possible of the session has a limit and
	 * we dont have a 1.0 session */
	if ((cmd->type == SML_COMMAND_TYPE_ADD || cmd->type == SML_COMMAND_TYPE_REPLACE) && smlAssemblerGetLimit(session->assembler) > 0 && session->version != SML_VERSION_10) {
		
		smlTrace(TRACE_INTERNAL, "Checking if command needs to be fragmented");
		
		char *data = NULL;
		unsigned int size = 0;
		if (!smlItemGetData(cmd->private.change.item, &data, &size, error))
			goto error;
		
		/* If the size of the item is larger than the max obj size, we have to return
		 * an error */
		if (size > smlSessionGetSendingMaxObjSize(session)) {
			smlErrorSet(error, SML_ERROR_SIZE_MISMATCH, "Item is larger than the limit");
			goto error;
		}
		
		if (!session->frag_command)
			cmd->size = size;
		cmd->private.change.item->moreData = TRUE;
		int space = 0;
		if (!smlAssemblerGetSpace(session->assembler, &space, parent, cmd, error))
			goto error;
		cmd->size = 0;
		cmd->private.change.item->moreData = FALSE;
		
		/* Check if item data fits into the current message */
		if (session->frag_command || (space && space < size - session->frag_size)) {
			smlTrace(TRACE_INTERNAL, "Space %i, size %i. Fragmenting. Already added: %i", space, size - session->frag_size, session->frag_size);
			/* We need to fragment */
			/* Store the command for processing */
			
			if (!session->frag_command) {
				session->frag_size = 0;
				
				session->frag_command = cmd;
				smlCommandRef(cmd);
				
				session->frag_callback = callback;
				session->frag_userdata = userdata;
			}
			
			if (!(cmd = _smlSessionFragmentSend(session, session->frag_command, parent, space, session->frag_size, size, error)))
				goto error;
			
			callback = _smlSessionFragmentStatus;
			userdata = NULL;
			fragmented = TRUE;
			
			if (cmd->private.change.item->moreData == FALSE) {
				smlTrace(TRACE_INTERNAL, "This is the last chunk");
				
				callback = session->frag_callback;
				userdata = session->frag_userdata;
				fragmented = FALSE;
				
				session->frag_size = 0;
				
				smlCommandUnref(session->frag_command);
				session->frag_command = NULL;
			}
			
			smlTrace(TRACE_INTERNAL, "%s: Fragmented. Added %i already", __func__, session->frag_size);
		}
	}
	
	/* We now increment the session ID */
	session->lastCommandID++;
	cmd->cmdID = session->lastCommandID;
	
	/* Now we can try to add the command to the assembler */
	switch (smlAssemblerStartCommand(session->assembler, parent, cmd, error)) {
		case SML_ASSEMBLER_RESULT_OK:
			/* We successfully added the command */
			cmd->pushedBack = FALSE;
			break;
		case SML_ASSEMBLER_RESULT_MISMATCH:
			if (cmd->pushedBack) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Command is too large");
				goto error;
			}
			cmd->pushedBack = TRUE;
			
			/* We werent able to add the command. So we have to flush the already
			 * added statuses/commands and then add it again later */
			if (!_smlSessionFlushInternal(session, FALSE, error)) {
				session->lastCommandID--;
				goto error;
			}
			
			smlTrace(TRACE_EXIT, "%s: Mismatch", __func__);
			return SML_ASSEMBLER_RESULT_MISMATCH;
		case SML_ASSEMBLER_RESULT_ERROR:
			session->lastCommandID--;
			goto error;
	}
	
	/* If the user requested a reply to the command we generate
	 * the pending status so that we can dispatch the incoming status
	 */
	if (callback) {
		pending = smlTryMalloc0(sizeof(SmlPendingStatus), error);
		if (!pending) {
			session->lastCommandID--;
			goto error;
		}
		
		pending->callback = callback;
		pending->userdata = userdata;
		pending->cmdID = cmd->cmdID;
		pending->msgID = session->lastMessageID;
		
		smlTrace(TRACE_INTERNAL, "Appending pending status with cmdID %i and msgID %i", pending->cmdID, pending->msgID);
		session->pendingReplies = g_list_append(session->pendingReplies, pending);
	} else
		cmd->noResp = TRUE;
	
	/* We will get a status and a result to a get command
	 * so we register a second callback for the result */
	if (cmd->type == SML_COMMAND_TYPE_GET) {
		pending = smlTryMalloc0(sizeof(SmlPendingStatus), error);
		if (!pending) {
			session->lastCommandID--;
			goto error;
		}
		
		pending->callback = callback;
		pending->userdata = userdata;
		pending->cmdID = cmd->cmdID;
		pending->msgID = session->lastMessageID;
		
		smlTrace(TRACE_INTERNAL, "Appending pending status for a result with cmdID %i and msgID %i", pending->cmdID, pending->msgID);
		session->pendingReplies = g_list_append(session->pendingReplies, pending);
	}
	
	if (fragmented) {
		/* We werent able to add the command. So we have to flush the already
		 * added statuses/commands and then add it again later */
		if (!_smlSessionFlushInternal(session, FALSE, error)) {
			session->lastCommandID--;
			goto error;
		}
		
		smlCommandUnref(cmd);
		
		smlTrace(TRACE_EXIT, "%s: Mismatch but fragmented", __func__);
		return SML_ASSEMBLER_RESULT_MISMATCH;
	}
	
	if (addToCommandStack)
		session->commandStack = g_list_append(session->commandStack, cmd);
	session->hasCommand = TRUE;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return SML_ASSEMBLER_RESULT_OK;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return SML_ASSEMBLER_RESULT_ERROR;
}

static SmlBool _smlSessionEndCommandInternal(SmlSession *session, SmlCommand *parent, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, parent, error);
	smlAssert(session);
	
	if (!smlAssemblerEndCommand(session->assembler, parent, error))
		goto error;
	
	session->commandStack = g_list_delete_link(session->commandStack, g_list_last(session->commandStack));
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static void _smlSessionCommandFree(SmlSessionCommand *sesscmd)
{
	if (sesscmd->cmd)
		smlCommandUnref(sesscmd->cmd);
		
	if (sesscmd->parent)
		smlCommandUnref(sesscmd->parent);
		
	if (sesscmd->status)
		smlStatusUnref(sesscmd->status);
	
	g_free(sesscmd);
}

static void _smlSessionCommandHandler(void *message, void *userdata)
{
	smlAssert(message);
	smlAssert(userdata);
	SmlSessionCommand *sesscmd = message;
	SmlSession *session = userdata;
	SmlError *error = NULL;
	//SmlPendingStatus *pending = NULL;
	
	SmlCommand *parent = sesscmd->parent;
	SmlCommand *cmd = sesscmd->cmd;
	SmlStatus *status = sesscmd->status;
	
	switch (sesscmd->type) {
		case SML_SESSION_FLUSH:
			if (sesscmd->end) {
				session->end = TRUE;
				smlTrace(TRACE_INTERNAL, "End on flush was set");
			}
			if (!_smlSessionFlushInternal(session, sesscmd->final, &error))
				goto error;
			break;
		case SML_SESSION_COMMAND_END:
			if (!_smlSessionEndCommandInternal(session, parent, &error))
				goto error;
			break;
		case SML_SESSION_COMMAND_START:
			switch (_smlSessionStartCommandInternal(session, cmd, parent, sesscmd->callback, sesscmd->callbackUserdata, TRUE, &error)) {
				case SML_ASSEMBLER_RESULT_OK:
					/* We successfully added the command */
					break;
				case SML_ASSEMBLER_RESULT_MISMATCH:
					smlQueuePushHead(session->command_queue, message);
					return;
				case SML_ASSEMBLER_RESULT_ERROR:
					goto error;
			}
			break;
		case SML_SESSION_STATUS:;
			unsigned int cmdRef, msgRef = 0;
			if (!smlAssemblerGetNextCmdRef(session->assembler, &cmdRef, &msgRef)) {
				smlErrorSet(&error, SML_ERROR_GENERIC, "No more statuses were needed");
				goto error;
			}
			
			if (status->cmdRef != cmdRef || status->msgRef != msgRef) {
				//Put it on the end of the queue
				smlQueueSendPrio(session->command_queue, message);
				return;
			}
			
			/* We ignore if the added status violates the size limitation if
			 * a incoming buffer for a large object is open. the problem is that
			 * the status for the chunk has to go into the next message AND the status
			 * MUST be in the same order as the original commands. so the only solution is
			 * to send all statuses and ignore the size. */
			
			/* Now we can try to add the status to the assembler */
			switch (smlAssemblerAddStatusFull(session->assembler, status, session->incomingBuffer ? TRUE : FALSE, &error)) {
				case SML_ASSEMBLER_RESULT_OK:
					/* We successfully added the status */
					session->active = TRUE;
					break;
				case SML_ASSEMBLER_RESULT_MISMATCH:
					/* We werent able to add the command. So we have to flush the already
					 * added statuses/commands and then add it again later */
					if (!_smlSessionFlushInternal(session, FALSE, &error)) {
						session->lastCommandID--;
						goto error;
					}
					
					/* Push the message back into the queue */
					smlQueuePushHeadPrio(session->command_queue, message);
					return;
					break;
				case SML_ASSEMBLER_RESULT_ERROR:
					goto error;
			}
			break;
	}
	
	_smlSessionCommandFree(sesscmd);
	
	return;

error:
	smlSessionDispatchEvent(session, SML_SESSION_EVENT_ERROR, NULL, NULL, NULL, error);
	_smlSessionCommandFree(sesscmd);
	return;
}

SmlBool smlSessionReceiveHeader(SmlSession *session, SmlHeader *header, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, header, error);
	smlAssert(session);
	smlAssert(header);

	/* check source, target, version etc */
	if (header->messageID <= session->lastReceivedMessageID) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Message ID not incremented");
		goto error;
	}
	session->lastReceivedMessageID = header->messageID;
	session->lastMessageID = header->messageID + 1;
	session->lastCommandID++;
	
	smlAssemblerSetRequestedLimit(session->assembler, header->maxmsgsize);
	
	/* Reserve the status for header reply. It will always be cmdRef 0
	 * and it will always be the first command in the message */
	if (!smlAssemblerReserveStatus(session->assembler, 0, header->messageID, 1, error))
		goto error;

	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlSessionDispatchStatus(SmlSession *session, SmlStatus *status, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, status, error);
	smlAssert(status);
	
	if (status->cmdRef == 0) {
		smlSessionDispatchEvent(session, SML_SESSION_EVENT_HEADER_REPLY, NULL, NULL, status, NULL);
		goto out;
	}
	
	//Check if a handler for this object at this path has been installed.
	GList *o = NULL;
	for (o = session->pendingReplies; o; o = o->next) {
		SmlPendingStatus *pending = o->data;
		if (pending->cmdID == status->cmdRef && pending->msgID == status->msgRef) {
			smlTrace(TRACE_INTERNAL, "Found pending status %p", pending);
			session->pendingReplies = g_list_remove(session->pendingReplies, pending);
			
			smlAssert(pending->callback);
			pending->callback(session, status, pending->userdata);
			
			g_free(pending);
			
			goto out;
		}
	}

	/* We have to catch at this point if the status reply is the reply
	 * to our server alerted sync. Since this initial alert is not sent
	 * over a session, the status can of course not be wanted */
	if (status->type == SML_COMMAND_TYPE_ALERT && !status->anchor) {
		smlTrace(TRACE_INTERNAL, "Handling status for server alerted sync");
		goto out;
	}
	
	smlErrorSet(error, SML_ERROR_GENERIC, "Received unwanted status reply");
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
	
out:
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
}

SmlBool smlSessionCheck(SmlSession *session)
{
	smlAssert(session);
	
	if (session->waiting)
		return FALSE;
	
	if (!smlAssemblerIsStatusMissing(session->assembler))
		return smlQueueCheck(session->command_queue);
	else
		return smlQueueCheckPrio(session->command_queue);
}

void smlSessionDispatch(SmlSession *session)
{
	smlAssert(session);
	
	if (session->waiting)
		return;
	
	/* If we are still missing statuses, we just dispatch the statuses
	 * If we are complete on the status side, we dispatch the commands
	 */
	if (!smlAssemblerIsStatusMissing(session->assembler))
		smlQueueDispatch(session->command_queue);
	else
		smlQueueDispatchPrio(session->command_queue);
}

SmlBool smlSessionReceiveBody(SmlSession *session, SmlParser *parser, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, parser, error);
	smlAssert(session);
	smlAssert(parser);
	SmlStatus *status = NULL;
	SmlCommand *cmd = NULL;
	SmlParserResult result = SML_PARSER_RESULT_ERROR;
	
parse_status:
	/* Now lets go through the available statuses (if any) */
	while (1) {
		if (!smlParserGetStatus(parser, &status, error))
			goto error_free_status;
		
		if (!status)
			break;
		
		/* Call the callback*/
		if (!smlSessionDispatchStatus(session, status, error))
			goto error_free_status;
		
		smlStatusUnref(status);
	}
	
	/* Go through the remaining commands */
	while (1) {
		result = smlParserGetCommand(parser, &cmd, error);
		if (result == SML_PARSER_RESULT_OTHER)
			break;
		
		if (cmd)
			cmd->msgID = session->lastReceivedMessageID;
		
		switch (result) {
			case SML_PARSER_RESULT_OPEN:
				session->lastCommandID++;
				/* Reserve the place for the status in the assembler */
				if (!smlAssemblerReserveStatus(session->assembler, cmd->cmdID, session->lastReceivedMessageID, session->lastCommandID, error))
					goto error_free_cmd;
				
				/* Store the parent */
				if (session->parentCommand) {
					smlErrorSet(error, SML_ERROR_GENERIC, "Already had a parent");
					goto error_free_cmd;
				}
				
				/* If we are a server, we just mimick the behaviour of the client regarding
				 * large object handling and support of number of changes. This way we dont
				 * need to parse the devinf */
				if (cmd->type == SML_COMMAND_TYPE_SYNC) {
					smlTrace(TRACE_INTERNAL, "maxObjSize: %i, numbOfChanges: %i, %i", cmd->private.sync.maxObjSize, cmd->private.sync.hasNumChanged, cmd->private.sync.numChanged);
					/* If the sync has the maxObjSize set (>= 0), we set the requested max obj size.
					 * If the max obj size was set before (with the alert, we overwrite it */
					if (cmd->private.sync.maxObjSize >= 0)
						smlSessionSetRequestedMaxObjSize(session, cmd->private.sync.maxObjSize);
				
					if (session->sessionType == SML_SESSION_TYPE_SERVER) {
						if (cmd->private.sync.hasNumChanged == FALSE)
							smlSessionUseNumberOfChanges(session, FALSE);
					}
				}
				
				
				session->parentCommand = cmd;
				cmd->msgID = session->lastReceivedMessageID;
				
				if (!session->end) {
					smlSessionDispatchEvent(session, SML_SESSION_EVENT_COMMAND, cmd, NULL, NULL, NULL);
				} else {
					smlTrace(TRACE_INTERNAL, "Replying with 407 since the session has ended");
					SmlStatus *reply = smlCommandNewReply(cmd, SML_ERROR_AUTH_REQUIRED, error);
					if (!reply)
						goto error_free_cmd;
					
					if (!smlSessionSendReply(session, reply, error))
						goto error_free_cmd;
						
					smlStatusUnref(reply);
				}
				break;
			case SML_PARSER_RESULT_NORMAL:
				session->lastCommandID++;
				/* Reserve the place for the status in the assembler */
				if (!smlAssemblerReserveStatus(session->assembler, cmd->cmdID, session->lastReceivedMessageID, session->lastCommandID, error))
					goto error_free_cmd;
					
				/* Here we catch Alerts for the next message */
				if (cmd->type == SML_COMMAND_TYPE_ALERT) {
					if (cmd->private.alert.type == SML_ALERT_NEXT_MESSAGE) {
						SmlStatus *reply = smlCommandNewReply(cmd, SML_NO_ERROR, error);
						if (!reply)
							goto error_free_cmd;
						
						if (!smlSessionSendReply(session, reply, error))
							goto error_free_cmd;
							
						smlStatusUnref(reply);
						
						smlCommandUnref(cmd);
						break;
					} else {
						smlTrace(TRACE_INTERNAL, "Alert maxObjSize: %i", cmd->private.alert.maxObjSize);
						if (cmd->private.alert.maxObjSize >= 0)
							smlAssemblerSetRequestedMaxObjSize(session->assembler, cmd->private.alert.maxObjSize);
					}
				}
				
				/* Here we catch changes with moreData set */
				if (cmd->type == SML_COMMAND_TYPE_ADD || cmd->type == SML_COMMAND_TYPE_REPLACE) {
					
					if (session->incomingBuffer) {
						smlTrace(TRACE_INTERNAL, "Appending to item buffer");
						/* We have a command in the incoming buffer. So we append the data to it.
						 * If the new command does not have moreData set, we dispatch the command */

						/* We already have a item in the buffer. So the cmd cannot have the size set */
						if (cmd->size) {
							smlErrorSet(error, SML_ERROR_GENERIC, "Only the first chunk can have the size set");
							goto error_free_cmd;
						}
						 
						char *data = NULL;
						unsigned int size = 0;
						if (!smlItemGetData(cmd->private.change.item, &data, &size, error))
							goto error_free_cmd;
						
						if (!smlItemAddData(session->incomingBuffer, data, size, error))
							goto error_free_cmd;
						
						smlTrace(TRACE_INTERNAL, "Appended %i to buffer. Buffer size: %i. Required: %i", size, xmlBufferLength(session->incomingBuffer->buffer), session->incomingBuffer->size);
						
						if (!cmd->private.change.item->moreData) {
							smlTrace(TRACE_INTERNAL, "Item buffer complete. Dispatching.");
							/* This cmd doesnt have more data set. So we check if the data
							 * is complete and dispatch it if it is */
							if (!smlItemCheck(session->incomingBuffer)) {
								/* We have a size mismatch */
								SmlStatus *reply = smlCommandNewReply(cmd, SML_ERROR_SIZE_MISMATCH, error);
								if (!reply)
									goto error_free_cmd;
								
								if (!smlSessionSendReply(session, reply, error))
									goto error_free_cmd;
									
								smlStatusUnref(reply);
								
								smlErrorSet(error, SML_ERROR_GENERIC, "Chunked item size mismatch");
								goto error_free_cmd;
							}
							
							/* Here we transfer the buffered item to the last command which gets dispatched */
							smlItemUnref(cmd->private.change.item);
							session->incomingBuffer->moreData = FALSE;
							cmd->private.change.item = session->incomingBuffer;
							session->incomingBuffer = NULL;
						}
					} else if (cmd->private.change.item->moreData) {
						/* We dont have a command in the buffer yet. so we set the item of current command as
						 * the buffer */
						
						if (!cmd->size) {
							smlErrorSet(error, SML_ERROR_GENERIC, "First MoreData item didnt have size set");
							goto error_free_cmd;
						}
						
						session->incomingBuffer = cmd->private.change.item;
						smlItemRef(session->incomingBuffer);
						smlTrace(TRACE_INTERNAL, "New Buffer started. Buffer size: %i. Required: %i", xmlBufferLength(session->incomingBuffer->buffer), session->incomingBuffer->size);
						
						smlTrace(TRACE_INTERNAL, "Content so far: %s\n", smlPrintBinary((char *)xmlBufferContent(session->incomingBuffer->buffer), xmlBufferLength(session->incomingBuffer->buffer)));
					}
					
					if (cmd->private.change.item->moreData == TRUE) {
						smlTrace(TRACE_INTERNAL, "Got item with moreData %i", cmd->msgID);
						
						SmlStatus *reply = smlCommandNewReply(cmd, SML_CHUNK_ACCEPTED, error);
						if (!reply)
							goto error_free_cmd;
						
						if (!smlSessionSendReply(session, reply, error))
							goto error_free_cmd;
							
						smlStatusUnref(reply);
						
						smlCommandUnref(cmd);
						break;
					}
				}
				
				/* if the command is a result, we treat it as a status */
				if (cmd->type == SML_COMMAND_TYPE_RESULTS) {
					/* Call the callback*/
					if (!smlSessionDispatchStatus(session, cmd->private.results.status, error))
						goto error_free_status;
				} else {
					/* Dispatch the command */
					cmd->msgID = session->lastReceivedMessageID;
					if (!session->end) {
						if (!session->parentCommand) {
							smlSessionDispatchEvent(session, SML_SESSION_EVENT_COMMAND, cmd, NULL, NULL, NULL);
						} else {
							smlSessionDispatchEvent(session, SML_SESSION_EVENT_CHILD_COMMAND, cmd, session->parentCommand, NULL, NULL);
						}
					} else {
						smlTrace(TRACE_INTERNAL, "Replying with 407 since the session has ended2");
						SmlStatus *reply = smlCommandNewReply(cmd, SML_ERROR_AUTH_REQUIRED, error);
						if (!reply)
							goto error_free_cmd;
						
						if (!smlSessionSendReply(session, reply, error))
							goto error_free_cmd;
							
						smlStatusUnref(reply);
					}
				}
				
				smlCommandUnref(cmd);
				break;
			case SML_PARSER_RESULT_CLOSE:
				/* Remove the parent */
				smlCommandUnref(session->parentCommand);
				session->parentCommand = NULL;
				break;
			case SML_PARSER_RESULT_STATUS:
				goto parse_status;
			default:
				goto error_free_cmd;
		}
	}
	
	/* Check that the message closes correctly */
	SmlBool final = FALSE;
	SmlBool end = FALSE;
	if (!smlParserEnd(parser, &final, &end, error))
		goto error_free_cmd;
	
	session->waiting = FALSE;

	/* Dispatch the final event before we enter the following loop. Otherwise if upper
	   layers depend on the final event we might end in a deadlock. */
	if (final) {
		session->sending = TRUE;
		smlSessionDispatchEvent(session, SML_SESSION_EVENT_FINAL, NULL, NULL, NULL, NULL);
	}
	
	/* We have to wait at this point until no status are pending anymore
	 * if there is a incoming buffer for large objs open,
	 * the other side expects to see the status for the open buffer of course. so we have to
	 * wait here for all open statuses, since they have to be added in the correct order and
	 * the status for the open buffer is obviously last */	
	while (1) {
		unsigned int cmdRef, msgRef = 0;
		if (!smlAssemblerGetNextCmdRef(session->assembler, &cmdRef, &msgRef))
			break;
		
		usleep(10000);
		
		int i = 0;
		unsigned int num = smlQueueLengthPrio(session->command_queue);
		smlTrace(TRACE_INTERNAL, "Now dispatching %i statuses. Session waiting: %i, prio: %i", num, session->waiting, smlQueueCheckPrio(session->command_queue));
		while (!session->waiting && smlQueueCheckPrio(session->command_queue) && i < num) {
			smlSessionDispatch(session);
			i++;
		}
	}
	
	smlTrace(TRACE_INTERNAL, "Final %i, waiting %i, sending %i", final, session->waiting, session->sending);
	if (final) {
		if (session->pendingReplies) {
			SmlPendingStatus *pending = (SmlPendingStatus *)session->pendingReplies->data;
			smlErrorSet(error, SML_ERROR_GENERIC, "Didnt receive a reply for pending status (cmdID %i, msgID %i)", pending->cmdID, pending->msgID);
			_smlSessionFreePendingReplies(session);
			goto error;
		}
		
		if (session->incomingBuffer) {
			smlErrorSet(error, SML_ERROR_GENERIC, "Incoming buffer was open when final arrived");
			smlItemUnref(session->incomingBuffer);
			session->incomingBuffer = NULL;
			goto error;
		}
	} else if (!session->waiting && !session->sending) {
		smlTrace(TRACE_INTERNAL, "This is not the final message. Empty? %i", smlAssemblerIsEmpty(session->assembler));
		/* This was not the final message. So we need to either
		 * send already added statuses or send an alert for the
		 * next message */
		if (smlAssemblerIsEmpty(session->assembler)) {
			/* The assembler is empty so we need to add an alert */
			SmlCommand *alert = smlCommandNewAlert(SML_ALERT_NEXT_MESSAGE, session->target, session->source, NULL, NULL, NULL, error);
			if (!alert)
				goto error_free_cmd;
				
			if (_smlSessionStartCommandInternal(session, alert, NULL, NULL, NULL, TRUE, error) != SML_ASSEMBLER_RESULT_OK)
				goto error_free_cmd;
			if (!_smlSessionEndCommandInternal(session, NULL, error))
				goto error_free_cmd;
			
			smlCommandUnref(alert);
		}
		if (!_smlSessionFlushInternal(session, FALSE, error))
			goto error_free_cmd;
	}
	
	if (end) {
		smlTrace(TRACE_INTERNAL, "Ending session now");
		session->end = TRUE;
		
		if (session->sessionType != SML_SESSION_TYPE_SERVER) {
			smlSessionDispatchEvent(session, SML_SESSION_EVENT_END, NULL, NULL, NULL, NULL);
		} else {
			/* if we are a server we have to flush here since we have to reply
			 * with a header reply */
			if (!_smlSessionFlushInternal(session, TRUE, error))
				goto error;
		}
	}

	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error_free_cmd:
	if (cmd)
		smlCommandUnref(cmd);
error_free_status:
	if (status)
		smlStatusUnref(status);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

void smlSessionSetEventCallback(SmlSession *session, SmlSessionEventCallback callback, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, callback, userdata);
	smlAssert(session);
	
	session->eventCallback = callback;
	session->eventCallbackUserdata = userdata;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

void smlSessionSetDataCallback(SmlSession *session, SmlSessionDataCallback callback, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, callback, userdata);
	smlAssert(session);
	
	session->dataCallback = callback;
	session->dataCallbackUserdata = userdata;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

const char *smlSessionGetSessionID(SmlSession *session)
{
	smlAssert(session);
	
	return session->sessionID;
}

void smlSessionSetSessionID(SmlSession *session, const char *sessionID)
{
	smlAssert(session);
	
	if (session->sessionID)
		g_free(session->sessionID);
	session->sessionID = g_strdup(sessionID);
}

SmlProtocolVersion smlSessionGetVersion(SmlSession *session)
{
	smlAssert(session);
	
	return session->version;
}

SmlLocation *smlSessionGetSource(SmlSession *session)
{
	smlAssert(session);
	return session->source;
}

SmlLocation *smlSessionGetTarget(SmlSession *session)
{
	smlAssert(session);
	return session->target;
}

/*@}*/

/**
 * @defgroup SessionPublic SyncML Session
 * @ingroup PublicAPI
 * @brief Interfaces to create, manipulate and delete sessions
 * 
 * A session has a unique id and consists of several packages which again contain several commands.
 * 
 */
/*@{*/

/**
 * @name Session Management
 * These function allow to create and delete sessions
 */
/*@{*/

/** @brief Creates a new session
 * 
 * Note that if you set the new session callback on the transport, you will
 * get a callback that this session was created (so that it gets easier to register
 * objects at a central place)
 * 
 * @param sessionType If the session is a server or a client.
 * @param mimetype The desired mime type. Choose if you want xml or wbxml
 * @param version What version of the protocol do you want to use?
 * @param protocol Which syncml protocol do you want to use?
 * @param target The remote target of this session. This is used in the header
 * @param source The source of this session. Used in the header
 * @param sessionID If you already have a sessionID, you can enter it here. Set it to NULL if you want to generate a new ID
 * @param messageID The message Id where this session should start. set to 0 to take the default
 * @param error A pointer to a error struct
 * @returns The new session or NULL if an error occured
 * 
 */
SmlSession *smlSessionNew(SmlSessionType sessionType, SmlMimeType mimetype, SmlProtocolVersion version, SmlProtocolType protocol, SmlLocation *target, SmlLocation *source, const char *sessionID, unsigned int messageID, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%i, %i, %i, %i, %p, %p, %s, %i, %p)", __func__, sessionType, mimetype, version, protocol, target, source, sessionID, messageID, error);
	
	/* Malloc the new session object */
	SmlSession *session = smlTryMalloc0(sizeof(SmlSession), error);
	if (!session)
		goto error;
	
	session->refCount = 1;
	session->version = version;
	session->protocol = protocol;
	session->type = mimetype;
	session->sessionID = g_strdup(sessionID);
	if (messageID)
		session->lastMessageID = messageID;
	else
		session->lastMessageID = 1;
	
	session->end = FALSE;
	session->sessionType = sessionType;
	session->lastCommandID = 0;
	
	if (sessionType == SML_SESSION_TYPE_CLIENT)
		session->sending = TRUE;
	else
		session->sending = FALSE;
	
	session->source = source;
	smlLocationRef(source);
		
	session->target = target;
	smlLocationRef(target);
	
	/* Create the assembler that will parse messages send
	 * from this session */
	session->assembler = smlAssemblerNew(mimetype, 0, error);
	if (!session->assembler)
		goto error_free_session;
	smlAssemblerSetOption(session->assembler, "USE_STRTABLE", "1");
	
	if (!smlAssemblerStart(session->assembler, session, error))
		goto error_free_session;
	
	if (!smlAssemblerAddHeader(session->assembler, session, error))
		goto error_free_session;
	
	session->command_queue = smlQueueNew(error);
	if (!session->command_queue)
		goto error_free_session;
	smlQueueSetHandler(session->command_queue, (SmlQueueHandler)_smlSessionCommandHandler, session);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, session);
	return session;

error_free_session:
	smlSessionUnref(session);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

void smlSessionRef(SmlSession *session)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, session);
	smlAssert(session);
	
	g_atomic_int_inc(&(session->refCount));
	
	smlTrace(TRACE_EXIT, "%s: New refcount: %i", __func__, session->refCount);
}

void smlSessionUnref(SmlSession *session)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, session);
	smlAssert(session);
	
	if (g_atomic_int_dec_and_test(&(session->refCount))) {
		smlTrace(TRACE_INTERNAL, "Refcount == 0!");
		
		if (session->target)
			smlLocationUnref(session->target);
		
		if (session->source)
			smlLocationUnref(session->source);
		
		if (session->sessionID)
			g_free(session->sessionID);
		
		if (session->command_queue) {
			SmlSessionCommand *sesscmd = NULL;
			while ((sesscmd = smlQueueTryPop(session->command_queue)))
				_smlSessionCommandFree(sesscmd);
			
			smlQueueFree(session->command_queue);
		}
		
		_smlSessionFreePendingReplies(session);
		
		if (session->assembler)
			smlAssemblerFree(session->assembler);
		
		if (session->incomingBuffer)
			smlItemUnref(session->incomingBuffer);
		
		if (session->parentCommand)
			smlCommandUnref(session->parentCommand);
		
		g_free(session);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

/*@}*/

/**
 * @name Session Control
 * These function allow to control a session
 */
/*@{*/

/** @brief Flushes a session
 * 
 * This flushes the already added statuses and/or commands. Note that if the session
 * has a size limitation, it will get flushed automatically if the size of the already
 * added commands exceeds the limit. Use this function if you want for some reason
 * to flush manually.
 * 
 * @param session The session to flush
 * @param final Set to TRUE if you want to have the final flag added. Otherwise it will just flush
 * the already added statuses and commands
 * @param error A error struct
 * @return TRUE if successful, FALSE otherwise
 * 
 */
SmlBool smlSessionFlush(SmlSession *session, SmlBool final, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p)", __func__, session, final, error);
	smlAssert(session);
	
	SmlSessionCommand *sesscmd = smlTryMalloc0(sizeof(SmlSessionCommand), error);
	if (!sesscmd)
		goto error;
	
	sesscmd->type = SML_SESSION_FLUSH;
	sesscmd->final = final;
	
	smlTrace(TRACE_INTERNAL, "sending command %p", sesscmd);

	smlQueueSend(session->command_queue, sesscmd);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/** @brief Ends the session
 * 
 * This function will end the session. This means that no more commands will be sent.
 * If there are replies waiting they will be sent. if there are commands waiting from the
 * remote side which have not yet been answered, they will get a error reply. If you already
 * sent commands in this session, you will receive error replies for your callbacks.
 * 
 * If you registered a session end handler it will get called. if the session is a server type
 * and it receives a command from the remote side for the ended session libsyncml will
 * automatically send an error reply.
 * 
 * Sessions will be deleted from the server session after a timeout.
 * 
 * @param session The session to flush
 * @param error A error struct
 * @return TRUE if successful, FALSE otherwise
 * 
 */
SmlBool smlSessionEnd(SmlSession *session, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, session, error);
	smlAssert(session);
	
	SmlSessionCommand *sesscmd = smlTryMalloc0(sizeof(SmlSessionCommand), error);
	if (!sesscmd)
		goto error;
	
	sesscmd->type = SML_SESSION_FLUSH;
	sesscmd->final = TRUE;
	sesscmd->end = TRUE;
	smlTrace(TRACE_INTERNAL, "sending command %p", sesscmd);

	smlQueueSend(session->command_queue, sesscmd);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

void smlSessionUseStringTable(SmlSession *session, SmlBool useStringtable)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i)", __func__, session, useStringtable);
	smlAssert(session);
	
	smlAssemblerSetOption(session->assembler, "USE_STRTABLE", useStringtable ? "1" : "0");
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

void smlSessionUseNumberOfChanges(SmlSession *session, SmlBool support)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i)", __func__, session, support);
	smlAssert(session);
	
	smlAssemblerSetOption(session->assembler, "USE_NUMBEROFCHANGES", support ? "1" : "0");
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

void smlSessionUseLargeObjects(SmlSession *session, SmlBool support)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i)", __func__, session, support);
	smlAssert(session);
	
	smlAssemblerSetOption(session->assembler, "USE_LARGEOBJECTS", support ? "1" : "0");
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

void smlSessionUseOnlyReplace(SmlSession *session, SmlBool onlyReplace)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i)", __func__, session, onlyReplace);
	smlAssert(session);
	
	smlAssemblerSetOption(session->assembler, "ONLY_REPLACE", onlyReplace ? "1" : "0");
	session->onlyReplace = onlyReplace;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

void smlSessionSetSendingLimit(SmlSession *session, unsigned int limit)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %u)", __func__, session, limit);
	smlAssert(session);
	
	smlAssemblerSetLimit(session->assembler, limit);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

unsigned int smlSessionGetSendingLimit(SmlSession *session)
{
	smlAssert(session);
	return smlAssemblerGetLimit(session->assembler);
}

void smlSessionSetReceivingLimit(SmlSession *session, unsigned int limit)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %u)", __func__, session, limit);
	smlAssert(session);
	
	session->incomingMaxMsgSize = limit;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

/** Sets the maximum obj size we want to send. If we try to send a item to the remote
 * side which is larger than this limit we return an error 
 * 
 * Possible values for limit are >= -1
 * 
 * -1 means that no sending limit is requested
 * 0 means that we can send objects of any size
 * > 0 means that a limit is set
 * 
 * */
void smlSessionSetSendingMaxObjSize(SmlSession *session, int limit)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i)", __func__, session, limit);
	smlAssert(session);
	smlAssert(limit >= -1);
	
	smlAssemblerSetSendingMaxObjSize(session->assembler, limit);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

/** Sets the maximum obj size as requested by the other side. We cannot send an item larger
 * than this limit 
 * 
 * Possible values for limit are >= -1
 * 
 * -1 means that no sending limit is requested
 * 0 means that we can send objects of any size
 * > 0 means that a limit is set
 * 
 * */
void smlSessionSetRequestedMaxObjSize(SmlSession *session, int limit)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i)", __func__, session, limit);
	smlAssert(session);
	smlAssert(limit >= -1);
	
	smlAssemblerSetRequestedMaxObjSize(session->assembler, limit);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

/** Returns the maximum obj size that we can send. If both the sending and the
 * requested limit are set, this function will return the smaller one of both.
 * 
 * Otherwise it will return the larger number. */
int smlSessionGetSendingMaxObjSize(SmlSession *session)
{
	smlAssert(session);
	return smlAssemblerGetSendingMaxObjSize(session->assembler);
}

/** Sets the obj size that will get transmitted to the remote side in the
 * <MaxObjSize> tag. So this sets the maximum size of the object we want to receive.
 * If an item is larger, we return an error 
 * 
 * In this function, 0 means that no limit is set.
 * a value > 0 means that a limit is set. note that this is required if large object
 * handling has to be supported.
 * 
 * The value is stored in the session since it has to be used by the assembler (to fill
 * the MaxObjSize tag) as well as by the session (to determine when to send an error if the
 * limit is violated)
 * 
 * */
void smlSessionSetReceivingMaxObjSize(SmlSession *session, unsigned int limit)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %u)", __func__, session, limit);
	smlAssert(session);
	
	session->receivingMaxObjSize = limit;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

unsigned int smlSessionGetReceivingMaxObjSize(SmlSession *session)
{
	smlAssert(session);
	return session->receivingMaxObjSize;
}

/*@}*/

/** @brief Sends a command with a parent
 * 
 * This function can be used to send subcommands (for example a "Add" inside a "Sync"). You
 * are guaranteed that you will get a callback if this function returns TRUE. So either this function
 * returns FALSE (might occur if the command is not complete or if a OOM occured) or you will later
 * get a reply or an error in the callback.
 * 
 * Note that if you dont set a callback function, libsyncml will automatically add a "NoResp".
 * 
 * @param session The session
 * @param cmd The command to send
 * @param parent The parent of this command. NULL for syncbody
 * @param callback The callback that will receive the reply. Set to NULL if you dont want any replies.
 * @param userdata The userdata to pass to the callback
 * @param error A error struct
 * @return TRUE if successful, FALSE otherwise
 * 
 */
SmlBool smlSessionSendCommand(SmlSession *session, SmlCommand *cmd, SmlCommand *parent, SmlStatusReplyCb callback, void *userdata, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p, %p, %p)", __func__, session, cmd, parent, callback, userdata,  error);
	smlAssert(session);
	smlAssert(cmd);
	
	if (!smlSessionStartCommand(session, cmd, parent, callback, userdata, error))
		goto error;
		
	if (!smlSessionEndCommand(session, parent, error))
		goto error;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/** @brief starts a command with a parent
 * 
 * @param session The session
 * @param cmd The command to send
 * @param parent The parent of this command
 * @param callback The callback that will receive the reply. Set to NULL if you dont want any replies.
 * @param userdata The userdata to pass to the callback
 * @param error A error struct
 * @return TRUE if successful, FALSE otherwise
 * 
 */
SmlBool smlSessionStartCommand(SmlSession *session, SmlCommand *cmd, SmlCommand *parent, SmlStatusReplyCb callback, void *userdata, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p, %p, %p)", __func__, session, cmd, parent, callback, userdata,  error);
	smlAssert(session);
	smlAssert(cmd);
	
	SmlSessionCommand *sesscmd = smlTryMalloc0(sizeof(SmlSessionCommand), error);
	if (!sesscmd)
		goto error;
	
	sesscmd->type = SML_SESSION_COMMAND_START;
	sesscmd->cmd = cmd;
	smlCommandRef(cmd);
	
	if (parent) {
		sesscmd->parent = parent;
		smlCommandRef(parent);
	}
	
	sesscmd->callback = callback;
	sesscmd->callbackUserdata = userdata;
	
	smlQueueSend(session->command_queue, sesscmd);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/** @brief ends a command
 * 
 * @param session The session
 * @param parent The parent of this command, NULL for syncbody
 * @param error A error struct
 * @return TRUE if successful, FALSE otherwise
 * 
 */
SmlBool smlSessionEndCommand(SmlSession *session, SmlCommand *parent, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, parent, error);
	smlAssert(session);
	
	SmlSessionCommand *sesscmd = smlTryMalloc0(sizeof(SmlSessionCommand), error);
	if (!sesscmd)
		goto error;
	
	sesscmd->type = SML_SESSION_COMMAND_END;
	
	if (parent) {
		sesscmd->parent = parent;
		smlCommandRef(parent);
	}
	
	smlQueueSend(session->command_queue, sesscmd);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}


/** @brief Sends a reply to a command
 * 
 * This function sends a status reply to a command.
 * 
 * This function might return an error in the case of an OOM situation or if you forgot to
 * fill in parts of the reply
 * 
 * @param session The session
 * @param status The status to send
 * @param error A error struct
 * @return TRUE if successful, FALSE otherwise
 * 
 */
SmlBool smlSessionSendReply(SmlSession *session, SmlStatus *status, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, status, error);
	smlAssert(session);
	smlAssert(status);
	
	SmlSessionCommand *sesscmd = smlTryMalloc0(sizeof(SmlSessionCommand), error);
	if (!sesscmd)
		goto error;
	
	sesscmd->type = SML_SESSION_STATUS;
	sesscmd->status = status;
	smlStatusRef(status);
	
	smlQueueSendPrio(session->command_queue, sesscmd);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/*@}*/

/*@}*/
