/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <errno.h>
#include <kstat.h>
#include <strings.h>

#include "libsysactivity.h"

static kstat_t* get_next_ksp(kstat_t* ksp);
static int assign_dst(struct sa_data_storage* dst, kstat_t* ksp);

__thread kstat_ctl_t* data_storage_kc;

int sa_open_data_storage() {
	data_storage_kc = kstat_open();
	if (data_storage_kc == NULL)
		return ENOSYS;

	return 0;
}

int sa_close_data_storage() {
	if (data_storage_kc != NULL)
		kstat_close(data_storage_kc);

	return 0;
}

int sa_count_data_storages(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	if(kstat_chain_update(data_storage_kc) == -1)
		return errno;

	kstat_t* ksp = NULL;
	*number = 0;

	while ((ksp = get_next_ksp(ksp)) != NULL) {
		(*number)++;
	}
	return 0;
}

int sa_get_data_storage(char* name, struct sa_data_storage* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	if(kstat_chain_update(data_storage_kc) == -1)
		return errno;

	kstat_t* ksp = NULL;

	while ((ksp = get_next_ksp(ksp)) != NULL) {
		if (strncmp(ksp->ks_name, name, sizeof ksp->ks_name) != 0)
			continue;

		if (assign_dst(dst, ksp) != 0)
			return ENOSYS;
		return 0;
	}

	return ENODEV;
}

int sa_get_data_storages(struct sa_data_storage* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	if(kstat_chain_update(data_storage_kc) == -1) // TODO Is this call really needed?
		return errno;

	uint16_t i = 0;
	kstat_t* ksp = NULL;

	*written = 0;
	while (i < dst_size) {
		ksp = get_next_ksp(ksp);
		if (ksp == NULL)
			return ENODEV;

		if (assign_dst(&dst[i], ksp) != 0)
			return ENOSYS;

		(*written)++;
		i++;
	}
	return 0;
}

static kstat_t* get_next_ksp(kstat_t* ksp) {
	do {
		if (ksp == NULL)
			ksp = data_storage_kc->kc_chain;
		else
			ksp = ksp->ks_next;

		if (ksp == NULL)
			return ksp;
	} while (ksp->ks_type != KSTAT_TYPE_IO);

	return ksp;
}

static int assign_dst(struct sa_data_storage* dst, kstat_t* ksp) {
	kstat_io_t kio;
	if (kstat_read(data_storage_kc, ksp, &kio) == -1)
		return errno;

	strncpy(dst->name, ksp->ks_name, sizeof dst->name);
	dst->reads = kio.reads;
	dst->writes = kio.writes;
	dst->bytes_read = kio.nread;
	dst->bytes_written = kio.nwritten;
	return 0;
}
