// TeOperationStrategy.h: interface for the TeOperationStrategy class.
//
//////////////////////////////////////////////////////////////////////


/*! \file TeOperationStrategy.h
    This file contains functions to compute attributes based on 
	alternative strategies. These strategies (Strategy Design Pattern)
	correspond to operations to derive information based on other  layers attributes and/or
	spatial components (polygonal, raster, point, raster or other cells). 
*/



#ifndef TeComputeAttrStrategies_H
#define TeComputeAttrStrategies_H



#include <TeAttribute.h>
#include <TeLayer.h>
#include <TeTimeInterval.h>
#include <TeGeometryAlgorithms.h>
#include <TeSTEFunctionsDB.h>


TePolygonSet  TeBox2PolygonSet (const TeBox& box);		
TePolygon	  TeBox2Polygon (const TeBox& box);		


///////////////////////////////////////////// Non-spatial Strategy Hierarchy
//!  Abstract class in the strategies hierarchy for computing attribute values based on other layers (Strategy Design Pattern). 


//!  Abstract class in the strategies hierarchy for computing attribute values based on other layers (Strategy Design Pattern). 
template<class Iterator>
class TeComputeAttrStrategy  
{
protected:
	TeComputeAttrStrategy () {};
	TeProperty	defaultValue_;
public:
	TeProperty  defaultValue () {return defaultValue_;}
	virtual  vector<TeProperty> compute (Iterator /*first*/, Iterator /*last*/, const string& /*columnName*/)
		{
			vector<TeProperty> result;
			result.push_back (defaultValue_);
			return result;
		}
	virtual ~TeComputeAttrStrategy(){}

};



template<class Iterator>
class TeAverageStrategy: public TeComputeAttrStrategy<Iterator>  
{
public:

	TeAverageStrategy (double defaultValue = 0.0) {this->defaultValue_.value_ = Te2String (defaultValue);}
	virtual ~TeAverageStrategy() {}
	virtual  vector<TeProperty> compute (Iterator first, Iterator last, const string& columnName)
	{

		double	tot_val = 0.0;
		int		num = 0;

		Iterator it = first;
		while (it != last)
		{
			tot_val += (*it);
			num++;
			++it;
		}

		TeProperty  prop;
		prop.attr_.rep_.name_ = columnName;
		prop.attr_.rep_.type_ = TeREAL;
		prop.attr_.rep_.numChar_ = 48;
		if (num > 0)
		{
			double val = tot_val/num;
			prop.value_ = Te2String (val);
		}
		else prop.value_ = this->defaultValue_.value_;

		TePropertyVector  result;
		result.push_back (prop);
		return result;
	}

};


template<class Iterator>
class TeSumStrategy: public TeComputeAttrStrategy<Iterator>   
{
public:

	TeSumStrategy (double defaultValue = 0.0) {this->defaultValue_.value_ = Te2String (defaultValue);}
	virtual ~TeSumStrategy() {}
	virtual  vector<TeProperty> compute (Iterator first, Iterator last, const string& columnName)
	{  
		double	sum = 0.0;
		Iterator it = first;
		while (it != last)
		{
			sum += (*it);
			++it;
		}

		TeProperty  prop;
	

		if (first == last) prop = this->defaultValue_;
		else  prop.value_ = Te2String (sum);

		prop.attr_.rep_.name_ = columnName;
		prop.attr_.rep_.type_ = TeREAL;
		prop.attr_.rep_.numChar_ = 48;
		TePropertyVector  result;
		result.push_back (prop);
		return result;
	}
};


template< class Iterator>
class TeMinimumStrategy: public TeComputeAttrStrategy<Iterator>   
{
public:

	TeMinimumStrategy (double defaultValue = 0.0) {this->defaultValue_.value_ = Te2String (defaultValue);}
	virtual ~TeMinimumStrategy() {}
	virtual  vector<TeProperty> compute (Iterator first, Iterator last, const string& columnName)
	{
		double	min =  TeMAXFLOAT;
		Iterator it = first;
		while (it != last)
		{
			double val = (*it);
			if (val < min) min = val;
			++it;
		}


		TeProperty  prop;
		prop.attr_.rep_.name_ = columnName;
		prop.attr_.rep_.type_ = TeREAL;
		prop.attr_.rep_.numChar_ = 48;

		if (first == last)
			prop.value_ = this->defaultValue_.value_; 
		else  
			prop.value_ = Te2String (min);

		TePropertyVector  result;
		result.push_back (prop);
		return result;
	}
};



template< class Iterator>
class TeMaximumStrategy: public TeComputeAttrStrategy<Iterator>   
{
public:

	TeMaximumStrategy (double defaultValue = 0.0) {this->defaultValue_.value_ = Te2String (defaultValue);}
	virtual ~TeMaximumStrategy() {}
	virtual  vector<TeProperty> compute (Iterator first, Iterator last, const string& columnName)
	{
		double	max =  TeMINFLOAT;
		Iterator it = first;
		while (it != last)
		{
			double val = (*it);
			if (val > max) max = val;
			++it;
		}


		TeProperty  prop;
		prop.attr_.rep_.name_ = columnName;
		prop.attr_.rep_.type_ = TeREAL;
		prop.attr_.rep_.numChar_ = 48;

		if (first == last)
			prop.value_ = this->defaultValue_.value_; 
		else  
			prop.value_ = Te2String (max);

		TePropertyVector  result;
		result.push_back (prop);
		return result;
	}
};


template<class Iterator>
class TeCategoryPercentageStrategy: public TeComputeAttrStrategy<Iterator>   
{
private:
	map<string, string> classesMap_;


public:
	TeCategoryPercentageStrategy (map<string, string>& classes, double defaultValue = 0.0) 
	{
		this->defaultValue_.value_ = Te2String (defaultValue); 
		classesMap_ = classes;
	}
	virtual ~TeCategoryPercentageStrategy() {}
	virtual  vector<TeProperty> compute (Iterator first, Iterator last, const string& columnName)
	{
		TeProperty category;
		int num = 0;

		// initialize count
		map<string, int>  count;
		map<string, string>::iterator itMap = classesMap_.begin();
		while (itMap != classesMap_.end())
		{
			count[(*itMap).second] = 0;
			++itMap;
		}

		Iterator it = first;
		while (it != last)
		{
			if (it.getProperty (category))
			{
				count[classesMap_[category.value_]]++;
				num++;
			}
			++it;
		}

		TePropertyVector  result;
		map <string, int>:: iterator count_it = count.begin();
		while(count_it != count.end())  
		{
			TeProperty prop;
			prop.attr_.rep_.name_ = columnName + (*count_it).first; // category
			prop.attr_.rep_.type_ = TeREAL;
			prop.attr_.rep_.numChar_ = 48;

			double percent = 0.0;
			string cat = (*count_it).first;
			double value = (double)(*count_it).second;
			if (num != 0) percent = value/num;
			prop.value_    = Te2String (percent); // count
			result.push_back (prop);
			++count_it;
		}
		return result;
	}

};


template<class Iterator>
class TePresenceStrategy: public TeComputeAttrStrategy<Iterator>   
{
public:
	TePresenceStrategy (bool defaultValue = 0) 
	{
		this->defaultValue_.attr_.rep_.type_ = TeINT;
		this->defaultValue_.value_ = Te2String (defaultValue); 
		this->defaultValue_.attr_.rep_.numChar_ = 48;

	}
	virtual ~TePresenceStrategy() {}
	virtual  vector<TeProperty> compute (Iterator first, Iterator last, const string& columnName)
	{
		TeProperty prop;
		if (first != last) 
			prop.value_ = "1";
		else
			prop = this->defaultValue_;

		TePropertyVector  result;
		prop.attr_.rep_.name_ = columnName; 
		prop.attr_.rep_.type_ = TeINT;
		prop.attr_.rep_.numChar_ = 48;
		result.push_back (prop);
		return result;
	}

};
	
template<class Iterator>
class TeMajorityStrategy: public TeComputeAttrStrategy<Iterator>   
{
public:
	TeMajorityStrategy (TeAttrDataType defaultType = TeINT) 
	{
		this->defaultValue_.attr_.rep_.type_ = defaultType;
		this->defaultValue_.value_ = "0"; 
	}
	virtual ~TeMajorityStrategy() {}
	virtual  vector<TeProperty> compute (Iterator first, Iterator last, const string& columnName)
	{
		TeProperty category;
		category.attr_.rep_.type_ = this->defaultValue_.attr_.rep_.type_;

		map<string, int>  count;
		Iterator it = first;
		while (it != last)
		{
			if (it.getProperty (category))
				count[category.value_]++;
			++it;
		}

		TeProperty prop;
		prop.attr_.rep_.type_ = category.attr_.rep_.type_;
		prop.value_ = this->defaultValue_.value_;
		prop.attr_.rep_.name_ = columnName; 
		prop.attr_.rep_.numChar_ = 48;


		int max = 0;
		map <string, int>:: iterator count_it = count.begin();
		while(count_it != count.end())  
		{
			if ((*count_it).second > max)
			{
				prop.value_ = (*count_it).first;
				max = (*count_it).second;
			}
			++count_it;		
		}
		TePropertyVector  result;
		result.push_back (prop);
		return result;
	}

};
	
template<class Iterator>
class TeMajorityCategoryStrategy: public TeComputeAttrStrategy<Iterator>   
{
	protected:
			map<string, string> classesMap_;

public:
	TeMajorityCategoryStrategy (const map<string, string>& classesMap, TeAttrDataType defaultType = TeINT) 
	{
		this->defaultValue_.attr_.rep_.type_ = defaultType;
		this->defaultValue_.value_ = "0"; 
		classesMap_ = classesMap;
	}
	virtual ~TeMajorityCategoryStrategy() {}
	virtual  vector<TeProperty> compute (Iterator first, Iterator last, const string& columnName)
	{
		TeProperty category;
		category.attr_.rep_.type_ = this->defaultValue_.attr_.rep_.type_;


			// initialize count
		map<string, int>  count;
		map<string, string>::iterator itMap = classesMap_.begin();
		while (itMap != classesMap_.end())
		{
			count[(*itMap).second] = 0;
			++itMap;
		}

		Iterator it = first;
		while (it != last)
		{
			if (it.getProperty (category))
			{
				count[classesMap_[category.value_]]++;
	
			}
			++it;
		}



	TePropertyVector  result;
	TeProperty prop;
	prop.attr_.rep_.type_ = TeSTRING;
//	prop.value_ = defaultValue_.value_;
	prop.attr_.rep_.name_ = columnName; 
	prop.attr_.rep_.numChar_ = 48;

	int max = 0;
	map <string, int>:: iterator count_it = count.begin();
	while(count_it != count.end())  
	{
		if ((*count_it).second > max)
		{
			prop.value_ = (*count_it).first;
			max = (*count_it).second;
		}
		++count_it;		
	}


	result.push_back (prop);
	return result;

}
};
	
///////////////////////////////////////////////////// Spatial Strategy Hierarchy

		

//!  Abstract class in the strategies hierarchy for computing attribute values based on other layers (Strategy Design Pattern). 

class TeComputeAttrSpatialStrategy  
{
	protected:
		TeComputeAttrSpatialStrategy () {}
	public:
		virtual ~TeComputeAttrSpatialStrategy() {}
		virtual  TePropertyVector compute (TeTheme* /* theme */, const string& /* attrName */, TeBox /* box */, const string& /* columnName */)
		{
			TePropertyVector result;
			return result;
		}

};


class TeCategoryAreaPercentageStrategy: public TeComputeAttrSpatialStrategy  
{
protected:
			map<string, string> classesMap_;
public:
		TeCategoryAreaPercentageStrategy (const map<string, string>& classesMap) {classesMap_ = classesMap;};
		virtual ~TeCategoryAreaPercentageStrategy() {}
		virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
		
};



class TeAverageWeighbyAreaStrategy: public TeComputeAttrSpatialStrategy  
{
public:
	TeAverageWeighbyAreaStrategy () {}
	virtual ~TeAverageWeighbyAreaStrategy() {}
	virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
};



class TeCategorySoilsAreaPercentageStrategy: public TeComputeAttrSpatialStrategy  
{
protected:
			map<string, string> classesMap_;
public:
		TeCategorySoilsAreaPercentageStrategy (const map<string, string>& classesMap) {classesMap_ = classesMap;};
		virtual ~TeCategorySoilsAreaPercentageStrategy() {}
		virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);	
		
};


class TeCategoryMajorityStrategy: public TeComputeAttrSpatialStrategy  
{
protected:
			map<string, string> classesMap_;
public:
		TeCategoryMajorityStrategy (const map<string, string>& classesMap) {classesMap_ = classesMap;};
		virtual ~TeCategoryMajorityStrategy() {}
		virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);	
		
};






class TeTotalAreaPercentageStrategy: public TeComputeAttrSpatialStrategy  
{

public:
		TeTotalAreaPercentageStrategy () {}
		virtual ~TeTotalAreaPercentageStrategy() {}
		virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
};





class TeCountPolygonalObjectsStrategy: public TeComputeAttrSpatialStrategy  
{

public:
		TeCountPolygonalObjectsStrategy () {}
		virtual ~TeCountPolygonalObjectsStrategy() {}
		virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
};





class TeCountLineObjectsStrategy: public TeComputeAttrSpatialStrategy  
{

public:
		TeCountLineObjectsStrategy () {}
		virtual ~TeCountLineObjectsStrategy() {}
		virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
};




class TeCountPointObjectsStrategy: public TeComputeAttrSpatialStrategy  
{

public:
		TeCountPointObjectsStrategy () {}
		virtual ~TeCountPointObjectsStrategy() {}
		virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
};





class TeLineLengthStrategy: public TeComputeAttrSpatialStrategy  
{
public:
	TeLineLengthStrategy () {}
		virtual ~TeLineLengthStrategy() {}
		virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
};

class TeMinimumDistancePolygonsStrategy: public TeComputeAttrSpatialStrategy  
{
public:
	TeMinimumDistancePolygonsStrategy () {}
		virtual ~TeMinimumDistancePolygonsStrategy() {}
		virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
};

class TeMinimumDistanceLinesStrategy: public TeComputeAttrSpatialStrategy  
{
public:
	TeMinimumDistanceLinesStrategy () {}
		virtual ~TeMinimumDistanceLinesStrategy() {}
		virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
};

class TeMinimumDistancePointsStrategy: public TeComputeAttrSpatialStrategy  
{
public:
	TeMinimumDistancePointsStrategy () {}
		virtual ~TeMinimumDistancePointsStrategy() {}
		virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
};



class TeSumWeighByAreaStrategy: public TeComputeAttrSpatialStrategy  
{
public:
	TeSumWeighByAreaStrategy () {}
	virtual ~TeSumWeighByAreaStrategy() {}
	virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
};




class TeSumWeighByRestrictedAreaStrategy: public TeComputeAttrSpatialStrategy  
{
private:
		TeRaster* raster_;
		map<int, int> classesSelection_;
		int		band_;
public:
	TeSumWeighByRestrictedAreaStrategy (TeRaster* raster, map<int, int> classesSelection, int band = 0){
		raster_ = raster; 
		classesSelection_ = classesSelection;
		band_ = band;}
	virtual ~TeSumWeighByRestrictedAreaStrategy() {}
	virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
};


class TeAverageWeighByRestrictedAreaStrategy: public TeComputeAttrSpatialStrategy  
{
private:
		TeRaster* raster_;
		map<int, int> classesSelection_;
		int		band_;
public:
	TeAverageWeighByRestrictedAreaStrategy (TeRaster* raster, map<int, int> classesSelection, int band = 0){
		raster_ = raster; 
		classesSelection_ = classesSelection;
		band_ = band;}
	virtual ~TeAverageWeighByRestrictedAreaStrategy() {}
	virtual  TePropertyVector compute (TeTheme* theme, const string& attrName, TeBox box, const string& columnName);
};

#endif
