/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeDecoderMemoryMap.h"
#include "TeUtils.h"
#include "TeAsciiFile.h"
#include <map>
#include <string>

static inline short swaps(short value)
{
    short svalue = ((value & 0x00ff) << 8) | ((value >> 8) & 0x00ff);
	return svalue;
}

TeDecoderMemoryMap::TeDecoderMemoryMap ( const TeRasterParams& par )
{
	params_ = par;
	dataInitPos_ = 0;
	dataInitPos_ = par.offset_;
	m_lpszFile = 0;
	m_hFile = 0;
	params_.decoderIdentifier_ = "MEMMAP";
}

TeDecoderMemoryMap::~TeDecoderMemoryMap ()
{
	if (m_hFile != 0)
		clear();
}
bool
TeDecoderMemoryMap::getElement (int col,int lin, double& val,int band)
{
	unsigned long position = dataInitPos_+(unsigned long)params_.nBands()*(unsigned long)(params_.ncols_*lin+col)+band;
	switch (params_.dataType_[0])
	{
	case (TeUNSIGNEDCHAR):
		val = ((unsigned char*)m_lpszFile)[position];
		break;
	case (TeCHAR) :
		val = ((char*) m_lpszFile)[position];
		break;
	case (TeUNSIGNEDSHORT):
		unsigned short uval;
		if (params_.swap_)
			uval = swaps(((unsigned short*)m_lpszFile)[position]);
		else
			uval = ((unsigned short*)m_lpszFile)[position];
		val = uval;
		break;
	case (TeSHORT):
		if (params_.swap_)
			val = swaps(((short*)m_lpszFile)[position]);
		else
			val = ((short*)m_lpszFile)[position];
		break;
	case (TeINTEGER):
		val = ((int*)m_lpszFile)[position];
	case (TeUNSIGNEDLONG):
		val = ((unsigned long*)m_lpszFile)[position];
		break;
	case (TeLONG):
		val = ((long*)m_lpszFile)[position];
		break;
	case (TeFLOAT):
		val = ((float*)m_lpszFile)[position];
		break;
	case (TeDOUBLE):
		val = ((double*)m_lpszFile)[position];
		break;
	default:
		return false;
	}
	return true;
}

bool
TeDecoderMemoryMap::setElement (int col,int lin, double val, int band)
{
	unsigned int position = dataInitPos_+params_.nBands()*(params_.ncols_*lin+col)+band;
	switch (params_.dataType_[0])
	{
	case (TeUNSIGNEDCHAR):
		((unsigned char*)m_lpszFile)[position] = (unsigned char) val;
		break;
	
	case (TeCHAR) :
		((char*) m_lpszFile)[position] = (char) val;
		break;

	case (TeUNSIGNEDSHORT):
		((unsigned short*)m_lpszFile)[position] = (unsigned short) val;
		break;

	case (TeSHORT):
		if (params_.swap_)
			((short*)m_lpszFile)[position] = swaps((short) val);
		else
			((short*)m_lpszFile)[position] = (short) val;
		break;

	case (TeINTEGER):
		((int*)m_lpszFile)[position] = (int) val;
		break;

	case (TeUNSIGNEDLONG):
		((unsigned long*)m_lpszFile)[position] = (unsigned long) val;
		break;
	
	case (TeLONG):
		((long*)m_lpszFile)[position] = (long) val;
		break;

	case (TeFLOAT):
		((float*)m_lpszFile)[position] = (float) val;
		break;
	
	case (TeDOUBLE):
		((double*)m_lpszFile)[position] = val;
	default:
		return false;
	}
	return true;
}

void
TeDecoderMemoryMap::writeMetadataFile()
{
	string metFileName = TeGetName (params_.fileName_.c_str())+".met";
	try {
		TeAsciiFile metFile(metFileName,"w");
		string line = "%Raw raster file metadata\n";
		metFile.writeString(line);
		line = "NROWS " + Te2String(params_.nlines_);
		metFile.writeString(line);
		metFile.writeNewLine();
		line = "NCOLS " + Te2String(params_.ncols_);
		metFile.writeString(line);
		metFile.writeNewLine();
		line = "NBANDS " + Te2String(params_.nBands());
		metFile.writeString(line);
		metFile.writeNewLine();

		if (params_.nBands() > 1)
		{
			line = "INTERLEAVING ";
			if (params_.interleaving_ == TePerPixel)
				line += "perpixel";
			else if (params_.interleaving_ == TePerLine)
				line += "perline";
			else
				line += "perband";
			metFile.writeString(line);
			metFile.writeNewLine();
		}

		line = "DATATYPE ";
		if (params_.dataType_[0] == TeBIT)
			line += "bit";
		else if (params_.dataType_[0] == TeUNSIGNEDCHAR)
			line += "unsignedchar";
		else if (params_.dataType_[0] == TeCHAR)
			line += "char";
		else if (params_.dataType_[0] == TeUNSIGNEDSHORT)
			line += "unsignedshort";
		else if (params_.dataType_[0] == TeSHORT)
			line += "short";
		else if (params_.dataType_[0] == TeINTEGER)
			line += "integer";
		else if (params_.dataType_[0] == TeUNSIGNEDLONG)
			line += "unsignedlong";
		else if (params_.dataType_[0] == TeLONG)
			line += "long";
		else if (params_.dataType_[0] == TeFLOAT)
			line += "float";
		else if (params_.dataType_[0] == TeDOUBLE)
			line += "double";		
		metFile.writeString(line);
		metFile.writeNewLine();
		
		if (params_.swap_)
		{
			line = "SWAPPED yes";
			metFile.writeString(line);
			metFile.writeNewLine();
		}

		if (params_.useDummy_)
		{
			line += "NO_DATA " + Te2String(params_.dummy_[0]);
			metFile.writeString(line);
			metFile.writeNewLine();
		}
		
		line = "RESOLUTION_X " + Te2String(params_.resx_);
		metFile.writeString(line);
		metFile.writeNewLine();
		line = "RESOLUTION_Y " + Te2String(params_.resy_);
		metFile.writeString(line);
		metFile.writeNewLine();
		line = "LOWERLEFT_X " + Te2String(params_.box().x1_);
		metFile.writeString(line);
		metFile.writeNewLine();
		line = "LOWERLEFT_Y " + Te2String(params_.box().y1_);
		metFile.writeString(line);
		metFile.writeNewLine();
		line = "PROJECTION " + params_.projection()->name() + " ";
		line += params_.projection()->describe();
		metFile.writeString(line);
		metFile.writeNewLine();
		line = "DATUM " +  params_.projection()->datum().name();
		metFile.writeString(line);
		metFile.writeNewLine();
	}
	catch(...)
	{
	}
}

void 
TeDecoderMemoryMap::readMetadataFile()
{
	// read all keys in the metadata file (there is no order expected) 
	string metFileName = TeGetName (params_.fileName_.c_str())+".met";
	map<string,string> metadata;
	try {
		TeAsciiFile metFile(metFileName,"r");
		string key, value;
		while (metFile.isNotAtEOF())
		{
			key = metFile.readString();
			if (key[0] == '%')
			{
				metFile.findNewLine();
				continue;
			}
			value = metFile.readString();
			metadata.insert(make_pair(key,value));
			metFile.findNewLine();
		}
	}
	catch(...)
	{
	}
	if (!metadata.empty())
	{
		double xll=0, yll=0;
		map<string,string>::iterator it;
		it = metadata.find("NROWS");
		if (it != metadata.end())
			params_.nlines_ = atoi(it->second.c_str());
		it = metadata.find("NCOLS");
		if (it != metadata.end())
			params_.ncols_ = atoi(it->second.c_str());
		it = metadata.find("NBANDS");
		if (it != metadata.end())
			params_.nBands(atoi(it->second.c_str()));
		else
			params_.nBands(1);
		it = metadata.find("RESOLUTION_X");
		if (it != metadata.end())
			params_.resx_ = atof(it->second.c_str());
		it = metadata.find("RESOLUTION_Y");
		if (it != metadata.end())
			params_.resy_ = atof(it->second.c_str());
		it = metadata.find("LOWERLEFT_X");
		if (it != metadata.end())
		{
			xll = atof(it->second.c_str());
			it = metadata.find("LOWERLEFT_Y");
			if (it != metadata.end())
			{
				yll = atof(it->second.c_str());
				params_.lowerLeftResolutionSize(xll,yll, params_.resx_, params_.resy_,
										params_.ncols_, params_.nlines_);
			}
		}

		it = metadata.find("INTERLEAVING");
		if (it != metadata.end())
		{
			if (it->second == "perline")
				params_.interleaving_ = TePerLine;
			else if (it->second == "perband")
				params_.interleaving_ = TePerBand;
			else params_.interleaving_ = TePerPixel;
		}

		it = metadata.find("DATATYPE");
		if (it != metadata.end())
		{
			if (it->second == "bit")
				params_.setDataType(TeBIT);
			else if (it->second == "unsignedchar")
				params_.setDataType(TeBIT);
			else if (it->second == "char")
				params_.setDataType(TeCHAR);
			else if (it->second == "unsignedshort")
				params_.setDataType(TeUNSIGNEDSHORT);
			else if (it->second == "short")
				params_.setDataType(TeSHORT);
			else if (it->second == "integer")
				params_.setDataType(TeINTEGER);
			else if (it->second == "unsignedlong")
				params_.setDataType(TeUNSIGNEDLONG);
			else if (it->second == "long")
				params_.setDataType(TeLONG);
			else if (it->second == "float")
				params_.setDataType(TeFLOAT);
			else if (it->second == "double")
				params_.setDataType(TeDOUBLE);
		}
		it = metadata.find("SWAPPED");
		if (it != metadata.end() && it->second == "yes")
			params_.swap_ = true;

		it = metadata.find("NO_DATA");
		if (it != metadata.end())
		{
			params_.useDummy_ = true;
			params_.setDummy(atof(it->second.c_str()));
		}
			
		it = metadata.find("PROJECTION");
		if (it != metadata.end())
		{
			string projdesc = it->second;
			TeProjectionParams pars;
			it = metadata.find("DATUM");
			if (it != metadata.end())
			{
				TeDatum dat = TeDatumFactory::make(it->second);
				pars.datum = dat;
			}
			if (decodifyDescription(projdesc,pars))
			{
				TeProjection* proj = TeProjectionFactory::make(pars);
				params_.projection(proj);
			}
		}
	}
}

#ifdef WIN32
void
TeDecoderMemoryMap::init()
{
	clear();
	params_.status_= TeNOTREADY;

	DWORD dwDesiredAccess, dwCreationDisposition, flProtect,dwDesiredAccessV ;
	
	if (params_.mode_ == 'c')
	{
		dwCreationDisposition = CREATE_ALWAYS;
		dwDesiredAccess = GENERIC_READ | GENERIC_WRITE;
		flProtect = PAGE_READWRITE;
		dwDesiredAccessV = FILE_MAP_WRITE;
	}
	else if (params_.mode_ == 'w')
	{
		dwCreationDisposition = OPEN_ALWAYS;
		dwDesiredAccess = GENERIC_READ | GENERIC_WRITE;		
		flProtect = PAGE_READWRITE;
		dwDesiredAccessV = FILE_MAP_WRITE;
	}
	else
	{
		dwCreationDisposition = OPEN_EXISTING;
		dwDesiredAccess = GENERIC_READ;		
		flProtect = PAGE_READONLY;
		dwDesiredAccessV = FILE_MAP_READ;
	}

	// First open the file
	m_hFile = CreateFile(
		params_.fileName_.c_str(),	// File name
		dwDesiredAccess,			// Desired access
		FILE_SHARE_READ
		| FILE_SHARE_WRITE,		// Allow sharing-- we're only doing a quick scan
		NULL,					// No security attributes
		dwCreationDisposition,	// Creation disposition
		0,						// Ignore file attributes
		NULL);					// Ignore hTemplateFile

	if (m_hFile == INVALID_HANDLE_VALUE)
			return;		// could not open file
	
	// Get the file's size
	m_dwSize = GetFileSize(m_hFile, NULL);
	if (m_dwSize == 0xffffffff)
	{
		m_hFile = NULL;
		return;
	}
	
	bool fillDummy = false;
	if (params_.mode_ == 'c' ||  m_dwSize == 0)	
	{
		if (!create())
			return;
		fillDummy = true;
		// if creating a raw file write the metata file
		writeMetadataFile();
	}
	else
	{
		// if reading an existing raw file try to read the metadata file
		readMetadataFile();
	}

	// Create a mapping object from the file
	m_hMapping = CreateFileMapping(
		m_hFile,				// Handle we got from CreateFile
		NULL,					// No security attributes
		flProtect,			// read-write
		0, 0,					// Max size = current size of file
		NULL);					// Don't name the mapping object
	if ( m_hMapping == NULL )
	{
		m_hFile = NULL;
		return;
	}

// Map the file to memory
	switch (params_.dataType_[0]) {
	case (TeUNSIGNEDCHAR):
		m_lpszFile = (unsigned char*) MapViewOfFile(
		m_hMapping,					// Handle we got from CreateFileMapping
		dwDesiredAccessV,				// full access
		0, 0,						// Offset into file = beginning of file
		0);
		break;
	case (TeCHAR) :
		m_lpszFile = (char*) MapViewOfFile(
		m_hMapping,					// Handle we got from CreateFileMapping
		dwDesiredAccessV,				// full access
		0, 0,						// Offset into file = beginning of file
		0);
		break;
	case (TeUNSIGNEDSHORT):
		m_lpszFile = (unsigned short*) MapViewOfFile(
		m_hMapping,					// Handle we got from CreateFileMapping
		dwDesiredAccessV,				// full access
		0, 0,						// Offset into file = beginning of file
		0);
		break;
	case (TeSHORT):
		m_lpszFile = (short*) MapViewOfFile(
		m_hMapping,					// Handle we got from CreateFileMapping
		dwDesiredAccessV,				// full access
		0, 0,						// Offset into file = beginning of file
		0);
		break;
	case (TeUNSIGNEDLONG):
		m_lpszFile = (unsigned long*) MapViewOfFile(
		m_hMapping,					// Handle we got from CreateFileMapping
		dwDesiredAccessV,				// full access
		0, 0,						// Offset into file = beginning of file
		0);
		break;
	case (TeLONG):
		m_lpszFile = (long*) MapViewOfFile(
		m_hMapping,					// Handle we got from CreateFileMapping
		dwDesiredAccessV,				// full access
		0, 0,						// Offset into file = beginning of file
		0);
		break;
	case (TeFLOAT):
		m_lpszFile = (float*) MapViewOfFile(
		m_hMapping,					// Handle we got from CreateFileMapping
		dwDesiredAccessV,				// full access
		0, 0,						// Offset into file = beginning of file
		0);
		break;
	case (TeDOUBLE):
		m_lpszFile = (double*) MapViewOfFile(
		m_hMapping,					// Handle we got from CreateFileMapping
		dwDesiredAccessV,				// full access
		0, 0,						// Offset into file = beginning of file
		0);
		break;
	}
	if ( m_lpszFile == NULL )
		return ;

	if (fillDummy)
	{
		for (int b=0; b<params_.nBands();b++)
			for (int l=0; l<params_.nlines_;l++)
				for (int c=0; c<params_.ncols_;c++)
					setElement ( c, l, params_.dummy_[b], b);
	}
	if (params_.mode_ =='c' || params_.mode_ == 'w')
		params_.status_ = TeREADYTOWRITE;
	else 
		params_.status_ = TeREADYTOREAD;	
	return;
}


bool
TeDecoderMemoryMap::create()
{
	char			lpBuffer[1024];
	unsigned long	nNumberOfBytesToWrite=(long)params_.nBands() * (long)params_.ncols_ * (long)params_.nlines_;    // number of bytes to write
	nNumberOfBytesToWrite *= (long)params_.nbitsperPixel_[0]/8;    // number of bytes to write
	unsigned long	nNumberOfBytesWritten;  // pointer to number of bytes written

	for (;nNumberOfBytesToWrite > 1024; nNumberOfBytesToWrite-=1024)
	{
		if ( !WriteFile(
			m_hFile,                // handle to file to write to
			lpBuffer,               // pointer to data to write to file
			1024,					// number of bytes to write
			&nNumberOfBytesWritten,  // pointer to number of bytes written
			NULL					// pointer to structure for overlapped I/O
			))
 			return false;			// could not write to file
	}

	if ( !WriteFile(
		m_hFile,                // handle to file to write to
		lpBuffer,               // pointer to data to write to file
		nNumberOfBytesToWrite,	// number of bytes to write
		&nNumberOfBytesWritten,  // pointer to number of bytes written
		NULL					// pointer to structure for overlapped I/O
		))
 		return false;			// could not write to file
	return true;
}

bool
TeDecoderMemoryMap::clear()
{
	if (m_hFile == NULL)
		return true;

	if ( !UnmapViewOfFile(m_lpszFile) )
		return false;

	CloseHandle(m_hMapping);
	CloseHandle(m_hFile);
	m_hFile = NULL;
	params_.status_ = TeNOTREADY;
	return true;
}
#else
// Linux version: uses mmap
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
void
TeDecoderMemoryMap::init()
{
	clear();

	// First open the file
	m_hFile = open(params_.fileName_.c_str(),	// File name
		              O_RDWR,				// Flags
			      S_IRWXU);				// Mode

	if (m_hFile == -1)
		return;				// could not open file

	// Get the file's size
	struct stat aux;
	if (stat(params_.fileName_.c_str(), &aux) == -1)
		return;
	m_dwSize = aux.st_size;

	m_lpszFile  =  mmap(0, m_dwSize, PROT_READ|PROT_WRITE , MAP_PRIVATE, m_hFile, 0);
	if ( ((int) m_lpszFile)  == -1 )
		return;
}

bool
TeDecoderMemoryMap::create()
{
	clear();
	// First open the file
	m_hFile = open(params_.fileName_.c_str(),	// File name
		              O_RDWR,				// Flags
			      S_IRWXU);				// Mode

	if (m_hFile == -1)
		return false;				// could not open file

	char			lpBuffer[1024];
	long	nNumberOfBytesToWrite=(long)params_.nBands() * (long)params_.ncols_ * (long)params_.nlines_;    // number of bytes to write
	nNumberOfBytesToWrite *= (long)params_.nbitsperPixel_[0]/8;    // number of bytes to write
	long	nNumberOfBytesWritten;  // pointer to number of bytes written

	for (;nNumberOfBytesToWrite > 1024; nNumberOfBytesToWrite-=1024)
	{
		nNumberOfBytesWritten = write(m_hFile, lpBuffer,1024);
		if (nNumberOfBytesWritten == -1)
 		   return false;			// could not write to file
	}

	nNumberOfBytesWritten = write(m_hFile, lpBuffer,nNumberOfBytesToWrite);
	if (nNumberOfBytesWritten == -1)
 	    return false;

	// Create a mapping object from the file

	struct stat aux;
	if (stat(params_.fileName_.c_str(), &aux) == -1) 	// Get the file's size
		return false;
	m_dwSize = aux.st_size;

	m_lpszFile =  mmap(0, m_dwSize, PROT_READ|PROT_WRITE , MAP_PRIVATE, m_hFile, 0);
	if ( ((int) m_lpszFile)  == -1 )
		return false;

	// Writes the default values
	if (params_.dummy_[0])
	{
		for (int b=0; b<params_.nBands();b++)
			for (int l=0; l<params_.nlines_;l++)
				for (int c=0; c<params_.ncols_;c++)
					setElement ( c, l, params_.dummy_[b], b);
	}
	return true;
}


bool
TeDecoderMemoryMap::clear()
{
	if (m_lpszFile == 0)
		return true;

	if ( munmap(m_lpszFile, m_dwSize) == -1)
		return false;
	m_lpszFile = 0;
	close(m_hFile);
	m_hFile = 0;
	return true;
}
#endif


