/*  tiffep - Ti File Format Engine & Proxy
 *  Copyright (C) 2000  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
  (Linux part)
  This unit is used to create and manage sockets via
  Linux sockets.

  Remark: this unit has been primarily written with sockets and rewritten with
  pipes because AF_UNIX & SOCK_STREAM combination protocol is not supported by
  Windows Sockets !
  Now, I use named pipes for Linux & anonymous pipes for Windows (named pipes 
  are only supported under WinNT. Anonymous pipes are supported both under 
  Win9x and NT even if they are emulated by named pipes under NT).
  Windows: I believe that I am going to be mad !!!
*/

#ifdef __LINUX__

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <fcntl.h>
#include <unistd.h>

#include "hfiles.h"
#include "tiffep.h"

//static TiffepSockAddr name;
enum PIPES { READ, WRITE };

#define SOCKET_NAME_S SOCKET_NAME"_s"
#define SOCKET_NAME_C SOCKET_NAME"_c"

/* 
   Create a socket
   - sd: a socket descriptor
   Returns 0 if successful.
*/
int socket_create(TiffepSock *sd, int server)
{
  char name[MAXCHARS];

  DISPLAY2("socket_create\n");

  if(server == SERVER)
    {  
      strcpy(name, SOCKET_NAME);
      strcat(name, "_s");
    }
  else
    {
      strcpy(name, SOCKET_NAME);
      strcat(name, "_c");
    }

  /* Check if the pipes already exist else create them */
  unlink(name);
  if(access(name, F_OK))
    {
      if(mkfifo(name, S_IRWXU) == -1)
	{
	  print_last_error("socket_create");
	  return ERR_SCK_SOCKET;
	}
    }

  return 0;
}

/*
  Bind the socket: assign a name to it
  - sd: a socket descriptor
  Returns 0 if successful.
*/

static int tfd = -1;

int socket_bind(TiffepSock sd)
{ 
  DISPLAY2("socket_bind\n");

  return 0;
}

/*
  Listen the socket for connections
  - sd: a socket descriptor
  - max_log: the maximum number of authorized connections
  Returns 0 if successful
*/
int socket_listen(TiffepSock sd, int max_log)
{
  char name[MAXCHARS];

  DISPLAY2("socket_listen\n");

  strcpy(name, SOCKET_NAME);
  strcat(name, "_s");

  /* Open the server pipe in reading */
  tfd = open(name, O_RDONLY); // blocking mode
  if(tfd == -1)
    {
      print_last_error("socket_listen");
      return ERR_SCK_LISTEN;
    }
  //DISPLAY("tfd = %i\n", tfd);
  
  return 0;
}

/*
  Accept a connection from a socket
  - sd: a socket descriptor
  - name: return the address to assign the socket
  - nsd: return  the new socket descriptor
  Returns 0 if success.
*/
int socket_accept(TiffepSock sd, TiffepSock *nsd)
{  
  int r;
  int fd;
  char buf[MAXCHARS];

  DISPLAY2("socket_accept\n");

  /* Retrieve name of client pipe */
  while( (r = read(tfd, buf, 10)) == 0);
  if (r <= 0)
    {
      print_last_socket_error("socket_accept");
      return ERR_SCK_ACCEPT;
    }
  //DISPLAY2("name = %s\n", buf);

  /* Open client pipe in writing */
  if((fd = open(buf, O_WRONLY)) == -1) // blocking mode
    {
      print_last_socket_error("socket_accept");
      return ERR_SCK_ACCEPT;
    }
  nsd->wfd = fd;
  nsd->rfd = dup(tfd);
  
  //DISPLAY2("Accepted: r=%i, w=%i\n", nsd->rfd, nsd->wfd);

  return 0;
}

/*
  Send data to a socket
  - sd: a connected socket descriptor
  - data: data to send
  - len: length of data
  Return 0 if success.
*/
int socket_send(TiffepSock sd, void *data, int len)
{
  //DISPLAY2("socket_send: %i\n", len);
  if(write(sd.wfd, data, len) < 0)
    {
      print_last_socket_error("socket_send");
      return ERR_SCK_SEND;
    }
  
  return 0;
}

/*
  Receive data from a socket
  - sd: a connected socket descriptor
  - data: data to receive
  - len: length of data buffer
  Return the number of received bytes.
*/
int socket_recv(TiffepSock nsd, void *data, int len, int *n)
{
  int r = 0;

  while( (r = read(nsd.rfd, data, len)) == 0);
  *n = r;
  if(r > 0)
    {
      //fprintf(stdout, "# of received bytes: %i\n", n);
    }
  else
    {
      print_last_socket_error("socket_recv");
      return ERR_SCK_RECEIVE;
    }
  
  //DISPLAY2("socket_recv: %i\n", r);

  return 0;
}

/*
  Connect the socket
  - sd: a socket descriptor
  - name: the address to assign to the socket
  Returns 0 if successful.
*/
/* (TiffepSockAddr *) &name */
int socket_connect(TiffepSock *sd)
{
  int r;
  char name[MAXCHARS];

  DISPLAY2("socket_connect\n");

  /* Open the server pipe in writing */
  strcpy(name, SOCKET_NAME);
  strcat(name, "_s");
  sd->wfd = open(name, O_WRONLY); // blocking mode
  if(sd->wfd == -1)
    {
      print_last_error("socket_connect");
      return ERR_SCK_CONNECT;
    }

  /* Transmit name of client pipe trough server pipe */
  strcpy(name, SOCKET_NAME);
  strcat(name, "_c");
  r =write(sd->wfd, name, strlen(name) + 1);
  if(r == -1)
    {
      print_last_error("socket_connect");
      return ERR_SCK_CONNECT;
    }

  /* Open the client pipe in reading */
  strcpy(name, SOCKET_NAME);
  strcat(name, "_c");
  sd->rfd = open(name, O_RDONLY); // blocking mode
  if(sd->rfd == -1)
    {
      print_last_error("socket_connect");
      return ERR_SCK_CONNECT;
    }
  //DISPLAY2("Connected: r=%i, w=%i\n", sd->rfd, sd->wfd);

  return 0;
}

int socket_send_msg(TiffepSock sd, TiffepMsg *m)
{
  //DISPLAY2("socket_send_msg: %i\n", sizeof(TiffepMsg));
  if(write(sd.wfd, m, sizeof(TiffepMsg)) < 0)
  {
      print_last_socket_error("socket_send_msg");
      return ERR_SCK_SEND;
    }

  return 0;
}

int socket_recv_msg(TiffepSock sd, TiffepMsg *m, int *read)
{
  int err = 0;
  int n;

  *read = 0;
  err = socket_recv(sd, m, sizeof(TiffepMsg), &n);
  //n = n / sizeof(TiffepMsg);
  *read = n;

  return err;
}

/* 
   Close a socket
   - sd: a socket descriptor
   Returns 0 if successful.
*/
int socket_close(TiffepSock sd)
{
  DISPLAY2("socket_close\n");
  //DISPLAY("r=%i w=%i\n", sd.rfd, sd.wfd);
  close(sd.rfd);
  close(sd.wfd);
  
  return 0;
}

#endif




